% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Functions.R
\name{obj_value}
\alias{obj_value}
\title{Estimate the average response value of the input data given a treatment regime}
\usage{
obj_value(
  X,
  y,
  a,
  object,
  beta,
  prob = 0.5,
  m0 = mean(y[a == 0]),
  m1 = mean(y[a == 1])
)
}
\arguments{
\item{X}{Input matrix, of dimension n_obs x n_vars; each row is an observation vector.}

\item{y}{Response variable to be maximized on average if every subject follows
the treatment recommended by the optimal regime.}

\item{a}{Received treatments for n_obs subjects. Must be bivariate, and labeled as \{0,1\}.}

\item{object}{Fitted "DTR.KernSmooth" or "DTR.Boots.KernSmooth" model object.}

\item{beta}{The treatment regime vector. Cannot be missing if "object" is not provided.}

\item{prob}{The propensity score for n_obs subjects, i.e., P(a=1|X). If \code{NULL},
it would be estimated by logistic regression a~X.}

\item{m0}{The estimated response values if the subjects receive treatment 0.
The default is the average response value of all subjects who receive treatment 0.}

\item{m1}{The estimated response values if the subjects receive treatment 1.
The default is the average response value of all subjects who receive treatment 1.}
}
\value{
The estimated average response value if all n_obs subjects follows the
treatment recommendations according to the fitted model or the estimated
treatment regime.
}
\description{
This function estimates the average response value of the input
data given a 'DTR.KernSmooth' / 'DTR.Boots.KernSmooth' model object or an
estimated optimal treatment regime vector, with doubly robust correction
}
\details{
\code{object} and \code{beta} cannot be both missing. If the input
data (X, y, a) is missing but \code{object} is provided, the function will
return the optimal value of the input object.
}
\examples{
n <- 500; p <- 3
beta <- c(0.2,1,-0.5,-0.8)*0.7
beta1 <- c(1,-0.5,-0.5,0.5)

set.seed(12345)
X <- matrix(rnorm(n*p),n)
a <- rbinom(n,1,0.7)
mean1 <- exp(cbind(1,X) \%*\% beta1)
mean2 <- 8/(1 + exp(-cbind(1,X) \%*\% beta)) - 4
y <- mean1 + a * mean2 + rnorm(n)

smooth_model <- DTR.KernSmooth(X, y, a, prob = 0.3 + 0.4*a)
boots_smooth_model <- DTR.Boots.KernSmooth(X, y, a, prob = 0.3 + 0.4*a, B = 100)

newn <- 1e4
newX <- matrix(rnorm(newn*p),newn)
newa <- rbinom(newn,1,0.5)
newmean1 <- exp(cbind(1,newX) \%*\% beta1)
newmean2 <- 8/(1 + exp(-cbind(1,newX) \%*\% beta)) - 4
newy <- newmean1 + newa * newmean2 + rnorm(newn)

obj_value(newX, newy, newa, smooth_model)
obj_value(newX, newy, newa, boots_smooth_model)
obj_value(newX, newy, newa, beta = smooth_model$beta_smooth)

}
\references{
Wu, Y. and Wang, L. (2021),
\emph{Resampling-based Confidence Intervals for Model-free Robust Inference
on Optimal Treatment Regimes, Biometrics, 77: 465– 476}, \doi{10.1111/biom.13337}.
}
\seealso{
\code{\link{DTR.KernSmooth}}, \code{\link{DTR.Boots.KernSmooth}}
}
\author{
Yunan Wu and Lan Wang\cr Maintainer:
Yunan Wu <yunan.wu@utdallas.edu>
}
