# Density Ratio class functions:
# ==============================


# Calculate Class Boundaries of Marginal Distributions (for posteriors):
# ======================================================================

#' Calculate marginal class bounding functions for a special Density Ratio
#' Class for which the lower and upper bounding functions are proportional.
#' 
#' This function is more efficient than 'DRclass_lu_Pdf' as it does not
#' need the evaluation of the bounding functions, l and u. 
#' It is thus recommended to use this function if l and u are proportional.
#'
#' @param sample_u Sample from a distribution proportional to the upper bound
#'                 of the class, often from the posterior of the upper bound
#'                 of the prior in Bayesian inference.
#'                 Columns represent variables, rows go across the sample.
#' @param k        Factor of proportionality between upper (u) and lower (l)
#'                 bound: u = k * l
#' @param nout     Number of equally spaced output intervals for the marginal
#'                 densities.
#' @param ...      Further arguments passed to the function 'density'
#'
#' @returns        Three dimensional array with the following dimensions:
#'                 1: variable corresponding to column of the sample
#'                 2: equidistant spacing of that variable
#'                 3: three columns for
#'                    variable values,
#'                    upper normalized density of the marginal class,
#'                    lower non-normalized density of the marginal class
#'                    
#' @examples
#' # example of the application of DRclass functions:
#' # ------------------------------------------------
#' 
#' # parameter values
#' k        <- 10
#' sd       <- 0.5
#' sampsize <- 10000
#' 
#' # upper and lower class boundaries:
#' u <- function(x) { return(    dnorm(x,0,sd)) }
#' l <- function(x) { return(1/k*dnorm(x,0,sd)) }
#' 
#' # generate sample:
#' sample_u <- cbind(rnorm(sampsize,0,sd),rnorm(sampsize,0,sd))  # example of 2d sample
#' 
#' # get class boundaries (back from sample):
#' pdf1   <- DRclass_k_Pdf(sample_u,k=k,adjust=2)       # faster for l proportional to u
#' pdf2   <- DRclass_lu_Pdf(sample_u,l=l,u=u,adjust=2)  # l and u could have different shapes
#' 
#' # get cdf bounds:
#' cdf1   <- DRclass_k_Cdf(sample_u,k=k)
#' cdf2   <- DRclass_lu_Cdf(sample_u,l=l,u=u)
#' 
#' # get quantile bounds:
#' quant1 <- DRclass_k_Quantile(sample_u,k=k,probs=c(0.025,0.5,0.975))
#' quant2 <- DRclass_lu_Quantile(sample_u,l=l,u=u,probs=c(0.025,0.5,0.975))
#' 
#' # plot selected features of the first component of the sample:
#' oldpar <- par(no.readonly=TRUE)
#' par(mar=c(5, 4, 1, 4) + 0.1)  # c(bottom, left, top, right)
#' plot(pdf1[1,,c("x","u")],type="l",xaxs="i",yaxs="i",xlim=c(-2,2),xlab="x",ylab="pdf")
#' lines(pdf2[1,,c("x","l")])
#' par(new=TRUE)
#' plot(cdf1[1,,c("x","F_upper")],type="l",xaxs="i",yaxs="i",axes=FALSE,
#'      xlim=c(-2,2),ylim=c(0,1),ylab="",lty="dashed")
#' axis(4); mtext("cdf",4,2)
#' lines(cdf2[1,,c("x","F_lower")],lty="dashed")
#' abline(v=quant1["quant_lower_0.5",1],lty="dotted")
#' abline(v=quant1["quant_upper_0.5",1],lty="dotted")
#' abline(v=quant1["quant_lower_0.025",1],lty="dotdash")
#' abline(v=quant1["quant_upper_0.975",1],lty="dotdash")
#' par(oldpar)

DRclass_k_Pdf <- function(sample_u,k=1,nout=512,...) {

  # reformat as matrix if needed:

  if ( is.vector(sample_u) ) sample_u <- matrix(sample_u,ncol=1)

  # calculate pdfs:

  pdf <- array(NA,dim=c(ncol(sample_u),nout,3))
  dimnames(pdf) <- list(colnames(sample_u),rep("",nout),c("x","u","l"))
  for ( i in 1:ncol(sample_u) ) {
    res <- density(sample_u[,i],n=nout,...)
    pdf[i,,1] <- res$x
    pdf[i,,2] <- res$y
    pdf[i,,3] <- pdf[i,,2]/k
  }

  return(pdf)
}

#' Calculate marginal class bounding functions for the general case of a
#' Density Ratio Class.
#' 
#' See the function 'DRclass_k_Pdf' for the case in which the upper and lower
#' bounding functions of the class are proportional.
#'
#' @param sample_u Sample from a distribution proportional to the upper bound
#'                 of the class, often from the posterior of the upper bound
#'                 of the prior in Bayesian inference.
#'                 Columns represent variables, rows go across the sample.
#' @param l        Either a function to evaluate the lower bound of the Density
#'                 Ratio Class or a vector of values of this function evaluated
#'                 for the rows of 'sample_u'.
#'                 Note that in the context of Bayesian inference the upper
#'                 bound of the prior can be provided as only the ratio of
#'                 l/u is needed and the likelihood cancels in this fraction.
#'                 This saves computation time as the prior is usually
#'                 computationally much cheaper to evaluate than the likelihood.
#' @param u        Either a function to evaluate the upper bound of the Density
#'                 Ratio Class or a vector of values of this function evaluated
#'                 for the rows of 'sample_u'.
#'                 Note that in the context of Bayesian inference the lower
#'                 bound of the prior can be provided as only the ratio of
#'                 l/u is needed and the likelihood cancels in this fraction.
#'                 This saves computation time as the prior is usually
#'                 computationally much cheaper to evaluate than the likelihood.
#' @param nout     Number of equally spaced output intervals for the marginal
#'                 densities.
#' @param ...      Further arguments passed to the function 'density'
#'
#' @returns        Three dimensional array with the following dimensions:
#'                 1: variable corresponding to column of the sample
#'                 2: equidistant spacing of that variable
#'                 3: three columns for
#'                    variable values,
#'                    upper normalized density of the marginal class,
#'                    lower non-normalized density of the marginal class
#'                    
#' @examples
#' # example of the application of DRclass functions:
#' # ------------------------------------------------
#' 
#' # parameter values
#' k        <- 10
#' sd       <- 0.5
#' sampsize <- 10000
#' 
#' # upper and lower class boundaries:
#' u <- function(x) { return(    dnorm(x,0,sd)) }
#' l <- function(x) { return(1/k*dnorm(x,0,sd)) }
#' 
#' # generate sample:
#' sample_u <- cbind(rnorm(sampsize,0,sd),rnorm(sampsize,0,sd))  # example of 2d sample
#' 
#' # get class boundaries (back from sample):
#' pdf1   <- DRclass_k_Pdf(sample_u,k=k,adjust=2)       # faster for l proportional to u
#' pdf2   <- DRclass_lu_Pdf(sample_u,l=l,u=u,adjust=2)  # l and u could have different shapes
#' 
#' # get cdf bounds:
#' cdf1   <- DRclass_k_Cdf(sample_u,k=k)
#' cdf2   <- DRclass_lu_Cdf(sample_u,l=l,u=u)
#' 
#' # get quantile bounds:
#' quant1 <- DRclass_k_Quantile(sample_u,k=k,probs=c(0.025,0.5,0.975))
#' quant2 <- DRclass_lu_Quantile(sample_u,l=l,u=u,probs=c(0.025,0.5,0.975))
#' 
#' # plot selected features of the first component of the sample:
#' oldpar <- par(no.readonly=TRUE)
#' par(mar=c(5, 4, 1, 4) + 0.1)  # c(bottom, left, top, right)
#' plot(pdf1[1,,c("x","u")],type="l",xaxs="i",yaxs="i",xlim=c(-2,2),xlab="x",ylab="pdf")
#' lines(pdf2[1,,c("x","l")])
#' par(new=TRUE)
#' plot(cdf1[1,,c("x","F_upper")],type="l",xaxs="i",yaxs="i",axes=FALSE,
#'      xlim=c(-2,2),ylim=c(0,1),ylab="",lty="dashed")
#' axis(4); mtext("cdf",4,2)
#' lines(cdf2[1,,c("x","F_lower")],lty="dashed")
#' abline(v=quant1["quant_lower_0.5",1],lty="dotted")
#' abline(v=quant1["quant_upper_0.5",1],lty="dotted")
#' abline(v=quant1["quant_lower_0.025",1],lty="dotdash")
#' abline(v=quant1["quant_upper_0.975",1],lty="dotdash")
#' par(oldpar)

DRclass_lu_Pdf <- function(sample_u,l,u,nout=512,...) {

  # reformat as matrix if needed:

  if ( is.vector(sample_u) ) sample_u <- matrix(sample_u,ncol=1)

  # calculate sample values, l_s and u_s, by applying l and u across rows:

  if ( is.function(l) ) l_s <- apply(sample_u,1,l) else l_s <- l
  if ( is.function(u) ) u_s <- apply(sample_u,1,u) else u_s <- u
  
  # calculate pdfs:

  pdf <- array(NA,dim=c(ncol(sample_u),nout,3))
  dimnames(pdf) <- list(colnames(sample_u),rep("",nout),c("x","u","l"))
  for ( i in 1:ncol(sample_u) ) {
    ind <- order(sample_u[,i])
    res <- density(sample_u[,i],n=nout,...)
    pdf[i,,1] <- res$x
    pdf[i,,2] <- res$y
    pdf[i,,3] <- pdf[i,,2]*approx(sample_u[ind,i],(l_s/u_s)[ind],xout=res$x,rule=2)$y
  }

  return(pdf)
}


# Calculate Bounding Cdfs:
# ========================

#' Calculate lower and upper bounds of the cumulative density function for the
#' marginals of a Density Ratio Class for which the lower and upper bounding
#' functions are proportional.
#' 
#' The cumulative density functions for the marginals of a distributions
#' proportional to the lower and upper bounding functions are also provided.
#' 
#' This function is more efficient than 'DRclass_lu_Pdf' as it does not
#' need the evaluation of the bounding functions, l and u. 
#' It is thus recommended to use this function if l and u are proportional.
#'
#' @param sample_u Sample from a distribution proportional to the upper bound
#'                 of the class, often from the posterior of the upper bound
#'                 of the prior in Bayesian inference.
#'                 Columns represent variables, rows go across the sample.
#' @param k        Factor of proportionality between upper (u) and lower (l)
#'                 bound: u = k * l
#' @param nout     Number of equally spaced output intervals for the marginal
#'                 densities.
#'
#' @returns        Three dimensional array with the following dimensions:
#'                 1: variable corresponding to column of the sample
#'                 2: equidistant spacing of that variable
#'                 3: five columns for
#'                    variable values,
#'                    lower bound of the cdf of the marginal of the class,
#'                    cdf of the marginal corresponding to the lower class boundary,
#'                    cdf of the marginal corresponding to the upper class boundary,
#'                    upper bound of the cdf of the marginal of the class
#'                    
#' @examples
#' # example of the application of DRclass functions:
#' # ------------------------------------------------
#' 
#' # parameter values
#' k        <- 10
#' sd       <- 0.5
#' sampsize <- 10000
#' 
#' # upper and lower class boundaries:
#' u <- function(x) { return(    dnorm(x,0,sd)) }
#' l <- function(x) { return(1/k*dnorm(x,0,sd)) }
#' 
#' # generate sample:
#' sample_u <- cbind(rnorm(sampsize,0,sd),rnorm(sampsize,0,sd))  # example of 2d sample
#' 
#' # get class boundaries (back from sample):
#' pdf1   <- DRclass_k_Pdf(sample_u,k=k,adjust=2)       # faster for l proportional to u
#' pdf2   <- DRclass_lu_Pdf(sample_u,l=l,u=u,adjust=2)  # l and u could have different shapes
#' 
#' # get cdf bounds:
#' cdf1   <- DRclass_k_Cdf(sample_u,k=k)
#' cdf2   <- DRclass_lu_Cdf(sample_u,l=l,u=u)
#' 
#' # get quantile bounds:
#' quant1 <- DRclass_k_Quantile(sample_u,k=k,probs=c(0.025,0.5,0.975))
#' quant2 <- DRclass_lu_Quantile(sample_u,l=l,u=u,probs=c(0.025,0.5,0.975))
#' 
#' # plot selected features of the first component of the sample:
#' oldpar <- par(no.readonly=TRUE)
#' par(mar=c(5, 4, 1, 4) + 0.1)  # c(bottom, left, top, right)
#' plot(pdf1[1,,c("x","u")],type="l",xaxs="i",yaxs="i",xlim=c(-2,2),xlab="x",ylab="pdf")
#' lines(pdf2[1,,c("x","l")])
#' par(new=TRUE)
#' plot(cdf1[1,,c("x","F_upper")],type="l",xaxs="i",yaxs="i",axes=FALSE,
#'      xlim=c(-2,2),ylim=c(0,1),ylab="",lty="dashed")
#' axis(4); mtext("cdf",4,2)
#' lines(cdf2[1,,c("x","F_lower")],lty="dashed")
#' abline(v=quant1["quant_lower_0.5",1],lty="dotted")
#' abline(v=quant1["quant_upper_0.5",1],lty="dotted")
#' abline(v=quant1["quant_lower_0.025",1],lty="dotdash")
#' abline(v=quant1["quant_upper_0.975",1],lty="dotdash")
#' par(oldpar)

DRclass_k_Cdf <- function(sample_u,k=1,nout=512) {

  # functions to calculate upper and lower cumulative distribution functions in 1d:

  F_upper <- function(x,s,k=1) return(sum(s<=x)/(sum(s<=x)+sum(s>x)/k))
  F_lower <- function(x,s,k=1) return(sum(s<=x)/(sum(s<=x)+sum(s>x)*k))

  # reformat as matrix if needed:

  if ( is.vector(sample_u) ) sample_u <- matrix(sample_u,ncol=1)

  # calculate cdfs (by applying the 1d function from above across sample columns):

  cdf <- array(NA,dim=c(ncol(sample_u),nout,5))
  dimnames(cdf) <- list(colnames(sample_u),rep("",nout),c("x","F_lower","F_l","F_u","F_upper"))
  for ( i in 1:ncol(sample_u) ) {
    xmin <- min(sample_u[,i]); xmax <- max(sample_u[,i])
    xmin <- xmin - (xmax-xmin)/nout; xmax <- xmax + (xmax-xmin)/nout
    x <- seq(xmin,xmax,length=nout)
    cdf[i,,1] <- x
    cdf[i,,2] <- sapply(x,F_lower,s=sample_u[,i],k=k)
    cdf[i,,3] <- sapply(x,F_upper,s=sample_u[,i],k=1)
    cdf[i,,4] <- cdf[i,,3]  # for consistency with case of different l and u
    cdf[i,,5] <- sapply(x,F_upper,s=sample_u[,i],k=k)
  }

  return(cdf)
}

#' Calculate lower and upper bounds of the cumulative density function for the
#' marginals of a general Density Ratio Class.
#' The cumulative density functions for the marginals of a distributions
#' proportional to the lower and upper bounding functions are also provided.
#' See the function 'DRclass_k_Cdf' for the case in which the upper and lower
#' bounding functions of the class are proportional.
#'
#' @param sample_u Sample from a distribution proportional to the upper bound
#'                 of the class, often from the posterior of the upper bound
#'                 of the prior in Bayesian inference.
#'                 Columns represent variables, rows go across the sample.
#' @param l        Either a function to evaluate the lower bound of the Density
#'                 Ratio Class or a vector of values of this function evaluated
#'                 for the rows of 'sample_u'.
#'                 Note that in the context of Bayesian inference the upper
#'                 bound of the prior can be provided as only the ratio of
#'                 l/u is needed and the likelihood cancels in this fraction.
#'                 This saves computation time as the prior is usually
#'                 computationally much cheaper to evaluate than the likelihood.
#' @param u        Either a function to evaluate the upper bound of the Density
#'                 Ratio Class or a vector of values of this function evaluated
#'                 for the rows of 'sample_u'.
#'                 Note that in the context of Bayesian inference the lower
#'                 bound of the prior can be provided as only the ratio of
#'                 l/u is needed and the likelihood cancels in this fraction.
#'                 This saves computation time as the prior is usually
#'                 computationally much cheaper to evaluate than the likelihood.
#' @param nout     Number of equally spaced output intervals for the marginal
#'                 densities.
#'
#' @returns        Three dimensional array with the following dimensions:
#'                 1: variable corresponding to column of the sample
#'                 2: equidistant spacing of that variable
#'                 3: five columns for
#'                    variable values,
#'                    lower bound of the cdf of the marginal of the class,
#'                    cdf of the marginal corresponding to the lower class boundary,
#'                    cdf of the marginal corresponding to the upper class boundary,
#'                    upper bound of the cdf of the marginal of the class
#'                    
#' @examples
#' # example of the application of DRclass functions:
#' # ------------------------------------------------
#' 
#' # parameter values
#' k        <- 10
#' sd       <- 0.5
#' sampsize <- 10000
#' 
#' # upper and lower class boundaries:
#' u <- function(x) { return(    dnorm(x,0,sd)) }
#' l <- function(x) { return(1/k*dnorm(x,0,sd)) }
#' 
#' # generate sample:
#' sample_u <- cbind(rnorm(sampsize,0,sd),rnorm(sampsize,0,sd))  # example of 2d sample
#' 
#' # get class boundaries (back from sample):
#' pdf1   <- DRclass_k_Pdf(sample_u,k=k,adjust=2)       # faster for l proportional to u
#' pdf2   <- DRclass_lu_Pdf(sample_u,l=l,u=u,adjust=2)  # l and u could have different shapes
#' 
#' # get cdf bounds:
#' cdf1   <- DRclass_k_Cdf(sample_u,k=k)
#' cdf2   <- DRclass_lu_Cdf(sample_u,l=l,u=u)
#' 
#' # get quantile bounds:
#' quant1 <- DRclass_k_Quantile(sample_u,k=k,probs=c(0.025,0.5,0.975))
#' quant2 <- DRclass_lu_Quantile(sample_u,l=l,u=u,probs=c(0.025,0.5,0.975))
#' 
#' # plot selected features of the first component of the sample:
#' oldpar <- par(no.readonly=TRUE)
#' par(mar=c(5, 4, 1, 4) + 0.1)  # c(bottom, left, top, right)
#' plot(pdf1[1,,c("x","u")],type="l",xaxs="i",yaxs="i",xlim=c(-2,2),xlab="x",ylab="pdf")
#' lines(pdf2[1,,c("x","l")])
#' par(new=TRUE)
#' plot(cdf1[1,,c("x","F_upper")],type="l",xaxs="i",yaxs="i",axes=FALSE,
#'      xlim=c(-2,2),ylim=c(0,1),ylab="",lty="dashed")
#' axis(4); mtext("cdf",4,2)
#' lines(cdf2[1,,c("x","F_lower")],lty="dashed")
#' abline(v=quant1["quant_lower_0.5",1],lty="dotted")
#' abline(v=quant1["quant_upper_0.5",1],lty="dotted")
#' abline(v=quant1["quant_lower_0.025",1],lty="dotdash")
#' abline(v=quant1["quant_upper_0.975",1],lty="dotdash")
#' par(oldpar)

DRclass_lu_Cdf <- function(sample_u,l,u,nout=512) {

  # functions to calculate upper and lower cumulative distribution functions in 1d:

  F_upper <- function(x,s,l,u) return(sum(s<=x)/(sum(s<=x)+sum((l/u)[s>x])))
  F_u     <- function(x,s)     return(sum(s<=x)/length(s))
  F_l     <- function(x,s,l,u) return(sum((l/u)[s<=x])/sum(l/u))
  F_lower <- function(x,s,l,u) return(sum((l/u)[s<=x])/(sum((l/u)[s<=x])+sum(s>x)))

  # reformat as matrix if needed:

  if ( is.vector(sample_u) ) sample_u <- matrix(sample_u,ncol=1)

  # calculate sample values, l_s and u_s, by applying l and u across rows:

  if ( is.function(l) ) l_s <- apply(sample_u,1,l) else l_s <- l
  if ( is.function(u) ) u_s <- apply(sample_u,1,u) else u_s <- u

  # calculate cdfs (by applying the 1d function from above across sample columns):

  cdf <- array(NA,dim=c(ncol(sample_u),nout,5))
  dimnames(cdf) <- list(colnames(sample_u),rep("",nout),c("x","F_lower","F_l","F_u","F_upper"))
  for ( i in 1:ncol(sample_u) ) {
    xmin <- min(sample_u[,i]); xmax <- max(sample_u[,i])
    xmin <- xmin - (xmax-xmin)/nout; xmax <- xmax + (xmax-xmin)/nout
    x <- seq(xmin,xmax,length=nout)
    cdf[i,,1] <- x
    cdf[i,,2] <- sapply(x,F_lower,s=sample_u[,i],l=l_s,u=u_s)
    cdf[i,,4] <- sapply(x,F_l,s=sample_u[,i],l=l_s,u=u_s)
    cdf[i,,3] <- sapply(x,F_u,s=sample_u[,i])
    cdf[i,,5] <- sapply(x,F_upper,s=sample_u[,i],l=l_s,u=u_s)
  }

  return(cdf)
}


# Calculate Bounds of Quantiles:
# ==============================

#' Calculate lower and upper bounds of quantiles of marginals of a Density
#' Ratio Class for which the lower and upper bounding functions are proportional.
#' 
#' Quantiles of the marginals of distributions proportional to the lower and
#' upper bounding functions are also provided.
#' 
#' This function is more efficient than 'DRclass_lu_Pdf' as it does not
#' need the evaluation of the bounding functions, l and u. 
#' It is thus recommended to use this function if l and u are proportional.
#'
#' @param sample_u Sample from a distribution proportional to the upper bound
#'                 of the class, often from the posterior of the upper bound
#'                 of the prior in Bayesian inference.
#'                 Columns represent variables, rows go across the sample.
#' @param k        Factor of proportionality between upper (u) and lower (l)
#'                 bound: u = k * l
#' @param probs    Vector of probabilities for which the quantile bounds
#'                 are to be provided.
#' @param tol      Tolerance in quantile value for approximating the solution
#'                 of the implicit equation for quantiles with the bisection
#'                 algorithm.
#' @param maxiter  Maximum number of iterations for approximating the solution
#'                 of the implicit equation for quantiles with the bisection
#'                 algorithm.
#'
#' @returns        Matrix of quantile bounds and quantiles (rows) for different
#'                 marginal variables (columns).
#'                    
#' @examples
#' # example of the application of DRclass functions:
#' # ------------------------------------------------
#' 
#' # parameter values
#' k        <- 10
#' sd       <- 0.5
#' sampsize <- 10000
#' 
#' # upper and lower class boundaries:
#' u <- function(x) { return(    dnorm(x,0,sd)) }
#' l <- function(x) { return(1/k*dnorm(x,0,sd)) }
#' 
#' # generate sample:
#' sample_u <- cbind(rnorm(sampsize,0,sd),rnorm(sampsize,0,sd))  # example of 2d sample
#' 
#' # get class boundaries (back from sample):
#' pdf1   <- DRclass_k_Pdf(sample_u,k=k,adjust=2)       # faster for l proportional to u
#' pdf2   <- DRclass_lu_Pdf(sample_u,l=l,u=u,adjust=2)  # l and u could have different shapes
#' 
#' # get cdf bounds:
#' cdf1   <- DRclass_k_Cdf(sample_u,k=k)
#' cdf2   <- DRclass_lu_Cdf(sample_u,l=l,u=u)
#' 
#' # get quantile bounds:
#' quant1 <- DRclass_k_Quantile(sample_u,k=k,probs=c(0.025,0.5,0.975))
#' quant2 <- DRclass_lu_Quantile(sample_u,l=l,u=u,probs=c(0.025,0.5,0.975))
#' 
#' # plot selected features of the first component of the sample:
#' oldpar <- par(no.readonly=TRUE)
#' par(mar=c(5, 4, 1, 4) + 0.1)  # c(bottom, left, top, right)
#' plot(pdf1[1,,c("x","u")],type="l",xaxs="i",yaxs="i",xlim=c(-2,2),xlab="x",ylab="pdf")
#' lines(pdf2[1,,c("x","l")])
#' par(new=TRUE)
#' plot(cdf1[1,,c("x","F_upper")],type="l",xaxs="i",yaxs="i",axes=FALSE,
#'      xlim=c(-2,2),ylim=c(0,1),ylab="",lty="dashed")
#' axis(4); mtext("cdf",4,2)
#' lines(cdf2[1,,c("x","F_lower")],lty="dashed")
#' abline(v=quant1["quant_lower_0.5",1],lty="dotted")
#' abline(v=quant1["quant_upper_0.5",1],lty="dotted")
#' abline(v=quant1["quant_lower_0.025",1],lty="dotdash")
#' abline(v=quant1["quant_upper_0.975",1],lty="dotdash")
#' par(oldpar)

DRclass_k_Quantile <- function(sample_u,k=1,probs=c(0.025,0.25,0.5,0.75,0.975),tol=0.001,maxiter=100) {

  # functions to calculate upper and lower cumulative distribution functions in 1d:

  F_upper <- function(x,s,k=1,prob=0) return(sum(s<=x)/(sum(s<=x)+sum(s>x)/k)-prob)
  F_lower <- function(x,s,k=1,prob=0) return(sum(s<=x)/(sum(s<=x)+sum(s>x)*k)-prob)

  # function to calculate quantiles in 1d:

  DRclass_Quantile <- function(s,k,probs,tol,maxiter) {
    
    # solve implicit equations for quantiles:
    
    min_s <- min(s)
    max_s <- max(s)
    res <- rep(NA,4*length(probs))
    for ( i in 1:length(probs) ) {
      res[(i-1)*4+1] <- bisection(F_upper,a=min_s,b=max_s,tol=tol,maxiter=maxiter,
                                  s=s,k=k,prob=probs[i])
      res[(i-1)*4+2] <- bisection(F_upper,a=min_s,b=max_s,tol=tol,maxiter=maxiter,
                                  s=s,k=1,prob=probs[i])
      res[(i-1)*4+3] <- res[(i-1)*4+2]
      res[(i-1)*4+4] <- bisection(F_lower,a=min_s,b=max_s,tol=tol,maxiter=maxiter,
                                  s=s,k=k,prob=probs[i])
    }
    names(res) <- as.vector(outer(c("quant_lower_","quant_u_","quant_l_","quant_upper_"),probs,paste0))

    return(res)
  }

  # reformat as matrix if needed:

  if ( is.vector(sample_u) ) sample_u <- matrix(sample_u,ncol=1)

  # calculate quantiles (by applying the 1d function from above across sample columns):

  q <- apply(sample_u,2,DRclass_Quantile,k=k,probs=probs,tol=tol,maxiter=maxiter)
  
  # return results:

  return(q)
}

#' Calculate lower and upper bounds of quantiles of marginals of a Density
#' Ratio Class.
#' Quantiles of the marginals of distributions proportional to the lower and
#' upper bounding functions are also provided.
#' See the function 'DRclass_k_Quantile' for the case in which the upper and
#' lower bounding functions of the class are proportional.
#'
#' @param sample_u Sample from a distribution proportional to the upper bound
#'                 of the class, often from the posterior of the upper bound
#'                 of the prior in Bayesian inference.
#'                 Columns represent variables, rows go across the sample.
#' @param l        Either a function to evaluate the lower bound of the Density
#'                 Ratio Class or a vector of values of this function evaluated
#'                 for the rows of 'sample_u'.
#'                 Note that in the context of Bayesian inference the upper
#'                 bound of the prior can be provided as only the ratio of
#'                 l/u is needed and the likelihood cancels in this fraction.
#'                 This saves computation time as the prior is usually
#'                 computationally much cheaper to evaluate than the likelihood.
#' @param u        Either a function to evaluate the upper bound of the Density
#'                 Ratio Class or a vector of values of this function evaluated
#'                 for the rows of 'sample_u'.
#'                 Note that in the context of Bayesian inference the lower
#'                 bound of the prior can be provided as only the ratio of
#'                 l/u is needed and the likelihood cancels in this fraction.
#'                 This saves computation time as the prior is usually
#'                 computationally much cheaper to evaluate than the likelihood.
#' @param probs    Vector of probabilities for which the quantile bounds
#'                 are to be provided.
#' @param tol      Tolerance in quantile value for approximating the solution
#'                 of the implicit equation for quantiles with the bisection
#'                 algorithm.
#' @param maxiter  Maximum number of iterations for approximating the solution
#'                 of the implicit equation for quantiles with the bisection
#'                 algorithm.
#'
#' @returns        Matrix of quantile bounds and quantiles (rows) for different
#'                 marginal variables (columns).
#'                    
#' @examples
#' # example of the application of DRclass functions:
#' # ------------------------------------------------
#' 
#' # parameter values
#' k        <- 10
#' sd       <- 0.5
#' sampsize <- 10000
#' 
#' # upper and lower class boundaries:
#' u <- function(x) { return(    dnorm(x,0,sd)) }
#' l <- function(x) { return(1/k*dnorm(x,0,sd)) }
#' 
#' # generate sample:
#' sample_u <- cbind(rnorm(sampsize,0,sd),rnorm(sampsize,0,sd))  # example of 2d sample
#' 
#' # get class boundaries (back from sample):
#' pdf1   <- DRclass_k_Pdf(sample_u,k=k,adjust=2)       # faster for l proportional to u
#' pdf2   <- DRclass_lu_Pdf(sample_u,l=l,u=u,adjust=2)  # l and u could have different shapes
#' 
#' # get cdf bounds:
#' cdf1   <- DRclass_k_Cdf(sample_u,k=k)
#' cdf2   <- DRclass_lu_Cdf(sample_u,l=l,u=u)
#' 
#' # get quantile bounds:
#' quant1 <- DRclass_k_Quantile(sample_u,k=k,probs=c(0.025,0.5,0.975))
#' quant2 <- DRclass_lu_Quantile(sample_u,l=l,u=u,probs=c(0.025,0.5,0.975))
#' 
#' # plot selected features of the first component of the sample:
#' oldpar <- par(no.readonly=TRUE)
#' par(mar=c(5, 4, 1, 4) + 0.1)  # c(bottom, left, top, right)
#' plot(pdf1[1,,c("x","u")],type="l",xaxs="i",yaxs="i",xlim=c(-2,2),xlab="x",ylab="pdf")
#' lines(pdf2[1,,c("x","l")])
#' par(new=TRUE)
#' plot(cdf1[1,,c("x","F_upper")],type="l",xaxs="i",yaxs="i",axes=FALSE,
#'      xlim=c(-2,2),ylim=c(0,1),ylab="",lty="dashed")
#' axis(4); mtext("cdf",4,2)
#' lines(cdf2[1,,c("x","F_lower")],lty="dashed")
#' abline(v=quant1["quant_lower_0.5",1],lty="dotted")
#' abline(v=quant1["quant_upper_0.5",1],lty="dotted")
#' abline(v=quant1["quant_lower_0.025",1],lty="dotdash")
#' abline(v=quant1["quant_upper_0.975",1],lty="dotdash")
#' par(oldpar)

DRclass_lu_Quantile <- function(sample_u,l,u,probs=c(0.025,0.25,0.5,0.75,0.975),tol=0.001,maxiter=100) {

  # functions to calculate upper and lower cumulative distribution functions in 1d:

  F_upper <- function(x,s,l,u,prob=0) return(sum(s<=x)/(sum(s<=x)+sum((l/u)[s>x]))-prob)
  F_u     <- function(x,s,prob=0)     return(sum(s<=x)/length(s)-prob)
  F_l     <- function(x,s,l,u,prob=0) return(sum((l/u)[s<=x])/sum(l/u)-prob)
  F_lower <- function(x,s,l,u,prob=0) return(sum((l/u)[s<=x])/(sum((l/u)[s<=x])+sum(s>x))-prob)

  # function to calculate quantiles in 1d:

  DRclass_Quantile <- function(s,l,u,probs,tol,maxiter) {

    # solve implicit equations for quantiles:
    
    min_s <- min(s)
    max_s <- max(s)
    res <- rep(NA,4*length(probs))
    for ( i in 1:length(probs) ) {
      res[(i-1)*4+1] <- bisection(F_upper,a=min_s,b=max_s,tol=tol,maxiter=maxiter,
                                  s=s,l=l,u=u,prob=probs[i])
      res[(i-1)*4+2] <- bisection(F_u,a=min_s,b=max_s,tol=tol,maxiter=maxiter,
                                  s=s,prob=probs[i])
      res[(i-1)*4+3] <- bisection(F_l,a=min_s,b=max_s,tol=tol,maxiter=maxiter,
                                  s=s,l=l,u=u,prob=probs[i])
      res[(i-1)*4+4] <- bisection(F_lower,a=min_s,b=max_s,tol=tol,maxiter=maxiter,
                                  s=s,l=l,u=u,prob=probs[i])
    }
    names(res) <- as.vector(outer(c("quant_lower_","quant_u_","quant_l_","quant_upper_"),probs,paste0))
    
    return(res)

  }

  # reformat as matrix if needed:

  if ( is.vector(sample_u) ) sample_u <- matrix(sample_u,ncol=1)

  # calculate sample values, l_s and u_s, by applying l and u across rows:

  if ( is.function(l) ) l_s <- apply(sample_u,1,l) else l_s <- l
  if ( is.function(u) ) u_s <- apply(sample_u,1,u) else u_s <- u

  # calculate quantiles:

  q <- apply(sample_u,2,DRclass_Quantile,l=l_s,u=u_s,probs=probs,tol=tol,maxiter=maxiter)

  # return results:

  return(q)
}




