% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dist_match_trans_learn.R
\name{DMTL}
\alias{DMTL}
\title{Distribution Mapping based Transfer Learning}
\usage{
DMTL(
  target_set,
  source_set,
  use_density = FALSE,
  pred_model = "RF",
  model_optimize = FALSE,
  sample_size = 1000,
  random_seed = NULL,
  all_pred = FALSE,
  get_verbose = FALSE,
  allow_parallel = FALSE
)
}
\arguments{
\item{target_set}{List containing the target datasets. A named list with
components \code{X} (predictors) and \code{y} (response). The predictions are performed
to estimate the response values corresponding to \code{X} while \code{y} is only used
to estimate the response distribution parameters.}

\item{source_set}{List containing the source datasets. A named list with
components \code{X} (predictors) and \code{y} (response). These two sets must be matched
and used in both distribution estimation and predictive modeling.}

\item{use_density}{Flag for using kernel density as distribution estimate
instead of histogram counts. Defaults to \code{FALSE}.}

\item{pred_model}{String indicating the underlying predictive model. The
currently available options are -
\itemize{
\item \code{RF} for random forest regression. If \code{model_optimize = FALSE}, builds a
model with \code{n_tree = 200} and \code{m_try = 0.4}.
\item \code{SVM} for support vector regression. If \code{model_optimize = FALSE}, builds a
model with \code{kernel = "poly"}, \code{C = 2}, and \code{degree = 3}.
\item \code{EN} for elastic net regression. If \code{model_optimize = FALSE}, builds a
model with \code{alpha = 0.8} and \code{lambda} generated from a 5-fold cross
validation.
}}

\item{model_optimize}{Flag for model parameter tuning. If \code{TRUE}, performs a
grid search to optimize parameters and train with the resulting model.
If \code{FALSE}, uses a set of predefined parameters. Defaults to \code{FALSE}.}

\item{sample_size}{Sample size for estimating distributions of target and
source datasets. Defaults to \code{1e3}.}

\item{random_seed}{Seed for random number generator (for reproducible
outcomes). Defaults to \code{NULL}.}

\item{all_pred}{Flag for returning the prediction values in the source space.
If \code{TRUE}, the function returns a named list with two components- \code{target}
and \code{source} (predictions in the target space and source space,
respectively). Defaults to \code{FALSE}.}

\item{get_verbose}{Flag for displaying the progress when optimizing the
predictive model \emph{i.e.}, \code{model_optimize = TRUE}. Defaults to \code{FALSE}.}

\item{allow_parallel}{Flag for allowing parallel processing when performing
grid search \emph{i.e.}, \code{model_optimimze = TRUE}. Defaults to \code{FALSE}.}
}
\value{
If \code{all_pred = FALSE}, a vector containing the final prediction values.

If \code{all_pred = TRUE}, a named list with two components \code{target} and \code{source}
\emph{i.e.}, predictions in the original target space and in source space,
respectively.
}
\description{
This function performs distribution mapping based transfer learning (DMTL)
regression for given target (primary) and source (secondary) datasets. The
data available in the source domain are used to design an appropriate
predictive model. The target features with unknown response values are
transferred to the source domain \emph{via} distribution matching and then the
corresponding response values in the source domain are predicted using the
aforementioned predictive model. The response values are then transferred to
the original target space by applying distribution matching again. Hence,
this function needs an \strong{unmatched} pair of target datasets (features and
response values) and a \strong{matched} pair of source datasets.
}
\note{
\itemize{
\item The datasets in \code{target_set} (\emph{i.e.}, \code{X} and \code{y}) do not need to be
matched (\emph{i.e.}, have the same number of rows) since the response values are
used only to estimate distribution for mapping while the feature values are
used for both mapping and final prediction. In contrast, the datasets in
\code{source_set} (\emph{i.e.}, \code{X} and \code{y}) must have matched samples.
\item It is recommended to normalize the two response values (\code{y}) so that
they will be in the same range. If normalization is not performed, \code{DMTL()}
uses the range of target \code{y} values as the prediction range.
}
}
\examples{
set.seed(8644)

## Generate two dataset with different underlying distributions...
x1 <- matrix(rnorm(3000, 0.3, 0.6), ncol = 3)
dimnames(x1) <- list(paste0("sample", 1:1000), paste0("f", 1:3))
y1 <- 0.3*x1[, 1] + 0.1*x1[, 2] - x1[, 3] + rnorm(1000, 0, 0.05)
x2 <- matrix(rnorm(3000, 0, 0.5), ncol = 3)
dimnames(x2) <- list(paste0("sample", 1:1000), paste0("f", 1:3))
y2 <- -0.2*x2[, 1] + 0.3*x2[, 2] - x2[, 3] + rnorm(1000, 0, 0.05)

## Model datasets using DMTL & compare with a baseline model...
library(DMTL)

target <- list(X = x1, y = y1)
source <- list(X = x2, y = y2)
y1_pred <- DMTL(target_set = target, source_set = source, pred_model = "RF")
y1_pred_bl <- RF_predict(x_train = x2, y_train = y2, x_test = x1)

print(performance(y1, y1_pred, measures = c("MSE", "PCC")))
print(performance(y1, y1_pred_bl, measures = c("MSE", "PCC")))

}
\keyword{density-matching}
\keyword{distribution-matching}
\keyword{domain-transfer}
\keyword{histogram-matching}
\keyword{transfer-learning}
