\name{f}
\alias{f}
\title{
Defining smooth terms in DDPstar formulae
}
\description{
Auxiliary function used to define smooth terms using Bayesian P-splines within \code{DDPstar} model formulae. The function does not evaluate the smooth - it exists purely to help set up a model using P-spline based smoothers.
}
\usage{
f(..., by = NULL, nseg = 5, bdeg = 3, pord = 2, atau = 1, btau = 0.005,
  prior.2D = c("anisotropic", "isotropic"))
}
\arguments{
  \item{\dots}{a list of up to two covariates to construct the smooth term.}
  \item{by}{a numeric or factor variable of the same dimension as each covariate. Following package \code{mgcv}, in the numeric case the elements multiply the smooth, evaluated at the corresponding covariate values (a `varying coefficient model'). Here, the smooth function does not exclude the intercept, so the \code{by} variable should not be added as an additional main effect. In the factor case, a replicate of the smooth is produced for each factor level. In this case, these smooths will exclude the intercept, so the factor needs to be added as a main effect as well (see details).}
  \item{nseg}{the number of segments for the (marginal) B-spline bases used to represent the smooth term. Numerical vector of length equal to the number of covariates. Atomic values are also valid, being recycled. The default value is 5.}
  \item{bdeg}{the order of the polynomial for the (marginal) B-spline bases for this term. Numerical vector of length equal to the number of covariates. Atomic values are also valid, being recycled. The default value is 3 (cubic B-splines).}
  \item{pord}{penalty order. Numerical vector of length equal to the number of covariates. Atomic values are also valid, being recycled. The default value is 2 (second-order penalty).}
  \item{atau}{a numeric value. Hyperparameter; shape parameter of the gamma prior distribution for the precisions (inverse variances) of the smooth term. The default is 1.}
  \item{btau}{a numeric value. Hyperparameter; rate parameter of the gamma prior distribution for the precisions (inverse variances) of the smooth term. The default is 0.005.}
  \item{prior.2D}{character string. Indicates whether the same precision parameter should be used for both variables in two-dimensional smooth functions (isotropy), or if each variable should have a different precision parameter (anisotropy). In the latter case, the Bayesian extension of the PS-ANOVA model by Lee et al. (2013) is used. The default is "anisotropic".}
}
\details{
The functions \code{f()} is designed to represent either one dimensional smooth functions for main effects of continuous explanatory variables or two dimensional smooth functions representing two way interactions of continuous variables using Bayesian P-splines. By default, the values of the arguments \code{nseg}, \code{pord} and \code{bdeg} are repeated to the length of the explanatory covariates. The two dimensional smooth terms are constructed using the tensor-product of marginal (one-dimensional) B-spline bases (Eilers and Marx, 2003). In this case, the Bayesian extension of the PS-ANOVA model by Lee et al. (2013) is considered when \code{prior.2D = "anisotropic"}. It is also possible to include factor-by-curve interactions between a continuous covariate (e.g., \code{x1}) and a categorical covariate (e.g., \code{x2}) by means of argument \code{by}: \code{y ~ x2 + f(x1, by = x2)}.
}
\value{
The function is interpreted in the formula of a \code{DDPstar} model and creates the right framework for fitting the smoother. List containing (among others) the following elements:

\item{vars}{names of the covariates involved in the smooth term.}
\item{nseg}{the number of segments for the (marginal) B-spline basis for each covariate (numerical vector of length equal to the number of covariates).}
\item{pord}{the penalty order (numerical vector of length equal to the number of covariates).}
\item{bdeg}{the order of the polynomial for the (marginal) B-Spline bases for this term (numerical vector of length equal to the number of covariates)}
\item{atau}{shape parameter of the gamma prior distribution}
\item{btau}{rate parameter of the gamma prior distribution}
}
\references{
Eilers, P.H.C. and Marx, B.D. (1996). Flexible smoothing with B-splines and penalties. Statistical Science, \bold{11}(2), 89-121.

Eilers, P.H.C. and Marx, B.D. (2003). Multidimensional calibration with temperature interaction using two- dimensional penalized signal regression. Chemometrics and Intelligence Laboratory Systems, \bold{66}, 159-174.

Lang, S. and Brezger, A. (2004). Bayesian P-splines. Journal of Computational and Graphical Statistics, \bold{13}(1), 183-212.

Lee, D.-J., Durban, M., and Eilers, P. (2013). Efficient two-dimensional smoothing with P-
spline ANOVA mixed models and nested bases. Computational Statistics & Data Analysis, \bold{61}, 22-37.
}
\seealso{
\code{\link{rae}}, \code{\link{DDPstar}}
}
\examples{
library(DDPstar)
data(dde)
dde$GAD <- dde$GAD/7 # GAD in weeks
\donttest{
formula.dde <- GAD ~ f(DDE, bdeg = 3, nseg = 20, pord = 2, atau = 1, btau = 0.005)

set.seed(10) # For reproducibility
fit_dde <- DDPstar(formula = formula.dde, 
  data = dde, mcmc = list(nburn = 20000, nsave = 15000, nskip = 1), 
  prior = list(a = 2, b = 0.5, aalpha = 2, balpha = 2, L = 20), 
  standardise = TRUE, compute.lpml = TRUE, compute.WAIC = TRUE, compute.DIC = TRUE)
summary(fit_dde)
}
\dontshow{
formula.dde <- GAD ~ f(DDE, bdeg = 3, nseg = 20, pord = 2, atau = 1, btau = 0.005)

set.seed(10) # For reproducibility
fit_dde <- DDPstar(formula = formula.dde, 
  data = dde, mcmc = list(nburn = 20, nsave = 15, nskip = 1), 
  prior = list(a = 2, b = 0.5, aalpha = 2, balpha = 2, L = 20), 
  standardise = TRUE, compute.lpml = TRUE, compute.WAIC = TRUE)
summary(fit_dde)
}
}