% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/xml_convert.r
\name{xml_to_df}
\alias{xml_to_df}
\title{Converting XML to data frames and vice versa}
\usage{
xml_to_df(
  file = NULL,
  text = NULL,
  xml.encoding = "",
  records.tags = NULL,
  records.xpath = NULL,
  fields = "tags",
  field.names = NULL,
  only.fields = NULL,
  exclude.fields = NULL,
  check.datatypes = TRUE,
  data.dec = ".",
  data.thds = ",",
  stringsAsFactors = FALSE,
  na = NA,
  non.exist = na,
  no.hierarchy = FALSE,
  hierarchy.field.delim = "|",
  hierarchy.value.sep = "~",
  no.return = FALSE,
  excel.filename = NULL,
  excel.sheetname = NULL,
  excel.pw = NULL,
  csv.filename = NULL,
  csv.sep = ",",
  csv.dec = ".",
  csv.encoding = "",
  ...
)
}
\arguments{
\item{file}{XML file to be converted. Instead of specifying a file, the XML
code can put in directly via the \code{text} argument}

\item{text}{XML code to be converted. Instead of providing the XML code, an
XML file can be specified with the \code{file} argument}

\item{xml.encoding}{Encoding of the XML file (optional), e.g. (\code{"UTF-8"})}

\item{records.tags}{Name (or vector of names) of the tags that represent the
data records in the XML (i.e. each record has one element with this tag
name). All elements with this tag name will be considered data records.
Instead of specifying the tag name, an XPatch expression can be used to
identify the data records (see \code{records.xpath})}

\item{records.xpath}{XPath expression that specifies the XML element to be
selected as data records; can be used instead of specifying the data record
XML tags directly with the \code{data.records} argument. If both,
\code{records.tags} and \code{records.path} are provided, only the XPath
expressions determines the tags to be selected.}

\item{fields}{A character value, either \code{"tags"} or \code{"attributes"}.
Specifies whether the fields of each data record are represented as XML tags
or as attributes. See \emph{Details} below for more on this topic. Default
is \code{"tags"}}

\item{field.names}{If the data fields are represented by XML elements/tags
(i.e. \code{fields = "tags"}) and it is not the tag name that identifies the
name of the data field but an attribute of that field tag then the name of
the attribute that provides the field name can be specified here. If
\code{NULL}, the tag names will be used as field names. See \emph{Details}
for more information.}

\item{only.fields}{Optional vector of tag or attribute names (depending on the
\code{fields} argument) of an XML record that will be included in the
resulting dataframe. \code{NULL} means all fields found in the data will end
up as columns in the dataframe.}

\item{exclude.fields}{Optional vector of fields that will be excluded from the
conversion; fields specified here will not end up as columns in the
resulting dataframe}

\item{check.datatypes}{Logical option that specifies if \code{xml_to_df()}
tries to identify the data types of the fields in the XML data. If
\code{TRUE} (default), \code{xml_to_df()} tries to identify numeric fields
and changes the class of the respective columns in the resulting dataframe
accordingly. Use the \code{data.dec} and \code{data.thds} arguments to
specify a number format different from the standard US/EN scheme. At this
point, there is no data type identification for logical and time/date values
available. If \code{check.datatypes} equals \code{FALSE} then all variables
in the resulting dataframe will be of class \code{character}}

\item{data.dec}{A decimal separator used in the identification of numeric data
when \code{check.datatypes = TRUE}. Default is dot (\code{.})}

\item{data.thds}{A thousands separator used in the identification of numeric
data when \code{check.datatypes = TRUE}. Default is comma (\code{,})}

\item{stringsAsFactors}{Logical option specifying if character values will be
converted to factors in the resulting data frame. Is only applied if
\code{check.datatypes = TRUE} which is the default}

\item{na}{Value that will be put into the resulting dataframe if the XML data
field is \emph{empty}. Default is \code{NA}. If a data record in the XML
does not have a specific field at all it is filled with the value provided
via the \code{non.exist} argument}

\item{non.exist}{Value that will be put into the resulting dataframe if a data
record in the XML data does not have a specific field at all. Default is the
value of the \code{na} (the default of which is \code{NA}). If instead a
field is present in the XML data but empty, then it will have the value of
the \code{na} argument in the resulting data frame}

\item{no.hierarchy}{If the fields in the XML data are represented by XML
elements/tags (i.e. argument \code{fields = "tags"}) and there is a
hierarchical structure below a data field element then this hierarchical
structure gets 'flattened', i.e. it will be represented by a single
character value. See \emph{Details} for an example}

\item{hierarchy.field.delim}{One or two-element character vector specifying
the tag delimiters used when 'flattening' a hierarchy in the XML data. If
only one delimiter is specified then this delimiter will be used for both,
the beginning of the tag and the end of the tag. See \emph{Details} for an
example}

\item{hierarchy.value.sep}{Character value that is used as the separator
between the tag name and the value of the tag when 'flattening' a hierarchy
in the XML data. See \emph{Details} for an example}

\item{no.return}{Logical option to prevent \code{xml_to_df()} from returning
the dataframe it creates; use this if you are only interested in saving the
dataframe as Excel or CSV.}

\item{excel.filename}{Name of an Excel file the resulting dataframe will be
exported to. If \code{NULL} (default) there will be no Excel export.}

\item{excel.sheetname}{Name of the worksheet the resulting dataframe will be
exported to when using the Excel export via the \code{excel.filename}
argument. If \code{NULL}, \code{xml_to_df()} will figure out a name,}

\item{excel.pw}{Password that is applied to the Excel workbook when the
resulting data.frame is exported via the \code{excel.filename} argument.
Default {NULL} means the workbook will not be protected with a password}

\item{csv.filename}{Name of a CSV file the resulting dataframe will be
exported to. If \code{NULL} there will be no CSV export.}

\item{csv.sep}{Separator used to separate fields in the CSV file when
exporting the resulting dataframe via the \code{csv.filename} argument.
Default is comma (\code{","})}

\item{csv.dec}{Decimal separator used when exporting the resulting dataframe
via the \code{csv.filename} argument, Default is dot (\code{.})}

\item{csv.encoding}{Text encoding used when exporting the resulting dataframe
via the \code{csv.filename} argument}

\item{...}{Additional arguments passed on the \code{write.table()} when
exporting the resulting dataframe via the \code{csv.filename} argument,
Default is dot (\code{.})}
}
\value{
The resulting dataframe. There is no return value if the
 \code{no.return} argument is set to \code{TRUE}.
}
\description{
\code{xml_to_df()} converts XML data to a dataframe. It
  provides a lot of flexibility with its arguments but can usually be
  used with just a couple of them to achieve the desired results. See the
  examples below for simple applications.
}
\details{
This section provides some more details on how \code{xml_to_df()}
 works with different ways of representing data fields in the XML (tags
 versus attributes) and on working with nested XML field structures.\cr\cr
 \subsection{Two ways of representing the data: Tags and attributes}{
 For \code{xml_to_df()} records are always represented by tags (i.e XML
 elements). Data fields within a record, however, can be represented by
 either tags or attributes.\cr\cr In the former case the XML would like like
 this:\cr \code{ <xml>} \cr \code{....<record>} \cr
 \code{........<field1>Value 1-1</field1>} \cr \code{........<field2>Value
 1-2</field2>} \cr \code{....</record>} \cr \code{....<record>} \cr
 \code{........<field1>Value 2-1</field1>} \cr \code{........<field2>Value
 2-2</field2>} \cr \code{....</record>} \cr \code{....</xml>} \cr\cr Here,
 each data field is represented by its own tag (e.g. \code{field1}). In this
 case the \code{records.tag} argument would need to be \code{"record"} (or we
 would specify an XPath expression with \code{records.xpath} that selects
 these elements) as this is the name of the tags that carry the data records;
 the \code{fields} argument would need to be \code{"tags"} because the actual
 data fields are represented by tags nested into the record elements.\cr A
 variant of this case would be if the fields are represented by tags but the
 field names are not the tag names but are provided by some attribute of
 these tags. This is the case in the following example:\cr \code{ <xml>} \cr
 \code{....<record>} \cr \code{........<data name="field1">Value 1-1</data>}
 \cr \code{........<data name="field2">Value 1-2</data>} \cr
 \code{....</record>} \cr \code{....<record>} \cr \code{........<data
 name="field1">Value 2-1</data>} \cr \code{........<data name"field2">Value
 2-2</data>} \cr \code{....</record>} \cr \code{....</xml>} \cr\cr  Here, we
 would use the optional \code{field.names} argument to tell
 \code{xml_to_df()} with \code{field.names = "name"} that each data tag has
 an attribute called \code{"name"} that specifies the field name for this
 data field tag.\cr\cr In contrast to these cases, data fields can also be
 represented by attributes of the record:\cr \code{<xml>}\cr
 \code{....<record field1="Value 1-1" field2="Value 1-2" />}\cr
 \code{....<record field1="Value 2-1" field2="Value 2-2" />}\cr
 \code{</xml>}\cr Here would need to change the \code{fields} argument to
 \code{"attributes"}.}
 \subsection{Working with the nested field values}{
 When data fields are represented by XML elements / tag then there may nested
 structures beneath them. If the \code{no.hierarchy} argument is \code{FALSE}
 (default) this nested structure within a field is recursively analyzed and
 represented in a single character value for this field. Each nested element
 is enclosed in the delimiters provided with the \code{hierarchy.field.delim}
 argument. By default, there is only one such delimiter (and that is
 \code{"|"}) which is used mark both, the start and the end of an element in
 the resulting value. However, it is possible to specify to different symbols
 in order to distinguish start and end of an element. The
 \code{hierarchy.value.sep} argument is the symbol that separates the name of
 the argument from its value. Consider the following example:\cr\cr
 \code{<xml>}\cr \code{....<ship>}\cr \code{........<name>Excelsior<name>}\cr
 \code{........<lastcaptain>Hikaru Sulu</lastcaptain>}\cr
 \code{....</ship>}\cr \code{....<ship>}\cr \code{........One proud ship name, many
 captains}\cr \code{........<name>Enterprise<name>}\cr
 \code{........<lastcaptain>}\cr \code{............<NCC-1701-A>James Tiberius
 Kirk</NCC-1701-A>}\cr \code{............<NCC-1701-B>John
 Harriman</NCC-1701-B>}\cr \code{............<NCC-1701-C>Rachel
 Garrett</NCC-1701-C>}\cr \code{............<NCC-1701-D>Jean-Luc
 Picard</NCC-1701-D>}\cr \code{........</lastcaptain>}\cr
 \code{....</ship>}\cr \code{</xml>}\cr\cr Calling \code{xml_to_df()} with
 the default values for \code{hierarchy.field.delim} and
 \code{hierarchy.value.sep} would result in the following value of the
 \code{lastcapatin} field for the \code{Enterprise} record:\cr \code{One
 proud name, many captains|NCC-1701-A~James Tiberius Kirk||NCC-1701-B~John
 Harriman||NCC-1701-C~Rachel Garrett||NCC-1701-D~Jean-Luc Picard|}\cr\cr If
 we would use \code{hierarchy.field.delim = c("[", "]")} then we would better
 see the start and of end of each element:\cr \code{One proud name, many
 captains[NCC-1701-A~James Tiberius Kirk][NCC-1701-B~John
 Harriman][NCC-1701-C~Rachel Garrett][NCC-1701-D~Jean-Luc Picard]}
 }
}
\examples{
# Data used: World population figures from the United Nations Statistics Division

# Read in the raw XML; two versions: one with data fields as XML
# elements/tags, one with data fields as attributes
example.tags <- system.file("worldpopulation_tags.xml", package="xmlconvert")
example.attr <- system.file("worldpopulation_attr.xml", package="xmlconvert")

# Convert XML data to dataframe
worldpop.tags <- xml_to_df(text = example.tags, records.tags = c("record"),
    fields = "tags", field.names = "name")
worldpop.attr <- xml_to_df(text = example.attr, records.tags = c("record"),
    fields = "attributes")

}
\seealso{
Other xmlconvert: 
\code{\link{df_to_xml}()}
}
\concept{xmlconvert}
