% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/crm_dtps.R
\name{crm_dtps}
\alias{crm_dtps}
\title{Calculate dose-transition pathways for a CRM study}
\usage{
crm_dtps(skeleton, target, model, cohort_sizes, previous_outcomes = "",
  next_dose = NULL, user_dose_func = NULL, verbose = FALSE,
  i_am_patient = FALSE, ...)
}
\arguments{
\item{skeleton}{a vector of the prior guesses of toxicity at doses.
This should be a monotonically-increasing vector of numbers between 0 and 1.}

\item{target}{the target toxicity probability, a number between 0 and 1.
This value would normally be one of the values in \code{skeleton}, but that
is not a requirement.}

\item{model}{Character string to denote desired model. One of \code{empiric},
\code{logistic}, \code{logistic_gamma}, or \code{logistic2}.
The choice of model determines which extra parameters are required by
\code{...}. See Details.}

\item{cohort_sizes}{vector of future cohort sizes, i.e. positive integers.
E.g. To calculate paths for the the next cohort of two followed by another
cohort of three, use \code{cohort_sizes = c(2, 3)}.}

\item{previous_outcomes}{Outcomes observed hitherto in the syntax required
by \code{\link{df_parse_outcomes}}.}

\item{next_dose}{optional, integer (1-based) dose-level to be given to the
next cohort. If omitted, the dose suggested by the model is used.}

\item{user_dose_func}{optional delegate for deciding dose. A function that
takes a \code{\link{crm_fit}} as the sole argument and returns the integer
(1-based) dose-level to be given next, or NA to show that no dose should be
chosen and the trial stopped. This function gives the user the opportunity to
build in custom behaviour to tailor the dose selection decision in response
to the insights garnered by the fit model, or recommend that a trial path
be halted immediately. If omitted, the dose ordinarily chosen by the model is
used. An example is given below.}

\item{verbose}{logical, TRUE to get log messages.}

\item{i_am_patient}{logical. The number of paths to analyse grows faster than
linearly in the number of future cohorts to resolve. Fitting many models by
MCMC can take a long time. This function will not proceed unless you signify
your patience when the number of paths to reolve exceeds 100.}

\item{...}{Extra parameters passed to \code{\link{stan_crm}}.}
}
\value{
A \code{\link{list}} of \code{\link{dose_finding_path_node}} objects.
}
\description{
Calculate dose-transition pathways (DTPs, Yap et al, 2017) for a dose-finding
trial using the continual reassessment method (CRM) design. DTPs are a
glimpse into the future for an in-progress trial. They tell us what the model
would advise for all feasible future outcomes. They can be used in the design
stages to detect possible undesirable behaviour. They can be used during the
trial to aid planning and understanding.
}
\details{
Different model choices require that different parameters are provided.
See below.
}
\section{Parameter requirements of \code{empiric} model}{

\itemize{
  \item \code{beta_sd}
}
}

\section{Parameter requirements of \code{logistic} model}{

\itemize{
  \item \code{a0}
  \item \code{beta_mean}
  \item \code{beta_sd}
}
}

\section{Parameter requirements of \code{logistic_gamma} model}{

\itemize{
  \item \code{a0}
  \item \code{beta_shape}
  \item \code{beta_inverse_scale}
}
}

\section{Parameter requirements of \code{logistic2} model}{

\itemize{
  \item \code{alpha_mean}
  \item \code{alpha_sd}
  \item \code{beta_mean}
  \item \code{beta_sd}
}
}

\examples{
\dontrun{
target <- 0.25
skeleton <- c(0.05, 0.15, 0.25, 0.4, 0.6)

# Run DTPs for the first two cohorts of two for new a trial:
paths <- crm_dtps(skeleton = skeleton, target = target, model = 'empiric',
                  cohort_sizes = c(2, 2), next_dose = 3, beta_sd = 1)
length(paths)  # 13

library(tibble)
df <- as_tibble(paths)
df


# Run DTPs for the next cohort of three in a trial that has already treated
# six patients, seeing some toxicity at dose-level 3:
paths2 <- crm_dtps(skeleton = skeleton, target = target, model = 'empiric',
                   cohort_sizes = c(3), previous_outcomes = '2NNN 3TTN',
                   beta_sd = 1)
length(paths2)  # 5
as_tibble(paths2)
# We see that de-escalation to dose-level 2 should occur now, and that any
# further toxicity will result in advice for further de-escalation to
# dose-level 1.


# An example with a custom dose selection function
paths3 <- crm_dtps(skeleton = skeleton, target = target, model = 'empiric',
                   cohort_sizes = c(3, 3), previous_outcomes = '2NN 3TN',
                   next_dose = 2, beta_sd = 1,
                   user_dose_func = function(x) {
                     careful_escalation(x, tox_threshold = target + 0.1,
                                        certainty_threshold = 0.7)
                   }, seed = 123, refresh = 0)
spread_paths(as_tibble(paths3) \%>\% select(-fit, -parent_fit, -dose_index))
# Stopping is recommended when the dose selection function returns NA.
}
}
\references{
Yap C, Billingham LJ, Cheung YK, Craddock C, O’Quigley J.
Dose transition pathways: The missing link between complex dose-finding
designs and simple decision-making. Clinical Cancer Research.
2017;23(24):7440-7447. doi:10.1158/1078-0432.CCR-17-0582
}
\seealso{
\code{\link{df_parse_outcomes}},
  \code{\link{stan_crm}},
  \code{\link{crm_path_analysis}},
  \code{\link{dose_finding_path_node}}
}
\author{
Kristian Brock
}
