% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/peps2.R
\name{peps2_process}
\alias{peps2_process}
\title{Process RStan samples from a BEBOP model fit to PePS2 data}
\usage{
peps2_process(dat, fit, min_eff = 0.1, max_tox = 0.3, eff_cert = 0.7,
  tox_cert = 0.9)
}
\arguments{
\item{dat}{An instance of \code{\link{peps2_params}}, a list of PePS2
data and parameters. An example is yielded by \code{\link{peps2_get_data}}.}

\item{fit}{An instance of \code{rstan::stanmodel}, derived by fitting data to
the BEBOP in PePS2 model.
Use \code{stan::sampling(stanmodels$BebopInPeps2, data = dat)}.}

\item{min_eff}{The lower efficacy probability threshold; a number between 0
and 1.}

\item{max_tox}{The upper toxicity probability threshold; a number between 0
and 1.}

\item{eff_cert}{Certainty required to infer the treatment is acceptable with
regards to being probably efficacious; a number between 0 and 1.}

\item{tox_cert}{Certainty required to infer the treatment is acceptable with
regards to being probably tolerable; a number between 0 and 1.}
}
\value{
a list with the following items:
\itemize{
\item \code{ProbEff}, the posterior mean probability of efficacy in the 6
cohorts.
\item \code{ProbAccEff}, the posterior mean probability that the probability
of efficacy exceeds \code{min_eff}, in the 6 cohorts.
\item \code{ProbTox}, the posterior mean probability of toxicity in the 6
cohorts.
\item \code{ProbAccTox}, the posterior mean probability that the probability
of toxicity is less than \code{max_tox}, in the 6 cohorts.
\item \code{Accept}, a vector of logical values to show whether treatment
should be accepted in the 6 cohorts. Treatment is acceptable when it is
probably efficacious and probably not toxic, with respect to the described
rules.
\item \code{alpha}, the posterior mean estimate of alpha.
\item \code{beta}, the posterior mean estimate of beta.
\item \code{gamma}, the posterior mean estimate of gamma.
\item \code{zeta}, the posterior mean estimate of zeta.
\item \code{lambda}, the posterior mean estimate of lambda.
\item \code{psi}, the posterior mean estimate of psi.
}
}
\description{
Process RStan samples from a BEBOP model fit to PePS2 data. This
step lets us make inferences about whether the modelled efficacy and toxicity
probabilities suggest the treatment is acceptable in each of the cohorts
under study.
The parameters have default values to match those used in the PePS2 trial.
See the accompanying vignette for a full description.
}
\examples{
set.seed(123)
dat <- peps2_get_data(num_patients = 60,
                      prob_eff = c(0.167, 0.192, 0.5,
                                   0.091, 0.156, 0.439),
                      prob_tox = rep(0.1, 6),
                      eff_tox_or = rep(1, 6))
samp = rstan::sampling(stanmodels$BebopInPeps2, data = dat)
decision <- peps2_process(dat, samp)
decision$Accept   # Accept in cohort 2, 3, 5, 6 but not 1, 4
decision$ProbEff  # The probability of efficacy is driving that decision

}
\seealso{
\code{\link{peps2_params}}

\code{\link{peps2_get_data}}
}
