% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/round.R
\name{time_round}
\alias{time_round}
\alias{time_floor}
\alias{time_ceiling}
\title{Round, floor and ceiling for date-time objects}
\usage{
time_round(
  time,
  unit = "second",
  week_start = getOption("timechange.week_start", 1)
)

time_floor(
  time,
  unit = "seconds",
  week_start = getOption("timechange.week_start", 1)
)

time_ceiling(
  time,
  unit = "seconds",
  change_on_boundary = inherits(time, "Date"),
  week_start = getOption("timechange.week_start", 1)
)
}
\arguments{
\item{time}{a date-time vector (\code{Date}, \code{POSIXct} or \code{POSIXlt})}

\item{unit}{a character string specifying a time unit or a multiple of a unit. Valid
base periods for civil time rounding are \code{second}, \code{minute}, \code{hour}, \code{day}, \code{week},
\code{month}, \code{bimonth}, \code{quarter}, \code{season}, \code{halfyear} and \code{year}. The only unit for
absolute time rounding is \code{asecond}. Other absolute units can be achieved with
multiples of \code{asecond} ("60a", "3600a" etc). See "Details" and examples. Arbitrary
unique English abbreviations are allowed. One letter abbreviations follow
\code{strptime} formats "y", "m", "d", "M", "H", "S". Multi-unit rounding of weeks is
currently not supported.

Rounding for a unit is performed from the parent's unit origin. For example when
rounding to seconds origin is start of the minute. When rounding to days, origin is
first date of the month. See examples.

With fractional sub-unit (unit < 1) rounding with child unit is performed
instead. For example 0.5mins == 30secs, .2hours == 12min etc.

Please note that for fractions which don't match exactly to integer number of the
child units only the integer part is used for computation. For example .7days =
16.8hours will use 16 hours during the computation.}

\item{week_start}{When unit is \code{weeks}, this is the first day of the week. Defaults
to 1 (Monday).}

\item{change_on_boundary}{If NULL (the default) don't change instants on the boundary
(\code{time_ceiling(ymd_hms('2000-01-01 00:00:00'))} is \verb{2000-01-01 00:00:00}), but
round up \code{Date} objects to the next boundary (\code{time_ceiling(ymd("2000-01-01"), "month")} is \code{"2000-02-01"}). When \code{TRUE}, instants on the boundary are rounded up
to the next boundary. When \code{FALSE}, date-time on the boundary are never rounded up
(this was the default for \pkg{lubridate} prior to \code{v1.6.0}. See section \verb{Rounding Up Date Objects} below for more details.}
}
\value{
An object of the same class as the input object. When input is a \code{Date}
object and unit is smaller than \code{day} a \code{POSIXct} object is returned.
}
\description{
\pkg{timechange} provides rounding to the nearest unit or multiple of a
unit with fractional support whenever makes sense. Units can be specified flexibly
as strings. All common abbreviations are supported - secs, min, mins, 2 minutes, 3
years, 2s, 1d etc.
}
\section{Civil Time vs Absolute Time rounding}{


Rounding in civil time is done on actual clock time (ymdHMS) and is affected
by civil time irregularities like DST. One important characteristic of civil
time rounding is that floor (ceiling) does not produce civil times that are
bigger (smaller) than the rounded civil time.

Absolute time rounding (with \code{aseconds}) is done on the absolute time (number
of seconds since origin), Thus, rounding of \code{aseconds} allows for fractional
seconds and multi-units larger than 60. See examples of rounding around DST
transition where rounding in civil time does not give same result as rounding
with the corresponding \verb{X aseconds}.

Please note that absolute rounding to fractions smaller than 1ms will result
to large precision errors due to the floating point representation of the
POSIXct objects.
}

\section{Note on \code{time_round()}}{


For rounding date-times which is exactly halfway between two consecutive units,
the convention is to round up. Note that this is in line with the behavior of R's
\code{\link[base:round.POSIXt]{base::round.POSIXt()}} function but does not follow the convention of the base
\code{\link[base:Round]{base::round()}} function which "rounds to the even digit" per IEC 60559.
}

\section{Ceiling of \code{Date} objects}{


By default rounding up \code{Date} objects follows 3 steps:
\enumerate{
\item Convert to an instant representing lower bound of the Date:
\code{2000-01-01} --> \verb{2000-01-01 00:00:00}
\item Round up to the \strong{next} closest rounding unit boundary. For example,
if the rounding unit is \code{month} then next closest boundary of \code{2000-01-01}
is \verb{2000-02-01 00:00:00}.

The motivation for this is that the "partial" \code{2000-01-01} is conceptually
an interval (\verb{2000-01-01 00:00:00} -- \verb{2000-01-02 00:00:00}) and the day
hasn't started clocking yet at the exact boundary \code{00:00:00}. Thus, it
seems wrong to round up a day to its lower boundary.

The behavior on the boundary can be changed by setting
\code{change_on_boundary} to a non-\code{NULL} value.
\item If rounding unit is smaller than a day, return the instant from step 2
(\code{POSIXct}), otherwise convert to and return a \code{Date} object.
}
}

\examples{

## print fractional seconds
options(digits.secs=6)

x <- as.POSIXct("2009-08-03 12:01:59.23")
time_round(x, ".5 asec")
time_round(x, "sec")
time_round(x, "second")
time_round(x, "asecond")
time_round(x, "minute")
time_round(x, "5 mins")
time_round(x, "5M") # "M" for minute "m" for month
time_round(x, "hour")
time_round(x, "2 hours")
time_round(x, "2H")
time_round(x, "day")
time_round(x, "week")
time_round(x, "month")
time_round(x, "bimonth")
time_round(x, "quarter") == time_round(x, "3 months")
time_round(x, "halfyear")
time_round(x, "year")

x <- as.POSIXct("2009-08-03 12:01:59.23")
time_floor(x, ".1 asec")
time_floor(x, "second")
time_floor(x, "minute")
time_floor(x, "M")
time_floor(x, "hour")
time_floor(x, "day")
time_floor(x, "week")
time_floor(x, "m")
time_floor(x, "month")
time_floor(x, "bimonth")
time_floor(x, "quarter")
time_floor(x, "season")
time_floor(x, "halfyear")
time_floor(x, "year")

x <- as.POSIXct("2009-08-03 12:01:59.23")
time_ceiling(x, ".1 asec")
time_ceiling(x, "second")
time_ceiling(x, "minute")
time_ceiling(x, "5 mins")
time_ceiling(x, "hour")
time_ceiling(x, "day")
time_ceiling(x, "week")
time_ceiling(x, "month")
time_ceiling(x, "bimonth") == time_ceiling(x, "2 months")
time_ceiling(x, "quarter")
time_ceiling(x, "season")
time_ceiling(x, "halfyear")
time_ceiling(x, "year")

## behavior on the boundary
x <- as.Date("2000-01-01")
time_ceiling(x, "month")
time_ceiling(x, "month", change_on_boundary = FALSE)

## As of R 3.4.2 POSIXct printing of fractional numbers is wrong
as.POSIXct("2009-08-03 12:01:59.3", tz = "UTC") ## -> "2009-08-03 12:01:59.2 UTC"
time_ceiling(x, ".1 asec") ## -> "2009-08-03 12:01:59.2 UTC"

## Civil Time vs Absolute Time Rounding

# "2014-11-02 01:59:59.5 EDT" before 1h backroll at 2AM
x <- .POSIXct(1414907999.5, tz = "America/New_York")
x
time_ceiling(x, "hour") # "2014-11-02 02:00:00 EST"
time_ceiling(x, "minute")
time_ceiling(x, "sec")
difftime(time_ceiling(x, "s"), x)
time_ceiling(x, "1a") # "2014-11-02 01:00:00 EST"
difftime(time_ceiling(x, "a"), x)

# "2014-11-02 01:00:00.5 EST" after 1h backroll at 2AM
x <- .POSIXct(1414908000.5, tz = "America/New_York")
x
time_floor(x, "hour") # "2014-11-02 01:00:00 EST"
difftime(time_floor(x, "hour"), x)
time_floor(x, "3600a") # "2014-11-02 01:00:00 EST"  - 25m
difftime(time_floor(x, "a"), x)

## behavior on the boundary when rounding multi-units

x <- as.POSIXct("2009-08-28 22:56:59.23", tz = "UTC")
time_ceiling(x, "3.4 secs")  # "2009-08-28 22:57:03.4"
time_ceiling(x, "50.5 secs")  # "2009-08-28 22:57:50.5"
time_ceiling(x, "57 min")  # "2009-08-28 22:57:00"
time_ceiling(x, "56 min")  # "2009-08-28 23:56:00"
time_ceiling(x, "7h")  # "2009-08-29 07:00:00"
time_ceiling(x, "7d")  # "2009-08-29 00:00:00"
time_ceiling(x, "8d")  # "2009-09-09 00:00:00"
time_ceiling(x, "8m")  # "2009-09-01 00:00:00"
time_ceiling(x, "6m")  # "2010-01-01 00:00:00"
time_ceiling(x, "7m")  # "2010-08-01 00:00:00"

x <- as.POSIXct("2010-11-25 22:56:57", tz = "UTC")
time_ceiling(x, "6sec") # "2010-11-25 22:57:00"
time_ceiling(x, "60sec") # "2010-11-25 22:57:00"
time_ceiling(x, "6min") # "2010-11-25 23:00:00"
time_ceiling(x, "60min") # "2010-11-25 23:00:00"
time_ceiling(x, "4h") # "2010-11-26 00:00:00"
time_ceiling(x, "15d") # "2010-12-01 00:00:00"
time_ceiling(x, "15d") # "2010-12-01 00:00:00"
time_ceiling(x, "6m") # "2011-01-01 00:00:00"

}
\seealso{
\code{\link[base:Round]{base::round()}}
}
