% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/refit.tidylda.R
\name{refit.tidylda}
\alias{refit.tidylda}
\title{Update a Latent Dirichlet Allocation topic model}
\usage{
\method{refit}{tidylda}(
  object,
  new_data,
  iterations = NULL,
  burnin = -1,
  prior_weight = 1,
  additional_k = 0,
  optimize_alpha = FALSE,
  calc_likelihood = FALSE,
  calc_r2 = FALSE,
  return_data = FALSE,
  threads = 1,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{object}{a fitted object of class \code{tidylda}.}

\item{new_data}{A document term matrix or term co-occurrence matrix of class dgCMatrix.}

\item{iterations}{Integer number of iterations for the Gibbs sampler to run.}

\item{burnin}{Integer number of burnin iterations. If \code{burnin} is greater than -1,
the resulting "beta" and "theta" matrices are an average over all iterations
greater than \code{burnin}.}

\item{prior_weight}{Numeric, 0 or greater or \code{NA}. The weight of the 
\code{beta} as a prior from the base model. See Details, below.}

\item{additional_k}{Integer number of topics to add, defaults to 0.}

\item{optimize_alpha}{Logical. Do you want to optimize alpha every iteration?
Defaults to \code{FALSE}. See 'details' of documentation for
\code{\link[textmineR]{FitLdaModel}}for more information.}

\item{calc_likelihood}{Logical. Do you want to calculate the log likelihood every iteration?
Useful for assessing convergence. Defaults to \code{FALSE}.}

\item{calc_r2}{Logical. Do you want to calculate R-squared after the model is trained?
Defaults to \code{FALSE}. This calls \code{\link[textmineR]{CalcTopicModelR2}}.}

\item{return_data}{Logical. Do you want \code{new_data} returned as part of the model object?}

\item{threads}{Number of parallel threads, defaults to 1.}

\item{verbose}{Logical. Do you want to print a progress bar out to the console?
Defaults to \code{TRUE}.}

\item{...}{Additional arguments, currently unused}
}
\value{
Returns an S3 object of class c("tidylda").
}
\description{
Update an LDA model using collapsed Gibbs sampling.
}
\details{
\code{refit} allows you to (a) update the probabilities (i.e. weights) of
  a previously-fit model with new data or additional iterations and (b) optionally
  use \code{beta} of a previously-fit LDA topic model as the \code{eta} prior
  for the new model. This is tuned by setting \code{beta_as_prior = FALSE} or
  \code{beta_as_prior = TRUE} respectively.
  
  \code{prior_weight} tunes how strong the base model is represented in the prior.
  If \code{prior_weight = 1}, then the tokens from the base model's training data
  have the same relative weight as tokens in \code{new_data}. In other words,
  it is like just adding training data. If \code{prior_weight} is less than 1,
  then tokens in \code{new_data} are given more weight. If \code{prior_weight}
  is greater than 1, then the tokens from the base model's training data are
  given more weight.

  If \code{prior_weight} is \code{NA}, then the new \code{eta} is equal to 
  \code{eta} from the old model, with new tokens folded in. 
  (For handling of new tokens, see below.) Effectively, this just controls
  how the sampler initializes (described below), but does not give prior
  weight to the base model.

  Instead of initializing token-topic assignments in the manner for new
  models (see \code{\link[tidylda]{tidylda}}), the update initializes in 2
  steps:

  First, topic-document probabilities (i.e. \code{theta}) are obtained by a
  call to \code{\link[tidylda]{predict.tidylda}} using \code{method = "dot"}
  for the documents in \code{new_data}. Next, both \code{beta} and \code{theta} are
  passed to an internal function, \code{\link[tidylda]{initialize_topic_counts}},
  which assigns topics to tokens in a manner approximately proportional to 
  the posteriors and executes a single Gibbs iteration.

  \code{refit} handles the addition of new vocabulary by adding a flat prior
  over new tokens. Specifically, each entry in the new prior is equal to the
  10th percentile of \code{eta} from the old model. The resulting model will
  have the total vocabulary of the old model plus any new vocabulary tokens.
  In other words, after running \code{refit.tidylda} \code{ncol(beta) >= ncol(new_data)}
  where \code{beta} is from the new model and \code{new_data} is the additional data.

  You can add additional topics by setting the \code{additional_k} parameter
  to an integer greater than zero. New entries to \code{alpha} have a flat
  prior equal to the median value of \code{alpha} in the old model. (Note that
  if \code{alpha} itself is a flat prior, i.e. scalar, then the new topics have
  the same value for their prior.) New entries to \code{eta} are the average
  of all previous topics in \code{eta}.
}
\note{
Updates are, as of this writing, are almost-surely useful but their behaviors
 have not been optimized or well-studied. Caveat emptor!
}
\examples{
\donttest{
# load a document term matrix
data(nih_sample_dtm, package = "textmineR")

d1 <- nih_sample_dtm[1:50, ]

d2 <- nih_sample_dtm[51:100, ]

# fit a model
m <- tidylda(d1,
  k = 10,
  iterations = 200, burnin = 175
)

# update an existing model by adding documents using old model as prior
m2 <- refit(
  object = m,
  new_data = rbind(d1, d2),
  iterations = 200,
  burnin = 175,
  prior_weight = 1
)

# use an old model to initialize new model and not use old model as prior
m3 <- refit(
  object = m,
  new_data = d2, # new documents only
  iterations = 200,
  burnin = 175,
  prior_weight = NA
)

# add topics while updating a model by adding documents
m4 <- refit(
  object = m,
  new_data = rbind(d1, d2),
  additional_k = 3,
  iterations = 200,
  burnin = 175
)
}
}
