% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bijectors.R
\name{tfb_glow}
\alias{tfb_glow}
\title{Implements the Glow Bijector from Kingma & Dhariwal (2018).}
\usage{
tfb_glow(
  output_shape = c(32, 32, 3),
  num_glow_blocks = 3,
  num_steps_per_block = 32,
  coupling_bijector_fn = NULL,
  exit_bijector_fn = NULL,
  grab_after_block = NULL,
  use_actnorm = TRUE,
  seed = NULL,
  validate_args = FALSE,
  name = "glow"
)
}
\arguments{
\item{output_shape}{A list of integers, specifying the event shape of the
output, of the bijectors forward pass (the image).  Specified as
\verb{[H, W, C]}. Default Value: (32, 32, 3)}

\item{num_glow_blocks}{An integer, specifying how many downsampling levels to
include in the model. This must divide equally into both H and W,
otherwise the bijector would not be invertible. Default Value: 3}

\item{num_steps_per_block}{An integer specifying how many Affine Coupling and
1x1 convolution layers to include at each level of the spatial
hierarchy. Default Value: 32 (i.e. the value used in the original glow paper).}

\item{coupling_bijector_fn}{A function which takes the argument \code{input_shape}
and returns a callable neural network (e.g. a \code{keras_model_sequential()}). The
network should either return a tensor with the same event shape as
\code{input_shape} (this will employ additive coupling), a tensor with the
same height and width as \code{input_shape} but twice the number of channels
(this will employ affine coupling), or a bijector which takes in a
tensor with event shape \code{input_shape}, and returns a tensor with shape
\code{input_shape}.}

\item{exit_bijector_fn}{Similar to coupling_bijector_fn, exit_bijector_fn is
a function which takes the argument \code{input_shape} and \code{output_chan}
and returns a callable neural network. The neural network it returns
should take a tensor of shape \code{input_shape} as the input, and return
one of three options: A tensor with \code{output_chan} channels, a tensor
with \code{2 * output_chan} channels, or a bijector. Additional details can
be found in the documentation for ExitBijector.}

\item{grab_after_block}{A tuple of floats, specifying what fraction of the
remaining channels to remove following each glow block. Glow will take
the integer floor of this number multiplied by the remaining number of
channels. The default is half at each spatial hierarchy.
Default value: None (this will take out half of the channels after each block.}

\item{use_actnorm}{A boolean deciding whether or not to use actnorm. Data-dependent
initialization is used to initialize this layer. Default value: \code{FALSE}}

\item{seed}{A seed to control randomness in the 1x1 convolution initialization.
Default value: \code{NULL} (i.e., non-reproducible sampling).}

\item{validate_args}{Logical, default FALSE. Whether to validate input with asserts. If validate_args is
FALSE, and the inputs are invalid, correct behavior is not guaranteed.}

\item{name}{name prefixed to Ops created by this class.}
}
\value{
a bijector instance.
}
\description{
Overview: \code{Glow} is a chain of bijectors which transforms a rank-1 tensor
(vector) into a rank-3 tensor (e.g. an RGB image). \code{Glow} does this by
chaining together an alternating series of "Blocks," "Squeezes," and "Exits"
which are each themselves special chains of other bijectors. The intended use
of \code{Glow} is as part of a \code{tfd_transformed_distribution}, in
which the base distribution over the vector space is used to generate samples
in the image space. In the paper, an Independent Normal distribution is used
as the base distribution.
}
\details{
A "Block" (implemented as the \code{GlowBlock} Bijector) performs much of the
transformations which allow glow to produce sophisticated and complex mappings
between the image space and the latent space and therefore achieve rich image
generation performance. A Block is composed of \code{num_steps_per_block} steps,
which are each implemented as a \code{Chain} containing an
\code{ActivationNormalization} (ActNorm) bijector, followed by an (invertible)
\code{OneByOneConv} bijector, and finally a coupling bijector. The coupling
bijector is an instance of a \code{RealNVP} bijector, and uses the
\code{coupling_bijector_fn} function to instantiate the coupling bijector function
which is given to the \code{RealNVP}. This function returns a bijector which
defines the coupling (e.g. \code{Shift(Scale)} for affine coupling or \code{Shift} for
additive coupling).

A "Squeeze" converts spatial features into channel features. It is
implemented using the \code{Expand} bijector. The difference in names is
due to the fact that the \code{forward} function from glow is meant to ultimately
correspond to sampling from a \code{tfp$util$TransformedDistribution} object,
which would use \code{Expand} (Squeeze is just Invert(Expand)). The \code{Expand}
bijector takes a tensor with shape \verb{[H, W, C]} and returns a tensor with shape
\verb{[2H, 2W, C / 4]}, such that each 2x2x1 spatial tile in the output is composed
from a single 1x1x4 tile in the input tensor, as depicted in the figure below.

Forward pass (Expand)

\if{html}{\out{<div class="sourceCode">}}\preformatted{\\     \\       \\    \\    \\
\\\\     \\ ----> \\  1 \\  2 \\
\\\\\\__1__\\       \\____\\____\\
\\\\\\__2__\\        \\    \\    \\
\\\\__3__\\  <----  \\  3 \\  4 \\
\\__4__\\          \\____\\____\\
}\if{html}{\out{</div>}}

Inverse pass (Squeeze)
This is implemented using a chain of \code{Reshape} -> \code{Transpose} -> \code{Reshape}
bijectors. Note that on an inverse pass through the bijector, each Squeeze
will cause the width/height of the image to decrease by a factor of 2.
Therefore, the input image must be evenly divisible by 2 at least
\code{num_glow_blocks} times, since it will pass through a Squeeze step that many
times.

An "Exit" is simply a junction at which some of the tensor "exits" from the
glow bijector and therefore avoids any further alteration. Each exit is
implemented as a \code{Blockwise} bijector, where some channels are given to the
rest of the glow model, and the rest are given to a bypass implemented using
the \code{Identity} bijector. The fraction of channels to be removed at each exit
is determined by the \code{grab_after_block} arg, indicates the fraction of
remaining channels which join the identity bypass. The fraction is
converted to an integer number of channels by multiplying by the remaining
number of channels and rounding.
Additionally, at each exit, glow couples the tensor exiting the highway to
the tensor continuing onward. This makes small scale features in the image
dependent on larger scale features, since the larger scale features dictate
the mean and scale of the distribution over the smaller scale features.
This coupling is done similarly to the Coupling bijector in each step of the
flow (i.e. using a RealNVP bijector). However for the exit bijector, the
coupling is instantiated using \code{exit_bijector_fn} rather than coupling
bijector fn, allowing for different behaviors between standard coupling and
exit coupling. Also note that because the exit utilizes a coupling bijector,
there are two special cases (all channels exiting and no channels exiting).
The full Glow bijector consists of \code{num_glow_blocks} Blocks each of which
contains \code{num_steps_per_block} steps. Each step implements a coupling using
\code{bijector_coupling_fn}. Between blocks, glow converts between spatial pixels
and channels using the Expand Bijector, and splits channels out of the
bijector using the Exit Bijector. The channels which have exited continue
onward through Identity bijectors and those which have not exited are given
to the next block. After passing through all Blocks, the tensor is reshaped
to a rank-1 tensor with the same number of elements. This is where the
distribution will be defined.
A schematic diagram of Glow is shown below. The \code{forward} function of the
bijector starts from the bottom and goes upward, while the \code{inverse} function
starts from the top and proceeds downward.
}
\section{#' ```}{
Glow Schematic Diagram
Input Image     ########################   shape = [H, W, C]
\                      /<- Expand Bijector turns spatial
\                    /    dimensions into channels.
|  XXXXXXXXXXXXXXXXXXXX
|  XXXXXXXXXXXXXXXXXXXX
|  XXXXXXXXXXXXXXXXXXXX     A single step of the flow consists
Glow Block  - |  XXXXXXXXXXXXXXXXXXXX  <- of ActNorm -> 1x1Conv -> Coupling.
|  XXXXXXXXXXXXXXXXXXXX     there are num_steps_per_block
|  XXXXXXXXXXXXXXXXXXXX     steps of the flow in each block.
|_ XXXXXXXXXXXXXXXXXXXX
\                  / <-- Expand bijectors follow each glow
\                /      block
XXXXXXXX\\\\\\\\   <-- Exit Bijector removes channels
_                    _     from additional alteration.
|    XXXXXXXX !  |  !
|    XXXXXXXX !  |  !
|    XXXXXXXX !  |  !       After exiting, channels are passed
Glow Block  - |    XXXXXXXX !  |  !  <--- downward using the Blockwise and
|    XXXXXXXX !  |  !       Identify bijectors.
|    XXXXXXXX !  |  !
|_   XXXXXXXX !  |  !
\              / <---- Expand Bijector
\            /
XXX\\\    | !  <---- Exit Bijector
_
|      XXX ! |   | !
|      XXX ! |   | !
|      XXX ! |   | !
low Block  - |      XXX ! |   | !
|      XXX ! |   | !
|      XXX ! |   | !
|_     XXX ! |   | !
XX\ ! |   | ! <----- (Optional) Exit Bijector
|    |   |
v    v   v
Output Distribution    ##########          shape = [H * W * C]

\if{html}{\out{<div class="sourceCode">}}\preformatted{    Legend
}\if{html}{\out{</div>}}
}

\section{| XX  = Step of flow      |
| X\  = Exit bijector     |
| \/  = Expand bijector   |
| !|! = Identity bijector |
|                         |
| up  = Forward pass      |
| dn  = Inverse pass      |
|_________________________|}{
\if{html}{\out{<div class="sourceCode">}}\preformatted{
[H, W, C]: R:H,\%20W,\%20C
[2H, 2W, C / 4]: R:2H,\%202W,\%20C\%20/\%204
[H, W, C]: R:H,\%20W,\%20C
[H * W * C]: R:H\%20*\%20W\%20*\%20C
}\if{html}{\out{</div>}}
}

\section{References}{

\itemize{
\item \href{https://arxiv.org/abs/1807.03039}{Diederik P Kingma, Prafulla Dhariwal, Glow: Generative Flow with Invertible 1x1 Convolutions. In \emph{Neural Information Processing Systems}, 2018.}
\item \href{https://arxiv.org/abs/1605.08803}{Laurent Dinh, Jascha Sohl-Dickstein, and Samy Bengio. Density Estimation using Real NVP. In \emph{International Conference on Learning Representations}, 2017.}
}
}

\seealso{
For usage examples see \code{\link[=tfb_forward]{tfb_forward()}}, \code{\link[=tfb_inverse]{tfb_inverse()}}, \code{\link[=tfb_inverse_log_det_jacobian]{tfb_inverse_log_det_jacobian()}}.

Other bijectors: 
\code{\link{tfb_absolute_value}()},
\code{\link{tfb_affine_linear_operator}()},
\code{\link{tfb_affine_scalar}()},
\code{\link{tfb_affine}()},
\code{\link{tfb_ascending}()},
\code{\link{tfb_batch_normalization}()},
\code{\link{tfb_blockwise}()},
\code{\link{tfb_chain}()},
\code{\link{tfb_cholesky_outer_product}()},
\code{\link{tfb_cholesky_to_inv_cholesky}()},
\code{\link{tfb_correlation_cholesky}()},
\code{\link{tfb_cumsum}()},
\code{\link{tfb_discrete_cosine_transform}()},
\code{\link{tfb_expm1}()},
\code{\link{tfb_exp}()},
\code{\link{tfb_ffjord}()},
\code{\link{tfb_fill_scale_tri_l}()},
\code{\link{tfb_fill_triangular}()},
\code{\link{tfb_gompertz_cdf}()},
\code{\link{tfb_gumbel_cdf}()},
\code{\link{tfb_gumbel}()},
\code{\link{tfb_identity}()},
\code{\link{tfb_inline}()},
\code{\link{tfb_invert}()},
\code{\link{tfb_iterated_sigmoid_centered}()},
\code{\link{tfb_kumaraswamy_cdf}()},
\code{\link{tfb_kumaraswamy}()},
\code{\link{tfb_lambert_w_tail}()},
\code{\link{tfb_masked_autoregressive_default_template}()},
\code{\link{tfb_masked_autoregressive_flow}()},
\code{\link{tfb_masked_dense}()},
\code{\link{tfb_matrix_inverse_tri_l}()},
\code{\link{tfb_matvec_lu}()},
\code{\link{tfb_normal_cdf}()},
\code{\link{tfb_ordered}()},
\code{\link{tfb_pad}()},
\code{\link{tfb_permute}()},
\code{\link{tfb_power_transform}()},
\code{\link{tfb_rational_quadratic_spline}()},
\code{\link{tfb_rayleigh_cdf}()},
\code{\link{tfb_real_nvp_default_template}()},
\code{\link{tfb_real_nvp}()},
\code{\link{tfb_reciprocal}()},
\code{\link{tfb_reshape}()},
\code{\link{tfb_scale_matvec_diag}()},
\code{\link{tfb_scale_matvec_linear_operator}()},
\code{\link{tfb_scale_matvec_lu}()},
\code{\link{tfb_scale_matvec_tri_l}()},
\code{\link{tfb_scale_tri_l}()},
\code{\link{tfb_scale}()},
\code{\link{tfb_shifted_gompertz_cdf}()},
\code{\link{tfb_shift}()},
\code{\link{tfb_sigmoid}()},
\code{\link{tfb_sinh_arcsinh}()},
\code{\link{tfb_sinh}()},
\code{\link{tfb_softmax_centered}()},
\code{\link{tfb_softplus}()},
\code{\link{tfb_softsign}()},
\code{\link{tfb_split}()},
\code{\link{tfb_square}()},
\code{\link{tfb_tanh}()},
\code{\link{tfb_transform_diagonal}()},
\code{\link{tfb_transpose}()},
\code{\link{tfb_weibull_cdf}()},
\code{\link{tfb_weibull}()}
}
\concept{bijectors}
