% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cox_regression.R
\name{tidy_coxreg}
\alias{tidy_coxreg}
\alias{tidy.summary.coxph}
\alias{tidy.coxreg.univar}
\alias{tidy.coxreg.multivar}
\title{Custom tidy methods for Cox regression}
\usage{
\method{tidy}{summary.coxph}(x, ...)

\method{tidy}{coxreg.univar}(x, ...)

\method{tidy}{coxreg.multivar}(x, ...)
}
\arguments{
\item{x}{(\code{list})\cr result of the Cox regression model fitted by \code{\link[=fit_coxreg_univar]{fit_coxreg_univar()}} (for univariate models)
or \code{\link[=fit_coxreg_multivar]{fit_coxreg_multivar()}} (for multivariate models).}

\item{...}{additional arguments for the lower level functions.}
}
\value{
\code{\link[=tidy]{tidy()}} returns:
\itemize{
\item For \code{summary.coxph} objects,  a \code{data.frame} with columns: \verb{Pr(>|z|)}, \code{exp(coef)}, \code{exp(-coef)}, \verb{lower .95},
\verb{upper .95}, \code{level}, and \code{n}.
\item For \code{coxreg.univar} objects, a \code{data.frame} with columns: \code{effect}, \code{term}, \code{term_label}, \code{level}, \code{n}, \code{hr},
\code{lcl}, \code{ucl}, \code{pval}, and \code{ci}.
\item For \code{coxreg.multivar} objects, a \code{data.frame} with columns: \code{term}, \code{pval}, \code{term_label}, \code{hr}, \code{lcl}, \code{ucl},
\code{level}, and \code{ci}.
}
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#stable}{\figure{lifecycle-stable.svg}{options: alt='[Stable]'}}}{\strong{[Stable]}}
}
\section{Functions}{
\itemize{
\item \code{tidy(summary.coxph)}: Custom tidy method for \code{\link[survival:coxph]{survival::coxph()}} summary results.

Tidy the \code{\link[survival:coxph]{survival::coxph()}} results into a \code{data.frame} to extract model results.

\item \code{tidy(coxreg.univar)}: Custom tidy method for a univariate Cox regression.

Tidy up the result of a Cox regression model fitted by \code{\link[=fit_coxreg_univar]{fit_coxreg_univar()}}.

\item \code{tidy(coxreg.multivar)}: Custom tidy method for a multivariate Cox regression.

Tidy up the result of a Cox regression model fitted by \code{\link[=fit_coxreg_multivar]{fit_coxreg_multivar()}}.

}}
\examples{
library(survival)
library(broom)

set.seed(1, kind = "Mersenne-Twister")

dta_bladder <- with(
  data = bladder[bladder$enum < 5, ],
  data.frame(
    time = stop,
    status = event,
    armcd = as.factor(rx),
    covar1 = as.factor(enum),
    covar2 = factor(
      sample(as.factor(enum)),
      levels = 1:4, labels = c("F", "F", "M", "M")
    )
  )
)
labels <- c("armcd" = "ARM", "covar1" = "A Covariate Label", "covar2" = "Sex (F/M)")
formatters::var_labels(dta_bladder)[names(labels)] <- labels
dta_bladder$age <- sample(20:60, size = nrow(dta_bladder), replace = TRUE)

formula <- "survival::Surv(time, status) ~ armcd + covar1"
msum <- summary(coxph(stats::as.formula(formula), data = dta_bladder))
tidy(msum)

## Cox regression: arm + 1 covariate.
mod1 <- fit_coxreg_univar(
  variables = list(
    time = "time", event = "status", arm = "armcd",
    covariates = "covar1"
  ),
  data = dta_bladder,
  control = control_coxreg(conf_level = 0.91)
)

## Cox regression: arm + 1 covariate + interaction, 2 candidate covariates.
mod2 <- fit_coxreg_univar(
  variables = list(
    time = "time", event = "status", arm = "armcd",
    covariates = c("covar1", "covar2")
  ),
  data = dta_bladder,
  control = control_coxreg(conf_level = 0.91, interaction = TRUE)
)

tidy(mod1)
tidy(mod2)

multivar_model <- fit_coxreg_multivar(
  variables = list(
    time = "time", event = "status", arm = "armcd",
    covariates = c("covar1", "covar2")
  ),
  data = dta_bladder
)
broom::tidy(multivar_model)

}
\seealso{
\link{cox_regression}
}
