#' conflate: Calculate the conflation of multiple distributions
#' pdfs, plot = TRUE, from, to, n, add = FALSE
#' @param pdfs A vector of calls to density_fun for defininf each individual distribution.
#' @param plot Whether to plot using curve
#' @param from,to,n The appropriate values from and to which to calculate the conflation, and a number of points n. These are the same used by the function curve but are still necessary even if no plot is required.
#' @param add Whether to add the curve to an existing plot.
#' 
#' @return A tree of class phylo with summary branch lengths in tree$edge.length.
#'
#' @details Produces either a plot or a data frame
#' with the x and y values for the conflated PDF. It uses as input
#' a vector of densities constructed with
#' density_fun, and further parameters to be pased to curve
#' if no plot is desired these are still used for returning
#' a data frame with the x and y values from evaluation the
#' conflated PDF on the sequence of x values determined by
#' a number n of equidistant points between from and to.
#' @examples
#' \donttest{
#' conflated_normals <- conflate(c("density_fun(x, 'dnorm', mean=0, sd=1)",
#'                                 "density_fun(x, 'dnorm', mean=3, sd=1)"),
#'                               from=-4, to=4, n=101, plot=FALSE)
#' plot(conflated_normals)
#' }
#' 
#' @export
#' @importFrom graphics curve

conflate <- function(pdfs, plot = TRUE, from, to, n, add = FALSE) {
    conflated <- .str_conflate(pdfs, from = from, to = to)
    # if plot, use curve with args from, to, and n as usual
    if (plot) {
        curve(expr = .wrapper_conflation(x, conflated),
              from=from, to=to, n=n, add=add, ylab="Density")
    # otherwise, calculate the same x and y vals for the conflation
    # and return a data frame with these coordinates 
    } else {
        xx <- seq(from, to, by=(to - from)/(n - 1))
        yy <- .wrapper_conflation(xx, conflated)
        return(data.frame(x = xx, y = yy))
    }
}

#' density_fun: A way to represent distributions to be conflated
#' @param x a symbol which needs to be present in order to allow passing values towards the distruibution generator. It should be just x, without quotation marks (see examples).
#' @param dist A character (using single quotes!) with the name of the distribution to use.
#' @param ... Parameters to be passed on to dist. See examples
#' 
#' @return A call from the elements of the distribution to be used.
#'
#' @details Produces a definition of each individual distribution to be conflated
#' and provides the symbol for non-standard evaluation (x). Single quotes in dist
#' are mandatory in order to avoid issues when calling expressions under the hood.
#' See the documentation for each individual distribution to call their parameters
#' adequately. Argument names and values should be used.
#' @examples
#' \donttest{
#' c("density_fun(x, 'dnorm', mean=0, sd=1)",
#'   "density_fun(x, 'dnorm', mean=-1, sd=1)",
#'   "density_fun(x, 'dnorm', mean=1, sd=1)")
#' }
#' 
#' @export
#' @importFrom graphics curve

density_fun <- function(x, dist, ...) {
    params <- list(...)
    callpars <- vector("list")
    callpars[[1]] <- x
    for (i in params) {
        callpars <- c(callpars, i)
    }
    return(do.call(dist, callpars))
}

#' .str_conflate: Build calls for density conflation
#' @param ... Densities to be generated by density_fun.
#' @param from,to The appropriate values from and to which to calculate the conflation.
#' 
#' @return A string with the conflated density to be used as expression later.
#'
#' @details This function should not be called by the user.
#' now we need to use individual calls to density_fun
#' and evaluate them in the conflation
#' this function will take a vector of densities specified
#' with densiti_fun and build the conflation string call
#' with densities / int(densities) in an appropriate
#' interval from--to
#' @examples
#' \donttest{
#' conflated <- .str_conflate("density_fun(x, 'dnorm', mean=0, sd=1)", "density_fun(x, 'dnorm', mean=3, sd=1)", from = -1, to = 1)
#' }
#' 
#' @noRd
#' @importFrom graphics curve

.str_conflate <- function(..., from, to) {
    densities <- list(...)
    densities <- unlist(densities)
    numerator <- paste(densities, collapse=" * ")
    denominator <- paste("integrate(function(x) ", numerator, ", lower = ", from, ", upper = ", to, ")[['value']]", sep="")
    fun_string <- paste(numerator, denominator, sep=" / ")
    #return(numerator)
    return(fun_string)
}

#' .wrapper_conflation: Wrapper for using curve
#' @param x A symbol, unquoted x should be just fine.
#' @param expr The expression from .str_conflate to be evaluated.
#' 
#' @return The evaluation of an expression.
#'
#' @details 
#' we need to wrap the string call in order to use curve, but not for integrate
#' this wrapper is just to make curve happy when plotting the
#' conflation but it is also useful for calculating a series
#' of x values of the conflated PDF from a sequence of x values
#' @examples
#' \donttest{
#' conflated <- .str_conflate("density_fun(x, 'dnorm', mean=0, sd=1)", "density_fun(x, 'dnorm', mean=3, sd=1)", from = -1, to = 1)
#' }
#' 
#' @noRd

.wrapper_conflation <- function(x, expr) {
    eval(str2expression(expr))
}
