#' Summary for a Categorical Row
#' 
#' Summarizes a categorical row using counts and column proportions.
#' @param dt the name of the dataframe object.
#' @param rowlabels the label for the table row name, if different from row_var.
#' @param missing logical: if TRUE, missing data is considered; FALSE only uses complete cases.
#' @param digits significant digits to use.
#' @return A dataframe with summary statistics for a categorical variable.
#' @import dplyr
#' @keywords tangram.pipe
#' @export

cat_default <- function(dt, rowlabels, missing, digits){
  rnd <- paste0("%.", digits, "f")
  if (!is.null(ncol(dt))){
    dt <- filter(dt, !is.na(dt[,2]))
    ct <- dt %>%
      table(useNA=ifelse(missing==TRUE, "ifany", "no")) %>%
      as.matrix() %>%
      cbind(Overall=dt %>%
              table(useNA=ifelse(missing==TRUE, "ifany", "no")) %>%
              rowSums())
    prop <- dt %>%
      table(useNA=ifelse(missing==TRUE, "ifany", "no")) %>%
      prop.table(margin=2) %>%
      as.matrix() %>%
      cbind(Overall=dt %>%
              table(useNA=ifelse(missing==TRUE, "ifany", "no")) %>%
              prop.table() %>%
              rowSums())

    cols <- unlist(dimnames(prop)[2])
    out <- matrix(paste0(sprintf(rnd, prop), " (", ct, ")"), nrow=nrow(prop), dimnames = list(NULL,cols)) %>%
      as.data.frame() 

    out <- cbind(dimnames(prop)[1], out)

    row1 <- c(paste(rowlabels), rep("", ncol(out)-1))
    out <- rbind(row1, out)
    if (missing == TRUE){
      out[is.na(out[,1]),1] <- "Missing"
    }
    out <- cbind(out[,1], Measure="", out[,(2:ncol(out))])
    out$Measure[1] <- "Col. Prop. (N)"
    colnames(out)[1] <- "Variable"
  } else {
    ct=dt %>%
      table(useNA=ifelse(missing==TRUE, "ifany", "no"))
    prop=dt %>%
      table(useNA=ifelse(missing==TRUE, "ifany", "no")) %>%
      prop.table()
    
    out <- matrix(paste0(sprintf(rnd, prop), " (", ct, ")"), nrow=nrow(prop)) %>%
      as.data.frame()

    out <- cbind(dimnames(prop), out)

    row1 <- c(paste(rowlabels), "Overall")
    out <- rbind(row1, out)
    if (missing == TRUE){
      out[is.na(out[,1]),1] <- "Missing"
    }

    out <- data.frame(out[,1], Measure="", out[,2])
    out$Measure[1] <- "Col. Prop. (N)"
    colnames(out)[1] <- "Variable"
    colnames(out)[3] <- "Overall"
  }
  out
}
