/**
 * [GridCluster.java]
 * 
 * Cluster class for DStream.
 * 
 * @author Yunsu Kim
 * Data Management and Data Exploration Group, RWTH Aachen University
 */

package moa.clusterers.dstream;

import java.util.ArrayList;
import java.util.List;
import java.util.Random;

import moa.cluster.Cluster;
import weka.core.Instance;

public class GridCluster extends Cluster {
	
	private static final long serialVersionUID = 1L;
	
	protected List<Grid> members;
	
	
	/* Constructors */
	
	public GridCluster(double label) {
		members = new ArrayList<Grid>();
		setId(label);
	}
	
	public GridCluster(Grid g, double label) {
		this(label);
		addGrid(g);
	}
	
	public GridCluster(List<Grid> grids, double label) {
		this(label);
		addGrids(grids);
	}
	
	
	/* Updates */
	
	public void addGrid(Grid g) {
		members.add(g);
		g.setClassLabel(getId());
	}
	
	public void addGrids(List<Grid> grids) {
		for (Grid g : grids) {
			addGrid(g);
		}
	}
	
	public void removeGrid(Grid g) {
		members.remove(g);
		g.setNoClass();
	}
	
	public void removeGrids(List<Grid> grids) {
		for (Grid g : grids) {
			removeGrid(g);
		}
	}
	
	public void mergeCluster(GridCluster G) {
		List<Grid> clustersToBeMerged = G.getGrids();
		for (Grid g : clustersToBeMerged) {
			g.setClassLabel(getId());
		}
		
		// Avoid duplicates
		members.removeAll(clustersToBeMerged);
		members.addAll(clustersToBeMerged);
	}
	
	
	/* Checking functions */
	
	public boolean containsGrid(Grid g) {
		return members.contains(g);
	}
	
	public List<Grid> getSplitGroup() {
		// Connectivity check
		List<Integer> unexplored = new ArrayList<Integer>();
		List<Integer> frontiers = new ArrayList<Integer>();
		
		for (int i = 0; i < members.size(); i++) {
			unexplored.add(i);
		}
		
		if (unexplored.size() == 0) {
			return null;
		} else {
			frontiers.add(unexplored.remove(0));
			
			while (!frontiers.isEmpty()) {
				Grid f = members.get(frontiers.get(0));
				
				for (int i : unexplored) {
					if (f.isNeighboringGridOf(members.get(i))) {
						frontiers.add(i);
					}
				}
				
				unexplored.removeAll(frontiers);
				frontiers.remove(0);
			}
			
			if (unexplored.isEmpty()) {
				return null;
			} else {	// Not connected
				List<Grid> splitGroup = new ArrayList<Grid>();
				for (int i : unexplored) {
					splitGroup.add(members.get(i));
				}
				
				return splitGroup;
			}
		}
	}
	
	public boolean isNeighboringClusterOf(Grid g) {
		for (Grid h : members) {
			if (h.isNeighboringGridOf(g)) {
				return true;
			}
		}
		
		return false;
	}
	
	public boolean isReallyCluster() {
		for (Grid g : members) {
			if (g.isInsideGridIn(this)) {
				if (!g.isDenseGrid()) {
					return false;
				}
			} else {
				if (g.isSparseGrid()) {
					return false;
				}
			}
		}
		
		return true;
	}
	
	
	/* Getters */
	
	public List<Grid> getGrids() {
		return members;
	}
	
	public int size() {
		return members.size();
	}
	

	/* Cluster properties */
	
	@Override
	public double getInclusionProbability(Instance instance) {
		for (Grid g : members) {
			if (g.includes(instance)) {
				return 1;
			}
		}
		
		return 0;
	}
	
	
	/* Redundant overrides */
	
	@Override
	public double[] getCenter() {
		return null;
	}

	@Override
	public double getWeight() {
		return -1;
	}

	@Override
	public Instance sample(Random random) {
		return null;
	}
}
