#' @export jmodeltest.run
#' 
#' @title jModelTest
#' @description Run jModelTest to determine appropriate substitution model.
#' 
#' @param x a list of DNA sequences.
#' @param sub.schemes number of substitution schemes to test. Can be one of 3, 5, 7, 11, or 203.
#' @param unequal.base.freq logical. Include models with unequal base frequencies?
#' @param prop.inv.sites logical. Include models with a proportion of invariable sites?
#' @param rate.var number of categories for models with rate variation among sites.
#' @param AIC,AICc,BIC,DT logical. Calculate respective information criterion metrics?
#' @param param.imp logical. Calculate parameter importances?
#' @param model.average logical. Do model averaging and parameter importances?
#' @param path path where \code{jModelTest.jar} is located.
#' @param java.opts options to \code{java} command line.
#' 
#' @details Requires that jModelTest is properly installed on the system in \code{path}.
#' 
#' @references Darriba D, Taboada GL, Doallo R, Posada D. 2012. jModelTest 2: more models, 
#'   new heuristics and parallel computing. Nature Methods 9(8), 772.\cr
#'   Guindon S and Gascuel O (2003). A simple, fast and accurate method to estimate large 
#'   phylogenies by maximum-likelihood". Systematic Biology 52: 696-704.\cr
#'   Software available at: \url{https://code.google.com/p/jmodeltest2/}
#' 
#' @author Eric Archer \email{eric.archer@@noaa.gov}

jmodeltest.run <- function(x, sub.schemes = 3, unequal.base.freq = FALSE,
  prop.inv.sites = FALSE, rate.var = NULL, AIC = FALSE, AICc = FALSE,
  BIC = FALSE, DT = FALSE, param.imp = FALSE, model.average = FALSE,
  path = ifelse(.Platform$OS.type == "windows", "C:/Program Files/jModelTest", "/usr/local/bin/jmodeltest"),
  java.opts = NULL) {
  
  if(!sub.schemes %in% c(3, 5, 7, 11, 203)) stop("'sub.schemes' not equal to 3, 5, 7, 11, or 203.")
  
  stopifnot.aligned(x, show.warnings = TRUE)
  
  wd <- getwd()
  setwd(path)
  in.file <- write.fasta(x, file = "jModelTest.fasta")
  output.file <- gsub(".fasta", ".results.txt", in.file)
  
  modeltest.call <- paste("java", java.opts, "-jar jModelTest.jar",
    "-d", in.file,
    "-s", sub.schemes,
    ifelse(unequal.base.freq, "-f", ""),
    ifelse(prop.inv.sites, "-i", ""),
    ifelse(!is.null(rate.var), paste("-g", rate.var), ""),
    ifelse(AIC, "-AIC", ""),
    ifelse(AICc, "-AICc", ""),
    ifelse(BIC, "-BIC", ""),
    ifelse(DT, "-DT", ""),
    ifelse(param.imp, "-p", ""),
    ifelse(model.average, "-v", ""),
    ">", output.file
  )
  system(modeltest.call, intern = F)
  
  file.remove(in.file)
  file.rename(file.path(path, output.file), file.path(wd, output.file))
  setwd(wd)
  
  output.file
}