% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fitModels.R
\name{fitModels}
\alias{fitModels}
\title{Fit spatial models per time point}
\usage{
fitModels(
  TP,
  trait,
  timePoints = names(TP),
  extraFixedFactors = NULL,
  geno.decomp = NULL,
  what = c("random", "fixed"),
  useCheck = FALSE,
  useRepId = FALSE,
  engine = c("SpATS", "asreml"),
  spatial = FALSE,
  quiet = FALSE
)
}
\arguments{
\item{TP}{An object of class \code{TP}.}

\item{trait}{A character string indicating the trait used as response
variable in the model.}

\item{timePoints}{A character or numeric vector indicating the time points
to be modeled. When using a character string to reference a time point, the
value has to be an exact match to one of the existing time points. When using
a number it will be matched by its number ("timeNumber") in the timePoints
attribute of the \code{TP} object.}

\item{extraFixedFactors}{A character vector indicating the variables to use
as extra fixed effects in the model.}

\item{geno.decomp}{A character vector indicating the variables to use to
group the genotypic variance in the model.}

\item{what}{A character vector specifying whether "genotype" should
be fitted as "random" or "fixed" effect. Note that when using
\code{geno.decomp}, fitting a model with genotype as "fixed" effect is not
possible.}

\item{useCheck}{Should check genotypes be used as an extra factor in the
model?}

\item{useRepId}{Should repId be used as a fixed effect in the model? When
fitting a spatial model rowId and colId are also nested within repId in the
random part of the model.}

\item{engine}{A character string indicating the engine used to fit the
models.}

\item{spatial}{Should a spatial model be fitted for asreml?}

\item{quiet}{Should printed progress messages be suppressed?}
}
\value{
An object of class \code{fitMod}, a list of fitted models.
}
\description{
Perform REML analysis at each time point using either SpATS or asreml. The
idea is to is to accurately separate the genetic effects from the spatial
effects at each time point. SpATS is used as a default method. See details
for the exact models fitted.
}
\details{
The actual model fitted depends on the function parameters specified. The
basic model is the following:\cr
trait = \strong{genotype} + e\cr
In case \code{useCheck = TRUE}, instead of genotype, genoCheck is used as
genotype and check is used as an extra fixed effect. So then the model
becomes:\cr
trait = \emph{check} + \strong{genoCheck} + e\cr
Variables in \code{extraFixedFactors} are fitted as extra fixed
effects.\cr\cr
When \code{SpATS} is used for modeling, an extra spatial term is always
included in the model. This term is constructed using the function
\code{\link[SpATS]{PSANOVA}} from the SpATS package as\cr
\code{PSANOVA(colNum, rowNum, nseg = nSeg, nest.div = 2)}
where\cr \code{nSeg = c(number of columns, number of rows)}.\cr\cr
When \code{asreml} is used for modeling and \code{spatial = TRUE},
four models are fitted with different covariance structures.
The best model is determined based on a goodness-of-fit criterion, AIC,
on 20\% of the time points or at least 10 time points. The best model is then
run on all time points.
The following combinations of random and spatial terms are fitted
\itemize{
\item{random = repId:rowId + repId:colId, spatial = NULL}
\item{random = repId:rowId + repId:colId, spatial = ar1(rowId):colId}
\item{random = repId:colId + repId:colId, spatial = rowId:ar1(colId)}
\item{random = repId:rowId + repId:colId, spatial = ar1(rowId):ar1(colId)}
}
If there are no replicates in the model, repId is left out from the random
parts above.\cr\cr
When \code{geno.decomp} is specified, the genotypic variance is decomposed
following the variable(s) chosen. For example, when a treatment is used in
\code{geno.decomp}, the initial model becomes:\cr
trait = \emph{treatment} + \strong{treatment:genotype} + e\cr
}
\examples{
## Using the first example dataset (PhenovatorDat1):
## Fit a model using SpATS on few time points:
\donttest{
## Create an object of class TP.
phenoTP <- createTimePoints(dat = PhenovatorDat1,
                            experimentName = "Phenovator",
                            genotype = "Genotype",
                            timePoint = "timepoints",
                            repId = "Replicate",
                            plotId = "pos",
                            rowNum = "y", colNum = "x",
                            addCheck = TRUE,
                            checkGenotypes = c("check1", "check2",
                                               "check3", "check4"))
## Fit a model with SpATS for three time points.
modPhenoSp <- fitModels(TP = phenoTP,
                        trait = "EffpsII",
                        timePoints = c(3, 6, 20))
summary(modPhenoSp)

## Fit a model with SpATS for a single time point with extra fixed factors
## and check genotypes:
modPhenoSpCheck <- fitModels(TP = phenoTP,
                             trait = "EffpsII",
                             extraFixedFactors = c("repId", "Image_pos"),
                             useCheck = TRUE,
                             timePoints = 3)


## Fit a model with asreml on few time points with a spatial function:
if (requireNamespace("asreml", quietly = TRUE)) {
  modPhenoSpAs <- fitModels(TP = phenoTP,
                            trait = "EffpsII",
                            timePoints = c(1, 6, 20),
                            engine = "asreml",
                            spatial = TRUE)
}

## Using the second example dataset (PhenoarchDat1):
## Fit a model with SpATS on one time points with two variables for
## geno.decomp:
data("PhenoarchDat1")
phenoTParch <- createTimePoints(dat = PhenoarchDat1,
                                experimentName = "Phenoarch",
                                genotype = "Genotype",
                                timePoint = "Date",
                                plotId = "pos",
                                rowNum = "Row",
                                colNum = "Col")

modPhenoSpGD <- fitModels(TP = phenoTParch,
                          trait = "LeafArea",
                          geno.decomp = c("Scenario", "population"),
                          timePoints = 16)
}


}
\references{
Maria Xose Rodriguez-Alvarez, Martin P. Boer, Fred A. van Eeuwijk, Paul H.C.
Eilers (2017). Correcting for spatial heterogeneity in plant breeding
experiments with P-splines. Spatial Statistics
\doi{10.1016/j.spasta.2017.10.003}
Butler, D. G., et al. (2018). ASReml-R Reference Manual Version 4. VSN
International Ltd, http://asreml.org
}
\seealso{
Other functions for spatial modeling: 
\code{\link{getCorrected}()},
\code{\link{getEffDims}()},
\code{\link{getGenoPred}()},
\code{\link{getHerit}()},
\code{\link{getVar}()},
\code{\link{plot.fitMod}()},
\code{\link{summary.fitMod}()}
}
\concept{functions for spatial modeling}
