% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sample_lcomp.R
\name{sample_lcomp}
\alias{sample_lcomp}
\title{Sample length compositions from expected values}
\usage{
sample_lcomp(dat_list, outfile, fleets = c(1, 2), Nsamp, years, cpar = 1,
  ESS = NULL, write_file = TRUE)
}
\arguments{
\item{dat_list}{An SS data list object as read in from
\code{\link[r4ss]{SS_readdat}} in the \pkg{r4ss} package. Make sure
you select option \code{section=2}.}

\item{outfile}{A character string of the new file name to be created.
Must end in \code{.dat} or equal \code{wtatage.ss}.}

\item{fleets}{*A numeric vector giving the fleets to be used. This order also
pertains to other arguments. A missing value excludes that fleet from
\code{outfile} (i.e. it turns it off so no samples are written). If none of
the fleet collected samples, keep the value to \code{fleets=NULL}.}

\item{Nsamp}{*A numeric list of the same length as \code{fleets}.
Either single values or vectors of the same length as the number of
years can be passed through. Single values are repeated for all
years. If no fleet collected samples, keep the value to
\code{Nsamp=NULL}.}

\item{years}{*A list the same length as \code{fleets} giving the years as
numeric vectors. If no fleet collected samples, keep the value to
\code{years=NULL}.}

\item{cpar}{*A numeric value or vector the same length as
\code{fleets} controlling the variance of the Dirichlet
distribution used for sampling. A value of \code{1} indicates the
same standard deviation as a multinomial of the given \code{Nsamp},
\code{2} indicates twice, etc. Values greater than one indicate
overdispersion, and less underdispersion.}

\item{ESS}{An effective sample size to write to file (but not use in
sampling). The default of NULL means to use the true (internally
calculated) ESS, which is \code{Nsamp} for the multinomial case or given
by the formula under \code{cpar} for the Dirichlet case. Must match the
structure of the \code{years} arguments.}

\item{write_file}{A logical switch for whether to write \code{outfile} to
disk. Can be turned off to speed up testing or exploration of the
function. The new data are returned invisibly, as in the examples
below.}
}
\value{
A modified \code{.dat} file if \code{write_file=TRUE}. A list object
containing the modified \code{.dat} file is returned invisibly.
}
\description{
Take a \code{data.SS_new} file containing expected values and sample to
create observed length compositions which are then written to file for use by
the estimation model.
If used with \code{\link{run_ss3sim}} the case file should be named
\code{lcomp}. A suggested (default) case letter is \code{D} for data.
}
\section{Which arguments to specifiy in case files}{

All function argument descriptions that start with an asterisk (*) will be passed
through the case files to \code{\link{run_ss3sim}}. If one of these arguments
is not specified in a case file, then a value of \code{NULL} will be passed,
which may or may not be an appropriate value. Other arguments will be ignored
if specified.
}

\examples{
d <- system.file("extdata", package = "ss3sim")
f_in <- paste0(d, "/models/cod-om/codOM.dat")
dat_list <- r4ss::SS_readdat(f_in, verbose = FALSE)
dat_list <- change_fltname(dat_list)

## Generate with constant sample size across years
ex1 <- sample_lcomp(dat_list=dat_list, outfile="test1.dat", fleets=c(1,2),
                    Nsamp=list(100,50), years=list(seq(26, 100, by=2),
                                            80:100), write_file = FALSE)

## Generate with varying Nsamp by year for first fleet
ex2 <- sample_lcomp(dat_list=dat_list, outfile="test2.dat", fleets=c(1,2),
                    Nsamp=list(c(rep(50, 5), rep(100, 5)), 50),
                    years=list(seq(26, 44, by=2),
                        80:100), write_file = FALSE)

\dontrun{
## Plot distributions for a particular year to compare multinomial
## vs. overdispersed Dirichlet
temp.list <- temp.list2 <- list()
for(i in 1:40){
    temp.list[[i]] <-
      sample_lcomp(dat_list=dat_list, outfile="test1.dat", fleets=c(2), cpar=c(3),
                     Nsamp=list(100), years=list(1995),
                     write_file=FALSE)
    temp.list2[[i]] <-
        sample_lcomp(dat_list=dat_list, outfile="test1.dat", fleets=c(2),
                     cpar=c(NA), Nsamp=list(100), years=list(1995),
                     write_file=FALSE)
}
## Organize the data for plotting
x1 <- reshape2::melt(do.call(rbind, temp.list)[,-(1:6)[-3]], id.vars="FltSvy")
x2 <- reshape2::melt(do.call(rbind, temp.list2)[,-(1:6)[-3]], id.vars="FltSvy")
op <- par(mfrow=c(2,1))
with(x1, boxplot(value~variable, las=2, ylim=c(0,.6), ylab="Proportion",
                 main="Overdispersed (cpar=3)",  xlab="length bin"))
temp <- as.numeric(subset(dat_list$lencomp, Yr==1995 & FltSvy == 2)[-(1:6)])
points(temp/sum(temp), pch="-", col="red")
with(x2, boxplot(value~variable, las=2, ylim=c(0,.6), ylab="Proportion",
                 main="Multinomial", xlab="length bin"))
temp <- as.numeric(subset(dat_list$lencomp, Yr==1995 & FltSvy == 2)[-(1:6)])
points(temp/sum(temp), pch="-", col="red")
par(op)
}

}
\seealso{
Other sampling functions: \code{\link{clean_data}},
  \code{\link{sample_agecomp}},
  \code{\link{sample_calcomp}}, \code{\link{sample_index}},
  \code{\link{sample_mlacomp}},
  \code{\link{sample_wtatage}}
}
\author{
Cole Monnahan and Kotaro Ono; modified from a version by Roberto
  Licandeo and Felipe Hurtado-Ferro
}
