\name{envelope}
\alias{envelope}
\title{Simulation envelopes of summary function}
\description{
  Computes simulation envelopes of a summary function.
}
\usage{
  envelope(Y, fun=Kest, nsim=99, nrank=1, verbose=TRUE, \dots,
  simulate=NULL, clipdata=TRUE, start=NULL, control=list(nrep=1e5, expand=1.5))
}
\arguments{
  \item{Y}{Either a point pattern (object of class
    \code{"ppp"}) or a fitted point process model
    (object of class \code{"ppm"}).
    }
    \item{fun}{
      Function that computes the desired summary statistic
      for a point pattern. 
    }
    \item{nsim}{
      Number of simulated point patterns to be generated
      when computing the envelopes.
    }
    \item{nrank}{
      Integer. Rank of the envelope value amongst the \code{nsim} simulated
      values. A rank of 1 means that the minimum and maximum
      simulated values will be used.
    }
    \item{verbose}{
      Logical flag indicating whether to print progress reports
      during the simulations.
    }
    \item{\dots}{
      Extra arguments passed to \code{fun}.
    }
    \item{simulate}{
      Optional. An expression. If this is present, then the simulated
      point patterns will be generated by evaluating this expression
      \code{nsim} times.
    }
    \item{clipdata}{
      Logical flag indicating whether the data point pattern should be
      clipped to the same window as the simulated patterns,
      before the summary function for the data is computed.
      This should usually be \code{TRUE} to ensure that the
      data and simulations are properly comparable.
    }
    \item{start,control}{
      Optional. These specify the arguments \code{start} and \code{control}
      of \code{rmh}, giving complete control over the
      simulation algorithm.
    }
}
\value{
  An object of class \code{"fv"}, see \code{\link{fv.object}},
  which can be plotted directly using \code{\link{plot.fv}}.

  Essentially a data frame containing columns
  \item{r}{the vector of values of the argument \eqn{r} 
    at which the summary function \code{fun} has been  estimated
  }
  \item{obs}{
    values of the summary function for the data point pattern
  }
  \item{lo}{
    lower envelope of simulations
  }
  \item{hi}{
    upper envelope of simulations
  }
}
\details{
  Simulation envelopes can be used to assess the goodness-of-fit of
  a point process model to point pattern data. See the References.

  This function first generates \code{nsim} random point patterns
  in one of the following ways. 
  \itemize{
    \item 
    If \code{Y} is a point pattern (an object of class \code{"ppp"})
    and \code{simulate=NULL},
    then this routine generates \code{nsim} simulations of
    Complete Spatial Randomness (i.e. \code{nsim} simulated point patterns
    each being a realisation of the uniform Poisson point process)
    with the same intensity as the pattern \code{Y}.
    (If \code{Y} is a multitype point pattern, then the simulated patterns
    are also given independent random marks; the probability
    distribution of the random marks is determined by the
    relative frequencies of marks in \code{Y}.) 
    \item
    If \code{Y} is a fitted point process model (an object of class
    \code{"ppm"}) and \code{simulate=NULL},
    then this routine generates \code{nsim} simulated
    realisations of that model.
    \item
    If \code{simulate} is supplied, then it must be
    an expression. It will be evaluated \code{nsim} times to
    yield \code{nsim} point patterns.
  }
  
  The summary statistic \code{fun} is applied to each of these simulated
  patterns. Typically \code{fun} is one of the functions
  \code{Kest}, \code{Gest}, \code{Fest}, \code{Jest}, \code{pcf},
  \code{Kcross}, \code{Kdot}, \code{Gcross}, \code{Gdot},
  \code{Jcross}, \code{Jdot}, \code{Kmulti}, \code{Gmulti},
  \code{Jmulti} or \code{Kinhom}. It may also be a character string
  containing the name of one of these functions.

  The statistic \code{fun} can also be a user-supplied function;
  if so, then it must have arguments \code{X} and \code{r}
  like those in the functions listed above, and it must return an object
  of class \code{"fv"}.

  Upper and lower pointwise envelopes are computed pointwise (i.e.
  for each value of the distance argument \eqn{r}), by sorting the
  \code{nsim} simulated values, and taking the \code{m}-th lowest
  and \code{m}-th highest values, where \code{m = nrank}.
  For example if \code{nrank=1}, the upper and lower envelopes
  are the pointwise maximum and minimum of the simulated values.

  The significance level of the associated Monte Carlo test is
  \code{alpha = 2 * nrank/(1 + nsim)}. 
  
  The return value is an object of class \code{"fv"} containing
  the summary function for the data point pattern
  and the upper and lower simulation envelopes. It can be plotted
  using \code{plot.fv}.

  Arguments can be passed to the function \code{fun} through
  \code{...}. This makes it possible to select the edge correction
  used to calculate the summary statistic. See the Examples.

  If \code{Y} is a fitted point process model, and \code{simulate=NULL},
  then the model is simulated
  by running the Metropolis-Hastings algorithm \code{\link{rmh}}.
  Complete control over this algorithm is provided by the 
  arguments \code{start} and \code{control} which are passed
  to \code{\link{rmh}}.

  Selecting only a single edge
  correction will make the code run much faster.
}
\references{
  Cressie, N.A.C. \emph{Statistics for spatial data}.
    John Wiley and Sons, 1991.

  Diggle, P.J. \emph{Statistical analysis of spatial point patterns}.
  Arnold, 2003.

  Ripley, B.D. \emph{Statistical inference for spatial processes}.
  Cambridge University Press, 1988.

  Stoyan, D. and Stoyan, H. (1994)
  Fractals, random shapes and point fields:
  methods of geometrical statistics.
  John Wiley and Sons.
} 
\seealso{
  \code{\link{fv.object}},
  \code{\link{plot.fv}},
  \code{\link{Kest}},
  \code{\link{Gest}},
  \code{\link{Fest}},
  \code{\link{Jest}},
  \code{\link{pcf}},
  \code{\link{ppp}},
  \code{\link{ppm}}
}
\examples{
 data(simdat)
 X <- simdat

 # Envelope of K function under CSR
 \dontrun{
 plot(envelope(X))
 }
 \testonly{
  plot(envelope(X, nsim=4))
 }

 # Translation edge correction (this is also FASTER):
 \dontrun{
 plot(envelope(X, correction="translate"))
 }
 \testonly{
  plot(envelope(X, nsim=4, correction="translate"))
 }

 # Envelope of K function for simulations from model 
 data(cells)
 fit <- ppm(cells, ~1, Strauss(0.05))
 \dontrun{
 plot(envelope(fit))
 }
 \testonly{
  plot(envelope(fit, nsim=4))
 }

 # Envelope of G function under CSR
 \dontrun{
 plot(envelope(X, Gest))
 }
 \testonly{
  plot(envelope(X, Gest, nsim=4))
 }

 # Envelope of L function under CSR
 #  L(r) = sqrt(K(r)/pi)
 \dontrun{
  E <- envelope(X, Kest)
  plot(E, sqrt(./pi) ~ r)
 }
 \testonly{
  E <- envelope(X, Kest, nsim=4)
  plot(E, sqrt(./pi) ~ r)
 }

 # How to pass arguments needed to compute the summary functions:
 # We want envelopes for Jcross(X, "A", "B") 
 # where "A" and "B" are types of points in the dataset 'demopat'

 data(demopat)
 \dontrun{
 plot(envelope(demopat, Jcross, i="A", j="B"))
 }
 \testonly{
 plot(envelope(demopat, Jcross, i="A", j="B", nsim=4))
 }
 
 # Use of `simulate'
 \dontrun{
 plot(envelope(cells, Gest, simulate=expression(runifpoint(42))))
 plot(envelope(cells, Gest, simulate=expression(rMaternI(100,0.02))))
 }
 \testonly{
  plot(envelope(cells, Gest, simulate=expression(runifpoint(42)), nsim=4))
  plot(envelope(cells, Gest, simulate=expression(rMaternI(100, 0.02)), nsim=4))
 }

 # Envelope under random toroidal shifts
 data(amacrine)
 \dontrun{
 plot(envelope(amacrine, Kcross, i="on", j="off",
               simulate=expression(rshift(amacrine, radius=0.25)))) 
 }
 \testonly{
 plot(envelope(amacrine, Kcross, i="on", j="off",
               simulate=expression(rshift(amacrine, radius=0.25)),
               nsim=4))
 }

 # Envelope under random shifts with erosion
 \dontrun{
 plot(envelope(amacrine, Kcross, i="on", j="off",
              simulate=expression(rshift(amacrine, radius=0.1, edge="erode"))))
 }
 \testonly{
 plot(envelope(amacrine, Kcross, i="on", j="off",
               simulate=expression(rshift(amacrine, radius=0.1, edge="erode")),
               nsim=4))
 }

 
 # Envelope of INHOMOGENEOUS K-function with fitted trend
 trend <- density.ppp(X, 1.5)
 \dontrun{
 plot(envelope(X, Kinhom, lambda=trend,
         simulate=expression(rpoispp(trend))))
 }
 \testonly{
 plot(envelope(X, Kinhom, lambda=trend,
         simulate=expression(rpoispp(trend)), nsim=4))
 }
 
}
\author{Adrian Baddeley
  \email{adrian@maths.uwa.edu.au}
  \url{http://www.maths.uwa.edu.au/~adrian/}
  and Rolf Turner
  \email{rolf@math.unb.ca}
  \url{http://www.math.unb.ca/~rolf}
}
\keyword{spatial}
 
 
