\name{rLGCP}
\alias{rLGCP}
\title{Simulate Log-Gaussian Cox Process}
\description{
  Generate a random point pattern, a realisation of the
  log-Gaussian Cox process.
}
\usage{
 rLGCP(model="exp", mu = 0, param = NULL, ..., win=NULL, saveLambda=TRUE)
}
\arguments{
  \item{model}{
    character string: the short name of a covariance model for
    the Gaussian random field. After adding the prefix \code{"RM"},
    the code will search for a function of this name
    in the \pkg{RandomFields} package.
  }
  \item{mu}{
    mean function of the Gaussian random field. Either a
    single number, a \code{function(x,y, ...)} or a pixel
    image (object of class \code{"im"}).
  }
  \item{param}{
    List of parameters for the covariance.
    Standard arguments are \code{var} and \code{scale}.
  }
  \item{\dots}{
    Additional parameters for the covariance,
    or arguments passed to \code{\link{as.mask}} to determine
    the pixel resolution.
  }
  \item{win}{
    Window in which to simulate the pattern.
    An object of class \code{"owin"}.
  }
  \item{saveLambda}{
    Logical. If \code{TRUE} (the default) then the
    simulated random intensity will also be saved,
    and returns as an attribute of the point pattern.
  }
}
\value{
  The simulated point pattern (an object of class \code{"ppp"}).

  Additionally, the simulated intensity function is
  returned as an attribute \code{"Lambda"},
  if \code{saveLambda=TRUE}.
}
\details{
  This function generates a realisation of a log-Gaussian Cox
  process (LGCP). This is a Cox point process in which
  the logarithm of the random intensity is a Gaussian random
  field with mean function \eqn{\mu} and covariance function
  \eqn{c(r)}. Conditional on the random intensity, the point process
  is a Poisson process with this intensity.

  The string \code{model} specifies the covariance 
  function of the Gaussian random field, and the parameters
  of the covariance are determined by \code{param} and \code{\dots}.

  To determine the covariance model, the string \code{model}
  is prefixed by \code{"RM"}, and a function of this name is
  sought in the \pkg{RandomFields} package. 
  For a list of available models see 
  \code{\link[RandomFields]{RMmodel}} in the
  \pkg{RandomFields} package. For example the
  \ifelse{latex}{\out{Mat\'ern}}{Matern}
  covariance is specified by \code{model="matern"}, corresponding
  to the function \code{RMmatern} in the \pkg{RandomFields} package.

  Standard variance parameters (for all functions beginning with
  \code{"RM"} in the \pkg{RandomFields} package) are \code{var}
  for the variance at distance zero, and \code{scale} for the scale
  parameter. Other parameters are specified in the help files
  for the individual functions beginning with \code{"RM"}. For example
  the help file for \code{RMmatern} states that \code{nu} is a parameter
  for this model.

  This algorithm uses the function \code{\link[RandomFields]{RFsimulate}} in the
  \pkg{RandomFields} package to generate values of
  a Gaussian random field, with the specified mean function \code{mu}
  and the covariance specified by the arguments \code{model} and
  \code{param}, on the points of a regular grid. The exponential
  of this random field is taken as the intensity of a Poisson point
  process, and a realisation of the Poisson process is then generated by the 
  function \code{\link{rpoispp}} in the \pkg{spatstat} package.
  
  If the simulation window \code{win} is missing or \code{NULL},
  then it defaults to 
  \code{Window(mu)} if \code{mu} is a pixel image,
  and it defaults to the unit square otherwise.
  
  The LGCP model can be fitted to data using \code{\link{kppm}}.
}
\seealso{
\code{\link{rpoispp}},
\code{\link{rMatClust}},
\code{\link{rGaussPoisson}},
\code{\link{rNeymanScott}},
\code{\link{lgcp.estK}},
\code{\link{kppm}}
}
\references{
  \ifelse{latex}{\out{M\o ller}}{Moller}, J., Syversveen, A. and Waagepetersen, R. (1998)
  Log Gaussian Cox Processes.
  \emph{Scandinavian Journal of Statistics} \bold{25}, 451--482.
}
\examples{
  if(require(RandomFields)) {
    # homogeneous LGCP with exponential covariance function
    X <- rLGCP("exp", 3, var=0.2, scale=.1)

    # inhomogeneous LGCP with Gaussian covariance function
    m <- as.im(function(x, y){5 - 1.5 * (x - 0.5)^2 + 2 * (y - 0.5)^2}, W=owin())
    X <- rLGCP("gauss", m, var=0.15, scale =0.5)
    plot(attr(X, "Lambda"))
    points(X)

    # inhomogeneous LGCP with Matern covariance function
    X <- rLGCP("matern", function(x, y){ 1 - 0.4 * x},
               var=2, scale=0.7, nu=0.5,
               win = owin(c(0, 10), c(0, 10)))
    plot(X)
  } else message("Simulation requires the RandomFields package")
}
\author{Abdollah Jalilian and Rasmus Waagepetersen.
  Modified by Adrian Baddeley
  \email{Adrian.Baddeley@uwa.edu.au}
  \url{http://www.maths.uwa.edu.au/~adrian/}
  and Rolf Turner
  \email{r.turner@auckland.ac.nz}
}
\keyword{spatial}
\keyword{datagen}

