% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bootstrap-projection.R
\name{bootstrap_CIs}
\alias{bootstrap_CIs}
\title{Calculate one or multiple bootstrap confidence intervals}
\usage{
bootstrap_CIs(
  boot_est,
  boot_se = NULL,
  est = NULL,
  se = NULL,
  CI_type = "percentile",
  level = 0.95,
  B_vals = length(boot_est),
  reps = 1L,
  format = "wide",
  seed = NULL
)
}
\arguments{
\item{boot_est}{vector of bootstrap replications of an estimator.}

\item{boot_se}{vector of estimated standard errors from each bootstrap
replication.}

\item{est}{numeric value of the estimate based on the original sample.
Required for \code{CI_type = "normal"}, \code{CI_type = "basic"}, and
\code{CI_type = "student"}.}

\item{se}{numeric value of the estimated standard error based on the original
sample. Required for \code{CI_type = "student"}.}

\item{CI_type}{Character string or vector of character strings indicating
types of confidence intervals to calculate. Options are \code{"normal"},
\code{"basic"}, \code{"student"}, and \code{"percentile"} (the default).}

\item{level}{numeric value between 0 and 1 for the desired coverage level,
with a default of \code{0.95}.}

\item{B_vals}{vector of sub-sample sizes for which to calculate confidence
intervals. Setting \code{B_vals = length(boot_est)} (the default) will
return bootstrap confidence intervals calculated on the full set of
bootstrap replications. For \code{B_vals < length(boot_est)}, confidence
intervals will be calculated after sub-sampling (without replacement) the
bootstrap replications.}

\item{reps}{integer value for the number of sub-sample confidence intervals
to generate when \code{B_vals < length(boot_est)}, with a default of
\code{reps = 1}.}

\item{format}{character string controlling the format of the output. If
\code{format = "wide"} (the default), then different types of confidence
intervals will be returned in separate columns. If \code{format = "long"},
then confidence intervals of different types will appear on different rows
of dataset. If \code{format = "wide-list"}, then different types of
confidence intervals will be returned in separate columns and the result
will be wrapped in an unnamed list.}

\item{seed}{Single numeric value to which the random number generator seed
will be set. Default is \code{NULL}, which does not set a seed.}
}
\value{
If \code{format = "wide"}, the function returns a \code{data.frame}
  with \code{reps} rows per entry of \code{B_vals}, where each row contains
  confidence intervals for one sub-sample replication.

  If \code{format = "long"}, the function returns a \code{data.frame} with
  one row for each \code{CI_type}, each replication, and each entry of
  \code{B_vals}, where each row contains a single confidence interval for one
  sub-sample replication.

  If \code{format = "wide-list"}, then the output will be structured as in
  \code{format = "wide"} but will be wrapped in an unnamed list, which makes
  it easier to sore the output in a tibble, and will be assigned the class
  \code{"bootstrap_CIs"}.
}
\description{
Calculate one or multiple bootstrap confidence intervals, given
  a sample of bootstrap replications.
}
\details{
Confidence intervals are calculated following the methods described
  in Chapter 5 of Davison and Hinkley (1997). For basic non-parametric
  bootstraps, the methods are nearly identical to the implementation in
  \code{\link[boot]{boot.ci}} from the \code{boot} package.
}
\examples{
# generate t-distributed data
N <- 50
mu <- 2
nu <- 5
dat <- mu + rt(N, df = nu)

# create bootstrap replications
f <- \(x) {
 c(
   M = mean(x, trim = 0.1),
   SE = sd(x) / sqrt(length(x))
 )
}

booties <- replicate(399, {
  sample(dat, replace = TRUE, size = N) |>
  f()
})

res <- f(dat)

# calculate bootstrap CIs from full set of bootstrap replicates
bootstrap_CIs(
  boot_est = booties[1,],
  boot_se = booties[2,],
  est = res[1],
  se = res[2],
  CI_type = c("normal","basic","student","percentile"),
  format = "long"
)

# calculate multiple bootstrap CIs using sub-sampling of replicates
bootstrap_CIs(
  boot_est = booties[1,],
  boot_se = booties[2,],
  est = res[1],
  se = res[2],
  CI_type = c("normal","basic","student","percentile"),
  B_vals = 199,
  reps = 4L,
  format = "long"
)

# calculate multiple bootstrap CIs using sub-sampling of replicates,
# for each of several sub-sample sizes.
bootstrap_CIs(
  boot_est = booties[1,],
  boot_se = booties[2,],
  est = res[1],
  se = res[2],
  CI_type = c("normal","basic","student","percentile"),
  B_vals = c(49,99,199),
  reps = 4L,
  format = "long"
)

}
\references{
Davison, A.C. and Hinkley, D.V. (1997). _Bootstrap Methods and
  Their Application_, Chapter 5. Cambridge University Press.
}
