% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/main_function.R
\name{simPATHy}
\alias{simPATHy}
\title{Simulate data from a graphical model}
\usage{
simPATHy(
  graph,
  path = NULL,
  S = NULL,
  min = 2,
  max = 3,
  prob = 1,
  n1 = 500,
  n2 = n1,
  digits = 5,
  mu1 = 0,
  mu2 = mu1,
  muRandom = FALSE
)
}
\arguments{
\item{graph}{A graphNEL object.}

\item{path}{A list of edges in edgesList format (see \pkg{gRbase}).}

\item{S}{The sample covariance matrix.}

\item{min, max}{Vectors of length 1 or of the same length as \code{path} containing the lower and upper limits of a uniform distribution.
The strength of dysregulation is sampled uniformly from the interval [\code{min}, \code{max}]: a value smaller than 1 represents deactivation, a value greater than 1 represents activation. If \code{path=NULL} only the first element is used.}

\item{prob}{A vector of size 1 or of the same length as path, giving the probability to change the sign of the correlation coefficient for each edge.
\code{prob=0} implying that the sign of the dysregulation should be changed, and \code{prob=1} implying that the sign should be left unaltered (default). Values between these two extremes allow for random sign switch: the sign is changed with probability \code{1-prob}.}

\item{n1, n2}{Number of observations to generate from the two conditions.}

\item{digits}{Integer indicating the number of decimal places to be used.}

\item{mu1, mu2}{A vector of size 1 or of the length equal to the number of nodes in the graph. Means of the multivariate normal distributions from which observations are generated. If \code{mu1} (and/or \code{mu2}) is a vector it has to be named in accordance with the names of the nodes of the graph.}

\item{muRandom}{Logical. If \code{muRandom=TRUE} the means of the variables are randomly generated.}
}
\value{
It returns a list containing:
\itemize{
 \item{\code{data}}       {random samples generated from multivariate normal distributions with covariance matrices \code{S1} (reference condition) and \code{S2} (dysregulated condition);}
 \item{\code{S1,S2}}      {two covariance matrices;}
 \item{\code{path}}       {the dysregulated path;}
 \item{\code{strength}}   {the dysregulation strength for each edge in the path;}
 \item{\code{mu1,mu2}}    {two mean vectors;}
 \item{\code{correction}} {correction details.}
}
}
\description{
Simulate data in two different conditions with a common structure of dependences. The two different conditions are characterized by different strengths of the links between nodes (dysregulation).
}
\details{
If the matrix \code{S} does not reflect conditional independence constraints imposed by the graph \code{simPATHy} uses the maximum likelihood estimation of covariance matrices for graphical models via internal function \code{\link[simPATHy]{fitSgraph}}.\cr \cr
When the dysregulation of the initial (reference condition) covariance matrix leads to a matrix that is no longer positive definite, the resulting matrix is corrected via internal function \code{\link[simPATHy]{makePositiveDefinite}}.\cr \cr
To avoid excessively strong dysregulations, the upper limit for the absolute value of the dysregulated correlation coefficient is set to: \deqn{min( 0.9, 1.25*max(abs(C[upper.tri(C)])) )} where C is the correlation matrix of the reference condition.
}
\examples{
if(require(gRbase) & require(graph)){

  ## Directed graph
  ## sub-graph Acute Myel... Leukemia
  graph<-gRbase::dag(~867:25+867:613+5295:867+5294:867+
                       + 207:5295+207:5294+4193:207+3551:207+
                       + 4792:3551+7157:4193+3265:6654+
                       + 3845:6654+6654:2885+2885:25+2885:613)
  genes<-graph::nodes(graph)

  # covariance matrix of the reference condition
  data<-t(chimera[genes,colnames(chimera)==1])
  S<-cov(data)
  S<-fitSgraph(graph,S)

  # select path to dysregulate
  path<-list(c("613","867"),c("867","5295"),c("5295","207"),
             c("207","4193"),c("4193","7157"))
  ## ..or select the path in an interactive plot
  # path<-getPathShiny(graph)

  # select parameters of the dysregulation
  min<-c(2,8,2,0.1,0.5)
  max<-c(2,10,2,4,0.5)
  prob<-c(1,0,0,0.5,1)

  # activation, switch, switch, random, deactivation
  dys<-cbind(min,max,prob)
  rownames(dys)<-sapply(path,paste,collapse = "~")
  dys

  set.seed(123)
  # main function
  Result<-simPATHy(graph,path,S,min,max,prob)
  class(Result)
  names(Result)

  # simulated data from two conditions
  round(Result$dataset[c(1:3,501:503),1:5],3)

  # Summary
  easyLookDys(Result)
  # ..or interactive summary
  # easyLookShiny(resObj=Result,graph=graph)


  # Visualization
  plotCorGraph(S1=Result$S1,S2 = Result$S2,graph = graph,path = path,colLim = c(-0.3,0.3))
  plotGraphNELD3(S1=Result$S1,S2 = Result$S2,graph = graph,colLim = c(-0.3,0.3))
  rm(list=ls())


  ## Undirected graph
  graph <- gRbase::ug(~a:b, ~a:c, ~c:d, ~b:d, ~b:c)
  # when reference condition covariance matrix is not supplied simPATHy generate a random one
  Result_ug<-simPATHy(graph)
  easyLookDys(Result_ug)
  plotGraphNELD3(S1=Result_ug$S1,S2 = Result_ug$S2,graph = graph,colLim = c(-0.5,0.5))

}
}
\seealso{
\code{\link[simPATHy]{easyLookDys}}, \code{\link[simPATHy]{easyLookShiny}}, \code{\link[simPATHy]{plotCorGraph}},  \code{\link[simPATHy]{plotGraphNELD3}}
}
