#=====================================================================
#=====================================================================

#' @title Fortran wrapper for pedigree reconstruction
#'
#' @description Call main Fortran part of sequoia, and convert its output to a
#'   list with dataframes.
#'
#' @param ParSib either "par" to call parentage assignment, or "sib" to call the
#'   rest of the algorithm.
#' @param Specs a named vector with parameter values, as generated by
#'   \code{\link{SeqPrep}}.
#' @param GenoM matrix with genotype data, size nInd x nSnp
#' @param LhIN  life history data: ID - sex - birth year
#' @param AgePriors matrix with agepriors, size Specs["nAgeClasses"] by 8.
#' @param Parents  matrix with rownumbers of assigned parents, size nInd by 2
#' @param quiet suppress messages
#'
#' @return A list with
#' \item{PedigreePar or Pedigree}{the pedigree}
#' \item{MaybeParent or MaybeRel}{Non-assigned likely relatives}
#' \item{MaybeTrio}{Non-assigned parent-parent-offspring trios}
#' \item{DummyIDs}{Info on dummies}
#' \item{TotLikParents or TotLikSib}{Total log-likelihood per iteration}
#'
#' For a detailed description of the output see \code{\link{sequoia}}
#'
#' @useDynLib sequoia, .registration = TRUE
# @useDynLib sequoia makeped deallocall
#'
#' @importFrom plyr dlply
#'
#' @keywords internal

SeqParSib <- function(ParSib = "par",
                      Specs = NULL,
                      GenoM = NULL,
                      LhIN = NULL,
                      AgePriors = NULL,
                      Parents = NULL,
                      quiet = FALSE)
{
  on.exit(.Fortran("deallocall")) #, PACKAGE = "sequoia"))

  Ng <- FacToNum(Specs[,"NumberIndivGenotyped"])
  SMax <- FacToNum(Specs[,"MaxSibshipSize"])
  gID <- rownames(GenoM)
  GenoV <- as.integer(GenoM)
  LHL <- orderLH(LhIN[LhIN$Sex %in% c(1:3), ], gID)
  DumPfx <- Specs[,c("DummyPrefixFemale", "DummyPrefixMale")]
  if (!is.null(Parents)) {
    PedPar <- IDToNum(Parents, gID, DumPrefix=paste0(DumPfx,"0"))[, 2:3]
    PedPar <- c(as.matrix(PedPar))
    if (length(PedPar) != Ng*2) stop("'PedPar' wrong length")
  } else {
    PedPar <- rep(0, Ng*2)
  }
  Complex <- switch(Specs[,"Complexity"], full = 2, simp = 1, mono = 0, herm = 4)
  UAge <- switch(Specs[,"UseAge"], extra = 2, yes = 1, no = 0)
  PrSb <- switch(ParSib, dup = 0, par = 1, sib = 2)

  SpecsInt <- c(ParSib = as.integer(PrSb),     						   # 1
                MaxSibIter = FacToNum(Specs[,"MaxSibIter"]), # 2
                nSnp = FacToNum(Specs[,"NumberSnps"]),       # 3
                MaxMis = FacToNum(Specs[,"MaxMismatch"]),    # 4
                SMax = SMax,                    					   # 5
                nAgeCl = FacToNum(Specs[,"nAgeClasses"]),    # 6
                Complx = as.integer(Complex),                # 7
                FindMaybe = as.integer(0), 									 # 8
                CalcLLR = as.integer(as.logical(Specs[,"CalcLLR"])),  # 9
                quiet = as.integer(quiet),        # 10
                nAmbMax = as.integer(0),    # 11
                UseAge = as.integer(UAge))        # 12
  SpecsDbl <- c(Er = FacToNum(Specs[,"GenotypingErrorRate"]),
                TF = FacToNum(Specs[,"Tfilter"]),
                TA = FacToNum(Specs[,"Tassign"]))

  if (is.null(AgePriors) & (UAge==0 | ParSib=="dup")) {
    AP <- matrix(1, Specs[,"nAgeClasses"], ncol=9)
  } else if (length(as.double(AgePriors)) != 9*FacToNum(Specs[,"nAgeClasses"])) {
    stop("'AgePriors' matrix should have size nAgeClasses * 9")
  } else {  # re-arrange columns (for historical reasons...)
    AP <- AgePriors[, c("MS", "PS", "MGM", "PGF", "MGF", "UA", "M", "P", "FS")]
  }

  TMP <- .Fortran("makeped",
                  Ng = as.integer(Ng),
                  SpecsInt = as.integer(SpecsInt),
                  SpecsDbl = as.double(SpecsDbl),
                  GenoV = as.integer(GenoV),
                  Sex = as.integer(LHL$Sex),
                  BY = as.integer(LHL$BY),
                  AP = as.double(AP),
									
                  parentsRF = as.integer(PedPar),
                  LrRF = double(3*Ng),
                  OhRF = integer(3*Ng),
                  Nd = integer(2),
                  DumParRF = integer(2*Ng),
                  DumLrRF = double(3*Ng),
                  DumBYRF = integer(3*Ng),
                  TotLik = double(42),

									nDupGenos = as.integer(0),
                  DupGenosFR = integer(2*Ng),
                  CntMism = integer(Ng),
                  DupLR = double(Ng))
#                  PACKAGE = "sequoia")

  TMP$LrRF[abs(TMP$LrRF - 999) < 0.1] <- NA
  TMP$DumLrRF[abs(TMP$DumLrRF - 999) < 0.1] <- NA
  TMP$LrRF <- round(TMP$LrRF, 2)
  TMP$DumLrRF <- round(TMP$DumLrRF, 2)
  TMP$OhRF[TMP$OhRF < 0] <- NA

	#=========================
	# duplicates
	if (PrSb==0) {
		Dup <- WrapDup(TMP, gID, LhIN)
		return( Dup )
	}

  #=========================
  # pedigree

	dID <- cbind(paste0(DumPfx[1], formatC(1:min(Ng,9999), width=4, flag=0)),
							 paste0(DumPfx[2], formatC(1:min(Ng,9999), width=4, flag=0)))

  Pedigree <- data.frame(id = gID,
                         VtoM(TMP$parentsRF),
                         VtoM(TMP$LrRF, nc=3),
												 VtoM(TMP$OhRF, nc=3),
                         stringsAsFactors=FALSE)
  names(Pedigree) <- c("id", "dam", "sire", "LLRdam", "LLRsire", "LLRpair",
	"OHdam", "OHsire", "MEpair")
  for (k in 1:2) Pedigree[, k+1] <- NumToID(Pedigree[, k+1], k, gID, dID)

  if (any(LhIN$Sex==4)) {  # hermaphrodites
    Pedigree <- herm_unclone_Ped(Pedigree, LH=LhIN, herm.suf=c("f", "m"))
  }

  if (quiet<1) {
    if (grepl("par", ParSib)) {
      message("assigned ", sum(!is.na(Pedigree$dam)), " dams and ",
           sum(!is.na(Pedigree$sire)), " sires to ", nrow(Pedigree), " individuals")
    } else {
     message("assigned ", sum(!is.na(Pedigree$dam)), " dams and ",
           sum(!is.na(Pedigree$sire)), " sires to ", Ng, " + ", sum(TMP$Nd),
           " individuals (real + dummy)")
    }
  }

  #=========================
  # dummies
  if (grepl("sib", ParSib) && any(TMP$Nd>0)) {
    nd <- TMP$Nd
    NgOdd <- Ng%%2==1
    for (k in 1:2) if (nd[k]==0)  nd[k] <- 1  # easier; remove below
		DumPfx <- paste0(DumPfx, "0")
		DPnc <- nchar(DumPfx)[1]
    NumOff <- with(Pedigree, list("mat" = table(dam[which(substr(dam,1,DPnc) == DumPfx[1])]),
                                  "pat" = table(sire[substr(sire,1,DPnc) == DumPfx[2]])))
    MaxOff <- max(unlist(NumOff))
    OffIDs <- c(dlply(Pedigree, "dam", function(df) df$id),
                dlply(Pedigree, "sire", function(df) df$id))
    OffIDs <- OffIDs[c(names(NumOff[[1]]), names(NumOff[[2]]))]

    DummyIDs <- data.frame(id=c(dID[1:nd[1], 1], dID[1:nd[2], 2]),
                           VtoM(TMP$DumParRF, sum(nd), 2, NgOdd),
                           VtoM(TMP$DumLrRF, sum(nd), 3, NgOdd),
                           sex=rep(1:2, nd),
                           VtoM(TMP$DumBYRF, sum(nd),3, NgOdd),
                           unlist(NumOff),
                           stringsAsFactors=FALSE)
    names(DummyIDs) <-  c("id", "dam", "sire", "LLRdam", "LLRsire", "LLRpair",
                          "sex", "BY.est", "BY.min", "BY.max", "NumOff")
    DummyIDs <- cbind(DummyIDs,
                      setNames(t(sapply(OffIDs, "[", i=1:MaxOff)),
                               paste0("O", 1:MaxOff)))
    if (TMP$Nd[1]==0) DummyIDs <- DummyIDs[-1, ]
    if (TMP$Nd[2]==0) DummyIDs <- DummyIDs[-nrow(DummyIDs), ]
    for (k in 1:2) DummyIDs[, k+1] <- NumToID(DummyIDs[, k+1], k, gID, dID)

    Pedigree <- rbind(Pedigree,
											cbind(DummyIDs[, 1:6],
														OHdam = NA, OHsire = NA, MEpair = NA) )
  } else  DummyIDs <- NULL


	#=========================
	# non-assigned putative relatives
  if (Specs[,"FindMaybeRel"]) {
    MaybeRelPairTrio <- GetMaybeRel(Ped = Pedigree,
                                     GenoM = GenoM,
                                     SeqList = list(Specs = Specs,
                                                    LifeHist = LhIN,
                                                    AgePriors = AgePriors),
                                     ParSib = ParSib,
                                     quiet = quiet)
  } else   MaybeRelPairTrio <- NULL


  #=========================
  # output
  rownames(Pedigree) <- 1:nrow(Pedigree)

  if (grepl("par", ParSib)) {
    OUT <- c(list(PedigreePar = Pedigree,
                  TotLikParents = TMP$TotLik[1:sum(TMP$TotLik!=0)]),
             MaybeRelPairTrio)

  } else if (grepl("sib", ParSib)) {
    OUT <- c(list(Pedigree = Pedigree,
                  DummyIDs = DummyIDs,
                  TotLikSib = TMP$TotLik[1:sum(TMP$TotLik!=0)]),
             MaybeRelPairTrio)
  }
  return(OUT[!sapply(OUT, is.null)])
}



#=====================================================================
#=====================================================================

#' @title Find putative relatives
#'
#' @description Identify pairs of individuals likely to be related, but not
#'   assigned as such in the provided pedigree.
#'
#' @param GenoM matrix with genotype data, size nInd x nSnp
#' @param SeqList list with output from \code{\link{sequoia}}. If provided, the
#' elements `Specs', `AgePriors' and 'LifeHist' are used, and all other input
#' parameters except 'GenoM', 'ParSib' and 'quiet' are ignored.
#' @param Ped dataframe with pedigree, with id - dam - sire in columns 1-3
#' @param LifeHistData dataframe with columns id - sex (1=female, 2=male,
#'   3=unknown) - birth year
#' @param ParSib either 'par' to check for putative parent-offspring pairs only,
#'   or 'sib' to check for all types of first and second degree relatives. When
#'   'par', all pairs are returned that are more likely parent-offspring than
#'   unrelated, including pairs that are even more likely to be otherwise
#'   related.
#' @param Complex either "full" (default), "simp" (simplified, no explicit
#'   consideration of inbred relationships), "mono" (monogamous) or "herm"
#'   (hermaphrodites, otherwise like "full").
#' @param Err estimated genotyping error rate. The error model aims to deal with
#'   scoring errors typical for SNP arrays.
#' @param MaxMismatch maximum number of loci at which candidate parent and
#'   offspring are allowed to be opposite homozygotes. Setting a more liberal
#'   threshold can improve performance if the error rate is high, at the cost of
#'   decreased speed.
#' @param Tassign minimum LLR required for acceptance of proposed relationship,
#'   relative to next most likely relationship. Higher values result in more
#'   conservative assignments. Must be zero or positive.
#' @param MaxPairs  The maximum number of putative pairs to return.
#' @param DumPrefix  character vector of length 2 with prefixes for dummy dams
#'   (mothers) and sires (fathers) used in \code{Ped}.
#' @param quiet suppress messages
#'
#' @return A list with
#'   \item{MaybeParent or MaybeRel}{Non-assigned likely relatives}
#'   \item{MaybeTrio}{Non-assigned parent-parent-offspring trios}
#'
#' @useDynLib sequoia, .registration = TRUE
#'
#' @examples
#' \dontrun{
#' data(SimGeno_example, LH_HSg5, package="sequoia")
#' SeqOUT <- sequoia(GenoM = SimGeno_example,
#'                   LifeHistData = LH_HSg5, MaxSibIter = 0)
#' MaybePO <- GetMaybeRel(GenoM = SimGeno_example,
#'                       SeqList = SeqOUT)
#'
#' # age differences limit which relationships are considered:
#' MaybePO3 <- GetMaybeRel(GenoM = SimGeno_example,
#'                       Ped = SeqOUT$PedigreePar)
#' }
#' @export

GetMaybeRel <- function(GenoM = NULL,
                        SeqList = NULL,
                        Ped = NULL,
                        LifeHistData = NULL,
                        ParSib = "par",
                        Complex = "full",
                        Err = 0.0001,
                        MaxMismatch = 3,
                        Tassign = 0.5,
                        MaxPairs =  7*nrow(GenoM),
                        DumPrefix = c("F0", "M0"),
                        quiet = FALSE)
{
  on.exit(.Fortran("deallocall"))

  # input check
	Excl <- CheckGeno(GenoM)
	if (ParSib == "full") ParSib <- "sib"
  if (!ParSib %in% c("par", "sib"))  stop("Invalid value for 'ParSib', choose 'par' or 'sib'")
	if (nchar(DumPrefix[1]) != nchar(DumPrefix[2])) stop("DumPrefix must have same number of characters")

	gID <- rownames(GenoM)
  GenoV <- as.integer(GenoM)

  if ("Pedigree" %in% names(SeqList)) {
    Ped <- SeqList$Pedigree
  } else if ("PedigreePar" %in% names(SeqList)) {
    Ped <- SeqList$PedigreePar
  }
  if (!is.null(Ped)) {
    names(Ped)[1:3] <- c("id","dam","sire")
    if(!all(gID %in% Ped$id))  stop("Not all genotyped individuals in Ped or SeqList pedigree")
    DPnc <- nchar(DumPrefix)[1]
    PedNum <- IDToNum(Ped[,1:3], gID, DumPrefix)
    PedPar <- as.matrix(PedNum[gID, 2:3])   # not dummy indivs
    PedPar[is.na(PedPar)] <- 0   # in case not all genotyped indivs in Ped
  } else {
    PedPar <- rep(0, 2*nrow(GenoM))
  }

  PrSb <- switch(ParSib, dup=0, par = 1, sib = 2)
  nAmbMax <- MaxPairs  # max no. of non-assigned relative pairs to return

  if ("Specs" %in% names(SeqList)) {
    Specs <- SeqList$Specs
    LhIN <- SeqList$LifeHist
    Ng <- FacToNum(Specs[,"NumberIndivGenotyped"])
		SMax <- FacToNum(Specs[,"MaxSibshipSize"])

    Cmplx <- switch(Specs[,"Complexity"], full = 2, simp = 1, mono = 0, herm = 4)
    UAge <- switch(Specs[,"UseAge"], extra = 2, yes = 1, no = 0)
		AP <- SeqList$AgePriors[, c("MS", "PS", "MGM", "PGF", "MGF", "UA", "M", "P", "FS")]
    SpecsInt <- c(ParSib = as.integer(PrSb),        		  	 # 1
									MaxSibIter = as.integer(-1),							 # 2
                  nSnp = FacToNum(Specs[,"NumberSnps"]),     # 3
                  MaxMis = FacToNum(Specs[,"MaxMismatch"]),  # 4
									SMax = SMax,                               # 5
                  nAgeCl = FacToNum(Specs[,"nAgeClasses"]),  # 6
                  Complx = as.integer(Cmplx),                # 7
									FindMaybe = as.integer(1),        # 8
									CalcLLR = as.integer(0),          # 9
                  quiet = as.integer(quiet),        # 10
									nAmbMax = as.integer(nAmbMax),    # 11
                  UseAge = as.integer(UAge))        # 12
    SpecsDbl <- c(Er = FacToNum(Specs[,"GenotypingErrorRate"]),
                  TF = FacToNum(Specs[,"Tfilter"]),
                  TA = FacToNum(Specs[,"Tassign"]))

  } else {
    Ng <- nrow(GenoM)
    LhIN <- LifeHistData
    if (!is.null(LhIN))   names(LhIN) <- c("ID", "Sex", "BY")
		if (is.null(Ped)) {
			SMax <- 100
		} else {
			SMax <- max(table(Ped$dam), table(Ped$sire)) +1
		}
    Cmplx <- switch(Complex, full = 2, simp = 1, mono = 0, herm = 4)
    if (!is.null(Ped) & !is.null(LifeHistData)) {
      AP <- MakeAgePrior(Ped, LifeHistData, Plot=FALSE, quiet=TRUE)
      UseAge <- 1
    } else {
		  AP <- matrix(1, nrow=3, ncol=9)
		  UseAge <- 0
    }
    SpecsInt <- c(ParSib = as.integer(PrSb),            # 1
									MaxSibIter = as.integer(-1),				  # 2
                  nSnp = as.integer(ncol(GenoM)),       # 3
                  MaxMis = as.integer(MaxMismatch),     # 4
									SMax = SMax,                          # 5
                  nAgeCl = as.integer(nrow(AP)),        # 6
                  Complx = as.integer(Cmplx),           # 7
									FindMaybe = as.integer(1),            # 8
									CalcLLR = as.integer(0),          		# 9
                  quiet = as.integer(quiet),      		  # 10
									nAmbMax = as.integer(nAmbMax),        # 11
                  UseAge = as.integer(UseAge))       				# 12
    SpecsDbl <- c(Er = as.double(Err),
                  TF = as.double(2.0),   # not used
                  TA = as.double(Tassign))
  }

	# Dummies
  Nd <- 0
  DumParRF <- rep(0, 2*Ng)
  dID <- NULL
  if (!is.null(Ped)) {
    Nd <- c(sum(substr(Ped$id,1,DPnc)==DumPrefix[1]),
            sum(substr(Ped$id,1,DPnc)==DumPrefix[2]))
    if (max(Nd)>0) {
      SibshipGPs <- array(0, dim=c(2,max(Nd),2),
                          dimnames=list(c("grandma", "granddad"), 1:max(Nd), c("mat", "pat")))
      for (k in 1:2) {
        SibshipGPs[,1:Nd[k],k] <- t(as.matrix(PedNum[substr(Ped$id,1,DPnc)==DumPrefix[k], 2:3]))
      }
      for (k in 1:2) {
        for (s in 1:Nd[k]) {
          for (g in 1:2) {
            x <- (k-1)*2*Ng/2 + (g-1)*Nd[1] + s
            DumParRF[x] <- SibshipGPs[g,s,k]
          }
        }
      }
      dID <- c(Ped$id[substr(Ped$id,1,DPnc)==DumPrefix[1]],
               Ped$id[substr(Ped$id,1,DPnc)==DumPrefix[2]])
    }
  }

  LHL <- orderLH(LhIN[LhIN$Sex %in% c(1:3), ], gID)

  #=========================
	# call fortran
  TMP <- .Fortran("FindAmbig",
                  Ng = as.integer(Ng),
                  SpecsInt = as.integer(SpecsInt),
                  SpecsDbl = as.double(SpecsDbl),
                  GenoV = as.integer(GenoV),
									Sex = as.integer(LHL$Sex),
                  BY = as.integer(LHL$BY),
									AP = as.double(AP),
									ParentsRF = as.integer(PedPar),
#									Nd = as.integer(Nd),   
                  DumParRF = as.integer(DumParRF),

									nAmb = as.integer(0),
                  AmbigID = integer(2*nAmbMax),
                  AmbigRel = integer(2*nAmbMax),
                  AmbigLR = double(2*nAmbMax),
                  AmbigOH = integer(nAmbMax),

									nTrio = as.integer(0),
                  trioID = integer(3*Ng),
                  trioLR = double(3*Ng),
                  trioOH = integer(3*Ng))

	TMP$AmbigLR[abs(TMP$AmbigLR - 999) < 0.1] <- NA
  TMP$AmbigLR <- round(TMP$AmbigLR, 2)
  TMP$AmbigOH[TMP$AmbigOH < 0] <- NA
  TMP$trioLR[abs(TMP$trioLR - 999) < 0.1] <- NA
  TMP$trioLR <- round(TMP$trioLR, 2)
  TMP$trioOH[TMP$trioOH < 0] <- NA

  #=========================
  # non-assigned probable relative pairs

	# TODO: look up sex & agedif if LH provided.

  if (TMP$nAmb > 0) {
    RelName <- c("PO", "FS", "HS", "GP", "FA", "HA", "U ", "XX", "2nd")
    Na <- TMP$nAmb
    TMP$AmbigID <- NumToID(TMP$AmbigID, 0, gID, dID)
    AmbigRel <- factor(TMP$AmbigRel, levels=1:9, labels=RelName)

    MaybeRel <- data.frame(VtoM(TMP$AmbigID, Na),
                           VtoM(AmbigRel, Na),
                           VtoM(TMP$AmbigLR, Na),
                           stringsAsFactors=FALSE)
    names(MaybeRel) <- c("ID1", "ID2", "Relx", "TopRel", "LLR_Rx_U", "LLR")
    MaybeRel <- MaybeRel[,-which(names(MaybeRel) %in% c("Relx", "LLR_Rx_U"))]  # drop; confusing.
    MaybeRel$OH <-  TMP$AmbigOH[1:Na]

    if (!is.null(LhIN) & nrow(MaybeRel)>0) {
      LhIN$BY[LhIN$BY<0] <- NA
      MaybeRel <- merge(MaybeRel, setNames(LhIN, c("ID1","Sex1","BY1")), all.x=TRUE)
      MaybeRel <- merge(MaybeRel, setNames(LhIN, c("ID2","Sex2","BY2")), all.x=TRUE)
      MaybeRel$AgeDif <- with(MaybeRel, BY1 - BY2)
      MaybeRel <- MaybeRel[, c("ID1", "ID2", "TopRel", "LLR", "OH",
                               "BY1", "BY2", "AgeDif", "Sex1", "Sex2")]
      for (i in 1:Na) {
        if (is.na(MaybeRel$AgeDif[i]))  next
        if (MaybeRel$AgeDif[i] < 0) {
          tmp <- MaybeRel[i,]
          tmp$AgeDif <- abs(tmp$AgeDif)
          MaybeRel[i,] <- tmp[,c("ID2","ID1","TopRel", "LLR", "OH",
                                 "BY2", "BY1","AgeDif", "Sex2", "Sex1")]
        }
      }
    }

    if (grepl("sib", ParSib)) {
      MaybeRel <- with(MaybeRel, MaybeRel[TopRel %in% c("PO", "FS", "HS","GG") |
                                          LLR > FacToNum(Specs[,"Tassign"]), ])
    }
    MaybeRel <- MaybeRel[order(ordered(MaybeRel$TopRel, levels=RelName),
                                     -MaybeRel$LLR),]

    if (any(LhIN$Sex==4)) {  # hermaphrodites
      MaybeRel <- herm_unclone_MaybeRel(MaybeRel, Ped[,1:3], LH=LhIN, herm.suf=c("f", "m"))
    }

    if (quiet<1 && nrow(MaybeRel)>0) {
      if (grepl("par", ParSib)) {
        message("there are  ", sum(MaybeRel$TopRel=="PO"),
             "  likely parent-offspring pairs and ", nrow(MaybeRel)-sum(MaybeRel$TopRel=="PO"),
             " other pairs of likely relatives  which are not assigned, ",
             "perhaps due to unknown birth year(s), please see 'MaybeParent'")
      } else {
        message("there are  ", nrow(MaybeRel), "  non-assigned pairs of possible relatives, ",
               "including  ", sum(MaybeRel$TopRel=="PO"), "  likely parent-offspring pairs; ",
               "please see 'MaybeRel'")
      }
    }
    if (nrow(MaybeRel)==0) {
      MaybeRel <- NULL
    } else {
      rownames(MaybeRel) <- 1:nrow(MaybeRel)
    }
  } else  MaybeRel <- NULL


  #=========================
  # non-assigned parent-parent-offspring trios
  if (TMP$nTrio>0) {
    trios <- data.frame(VtoM(TMP$trioID, nr=TMP$nTrio, nc=3),
                         VtoM(TMP$trioLR, nr=TMP$nTrio, nc=3),
                        VtoM(TMP$trioOH, nr=TMP$nTrio, nc=3),
                         stringsAsFactors=FALSE)
    names(trios) <- c("id", "parent1", "parent2", "LLRparent1", "LLRparent2", "LLRpair",
                      "OHparent1", "OHparent2", "MEpair")
    for (k in 1:3) trios[, k] <- NumToID(trios[, k], k-1, gID, dID)

    if (any(LhIN$Sex==4)) {  # hermaphrodites
      trios <- herm_unclone_Trios(trios, LH=LhIN, herm.suf=c("f", "m"))
    }

    if (quiet<1) {
      message("found ", nrow(trios), " parent-parent-offspring trios with parents of unknown sex")
    }
  } else  trios <- NULL

  if (grepl("par", ParSib)) {
    return( list(MaybePar = MaybeRel,
               MaybeTrio = trios) )
  } else {
    return( list(MaybeRel = MaybeRel,
               MaybeTrio = trios) )
  }
}


#============================================================================
#============================================================================

#' @title Check data for duplicates.
#'
#' @description Check the genotype and life history data for duplicate IDs (not
#'   permitted) and duplicated genotypes (not advised), and count how many
#'   individuals in the genotype data are not included in the life history data
#'   (permitted). The order of IDs in the genotype and life history data is not
#'   required to be identical.
#'
#' @param Specs The 1-row dataframe with parameter values
#' @param GenoM matrix with genotype data, size nInd x nSnp
#' @param LhIN  life history data
#' @param quiet suppress messages
#'
#' @return A list with one or more of the following elements:
#' \item{DupGenoID}{Dataframe, rownumbers of duplicated IDs in genotype data.
#'   Please do remove or relabel these to avoid downstream confusion.}
#' \item{DupGenotype}{Dataframe, duplicated genotypes (with or without
#'   identical IDs). The specified number of maximum mismatches is allowed,
#'   and this dataframe may include pairs of closely related individuals.
#'   Mismatch = number of SNPs at which genotypes differ, LLR = likelihood
#'   ratio between 'self' and most likely non-self.}
#' \item{DupLifeHistID}{Dataframe, rownumbers of duplicated IDs in life
#'   history data}
#' \item{NoLH}{Vector, IDs (in genotype data) for which no life history data is
#' provided}
#'
#' @useDynLib sequoia, .registration = TRUE
# @useDynLib sequoia duplicates
#'
#' @keywords internal

WrapDup <- function(FortList, gID, LhIN, quiet = FALSE)
{
  DUP <- FortList

  Duplicates <- list()
  if (any(duplicated(gID))) {
    r1 <- which(duplicated(gID))
    r2 <- which(duplicated(gID, fromLast=TRUE))
    Duplicates$DupGenoID <- data.frame(row1 = r1,
                                           row2 = r2,
                                           ID = gID[r1])
  }
  if(DUP$nDupGenos>0) {
    tmp <- VtoM(DUP$DupGenosFR, DUP$nDupGenos)
    Duplicates$DupGenotype <- data.frame(row1 = tmp[, 1],
                                       row2 = tmp[, 2],
                                       ID1 = gID[tmp[, 1]],
                                       ID2 = gID[tmp[, 2]],
                                       Mismatch = DUP$CntMism[1:DUP$nDupGenos],
                                       LLR = DUP$DupLR[1:DUP$nDupGenos])
  }
  if (!is.null(LhIN)) {
    names(LhIN) <- c("ID", "Sex", "BY")
    LhIN$ID <- as.character(LhIN$ID)
    if (any(duplicated(LhIN[,1]))) {
      r1 <- which(duplicated(LhIN[,1]))
      r2 <- which(duplicated(LhIN[,1], fromLast=TRUE))
      Duplicates$DupLifeHistID <- data.frame(row1 = r1,
                                  row2 = r2,
                                  ID = LhIN[r1, "ID"],
                                  Sex1 = LhIN[r1, "Sex"],
                                  Sex2 = LhIN[r2, "Sex"],
                                  BY1 = LhIN[r1, "BY"],
                                  BY2 = LhIN[r2, "BY"])
    }
    NoLH <- setdiff(gID, LhIN$ID)
    if (length(NoLH)>0) Duplicates$NoLH <- NoLH
  }

  # print warnings
  if (quiet <1) {
    if (any(duplicated(gID))) message("duplicate IDs found in genotype data, please remove to avoid confusion")
    if (DUP$nDupGenos>0 && DUP$nDupGenos > sum(duplicated(gID))) {
      message("likely duplicate genotypes found, consider removing")
    }
    if (any(duplicated(LhIN[,1]))) message("duplicate IDs found in lifehistory data, first entry will be used")
  }

  return( Duplicates )
}


#=====================================================================
#=====================================================================

#' @title Order lifehistory data
#'
#' @description Order lifehistory data to match order of IDs in genotype data,
#'   filling in gaps with missing values
#'
#' @param LH dataframe with lifehistory information:
#' \itemize{
#'  \item{ID: }{max. 30 characters long,}
#'  \item{Sex: }{1 = females, 2 = males, other numbers = unkown,}
#'  \item{Birth Year: }{(or hatching year) Use negative numbers to denote
#'  missing values.}}
#' @param gID character vector with IDs in genotype data, in order of occurence
#'
#' @return
#' \item{BY}{Numeric vector with birth years, of same length as gID}
#' \item{Sex}{Numeric vector with genders, of same length as gID}
#'
#' @keywords internal

orderLH <- function(LH=NULL, gID=NULL) {
  if (!is.null(LH)) {
    names(LH) <- c("lhID", "Sex", "BY")
    LH$lhID <- as.character(LH$lhID)
    LH <- LH[!duplicated(LH$lhID), ]
    rownames(LH) <- LH$lhID
    LH <- LH[gID, ]
    Sex <- LH$Sex
    Sex[is.na(Sex)] <- 3
    Sex[Sex<1 | Sex>2] <- 3
    BY <- LH$BY
    BY[is.na(BY)] <- -999
    BY[BY<0] <- -999
  } else {
    Sex <- rep(3, length(gID))
    BY <- rep(-999, length(gID))
  }
  list(Sex=Sex, BY=BY)
}

