#' @title Add a Fit Object to a 'qgraph'
#' Object
#'
#' @description Add a fit object (e.g.,
#' 'lavaan' output) to the a 'qgraph'
#' object as an attribute.
#'
#' @details It adds an object to
#' a [qgraph::qgraph] object as the
#' attribute `"semptools_fit_object"`,
#' to be retrieved by other functions
#' that need to access the original
#' output used in [semPlot::semPaths()]
#' to create a diagram.
#'
#' @return
#' The original [qgraph::qgraph] object
#' set to `semPaths_plot`, with the
#' attribute `"semptools_fit_object"`
#' set to `object`.
#'
#' @param semPaths_plot A
#' [qgraph::qgraph] object generated by
#' [semPlot::semPaths()], or a similar
#' [qgraph::qgraph] object modified by
#' other [semptools] functions.
#'
#' @param object Should be the object,
#' such as the output of [lavaan::sem()]
#' or [lavaan::cfa()], used by
#' [semPlot::semPaths()] to generate
#' `semPaths_plot`. Note that this
#' function will not check whether
#' the object is appropriate because
#' there is no way to do so reliably.
#'
#' @seealso [semPlot::semPaths()]
#'
#' @examples
#'
#' library(lavaan)
#' library(semPlot)
#'
#' mod <-
#'   'f1 =~ x01 + x02 + x03 + x06
#'    f2 =~ x04 + x05 + x06 + x07
#'    f3 =~ x08 + x09 + x10 + x03
#'    f4 =~ x11 + x12 + x13 + x14
#'   '
#' fit <- lavaan::cfa(mod, cfa_example)
#' p <- semPaths(fit,
#'               whatLabels = "est",
#'               sizeMan = 3.25,
#'               node.width = 1,
#'               edge.label.cex = .75,
#'               mar = c(10, 5, 10, 5),
#'               DoNotPlot = TRUE)
#' p <- add_object(p, fit)
#' attr(p, "semptools_fit_object")
#'
#' @export

add_object <- function(semPaths_plot,
                       object) {
    if (!inherits(semPaths_plot, "qgraph")) {
        stop("'semPaths_plot' not a qgraph object.")
      }
    attr(semPaths_plot, "semptools_fit_object") <- object
    semPaths_plot
  }

