#' @importFrom ggplot2 .data
#' @export
plot.see_check_model <- function(x,
                                 style = theme_lucid,
                                 colors = c("#3aaf85", "#1b6ca8", "#cd201f"),
                                 ...) {
  p <- list()

  panel <- attr(x, "panel")
  check <- attr(x, "check")
  size_point <- attr(x, "dot_size")
  size_line <- attr(x, "line_size")
  show_labels <- attr(x, "show_labels") %||% TRUE
  size_text <- attr(x, "text_size")
  alpha_level <- attr(x, "alpha")
  dot_alpha_level <- attr(x, "dot_alpha")
  detrend <- attr(x, "detrend")
  model_info <- attr(x, "model_info")
  overdisp_type <- attr(x, "overdisp_type")

  if (missing(style) && !is.null(attr(x, "theme"))) {
    theme_style <- unlist(strsplit(attr(x, "theme"), "::", fixed = TRUE))
    style <- get(theme_style[2], asNamespace(theme_style[1]))
  }

  if (missing(colors)) {
    colors <- attr(x, "colors")
  }

  if (is.null(colors)) {
    colors <- c("#3aaf85", "#1b6ca8", "#cd201f")
  }

  colors <- unname(colors)

  if (is.null(alpha_level)) {
    alpha_level <- .2
  }

  if (is.null(dot_alpha_level)) {
    dot_alpha_level <- .8
  }

  if (is.null(check)) check <- "all"

  if ("PP_CHECK" %in% names(x) && !is.null(x$PP_CHECK) && any(c("pp_check", "all") %in% check)) {
    x$NORM <- NULL
    p$PP_CHECK <- plot.see_performance_pp_check(
      x$PP_CHECK,
      style = style,
      check_model = TRUE,
      adjust_legend = TRUE
    )
  }

  if ("NCV" %in% names(x) && !is.null(x$NCV) && any(c("ncv", "linearity", "all") %in% check)) {
    p$NCV <- .plot_diag_linearity(
      x$NCV,
      size_point,
      size_line,
      alpha_level,
      theme_style = style,
      colors = colors,
      dot_alpha_level = dot_alpha_level
    )
  }

  if ("BINNED_RESID" %in% names(x) && !is.null(x$BINNED_RESID) && any(c("binned_residuals", "all") %in% check)) {
    x$HOMOGENEITY <- NULL
    p$BINNED_RESID <- plot.see_binned_residuals(
      x$BINNED_RESID,
      style = style,
      colors = colors[3:2],
      adjust_legend = TRUE,
      check_model = TRUE
    )
  }

  if ("OVERDISPERSION" %in% names(x) && !is.null(x$OVERDISPERSION) && any(c("overdispersion", "all") %in% check)) {
    p$OVERDISPERSION <- .plot_diag_overdispersion(
      x$OVERDISPERSION,
      style = style,
      colors = colors[c(1, 2)],
      size_line = size_line,
      type = overdisp_type
    )
  }

  if ("HOMOGENEITY" %in% names(x) && !is.null(x$HOMOGENEITY) && any(c("homogeneity", "all") %in% check)) {
    p$HOMOGENEITY <- .plot_diag_homogeneity(
      x$HOMOGENEITY,
      size_point,
      size_line,
      alpha_level,
      theme_style = style,
      colors = colors,
      dot_alpha_level = dot_alpha_level
    )
  }

  if ("INFLUENTIAL" %in% names(x) && !is.null(x$INFLUENTIAL) && any(c("outliers", "influential", "all") %in% check)) {
    p$OUTLIERS <- .plot_diag_outliers_new(
      x$INFLUENTIAL,
      show_labels = show_labels,
      size_text = size_text,
      size_line = size_line,
      theme_style = style,
      colors = colors,
      dot_alpha_level = dot_alpha_level
    )
  }

  if ("VIF" %in% names(x) && !is.null(x$VIF) && any(c("vif", "all") %in% check)) {
    p$VIF <- .plot_diag_vif(
      x$VIF,
      theme_style = style,
      colors = colors
    )
  }

  if ("QQ" %in% names(x) && !is.null(x$QQ) && any(c("qq", "all") %in% check)) {
    p$QQ <- .plot_diag_qq(
      x$QQ,
      size_point,
      size_line,
      alpha_level = alpha_level,
      detrend = detrend,
      theme_style = style,
      colors = colors,
      dot_alpha_level = dot_alpha_level
    )
  }

  if ("NORM" %in% names(x) && !is.null(x$NORM) && any(c("normality", "all") %in% check)) {
    p$NORM <- .plot_diag_norm(
      x$NORM,
      size_line,
      alpha_level = alpha_level,
      theme_style = style,
      colors = colors
    )
  }

  if ("REQQ" %in% names(x) && !is.null(x$REQQ) && any(c("reqq", "all") %in% check)) {
    ps <- .plot_diag_reqq(
      x$REQQ,
      size_point,
      size_line,
      alpha_level = alpha_level,
      theme_style = style,
      colors = colors,
      dot_alpha_level = dot_alpha_level
    )

    for (i in 1:length(ps)) {
      p[[length(p) + 1]] <- ps[[i]]
    }
  }

  if (panel) {
    plots(p, n_columns = 2)
  } else {
    return(p)
  }
}



.plot_diag_vif <- function(x,
                           theme_style = theme_lucid,
                           colors = unname(social_colors(c("green", "blue", "red")))) {
  ylim <- max(x$y, na.rm = TRUE)
  if (ylim < 10) ylim <- 10

  # make sure legend is properly sorted
  x$group <- factor(x$group, levels = c("low", "moderate", "high"))
  levels(x$group) <- c("low (< 5)", "moderate (< 10)", "high (>= 10)")
  names(colors) <- c("low (< 5)", "moderate (< 10)", "high (>= 10)")

  p <- ggplot2::ggplot(x, ggplot2::aes(x = .data$x, y = .data$y, fill = .data$group))

  if (ylim > 5) {
    p <- p + ggplot2::geom_rect(
      xmin = -Inf,
      xmax = Inf,
      ymin = 0,
      ymax = 5,
      fill = colors[1],
      color = NA,
      alpha = .025
    )

    p <- p + ggplot2::geom_rect(
      xmin = -Inf,
      xmax = Inf,
      ymin = 5,
      ymax = ifelse(ylim > 10, 10, ylim),
      fill = colors[2],
      color = NA,
      alpha = .025
    )
  }

  if (ylim > 10) {
    p <- p + ggplot2::geom_rect(
      xmin = -Inf,
      xmax = Inf,
      ymin = 10,
      ymax = ylim,
      fill = colors[3],
      color = NA,
      alpha = .025
    )
  }

  p <- p +
    ggplot2::geom_col(width = 0.7) +
    ggplot2::labs(
      title = "Collinearity",
      subtitle = "Higher bars (>5) indicate potential collinearity issues",
      x = NULL,
      y = "Variance Inflation Factor (VIF)",
      fill = NULL
    ) +
    # geom_text(aes(label = round(.data$y, 1)), nudge_y = 1) +
    ggplot2::scale_fill_manual(values = colors) +
    theme_style(
      base_size = 10,
      plot.title.space = 3,
      axis.title.space = 5
    ) +
    ggplot2::ylim(c(0, ylim)) +
    ggplot2::theme(
      legend.position = "bottom",
      legend.margin = ggplot2::margin(0, 0, 0, 0),
      legend.box.margin = ggplot2::margin(-5, -5, -5, -5)
    )

  if ("facet" %in% colnames(x)) {
    p <- p + ggplot2::facet_wrap(~facet, nrow = 1, scales = "free")
  }

  p
}



.plot_diag_norm <- function(x,
                            size_line,
                            alpha_level = .2,
                            theme_style = theme_lucid,
                            colors = unname(social_colors(c("green", "blue", "red")))) {
  ggplot2::ggplot(x, ggplot2::aes(x = .data$x)) +
    ggplot2::geom_ribbon(
      mapping = ggplot2::aes(ymin = 0, ymax = .data$y),
      colour = NA,
      fill = colors[2],
      alpha = alpha_level
    ) +
    ggplot2::geom_line(
      mapping = ggplot2::aes(y = .data$curve),
      colour = colors[1],
      size = size_line
    ) +
    ggplot2::labs(
      x = "Residuals",
      y = "Density",
      title = "Normality of Residuals",
      subtitle = "Distribution should be close to the normal curve"
    ) +
    theme_style(
      base_size = 10,
      plot.title.space = 3,
      axis.title.space = 5
    ) +
    ggplot2::scale_y_continuous(labels = NULL)
}




.plot_diag_qq <- function(x,
                          size_point,
                          size_line,
                          alpha_level = .2,
                          detrend = FALSE,
                          theme_style = theme_lucid,
                          colors = unname(social_colors(c("green", "blue", "red"))),
                          dot_alpha_level = .8) {
  if (requireNamespace("qqplotr", quietly = TRUE)) {
    qq_stuff <- list(
      qqplotr::stat_qq_band(alpha = alpha_level, detrend = detrend),
      qqplotr::stat_qq_line(
        size = size_line,
        colour = colors[1],
        detrend = detrend
      ),
      qqplotr::stat_qq_point(
        shape = 16,
        stroke = 0,
        size = size_point,
        colour = colors[2], # "#2c3e50",
        alpha = dot_alpha_level,
        detrend = detrend
      )
    )
    y_lab <- "Sample Quantiles"
  } else {
    message(
      "For confidence bands",
      if (isTRUE(detrend)) " and detrending",
      ", please install `qqplotr`."
    )

    qq_stuff <- list(
      ggplot2::geom_qq(
        shape = 16, stroke = 0,
        size = size_point,
        colour = colors[2] # "#2c3e50"
      ),
      ggplot2::geom_qq_line(
        size = size_line,
        colour = colors[1]
      )
    )
    y_lab <- "Sample Quantiles"
  }
  ggplot2::ggplot(x, ggplot2::aes(sample = .data$y)) +
    qq_stuff +
    ggplot2::labs(
      title = "Normality of Residuals",
      subtitle = "Dots should fall along the line",
      y = y_lab,
      x = "Standard Normal Distribution Quantiles"
    ) +
    theme_style(
      base_size = 10,
      plot.title.space = 3,
      axis.title.space = 5
    )
}




.plot_diag_pp <- function(x,
                          size_point,
                          size_line,
                          alpha_level = .2,
                          detrend = FALSE,
                          theme_style = theme_lucid,
                          colors = unname(social_colors(c("green", "blue", "red"))),
                          dot_alpha_level = .8) {
  if (requireNamespace("qqplotr", quietly = TRUE)) {
    p_plot <- ggplot2::ggplot(x, ggplot2::aes(sample = .data$res)) +
      qqplotr::stat_pp_band(alpha = alpha_level, detrend = detrend) +
      qqplotr::stat_pp_line(
        size = size_line,
        colour = colors[1],
        detrend = detrend
      ) +
      qqplotr::stat_pp_point(
        shape = 16, stroke = 0,
        size = size_point,
        colour = colors[2], # "#2c3e50",
        alpha = dot_alpha_level,
        detrend = detrend
      )
  } else if (requireNamespace("MASS", quietly = TRUE)) {
    message(
      "For confidence bands",
      if (isTRUE(detrend)) " and detrending",
      ", please install `qqplotr`."
    )


    x$probs <- stats::ppoints(x$res)
    dparms <- MASS::fitdistr(x$res, densfun = "normal")
    x$y <- do.call(stats::pnorm, c(list(q = x$res), dparms$estimate))

    p_plot <- ggplot2::ggplot(x, ggplot2::aes(x = .data$probs, y = .data$y)) +
      ggplot2::geom_abline(
        slope = 1,
        size = size_line,
        colour = colors[1]
      ) +
      geom_point2(
        colour = colors[2],
        size = size_point,
        alpha = dot_alpha_level
      ) # "#2c3e50"
  } else {
    stop("Package 'qqplotr' OR 'MASS' required for PP-plots. Please install one of them.", call. = FALSE)
  }

  p_plot +
    ggplot2::labs(
      title = "Normality of Residuals (PP plot)",
      subtitle = "Dots should fall along the line",
      y = "Cummulative Probability",
      x = "Probability Points"
    ) +
    theme_style(
      base_size = 10,
      plot.title.space = 3,
      axis.title.space = 5
    )
}




.plot_diag_homogeneity <- function(x,
                                   size_point,
                                   size_line,
                                   alpha_level = .2,
                                   theme_style = theme_lucid,
                                   colors = unname(social_colors(c("green", "blue", "red"))),
                                   dot_alpha_level = .8) {
  ggplot2::ggplot(x, ggplot2::aes(x = .data$x, .data$y)) +
    geom_point2(
      colour = colors[2],
      size = size_point,
      alpha = dot_alpha_level
    ) +
    ggplot2::stat_smooth(
      method = "loess",
      se = TRUE,
      alpha = alpha_level,
      formula = y ~ x,
      size = size_line,
      colour = colors[1]
    ) +
    ggplot2::labs(
      title = "Homogeneity of Variance",
      subtitle = "Reference line should be flat and horizontal",
      y = expression(sqrt("|Std. residuals|")),
      x = "Fitted values"
    ) +
    theme_style(
      base_size = 10,
      plot.title.space = 3,
      axis.title.space = 5
    )
}



.plot_diag_linearity <- function(x,
                                 size_point,
                                 size_line,
                                 alpha_level = .2,
                                 theme_style = theme_lucid,
                                 colors = unname(social_colors(c("green", "blue", "red"))),
                                 dot_alpha_level = .8) {
  ggplot2::ggplot(x, ggplot2::aes(x = .data$x, y = .data$y)) +
    geom_point2(
      colour = colors[2],
      size = size_point,
      alpha = dot_alpha_level
    ) +
    ggplot2::geom_smooth(
      method = "loess",
      se = TRUE,
      formula = y ~ x,
      alpha = alpha_level,
      size = size_line,
      colour = colors[1]
    ) +
    ggplot2::geom_hline(yintercept = 0, linetype = "dashed") +
    ggplot2::labs(
      x = "Fitted values",
      y = "Residuals",
      title = "Linearity",
      subtitle = "Reference line should be flat and horizontal"
    ) +
    theme_style(
      base_size = 10,
      plot.title.space = 3,
      axis.title.space = 5
    )
}



.plot_diag_reqq <- function(x,
                            size_point,
                            size_line,
                            panel = TRUE,
                            alpha_level = .2,
                            theme_style = theme_lucid,
                            colors = unname(social_colors(c("green", "blue", "red"))),
                            dot_alpha_level = .8) {
  lapply(names(x), function(i) {
    dat <- x[[i]]
    p <- ggplot2::ggplot(dat, ggplot2::aes(x = .data$x, y = .data$y)) +
      ggplot2::labs(
        x = "Theoretical Quantiles",
        y = "RE Quantiles",
        title = sprintf("Normality of Random Effects (%s)", i),
        subtitle = "Dots should be plotted along the line"
      ) +
      ggplot2::stat_smooth(
        method = "lm",
        alpha = alpha_level,
        size = size_line,
        formula = y ~ x,
        colour = colors[1]
      ) +
      ggplot2::geom_errorbar(
        ggplot2::aes(ymin = .data$conf.low, ymax = .data$conf.high),
        width = 0,
        colour = colors[2],
        alpha = dot_alpha_level
      ) +
      geom_point2(
        colour = colors[2],
        size = size_point,
        alpha = dot_alpha_level
      ) +
      theme_style(
        base_size = 10,
        plot.title.space = 3,
        axis.title.space = 5
      )

    if (nlevels(dat$facet) > 1 && isTRUE(panel)) {
      p <- p + ggplot2::facet_wrap(~facet, scales = "free")
    }

    p
  })
}



.plot_diag_overdispersion <- function(x,
                                      theme_style = theme_lucid,
                                      colors = c("#3aaf85", "#1b6ca8"),
                                      size_line = .8,
                                      type = 1,
                                      ...) {
  if (is.null(type) || type == 1) {
    p <- ggplot2::ggplot(x) +
      ggplot2::aes(x = .data$Predicted) +
      ggplot2::geom_smooth(ggplot2::aes(y = .data$V), size = size_line, color = colors[2], se = FALSE) +
      ggplot2::geom_smooth(ggplot2::aes(y = .data$Res2), size = size_line, color = colors[1]) +
      ggplot2::labs(
        title = "Overdispersion and zero-inflation",
        subtitle = "Observed residual variance (green) should follow predicted residual variance (blue)",
        x = "Predicted mean",
        y = "Residual variance"
      ) +
      theme_style(
        base_size = 10,
        plot.title.space = 3,
        axis.title.space = 5
      )
  } else {
    p <- ggplot2::ggplot(x) +
      ggplot2::aes(x = .data$Predicted) +
      ggplot2::geom_point(ggplot2::aes(y = .data$StdRes)) +
      ggplot2::geom_hline(
        yintercept = c(-2, 2, -4, 4),
        linetype = c("solid", "solid", "dashed", "dashed"),
        color = c(rep(colors[1], 2), rep(colors[2], 2))
      ) +
      ggplot2::labs(
        title = "Overdispersion and zero-inflation",
        subtitle = "Most points should be within solid lines, few points outside dashed lines",
        x = "Predicted mean",
        y = "Standardized resiuduals"
      ) +
      theme_style(
        base_size = 10,
        plot.title.space = 3,
        axis.title.space = 5
      )
  }

  p
}
