% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/seagull.R
\docType{package}
\name{seagull}
\alias{seagull}
\title{Mixed model fitting with lasso, group lasso, or sparse-group lasso
regularization}
\usage{
seagull(y, X, Z, weights_u, groups, alpha, rel_acc, max_lambda, xi,
        loops_lambda, max_iter, gamma_bls, trace_progress)
}
\arguments{
\item{y}{numeric vector of observations.}

\item{X}{(optional) numeric design matrix relating y to fixed effects b.}

\item{Z}{numeric design matrix relating y to random effects u.}

\item{weights_u}{(optional) numeric vector of weights for the vector of
random effects u. If no weights are passed by the user, all weights for u are
set to \code{1}.}

\item{groups}{(not required for the lasso) integer vector specifying which
effect belongs to which group. If the lasso is called, this vector is without
use and may remain empty. For group and sparse-group lasso, at least each
random effect variable needs to be assigned to one group. If in this case
also fixed effect variables are present in the model, but without assigned
group values, then the same value will be assigned to all fixed effect
variables automatically. Float or double input values will be truncated.}

\item{alpha}{mixing parameter for the sparse-group lasso. Has to satisfy:
\eqn{0 \le \alpha \le 1}. The penalty term looks as follows: \deqn{\alpha *
'"lasso penalty" + (1-\alpha) * "group lasso penalty".} If \code{alpha=1},
the lasso is called. If \code{alpha=0}, the group lasso is called. Default
value is \code{0.9}.}

\item{rel_acc}{(optional) relative accuracy of the solution to stop the
algorithm for the current value of \eqn{\lambda}. The algorithm stops after
iteration m, if: \deqn{||sol^{(m)} - sol^{(m-1)}||_\infty < rel_acc *
||sol1{(m-1)}||_2.} Default value is \code{0.0001}.}

\item{max_lambda}{(optional) maximum value for the penalty parameter. Default
option is an integrated algorithm to calculate this value. This is the start
value for the grid search of the penalty parameter \eqn{\lambda}. (More
details about the integrated algorithms are available here:
\code{\link[seagull]{lambda_max}}.)}

\item{xi}{(optional) multiplicative parameter to determine the minimum value
of \eqn{\lambda} for the grid search, i.e. \eqn{\lambda_{min} = \xi *
\lambda_{max}}. Has to satisfy: \eqn{0 < \xi \le 1}. If \code{xi=1}, only a
single solution for \eqn{\lambda = \lambda_{max}} is calculated. Default
value is \code{0.01}.}

\item{loops_lambda}{(optional) number of lambdas for the grid search between
\eqn{\lambda_{max}} and \eqn{\xi * \lambda_{max}}. Loops are performed on a
logarithmic grid. Float or double input values will be truncated. Default
value is \code{50}.}

\item{max_iter}{(optional) maximum number of iterations for each value of the
penalty parameter \eqn{\lambda}. Determines the end of the calculation if the
algorithm didn't converge according to \code{rel_acc} before. Float or double
input values will be truncated. Default value is \code{1000}.}

\item{gamma_bls}{(optional) multiplicative parameter to decrease the step
size during backtracking line search. Has to satisfy: \eqn{0 < \gamma_{bls}
< 1}. Default value is \code{0.8}.}

\item{trace_progress}{(optional) if \code{TRUE}, a message will occur on the
screen after each finished loop of the \eqn{\lambda} grid. This is
particularly useful for larger data sets.}
}
\value{
A list of estimates and parameters relevant for the computation:
\describe{
\item{fixed_effects}{estimates for the fixed effects b, if present in the
model. Each row corresponds to a particular value of \eqn{\lambda}.}
\item{random_effects}{predictions for the random effects u. Each row
corresponds to a particular value of \eqn{\lambda}.}
\item{lambda}{all values for \eqn{\lambda} which were used during the grid
search.}
\item{iterations}{a sequence of actual iterations for each value of
\eqn{\lambda}. If an occurring number is equal to \code{max_iter}, then the
algorithm most likely did not converge to \code{rel_acc} during the
corresponding run of the grid search.}
}
The following parameters are also returned. But primarily for the purpose of
comparison and repetition: \code{alpha} (only for the sparse-group lasso),
\code{max_iter}, \code{gamma_bls}, \code{xi}, and \code{loops_lambda}.
}
\description{
Fit a mixed model with lasso, group lasso, or sparse-group lasso
via proximal gradient descent. As this is an iterative algorithm, the step
size for each iteration is determined via backtracking line search. A grid
search for the regularization parameter \eqn{\lambda} is performed using warm
starts. Depending on the input parameter \code{alpha} this function
subsequently calls one of the three implemented
\code{\link[seagull]{lasso_variants}}. None of the input variables will be
centered or standardized throughout any of the calculations of this package.
}
\details{
The underlying mixed model has the form: \deqn{y = X b + Z u +
residual,} where \eqn{b} is the vector of fixed effects and \eqn{u} is the
vector of random effects. The penalty of the sparse-group lasso (without
additional weights for features) is then: \deqn{\alpha \lambda ||u||_1 + (1 -
\alpha) \lambda \sum_l \omega^G_l ||u^{(l)}||_2.} The variable
\eqn{\omega^G_l} is a weight for group \eqn{l}. If \eqn{\alpha = 1}, this
leads to the lasso. If \eqn{\alpha = 0}, this leads to the group lasso.

The above penalty can be enhanced by introducing additional weights
\eqn{\omega^F} for features in the lasso penalty: \deqn{\alpha \lambda
\sum_j | \omega^F_j u_j | + (1 - \alpha) \lambda \sum_l \omega^G_l
||u^{(l)}||_2.}

The group weights \eqn{\omega^G_l} are implemented as follows:
\eqn{\omega^G_l = \sum_{j, j \in G} \sqrt{\omega^F_{j}}}. So, in the case
that the weights for features in a particular group are all equal to one,
this term collapses to the square root of the group size.

In addition, the above penalty can be formally rewritten by including the
fixed effects and assigning weights equal to zero to all these features. This
is how the algorithms are implemented. Consequently, if a weight for any
random effect is set to zero by the user, the function drops an error and the
calculation will not start.
}
\examples{

\donttest{

set.seed(62)
n <- 50  ## observations
p <- 8   ## variables

## Create a design matrix X for fixed effects to model the intercept:
X <- matrix(1, nrow = n, ncol = 1)

## Create a design matrix Z for random effects:
Z <- matrix(rnorm(p * n, mean = 0, sd = 1), nrow = n)

## Intercept b, random effect vector u, and response y:
b <- 0.2
u <- c(0, 1.5, 0, 0.5, 0, 0, -2, 1)
y <- X\%*\%b + Z\%*\%u + rnorm(n, mean = 0, sd = 1)

## Create a vector of three groups corresponding to vector u:
group_indices <- c(1L, 2L, 2L, 2L, 1L, 1L, 3L, 1L)

## Calculate the solution:
fit_l   <- seagull(y = y, X = X, Z = Z, alpha = 1.0)
fit_gl  <- seagull(y = y, X = X, Z = Z, alpha = 0.0, groups = group_indices)
fit_sgl <- seagull(y = y, X = X, Z = Z, groups = group_indices)

## Combine the estimates for fixed and random effects:
estimates_l   <- cbind(fit_l$fixed_effects, fit_l$random_effects)
estimates_gl  <- cbind(fit_gl$fixed_effects, fit_gl$random_effects)
estimates_sgl <- cbind(fit_sgl$fixed_effects, fit_sgl$random_effects)
true_effects  <- c(b, u)

## Calculate mean squared errors along the solution paths:
MSE_l   <- rep(as.numeric(NA), fit_l$loops_lambda)
MSE_gl  <- rep(as.numeric(NA), fit_l$loops_lambda)
MSE_sgl <- rep(as.numeric(NA), fit_l$loops_lambda)

for (i in 1:fit_l$loops_lambda) {
  MSE_l[i] <- t(estimates_l[i,] - true_effects)\%*\%(estimates_l[i,] - true_effects)/(p+1)
  MSE_gl[i] <- t(estimates_gl[i,] - true_effects)\%*\%(estimates_gl[i,] - true_effects)/(p+1)
  MSE_sgl[i] <- t(estimates_sgl[i,] - true_effects)\%*\%(estimates_sgl[i,] - true_effects)/(p+1)
}

## Plot a fraction of the results of the MSEs:
plot(x = seq(1, fit_l$loops_lambda, 1)[25:50], MSE_l[25:50], type = "l", lwd = 2)
points(x = seq(1, fit_l$loops_lambda, 1)[25:50], MSE_gl[25:50], type = "l", lwd = 2, col = "blue")
points(x = seq(1, fit_l$loops_lambda, 1)[25:50], MSE_sgl[25:50], type = "l", lwd = 2, col = "red")

## A larger example with simulated genetic data:
data(seagull_data)

fit_l1 <- seagull(y = phenotypes[,1], Z = genotypes, alpha = 1.0)
fit_l2 <- seagull(y = phenotypes[,2], Z = genotypes, alpha = 1.0)
fit_l3 <- seagull(y = phenotypes[,3], Z = genotypes, alpha = 1.0,
          trace_progress = T)

fit_gl1 <- seagull(y = phenotypes[,1], Z = genotypes, alpha = 0.0,
           groups = groups)
fit_gl2 <- seagull(y = phenotypes[,2], Z = genotypes, alpha = 0.0,
           groups = groups)
fit_gl3 <- seagull(y = phenotypes[,3], Z = genotypes, alpha = 0.0,
           groups = groups, trace_progress = T)

fit_sgl1 <- seagull(y = phenotypes[,1], Z = genotypes, groups = groups)
fit_sgl2 <- seagull(y = phenotypes[,2], Z = genotypes, groups = groups)
fit_sgl3 <- seagull(y = phenotypes[,3], Z = genotypes, groups = groups,
            trace_progress = T)
}

}
\seealso{
\code{\link{plot}}
}
\author{
Jan Klosa
}
\keyword{models}
\keyword{regression}
