% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cross-val.R
\name{cv_to_waywiser}
\alias{cv_to_waywiser}
\title{Convert \code{\link[=sdmTMB_cv]{sdmTMB_cv()}} objects to \pkg{sf} format for spatial assessment with \pkg{waywiser}}
\usage{
cv_to_waywiser(
  object,
  ll_names = c("longitude", "latitude"),
  ll_crs = 4326,
  utm_crs = get_crs(object$data, ll_names)
)
}
\arguments{
\item{object}{An object of class \code{sdmTMB_cv} from \code{\link[=sdmTMB_cv]{sdmTMB_cv()}}.}

\item{ll_names}{Column names for longitude and latitude in the original data.
\strong{Note the order: longitude first, then latitude.}}

\item{ll_crs}{The coordinate reference system (CRS) for the \code{ll_names} columns.
Defaults to \code{4326} (WGS84 lon/lat).}

\item{utm_crs}{The projected coordinate reference system (CRS) for the output
sf object. By default (if you're feeling lucky!) automatically detected
using \code{\link[=get_crs]{get_crs()}} based on \code{ll_names}. Can be manually specified as an
EPSG code (e.g., \code{32609}) or any format accepted by \code{\link[sf:st_crs]{sf::st_crs()}}.}
}
\value{
An \code{\link[sf:sf]{sf::sf()}} object with POINT geometry containing:
\describe{
\item{truth}{The observed response values}
\item{estimate}{The cross-validated predictions}
\item{geometry}{Spatial point locations}
}
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#experimental}{\figure{lifecycle-experimental.svg}{options: alt='[Experimental]'}}}{\strong{[Experimental]}}
Converts cross-validation results to an \code{\link[sf:sf]{sf::sf()}} object for use with
spatial model assessment tools such as those in the \pkg{waywiser} package.
This enables multi-scale spatial assessment of model predictions.
}
\details{
This function is particularly useful for assessing spatial models at multiple
scales using the \pkg{waywiser} package. After converting to sf format, you
can use functions like \code{\link[waywiser:ww_multi_scale]{waywiser::ww_multi_scale()}} to evaluate how model
performance changes when predictions are aggregated to different spatial
scales.

For delta/hurdle models, the combined predictions are returned (e.g., the
product of the encounter probability and positive catch rate).
}
\examples{
\dontshow{if (require("sf", quietly = TRUE) && require("waywiser", quietly = TRUE)) withAutoprint(\{ # examplesIf}
mesh <- make_mesh(pcod_2011, c("X", "Y"), cutoff = 12)

# Run cross-validation
set.seed(123)
m_cv <- sdmTMB_cv(
  density ~ depth_scaled,
  data = pcod_2011,
  mesh = mesh,
  family = tweedie(),
  k_folds = 2
)

# Convert with default auto-detected CRS based on lon/lat columns:
cv_sf <- cv_to_waywiser(m_cv, ll_names = c("lon", "lat"))

# Or manually specify the desired UTM CRS:
cv_sf <- cv_to_waywiser(m_cv, ll_names = c("lon", "lat"), utm_crs = 32609)

# Use with waywiser for multi-scale assessment
waywiser::ww_multi_scale(
  cv_sf,
  truth,    # column name (unquoted)
  estimate, # column name (unquoted)
  n = list(c(5, 5), c(2, 2)) # 5x5 and 2x2 grid blocks
)
\dontshow{\}) # examplesIf}
}
\seealso{
\code{\link[=sdmTMB_cv]{sdmTMB_cv()}}, \code{\link[=get_crs]{get_crs()}}, \url{https://sdmTMB.github.io/sdmTMB/articles/cross-validation.html}
}
