% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/model_LinearFlow.R
\docType{data}
\name{LinearFlow}
\alias{LinearFlow}
\title{Creates Linear-FLow model for one-class collaborative filtering}
\format{\code{R6Class} object.}
\usage{
LinearFlow
}
\description{
Creates \bold{Linear-FLow} model described in
\href{http://www.bkveton.com/docs/ijcai2016.pdf}{Practical Linear Models for Large-Scale One-Class Collaborative Filtering}.
The goal is to find item-item (or user-user) similarity matrix which is \bold{low-rank and has small Frobenius norm}. Such
double regularization allows to better control the generalization error of the model.
Idea of the method is somewhat similar to \bold{Sparse Linear Methods(SLIM)} but scales to large datasets much better.
}
\section{Usage}{

For usage details see \bold{Methods, Arguments and Examples} sections.
\preformatted{
  model = LinearFlow$new( rank = 8L,
                          lambda = 0,
                          init = NULL,
                          preprocess = identity,
                          solve_right_singular_vectors = c("soft_impute", "svd")
                          ...)
  model$fit_transform(x, ...)
  model$transform(x, ...)
  model$predict(x, k, not_recommend = x, ...)
  model$components
  model$v
  model$cross_validate_lambda(x, x_train, x_test, lambda = "auto@10",
                       metric = "map@10", not_recommend = x_train, ...)
}


}

\section{Methods}{

\describe{
  \item{\code{$new(rank = 8L, lambda = 0,
              init = NULL,
              preprocess = identity,
              solve_right_singular_vectors = c("svd", "soft_impute"),
              ...)}}{ creates Linear-FLow model with \code{rank} latent factors.
    If \code{init} (right singular vectors of the user-item interactions matrix)
    is provided then model initialized with its values.}
  \item{\code{$fit_transform(x, ...)}}{ fits model to
    an input user-item interaction matrix.
    \bold{Returns user embeddings matrix of the size \code{n_users * rank}}}
  \item{\code{$transform(x, ...)}}{transforms user-item interaction matrix into user-embeddings matrix.}
  \item{\code{$predict(x, k, not_recommend = x, ...)}}{predicts \bold{top k}
    item ids for users \code{x}. Users features should be defined the same way as they were defined in
    training data - as \bold{sparse matrix}. Column names (=item ids) should be in the same order as
    in the \code{fit_transform()}.}
  \item{preprocess}{\code{function} = \code{identity()} by default. User spectified function which will
    be applied to user-item interaction matrix before running matrix factorization
    (also applied in inference time before making predictions).}
  \item{\code{$cross_validate_lambda(x, x_train, x_test, lambda = "auto@10", metric = "map@10",
                              not_recommend = x_train, ...)}}{perfroms search of the
  best regularization parameter \code{lambda}:
  \enumerate{
    \item Model is trained on \code{x} data
    \item Then model makes predictions based on \code{x_train} data
    \item And finally these predications are validated using specified \code{metric} against \code{x_test} data
  }
  Note that this is implemented smartly with \bold{"warm starts"}.
  So it is very cheap - \bold{cost is almost the same as for single fit} of the model. The only considerable additional cost is
  time to predict \emph{top k} items. In most cases automatic lambda like \code{lambda = "auto@20"} is able to find good value of the parameter}
  \item{\code{$components}}{item factors matrix of size \code{rank * n_items}. In the paper this matrix is called \bold{Y}}
  \item{\code{$v}}{right singular vector of the user-item matrix. Size is \code{n_items * rank}. In the paper this matrix is called \bold{v}}
}
}

\section{Arguments}{

\describe{
 \item{model}{A \code{LinearFlow} model.}
 \item{x}{An input sparse user-item matrix (inherits from \code{sparseMatrix})}
 \item{rank}{\code{integer} - number of latent factors}
 \item{lambda}{\code{numeric} - regularization parameter or sequence of regularization values for \code{cross_validate_lambda} method.}
 \item{not_recommend}{\code{sparse matrix} or \code{NULL} - points which items should be excluided from recommendations for a user.
   By default it excludes previously seen/consumed items.}
 \item{metric}{metric to use in evaluation of top-k recommendations.
   Currently only \code{map@k} and \code{ndcg@k} are supported (\code{k} can be any integer).}
 \item{...}{other arguments (not used at the moment)}
}
}

\examples{
data('movielens100k')
train = movielens100k[1:900, ]
cv = movielens100k[901:nrow(movielens100k), ]
model = LinearFlow$new(rank = 10, lambda = 0, init = NULL,
                       solve_right_singular_vectors = "svd")
user_emb = model$fit_transform(train)
preds = model$predict(cv, k = 10)
}
\seealso{
\itemize{
  \item{\url{http://www.bkveton.com/docs/ijcai2016.pdf}}
  \item{\url{http://www-users.cs.umn.edu/~xning/slides/ICDM2011_slides.pdf}}
}
}
\keyword{datasets}
