##' Summary information about the Open Tree Taxaonomy (OTT)
##'
##' Return metadata and information about the taxonomy
##' itself. Currently, the available metadata is fairly sparse, but
##' includes (at least) the eversion, and the location from which the
##' complete taxonomy source files can be downloaded.
##' @title About the Open Tree Taxonomy
##' @param ... additional arguments to customize the API request (see
##'     \code{\link{rotl}} package documentation).
##' @return A list with at least the version and the location of the
##'     taxonomy source files
##' @examples
##' \dontrun{
##' taxonomy_about()
##' }
##' @export
taxonomy_about <- function (...) {
    res <- .taxonomy_about(...)
    return(res)
}


##' Information about taxa.
##'
##' Given a vector of ott ids, \code{taxonomy_taxon} returns
##' information about the specified taxa.
##'
##' The functions \code{tax_rank}, \code{ott_taxon_name}, and
##' \code{synonyms} can extract this information from an object
##' created by the \code{taxonomy_taxon} function.
##'
##' @title Taxon information
##' @param ott_ids the ott ids of the taxon of interest (numeric or
##'     character containing only numbers)
##' @param include_lineage whether to include information about
##'     all the higher level taxa that include the \code{ott_ids}.
##'     Default \code{FALSE}.
##' @param list_terminal_descendants whether to include the list of
##'     terminal \code{ott_ids} contained in the \code{ott_ids}
##'     provided.
##' @param ... additional arguments to customize the API request (see
##'     \code{\link{rotl}} package documentation).
##' @param tax an object generated by the \code{taxonomy_taxon}
##'     function
##' @return \code{taxonomy_taxon} returns a list detailing information
##'     about the taxa. \code{tax_rank} and \code{ott_taxon_name}
##'     return a vector. \code{synonyms} returns a list whose elements
##'     are the synonyms for each of the \code{ott_id} requested.
##'
##' @seealso \code{\link{tnrs_match_names}} to obtain \code{ott_id}
##'     from a taxonomic name.
##' @examples
##' \dontrun{
##' req <- taxonomy_taxon(ott_id=515698)
##' tax_rank(req)
##' ott_taxon_name(req)
##' synonyms(req)
##' }
##' @export
taxonomy_taxon <- function (ott_ids, include_lineage = FALSE,
                            list_terminal_descendants = FALSE, ...) {
    res <- lapply(ott_ids, function(x) {
        .taxonomy_taxon(
            ott_id = x,
            include_lineage = include_lineage,
            list_terminal_descendants = list_terminal_descendants,
            ...
        )
    })
    names(res) <- ott_ids
    class(res) <- "taxon_info"
    return(res)
}


##' Given an ott id, return the inclusive taxonomic subtree descended
##' from the specified taxon.
##'
##' If the output of this function is exported to a file, the only
##' possible value for the \code{output_format} argument is
##' \dQuote{\code{newick}}. If the file provided already exists, it
##' will be silently overwritten.
##'
##' @title Taxonomy subtree
##' @param ott_id The ott id of the taxon of interest.
##' @param output_format the format of the object to be returned. See
##'     the \sQuote{Return} section.
##' @param ... additional arguments to customize the API request (see
##'     \code{\link{rotl}} package documentation).
##' @param file the file name where to save the output of the
##'     function. Ignored unless \code{output_format} is set to
##'     \dQuote{\code{phylo}}.
##' @return If the \code{file} argument is missing: \itemize{
##'
##'     \item{\dQuote{\code{taxa}}} { a list of the taxa names
##'     (species) in slot \code{tip_label}, and higher-level taxanomy
##'     (e.g., families, genera) in slot \code{edge_label}, descending
##'     from the taxa corresponding to the \code{ott_id} provided. }
##'
##'     \item{\dQuote{\code{newick}}} { a character vector containing
##'     the newick formatted string corresponding to the taxonomic
##'     subtree for the \code{ott_id} provided. }
##'
##'     \item{\dQuote{\code{phylo}}} { an object of the class
##'     \code{phylo} from the \code{\link[ape]{ape}} package. }
##'
##'     \item{\dQuote{\code{raw}}} { the direct output from the API,
##'     i.e., a list with an element named \sQuote{subtree} that
##'     contains the subtree as a newick formatted string. }
##'
##'     }
##'
##'     If a \code{file} argument is provided (and
##'     \code{output_format} is set to \dQuote{\code{phylo}}), a
##'     logical indicating whether the file was successfully created.
##'
##' @examples
##' \dontrun{
##' req <- taxonomy_subtree(ott_id=515698)
##' plot(taxonomy_subtree(ott_id=515698, output_format="phylo"))
##' }
##' @export
taxonomy_subtree <- function (ott_id=NULL,
                              output_format = c("taxa", "newick", "phylo", "raw"),
                              file, ...) {
    output_format <- match.arg(output_format)
    res <- .taxonomy_subtree(ott_id = ott_id, ...)
    if (!missing(file) && !identical(output_format, "newick"))
        warning("'file' argument is ignored, you can only write newick tree strings to a file.")
    if (identical(output_format, "raw")) {
        return(res)
    } else if (identical(output_format, "newick")) {
        res <- res$subtree
        if (!missing(file)) {
            unlink(file)
            cat(res, file = file)
            invisible(return(file.exists(file)))
        }
    } else if (identical(output_format, "phylo")) {
        res <- phylo_from_otl(res)
    } else { ## in all other cases use tree_to_labels
        res <- tree_to_labels(res)
    }
    return(res)
}


##' Taxonomic Least Inclusive Common Ancestor
##'
##' Given a set of ott ids, get the taxon that is the least inclusive common ancestor
##' (the LICA) of all the identified taxa. A taxonomic LICA is analogous to a most recent
##' common ancestor (MRCA) in a phylogenetic tree.
##'
##' @title Taxonomic LICA
##' @param ott_ids a vector of ott ids for the taxa whose LICA is to
##'     be found (numeric).
##' @param tax an object generated by the \code{taxonomy_lica}
##'     function
##' @param ... additional arguments to customize the API request (see
##'     \code{\link{rotl}} package documentation).
##' @return \itemize{
##'
##'     \item{\code{taxonomy_lica}} { returns a list about the
##'     taxonomic information relating to the LICA for the ott_ids
##'     provided. }
##'
##'     \item{\code{tax_rank}} { returns a character vector of the
##'     taxonomic rank for the LICA. }
##'
##'     \item{\code{ott_taxon_name}} { returns a character vector the
##'     Open Tree Taxonomy name for the LICA. }
##'
##'     \item{\code{ott_id}} { returns a numeric vector of the ott id
##'     for the LICA. }
##'
##' }
##' @examples
##' \dontrun{
##' req <- taxonomy_lica(ott_ids=c(515698,590452,409712,643717))
##' tax_rank(req)
##' ott_taxon_name(req)
##' ott_id(req)
##' }
##' @export
taxonomy_lica <- function (ott_ids=NULL, ...) {
    res <- .taxonomy_lica(ott_ids = ott_ids, ...)
    class(res) <- "taxon_lica"
    return(res)
}

############################################################################
## methods                                                                ##
############################################################################

##' Methods for dealing with objects returned by functions dealing
##' with the Taxonomy and the Taxonomic Name Resolution Services APIs.
##'
##' This is the page for the generic methods. See the help pages for
##' \code{\link{taxonomy_taxon}}, \code{\link{taxonomy_lica}}, and
##' \code{\link{tnrs_match_names}} for more information.
##'
##' @title Methods for Taxonomy
##' @param tax an object returned by \code{\link{taxonomy_taxon}},
##'     \code{\link{taxonomy_lica}}, or \code{\link{tnrs_match_names}}
##' @param ... additional arguments (see
##'     \code{\link{tnrs_match_names}})
##' @rdname taxonomy-methods
##' @export

tax_rank <- function(tax) { UseMethod("tax_rank") }

##' @export
##' @rdname taxonomy-methods
ott_taxon_name <- function(tax) { UseMethod("ott_taxon_name") }

##' @export
##' @rdname taxonomy-methods
ott_id <- function(tax, ...) { UseMethod("ott_id") }

##' @export
##' @rdname taxonomy-methods
synonyms <- function(tax, ...) { UseMethod("synonyms") }


### methods for taxonomy_taxon ---------------------------------------------

##' @export
##' @rdname taxonomy_taxon
tax_rank.taxon_info <- function(tax) {
    vapply(tax, function(x) x[["rank"]], character(1))
}

##' @export
##' @rdname taxonomy_taxon
ott_taxon_name.taxon_info <- function(tax) {
    vapply(tax, function(x) x[["ot:ottTaxonName"]], character(1))
}

##' @export
##' @rdname taxonomy_taxon
synonyms.taxon_info <- function(tax, ...) {
    sapply(tax, function(x) {
        tt <- x[["synonyms"]]
        unlist(as.character(tt))
    }, simplify = FALSE)
}

### methods for taxonomy_lica ----------------------------------------------

##' @export
##' @rdname taxonomy_lica
tax_rank.taxon_lica <- function(tax) {
    tax[["lica"]][["rank"]]
}

##' @export
##' @rdname taxonomy_lica
ott_taxon_name.taxon_lica <- function(tax) {
    tax[["lica"]][["ot:ottTaxonName"]]
}

##' @export
##' @rdname taxonomy_lica
ott_id.taxon_lica <- function(tax, ...) {
    tax[["lica"]][["ot:ottId"]]
}
