

tile.raster.autozoom <- function(bbox, epsg, minnumtiles=12) {
  for(zoom in 1:19) {
    tiles <- tiles.bybbox(bbox, zoom, epsg) #always latlon
    if(nrow(tiles) >= minnumtiles) {
      return(zoom)
    }
  }
  return(20)
}


#' Get Open Street Map Tiles As A RasterStack
#'
#' Get Open Street Map tiles as RasterStack object (requires package
#' \code{raster} to be installed).
#'
#' @param x A bounding box as generated by \code{sp::bbox()} or \code{prettymapr::searchbbox()}.
#'          Must be in lon/lat (epsg:4326)! Alternatively, pass a \code{Spatial*} object to use
#'          the bounding box of that
#' @param zoomin The amount by which to adjust the automatically calculated zoom (or
#' manually specified if the \code{zoom} parameter is passed). Use +1 to zoom in, or -1 to zoom out.
#' @param zoom Manually specify the zoom level (not recommended; adjust \code{zoomin} instead.
#' @param type A map type; one of that returned by \link{osm.types}. User defined types are possible
#' by defining \code{tile.url.TYPENAME <- function(xtile, ytile, zoom){}} and passing TYPENAME
#' as the \code{type} argument.
#' @param forcedownload \code{TRUE} if cached tiles should be re-downloaded. Useful if
#' some tiles are corrupted.
#' @param cachedir The directory in which tiles should be cached. Defaults to \code{getwd()/rosm.cache}.
#' @param projection A map projection in which to reproject the RasterStack as generated by \code{CRS()} or
#'                  \code{Spatial*@@proj4string}. If a \code{Spatial*} object is passed as the first argument,
#'                  this argument will be ignored.
#' @param crop \code{TRUE} if results should be cropped to the specified bounding box (see \code{x}),
#'              \code{FALSE} otherwise.
#' @param filename A filename to which the raster should be written (see \code{raster::writeRaster()}). Use a ".tif"
#'                extension to write as a GeoTIFF.
#' @param ... Arguments passed on to \code{raster::writeRaster()} if \code{filename} is specified.
#' @return A projected RasterStack of the fused tiles.
#' @examples
#' \donttest{
#' library(cartography)
#' library(raster)
#' library(prettymapr)
#'
#' ns <- makebbox(47.2, -59.7, 43.3, -66.4)
#' x <- osm.raster(ns, projection=CRS("+init=epsg:26920"), crop=TRUE)
#' #plot using plotRGB (from the raster package)
#' plotRGB(x)
#'
#' #use a Spatial* object as the first argument to automatically set the bounding
#' #box and projection
#' data(nuts2006)
#' spdf <- nuts0.spdf[nuts0.spdf$id=="DE",]
#' x <- osm.raster(spdf, type="thunderforestlandscape")
#' plotRGB(x)
#'
#' #write to disk by passing a filename argument (use .tif extension to write GeoTIFF)
#' osm.raster(ns, projection=CRS("+init=epsg:26920"), crop=TRUE, filename="ns.tif")
#'
#' #can also write Raster* objects using osm.raster
#' osm.raster(x, filename="germany.tif")
#'
#' }
#' @export
osm.raster <- function(x, zoomin=0, zoom=NULL, type="osm", forcedownload=FALSE, cachedir=NULL,
                       projection=NULL, crop=FALSE, filename=NULL, ...) {
  if(!("raster" %in% rownames(utils::installed.packages()))) {
    stop("package 'raster' must be installed for call to osm.raster()")
  }

  if(!is.null(filename)) {
    if(methods::is(x, "Raster")) {
      return(invisible(raster::writeRaster(x, filename=filename, datatype="INT1U", ...)))
    } else {
      return(invisible(raster::writeRaster(osm.raster(x=x, zoomin=zoomin, zoom=zoom,
                                       type=type, forcedownload=forcedownload,
                                       cachedir=cachedir, projection=projection,
                                       crop=crop, filename=NULL),
                filename=filename, datatype="INT1U", ...)))
    }
  }

  if(methods::is(x, "Spatial")) {
    crop.bbox <- sp::bbox(x)
    if(!is.na(rgdal::CRSargs(x@proj4string))) {
      projection <- x@proj4string
      lookup.bbox <- sp::bbox(sp::spTransform(x, sp::CRS("+init=epsg:4326")))
    }
  } else if(!is.null(projection)) {
    lookup.bbox <- x
    crop.bbox <- .projectbbox(lookup.bbox, projection=projection)
  } else {
    lookup.bbox <- x
    crop.bbox <- .projectbbox(lookup.bbox, projection=sp::CRS("+init=epsg:3857"))
  }

  if(is.null(zoom)) {
    zoom <- tile.raster.autozoom(lookup.bbox, epsg=4326)
  }

  zoom <- min(zoom+zoomin, tile.maxzoom(type))
  zoom <- max(1, zoom) #global min zoom set to 1
  message("Zoom: ", zoom)

  tiles <- tiles.bybbox(lookup.bbox, zoom, epsg=4326)
  tile.download(tiles, zoom, type=type, forcedownload=forcedownload, cachedir=cachedir)

  fused <- tile.fuse(tiles, zoom, type=type, epsg=3857, cachedir=cachedir)
  arr <- fused[[1]]
  box <- fused[[2]]
  nbrow <- dim(arr)[1]
  nbcol <- dim(arr)[2]
  bands <- dim(arr)[3]
  .makeraster <- function(i) {
    raster::raster(matrix(arr[,,i]*255, nrow = nbrow, ncol = nbcol),
                   xmn = box[1,1], xmx = box[1,2],
                   ymn = box[2,1], ymx = box[2,2],
                   crs = "+init=epsg:3857")
  }
  if(bands==3) {
    rstack <- raster::stack(.makeraster(1), .makeraster(2), .makeraster(3))
  } else if(bands==4) {
    rstack <- raster::stack(.makeraster(1), .makeraster(2), .makeraster(3), .makeraster(4))
  } else {
    stop("Invalid number of bands in image: ", bands)
  }

  if(!is.null(projection)) {
    if(crop) {
      return(osm.proj(rstack, projection, crop.bbox))
    } else {
      return(osm.proj(rstack, projection))
    }
  } else {
    if(crop) {
      return(osm.proj(rstack, sp::CRS("+init=epsg:3857"), crop.bbox))
    } else {
      return(rstack)
    }
  }
}


# @title Project an OSM RasterStack
# @name osm.proj
# @description  Project an OSM RasterStack
# @param osm.raster A RasterStack
# @param projection A projection as generated by \code{Spatial*@@proj4String} or \code{sp::CRS()}.
# @param crop.bbox (optional) a bbox to crop the image to (in projected coordinates)
# @examples
# \donttest{
# library(cartography)
# library(raster)
# library(prettymapr)
#
# data(nuts2006)
# spdf <- nuts0.spdf[nuts0.spdf$id=="DE",]
# spdf2 <- spTransform(spdf, CRSobj = CRS("+init=epsg:4326"))
# x <- osm.raster(bbox(spdf2), type="thunderforestlandscape")
# x2 <- osm.proj(x, spdf@@proj4string, crop.bbox=bbox(spdf))
# osm.proj.plot(x2)
#
# ns <- makebbox(47.2, -59.7, 43.3, -66.4)
# x <- osm.raster(ns)
# osm.raster.plot(osm.proj(x, CRS("+init=epsg:26920")))
# }

osm.proj <- function(osm.raster, projection, crop.bbox=NULL) {

  rstackproj <- raster::projectRaster(osm.raster, crs = projection)
  rstackproj@data@values[rstackproj@data@values > 255 ] <- 255
  rstackproj@data@values[rstackproj@data@values < 0 ] <- 0

  if(!is.null(crop.bbox)) {
    k <- min(c(0.052 * (crop.bbox[2,2] - crop.bbox[2,1]),
               0.052 * (crop.bbox[1,2] - crop.bbox[1,1])))
    crop.bbox[2,2] <- crop.bbox[2,2] + k
    crop.bbox[1,1] <- crop.bbox[1,1] - k
    crop.bbox[2,1] <- crop.bbox[2,1] - k
    crop.bbox[1,2] <- crop.bbox[1,2] + k

    return(raster::crop(rstackproj, crop.bbox))
  } else {
    return(rstackproj)
  }
}

