% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ci.R
\name{ci}
\alias{ci}
\title{Calculates different type of concentration indexes}
\usage{
ci(
  ineqvar,
  outcome,
  weights = NULL,
  type = c("CI", "CIg", "CIc", "CIw"),
  method = c("linreg_delta", "linreg_convenience", "cov_convenience", "direct"),
  df_correction = TRUE,
  robust_se = FALSE,
  rse_type = "HC3"
)
}
\arguments{
\item{ineqvar}{Used for ranking, usually relates to the socioeconomic position, for example income.}

\item{outcome}{The variable in which the inequality should be measures, for example health.}

\item{weights}{Optional, used to weigh the observations. Defaults to equal weights for all observations.}

\item{type}{Character, the type of concentration index to be calculated: relative concentration index (\code{CI}, default), generalized concentration index (\code{CIg}), concentration index with Erreygers Correction \code{CIc}, or Wagstaff concentration index suitable for bounded and binary outcomes \code{CIw}}

\item{method}{Character, defines the calculation method. One of:
\itemize{
\item \code{linreg_delta}: Based on linear regression without transforming the left hand side variable. Computes correct standard errors that take into account the sampling variability of the estimate of the mean (O’Donnell et al. 2008, Owen et al. 2016)
\item \code{linreg_convenience}): Based on simpler regression with transformed left hand side variable. Standard errors do not take into account thee sampling variability of the estimate of the mean(O’Donnell et al. 2008, Owen et al. 2016)
\item \code{cov_convenience}: Based on covariance. Equivalent to \code{linreg_convenience} (O’Donnell et al. 2008, Owen et al. 2016)
\item \code{direct}: Using direct formula, standard errors do no take weighting appropriately into account  (O’Donnell et al. 2008, Kakwani et al. 1997)
}}

\item{df_correction}{If \code{TRUE} (default), calculates the concentration index based on the population variance (derived from the sample variance).}

\item{robust_se}{Uses robust standard errors if \code{TRUE}. Only available for the \verb{linreg_*} type methods. Requires the \code{sandwich} package.}

\item{rse_type}{Character, type argument for the \code{vcovHC()}. \code{HC3}' is suggested as default, set to \code{HC1} for Stata compatibility. See \code{?sandwich::vcovHC()} for options.}
}
\value{
An  S3 object of class \code{hci}. Contains:
\itemize{
\item \code{concentration_index} The concentration index
\item \code{type} The type
\item \code{method} The method used for calculation
\item \code{variance} The variance,used for calculation of confidence intervals
\item \code{fractional_rank} Computed fractional rank \code{NA}
\item \code{outcome} Outcome after removing \code{NA}
\item \code{call} Call signature
\item \code{n} Number of observations after removing \code{NA}
\item \code{robust_se} Were robust standard errors calculated?
\item \code{rse_type} Type of robust standard errors.
\item \code{df_correction} Do the degrees of freedom correspond to a sample?
}
}
\description{
This function calculates the relative concentration index (Kakwani et al.), the generalized concentration index (Clarke et al., 2002), the Wagstaff index for bounded variables (Owen et al. 2016), and the concentration index with Erreygers' correction (Erreygers et al., 2009).  It returns an object of class \code{hci} for which confidence intervals, summaries and plots are defined.
}
\examples{
# Direct
data(housing)
ci.bmi <- ci(ineqvar = housing$income, outcome = housing$bmi, method = "direct")
summary(ci.bmi)

# retrieve value
ci.bmi$concentration_index

# obtain confidence intervals
confint(ci.bmi, level = 0.95) 

plot(ci.bmi)

# Wagstaff type with binary outcome and robust standard errors 
# that should correspond to Stata (depends on 'sandwich'):


ci.bmi.b <- ci(housing$income, housing$high.bmi, type = "CIw", robust_se = TRUE, 
   rse_type = "HC1")

}
\references{
Clarke, P. M., Gerdtham, U. G., Johannesson, M., Bingefors, K., & Smith, L. (2002). On the measurement of relative and absolute income-related health inequality. Social Science & Medicine, 55(11), 1923-1928

Erreygers, G. (2009). Correcting the concentration index. Journal of health economics, 28(2), 504-515

Kakwani, N., Wagstaff, A., & Van Doorslaer, E. (1997). Socioeconomic inequalities in health: measurement, computation, and statistical inference. Journal of econometrics, 77(1), 87-103.

O'Donnel, O., O'Neill S., Van Ourti T., & Walsh B. (2016). Conindex: Estimation of Concentration Indices. The Stata Journal 16(1): 112-138.

O’Donnell, O., Van Doorslaer, E. , Wagstaff, A., Lindelow, M., 2008. Analyzing Health Equity Using Household Survey Data: A Guide to Techniques and Their Implementation, World Bank Publications. The World Bank.
}
