#' Load plain-text data from a URL (either http or https)
#' 
#' \code{source_data} loads plain-text formatted data stored at a URL (both http and https) into R.
#' @param url The plain-text formatted data's URL.
#' @param sha1 Character string of the file's SHA-1 hash, generated by \code{source_data}. Note if you are using data stored using Git, this is not the file's commit SHA-1 hash.
#' @param sep The separator method for the data. For example, to load comma-separated values data (CSV) use \code{sep = ","} (the default). To load tab-separated values data (TSV) use \code{sep = "\t"}.
#' @param quote  the set of quoting characters. To disable quoting altogether, use \code{quote = ""}. See \code{\link{scan}} for the behaviour on quotes embedded in quotes. 
#' @param header Logical, whether or not the first line of the file is the header (i.e. variable names). The default is \code{header = TRUE}.
#' @param stringsAsFactors logical. Should character vectors be converted to factors? Note that this is overridden by \code{as.is} and \code{colClasses}, both of which allow finer control.
#' @param ... additional arguments passed to \code{\link{read.table}}.

#' @return a data frame
#' @details Loads plain-text data (e.g. CSV, TSV) data from a URL. Works with both HTTP and HTTPS sites. Note: the URL you give for the \code{url} argument must be for the RAW version of the file. The function should work to download plain-text data from any secure URL (https), though I have not verified this.
#'
#' From the source_url documentation: "If a SHA-1 hash is specified with the sha1 argument, then this function will check the SHA-1 hash of the downloaded file to make sure it matches the expected value, and throw an error if it does not match. If the SHA-1 hash is not specified, it will print a message displaying the hash of the downloaded file. The purpose of this is to improve security when running remotely-hosted code; if you have a hash of the file, you can be sure that it has not changed."
#' @examples
#' # Download electoral disproportionality data stored on GitHub
#' # Note: Using shortened URL created by bitly
#' DisData <- source_data("http://bit.ly/156oQ7a")
#'
#' # Check to see if SHA-1 hash matches downloaded file
#' DisDataHash <- source_data("http://bit.ly/Ss6zDO", 
#'  sha1 = "dc8110d6dff32f682bd2f2fdbacb89e37b94f95d")
#' @source Based on source_url from the Hadley Wickham's devtools package.
#' @seealso \link{httr} and \code{\link{read.table}}
#' @importFrom digest digest
#' @importFrom httr GET stop_for_status text_content content
#' @export

source_data <-function(url, sha1 = NULL, sep = ",", quote = "\"'", header = TRUE, stringsAsFactors = default.stringsAsFactors(), ...)
{
  stopifnot(is.character(url), length(url) == 1)
  
  temp_file <- tempfile()
  on.exit(unlink(temp_file))
  
  request <- GET(url)
  stop_for_status(request)
  writeBin(content(request, type = "raw"), temp_file)
  
  file_sha1 <- digest(file = temp_file, algo = "sha1")
  
  if (is.null(sha1)) {
    message("SHA-1 hash of file is ", file_sha1)
  }
  else {
    if (!identical(file_sha1, sha1)) {
      stop("SHA-1 hash of downloaded file (", file_sha1, 
           ")\n  does not match expected value (", sha1, 
           ")", call. = FALSE)
    }
  }
  
  read.table(temp_file, sep = sep, quote = quote, header = header, stringsAsFactors = stringsAsFactors, ...)
}
