% Generated by roxygen2 (4.1.0): do not edit by hand
% Please edit documentation in R/fpca2s.R
\name{fpca2s}
\alias{fpca2s}
\title{Functional principal component analysis by a two-stage method}
\usage{
fpca2s(Y = NULL, ydata = NULL, argvals = NULL, npc = NA,
  center = TRUE, smooth = TRUE)
}
\arguments{
\item{Y}{data matrix (rows: observations; columns: grid of eval. points)}

\item{ydata}{a data frame \code{ydata} representing
irregularly observed functions. NOT IMPLEMENTED for this method.}

\item{argvals}{the argument values of the function evaluations in \code{Y},
defaults to a equidistant grid from 0 to 1. See Details.}

\item{npc}{how many smooth SVs to try to extract, if \code{NA} (the default)
the hard thresholding rule of Donoho, Gavish (2013) is used (see Details,
References).}

\item{center}{center \code{Y} so that its column-means are 0? Defaults to
\code{TRUE}}

\item{smooth}{logical; defaults to TRUE, if NULL, no smoothing of
eigenvectors.}
}
\value{
an \code{fpca} object like that returned from \code{\link{fpca.sc}},
 with entries \code{Yhat}, the smoothed trajectories, \code{Y}, the observed
 data, \code{scores}, the estimated FPC loadings, \code{mu}, the column means
 of \code{Y} (or a vector of zeroes if \code{!center}),  \code{efunctions},
 the estimated smooth FPCs (note that these are orthonormal vectors, not
 evaluations of orthonormal functions if \code{argvals} is not equidistant),
 \code{evalues}, their associated eigenvalues, and \code{npc}, the number of
 smooth components that were extracted.
}
\description{
This function performs functional PCA by performing an ordinary singular
value decomposition on the functional data matrix, then smoothing the right
singular vectors by smoothing splines.
}
\details{
Note that \code{fpca2s} computes smoothed orthonormal eigenvectors
 of the supplied function evaluations (and associated scores), not (!)
 evaluations of the smoothed orthormal eigenfunctions. The smoothed
 orthonormal eigenvectors are then rescaled by the length of the domain
 defined by \code{argvals} to have a quadratic integral approximately equal
 to one (instead of crossproduct equal to one), so they approximate the behavior
 of smooth eigenfunctions. If \code{argvals} is not equidistant,
 \code{fpca2s} will simply return the smoothed eigenvectors without rescaling,
 with a warning.
}
\examples{
#### settings
  I <- 50 # number of subjects
  J <- 3000 # dimension of the data
  t <- (1:J)/J # a regular grid on [0,1]
  N <- 4 #number of eigenfunctions
  sigma <- 2 ##standard deviation of random noises
  lambdaTrue <- c(1,0.5,0.5^2,0.5^3) # True eigenvalues

  case = 1
  ### True Eigenfunctions

  if(case==1) phi <- sqrt(2)*cbind(sin(2*pi*t),cos(2*pi*t),
                                   sin(4*pi*t),cos(4*pi*t))
  if(case==2) phi <- cbind(rep(1,J),sqrt(3)*(2*t-1),
                           sqrt(5)*(6*t^2-6*t+1),
                           sqrt(7)*(20*t^3-30*t^2+12*t-1))

  ###################################################
  ########     Generate Data            #############
  ###################################################
  xi <- matrix(rnorm(I*N),I,N);
  xi <- xi\%*\%diag(sqrt(lambdaTrue))
  X <- xi\%*\%t(phi); # of size I by J
  Y <- X + sigma*matrix(rnorm(I*J),I,J)

  results <- fpca2s(Y,npc=4,argvals=t)
  ###################################################
  ####               SVDS               ########
  ###################################################
  Phi <- results$efunctions
  eigenvalues <- results$evalues

  for(k in 1:N){
    if(Phi[,k]\%*\%phi[,k]< 0)
      Phi[,k] <- - Phi[,k]
  }

 ### plot eigenfunctions
 par(mfrow=c(N/2,2))
 seq <- (1:(J/10))*10
 for(k in 1:N){
      plot(t[seq],Phi[seq,k]*sqrt(J),type='l',lwd = 3,
           ylim = c(-2,2),col = 'red',
           ylab = paste('Eigenfunction ',k,sep=''),
           xlab='t',main='SVDS')

      lines(t[seq],phi[seq,k],lwd = 2, col = 'black')
      }
}
\author{
Luo Xiao \email{lxiao@jhsph.edu}, Fabian Scheipl
}
\references{
Xiao, L., Ruppert, D., Zipunnikov, V., and Crainiceanu, C., (2013), Fast
covariance estimation for high-dimensional functional data. (submitted)
\url{http://arxiv.org/abs/1306.5718}.

Gavish, M., and Donoho, D. L.  (2014). The optimal hard threshold for
singular values is 4/sqrt(3).  \emph{IEEE Transactions on Information Theory}, 60(8), 5040--5053.
}
\seealso{
\code{\link{fpca.sc}} and \code{\link{fpca.face}} for FPCA based
on smoothing a covariance estimate; \code{\link{fpca.ssvd}} for another
SVD-based approach.
}

