#' @name exportMetaData
#' @aliases exportMetaData.redcapApiConnection
#' @aliases exportMetaData.redcapDbConnection
#' @export exportMetaData
#' @importFrom DBI dbGetQuery
#' 
#' @title Export Meta Data from a REDCap Database
#' @description Retrieves the meta data for a REDcap database, including 
#' field names, labels, types, formulas, etc.  This file can be used to parse 
#' levels of factors, apply labels, and other data management tasks once the 
#' data are retrieved
#' 
#' @param rcon A REDCap connection object as generated by \code{redcapConnection.}
#' @param ... Arguments to be passed to other methods.
#' 
#' @details A record of this export is placed in the REDCap logging page, 
#' but the file that is exported is not stored in the database.
#' 
#' @author Jeffrey Horner
#' 
#' @references
#' This functionality were originally developed by Jeffrey Horner in the \code{redcap} package.
#' \url{https://github.com/vubiostat/redcap}
#' 
#' Please refer to your institution's API documentation.
#' 
#' Additional details on API parameters are found on the package wiki at
#' \url{https://github.com/nutterb/redcapAPI/wiki/REDCap-API-Parameters}
#' 
#' @examples
#' \dontrun{
#' #*** Note: I cannot provide working examples without
#' #*** compromising security.  Instead, I will try to 
#' #*** offer up sample code with the matching results
#' 
#' #*** Create the connection object
#' rcon <- redcapConnection(url=[YOUR_REDCAP_URL], token=[API_TOKEN])
#' 
#' exportMetaData(rcon)
#' }
#' 

exportMetaData <- function(rcon, ...) UseMethod("exportMetaData")

#' @rdname exportMetaData
#' @export

exportMetaData.redcapDbConnection <- 
function(rcon, ...)
{
.SQL_PROJECT_META_DATA <- "
SELECT 
   field_name, 
   form_name, 
   element_preceding_header as section_header, 
   if(element_type='textarea','notes',if(element_type='select','dropdown',element_type)) as field_type, 
   element_label as field_label, 
   element_enum as select_choices_or_calculations, 
   element_note as field_note,
   if(element_validation_type='int','integer',if(element_validation_type='float','number',element_validation_type)) as text_validation_type_or_show_slider_number, 
   element_validation_min as text_validation_min, 
   element_validation_max as text_validation_max, 
   if(field_phi='1','Y','') as identifier, branching_logic, 
   if(field_req='0','','Y') as required_field, 
   custom_alignment, 
   question_num as question_number
FROM 
   %s
WHERE 
   project_id = %d AND 
   field_name != concat(form_name,'_complete')
ORDER BY field_order"

   suppressWarnings(
   x <- DBI::dbGetQuery(rcon$con,
           sprintf(.SQL_PROJECT_META_DATA,
                   getOption('redcap.metaDataTable'),
                   rcon$project))
   )
   suppressWarnings(
   x$required_field <- as.integer(x$required_field)
   )

   # MySQL likes to escape stuff
   x$select_choices_or_calculations <- gsub('\\\\n','\n',x$select_choices_or_calculations)

   x
}

#' @rdname exportMetaData
#' @export

exportMetaData.redcapApiConnection <-
function(rcon, ...)
{
   x <- apiCall(url=rcon$url, 
                body=list(token=rcon$token, content='metadata',
                          format='csv', returnFormat='csv'),
                config=rcon$config)          
if (x$status_code == 200){
     x <- read.csv(textConnection(as.character(x)), stringsAsFactors=FALSE, na.strings="")
     return(x)
   }
   else stop(paste0(x$status_code, ": ", as.character(x)))
}
