% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pipeline_generated.R
\name{GeneratedPipeline}
\alias{GeneratedPipeline}
\title{Chain multiple transformations and generate argument values per group}
\description{
\Sexpr[results=rd, stage=render]{lifecycle::badge("experimental")}

Build a pipeline of transformations to be applied sequentially.

Generate argument values for selected arguments with a given set of generators.
E.g. randomly generate argument values for each group in a \code{data.frame}.

Groupings are reset between each transformation. See \code{group_cols}.

\strong{Standard workflow}: Instantiate pipeline -> Add transformations -> Apply to data

To apply the same arguments to all groups, see
\code{\link[rearrr:Pipeline]{Pipeline}}.

To apply different but specified argument values to a fixed set of groups,
see \code{\link[rearrr:FixedGroupsPipeline]{FixedGroupsPipeline}}.
}
\examples{
# Attach package
library(rearrr)

# Create a data frame
df <- data.frame(
  "Index" = 1:12,
  "A" = c(1:4, 9:12, 15:18),
  "G" = rep(1:3, each = 4)
)

# Create new pipeline
pipe <- GeneratedPipeline$new()

# Add 2D rotation transformation
# Note that we specify the grouping via `group_cols`
pipe$add_transformation(
  fn = rotate_2d,
  args = list(
    x_col = "Index",
    y_col = "A",
    suffix = "",
    overwrite = TRUE
  ),
  generators = list(degrees = function(){sample.int(360, 1)},
                    origin = function(){rnorm(2)}),
  name = "Rotate",
  group_cols = "G"
)

# Add the `cluster_group` transformation
# Note that this function requires the entire input data
# to properly scale the groups. We therefore specify `group_cols`
# as part of `args`. This works as `cluster_groups()` accepts that
# argument.
# Also note the `.apply` generator which generates a TRUE/FALSE scalar
# for whether the transformation should be applied to the current group
pipe$add_transformation(
  fn = cluster_groups,
  args = list(
    cols = c("Index", "A"),
    suffix = "",
    overwrite = TRUE,
    group_cols = "G"
  ),
  generators = list(
    multiplier = function() {
      0.1 * runif(1) * 3 ^ sample.int(5, 1)
    },
    .apply = function(){sample(c(TRUE, FALSE), 1)}
  ),
  name = "Cluster"
)

# Check pipeline object
pipe

# Apply pipeline to data.frame
# Enable `verbose` to print progress
pipe$apply(df, verbose = TRUE)


## ------------------------------------------------
## Method `GeneratedPipeline$add_transformation`
## ------------------------------------------------

# `generators` is a list of functions for generating
# argument values for a chosen set of arguments
# `.apply` can be used to disable the transformation
generators = list(degrees = function(){sample.int(360, 1)},
                  origin = function(){rnorm(2)},
                  .apply = function(){sample(c(TRUE, FALSE), 1)})
}
\seealso{
Other pipelines: 
\code{\link{FixedGroupsPipeline}},
\code{\link{Pipeline}}
}
\author{
Ludvig Renbo Olsen, \email{r-pkgs@ludvigolsen.dk}
}
\concept{pipelines}
\section{Super class}{
\code{\link[rearrr:Pipeline]{rearrr::Pipeline}} -> \code{GeneratedPipeline}
}
\section{Public fields}{
\if{html}{\out{<div class="r6-fields">}}
\describe{
\item{\code{transformations}}{\code{list} of transformations to apply.}

\item{\code{names}}{Names of the transformations.}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-GeneratedPipeline-add_transformation}{\code{GeneratedPipeline$add_transformation()}}
\item \href{#method-GeneratedPipeline-print}{\code{GeneratedPipeline$print()}}
\item \href{#method-GeneratedPipeline-clone}{\code{GeneratedPipeline$clone()}}
}
}
\if{html}{\out{
<details open><summary>Inherited methods</summary>
<ul>
<li><span class="pkg-link" data-pkg="rearrr" data-topic="Pipeline" data-id="apply"><a href='../../rearrr/html/Pipeline.html#method-Pipeline-apply'><code>rearrr::Pipeline$apply()</code></a></span></li>
</ul>
</details>
}}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GeneratedPipeline-add_transformation"></a>}}
\if{latex}{\out{\hypertarget{method-GeneratedPipeline-add_transformation}{}}}
\subsection{Method \code{add_transformation()}}{
Add a transformation to the pipeline.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GeneratedPipeline$add_transformation(
  fn,
  args,
  generators,
  name,
  group_cols = NULL
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{fn}}{Function that performs the transformation.}

\item{\code{args}}{Named \code{list} with arguments for the \code{`fn`} function.}

\item{\code{generators}}{Named \code{list} of functions for generating argument values
for a single call of \code{`fn`}.

It is possible to include an \emph{apply generator} for deciding whether
the transformation should be applied to the current group or not.
This is done by adding a function with the name \code{`.apply`} to the \code{`generators`} list.
E.g. \code{".apply" = function(){sample(c(TRUE, FALSE), 1)}}.}

\item{\code{name}}{Name of the transformation step. Must be unique.}

\item{\code{group_cols}}{Names of the columns to group the input
data by before applying the transformation.

Note that the transformation function is applied separately to each group (subset).
If the \code{`fn`} function requires access to the entire \code{data.frame}, the
grouping columns should be specified as part of \code{`args`} and
handled by the \code{`fn`} function.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
The pipeline. To allow chaining of methods.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{# `generators` is a list of functions for generating
# argument values for a chosen set of arguments
# `.apply` can be used to disable the transformation
generators = list(degrees = function(){sample.int(360, 1)},
                  origin = function(){rnorm(2)},
                  .apply = function(){sample(c(TRUE, FALSE), 1)})
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GeneratedPipeline-print"></a>}}
\if{latex}{\out{\hypertarget{method-GeneratedPipeline-print}{}}}
\subsection{Method \code{print()}}{
Print an overview of the pipeline.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GeneratedPipeline$print(...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{further arguments passed to or from other methods.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
The pipeline. To allow chaining of methods.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GeneratedPipeline-clone"></a>}}
\if{latex}{\out{\hypertarget{method-GeneratedPipeline-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GeneratedPipeline$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
