% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/inla_rspde.R
\name{rspde.matern.precision}
\alias{rspde.matern.precision}
\title{Precision matrix of the covariance-based rational approximation of
stationary Gaussian Matern random fields}
\usage{
rspde.matern.precision(
  kappa,
  nu,
  tau = NULL,
  sigma = NULL,
  rspde_order,
  dim,
  fem_mesh_matrices,
  only_fractional = FALSE,
  return_block_list = FALSE,
  type_rational_approx = "chebfun"
)
}
\arguments{
\item{kappa}{Range parameter of the covariance function.}

\item{nu}{Shape parameter of the covariance function.}

\item{tau}{Scale parameter of the covariance function. If sigma is
not provided, tau should be provided.}

\item{sigma}{Standard deviation of the covariance function. If tau is
not provided, sigma should be provided.}

\item{rspde_order}{The order of the rational approximation}

\item{dim}{The dimension of the domain}

\item{fem_mesh_matrices}{A list containing the FEM-related matrices. The
list should contain elements c0, g1, g2, g3, etc.}

\item{only_fractional}{Logical. Should only the fractional-order part of
the precision matrix be returned?}

\item{return_block_list}{Logical. For \code{type = "covariance"}, should the
block parts of the precision matrix be returned separately as a list?}

\item{type_rational_approx}{Which type of rational approximation should be
used? The current types are "chebfun", "brasil" or "chebfunLB".}
}
\value{
The precision matrix
}
\description{
\code{rspde.matern.precision} is used for computing the
precision matrix of the
covariance-based rational SPDE approximation of a stationary Gaussian random
fields on \eqn{R^d} with a Matern covariance function
\deqn{C(h) = \frac{\sigma^2}{2^(\nu-1)\Gamma(\nu)}(\kappa h)^\nu
K_\nu(\kappa h)}{C(h) = (\sigma^2/(2^(\nu-1)\Gamma(\nu))(\kappa h)^\nu
K_\nu(\kappa h)}
}
\examples{
set.seed(123)
nobs <- 101
x <- seq(from = 0, to = 1, length.out = nobs)
fem <- rSPDE.fem1d(x)
kappa <- 40
sigma <- 1
d <- 1
nu <- 2.6
tau <- sqrt(gamma(nu) / (kappa^(2 * nu) * (4 * pi)^(d / 2) *
gamma(nu + d / 2)))
op_cov <- matern.operators(
  C = fem$C, G = fem$G, nu = nu, kappa = kappa, sigma = sigma,
  d = 1, m = 2, compute_higher_order = TRUE
)
v <- t(rSPDE.A1d(x, 0.5))
c.true <- matern.covariance(abs(x - 0.5), kappa, nu, sigma)
Q <- rspde.matern.precision(
  kappa = kappa, nu = nu, tau = tau, rspde_order = 2, d = 1,
  fem_mesh_matrices = op_cov$fem_mesh_matrices
)
A <- Diagonal(nobs)
Abar <- cbind(A, A, A)
w <- rbind(v, v, v)
c.approx_cov <- (Abar) \%*\% solve(Q, w)

# plot the result and compare with the true Matern covariance
plot(x, matern.covariance(abs(x - 0.5), kappa, nu, sigma),
  type = "l", ylab = "C(h)",
  xlab = "h", main = "Matern covariance and rational approximations"
)
lines(x, c.approx_cov, col = 2)
}
