% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Class-FreqRep.R
\docType{class}
\name{FreqRep-class}
\alias{FreqRep-class}
\alias{FreqRep}
\title{Class for Frequency Representation.}
\description{
\code{FreqRep} is an S4 class that encapsulates, for a multivariate time
series \eqn{(Y_{t,i})_{t=0,\ldots,n-1}}{Y_0i,\dots,Y_{n-1,i}},
\eqn{i=1,\ldots,d}{i=1,...,d}
the data structures for the storage of a frequency representation. Examples
of such frequency representations include
\itemize{
  \item the Fourier transformation of the clipped time series
        \eqn{(\{I\{Y_{t,i} \leq q\})}{(I{Y_ti <= q})}, or
  \item the weighted \eqn{L_1}{L1}-projection of \eqn{(Y_{t,i})} onto an harmonic
        basis.
}
Examples are realized by implementing a sub-class to
\code{FreqRep}.
Currently, implementations for the two examples mentioned above are available:
    \code{\link{ClippedFT}} and
    \code{\link{QRegEstimator}}.
}
\details{
It is always an option to base the calculations on the pseudo data
\eqn{R_{t,n,i} / n}{R_tni / n} where \eqn{R_{t,n,i}}{R_tnj} denotes the rank of
\eqn{Y_{t,i}}{Y_ti} among \eqn{(Y_{t,i})_{t=0,\ldots,n-1}}{Y_0,\dots,Y_{n-1}}.

To allow for a block bootstrapping procedure a number of \code{B} estimates
determined from bootstrap replications of the time series which are yield by
use of a \code{\link{BootPos}}-object can be stored on initialization.

The data in the frequency domain is stored in the array \code{values}, which
has dimensions \code{(J,P,K,B+1)}, where \code{J} is the number of
\code{frequencies}, \code{P} is the dimension of the time series,
\code{K} is the number of \code{levels} and \code{B} is
the number of bootstrap replications requested on intialization.
In particular, \code{values[j,i,k,1]} corresponds to the time series' frequency
representation with \code{frequencies[j]}, dimension \code{i} and \code{levels[k]}, while
\code{values[j,i,k,b+1]} is the for the same, but determined from the
\code{b}th block bootstrapped replicate of the time series.
}
\section{Slots}{

\describe{
\item{\code{Y}}{The time series of which the frequency representation is to be determined.}

\item{\code{frequencies}}{The frequencies for which the frequency representation will
be determined. On initalization
\code{\link{frequenciesValidator}} is called, so that it
will always be a vector of reals from \eqn{[0,\pi]}{[0,pi]}.
Also, only Fourier frequencies of the form
\eqn{2\pi j / n}{2 pi j / n} with integers \eqn{j} and \eqn{n}
the \code{length(Y)} are allowed.}

\item{\code{levels}}{The levels for which the frequency representation will be
determined. If the flag \code{isRankBased} is set to \code{FALSE},
then it can be any vector of reals. If \code{isRankBased} is set
to \code{TRUE}, then it has to be from \eqn{[0,1]}.}

\item{\code{values}}{The array holding the determined frequency representation. Use a
\code{getValues} method of the relevant subclass to access it.}

\item{\code{isRankBased}}{A flag that is \code{FALSE} if the determined \code{values}
are based on the original time series and \code{TRUE} if it
is based on the pseudo data as described in the Details
section of this topic.}

\item{\code{positions.boot}}{An object of type \code{\link{BootPos}},
that is used to determine the block bootstrapped
replicates of the time series.}

\item{\code{B}}{Number of bootstrap replications to perform.}
}}

\examples{
Y       <- rnorm(32)
freq    <- 2*pi*c(0:31)/32 
levels  <- c(0.25,0.5,0.75)
cFT     <- clippedFT(Y, freq, levels)

plot(cFT)

# Get values for all Fourier frequencies and all levels available.
V.all    <- getValues(cFT)

# Get values for every second frequency available
V.coarse <- getValues(cFT, frequencies = 2*pi*c(0:15)/16, levels = levels)

# Trying to get values on a finer grid of frequencies than available will
# yield a warning and then all values with frequencies closest to that finer
# grid.
V.fine   <- getValues(cFT, frequencies = 2*pi*c(0:63)/64, levels = levels)

# Finally, get values for the available Fourier frequencies from [0,pi] and
# only for tau=0.25
V.part   <- getValues(cFT, frequencies = 2*pi*c(0:16)/32, levels = c(0.25))

# Alternatively this can be phrased like this:
V.part.alt <- getValues(cFT, frequencies = freq[freq <= pi], levels = c(0.25))
}
\keyword{S4-classes}
