#' Save attribute and level labels to a CSV file
#'
#' Saves the attributes and levels (and their order) from a \code{\link{projoint_data}}
#' object, as generated by \code{\link{reshape_projoint}}, to a CSV file. This
#' enables manual reordering and later re-import via \code{\link{read_labels}}.
#'
#' @import readr
#' @import dplyr
#'
#' @param .data A \code{\link{projoint_data}} object.
#' @param .filename A character string giving the name of a CSV file to be written.
#'
#' @return No return value, called for side effects (writes a CSV file).
#'
#' @seealso \code{\link{read_labels}}, \code{\link{reshape_projoint}}
#'
#' @examples
#' \donttest{
#' library(projoint)
#' data(exampleData1)
#' reshaped <- reshape_projoint(
#'   exampleData1,
#'   .outcomes = c(paste0("choice", 1:8), "choice1_repeated_flipped")
#' )
#' tmpfile <- tempfile(fileext = ".csv")
#' save_labels(reshaped, tmpfile)
#' readLines(tmpfile, n = 5)  # show first few lines
#' }
#'
#' @export
save_labels <- function(
    .data,
    .filename
){
  
  (.data$labels |> 
      dplyr::select(level_id, attribute, level) |> 
      dplyr::arrange(level_id) |> 
      dplyr::mutate(order = row_number())
  )|>
    readr::write_csv(.filename)
}  