#' plot.phenology plots the phenology.
#' @title Plot the phenology from a result.
#' @author Marc Girondot
#' @return A list with four objects: synthesis is a data.frame with global estimate of nesting.\cr
#' details_MCMC, details_ML and details_mean are lists with day by day information for each series.
#' @param x A result file generated by fit_phenology
#' @param ... Parameters used by plot
#' @param series Name or number of series to be plotted or 'all'
#' @param moon If TRUE, the moon phase is ploted. Default is FALSE
#' @param replicate.CI Number of replicates for estimation of confidence interval
#' @param resultmcmc A mcmc object
#' @param chain The number of chain to be used in resultmcmc
#' @param replicate.CI.mcmc Number of iterations to be used or "all"
#' @param level Level to estimate confidence interval or credibility interval
#' @param plot.objects What to plot?
#' @param col.ML Color of the ML mean curve
#' @param col.SD Color of the SD curve (distribution of observations)
#' @param col.MCMC.quantiles Color of the quantiles curve based on mcmc
#' @param col.ML.quantiles Color of the SE curve based on ML
#' @param col.observations Color of the points
#' @param col.grouped.observations Color of the lines indicating grouped observations
#' @description The function plot.phenology plots the phenology graph from a result object.\cr
#' If cofactors have been added, the plot does not show their effects.\cr
#' plot.objects can be "observations", "ML" for maximum likelihood, "ML.SD" for dispersion of observations, "ML.quantiles" or "MCMC.quantiles" if a mcmc object is given
#' @examples
#' \dontrun{
#' library(phenology)
#' # Read a file with data
#' Gratiot <- read.delim("http://max2.ese.u-psud.fr/epc/conservation/BI/Complete.txt", header=FALSE)
#' data(Gratiot)
#' # Generate a formatted list nammed data_Gratiot 
#' data_Gratiot <- add_phenology(Gratiot, name = "Complete", 
#' 		reference = as.Date("2001-01-01"), format="%d/%m/%Y")
#' # Generate initial points for the optimisation
#' parg <- par_init(data_Gratiot, fixed.parameters=NULL)
#' # Run the optimisation
#' result_Gratiot <- fit_phenology(data=data_Gratiot, 
#' 		fitted.parameters=parg, fixed.parameters=NULL)
#' data(result_Gratiot)
#' # Plot the phenology and get some stats
#' output <- plot(result_Gratiot)
#' # Plot only part of the nesting season
#' ptoutput <- plot(result_Gratiot, xlim=c(as.Date("2001-03-01"),as.Date("2001-08-31")))
#' # Use month names in English
#' Sys.setlocale(category = "LC_TIME", locale = "en_GB.UTF-8")
#' output <- plot(result_Gratiot)
#' # set back the month name in local R language
#' Sys.setlocale(category = "LC_TIME", locale = "")
#' # plot based on quantiles of mcmc object
#' plot(result_Gratiot, resultmcmc=result_Gratiot_mcmc, 
#'             plot.objects=c("observations", "MCMC.quantiles"))
#' plot(result_Gratiot, resultmcmc=result_Gratiot_mcmc, 
#'             plot.objects=c("observations", "ML.SD", "ML.quantiles"))
#' plot(result_Gratiot, resultmcmc=result_Gratiot_mcmc, 
#'             plot.objects=c("observations", "ML.SD", "MCMC.quantiles"))
#' plot(result_Gratiot, resultmcmc=result_Gratiot_mcmc, 
#'             plot.objects=c("observations", "ML.quantiles", "MCMC.quantiles"))
#' }
#' @method plot phenology
#' @export

#plot.phenology <- function(x, ...) {

plot.phenology <- 
  function(x, ..., 
           series="all", moon=FALSE, replicate.CI=10000, 
           resultmcmc = NULL,
           chain = 1,
           replicate.CI.mcmc = "all",
           level=0.95, 
           plot.objects = c("observations", "ML", "ML.SD", "ML.quantiles", "MCMC.quantiles"), 
           col.ML="black", 
           col.SD="red", 
           col.MCMC.quantiles="purple",
           col.ML.quantiles="black",
           col.observations = "black", 
           col.grouped.observations = "green") {
    
    # x=NULL; series="all"; moon=FALSE; level=0.95; replicate.CI=1000; progressbar=TRUE; growlnotify=TRUE; show.plot=TRUE; resultmcmc = NULL; chain = 1; replicate.CI.mcmc = "all"; plot.objects = c("observations", "ML", "ML.SD", "ML.quantiles", "MCMC.quantiles"); col.ML="black"; col.SD="red"; col.MCMC.quantiles="purple"; col.ML.quantiles="black"; col.observations = "black"; col.grouped.observations = "green"
    # x <- result_Gratiot
    # p3p <- ifelse(class(try(list(...), silent=TRUE))=="try-error", list(), list(...))
    p3p <- list(...)
    
    # result <- x
    
    data <- x$data
    
    out <- summary(object = x, resultmcmc=resultmcmc, 
                   series=series, replicate.CI=replicate.CI, 
                   replicate.CI.mcmc=replicate.CI.mcmc, 
                   level=level, chain=chain, print=FALSE)

    
    # kseries <- 1
    for(nmser in rownames(out$synthesis)) {
      
      reference <- data[[nmser]]$Date[1]-data[[nmser]]$ordinal[1]+1
      nday <- ifelse(as.POSIXlt(reference+365)$mday==as.POSIXlt(reference)$mday, 365, 366)
      
      vmaxx <- c(reference, reference+nday-1)
      
      vmaxy <- c(0, 0.1)
          if (any((is.na(data[[nmser]]$ordinal2)))) {
      vmaxy[2] <- max(data[[nmser]]$nombre[(is.na(data[[nmser]]$ordinal2)) & 
                                             (!is.na(data[[nmser]]$nombre))])
      }

      vmaxy[2] <- max(vmaxy[2], out$details_ML[[nmser]]["97.5%", ], out$details_Mean[[nmser]]["SD.High",])

        x <- seq(from=reference, to=reference+nday-1, by="1 day")
      

        # je prépare une base
        par(new=FALSE)
        pnp <- modifyList(list(xlab="Months", ylab="Counts", main=nmser, 
                               pch=16, cex=0.5, xlim=vmaxx, ylim=vmaxy, type="n", bty="n"), p3p)
        do.call("plot", modifyList(pnp, list(x=x, y=rep(0, length(x)))))
      
      if (moon) {
        moony<-vmaxy[2]*1.06
        mp<-moon.info(x, phase=TRUE)
        mpT1<-ifelse((mp!="FM") | (is.na(mp)), FALSE, TRUE)
        mpT2<-ifelse((mp!="NM") | (is.na(mp)), FALSE, TRUE)
        
        xnewmoon <- ifelse(x[mpT1]>=ScalePreviousPlot()$xlim["begin"] & x[mpT1]<=ScalePreviousPlot()$xlim["end"], TRUE, FALSE)
        xfullmoon <- ifelse(x[mpT2]>=ScalePreviousPlot()$xlim["begin"] & x[mpT2]<=ScalePreviousPlot()$xlim["end"], TRUE, FALSE)
        points(x[mpT1][xnewmoon], rep(moony, length(x[mpT1]))[xnewmoon], cex=1, bg="black", col="black", pch=21, xpd=TRUE)
        points(x[mpT2][xfullmoon], rep(moony, length(x[mpT2]))[xfullmoon], cex=1, bg="white", col="black", pch=21, xpd=TRUE)
      }
      
      if (!is.null(data) & any(plot.objects == "observations")) {
          points(x = data[[nmser]]$Date[is.na(data[[nmser]]$Date2)], 
                 y=data[[nmser]]$nombre[is.na(data[[nmser]]$Date2)], 
                 pch=16, col=col.observations, cex=0.5)

          for(i in 1:dim(data[[nmser]])[1]) {
        
            if (!is.na(data[[nmser]]$ordinal2[i])) {
              x0<-data[[nmser]]$Date[i]
              x1<-data[[nmser]]$Date2[i]
              lgt01<-as.numeric(data[[nmser]]$Date2[i]-data[[nmser]]$Date[i]+1)
              y0<-data[[nmser]]$nombre[i]/lgt01
              y1<-y0
              segments(x0, y0, x1=x1, y1=y1, col=col.grouped.observations, lwd=2)
            }
          }
      }
      

        if (any(plot.objects == "ML")) {
          lines(x=x, 
                y=out$details_Mean[[nmser]]["Mean", ], col=col.ML)
        }
       if (any(plot.objects == "ML.SD")) {
          lines(x=x, 
                y=out$details_Mean[[nmser]]["SD.Low", ], lty=2, col=col.SD)
          lines(x=x, 
                y=out$details_Mean[[nmser]]["SD.High", ], lty=2, col=col.SD)
        } 
        if (any(plot.objects == "ML.quantiles") & !identical(NA, out$details_ML[[nmser]])) {
          lines(x=x, 
                y=out$details_ML[[nmser]]["2.5%", ], lty=2, col=col.ML.quantiles)
          lines(x=x, 
                y=out$details_ML[[nmser]]["50%", ], lty=1, col=col.ML.quantiles)
          lines(x=x, 
                y=out$details_ML[[nmser]]["97.5%", ], lty=2, col=col.ML.quantiles)
        }
        if (any(plot.objects == "MCMC.quantiles") & !identical(NA, out$details_mcmc[[nmser]])) {
          lines(x=x, 
                y=out$details_mcmc[[nmser]]["2.5%", ], lty=2, col=col.MCMC.quantiles)
          lines(x=x, 
                y=out$details_mcmc[[nmser]]["50%", ], lty=1, col=col.MCMC.quantiles)
          lines(x=x, 
                y=out$details_mcmc[[nmser]]["97.5%", ], lty=2, col=col.MCMC.quantiles)
        }
      }

    

    class(out) <- "phenologyout"
    
    return(invisible(out))
    
  }
