#' fit_phenology fits parameters to timeseries.
#' @title Fit the phenology parameters to timeseries of counts.
#' @author Marc Girondot
#' @return Return a list of with data and result
#' @param data A dataset generated by add_format
#' @param parametersfixed Set of fixed parameters
#' @param parametersfit Set of parameters to be fitted
#' @param trace If 1, it shows the progression of fit; 0 is silent (don't be afraid if it is long !)
#' @param maxit Number of iterations for search before checking if it converges. If it does not converge, it will continue to search for. Default is 500.
#' @param method_incertitude 'multinomial' [default] is the correct one from a statistical point of view;\cr
#'                           'binomial' is an aproximate method more rapid and using less memory;\cr
#'                           'sum' is an alternative more rapid but potentially biased (not sure).
#' @param zero_counts example c(TRUE, TRUE, FALSE) indicates whether the zeros have 
#'                    been recorder for each of these timeseries. Defaut is TRUE for all.
#' @param hessian If FALSE does not estimate se of parameters
#' @param silent If TRUE does not show any message
#' @param growlnotify If False, does not send growl notification
#' @description Function of the package phenology to fit parameters to timeseries.\cr
#' To fit data, the syntaxe is :\cr
#' Result<-fit_phenology(data=dataset, parametersfit=par, parametersfixed=pfixed, trace=1, method_incertitude=2, zero_counts=TRUE, hessian=TRUE)\cr
#' or if no parameter is fixed :\cr
#' Result<-fit_phenology(data=dataset, parametersfit=par)\cr
#' Add trace=1 [default] to have information on the fit progression or trace=0 to hide information on the fit progression.\cr
#' method_incertitude='multinomial' [default] is the correct one from a statistical point of view.\cr
#' method_incertitude='binomial' is an alternative more rapid.\cr
#' method_incertitude='sum' is an aproximate method more rapid and using less memory.\cr
#' zero_counts=c(TRUE, TRUE, FALSE) indicates whether the zeros have been recorded for each of these timeseries. Defaut is TRUE for all.\cr
#' hessian=FALSE does not estimate se of parameters.
#' @examples
#' \dontrun{
#' library(phenology)
#' # Read a file with data
#' Gratiot<-read.delim("http://max2.ese.u-psud.fr/epc/conservation/BI/Complete.txt", header=FALSE)
#' data(Gratiot)
#' # Generate a formatted list nammed data_Gratiot 
#' data_Gratiot<-add_phenology(Gratiot, name="Complete", 
#' 		reference=as.Date("2001-01-01"), format="%d/%m/%Y")
#' # Generate initial points for the optimisation
#' parg<-par_init(data_Gratiot, parametersfixed=NULL)
#' # Run the optimisation
#' result_Gratiot<-fit_phenology(data=data_Gratiot, 
#' 		parametersfit=parg, parametersfixed=NULL, trace=1)
#' data(result_Gratiot)
#' # Plot the phenology and get some stats
#' output<-plot(result_Gratiot)
#' }
#' @export


fit_phenology <-
function(data=file.choose(), parametersfit=NULL, parametersfixed=NULL, 
         trace=1, maxit=500, method_incertitude="multinomial", 
         zero_counts=TRUE, hessian=TRUE, silent=FALSE, growlnotify=TRUE) {

# data=NULL; parametersfit=NULL; parametersfixed=NA; trace=1; maxit=500; method_incertitude="multinomial"; zero_counts=TRUE; hessian=TRUE; silent=FALSE; growlnotify=TRUE
# data=lf; parametersfit=parg; parametersfixed=pfixed; trace=1

#  if (is.null(parametersfixed)) {parametersfixed<-NA}

if (method_incertitude!="sum" & method_incertitude!="binomial" & method_incertitude!="multinomial" 
	& method_incertitude!=0 & method_incertitude!=1 & method_incertitude!=2) {
  stop("method_incertitude must be 'sum', 'binomial' or 'multinomial'")
}


if (class(data)=="character") {
# j'ai utilisé le file.choose
	data <- lapply(data,readLines, warn=FALSE)
}

if (class(data)!="phenologydata") {
  print("Data should have been formated first using the function add_phenology(). I do it now.")
  data <- add_phenology(data)
}

if (is.null(parametersfit)) {
	print("No initial parameters set has been defined. I estimate one set using par_init().")
	parametersfit <- par_init(data, parametersfixed=parametersfixed)
}


	
if (length(zero_counts)==1) {zero_counts<-rep(zero_counts, length(data))}
if (length(zero_counts)!=length(data)) {
	stop("zero_counts parameter must be TRUE (the zeros are used for all timeseries) or FALSE (the zeros are not used for all timeseries) and with the same number of logical values (TRUE or FALSE) than the number of series analyzed.")
}


	repeat {
		resul<-optim(parametersfit, .Lnegbin, 
		             pt=list(data=data, fixed=parametersfixed, incertitude=method_incertitude, zerocounts=zero_counts) , 
		             method="BFGS",
		             control=list(trace=trace, REPORT=1, maxit=maxit),
		             hessian=FALSE)
#		resul<-optim(parametersfit, phenology:::.Lnegbin, pt=list(data=data, fixed=parametersfixed, incertitude=method_incertitude, zerocounts=zero_counts) , method="BFGS",control=list(trace=trace, REPORT=1, maxit=maxit),hessian=FALSE)
		if (resul$convergence==0) break
		parametersfit<-resul$par
		if (!silent) print("Convergence is not acheived. Optimization continues !")
	}
	
	resfit<-resul$par
	resfit[substr(names(resfit), 1, 4)=="Peak"]<-abs(resfit[substr(names(resfit), 1, 4)=="Peak"])
	resfit["Theta"]<-abs(resfit["Theta"])
	resfit["PMinE"]<-abs(resfit["PMinE"])
	resfit["PMinB"]<-abs(resfit["PMinB"])
	resfit["Flat"]<-abs(resfit["Flat"])
	resfit[substr(names(resfit), 1, 6)=="Length"]<-abs(resfit[substr(names(resfit), 1, 6)=="Length"])
	resfit[substr(names(resfit), 1, 3)=="Min"]<-abs(resfit[substr(names(resfit), 1, 3)=="Min"])
	resfit[substr(names(resfit), 1, 3)=="Max"]<-abs(resfit[substr(names(resfit), 1, 3)=="Max"])
	resfit<-resfit[!is.na(resfit)]
	if (!silent) cat("Fit done!\n")
	if (!silent) cat(paste("-Ln L=", format(resul$value, digits=max(3, trunc(log10(resul$value))+4)), "\n", sep=""))
	if (hessian) {
	if (!silent) cat("Estimation of the standard error of parameters. Be patient please.\n")
	
	resul<-optim(resfit, .Lnegbin, pt=list(data=data, fixed=parametersfixed, 
	incertitude=method_incertitude, zerocounts=zero_counts), method="BFGS",
	control=list(trace=0, REPORT=1, maxit=10),hessian=TRUE)

	resfit<-resul$par

	mathessian<-resul$hessian
	inversemathessian <- try(solve(mathessian), silent=TRUE)
	if (substr(inversemathessian[1], 1, 5)=="Error") {
		if (!silent) print("Error in the fit; probably one or more parameters are not estimable.")
		if (!silent) print("Standard errors cannot be estimated.")
		res_se<-rep(NA, length(resfit))
	
	} else {
		res_se_diag=diag(inversemathessian)
		
		res_se <- rep(NA, length(resfit))
		res_se[res_se_diag>=0]<-sqrt(res_se_diag[res_se_diag>=0])

	}
	} else {
	
		if (!silent) print("Standard errors are not estimated.")
		res_se<-rep(NA, length(resfit))
	
	}
		
	names(res_se)<-names(resfit)
	
	resul$se<-res_se
	
	resul$parametersfixed<-parametersfixed
	
	resul$method_incertitude<-method_incertitude
	
	resul$zero_counts<-zero_counts
	
	resul$data<-data
	
		
for(kl in 1:length(res_se)) {
	if (is.na(res_se[kl])) {
		if (!silent) cat(paste(names(resfit[kl]), "=", format(resfit[kl], digits=max(3, trunc(log10(abs(resfit[kl])))+4)), "  SE= NaN\n", sep=""))
	} else {
		if (!silent) cat(paste(names(resfit[kl]), "=", format(resfit[kl], digits=max(3, trunc(log10(abs(resfit[kl])))+4)), "  SE=", format(res_se[kl], , digits=max(3, trunc(log10(res_se[kl]))+4)), "\n", sep=""))
	}
}

dtout <- list()

for (kl in 1:length(resul$data)) {

if (!silent) cat(paste("Series: ", names(resul$data[kl]), "\n", sep=""))

# la date de référence est resul$data[[kl]][1, "Date"]-resul$data[[kl]][1, "ordinal"]+1
ref <- resul$data[[kl]][1, "Date"]-resul$data[[kl]][1, "ordinal"]+1
intdtout <- c(reference=ref)
# save(list = ls(all.names = TRUE), file = "total.RData", envir = environment())

par <- .format_par(c(resfit, parametersfixed), names(resul$data[kl]))
sepfixed <- parametersfixed[strtrim(names(parametersfixed), 3)=="sd#"]
if (!is.null(sepfixed)) names(sepfixed) <- substring(names(sepfixed), 4)
se <- c(res_se, sepfixed)

d1 <- ref+par["Peak"]
if (!silent) cat(paste("Peak: ", d1, "\n", sep=""))
intdtout <- c(intdtout, Peak=as.numeric(d1))
if (!is.na(se["Peak"])) {
	d2 <- d1-2*se["Peak"]
	d3 <- d1+2*se["Peak"]
if (!silent) 	cat(paste("confidence interval:", d2, " to ", d3, "\n", sep=""))
	intdtout <- c(intdtout, PeakCI1=as.numeric(d2), PeakCI2=as.numeric(d3))
} else {
if (!silent) 	cat(paste("confidence interval not available\n", sep=""))
	intdtout <- c(intdtout, PeakCI1=NA, PeakCI2=NA)
}

d1 <- ref+par["Begin"]
if (!silent) cat(paste("Begin: ", d1, "\n", sep=""))
intdtout <- c(intdtout, Begin=as.numeric(d1))
# pour l'intervalle de confiance, il faut modifier soit directement
# Begin
# Peak Length
# Peak LengthB
d2 <- NULL
d3 <- NULL
if (!is.na(se["Begin"])) {
	d2 <- ref+par["Begin"]-2*se["Begin"]
	d3 <- ref+par["Begin"]+2*se["Begin"]
} else {
	sel <- 0
	l <- NA
	if (!is.na(se["Length"])) {
		l <- par["Length"]
		sel <- se["Length"]
	} else {
		if (!is.na(se["LengthB"])) {
			l <- par["LengthB"]
			sel <- se["LengthB"]
		}
	}
	if (!is.na(se["Peak"])) {
		d2 <- ref+par["Peak"]-2*se["Peak"]-l-2*sel
		d3 <- ref+par["Peak"]+2*se["Peak"]-l+2*sel
	} else {
		d2 <- ref+par["Peak"]-l-2*sel
		d3 <- ref+par["Peak"]-l+2*sel
	}
}
if (!is.null(d2) & !is.na(d2)) {
if (!silent) 	cat(paste("confidence interval:", d2, " to ", d3, "\n", sep=""))
	intdtout <- c(intdtout, BeginCI1=as.numeric(d2), BeginCI2=as.numeric(d3))
} else {
if (!silent) 	cat(paste("confidence interval not available\n", sep=""))
	intdtout <- c(intdtout, BeginCI1=NA, BeginCI2=NA)
}


d1 <- ref+par["End"]
if (!silent) cat(paste("End: ", d1, "\n", sep=""))
intdtout <- c(intdtout, End=as.numeric(d1))
# pour l'intervalle de confiance, il faut modifier soit directement
# End
# Peak Length
# Peak LengthE
d2 <- NULL
d3 <- NULL
if (!is.na(se["End"])) {
	d2 <- ref+par["End"]-2*se["End"]
	d3 <- ref+par["End"]+2*se["End"]
} else {
	sel <- 0
	l <- NA
	if (!is.na(se["Length"])) {
		l <- par["Length"]
		sel <- se["Length"]
	} else {
		if (!is.na(se["LengthE"])) {
			l <- par["LengthE"]
			sel <- se["LengthE"]
		}
	}
	if (!is.na(se["Peak"])) {
		d2 <- ref+par["Peak"]-2*se["Peak"]+l-2*sel
		d3 <- ref+par["Peak"]+2*se["Peak"]+l+2*sel
	} else {
		d2 <- ref+par["Peak"]+l-2*sel
		d3 <- ref+par["Peak"]+l+2*sel
	}
}
if (!is.null(d2) & !is.na(d2)) {
if (!silent) 	cat(paste("confidence interval:", d2, " to ", d3, "\n", sep=""))
	intdtout <- c(intdtout, EndCI1=as.numeric(d2), EndCI2=as.numeric(d3))
} else {
if (!silent) 	cat(paste("confidence interval not available\n", sep=""))
	intdtout <- c(intdtout, EndCI1=NA, EndCI2=NA)
}

dtout <- c(dtout, list(intdtout))

}

names(dtout) <- names(resul$data)

resul$Dates <- dtout

class(resul) <- "phenology"
	
if (!silent) cat(paste("-Ln L=", format(resul$value, digits=max(3, trunc(log10(resul$value))+4)), "\n", sep=""))
if (!silent) cat(paste("Parameters=", format(length(resul$par), digits=max(3, trunc(log10(length(resul$par)))+4)), "\n", sep=""))
if (!silent) cat(paste("AIC=", format(2*resul$value+2*length(resul$par), digits=max(3, trunc(log10(2*resul$value+2*length(resul$par)))+4)), "\n", sep=""))

	
if (!silent & growlnotify) growlnotify('Fit is done!')
	

return(resul)

}

