#PCDFunctions.r;
#Contains the ancillary functions used in PCD calculations, such as equation of lines for two points

#####AUXILIARY FUNCTIONS#################
#in all these functions
#points are vectors, and data sets are either matrices or data frames
#'
#' @import graphics
#' @import stats
#' @import plot3D
#' @import interp
#' @importFrom Rdpack reprompt
#'
#' @title Check the argument is a point of a given dimension
#'
#' @description Returns \code{TRUE} if the argument \code{p} is a \code{numeric} point of dimension \code{dim}
#' (default is \code{dim=2}); otherwise returns \code{FALSE}.
#'
#' @param p A \code{vector} to be checked to see it is a point of dimension \code{dim} or not.
#' @param dim A positive integer representing the dimension of the argument \code{p}.
#'
#' @return \code{TRUE} if \code{p} is a \code{vector} of dimension \code{dim}.
#'
#' @seealso \code{\link{dimension}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(-1.22,-2.33); B<-c(2.55,3.75,4)
#' is.point(A)
#' is.point(A,2)
#' is.point(A,1)
#' is.point(A,3)
#'
#' is.point(B)
#' is.point(B,3)
#'
#' Dt<-rbind(A,B[1:2])
#' is.point(Dt)
#' is.point(as.vector(Dt),4)
#'
#' lis<-list(a=A,b=B)
#' is.point(lis)
#'
#' dat.fr<-data.frame(a=A,b=B[1:2])
#' is.point(dat.fr)
#'
#' is.point(c("a","b"))
#'
#' @export
is.point <- function(p,dim=2)
{
  res<-is.numeric(p)==TRUE && is.vector(p)==TRUE && !is.list(p) && length(p)==dim
  res
}

#################################################################

#' @title The dimension of a \code{vector} or matrix or a data frame
#'
#' @description
#' Returns the dimension (i.e., number of columns) of \code{x}, which is a matrix or a \code{vector} or a data
#' frame. This is different than the \code{dim} function in the base distribution of \code{R}, in the sense that,
#' \code{dimension} gives only the number of columns of the argument \code{x}, while \code{dim} gives the number of rows and
#' columns of \code{x}.
#' \code{dimension} also works for a scalar or a vector, while \code{dim} yields \code{NULL} for such arguments.
#'
#' @param x A \code{vector} or a matrix or a data frame whose dimension is to be determined.
#'
#' @return Dimension (i.e., number of columns) of \code{x}
#'
#' @seealso \code{\link{is.point}} and \code{\link{dim}} from the base distribution of \code{R}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' dimension(3)
#' dim(3)
#'
#' A<-c(1,2)
#' dimension(A)
#' dim(A)
#'
#' B<-c(2,3)
#' dimension(rbind(A,B,A))
#' dimension(cbind(A,B,A))
#'
#' M<-matrix(runif(20),ncol=5)
#' dimension(M)
#' dim(M)
#'
#' dat.fr<-data.frame(a=A,b=B)
#' dimension(dat.fr)
#' dim(dat.fr)
#'
#' dimension(c("a","b"))
#'
#' @export dimension
dimension <- function(x)
{
  if ( (!is.vector(x) && !is.matrix(x) && !is.data.frame(x) )  || (is.list(x) && !is.data.frame(x)) )
  {stop('x must be a vector or a matrix or a data frame')}

  if (is.vector(x))
  {dimen<-length(x)
  } else
  {x<-as.matrix(x)
  dimen<-ncol(x)
  }
  dimen
}

#################################################################

#' @title The slope of a line
#'
#' @description Returns the slope of the line joining two distinct 2D points \code{a} and \code{b}.
#'
#' @param a,b 2D points that determine the straight line (i.e., through which the straight line passes).
#'
#' @return Slope of the line joining 2D points \code{a} and \code{b}
#'
#' @seealso \code{\link{Line}}, \code{\link{paraline}}, and \code{\link{perpline}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(-1.22,-2.33); B<-c(2.55,3.75)
#' slope(A,B)
#'
#' slope(c(1,2),c(2,3))
#' slope(c(1,2),c(1,3))
#'
#' @export slope
slope <- function(a,b)
{
  if (!is.point(a) || !is.point(b))
  {stop('both arguments must be numeric points (vectors) of dimension 2')}

  if (all(a==b))
  {stop('both arguments are same; slope is not well defined')}

  (b[2]-a[2])/(b[1]-a[1])
}

#################################################################

#' @title The line joining two distinct 2D points \code{a} and \code{b}
#'
#' @description
#' An object of class \code{"Lines"}.
#' Returns the \code{equation, slope, intercept}, and \eqn{y}-coordinates of the line crossing two
#' distinct 2D points \code{a} and \code{b} with \eqn{x}-coordinates provided in \code{vector} \code{x}.
#'
#' This function is different from the \code{\link[stats]{line}} function in the standard \code{stats} package
#' in \code{R} in the sense that \code{Line(a,b,x)} fits the line passing
#' through points \code{a} and \code{b} and returns various quantities (see below) for this line and \code{x} is
#' the \eqn{x}-coordinates of the points we want to find on the \code{Line(a,b,x)} while \code{line(a,b)} fits the line robustly
#' whose \eqn{x}-coordinates are in \code{a} and \eqn{y}-coordinates are in \code{b}.
#'
#' \code{Line(a,b,x)} and \code{line(x,Line(A,B,x)$y)}
#' would yield the same straight line (i.e., the line with the same coefficients.)
#'
#' @param a,b 2D points that determine the straight line (i.e., through which the straight line passes).
#' @param x A scalar or a \code{vector} of scalars representing the \eqn{x}-coordinates of the line.
#'
#' @return A \code{list} with the elements
#' \item{desc}{A description of the line}
#' \item{mtitle}{The \code{"main"} title for the plot of the line}
#' \item{points}{The input points \code{a} and \code{b} through which the straight line passes
#' (stacked row-wise, i.e., row 1 is point \code{a} and row 2 is point \code{b}).}
#' \item{x}{The input scalar or \code{vector} which constitutes the \eqn{x}-coordinates of the point(s) of interest
#' on the line.}
#' \item{y}{The output scalar or \code{vector} which constitutes the \eqn{y}-coordinates of the point(s) of interest
#' on the line. If \code{x} is a scalar, then \code{y} will be a scalar and if \code{x} is a \code{vector} of scalars,
#' then \code{y} will be a \code{vector} of scalars.}
#' \item{slope}{Slope of the line, \code{Inf} is allowed, passing through points \code{a} and \code{b} }
#' \item{intercept}{Intercept of the line passing through points \code{a} and \code{b}}
#' \item{equation}{Equation of the line passing through points \code{a} and \code{b}}
#'
#' @seealso \code{\link{slope}}, \code{\link{paraline}}, \code{\link{perpline}}, \code{\link[stats]{line}}
#' in the generic \code{stats} package and and \code{\link{Line3D}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(-1.22,-2.33); B<-c(2.55,3.75)
#'
#' xr<-range(A,B);
#' xf<-(xr[2]-xr[1])*.1 #how far to go at the lower and upper ends in the x-coordinate
#' x<-seq(xr[1]-xf,xr[2]+xf,l=20)  #try also l=100
#'
#' lnAB<-Line(A,B,x)
#' lnAB
#' summary(lnAB)
#' plot(lnAB)
#'
#' line(A,B)  #this takes A as the x points and B as the y points and fits the line
#'
#' y<-lnAB$y
#' Xlim<-range(x,A,B)
#' if (!is.na(y[1])) {Ylim<-range(y,A,B)} else {Ylim<-range(A,B)}
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#' pf<-c(xd,-yd)*.025
#'
#' #plot of the line joining A and B
#' plot(rbind(A,B),pch=1,xlab="x",ylab="y",
#' xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' if (!is.na(y[1])) {lines(x,y,lty=1)} else {abline(v=A[1])}
#' text(rbind(A+pf,B+pf),c("A","B"))
#' int<-round(lnAB$intercept,2)  #intercept
#' sl<-round(lnAB$slope,2)  #slope
#' text(rbind((A+B)/2+pf*3),ifelse(is.na(int),paste("x=",A[1]),
#' ifelse(sl==0,paste("y=",int),
#' ifelse(sl==1,ifelse(sign(int)<0,paste("y=x",int),paste("y=x+",int)),
#' ifelse(sign(int)<0,paste("y=",sl,"x",int),paste("y=",sl,"x+",int))))))
#'
#' Line(c(1,2),c(2,3),3)
#'
#' @export Line
Line <- function(a,b,x)
{
  if (!is.point(a) || !is.point(b))
  {stop('a and b must be numeric points of dimension 2')}

  if (!is.point(x,length(x)))
  {stop('x must be a numeric vector')}

  if (all(a==b))
  {stop('a and b are same; line is not well defined')}

  sl<-slope(a,b)
  if (abs(sl)==Inf)
  {
   int<-ln<-NA
  } else
  {
   int<-(b[2]+(0-b[1])*sl);
    ln<-(b[2]+(x-b[1])*sl);
  }

  names(sl)<-"slope"
  names(int)<-"intercept"

  aname <-deparse(substitute(a))
  bname <-deparse(substitute(b))

  line.desc<-paste("The Line Passing through two Distinct Points", aname, "and", bname)
  main.txt<-paste("Line Passing through Points", aname, "and", bname)

  pts<-rbind(a,b)
  row.names(pts)<-c(aname,bname)

  if (abs(sl)==Inf)
  { vert<-a[1]
  eqn<-reqn<-paste("x=",vert)
  } else
  {
    vert<-NA
    eqn<-ifelse(sl==0 & int==0,"y=0",ifelse(sl!=0 & int==0,paste("y=",sl,"x",sep=""),
                                            ifelse(sl==0 & int!=0,paste("y=",int,sep=""),ifelse(sl==1,
                                                                                                ifelse(sign(int)<0,paste("y=x",int,sep=""),paste("y=x+",int,sep="")),
                                                                                                ifelse(sign(int)<0,paste("y=",sl,"x",int,sep=""),paste("y=",sl,"x+",int,sep="")))) ))
    rsl<-round(sl,2)
    rint<-round(int,2)
    reqn<-ifelse(rsl==0 & rint==0,"y=0",ifelse(rsl!=0 & rint==0,paste("y=",rsl,"x",sep=""),
                                               ifelse(rsl==0 & rint!=0,paste("y=",rint,sep=""),ifelse(rsl==1,
                                                                                                      ifelse(sign(rint)<0,paste("y=x",rint,sep=""),paste("y=x+",rint,sep="")),
                                                                                                      ifelse(sign(rint)<0,paste("y=",rsl,"x",rint,sep=""),paste("y=",rsl,"x+",rint,sep="")))) ))
  }

  res<-list(
    desc=line.desc,
    mtitle=main.txt,
    points=pts,
    vert=vert,
    x=x,
    y=ln,
    slope=sl,
   intercept=int,
    equation=eqn,
    eqnlabel=reqn
  )
  class(res)<-"Lines"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

#' @title The point of intersection of two lines
#'
#' @description Returns the intersection of two lines, first line passing through points \code{p1} and \code{q1}
#' and second line passing through points \code{p2} and \code{q2}. The points are chosen so that lines are well
#' defined.
#'
#' @param p1,q1 2D points that determine the first straight line (i.e., through which the first straight line
#' passes).
#' @param p2,q2 2D points that determine the second straight line (i.e., through which the second straight line
#' passes).
#'
#' @return The coordinates of the point of intersection of the two lines, first passing through points
#' \code{p1} and \code{q1} and second passing through points \code{p2} and \code{q2}.
#'
#' @seealso \code{\link{int.circ.line}} and \code{\link{dp2l}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(-1.22,-2.33); B<-c(2.55,3.75); C<-c(0,6); D<-c(3,-2)
#'
#' ip<-int.2lines(A,B,C,D)
#' ip
#' pts<-rbind(A,B,C,D,ip)
#' xr<-range(pts)
#' xf<-abs(xr[2]-xr[1])*.1 #how far to go at the lower and upper ends in the x-coordinate
#' x<-seq(xr[1]-xf,xr[2]+xf,l=20)  #try also l=100
#' lnAB<-Line(A,B,x)
#' lnCD<-Line(C,D,x)
#'
#' y1<-lnAB$y
#' y2<-lnCD$y
#' Xlim<-range(x,pts)
#' Ylim<-range(y1,y2,pts)
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#' pf<-c(xd,-yd)*.025
#'
#' #plot of the line joining A and B
#' plot(rbind(A,B,C,D),pch=1,xlab="x",ylab="y",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' lines(x,y1,lty=1,col=1)
#' lines(x,y2,lty=1,col=2)
#' text(rbind(A+pf,B+pf),c("A","B"))
#' text(rbind(C+pf,D+pf),c("C","D"))
#' text(rbind(ip+pf),c("intersection\n point"))
#'
#' int.2lines(c(1,2),c(1,3),C,D)
#'
#' @export int.2lines
int.2lines <- function(p1,q1,p2,q2)
{
  if (!is.point(p1) || !is.point(p2) || !is.point(q1) || !is.point(q2))
  {stop('All arguments must be numeric points of dimension 2')}

  if (all(p1==q1) || all(p2==q2))
  {stop('At least one of the lines is not well defined is not well defined')}

  sl1<-slope(p1,q1); sl2<-slope(p2,q2);
  if (sl1==sl2)
  {stop('The lines are parallel, hence do not intersect at a point')}

  p11<-p1[1]; p12<-p1[2]; p21<-p2[1]; p22<-p2[2];
  q11<-q1[1]; q12<-q1[2]; q21<-q2[1]; q22<-q2[2];
  if (p11==q11 && p12!=q12 && sl2!=Inf)
  {int.pnt<-c(p11,(p11*p22-p11*q22+p21*q22-p22*q21)/(-q21+p21))
  return(int.pnt);stop}

  if (p21==q21 && p22!=q22 && sl1!=Inf)
  {int.pnt<-c(p21,(p11*q12+p12*p21-p12*q11-p21*q12)/(-q11+p11))
  return(int.pnt);stop}

 int.pnt<-c((p11*p21*q12-p11*p21*q22+p11*p22*q21-p11*q12*q21-p12*p21*q11+p12*q11*q21+p21*q11*q22-p22*q11*q21)/(p11*p22-p11*q22-p12*p21+p12*q21+p21*q12-p22*q11+q11*q22-q12*q21),
             (p11*p22*q12-p11*q12*q22-p12*p21*q22-p12*p22*q11+p12*p22*q21+p12*q11*q22+p21*q12*q22-p22*q12*q21)/(p11*p22-p11*q22-p12*p21+p12*q21+p21*q12-p22*q11+q11*q22-q12*q21))

 int.pnt
} #end of the function
#'

#################################################################

#' @title The distance from a point to a line
#'
#' @description Returns the distance from a point \code{p} to the line joining
#' points \code{a} and \code{b} in 2D space.
#'
#' @param p A 2D point, distance from \code{p} to the line passing through points \code{a} and \code{b} are to be
#' computed.
#' @param a,b 2D points that determine the straight line (i.e., through which the straight line passes).
#'
#' @return A \code{list} with two elements
#' \item{dis}{Distance from point \code{p} to the line passing through \code{a} and \code{b}}
#' \item{cl2p}{The closest point on the line passing through \code{a} and \code{b} to the point \code{p}}
#'
#' @seealso \code{\link{dp2pl}}, \code{\link{dist.pt2set}} and \code{\link{Dist}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,2); B<-c(2,3); P<-c(3,1.5)
#'
#' dpl<-dp2l(P,A,B);
#' dpl
#' C<-dpl$cl2p
#' pts<-rbind(A,B,C,P)
#'
#' dp2l(A,A,B);
#' dp2l((A+B)/2,A,B);
#'
#' xr<-range(pts[,1])
#' xf<-(xr[2]-xr[1])*.25 #how far to go at the lower and upper ends in the x-coordinate
#' x<-seq(xr[1]-xf,xr[2]+xf,l=20)  #try also l=100
#' lnAB<-Line(A,B,x)
#' y<-lnAB$y
#' int<-lnAB$intercept #intercept
#' sl<-lnAB$slope #slope
#'
#' xsq<-seq(min(A[1],B[1],P[1])-xf,max(A[1],B[1],P[1])+xf,l=20)  #try also l=100
#' pline<-(-1/sl)*(xsq-P[1])+P[2] #line passing thru P and perpendicular to AB
#'
#' Xlim<-range(pts[,1],x)
#' Ylim<-range(pts[,2],y)
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(rbind(P),asp=1,pch=1,xlab="x",ylab="y",
#' xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' points(rbind(A,B),pch=1)
#' lines(x,y,lty=1,xlim=Xlim,ylim=Ylim)
#' int<-round(int,2);sl<-round(sl,2)
#' text(rbind((A+B)/2+xd*c(-.01,-.01)),ifelse(sl==0,paste("y=",int),
#' ifelse(sl==1,paste("y=x+",int),
#' ifelse(int==0,paste("y=",sl,"x"),paste("y=",sl,"x+",int)))))
#' text(rbind(A+xd*c(0,-.01),B+xd*c(.0,-.01),P+xd*c(.01,-.01)),c("A","B","P"))
#' lines(xsq,pline,lty=2)
#' segments(P[1],P[2], C[1], C[2], lty=1,col=2,lwd=2)
#' text(rbind(C+xd*c(-.01,-.01)),"C")
#'
#' text(rbind((P+C)/2),col=2,paste("d=",round(dpl$dis,2)))
#'
#' A<-c(1,2); B<-c(1,3)
#' P<-c(3,1.5)
#' dp2l(P,A,B);
#'
#' A<-c(1,2); B<-c(2,2)
#' P<-c(3,1.5)
#' dp2l(P,A,B);
#'
#' @export dp2l
dp2l <- function(p,a,b)
{
  if (!is.point(p) || !is.point(a) || !is.point(b))
  {stop('all arguments must be numeric points of dimension 2')}

  if (all(a==b))
  {stop('line is not well defined')}

  a1<-a[1]; a2<-a[2];
  b1<-b[1]; b2<-b[2];
  p1<-p[1]; p2<-p[2];

  if (a1==b1)
  {
    dp<-abs(a1-p1)
    C<-c(a1,p1)  #the point of intersection of perpendicular line from P to line joining A and B
  } else
  {
    sl<-slope(a,b);
    dp<-abs(p2-Line(a,b,p1)$y )/sqrt(sl^2+1)}
  c1<-(a1^2*p1-a1*a2*b2+a1*a2*p2-2*a1*b1*p1+a1*b2^2-a1*b2*p2+a2^2*b1-a2*b1*b2-a2*b1*p2+b1^2*p1+b1*b2*p2)/(a1^2-2*a1*b1+a2^2-2*a2*b2+b1^2+b2^2)
  c2<-(a1^2*b2-a1*a2*b1+a1*a2*p1-a1*b1*b2-a1*b2*p1+a2^2*p2+a2*b1^2-a2*b1*p1-2*a2*b2*p2+b1*b2*p1+b2^2*p2)/(a1^2-2*a1*b1+a2^2-2*a2*b2+b1^2+b2^2)
  C<-c(c1,c2)

  list(dis=dp,
       cl2p=C
  )
  #dis is tghe distance
  #cl2p closest point on the line to point p
} #end of the functionP
#'

#################################################################

#' @title The distance between two vectors, matrices, or data frames
#'
#' @description Returns the Euclidean distance between \code{x} and \code{y} which can be vectors
#' or matrices or data frames of any dimension (\code{x} and \code{y} should be of same dimension).
#'
#' This function is different from the \code{\link[stats]{dist}} function in the \code{stats} package of the standard \code{R} distribution.
#' \code{dist} requires its argument to be a data matrix and \code{\link[stats]{dist}} computes and returns the distance matrix computed
#' by using the specified distance measure to compute the distances between the rows of a data matrix
#' (\insertCite{S-Book:1998;textual}{pcds}),
#' while \code{Dist} needs two arguments to find the distances between. For two data matrices \eqn{A} and \eqn{B},
#' \code{dist(rbind(as.vector(A),as.vector(B)))} and \code{Dist(A,B)} yield the same result.
#'
#' @param x,y Vectors, matrices or data frames (both should be of the same type).
#'
#' @return Euclidean distance between \code{x} and \code{y}
#'
#' @seealso \code{\link[stats]{dist}} from the base package \code{stats}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' B<-c(1,0); C<-c(1/2,sqrt(3)/2);
#' Dist(B,C);
#' dist(rbind(B,C))
#' dist(rbind(as.vector(B),as.vector(C)))
#' Dist(B,B);
#'
#' x<-runif(10)
#' y<-runif(10)
#' Dist(x,y)
#'
#' xm<-matrix(x,ncol=2)
#' ym<-matrix(y,ncol=2)
#' Dist(xm,ym)
#' dist(rbind(as.vector(xm),as.vector(ym)))
#'
#' Dist(xm,xm)
#'
#' dat.fr<-data.frame(b=B,c=C)
#' Dist(dat.fr,dat.fr)
#' Dist(dat.fr,cbind(B,C))
#'
#' @export Dist
Dist <- function(x,y)
{
  x<-as.matrix(x)
  y<-as.matrix(y)
  dis<-sqrt(sum((x-y)^2))

  dis
} #end of the function
#'

#################################################################

#' @title Distance from a point to a set of finite cardinality
#'
#' @description Returns the Euclidean distance between a point \code{x} and set of points \code{Y} and the
#' closest point in set \code{Y} to \code{x}. Distance between a point and a set is by definition the distance
#' from the point to the closest point in the set. \code{x} should be of finite dimension and \code{Y} should
#' be of finite cardinality and \code{x} and elements of \code{Y} must have the same dimension.
#'
#' @param x A \code{vector} (i.e., a point in \eqn{R^d}).
#' @param Y A set of \eqn{d}-dimensional points.
#'
#' @return A \code{list} with the elements
#' \item{distance}{Distance from point \code{x} to set \code{Y}}
#' \item{ind.cl.point}{Index of the closest point in set \code{Y} to the point \code{x}}
#' \item{closest.point}{The closest point in set \code{Y} to the point \code{x}}
#'
#' @seealso \code{\link{dp2l}} and \code{\link{dp2pl}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(0,0); B<-c(1,0); C<-c(1/2,sqrt(3)/2);
#' Te<-rbind(A,B,C);
#' dist.pt2set(c(1,2),Te)
#'
#' X2<-cbind(runif(10),runif(10))
#' dist.pt2set(c(1,2),X2)
#'
#' dist.pt2set(C,C)
#' dist.pt2set(B,C)
#'
#' x<-runif(1)
#' y<-as.matrix(runif(10))
#' dist.pt2set(x,y)  #this works, because x is a 1D point, and y is treated as a set of 10 1D points
#' #but will give an error message if y<-runif(10) is used above
#'
#' dat.fr<-data.frame(b=B,c=C)
#' dist.pt2set(A,dat.fr)
#'
#' @export dist.pt2set
dist.pt2set <- function(x,Y)
{
  if (!is.point(x,length(x)))
  {stop('x must be a numeric vector')}

  dimx<-length(x)

  if (is.vector(Y))
  {dimY<-length(Y);
  if (dimY != dimx )
  {stop('Both arguments must be of the same dimension')
  } else
  {dis<-Dist(x,Y)
 ind.mdis<-1
  xcl<-Y}
  } else
  {
    if (!is.matrix(Y) && !is.data.frame(Y))
    {stop('Y must be a matrix or a data frame, each row representing a point')}
    Y<-as.matrix(Y)
    dimY<-ncol(Y);
    if (dimY != dimx)
    {stop('x and Y must be of the same dimension')}
    dst<-vector()
    xcl<-NULL
    nY<-nrow(Y)
    for (i in 1:nY)
    {
      dst<-c(dst,Dist(x,Y[i,]) )
    }

    dis<-min(dst)
   ind.mdis<-which(dst==dis)
    xcl<-Y[ind.mdis,]
  }

  list(
    distance=dis,
   ind.cl.point=ind.mdis,
    closest.point=xcl
  )
} #end of the function
#'

#################################################################

#' @title Generation of points segregated (in a radial or circular fashion) from a given set of points
#'
#' @description
#' An object of class \code{"Patterns"}.
#' Generates \code{n} 2D points uniformly in \eqn{(a_1-e,a_1+e) \times (a_1-e,a_1+e) \setminus B(y_i,e)}
#' (\eqn{a_1} and \eqn{b1} are denoted as \code{a1} and \code{b1} as arguments) where
#' \eqn{Y=(y_1,y_2,\ldots,y_{n_y})} with \eqn{n_y} being number of \code{Y} points for various values of \code{e}
#' under the segregation pattern and \eqn{B(y_i,e)} is the ball centered at \eqn{y_i} with radius \code{e}.
#'
#' Positive values of \code{e} yield realizations from the segregation pattern
#' and nonpositive values of \code{e} provide a type of complete spatial randomness (CSR),
#' \code{e} should not be too large to make the support of generated points empty,
#' \code{a1} is defaulted to the minimum of the \eqn{x}-coordinates of the \code{Y} points,
#' \code{a2} is defaulted to the maximum of the \eqn{x}-coordinates of the \code{Y} points,
#' \code{b1} is defaulted to the minimum of the \eqn{y}-coordinates of the \code{Y} points,
#' \code{b2} is defaulted to the maximum of the \eqn{y}-coordinates of the \code{Y} points.
#'
#' @param n A positive integer representing the number of points to be generated.
#' @param Y A set of 2D points representing the reference points. The generated points are segregated
#' (in a circular or radial fashion) from these points.
#' @param e A positive real number representing the radius of the balls centered at \code{Y} points. These balls
#' are forbidden for the generated points (i.e., generated points would be in the complement of union of these
#' balls).
#' @param a1,a2 Real numbers representing the range of \eqn{x}-coordinates in the region
#' (default is the range of \eqn{x}-coordinates of the \code{Y} points).
#' @param b1,b2 Real numbers representing the range of \eqn{y}-coordinates in the region
#' (default is the range of \eqn{y}-coordinates of the \code{Y} points).
#'
#' @return A \code{list} with the elements
#' \item{type}{The type of the point pattern}
#' \item{mtitle}{The \code{"main"} title for the plot of the point pattern}
#' \item{parameters}{Radial (i.e., circular) exclusion parameter of the segregation pattern}
#' \item{ref.points}{The input set of reference points \code{Y}, i.e., points from which generated points are
#' segregated.}
#' \item{gen.points}{The output set of generated points segregated from \code{Y} points}
#' \item{tri.Y}{Logical output for triangulation based on \code{Y} points should be implemented or not.
#' if \code{TRUE} triangulation based on \code{Y} points is to be implemented (default is set to \code{FALSE}).}
#' \item{desc.pat}{Description of the point pattern}
#' \item{num.points}{The \code{vector} of two numbers, which are the number of generated points and
#' the number of reference (i.e., \code{Y}) points.}
#' \item{xlimit,ylimit}{The possible ranges of the \eqn{x}- and \eqn{y}-coordinates of the generated points}
#'
#' @seealso \code{\link{rasc.disc}}, \code{\link{rsegTe}}, \code{\link{rsegIITe}}, and \code{\link{rsegMT}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' nx<-20; ny<-4;  #try also nx<-1000; ny<-10
#' e<-.15;
#' #with default bounding box (i.e., unit square)
#' Y<-cbind(runif(ny),runif(ny))
#'
#' Xdt<-rseg.disc(nx,Y,e)
#' Xdt
#' summary(Xdt)
#' plot(Xdt,asp=1)
#'
#' e<-.15;  #try also e<- -.1; #this provides a CSR realization
#'
#' #with default bounding box (i.e., unit square)
#' Y<-cbind(runif(ny),runif(ny))
#' Xdt<-rseg.disc(nx,Y,e)$gen.points
#' Xlim<-range(Xdt[,1],Y[,1]);
#' Ylim<-range(Xdt[,2],Y[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(Y,asp=1,pch=16,col=2,lwd=2, xlab="x",ylab="y",main="segregation of two classes",
#'      xlim=Xlim+xd*c(-.01,.01),ylim=Ylim+yd*c(-.01,.01))
#' points(Xdt)
#'
#' #with a rectangular bounding box
#' a1<-0; a2<-10;
#' b1<-0; b2<-5;
#' e<-1.5;
#' Y<-cbind(runif(ny,a1,a2),runif(ny,b1,b2))
#' #try also Y<-cbind(runif(ny,a1,a2/2),runif(ny,b1,b2/2))
#'
#' Xdt<-rseg.disc(nx,Y,e,a1,a2,b1,b2)$gen.points
#' Xlim<-range(Xdt[,1],Y[,1]); Ylim<-range(Xdt[,2],Y[,2])
#'
#' plot(Y,pch=16,asp=1,col=2,lwd=2, xlab="x",ylab="y",main="segregation of two classes",
#'      xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' points(Xdt)
#'
#' @export rseg.disc
rseg.disc <- function(n,Y,e,a1=min(Y[,1]),a2=max(Y[,1]),b1=min(Y[,2]),b2=max(Y[,2]))
{
  if (is.point(Y))
  { Y<-matrix(Y,ncol=2)
  } else
  {Y<-as.matrix(Y)
  if (ncol(Y)!=2 )
  {stop('Y must be of dimension nx2')}
  }

  ny<-nrow(Y)

  if (!is.point(e,1))
  {stop('e must be a scalar')}

  if (ny>1 && e>=min((a2-a1)/2, (b2-b1)/2))
  {warning("warning: e is taken large, so it might take a very long time!")}

  if (e<=0)
  {warning("e is taken nonpositive, so the pattern is actually uniform in the study region!")}

  i<-0
  Xdt<-matrix(0,n,2)
  while (i<=n)
  {
    x<-c(runif(1,a1-e,a2+e),runif(1,b1-e,b2+e))
    if (dist.pt2set(x,Y)$distance>=e)
    {Xdt[i,]<-x;
    i<-i+1}
  }

  ename <-deparse(substitute(e))
  param<-e
  names(param)<-"exclusion parameter"
  typ<-paste("Segregation of ",n, " X points from ", ny," Y points with circular (or radial) exclusion parameter ",ename, "=",e,sep="")

  npts<-c(n,ny)
  names(npts)<-c("nx","ny")

  txt<-"Segregation of Class X from Class Y"
  main.txt<-paste("Segregation of Two Classes \n with circular exclusion parameter ",ename,"=",e,sep="")
  res<-list(
    type=typ,
    parameters=param,
    gen.points=Xdt, #generated points segregated from Y points
    ref.points=Y, #reference points, i.e., points from which generated points are segregated
    tri.Y=FALSE,
    desc.pat=txt, #description of the pattern
    mtitle=main.txt,
    num.points=npts,
    xlimit=c(a1-e,a2+e),
    ylimit=c(b1-e,b2+e)
  )

  class(res)<-"Patterns"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

#' @title Generation of points associated (in a radial or circular fashion) with a given set of points
#'
#' @description
#' An object of class \code{"Patterns"}.
#' Generates \code{n} 2D points uniformly in \eqn{(a_1-e,a_1+e) \times (a_1-e,a_1+e) \cap U_i B(y_i,e)}
#' (\eqn{a_1} and \eqn{b1} are denoted as \code{a1} and \code{b1} as arguments)
#' where \eqn{Y=(y_1,y_2,\ldots,y_{n_y})} with
#' \eqn{n_y} being number of \code{Y} points for various values of e under the association pattern
#' and \eqn{B(y_i,e)} is the ball centered at \eqn{y_i} with radius \code{e}.
#'
#' \code{e} must be positive and very large values of e provide patterns close to CSR.
#' \code{a1} is defaulted to the minimum of the \eqn{x}-coordinates of the \code{Y} points,
#' \code{a2} is defaulted to the maximum of the \eqn{x}-coordinates of the \code{Y} points,
#' \code{b1} is defaulted to the minimum of the \eqn{y}-coordinates of the \code{Y} points,
#' \code{b2} is defaulted to the maximum of the \eqn{y}-coordinates of the \code{Y} points.
#' This function is also very similar to \code{\link{rasc.matern}}, where \code{rasc.disc} needs the study window to be specified,
#' while \code{\link{rasc.matern}} does not.
#'
#' @param n A positive integer representing the number of points to be generated.
#' @param Y A set of 2D points representing the reference points. The generated points are associated
#' (in a circular or radial fashion) with these points.
#' @param e A positive real number representing the radius of the balls centered at \code{Y} points. Only these
#' balls are allowed for the generated points (i.e., generated points would be in the union of these balls).
#' @param a1,a2 Real numbers representing the range of \eqn{x}-coordinates in the region
#' (default is the range of \eqn{x}-coordinates of the \code{Y} points).
#' @param b1,b2 Real numbers representing the range of \eqn{y}-coordinates in the region
#' (default is the range of \eqn{y}-coordinates of the \code{Y} points).
#'
#' @return A \code{list} with the elements
#' \item{type}{The type of the point pattern}
#' \item{mtitle}{The \code{"main"} title for the plot of the point pattern}
#' \item{parameters}{Radial attraction parameter of the association pattern}
#' \item{ref.points}{The input set of attraction points \code{Y},
#' i.e., points with which generated points are associated.}
#' \item{gen.points}{The output set of generated points associated with \code{Y} points}
#' \item{tri.Y}{Logical output for triangulation based on \code{Y} points should be implemented or not.
#' if \code{TRUE} triangulation based on \code{Y} points is to be implemented (default is set to \code{FALSE}).}
#' \item{desc.pat}{Description of the point pattern}
#' \item{num.points}{The \code{vector} of two numbers, which are the number of generated points and the number of
#' attraction (i.e., \code{Y}) points.}
#' \item{xlimit,ylimit}{The possible range of the \eqn{x}- and \eqn{y}-coordinates of the generated points.}
#'
#' @seealso \code{\link{rseg.disc}}, \code{\link{rascTe}}, \code{\link{rascIITe}},
#' \code{\link{rasc.matern}}, and \code{\link{rascMT}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' nx<-20; ny<-4;  #try also nx<-1000; ny<-10;
#'
#' e<-.15;
#' #with default bounding box (i.e., unit square)
#' Y<-cbind(runif(ny),runif(ny))
#'
#' Xdat<-rasc.disc(nx,Y,e)
#' Xdat
#' summary(Xdat)
#' plot(Xdat,asp=1)
#'
#' Xdt<-rasc.disc(nx,Y,e)$gen.points
#' Xlim<-range(Xdt[,1],Y[,1]);
#' Ylim<-range(Xdt[,2],Y[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(Y,asp=1,xlab="x",ylab="y",main="association of two classes",
#'      xlim=Xlim+xd*c(-.01,.01),ylim=Ylim+yd*c(-.01,.01),pch=16,col=2,lwd=2)
#' points(Xdt)
#'
#' e<-.25; #pattern is very close to CSR!
#' #try also e<-.1;
#'
#' #with default bounding box (i.e., unit square)
#' Y<-cbind(runif(ny),runif(ny))
#' Xdt<-rasc.disc(nx,Y,e)$gen.points
#'
#' Xlim<-range(Xdt[,1],Y[,1]);
#' Ylim<-range(Xdt[,2],Y[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(Y,asp=1,xlab="x",ylab="y",main="association of two classes",
#'      xlim=Xlim+xd*c(-.01,.01),ylim=Ylim+yd*c(-.01,.01),pch=16,col=2,lwd=2)
#' points(Xdt)
#'
#' #with a rectangular bounding box
#' a1<-0; a2<-10;
#' b1<-0; b2<-5;
#' e<-1.1;  #try also e<-5; #pattern very close to CSR!
#'
#' Y<-cbind(runif(ny,a1,a2),runif(ny,b1,b2))
#' #try also Y<-cbind(runif(ny,a1,a2/2),runif(ny,b1,b2/2))
#'
#' Xdt<-rasc.disc(nx,Y,e,a1,a2,b1,b2)$gen.points
#' Xlim<-range(Xdt[,1],Y[,1]);
#' Ylim<-range(Xdt[,2],Y[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(Y,asp=1,xlab="x",ylab="y",main="association of two classes",
#'      xlim=Xlim+xd*c(-.01,.01),ylim=Ylim+yd*c(-.01,.01),pch=16,col=2,lwd=2)
#' points(Xdt)
#'
#' e<-.15
#' Y<-data.frame(yp=cbind(runif(ny),runif(ny)) )
#' Xdt<-rasc.disc(nx,Y,e)
#'
#' @export rasc.disc
rasc.disc <- function(n,Y,e,a1=min(Y[,1]),a2=max(Y[,1]),b1=min(Y[,2]),b2=max(Y[,2]))
{
  if (is.point(Y))
  { Y<-matrix(Y,ncol=2)
  } else
  {Y<-as.matrix(Y)
  if (ncol(Y)!=2 )
  {stop('Y must be of dimension nx2')}
  }

  ny<-nrow(Y)

  if (!is.point(e,1) || e<=0)
  {stop('e must be a positive scalar')}

  if (ny>1 && e>=max((a2-a1)/2, (b2-b1)/2))
  {warning("e is taken too large, the pattern will be very close to CSR!")}

  i<-0
  Xdt<-matrix(0,n,2)
  while (i<=n)
  {
    x<-c(runif(1,a1-e,a2+e),runif(1,b1-e,b2+e))
    if (dist.pt2set(x,Y)$distance<=e)
    {Xdt[i,]<-x;
    i<-i+1}
  }
  ename <-deparse(substitute(e))

  param<-e
  names(param)<-"attraction parameter"
  typ<-paste("Association of ",n, " points with ", ny," Y points with circular (or radial) attraction parameter ",ename, "=",e,sep="")

  npts<-c(n,ny)
  names(npts)<-c("nx","ny")

  txt<-"Association of one Class with Class Y"
  main.txt<-paste("Association of Two Classes \n with circular attraction parameter ",ename,"=",e,sep="")
  res<-list(
    type=typ,
    parameters=param,
    gen.points=Xdt, #generated points associated with Y points
    ref.points=Y, #attraction points, i.e., points to which generated points are associated
    tri.Y=FALSE,
    desc.pat=txt, #description of the pattern
    mtitle=main.txt,
    num.points=npts,
    xlimit=c(a1-e,a2+e),
    ylimit=c(b1-e,b2+e)
  )

  class(res)<-"Patterns"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

#' @title Generation of points associated (in a Matern-like fashion) to a given set of points
#'
#' @description
#' An object of class \code{"Patterns"}.
#' Generates \code{n} 2D points uniformly in \eqn{\cup B(y_i,e)} where \eqn{Y=(y_1,y_2,\ldots,y_{n_y})}
#' with \eqn{n_y} being number of \code{Y} points for various values of e under the association pattern
#' and \eqn{B(y_i,e)} is the ball centered at \eqn{y_i} with radius \code{e}.
#'
#' The pattern resembles the Matern cluster pattern (see \code{\link[spatstat.random]{rMatClust}} in the
#' \code{spatstat.random} package for further information (\insertCite{baddeley:2005;textual}{pcds}).
#' \code{rMatClust(kappa, scale, mu, win)} in the simplest
#' case generates a uniform Poisson point process of "parent" points with intensity \code{kappa}.
#' Then each parent point is replaced by a random cluster of
#' "offspring" points, the number of points per cluster being Poisson(\code{mu}) distributed, and their positions
#' being placed and uniformly inside a disc of radius scale centered on the parent point. The resulting point
#' pattern is a realization of the classical "stationary Matern cluster process" generated inside the
#' window \code{win}.
#'
#' The main difference of \code{rasc.matern} and \code{\link[spatstat.random]{rMatClust}} is that the parent points are \code{Y} points
#' which are given beforehand and we do not discard them in the end in \code{rasc.matern} and the offspring points
#' are the points associated with the reference points, \code{Y};
#' \code{e} must be positive and very large values of e provide patterns close to CSR.
#'
#' This function is also very similar to \code{\link{rasc.disc}}, where \code{\link{rasc.disc}} needs the study window to be specified,
#' while \code{rasc.matern} does not.
#'
#' @param n A positive integer representing the number of points to be generated.
#' @param Y A set of 2D points representing the reference points. The generated points are associated
#' (in a Matern-cluster like fashion) with these points.
#' @param e A positive real number representing the radius of the balls centered at \code{Y} points. Only
#' these balls are allowed for the generated points (i.e., generated points would be in the union of
#' these balls).
#'
#' @return A \code{list} with the elements
#' \item{type}{The type of the point pattern}
#' \item{mtitle}{The \code{"main"} title for the plot of the point pattern}
#' \item{parameters}{Radial (i.e., circular) attraction parameter of the association pattern.}
#' \item{ref.points}{The input set of attraction points \code{Y},
#' i.e., points with which generated points are associated.}
#' \item{gen.points}{The output set of generated points associated with \code{Y} points.}
#' \item{tri.Y}{Logical output for triangulation based on \code{Y} points should be implemented or not.
#' if \code{TRUE} triangulation based on \code{Y} points is to be implemented (default is set to \code{FALSE}).}
#' \item{desc.pat}{Description of the point pattern}
#' \item{num.points}{The \code{vector} of two numbers, which are the number of generated points and the number of
#' attraction (i.e., \code{Y}) points.}
#' \item{xlimit,ylimit}{The possible ranges of the \eqn{x}- and \eqn{y}-coordinates of the generated points.}
#'
#' @seealso \code{\link{rasc.disc}}, \code{\link{rascTe}}, \code{\link{rascIITe}},
#' \code{\link{rascMT}}, \code{\link{rseg.disc}}, and \code{\link[spatstat.random]{rMatClust}}
#' in the \code{spatstat.random} package
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' nx<-20; ny<-4;  #try also nx<-1000; ny<-10;
#'
#' e<-.15;  #try also e<-1.1; #closer to CSR than association, as e is large
#'
#' #Y points uniform in unit square
#' Y<-cbind(runif(ny),runif(ny))
#'
#' Xdt<-rasc.matern(nx,Y,e)
#' Xdt
#' summary(Xdt)
#' plot(Xdt,asp=1)
#'
#' Xdt<-rasc.matern(nx,Y,e)$gen.points
#' Xlim<-range(Xdt[,1],Y[,1]);
#' Ylim<-range(Xdt[,2],Y[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(Y,asp=1,xlab="x",ylab="y",main="Association of two classes",
#'      xlim=Xlim+xd*c(-.01,.01),ylim=Ylim+yd*c(-.01,.01),pch=16,col=2,lwd=2)
#' points(Xdt)
#'
#' a1<-0; a2<-10;
#' b1<-0; b2<-5;
#' e<-1.1;
#'
#' #Y points uniform in a rectangle
#' Y<-cbind(runif(ny,a1,a2),runif(ny,b1,b2))
#' #try also Y<-cbind(runif(ny,a1,a2/2),runif(ny,b1,b2/2))
#'
#' Xdt<-rasc.matern(nx,Y,e)$gen.points
#' Xlim<-range(Xdt[,1],Y[,1]);
#' Ylim<-range(Xdt[,2],Y[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(Y,asp=1,xlab="x",ylab="y",main="association of two classes",
#'      xlim=Xlim+xd*c(-.01,.01),ylim=Ylim+yd*c(-.01,.01),pch=16,col=2,lwd=2)
#' points(Xdt)
#'
#' e<-.15
#' Y<-data.frame(yp=cbind(runif(ny),runif(ny)) )
#' Xdt<-rasc.matern(nx,Y,e)
#'
#' @export rasc.matern
rasc.matern <- function(n,Y,e)
{
  if (is.point(Y))
  { Y<-matrix(Y,ncol=2)
  } else
  {Y<-as.matrix(Y)
  if (ncol(Y)!=2 )
  {stop('Y must be of dimension nx2')}
  }

  if (!is.point(e,1) || e<=0)
  {stop('e must be a positive scalar')}

  ny<-nrow(Y)
 indy<-sample(1:ny, n, replace = T)
  Xdt<-matrix(0,n,2)

  for (i in 1:n)
  {
    Yp<-Y[indy[i],]
    t<-2*pi*runif(1)
    r<-e*sqrt(runif(1))
    Xdt[i,]<-Yp+ r*c(cos(t),sin(t))
  }

  ename <-deparse(substitute(e))

  param<-e
  names(param)<-"attraction parameter"
  typ<-paste("Matern-like Association of ",n, " points with ", ny," Y points with circular attraction parameter ",ename, "=",e,sep="")

  npts<-c(n,ny)
  names(npts)<-c("nx","ny")

  txt<-"Matern-like Association of one Class with Class Y"
  main.txt<-paste("Matern-like Association of Two Classes \n Circular Attraction Parameter ",ename,"=",e,sep="")
  Xlim<-range(Y[,1])+c(-e,e)
  Ylim<-range(Y[,2])+c(-e,e)
  res<-list(
    type=typ,
    parameters=param,
    gen.points=Xdt, #generated points associated with Y points
    ref.points=Y, #attraction points, i.e., points to which generated points are associated
    tri.Y=FALSE,
    desc.pat=txt, #description of the pattern
    mtitle=main.txt,
    num.points=npts,
    xlimit=Xlim,
    ylimit=Ylim
  )

  class(res)<-"Patterns"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

#' @title Generation of Uniform Points in the first one-sixth of standard equilateral triangle
#'
#' @description
#' An object of class \code{"Uniform"}.
#' Generates \code{k} points uniformly in the first 1/6th of the standard equilateral triangle \eqn{T_e=(A,B,C)}
#' with vertices with \eqn{A=(0,0)}; \eqn{B=(1,0)}, \eqn{C=(1/2,\sqrt{3}/2)} (see the examples below).
#' The first 1/6th of the standard equilateral triangle is the triangle with vertices
#' \eqn{A=(0,0)}, \eqn{(1/2,0)}, \eqn{C=(1/2,\sqrt{3}/6)}.
#'
#' @param k a positive integer representing number of uniform points to be generated
#' in the first one-sixth of \eqn{T_e}.
#'
#' @return A \code{list} with the elements
#' \item{type}{The type of the point pattern}
#' \item{mtitle}{The \code{"main"} title for the plot of the point pattern}
#' \item{support}{The vertices of the support of the uniformly generated points}
#' \item{gen.points}{The output set of uniformly generated points in the first 1/6th of
#' the standard equilateral triangle.}
#' \item{out.region}{The outer region for the one-sixth of \eqn{T_e}, which is just \eqn{T_e} here.}
#' \item{desc.pat}{Description of the point pattern}
#' \item{num.points}{The \code{vector} of two numbers, which are the number of generated points
#' and the number of vertices of the support (i.e., \code{Y}) points.}
#' \item{txt4pnts}{Description of the two numbers in \code{num.points}.}
#' \item{xlimit,ylimit}{The ranges of the \eqn{x}- and \eqn{y}-coordinates of the generated, support and outer region points}
#'
#' @seealso \code{\link{runifTe}}, \code{\link{runif.bastri}}, \code{\link{runif.tri}}, and \code{\link{runifMT}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(0,0); B<-c(1,0); C<-c(1/2,sqrt(3)/2);
#' Te<-rbind(A,B,C);
#' CM<-(A+B+C)/3;
#' D1<-(B+C)/2; D2<-(A+C)/2; D3<-(A+B)/2;
#' Ds<-rbind(D1,D2,D3)
#' nx<-100  #try also nx<-100
#'
#' #data generation step
#' set.seed(1)
#' Xdt<-runifTe.onesixth(nx)
#' Xdt
#' summary(Xdt)
#' plot(Xdt,asp=1)
#'
#' Xd<-runifTe.onesixth(nx)$gen.points
#'
#' #plot of the data with the regions in the equilateral triangle
#' Xlim<-range(Te[,1])
#' Ylim<-range(Te[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(Te,asp=1,pch=".",xlim=Xlim+xd*c(-.01,.01),ylim=Ylim+yd*c(-.01,.01),xlab=" ",ylab=" ",
#'      main="first 1/6th of the \n standard equilateral triangle")
#' polygon(Te)
#' L<-Te; R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#' polygon(rbind(A,D3,CM),col=5)
#' points(Xd)
#'
#' #letter annotation of the plot
#' txt<-rbind(A,B,C,CM,D1,D2,D3)
#' xc<-txt[,1]+c(-.02,.02,.02,.04,.05,-.03,0)
#' yc<-txt[,2]+c(.02,.02,.02,.03,0,.03,-.03)
#' txt.str<-c("A","B","C","CM","D1","D2","D3")
#' text(xc,yc,txt.str)
#'
#' @export
runifTe.onesixth <- function(k)
{
  X <-matrix(0,k,2);
  for (i in 1:k)
  {ct<-0;
  while (ct==0)
  {
    x<-runif(1,0,.5); y<-runif(1,0,0.2886751347);
    if (y<.5773502694*x)
    {X[i,]<-c(x,y);
    ct<-1;
    }
  }
  }
  A<-c(0,0); B<-c(1,0); C<-c(1/2,sqrt(3)/2)
  CM<-(A+B+C)/3
  Te<-rbind(A,B,C)  #std eq triangle
  Te1_6<-rbind(A,(A+B)/2,CM)  #std eq triangle

  typ<-"Uniform Distribution in one-sixth of the standard equilateral triangle"
  txt<-paste(k, "uniform points in one-sixth of the standard equilateral triangle")
  main.txt<-"Uniform points in one-sixth of the\n standard equilateral triangle"

  npts<-c(k,3)
  names(npts)<-c("nx","ny")
  txt4pnts<-paste(names(npts)[1], "is the number of Uniform points \n",names(npts)[2], "is the number of vertices of the support region")
  row.names(Te1_6)<-c()

  res<-list(
    type=typ,
    gen.points=X, #uniformly generated points in one-sixth of std eq triangle
    tess.points=Te1_6, #tessellation points whose convex hull consititues the support of the uniform points
    desc.pat=txt, #description of the pattern
    mtitle=main.txt,
    out.region=Te, #outer region for the one-sixth of Te
    num.points=npts,  txt4pnts=txt4pnts,
    xlimit=range(Te[,1]),
    ylimit=range(Te[,2])
  )

  class(res)<-"Uniform"

  res$call <-match.call()

  res
} #end of the function
#'

#################################################################

#' @title The line parallel to the line segment joining two distinct 2D points \code{a} and \code{b}
#'
#' @description
#' An object of class \code{"Lines"}.
#' Returns the \code{equation, slope, intercept}, and \eqn{y}-coordinates of the line crossing the
#' point \code{p} and parallel to the line passing through the points \code{a} and \code{b} with
#' \eqn{x}-coordinates are provided in \code{vector} \code{x}.
#'
#' @param p A 2D point at which the parallel line to line segment joining \code{a} and \code{b} crosses.
#' @param a,b 2D points that determine the line segment (the line will be parallel to this line segment).
#' @param x A scalar or a \code{vector} of scalars representing the \eqn{x}-coordinates of the line parallel to
#' \code{ab} and crossing \code{p}.
#'
#' @return A \code{list} with the elements
#' \item{desc}{Description of the line passing through point \code{p} and parallel to line segment joining
#' \code{a} and \code{b}}
#' \item{mtitle}{The \code{"main"} title for the plot of the line passing through point \code{p} and parallel to
#' line segment joining \code{a} and \code{b}}
#' \item{points}{The input points \code{p}, \code{a}, and \code{b} (stacked row-wise,
#' i.e., point \code{p} is in row 1, point \code{a} is in row 2 and point \code{b} is in row 3).
#' Line parallel to \code{ab} crosses p.}
#' \item{x}{The input vector. It can be a scalar or a \code{vector} of scalars,
#' which constitute the \eqn{x}-coordinates of the point(s) of interest on the line passing through point
#' \code{p} and parallel to line segment joining \code{a} and \code{b}.}
#' \item{y}{The output scalar or \code{vector} which constitutes the \eqn{y}-coordinates of the point(s) of interest
#' on the line passing through point \code{p} and parallel to line segment joining \code{a} and \code{b}.
#' If \code{x} is a scalar, then \code{y} will be a scalar and if \code{x} is a \code{vector} of scalars,
#' then \code{y} will be a \code{vector} of scalars.}
#' \item{slope}{Slope of the line, \code{Inf} is allowed, passing through point \code{p} and parallel to
#' line segment joining \code{a} and \code{b}}
#' \item{intercept}{Intercept of the line passing through point \code{p} and parallel to line segment
#' joining \code{a} and \code{b}}
#' \item{equation}{Equation of the line passing through point \code{p} and parallel to line segment joining
#' \code{a} and \code{b}}
#'
#' @seealso \code{\link{slope}}, \code{\link{Line}}, and \code{\link{perpline}}, \code{\link[stats]{line}}
#' in the generic \code{stats} package, \code{\link{paraline3D}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1.1,1.2); B<-c(2.3,3.4); p<-c(.51,2.5)
#'
#' paraline(p,A,B,.45)
#' paraline(A,A,B,.45)
#'
#' pts<-rbind(A,B,p)
#' xr<-range(pts)
#' xf<-(xr[2]-xr[1])*.25 #how far to go at the lower and upper ends in the x-coordinate
#' x<-seq(xr[1]-xf,xr[2]+xf,l=20)  #try also l=100
#'
#' plnAB<-paraline(p,A,B,x)
#' plnAB
#' summary(plnAB)
#' plot(plnAB)
#'
#' y<-plnAB$y
#' Xlim<-range(x,pts[,1])
#' if (!is.na(y[1])) {Ylim<-range(y,pts[,2])} else {Ylim<-range(pts[,2])}
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#' pf<-c(xd,-yd)*.025
#'
#' plot(A,pch=".",xlab="",ylab="",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' points(pts)
#' txt.str<-c("A","B","p")
#' text(pts+rbind(pf,pf,pf),txt.str)
#'
#' segments(A[1],A[2],B[1],B[2],lty=2)
#' if (!is.na(y[1])) {lines(x,y,type="l",lty=1,xlim=Xlim,ylim=Ylim)} else {abline(v=p[1])}
#' tx<-(A[1]+B[1])/2;
#' if (!is.na(y[1])) {ty<-paraline(p,A,B,tx)$y} else {ty=p[2]}
#' text(tx,ty,"line parallel to AB\n and crossing p")
#'
#' @export paraline
paraline <- function(p,a,b,x)
{
  if (!is.point(p) || !is.point(a) || !is.point(b))
  {stop('p, a and b must all be numeric points of dimension 2')}

  if (!is.vector(x))
  {stop('x must be a vector')}

  if (all(a==b))
  {stop('a and b are same, hence lines are not well defined')}

  sl<-slope(a,b)
  if (abs(sl)==Inf)
  {
   int<-pln<-NA
  } else
  {
   int<-p[2]+sl*(0-p[1]);
    pln<-p[2]+sl*(x-p[1])
  }

  names(sl)<-"slope"
  names(int)<-"intercept"

  aname <-deparse(substitute(a))
  bname <-deparse(substitute(b))
  pname <-deparse(substitute(p))

  pts<-rbind(a,b,p)
  row.names(pts)<-c(aname,bname,pname)

  line.desc<-paste("Line Passing thru Point ",pname, " Parallel to Line Segment [",aname,bname,"]",sep="")
  main.text<-paste("Line Passing thru Point ",pname, "\n Parallel to Line Segment [",aname,bname,"]",sep="")

  if (abs(sl)==Inf)
  { vert<-p[1]
  eqn<-reqn<-paste("x = ",vert)
  } else
  {
    vert<-NA
    eqn<-ifelse(sl==0 & int==0,"y=0",ifelse(sl!=0 & int==0,paste("y=",sl,"x",sep=""),
                                            ifelse(sl==0 & int!=0,paste("y=",int,sep=""),ifelse(sl==1,
                                                                                                ifelse(sign(int)<0,paste("y=x",int,sep=""),paste("y=x+",int,sep="")),
                                                                                                ifelse(sign(int)<0,paste("y=",sl,"x",int,sep=""),paste("y=",sl,"x+",int,sep="")))) ))
    rsl<-round(sl,2)
    rint<-round(int,2)
    reqn<-ifelse(rsl==0 & rint==0,"y=0",ifelse(rsl!=0 & rint==0,paste("y=",rsl,"x",sep=""),
                                               ifelse(rsl==0 & rint!=0,paste("y=",rint,sep=""),ifelse(rsl==1,
                                                                                                      ifelse(sign(rint)<0,paste("y=x",rint,sep=""),paste("y=x+",rint,sep="")),
                                                                                                      ifelse(sign(rint)<0,paste("y=",rsl,"x",rint,sep=""),paste("y=",rsl,"x+",rint,sep="")))) ))
  }

  res<-list(
    desc=line.desc, mtitle=main.text,
    points=pts, vert=vert,
    x=x,
    y=pln,
    slope=sl,
   intercept=int,
    equation=eqn,
    eqnlabel=reqn
  )
  class(res)<-"Lines"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

# funsAB2CMTe
#'
#' @title The lines joining two vertices to the center of mass in standard equilateral triangle
#'
#' @description Two functions, \code{lA_CM.Te} and \code{lB_CM.Te} of class \code{"TriLines"}.
#' Returns the \code{equation, slope, intercept}, and \eqn{y}-coordinates of the lines joining \eqn{A} and \eqn{CM} and
#' also \eqn{B} and \eqn{CM}.
#'
#' \code{lA_CM.Te} is the line joining \eqn{A} to the center of mass, \eqn{CM}, and
#'
#' \code{lB_CM.Te} is the line joining \eqn{B} to the center of mass, \eqn{CM}, in the standard equilateral triangle \eqn{T_e=(A,B,C)}
#' with \eqn{A=(0,0)}, \eqn{B=(1,0)}, \eqn{C=(1/2,\sqrt{3}/2)}; \eqn{x}-coordinates are provided in \code{vector} \code{x}.
#'
#' @param x A single scalar or a \code{vector} of scalars which is the argument of the functions
#' \code{lA_CM.Te} and \code{lB_CM.Te}.
#'
#' @return A \code{list} with the elements
#' \item{txt1}{Longer description of the line.}
#' \item{txt2}{Shorter description of the line (to be inserted over the line in the plot).}
#' \item{mtitle}{The \code{"main"} title for the plot of the line.}
#' \item{cent}{The center chosen inside the standard equilateral triangle.}
#' \item{cent.name}{The name of the center inside the standard equilateral triangle.
#' It is \code{"CM"} for these two functions.}
#' \item{tri}{The triangle (it is the standard equilateral triangle for this function).}
#' \item{x}{The input vector, can be a scalar or a \code{vector} of scalars,
#' which constitute the \eqn{x}-coordinates of the point(s) of interest on the line.}
#' \item{y}{The output vector, will be a scalar if \code{x} is a scalar or a \code{vector} of scalars
#' if \code{x} is a \code{vector} of scalar, constitutes the \eqn{y}-coordinates of the point(s) of interest on the line.}
#' \item{slope}{Slope of the line.}
#' \item{intercept}{Intercept of the line.}
#' \item{equation}{Equation of the line.}
#'
#' @name funsAB2CMTe
NULL
#'
#' @seealso \code{\link{lA_M.Te}}, \code{\link{lB_M.Te}}, and \code{\link{lC_M.Te}}
#'
#' @rdname funsAB2CMTe
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' #Examples for lA_CM.Te
#' A<-c(0,0); B<-c(1,0); C<-c(1/2,sqrt(3)/2);
#' Te<-rbind(A,B,C)
#' xfence<-abs(A[1]-B[1])*.25 #how far to go at the lower and upper ends in the x-coordinate
#' x<-seq(min(A[1],B[1])-xfence,max(A[1],B[1])+xfence,by=.1)  #try also by=.01
#'
#' lnACM<-lA_CM.Te(x)
#' lnACM
#' summary(lnACM)
#' plot(lnACM)
#'
#' CM<-(A+B+C)/3;
#' D1<-(B+C)/2; D2<-(A+C)/2; D3<-(A+B)/2;
#' Ds<-rbind(D1,D2,D3)
#'
#' Xlim<-range(Te[,1])
#' Ylim<-range(Te[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(Te,pch=".",xlab="",ylab="",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Te)
#' L<-Te; R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#'
#' txt<-rbind(Te,CM,D1,D2,D3,c(.25,lA_CM.Te(.25)$y),c(.75,lB_CM.Te(.75)$y))
#' xc<-txt[,1]+c(-.02,.02,.02,.05,.05,-.03,.0,0,0)
#' yc<-txt[,2]+c(.02,.02,.02,.02,0,.02,-.04,0,0)
#' txt.str<-c("A","B","C","CM","D1","D2","D3","lA_CM.Te(x)","lB_CM.Te(x)")
#' text(xc,yc,txt.str)
#'
#' lA_CM.Te(.25)$y
#'
#' @export
lA_CM.Te <- function(x)
{
  if (!is.vector(x))
  {stop('the argument must be a vector')}

  sl<-.5773502694
 int<-0
  ln<-int+sl*x

  A<-c(0,0); B<-c(1,0); C<-c(1/2,sqrt(3)/2); #the vertices of the standard equilateral triangle
  Tr<-rbind(A,B,C)
  M<-(A+B+C)/3;

  names(sl)<-"slope"
  names(int)<-"intercept"

  dname <-deparse(substitute(x))
  cname <-"CM"

  txt1<-"Line Passing through A and the Center of Mass, CM, in the Standard Equilateral Triangle Te=ABC\n with A=(0,0);  B=(1,0); C=(1/2,sqrt(3)/2)"
  main.text<-"Line Passing through A and the Center of Mass CM\n in the Standard Equilateral Triangle"
  txt2<-paste("lA_CM.Te(",dname,")",sep="")

  res<-list(
    txt1=txt1, txt2=txt2, mtitle=main.text,
    cent=M, cent.name=cname,
    tri=Tr,
    x=x,
    y=ln,
    slope=sl,
   intercept=int,
    equation=ifelse(sl==0 & int==0,"y=0",ifelse(sl!=0 & int==0,paste("y=",sl,"x",sep=""),ifelse(sl==0 & int!=0,paste("y=",int,sep=""),ifelse(sl==1,ifelse(sign(int)<0,paste("y=x",int,sep=""),paste("y=x+",int,sep="")),
                                                                                                                                             ifelse(sign(int)<0,paste("y=",sl,"x",int,sep=""),paste("y=",sl,"x+",int,sep="")))) ))
  )

  class(res)<-"TriLines"
  res$call <-match.call()
  res
} #end of the function
#'
#' @rdname funsAB2CMTe
#'
#' @examples
#' #Examples for lB_CM.Te
#' A<-c(0,0); B<-c(1,0); C<-c(1/2,sqrt(3)/2);
#' Te<-rbind(A,B,C)
#' xfence<-abs(A[1]-B[1])*.25 #how far to go at the lower and upper ends in the x-coordinate
#' x<-seq(min(A[1],B[1])-xfence,max(A[1],B[1])+xfence,by=.1)  #try also by=.01
#'
#' lnBCM<-lB_CM.Te(x)
#' lnBCM
#' summary(lnBCM)
#' plot(lnBCM,xlab="x",ylab="y")
#'
#' lB_CM.Te(.25)$y
#'
#' @export
lB_CM.Te <- function(x)
{
  if (!is.vector(x))
  {stop('the argument must be a vector')}

  sl<--.5773502694
 int<-.5773502694
  ln<-int+sl*x

  names(sl)<-"slope"
  names(int)<-"intercept"

  A<-c(0,0); B<-c(1,0); C<-c(1/2,sqrt(3)/2); #the vertices of the standard equilateral triangle
  Tr<-rbind(A,B,C)
  M<-(A+B+C)/3;

  dname <-deparse(substitute(x))
  cname <-"CM"

  txt1<-"Line Passing through B and the Center of Mass, CM, in the Standard Equilateral Triangle Te=ABC\n with A=(0,0);  B=(1,0); C=(1/2,sqrt(3)/2)"
  main.text<-"Line Passing through B and the Center of Mass CM\n in the Standard Equilateral Triangle"
  txt2<-paste("lB_CM.Te(",dname,")",sep="")

  res<-list(
    txt1=txt1, txt2=txt2, mtitle=main.text,
    cent=M, cent.name=cname,
    tri=Tr,
    x=x,
    y=ln,
    slope=sl,
   intercept=int,
    equation=ifelse(sl==0 & int==0,"y=0",ifelse(sl!=0 & int==0,paste("y=",sl,"x",sep=""),ifelse(sl==0 & int!=0,paste("y=",int,sep=""),ifelse(sl==1,ifelse(sign(int)<0,paste("y=x",int,sep=""),paste("y=x+",int,sep="")),
                                                                                                                                             ifelse(sign(int)<0,paste("y=",sl,"x",int,sep=""),paste("y=",sl,"x+",int,sep="")))) ))
  )

  class(res)<-"TriLines"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

# funsAB2MTe
#'
#' @title The lines joining the three vertices of the standard equilateral triangle to a center, \code{M}, of it
#'
#' @description
#' Three functions, \code{lA_M.Te}, \code{lB_M.Te} and \code{lC_M.Te} of class \code{"TriLines"}.
#' Returns the \code{equation, slope, intercept}, and \eqn{y}-coordinates of the lines joining \eqn{A} and \code{M},
#' \eqn{B} and \code{M}, and also \eqn{C} and \code{M}.
#'
#' \code{lA_M.Te} is the line joining \eqn{A} to the center, \code{M}, \code{lB_M.Te} is the line joining \eqn{B} to \code{M},
#' and \code{lC_M.Te} is the line joining C to \code{M}, in the standard equilateral triangle \eqn{T_e=(A,B,C)}
#' with \eqn{A=(0,0)}, \eqn{B=(1,0)}, \eqn{C=(1/2,\sqrt{3}/2)}; \eqn{x}-coordinates are provided in \code{vector} \code{x}
#'
#' @param x A single scalar or a \code{vector} of scalars.
#' @param M A 2D point in Cartesian coordinates or a 3D point in barycentric coordinates
#' which serves as a center in the interior of the standard equilateral triangle.
#'
#' @return A \code{list} with the elements
#' \item{txt1}{Longer description of the line.}
#' \item{txt2}{Shorter description of the line (to be inserted over the line in the plot).}
#' \item{mtitle}{The \code{"main"} title for the plot of the line.}
#' \item{cent}{The center chosen inside the standard equilateral triangle.}
#' \item{cent.name}{The name of the center inside the standard equilateral triangle.}
#' \item{tri}{The triangle (it is the standard equilateral triangle for this function).}
#' \item{x}{The input vector, can be a scalar or a \code{vector} of scalars,
#' which constitute the \eqn{x}-coordinates of the point(s) of interest on the line.}
#' \item{y}{The output vector, will be a scalar if \code{x} is a scalar or a \code{vector} of scalars if \code{x} is
#' a \code{vector} of scalar, constitutes the \eqn{y}-coordinates of the point(s) of interest on the line.}
#' \item{slope}{Slope of the line.}
#' \item{intercept}{Intercept of the line.}
#' \item{equation}{Equation of the line.}
#'
#' @name funsAB2MTe
NULL
#'
#' @seealso \code{\link{lA_CM.Te}} and \code{\link{lB_CM.Te}}
#'
#' @rdname funsAB2MTe
#'
#' @examples
#' #Examples for lA_M.Te
#' A<-c(0,0); B<-c(1,0); C<-c(1/2,sqrt(3)/2);
#' Te<-rbind(A,B,C)
#'
#' M<-c(.65,.2)  #try also M<-c(1,1,1)
#'
#' xfence<-abs(A[1]-B[1])*.25 #how far to go at the lower and upper ends in the x-coordinate
#' x<-seq(min(A[1],B[1])-xfence,max(A[1],B[1])+xfence,by=.1)  #try also by=.01
#'
#' lnAM<-lA_M.Te(x,M)
#' lnAM
#' summary(lnAM)
#' plot(lnAM)
#'
#' Ds<-cp2e.tri(Te,M)
#' #finds the projections from a point M=(m1,m2) to the edges on the
#' #extension of the lines joining M to the vertices in the triangle Te
#'
#' Xlim<-range(Te[,1])
#' Ylim<-range(Te[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(Te,pch=".",xlab="",ylab="",
#' xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Te)
#' L<-Te; R<-rbind(M,M,M)
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#' L<-Ds; R<-rbind(M,M,M)
#' segments(L[,1], L[,2], R[,1], R[,2], lty=3,col=2)
#'
#' txt<-rbind(Te,M,Ds,c(.25,lA_M.Te(.25,M)$y),c(.4,lB_M.Te(.4,M)$y),
#' c(.60,lC_M.Te(.60,M)$y))
#' xc<-txt[,1]+c(-.02,.02,.02,.02,.04,-.03,.0,0,0,0)
#' yc<-txt[,2]+c(.02,.02,.02,.05,.02,.03,-.03,0,0,0)
#' txt.str<-c("A","B","C","M","D1","D2","D3","lA_M.Te(x)","lB_M.Te(x)","lC_M.Te(x)")
#' text(xc,yc,txt.str)
#'
#' lA_M.Te(.25,M)
#'
#' @export
lA_M.Te <- function(x,M)
{
  if (!is.point(x,length(x)))
  {stop('x must be a numeric vector')}

  A<-c(0,0); B<-c(1,0); C<-c(0.5,sqrt(3)/2); Te<-rbind(A,B,C)

  if (!is.point(M) && !is.point(M,3) )
  {stop('M must be a numeric 2D point for Cartesian coordinates or 3D point for barycentric coordinates')}

  dname <-deparse(substitute(x))
  cname <-deparse(substitute(M))

  if (dimension(M)==3)
  {
    M<-bary2cart(M,Te)
    cname <-"M"
  }
  if (!in.triangle(M,Te)$in.tri)
  {stop('M must be inside standard equilateral triangle')}

  m1<-M[1]
  m2<-M[2]

  sl<-m2/m1
 int<-m2+(0-m1)*m2/m1
  ln<-int+sl*x

  names(sl)<-"slope"
  names(int)<-"intercept"

  txt1<-"Line Passing through A and Center, M, in the Standard Equilateral Triangle Te=ABC\n with A=(0,0);  B=(1,0); C=(1/2,sqrt(3)/2)"
  main.text<-"Line Passing through A and the Center M\n in the Standard Equilateral Triangle"
  txt2<-paste("lA_M.Te(",dname,")",sep="")

  res<-list(
    txt1=txt1, txt2=txt2, mtitle=main.text,
    cent=M, cent.name=cname,
    tri=Te,
    x=x,
    y=ln,
    slope=sl,
   intercept=int,
    equation=ifelse(sl==0 & int==0,"y=0",ifelse(sl!=0 & int==0,paste("y=",sl,"x",sep=""),ifelse(sl==0 & int!=0,paste("y=",int,sep=""),ifelse(sl==1,ifelse(sign(int)<0,paste("y=x",int,sep=""),paste("y=x+",int,sep="")),
                                                                                                                                             ifelse(sign(int)<0,paste("y=",sl,"x",int,sep=""),paste("y=",sl,"x+",int,sep="")))) ))
  )

  class(res)<-"TriLines"
  res$call <-match.call()
  res
} #end of the function
#'
#' @rdname funsAB2MTe
#'
#' @examples
#' #Examples for lB_M.Te
#' A<-c(0,0); B<-c(1,0); C<-c(1/2,sqrt(3)/2);
#' Te<-rbind(A,B,C)
#'
#' M<-c(.65,.2)  #try also M<-c(1,1,1)
#'
#' xfence<-abs(A[1]-B[1])*.25 #how far to go at the lower and upper ends in the x-coordinate
#' x<-seq(min(A[1],B[1])-xfence,max(A[1],B[1])+xfence,by=.1)  #try also by=.01
#'
#' lnBM<-lB_M.Te(x,M)
#' lnBM
#' summary(lnBM)
#' plot(lnBM)
#'
#' lB_M.Te(.25,M)
#'
#' @export
lB_M.Te <- function(x,M)
{
  if (!is.point(x,length(x)))
  {stop('x must be a numeric vector')}

  A<-c(0,0); B<-c(1,0); C<-c(0.5,sqrt(3)/2); Te<-rbind(A,B,C)
  if (!is.point(M) && !is.point(M,3) )
  {stop('M must be a numeric 2D point for Cartesian coordinates or 3D point for barycentric coordinates')}

  dname <-deparse(substitute(x))
  cname <-deparse(substitute(M))

  if (dimension(M)==3)
  {
    M<-bary2cart(M,Te)
    cname <-"M"
  }
  if (!in.triangle(M,Te)$in.tri)
  {stop('M must be inside standard equilateral triangle')}

  m1<-M[1]
  m2<-M[2]

  sl<-m2/(m1-1)
 int<-m2+(0-m1)*m2/(m1-1)
  ln<-int+sl*x

  names(sl)<-"slope"
  names(int)<-"intercept"

  txt1<-"Line Passing through B and the Center, M, in the Standard Equilateral Triangle Te=ABC\n with A=(0,0);  B=(1,0); C=(1/2,sqrt(3)/2)"
  main.text<-"Line Passing through B and the Center M\n in the Standard Equilateral Triangle"
  txt2<-paste("lB_M.Te(",dname,")",sep="")

  res<-list(
    txt1=txt1, txt2=txt2, mtitle=main.text,
    cent=M, cent.name=cname,
    tri=Te,
    x=x,
    y=ln,
    slope=sl,
   intercept=int,
    equation=ifelse(sl==0 & int==0,"y=0",ifelse(sl!=0 & int==0,paste("y=",sl,"x",sep=""),ifelse(sl==0 & int!=0,paste("y=",int,sep=""),ifelse(sl==1,ifelse(sign(int)<0,paste("y=x",int,sep=""),paste("y=x+",int,sep="")),
                                                                                                                                             ifelse(sign(int)<0,paste("y=",sl,"x",int,sep=""),paste("y=",sl,"x+",int,sep="")))) ))
  )

  class(res)<-"TriLines"
  res$call <-match.call()
  res
} #end of the function
#'
#' @rdname funsAB2MTe
#'
#' @examples
#' #Examples for lC_M.Te
#' A<-c(0,0); B<-c(1,0); C<-c(1/2,sqrt(3)/2);
#' Te<-rbind(A,B,C)
#'
#' M<-c(.65,.2)  #try also M<-c(1,1,1)
#'
#' xfence<-abs(A[1]-B[1])*.25 #how far to go at the lower and upper ends in the x-coordinate
#' x<-seq(min(A[1],B[1])-xfence,max(A[1],B[1])+xfence,by=.1)  #try also by=.01
#'
#' lnCM<-lC_M.Te(x,M)
#' lnCM
#' summary(lnCM)
#' plot(lnCM)
#'
#' lC_M.Te(.25,M)
#'
#' @export
lC_M.Te <- function(x,M)
{
  if (!is.vector(x))
  {stop('x must be a vector')}

  A<-c(0,0); B<-c(1,0); C<-c(0.5,sqrt(3)/2); Te<-rbind(A,B,C)
  if (!is.point(M) && !is.point(M,3) )
  {stop('M must be a numeric 2D point for Cartesian coordinates or 3D point for barycentric coordinates')}

  dname <-deparse(substitute(x))
  cname <-deparse(substitute(M))

  if (dimension(M)==3)
  {
    M<-bary2cart(M,Te)
    cname <-"M"
  }
  if (!in.triangle(M,Te)$in.tri)
  {stop('M must be inside standard equilateral triangle')}

  m1<-M[1];
  m2<-M[2];

  sl<-(m2*2-sqrt(3))/(2*m1-1)
 int<-(m2*(2*0-1)-sqrt(3)*(0-m1))/(2*m1-1)
  ln<-int+sl*x

  names(sl)<-"slope"
  names(int)<-"intercept"

  txt1<-"Line Passing through C and the Center, M, in the Standard Equilateral Triangle Te=ABC\n with A=(0,0);  B=(1,0); C=(1/2,sqrt(3)/2)"
  main.text<-"Line Passing through C and the Center M\n in the Standard Equilateral Triangle"
  txt2<-paste("lC_M.Te(",dname,")",sep="")

  res<-list(
    txt1=txt1, txt2=txt2, mtitle=main.text,
    cent=M, cent.name=cname,
    tri=Te,
    x=x,
    y=ln,
    slope=sl,
   intercept=int,
    equation=ifelse(sl==0 & int==0,"y=0",ifelse(sl!=0 & int==0,paste("y=",sl,"x",sep=""),ifelse(sl==0 & int!=0,paste("y=",int,sep=""),ifelse(sl==1,ifelse(sign(int)<0,paste("y=x",int,sep=""),paste("y=x+",int,sep="")),
                                                                                                                                             ifelse(sign(int)<0,paste("y=",sl,"x",int,sep=""),paste("y=",sl,"x+",int,sep="")))) ))
  )

  class(res)<-"TriLines"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

#' @title Generation of Uniform Points in the Standard Equilateral Triangle
#'
#' @description
#' An object of class \code{"Uniform"}.
#' Generates \code{k} points uniformly in the standard equilateral triangle \eqn{T_e=T(A,B,C)}
#' with vertices \eqn{A=(0,0)}, \eqn{B=(1,0)}, and \eqn{C=(1/2,\sqrt{3}/2)}.
#'
#' @param k A positive integer representing the number of uniform points to be generated in the
#' standard equilateral triangle \eqn{T_e}.
#'
#' @return A \code{list} with the elements
#' \item{type}{The type of the pattern from which points are to be generated}
#' \item{mtitle}{The \code{"main"} title for the plot of the point pattern}
#' \item{tess.points}{The vertices of the support region of the uniformly generated points, it is the
#' standard equilateral triangle \eqn{T_e} for this function}
#' \item{gen.points}{The output set of generated points uniformly in the standard equilateral triangle \eqn{T_e}.}
#' \item{out.region}{The outer region which contains the support region, \code{NULL} for this function.}
#' \item{desc.pat}{Description of the point pattern from which points are to be generated}
#' \item{num.points}{The \code{vector} of two numbers, which are the number of generated points and the number
#' of vertices of the support points (here it is 3).}
#' \item{txt4pnts}{Description of the two numbers in \code{num.points}}
#' \item{xlimit,ylimit}{The ranges of the \eqn{x}- and \eqn{y}-coordinates of the support, \eqn{T_e}}
#'
#' @seealso \code{\link{runif.bastri}}, \code{\link{runif.tri}}, and \code{\link{runifMT}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(0,0); B<-c(1,0); C<-c(1/2,sqrt(3)/2);
#' Te<-rbind(A,B,C);
#' n<-10  #try also n<-100
#'
#' set.seed(1)
#' Xdt<-runifTe(n)
#' Xdt
#' summary(Xdt)
#' plot(Xdt,asp=1)
#'
#' Xlim<-range(Te[,1])
#' Ylim<-range(Te[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' dat<-runifTe(n)$gen.points
#' plot(Te,asp=1,pch=".",xlab="",ylab="",xlim=Xlim+xd*c(-.01,.01),ylim=Ylim+yd*c(-.01,.01))
#' polygon(Te)
#' points(dat)
#'
#' @export runifTe
runifTe <- function(k)
{
  Xdt <-matrix(0,k,2);
  for (i in 1:k)
  {ct<-0;
  while (ct==0)
  {
    x<-runif(1,0,1); y<-runif(1,0,.8660254040);
    if (y<1.732050808*x && y<1.732050808-1.732050808*x)
    {Xdt[i,]<-c(x,y);
    ct<-1;
    }
  }
  }
  A<-c(0,0); B<-c(1,0); C<-c(1/2,sqrt(3)/2)
  Te<-rbind(A,B,C)  #std eq triangle

  typ<-"Uniform Distribution in the Standard Equilateral Triangle"
  main.txt<-"Uniform Points in the Standard Equilateral Triangle"
  txt<-paste(k, "uniform points in the standard equilateral triangle")

  npts<-c(k,3)
  names(npts)<-c("nx","ny")
  txt4pnts<-paste(names(npts)[1], "is the number of uniform points \n",names(npts)[2], "is the number of vertices of the support region")

  res<-list(
    type=typ,
    mtitle=main.txt,
    tess.points=Te, #tessellation points whose convex hull consititues the support of the uniform points
    gen.points=Xdt, #uniformly generated points in the std regular tetrahedron
    out.region=NULL, #outer region for Te
    desc.pat=txt, #description of the pattern
    num.points=npts,  txt4pnts=txt4pnts,
    xlimit=range(Te[,1]),
    ylimit=range(Te[,2])
  )

  class(res)<-"Uniform"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

#' @title Generation of Uniform Points in the basic triangle
#'
#' @description
#' An object of class \code{"Uniform"}.
#' Generates \code{k} points uniformly in the basic triangle \eqn{T_b=T((0,0),(1,0),(c_1,c_2))}
#' where \eqn{c_1} is in \eqn{[0,1/2]}, \eqn{c_2>0} and \eqn{(1-c_1)^2+c_2^2 \le 1}.
#'
#' Any given triangle can be mapped to the basic
#' triangle by a combination of rigid body motions (i.e., translation, rotation and reflection) and scaling,
#' preserving uniformity of the points in the original triangle
#' (\insertCite{ceyhan:Phd-thesis,ceyhan:arc-density-CS,ceyhan:arc-density-PE;textual}{pcds}).
#' Hence basic triangle is useful for simulation studies under the uniformity hypothesis.
#'
#' @param k A positive integer representing the number of uniform points to be generated in the basic triangle.
#' @param c1,c2 Positive real numbers representing the top vertex in basic triangle
#' \eqn{T_b=T((0,0),(1,0),(c_1,c_2))}, \eqn{c_1} must be in \eqn{[0,1/2]}, \eqn{c_2>0} and
#' \eqn{(1-c_1)^2+c_2^2 \le 1}.
#'
#' @return A \code{list} with the elements
#' \item{type}{The type of the pattern from which points are to be generated}
#' \item{mtitle}{The \code{"main"} title for the plot of the point pattern}
#' \item{tess.points}{The vertices of the support of the uniformly generated points, it is the basic triangle
#' \eqn{T_b} for this function}
#' \item{gen.points}{The output set of generated points uniformly in the basic triangle}
#' \item{out.region}{The outer region which contains the support region, \code{NULL} for this function.}
#' \item{desc.pat}{Description of the point pattern from which points are to be generated}
#' \item{num.points}{The \code{vector} of two numbers, which are the number of generated points and the number
#' of vertices of the support points (here it is 3).}
#' \item{txt4pnts}{Description of the two numbers in \code{num.points}.}
#' \item{xlimit,ylimit}{The ranges of the \eqn{x}- and \eqn{y}-coordinates of the support, Tb}
#'
#' @seealso \code{\link{runifTe}}, \code{\link{runif.tri}}, and \code{\link{runifMT}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' c1<-.4; c2<-.6
#' A<-c(0,0); B<-c(1,0); C<-c(c1,c2);
#' Tb<-rbind(A,B,C);
#' n<-10  #try also n<-100
#'
#' set.seed(1)
#' runif.bastri(1,c1,c2)
#' Xdt<-runif.bastri(n,c1,c2)
#' Xdt
#' summary(Xdt)
#' plot(Xdt)
#'
#' dat<-runif.bastri(n,c1,c2)$g
#'
#' Xlim<-range(Tb[,1])
#' Ylim<-range(Tb[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(Tb,xlab="",ylab="",xlim=Xlim+xd*c(-.01,.01),ylim=Ylim+yd*c(-.01,.01),type="n")
#' polygon(Tb)
#' points(dat)
#'
#' @export runif.bastri
runif.bastri <- function(k,c1,c2)
{
  if (!is.point(c1,1) || !is.point(c2,1))
  {stop('c1 and c2 must be scalars')}

  if (c1<0 || c1>1/2 || c2<=0 || (1-c1)^2+c2^2 >1)
  {stop('c1 must be in [0,1/2], c2>0 and (1-c1)^2+c2^2 <=1')}

  X <-matrix(0,k,2);
  for (i in 1:k)
  {ct<-0;
  while (ct==0)
  {
    x<-runif(1,0,1); y<-runif(1,0,c2);
    if (y<(c2/c1)*x && y<c2*(x-1)/(c1-1))
    {X[i,]<-c(x,y);
    ct<-1;
    }
  }
  }
  A<-c(0,0); B<-c(1,0); C<-c(c1,c2)
  Tb<-rbind(A,B,C)  #basic triangle

  typ<-"Uniform Distribution in the Basic Triangle"
  main.txt<-paste("Uniform Points in the Basic Triangle\n with c1=",c1,"and c2=",c2)
  txt<-paste(k, "uniform points in the basic triangle with c1=",c1,"and c2=",c2)

  npts<-c(k,3)
  names(npts)<-c("nx","ny")
  txt4pnts<-paste(names(npts)[1], "is the number of uniform points \n",names(npts)[2], "is the number of vertices of the support region")

  res<-list(
    type=typ,
    gen.points=X, #uniformly generated points in the basic triangle, Tb
    tess.points=Tb, #tessellation points whose convex hull consititues the support of the uniform points
    desc.pat=txt, #description of the pattern
    mtitle=main.txt,
    out.region=NULL, #outer region for Tb
    num.points=npts,  txt4pnts=txt4pnts,
    xlimit=range(Tb[,1]),
    ylimit=range(Tb[,2])
  )

  class(res)<-"Uniform"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

#' @title Check whether a point is inside a circle
#'
#' @description Checks if the point \code{pt} lies in the circle with center \code{cent} and radius \code{rad},
#' denoted as \code{C(cent,rad)}. So, it returns 1 or \code{TRUE} if \code{pt} is inside the circle, and 0 otherwise.
#'
#' \code{boundary} is a logical argument (default=\code{FALSE}) to include boundary or not, so if it is \code{TRUE},
#' the function checks if the point, \code{pt}, lies in the closure of the circle (i.e., interior and
#' boundary combined) else it checks if \code{pt} lies in the interior of the circle.
#'
#' @param pt A 2D point to be checked whether it is inside the circle or not.
#' @param cent A 2D point in Cartesian coordinates which serves as the center of the circle.
#' @param rad A positive real number which serves as the radius of the circle.
#' @param boundary A logical parameter (default=\code{FALSE}) to include boundary or not, so if it is \code{TRUE},
#' the function checks if the point, \code{pt}, lies in the closure of the circle (i.e., interior and
#' boundary combined) else it checks if \code{pt} lies in the interior of the circle.
#'
#' @return Indicator for the point \code{pt} being inside the circle or not, i.e., returns 1 or \code{TRUE}
#' if \code{pt} is inside the circle, and 0 otherwise.
#'
#' @seealso \code{\link{in.triangle}}, \code{\link{in.tetrahedron}},  and
#' \code{\link[interp]{on.convex.hull}} from the \code{interp} package for documentation for \code{in.convex.hull}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' cent<-c(1,1); rad<-1; p<-c(1.4,1.2)
#' #try also cent<-runif(2); rad<-runif(1); p<-runif(2);
#'
#' in.circle(p,cent,rad)
#'
#' p<-c(.4,-.2)
#' in.circle(p,cent,rad)
#'
#' p<-c(1,0)
#' in.circle(p,cent,rad)
#' in.circle(p,cent,rad,boundary=TRUE)
#'
#' #for a NA entry point
#' p<-c(NA,.2)
#' in.circle(p,cent,rad)
#'
#' @export in.circle
in.circle <- function(pt,cent,rad,boundary=FALSE)
{
  if (!is.point(pt) || !is.point(cent) )
  {stop('pt and cent must be numeric 2D points')}

  if (!is.point(rad,1) || rad<=0)
  {stop('rad must be a scalar greater than 0')}

  if (boundary==FALSE)
  {ind.circ<-ifelse(Dist(pt,cent)<rad, TRUE, FALSE)
  }  else
  {ind.circ<-ifelse(Dist(pt,cent)<=rad, TRUE, FALSE)
  }
 ind.circ
} #end of the function
#'

#################################################################

#' @title Check whether a point is inside a triangle
#'
#' @description Checks if the point \code{p} lies in the triangle, \code{tri}, using the barycentric
#' coordinates, generally denoted as \eqn{(\alpha,\beta,\gamma)}.
#'
#' If all (normalized or non-normalized)
#' barycentric coordinates are positive then the point \code{p} is inside the triangle,
#' if all are nonnegative with one or more are zero, then \code{p} falls in the boundary. If some of the
#' barycentric coordinates are negative, then \code{p} falls outside the triangle.
#'
#' \code{boundary} is a logical argument (default=\code{FALSE}) to include boundary or not, so if it is \code{TRUE},
#' the function checks if the point, \code{p}, lies in the closure of the triangle (i.e., interior and boundary
#' combined) else it checks if \code{p} lies in the interior of the triangle.
#'
#' @param p A 2D point to be checked whether it is inside the triangle or not.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param boundary A logical parameter (default=\code{FALSE}) to include boundary or not, so if it is \code{TRUE},
#' the function checks if the point, \code{p}, lies in the closure of the triangle (i.e., interior and boundary
#' combined) else it checks if \code{p} lies in the interior of the triangle.
#'
#' @return A \code{list} with two elements
#' \item{in.tri}{A logical output, it is \code{TRUE}, if the point, \code{p}, is inside the triangle, \code{tri},
#' else it is \code{FALSE}.}
#' \item{barycentric}{The barycentric coordinates \eqn{(\alpha,\beta,\gamma)} of the point \code{p} with respect to
#' the triangle, \code{tri}.}
#'
#' @seealso \code{\link{inTriAll}} and \code{\link[interp]{on.convex.hull}} from the \code{interp} package for documentation for \code{in.convex.hull}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2); p<-c(1.4,1.2)
#'
#' Tr<-rbind(A,B,C)
#'
#' in.triangle(p,Tr)
#'
#' p<-c(.4,-.2)
#' in.triangle(p,Tr)
#'
#' #for the vertex A
#' in.triangle(A,Tr)
#' in.triangle(A,Tr,boundary=TRUE)
#'
#' #for a point on the edge AB
#' D3<-(A+B)/2
#' in.triangle(D3,Tr)
#' in.triangle(D3,Tr,boundary=TRUE)
#'
#' #for a NA entry point
#' p<-c(NA,.2)
#' in.triangle(p,Tr)
#'
#' dat.fr<-data.frame(a=Tr)
#' in.triangle(p,dat.fr)
#'
#' @export
in.triangle <- function(p,tri,boundary=FALSE)
{
  if (!is.point(p))
  {stop('p must be a numeric 2D point')}

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  p1<-tri[1,]; p2<-tri[2,]; p3<-tri[3,];
  if (is.na(p[1]) || is.na(p[2]))
  {ind.tri<-FALSE; alpha<-beta<-gamma<-NA;
  } else
  {
    p.x<-p[1]; p.y<-p[2]
    p1.x<-p1[1]; p1.y<-p1[2]
    p2.x<-p2[1]; p2.y<-p2[2]
    p3.x<-p3[1]; p3.y<-p3[2]
    # barycentric coordinates
    alpha = ((p2.y - p3.y)*(p.x - p3.x) + (p3.x - p2.x)*(p.y - p3.y)) /
      ((p2.y - p3.y)*(p1.x - p3.x) + (p3.x - p2.x)*(p1.y - p3.y));
    beta = ((p3.y - p1.y)*(p.x - p3.x) + (p1.x - p3.x)*(p.y - p3.y)) /
      ((p2.y - p3.y)*(p1.x - p3.x) + (p3.x - p2.x)*(p1.y - p3.y));
    gamma = 1 - alpha - beta;
    #if all of alpha, beta, and gamma are greater than 0,
    #then the point p lies within the interior of the triangle made of points p1, p2, and p3.

    if (boundary==FALSE)
    {ind.tri<-ifelse(all(c(alpha,beta,gamma)>0), TRUE, FALSE)
    }  else
    {ind.tri<-ifelse(all(c(alpha,beta,gamma)>=0), TRUE, FALSE)}
  }
  list(
   in.tri=ind.tri,
    barycentric=c(alpha,beta,gamma)
  )
} #end of the function
#'

#################################################################

#' @title Check whether all points in a data set are inside the triangle
#'
#' @description Checks if all the data points in the 2D data set, \code{Dt}, lie in the triangle, \code{tri},
#' using the barycentric coordinates, generally denoted as \eqn{(\alpha,\beta,\gamma)}.
#'
#' If all (normalized or non-normalized) barycentric coordinates of a point are positive then the point is
#' inside the triangle, if all are nonnegative with one or more are zero, then the point falls in the boundary.
#' If some of the barycentric coordinates are negative, then the point falls outside the triangle.
#'
#' \code{boundary} is a logical argument (default=\code{FALSE}) to include boundary or not, so if it is \code{TRUE},
#' the function checks if a point lies in the closure of the triangle (i.e., interior and boundary combined)
#' else it checks if the point lies in the interior of the triangle.
#'
#' @param Dt A set of 2D points representing the set of data points.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param boundary A logical parameter (default=\code{FALSE}) to include boundary or not, so if it is \code{TRUE},
#' the function checks if a point lies in the closure of the triangle (i.e., interior and boundary combined)
#' else it checks if the point lies in the interior of the triangle.
#'
#' @return A logical output, if all data points in \code{Dt} are inside the triangle, \code{tri}, the output is \code{TRUE},
#' else it is \code{FALSE}.
#'
#' @seealso \code{\link{in.triangle}} and \code{\link[interp]{on.convex.hull}} from the \code{interp} package for documentation for \code{in.convex.hull}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2); p<-c(1.4,1.2)
#'
#' Tr<-rbind(A,B,C)
#'
#' inTriAll(p,Tr)
#'
#' #for the vertex A
#' inTriAll(A,Tr)
#' inTriAll(A,Tr,boundary=TRUE)
#'
#' #for a point on the edge AB
#' D3<-(A+B)/2
#' inTriAll(D3,Tr)
#' inTriAll(D3,Tr,boundary=TRUE)
#'
#' #data set
#' n<-10
#' dat<-cbind(runif(n),runif(n))
#' inTriAll(dat,Tr,boundary=TRUE)
#'
#' dat<-runifTe(n)$gen.points
#' inTriAll(dat,Tr)
#' inTriAll(dat,Tr,boundary=TRUE)
#'
#' dat<-runif.tri(n,Tr)$g
#' inTriAll(dat,Tr)
#' inTriAll(dat,Tr,boundary=TRUE)
#'
#' dat.fr<-data.frame(a=dat)
#' inTriAll(dat.fr,Tr)
#'
#' dat.fr<-data.frame(a=Tr)
#' inTriAll(dat,dat.fr)
#'
#' @export inTriAll
inTriAll <- function(Dt,tri,boundary=FALSE)
{
  if (!is.numeric(as.matrix(Dt)))
  {stop('Dt must be numeric')}

  if (is.point(Dt))
  { intri<-in.triangle(Dt,tri,boundary)$in.tri
  } else
  {Dt<-as.matrix(Dt)
  if (ncol(Dt)!=2 )
  {stop('Dt must be of dimension nx2')}

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  n<-nrow(Dt)
  cnt<-0; i<-1; intri<-TRUE
  while (i<=n & cnt==0)
  { pnt<-Dt[i,]
  if (!in.triangle(pnt,tri,boundary)$in.tri)
  {cnt<-1; intri<-FALSE}
  else
  {i<-i+1}
  }
  }
 intri
} #end of the function
#'

#################################################################

#' @title Generation of Uniform Points in a Triangle
#'
#' @description
#' An object of class \code{"Uniform"}.
#' Generates \code{k} points uniformly in a given triangle, \code{tri}
#'
#' @param k A positive integer representing the number of uniform points to be generated in the triangle.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#'
#' @return A \code{list} with the elements
#' \item{type}{The type of the pattern from which points are to be generated}
#' \item{mtitle}{The \code{"main"} title for the plot of the point pattern}
#' \item{tess.points}{The vertices of the support of the uniformly generated points, it is the triangle
#' \code{tri} for this function}
#' \item{gen.points}{The output set of generated points uniformly in the triangle, \code{tri}.}
#' \item{out.region}{The outer region which contains the support region, \code{NULL} for this function.}
#' \item{desc.pat}{Description of the point pattern from which points are to be generated}
#' \item{num.points}{The \code{vector} of two numbers, which are the number of generated points and the number
#' of vertices of the support points (here it is 3).}
#' \item{txt4pnts}{Description of the two numbers in \code{num.points}}
#' \item{xlimit,ylimit}{The ranges of the \eqn{x}- and \eqn{y}-coordinates of the support, \code{tri}}
#'
#' @seealso \code{\link{runifTe}}, \code{\link{runif.bastri}}, and \code{\link{runifMT}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' n<-10  #try also n<-100
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C)
#'
#' Xdt<-runif.tri(n,Tr)
#' Xdt
#' summary(Xdt)
#' plot(Xdt)
#'
#' dat<-runif.tri(n,Tr)$g
#' Xlim<-range(Tr[,1])
#' Ylim<-range(Tr[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(Tr,pch=".",xlab="",ylab="",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tr)
#' points(dat)
#' xc<-Tr[,1]+c(-.01,.01,.01)
#' yc<-Tr[,2]+c(.02,.02,.02)
#' txt.str<-c("A","B","C")
#' text(xc,yc,txt.str)
#'
#' @export runif.tri
runif.tri <- function(k,tri)
{
  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  x1<-min(tri[,1]); x2<-max(tri[,1])
  y1<-min(tri[,2]); y2<-max(tri[,2])

  X <-matrix(0,k,2);
  for (i in 1:k)
  {ct<-0;
  while (ct==0)
  {
    x<-runif(1,x1,x2); y<-runif(1,y1,y2);
    if (in.triangle(c(x,y),tri)$in.tri==TRUE)
    {X[i,]<-c(x,y);
    ct<-1;
    }
  }
  }

  typ<-"Uniform Distribution in the Triangle"
  main.txt<-"Uniform Points in the Triangle"
  txt<-paste(k, " uniform points in the triangle with vertices (",tri[1,1],",",tri[1,2],")",", ",
             "(",tri[2,1],",",tri[2,2],")"," and ","(",tri[3,1],",",tri[3,2],")",sep="")

  npts<-c(k,3)
  names(npts)<-c("nx","ny")
  txt4pnts<-paste(names(npts)[1], "is the number of uniform points \n",names(npts)[2], "is the number of vertices of the support region")

  res<-list(
    type=typ,
    mtitle=main.txt,
    gen.points=X, #uniformly generated points in the triangle, tri
    tess.points=tri, #tessellation points whose convex hull consititues the support of the uniform points
    desc.pat=txt, #description of the pattern
    out.region=NULL, #outer region for tri
    num.points=npts,  txt4pnts=txt4pnts,
    xlimit=range(tri[,1]),
    ylimit=range(tri[,2])
  )

  class(res)<-"Uniform"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

#' @title Generation of points segregated (in a Type I fashion) from the vertices of \eqn{T_e}
#'
#' @description
#' An object of class \code{"Patterns"}.
#' Generates \code{k} points uniformly in the standard equilateral triangle \eqn{T_e=T((0,0),(1,0),(1/2,\sqrt{3}/2))}
#' under the type I segregation alternative for \code{eps} in \eqn{(0,\sqrt{3}/3=0.5773503]}.
#'
#' In the type I segregation, the triangular forbidden regions around the vertices are determined by
#' the parameter \code{eps} which serves as the height of these triangles (see examples for a sample plot.)
#'
#' See also (\insertCite{ceyhan:arc-density-PE,ceyhan:arc-density-CS,ceyhan:dom-num-NPE-Spat2011;textual}{pcds}).
#'
#' @param k A positive integer representing the number of points to be generated.
#' @param eps A positive real number representing the parameter of type I segregation (which is the
#' height of the triangular forbidden regions around the vertices).
#'
#' @return A \code{list} with the elements
#' \item{type}{The type of the point pattern}
#' \item{mtitle}{The \code{"main"} title for the plot of the point pattern}
#' \item{parameters}{The exclusion parameter, \code{eps}, of the segregation pattern, which is the height
#' of the triangular forbidden regions around the vertices }
#' \item{ref.points}{The input set of points \code{Y};
#' reference points, i.e., points from which generated points are segregated (i.e., vertices of \eqn{T_e}).}
#' \item{gen.points}{The output set of generated points segregated from \code{Y} points (i.e., vertices of \eqn{T_e}).}
#' \item{tri.Y}{Logical output for triangulation based on \code{Y} points should be implemented or not.
#' if \code{TRUE} triangulation based on \code{Y} points is to be implemented (default is set to \code{FALSE}).}
#' \item{desc.pat}{Description of the point pattern}
#' \item{num.points}{The \code{vector} of two numbers, which are the number of generated points and
#' the number of reference (i.e., \code{Y}) points, which is 3 here.}
#' \item{xlimit,ylimit}{The ranges of the \eqn{x}- and \eqn{y}-coordinates of the reference points, which are the
#' vertices of \eqn{T_e} here}
#'
#' @seealso \code{\link{rseg.disc}}, \code{\link{rasc.disc}},  \code{\link{rsegIITe}}, and \code{\link{rsegMT}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(0,0); B<-c(1,0); C<-c(1/2,sqrt(3)/2);
#' Te<-rbind(A,B,C);
#' n<-10  #try also n<-20 or n<-100 or 1000
#' eps<-.3  #try also .15, .5, .75
#'
#' set.seed(1)
#' Xdt<-rsegTe(n,eps)
#' Xdt
#' summary(Xdt)
#' plot(Xdt,asp=1)
#'
#' Xlim<-range(Te[,1])
#' Ylim<-range(Te[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' set.seed(1)
#' dat<-rsegTe(n,eps)$gen.points
#'
#' plot(Te,asp=1,pch=".",xlab="",ylab="",
#' main="Type I segregation in the \n standard equilateral triangle",
#'      xlim=Xlim+xd*c(-.01,.01),ylim=Ylim+yd*c(-.01,.01))
#' polygon(Te)
#' points(dat)
#'
#' #The support for the Type I segregation alternative
#' sr<-eps/(sqrt(3)/2)
#' C1<-C+sr*(A-C); C2<-C+sr*(B-C)
#' A1<-A+sr*(B-A); A2<-A+sr*(C-A)
#' B1<-B+sr*(A-B); B2<-B+sr*(C-B)
#' supp<-rbind(A1,B1,B2,C2,C1,A2)
#'
#' plot(Te,asp=1,pch=".",xlab="",ylab="",main="Support of the Type I Segregation",
#'      xlim=Xlim+xd*c(-.01,.01),ylim=Ylim+yd*c(-.01,.01))
#' if (sr<=.5)
#' {
#'   polygon(Te)
#'   polygon(supp,col=5)
#' } else
#' {
#'   polygon(Te,col=5,lwd=2.5)
#'   polygon(rbind(A,A1,A2),col=0,border=NA)
#'   polygon(rbind(B,B1,B2),col=0,border=NA)
#'   polygon(rbind(C,C1,C2),col=0,border=NA)
#' }
#' points(dat)
#'
#' @export rsegTe
rsegTe <- function(k,eps)
{
  if (!is.point(eps,1) || eps<=0 || eps>=sqrt(3)/3)
  {stop('eps must be a scalar in (0,sqrt(3)/3=0.5773503)')}

  Y<-rbind(c(0,0),c(1,0),c(1/2,sqrt(3)/2))  #std eq triangle
  ny<-nrow(Y)

  X <-matrix(0,k,2);
  if (eps < 0.4330127 )  #(eps < sqrt(3)/4 )
  {
    for (i in 1:k)
    {ct<-0;
    while (ct==0)
    {
      x<-runif(1,0.5773503*eps,1-0.5773503*eps); y<-runif(1,0,0.8660254-eps);
      if (y < 1.732050808*x && y < 1.732050808-1.732050808*x && y > 1.732051*x+2*eps-1.732051 && y > -1.732051*x+2*eps)
      {X[i,]<-c(x,y);
      ct<-1;
      }
    }
    }
  }
  else
  {
    for (i in 1:k)
    {ct<-0;
    while (ct==0)
    {
      x<-runif(1,-.5+1.732050809*eps,1.5-1.732050809*eps); y<-runif(1,-.866025404+2*eps,.866025404-eps);
      if (y > 2*eps-1.732050808*x && y > 2*eps-1.732050808+1.732050808*x)
      {X[i,]<-c(x,y);
      ct<-1;
      }
    }
    }
  }
  ename <-deparse(substitute(eps))

  param<-eps
  names(param)<-"exclusion parameter"
  typ<-paste("Type I Segregation of ",k, " points from vertices of the standard equilateral triangle with exclusion parameter ",ename, "=",eps,sep="")

  npts<-c(k,ny)
  names(npts)<-c("nx","ny")

  txt<-"Type I Segregation of One Class from Vertices of the Standard Equilateral Triangle"
  main.txt<-paste("Type I Segregation in the Standard Equilateral Triangle \n with Exclusion Parameter ",ename,"=",eps,sep="")

  res<-list(
    type=typ,
    parameters=param,
    gen.points=X, #generated points segregated from Y points (vertices of std eq triangle)
    ref.points=Y, #reference points, i.e., points from which generated points are segregated (vertices of std eq triangle)
    desc.pat=txt, #description of the pattern
    mtitle=main.txt,
    tri.Y=TRUE,
    num.points=npts,
    xlimit=range(Y[,1]),
    ylimit=range(Y[,2])
  )

  class(res)<-"Patterns"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

#' @title Generation of points segregated (in a Type I fashion) from the vertices of a triangle
#'
#' @description
#' An object of class \code{"Patterns"}.
#' Generates \code{k} points uniformly in the support for Type I segregation in
#' a given triangle, \code{tri}.
#'
#' \code{delta} is the parameter of segregation (that is, \eqn{\delta 100} \% of the area around each vertex
#' in the triangle is forbidden for point generation). \code{delta} corresponds to \code{eps} in the
#' standard equilateral triangle \eqn{T_e} as \eqn{delta=4 eps ^2/3} (see \code{rsegTe}  function).
#'
#' See (\insertCite{ceyhan:arc-density-PE,ceyhan:arc-density-CS,ceyhan:dom-num-NPE-Spat2011;textual}{pcds}) for more on
#' the segregation pattern.
#'
#' @param k A positive integer representing the number of points to be generated from the segregation pattern
#' in the triangle, \code{tri}.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param delta A positive real number in \eqn{(0,4/9)}. \code{delta} is the parameter of segregation (that is,
#' \eqn{\delta 100} \% area around each vertex in each Delaunay triangle is forbidden for point generation).
#'
#' @return A \code{list} with the elements
#' \item{type}{The type of the pattern from which points are to be generated}
#' \item{mtitle}{The \code{"main"} title for the plot of the point pattern}
#' \item{parameters}{Exclusion parameter, \code{delta}, of the Type I segregation pattern. \code{delta} is in \eqn{(0,4/9)}
#' \eqn{\delta 100} \% area around each vertex in the triangle \code{tri} is forbidden for point generation.}
#' \item{ref.points}{The input set of points, i.e., vertices of \code{tri};
#' reference points, i.e., points from which generated points are segregated.}
#' \item{gen.points}{The output set of generated points segregated from the vertices of \code{tri}.}
#' \item{tri.Y}{Logical output, if \code{TRUE} the triangle \code{tri} is also plotted when the
#' corresponding plot function from the \code{Patterns} object is called.}
#' \item{desc.pat}{Description of the point pattern}
#' \item{num.points}{The \code{vector} of two numbers, which are the number of generated points
#' and the number of reference (i.e., vertex of \code{tri}, which is 3 here).}
#' \item{xlimit,ylimit}{The ranges of the \eqn{x}- and \eqn{y}-coordinates of the reference points, which are the
#' vertices of the triangle \code{tri}}
#'
#' @seealso \code{\link{rasc.tri}}, \code{\link{rsegTe}}, \code{\link{rsegIITe}}, and \code{\link{rsegMT}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' n<-10  #try also n<-100
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C)
#' del<-.4
#'
#' Xdt<-rseg.tri(n,Tr,del)
#' Xdt
#' summary(Xdt)
#' plot(Xdt)
#'
#' dat<-rseg.tri(n,Tr,del)$g
#' Xlim<-range(Tr[,1])
#' Ylim<-range(Tr[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(Tr,pch=".",xlab="",ylab="",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tr)
#' points(dat)
#' xc<-Tr[,1]+c(-.01,.01,.01)
#' yc<-Tr[,2]+c(.02,.02,.02)
#' txt.str<-c("A","B","C")
#' text(xc,yc,txt.str)
#'
#' dat.fr<-data.frame(a=Tr)
#' rseg.tri(n,dat.fr,del)
#'
#' @export
rseg.tri <- function(k,tri,delta)
{
  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  if (!is.point(delta,1) || delta<=0 || delta>=1)
  {stop('delta must be a scalar in (0,1)')}

  cnt<-1;ind<-0
  dat<-NULL
  while (cnt <= k)
  {
    pnt<-as.vector(runif.tri(1,tri)$g)
    seg.tri<-seg.tri.supp(delta,tri)
    if (in.triangle(pnt,seg.tri,boundary=TRUE)$in.tri)
    {
      dat<-rbind(dat,pnt);
      cnt<-cnt+1
    }
  }

  dname <-deparse(substitute(delta))

  param<-delta
  names(param)<-"exclusion parameter"
  typ<-paste("Type I Segregation of ",k, " points in the triangle with vertices (",tri[1,1],",",tri[1,2],")",", ",
             "(",tri[2,1],",",tri[2,2],")"," and ","(",tri[3,1],",",tri[3,2],") with exclusion parameter ",dname, "=",delta,sep="")

  npts<-c(k,3)
  names(npts)<-c("nx","ny")

  txt<-"Type I Segregation of One Class from Vertices of the Triangle"
  main.txt<-paste("Type I Segregation in the One-Triangle Case \n with  Exclusion Parameter ",dname,"=",delta,sep="")

  res<-list(
    type=typ,
    parameters=param,
    gen.points=dat, #generated points segregated from Y points (vertices of the triangle, tri)
    ref.points=tri, #reference points, i.e., points from which generated points are segregated (vertices of the triangle)
    desc.pat=txt, #description of the pattern
    mtitle=main.txt,
    tri.Y=TRUE,
    num.points=npts,
    xlimit=range(tri[,1]),
    ylimit=range(tri[,2])
  )

  class(res)<-"Patterns"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

#' @title Generation of points segregated (in a Type II fashion) from the vertices of \eqn{T_e}
#'
#' @description
#' An object of class \code{"Patterns"}.
#' Generates \code{k} points uniformly in the standard equilateral triangle \eqn{T_e=T((0,0),(1,0),(1/2,\sqrt{3}/2))}
#' under the type II segregation alternative for \code{eps} in \eqn{(0,\sqrt{3}/6=0.2886751]}.
#'
#' In the type II segregation, the annular forbidden regions around the edges are determined by
#' the parameter \code{eps} which is the distance from the interior triangle (i.e., support for the segregation)
#' to \eqn{T_e} (see examples for a sample plot.)
#'
#' @param k A positive integer representing the number of points to be generated.
#' @param eps A positive real number representing the parameter of type II segregation (which is the
#' distance from the interior triangle to \eqn{T_e}).
#'
#' @return A \code{list} with the elements
#' \item{type}{The type of the point pattern}
#' \item{mtitle}{The \code{"main"} title for the plot of the point pattern}
#' \item{parameters}{The exclusion parameter, \code{eps}, of the segregation pattern, which is the distance
#' from the interior triangle to \eqn{T_e}}
#' \item{ref.points}{The input set of points \code{Y};
#' reference points, i.e., points from which generated points are segregated (i.e., vertices of \eqn{T_e}).}
#' \item{gen.points}{The output set of generated points segregated from \code{Y} points (i.e., vertices of \eqn{T_e}).}
#' \item{tri.Y}{Logical output for triangulation based on \code{Y} points should be implemented or not.
#' if \code{TRUE} triangulation based on \code{Y} points is to be implemented (default is set to \code{FALSE}).}
#' \item{desc.pat}{Description of the point pattern}
#' \item{num.points}{The \code{vector} of two numbers, which are the number of generated points and
#' the number of reference (i.e., \code{Y}) points, which is 3 here.}
#' \item{xlimit,ylimit}{The ranges of the \eqn{x}- and \eqn{y}-coordinates of the reference points, which are the
#' vertices of \eqn{T_e} here}
#'
#' @seealso \code{\link{rseg.disc}}, \code{\link{rasc.disc}},  \code{\link{rsegTe}}, and \code{\link{rsegMT}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(0,0); B<-c(1,0); C<-c(1/2,sqrt(3)/2);
#' Te<-rbind(A,B,C);
#' n<-10  #try also n<-20 or n<-100 or 1000
#' eps<-.15  #try also .2
#'
#' set.seed(1)
#' Xdt<-rsegIITe(n,eps)
#' Xdt
#' summary(Xdt)
#' plot(Xdt,asp=1)
#'
#' Xlim<-range(Te[,1])
#' Ylim<-range(Te[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' dat<-rsegIITe(n,eps)$gen.points
#'
#' plot(Te,pch=".",xlab="",ylab="",main="Type II segregation in the \n standard equilateral triangle",
#'      xlim=Xlim+xd*c(-.01,.01),ylim=Ylim+yd*c(-.01,.01))
#' polygon(Te)
#' points(dat)
#'
#' #The support for the Type II segregation alternative
#' C1<-c(1/2,sqrt(3)/2-2*eps);
#' A1<-c(eps*sqrt(3),eps); B1<-c(1-eps*sqrt(3),eps);
#' supp<-rbind(A1,B1,C1)
#'
#' plot(Te,asp=1,pch=".",xlab="",ylab="",main="Support of the Type II Segregation",
#'      xlim=Xlim+xd*c(-.01,.01),ylim=Ylim+yd*c(-.01,.01))
#'   polygon(Te)
#'   polygon(supp,col=5)
#' points(dat)
#'
#' @export rsegIITe
rsegIITe <- function(k,eps)
{
  if (!is.point(eps,1) || eps<=0 || eps>=sqrt(3)/6)
  {stop('eps must be a scalar in (0,sqrt(3)/6=0.2886751)')}

  Y<-rbind(c(0,0),c(1,0),c(1/2,sqrt(3)/2))  #std eq triangle
  ny<-nrow(Y)

  X <-matrix(0,k,2);
  for (i in 1:k)
  {ct<-0;
  while (ct==0)
  {
    x<-runif(1,1.732050808*eps,1-1.732050808*eps); y<-runif(1,eps,0.8660254-2*eps);
    if (y < -2*eps+1.732050808*x && y < -2*eps+1.732050808-1.732050808*x)
    {X[i,]<-c(x,y);
    ct<-1;
    }
  }
  }
  ename <-deparse(substitute(eps))

  param<-eps
  names(param)<-"exclusion parameter"
  typ<-paste("Type II Segregation of ",k, " points from edges of the standard equilateral triangle with exclusion parameter ",ename, "=",eps,sep="")

  npts<-c(k,ny)
  names(npts)<-c("nx","ny")

  txt<-"Type II Segregation of One Class from Edges of the Standard Equilateral Triangle"
  main.txt<-paste("Type II Segregation in the \n Standard Equilateral Triangle \n with Exclusion Parameter ",ename,"=",eps,sep="")

  res<-list(
    type=typ,
    parameters=param,
    gen.points=X, #generated points segregated from Y points (vertices of std eq triangle)
    ref.points=Y, #reference points, i.e., points from which generated points are segregated (vertices of std eq triangle)
    desc.pat=txt, #description of the pattern
    mtitle=main.txt,
    tri.Y=TRUE,
    num.points=npts,
    xlimit=range(Y[,1]),
    ylimit=range(Y[,2])
  )

  class(res)<-"Patterns"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

#' @title Generation of points associated (in a Type I fashion) with the vertices of \eqn{T_e}
#'
#' @description
#' An object of class \code{"Patterns"}.
#' Generates \code{k} points uniformly in the standard equilateral triangle \eqn{T_e=T((0,0),(1,0),(1/2,\sqrt{3}/2))}
#' under the type I association alternative for \code{eps} in \eqn{(0,\sqrt{3}/3=0.5773503]}.
#' The allowed triangular regions around the vertices are determined by the parameter \code{eps}.
#'
#' In the type I association, the triangular support regions around the vertices are determined by
#' the parameter \code{eps} where \eqn{\sqrt{3}/3}-\code{eps} serves as the height of these triangles
#' (see examples for a sample plot.)
#'
#' See also (\insertCite{ceyhan:arc-density-PE,ceyhan:arc-density-CS,ceyhan:dom-num-NPE-Spat2011;textual}{pcds}).
#'
#' @param k A positive integer representing the number of points to be generated.
#' @param eps A positive real number representing the parameter of type I association (where \eqn{\sqrt{3}/3}-\code{eps}
#' serves as the height of the triangular support regions around the vertices).
#'
#' @return A \code{list} with the elements
#' \item{type}{The type of the point pattern}
#' \item{mtitle}{The \code{"main"} title for the plot of the point pattern}
#' \item{parameters}{The attraction parameter of the association pattern, \code{eps},
#' where \eqn{\sqrt{3}/3}-\code{eps} serves as the height of the triangular support regions around the vertices}
#' \item{ref.points}{The input set of points \code{Y};
#' reference points, i.e., points with which generated points are associated (i.e., vertices of \eqn{T_e}).}
#' \item{gen.points}{The output set of generated points associated with \code{Y} points (i.e., vertices of \eqn{T_e}).}
#' \item{tri.Y}{Logical output for triangulation based on \code{Y} points should be implemented or not.
#' if \code{TRUE} triangulation based on \code{Y} points is to be implemented (default is set to \code{FALSE}).}
#' \item{desc.pat}{Description of the point pattern.}
#' \item{num.points}{The \code{vector} of two numbers,
#' which are the number of generated points and the number of reference (i.e., \code{Y}) points.}
#' \item{xlimit,ylimit}{The ranges of the \eqn{x}- and \eqn{y}-coordinates of the reference points, which are the
#' vertices of \eqn{T_e} here}
#'
#' @seealso \code{\link{rseg.disc}}, \code{\link{rasc.disc}},  \code{\link{rsegIITe}}, and \code{\link{rsegMT}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(0,0); B<-c(1,0); C<-c(1/2,sqrt(3)/2);
#' Te<-rbind(A,B,C);
#' n<-10  #try also n<-20 or n<-100 or 1000
#' eps<-.25  #try also .15, .5, .75
#'
#' set.seed(1)
#' Xdt<-rascTe(n,eps)
#' Xdt
#' summary(Xdt)
#' plot(Xdt,asp=1)
#'
#' Xlim<-range(Te[,1])
#' Ylim<-range(Te[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' dat<-rascTe(n,eps)$gen.points
#' plot(Te,pch=".",xlab="",ylab="",
#' main="Type I association in the \n standard equilateral triangle",
#'      xlim=Xlim+xd*c(-.01,.01),ylim=Ylim+yd*c(-.01,.01))
#' polygon(Te)
#' points(dat)
#'
#' #The support for the Type I association alternative
#' sr<-(sqrt(3)/3-eps)/(sqrt(3)/2)
#' C1<-C+sr*(A-C); C2<-C+sr*(B-C)
#' A1<-A+sr*(B-A); A2<-A+sr*(C-A)
#' B1<-B+sr*(A-B); B2<-B+sr*(C-B)
#' supp<-rbind(A1,B1,B2,C2,C1,A2)
#'
#' plot(Te,asp=1,pch=".",xlab="",ylab="",
#' main="Support of the Type I Association",
#' xlim=Xlim+xd*c(-.01,.01),ylim=Ylim+yd*c(-.01,.01))
#' if (sr<=.5)
#' {
#'   polygon(Te,col=5)
#'   polygon(supp,col=0)
#' } else
#' {
#'   polygon(Te,col=0,lwd=2.5)
#'   polygon(rbind(A,A1,A2),col=5,border=NA)
#'   polygon(rbind(B,B1,B2),col=5,border=NA)
#'   polygon(rbind(C,C1,C2),col=5,border=NA)
#' }
#' points(dat)
#'
#' @export rascTe
rascTe <- function(k,eps)
{
  if (!is.point(eps,1) || eps<=0 || eps>=sqrt(3)/3 )
  {stop('eps must be a scalar in (0,sqrt(3)/3=0.5773503)')}

  Y<-rbind(c(0,0),c(1,0),c(1/2,sqrt(3)/2))  #std eq triangle
  ny<-nrow(Y)

  X<-matrix(0,k,2);
  Eps<-0.5773503-eps; #sqrt(3)/3-eps
  for (i in 1:k)
  {ct<-0;
  while (ct==0)
  {
    x<-runif(1,0,1); y<-runif(1,0,0.8660254);
    if ( (y < 1.732050808*x && y < 1.732050808-1.732050808*x) &&
         !(y > 1.732051*x+2*Eps-1.732051 & y > -1.732051*x+2*Eps & y < 0.8660254-Eps) )
    {X[i,]<-c(x,y);
    ct<-1;
    }
  }
  }
  ename <-deparse(substitute(eps))

  param<-eps
  names(param)<-"attraction parameter"
  typ<-paste("Type I Association of ",k, " points with vertices of the standard equilateral triangle with attraction parameter ",ename, "=",eps,sep="")

  npts<-c(k,ny)
  names(npts)<-c("nx","ny")

  txt<-"Type I Association of One Class with Vertices of the Standard Equilateral Triangle"
  main.txt<-paste("Type I Association in the \n Standard Equilateral Triangle \n with Attraction Parameter ",ename,"=",eps,sep="")

  res<-list(
    type=typ,
    parameters=param,
    gen.points=X, #generated points associated with Y points (vertices of std eq triangle)
    ref.points=Y, #reference points, i.e., points with which generated points are associated (vertices of std eq triangle)
    desc.pat=txt, #description of the pattern
    mtitle=main.txt,
    tri.Y=TRUE,
    num.points=npts,
    xlimit=range(Y[,1]),
    ylimit=range(Y[,2])
  )

  class(res)<-"Patterns"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

#' @title Generation of points associated (in a Type I fashion) with the vertices of a triangle
#'
#' @description
#' An object of class \code{"Patterns"}.
#' Generates \code{k} points uniformly in the support for Type I association in
#' a given triangle, \code{tri}.
#' \code{delta} is the parameter of association (that is, only \eqn{\delta 100} \% area around each vertex in
#' the triangle is allowed for point generation). \code{delta} corresponds to \code{eps} in the standard equilateral triangle
#' \eqn{T_e} as \eqn{delta=4eps^2/3} (see \code{rsegTe}  function).
#'
#' See (\insertCite{ceyhan:arc-density-PE,ceyhan:arc-density-CS,ceyhan:dom-num-NPE-Spat2011;textual}{pcds}) for more on
#' the association pattern.
#'
#' @param k A positive integer representing the number of points to be generated from the association pattern
#' in the triangle, \code{tri}.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param delta A positive real number in \eqn{(0,4/9)}. \code{delta} is the parameter of association (that is, only
#' \eqn{\delta 100} \% area around each vertex in the triangle is allowed for point generation).
#'
#' @return A \code{list} with the elements
#' \item{type}{The type of the pattern from which points are to be generated}
#' \item{mtitle}{The \code{"main"} title for the plot of the point pattern}
#' \item{parameters}{Attraction parameter, \code{delta}, of the Type I association pattern. \code{delta} is in \eqn{(0,4/9)}
#' only \eqn{\delta 100} \% of the area around each vertex in the triangle \code{tri} is allowed for point generation.}
#' \item{ref.points}{The input set of points, i.e., vertices of \code{tri};
#' reference points, i.e., points with which generated points are associated.}
#' \item{gen.points}{The output set of generated points associated with the vertices of \code{tri}.}
#' \item{tri.Y}{Logical output, \code{TRUE} if triangulation based on \code{Yp} points should be implemented.}
#' \item{desc.pat}{Description of the point pattern}
#' \item{num.points}{The \code{vector} of two numbers, which are the number of generated points
#' and the number of reference (i.e., \code{Yp}) points.}
#' \item{xlimit,ylimit}{The ranges of the \eqn{x}- and \eqn{y}-coordinates of the reference points, which are the
#' \code{Yp} points}
#'
#' @seealso \code{\link{rseg.tri}}, \code{\link{rascTe}}, \code{\link{rascIITe}}, and \code{\link{rascMT}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' n<-10  #try also n<-100
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C)
#' del<-.4
#'
#' Xdt<-rasc.tri(n,Tr,del)
#' Xdt
#' summary(Xdt)
#' plot(Xdt)
#'
#' dat<-rasc.tri(n,Tr,del)$g
#' Xlim<-range(Tr[,1])
#' Ylim<-range(Tr[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(Tr,pch=".",xlab="",ylab="",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tr)
#' points(dat)
#' xc<-Tr[,1]+c(-.01,.01,.01)
#' yc<-Tr[,2]+c(.02,.02,.02)
#' txt.str<-c("A","B","C")
#' text(xc,yc,txt.str)
#'
#' dat.fr<-data.frame(a=Tr)
#' rasc.tri(n,dat.fr,del)
#'
#' @export
rasc.tri <- function(k,tri,delta)
{
  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  if (!is.point(delta,1) || delta<=0 || delta>=1)
  {stop('delta must be a scalar in (0,1)')}

  cnt<-1;ind<-0
  dat<-NULL
  while (cnt <= k)
  {
    pnt<-as.vector(runif.tri(1,tri)$g)
    seg.tri<-seg.tri.supp(delta,tri)
    if (!in.triangle(pnt,seg.tri)$in.tri)
    {
      dat<-rbind(dat,pnt);
      cnt<-cnt+1
    }
  }

  dname <-deparse(substitute(delta))

  param<-delta
  names(param)<-"attraction parameter"
  typ<-paste("Type I Association of ",k, " points in the triangle with vertices (",tri[1,1],",",tri[1,2],")",", ",
             "(",tri[2,1],",",tri[2,2],")"," and ","(",tri[3,1],",",tri[3,2],") with attraction parameter ",dname, "=",delta,sep="")

  npts<-c(k,3)
  names(npts)<-c("nx","ny")

  txt<-"Type I Association of One Class with Vertices of the Triangle"
  main.txt<-paste("Type I Association in the One-Triangle Case \n with Attraction Parameter ",dname,"=",delta,sep="")

  res<-list(
    type=typ,
    parameters=param,
    gen.points=dat, #generated points associated with Y points (vertices of the triangle, tri)
    ref.points=tri, #reference points, i.e., points with which generated points are associated (vertices of the triangle)
    desc.pat=txt, #description of the pattern
    mtitle=main.txt,
    tri.Y=TRUE,
    num.points=npts,
    xlimit=range(tri[,1]),
    ylimit=range(tri[,2])
  )

  class(res)<-"Patterns"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

#' @title Generation of points associated (in a Type II fashion) with the edges of \eqn{T_e}
#'
#' @description
#' An object of class \code{"Patterns"}.
#' Generates \code{k} points uniformly in the standard equilateral triangle \eqn{T_e=T((0,0),(1,0),(1/2,\sqrt{3}/2))}
#' under the type II association alternative for \code{eps} in \eqn{(0,\sqrt{3}/6=0.2886751]}.
#'
#' In the type II association, the annular allowed regions around the edges are determined by
#' the parameter \code{eps} where \eqn{\sqrt{3}/6}-\code{eps} is the distance from the interior triangle
#' (i.e., forbidden region for association) to \eqn{T_e} (see examples for a sample plot.)
#'
#' @param k A positive integer representing the number of points to be generated.
#' @param eps A positive real number representing the parameter of type II association (where \eqn{\sqrt{3}/6}-\code{eps}
#' is the distance from the interior triangle distance from the interior triangle to \eqn{T_e}).
#'
#' @return A \code{list} with the elements
#' \item{type}{The type of the point pattern}
#' \item{mtitle}{The \code{"main"} title for the plot of the point pattern}
#' \item{parameters}{The attraction parameter, \code{eps}, of the association pattern, where \eqn{\sqrt{3}/6}-\code{eps}
#' is the distance from the interior triangle to \eqn{T_e}}
#' \item{ref.points}{The input set of points \code{Y};
#' reference points, i.e., points with which generated points are associated (i.e., vertices of \eqn{T_e}).}
#' \item{gen.points}{The output set of generated points associated with \code{Y} points (i.e., edges of \eqn{T_e}).}
#' \item{tri.Y}{Logical output for triangulation based on \code{Y} points should be implemented or not.
#' if \code{TRUE} triangulation based on \code{Y} points is to be implemented (default is set to \code{FALSE}).}
#' \item{desc.pat}{Description of the point pattern}
#' \item{num.points}{The \code{vector} of two numbers, which are the number of generated points and
#' the number of reference (i.e., \code{Y}) points, which is 3 here.}
#' \item{xlimit,ylimit}{The ranges of the \eqn{x}- and \eqn{y}-coordinates of the reference points, which are the
#' vertices of \eqn{T_e} here}
#'
#' @seealso \code{\link{rseg.disc}}, \code{\link{rasc.disc}},  \code{\link{rsegIITe}}, and \code{\link{rsegMT}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(0,0); B<-c(1,0); C<-c(1/2,sqrt(3)/2);
#' Te<-rbind(A,B,C);
#' n<-10  #try also n<-20 or n<-100 or 1000
#' eps<-.2  #try also .25, .1
#'
#' set.seed(1)
#' Xdt<-rascIITe(n,eps)
#' Xdt
#' summary(Xdt)
#' plot(Xdt,asp=1)
#'
#' Xlim<-range(Te[,1])
#' Ylim<-range(Te[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' dat<-rascIITe(n,eps)$gen.points
#' plot(Te,pch=".",xlab="",ylab="",
#' main="Type II association in the \n standard equilateral triangle",
#'      xlim=Xlim+xd*c(-.01,.01),ylim=Ylim+yd*c(-.01,.01))
#' polygon(Te)
#' points(dat)
#'
#' #The support for the Type II association alternative
#' A1<-c(1/2-eps*sqrt(3),sqrt(3)/6-eps); B1<-c(1/2+eps*sqrt(3),sqrt(3)/6-eps);
#' C1<-c(1/2,sqrt(3)/6+2*eps);
#' supp<-rbind(A1,B1,C1)
#'
#' plot(Te,asp=1,pch=".",xlab="",ylab="",main="Support of the Type II Association",
#' xlim=Xlim+xd*c(-.01,.01),ylim=Ylim+yd*c(-.01,.01))
#' polygon(Te,col=5)
#' polygon(supp,col=0)
#' points(dat)
#'
#' @export rascIITe
rascIITe <- function(k,eps)
{
  if (!is.point(eps,1) || eps<=0 || eps>=sqrt(3)/6 )
  {stop('eps must be a scalar in (0,sqrt(3)/6=0.2886751)')}

  Y<-rbind(c(0,0),c(1,0),c(1/2,sqrt(3)/2))  #std eq triangle
  ny<-nrow(Y)

  X <-matrix(0,k,2);
  Eps<-0.2886751-eps; #sqrt(3)/6-eps;
  for (i in 1:k)
  {ct<-0;
  while (ct==0)
  {
    x<-runif(1,0,1); y<-runif(1,0,0.8660254);
    if ( (y < 1.732050808*x && y < 1.732050808-1.732050808*x) &&
         !(y > Eps & y < -2*Eps+1.732050808*x  & y < -2*Eps+1.732050808-1.732050808*x) )
    {X[i,]<-c(x,y);
    ct<-1;
    }
  }
  }
  ename <-deparse(substitute(eps))

  param<-eps
  names(param)<-"attraction parameter"
  typ<-paste("Type II Association of ",k, " points with edges of the standard equilateral triangle with attraction parameter ",ename, "=",eps,sep="")

  npts<-c(k,ny)
  names(npts)<-c("nx","ny")

  txt<-"Type II Association of One Class with Edges of the Standard Equilateral Triangle"
  main.txt<-paste("Type II Association Pattern\n in the Standard Equilateral Triangle \n with Attraction Parameter ",ename,"=",eps,sep="")
  res<-list(
    type=typ,
    parameters=param,
    gen.points=X, #generated points associated with Y points (edges of std eq triangle)
    ref.points=Y, #reference points, i.e., points with which generated points are associated (edges eq triangle)
    desc.pat=txt, #description of the pattern
    mtitle=main.txt,
    tri.Y=TRUE,
    num.points=npts,
    xlimit=range(Y[,1]),
    ylimit=range(Y[,2])
  )

  class(res)<-"Patterns"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

#' @title The line passing through a point and perpendicular to the line segment joining two points
#'
#' @description
#' An object of class \code{"Lines"}.
#' Returns the \code{equation, slope, intercept}, and \eqn{y}-coordinates of the line crossing
#' the point \code{p} and perpendicular to the line passing through the points \code{a} and \code{b}
#' with \eqn{x}-coordinates are provided in \code{vector} \code{x}.
#'
#' @param p A 2D point at which the perpendicular line to line segment joining \code{a} and \code{b} crosses.
#' @param a,b 2D points that determine the line segment (the line will be perpendicular to this line segment).
#' @param x A scalar or a \code{vector} of scalars representing the \eqn{x}-coordinates of the line perpendicular to
#' line joining \code{a} and \code{b} and crossing \code{p}.
#'
#' @return A \code{list} with the elements
#' \item{desc}{Description of the line passing through point \code{p} and perpendicular to line joining
#' \code{a} and \code{b}}
#' \item{mtitle}{The \code{"main"} title for the plot of the line passing through point \code{p} and perpendicular to line joining
#' \code{a} and \code{b}}
#' \item{points}{The input points \code{a} and \code{b} (stacked row-wise,
#' i.e., row 1 is point \code{a} and row 2 is point \code{b}). Line passing through point \code{p} is perpendicular
#' to line joining \code{a} and \code{b}}
#' \item{x}{The input vector, can be a scalar or a \code{vector} of scalars,
#' which constitute the \eqn{x}-coordinates of the point(s) of interest on the line passing through point \code{p} and perpendicular to line joining
#' \code{a} and \code{b}}
#' \item{y}{The output \code{vector} which constitutes the \eqn{y}-coordinates of the point(s) of interest on the line passing through point \code{p} and perpendicular to line joining
#' \code{a} and \code{b}. If \code{x} is a scalar, then \code{y} will be a scalar and if \code{x} is a \code{vector} of scalars,
#' then \code{y} will be a \code{vector} of scalars.}
#' \item{slope}{Slope of the line passing through point \code{p} and perpendicular to line joining
#' \code{a} and \code{b}}
#' \item{intercept}{Intercept of the line passing through point \code{p} and perpendicular to line joining
#' \code{a} and \code{b}}
#' \item{equation}{Equation of the line passing through point \code{p} and perpendicular to line joining
#' \code{a} and \code{b}}
#'
#' @seealso \code{\link{slope}}, \code{\link{Line}}, and \code{\link{paraline}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1.1,1.2); B<-c(2.3,3.4); p<-c(.51,2.5)
#'
#' perpline(p,A,B,.45)
#' perpline(A,A,B,.45)
#'
#' pts<-rbind(A,B,p)
#' xr<-range(pts)
#' xf<-(xr[2]-xr[1])*.25 #how far to go at the lower and upper ends in the x-coordinate
#' x<-seq(xr[1]-xf,xr[2]+xf,l=20)  #try also l=100
#'
#' plnAB<-perpline(p,A,B,x)
#' plnAB
#' summary(plnAB)
#' plot(plnAB,asp=1)
#'
#' y<-plnAB$y
#' Xlim<-range(x,pts[,1])
#' if (!is.na(y[1])) {Ylim<-range(y,pts[,2])} else {Ylim<-range(pts[,2])}
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#' pf<-c(xd,-yd)*.025
#'
#' plot(A,asp=1,pch=".",xlab="",ylab="",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' points(pts)
#' txt.str<-c("A","B","p")
#' text(pts+rbind(pf,pf,pf),txt.str)
#'
#' segments(A[1],A[2],B[1],B[2],lty=2)
#' if (!is.na(y[1])) {lines(x,y,type="l",lty=1,xlim=Xlim,ylim=Ylim)} else {abline(v=p[1])}
#' tx<-p[1]+abs(xr-p[1])/2;
#' if (!is.na(y[1])) {ty<-perpline(p,A,B,tx)$y} else {ty=p[2]}
#' text(tx,ty,"line perpendicular to AB\n and crossing p")
#'
#' @export perpline
perpline <- function(p,a,b,x)
{
  if (!is.point(p) || !is.point(a) || !is.point(b))
  {stop('p, a and b must all be numeric points of dimension 2')}

  if (all(a==b))
  {stop('a and b are same, hence lines are not well defined')}

  if (!is.vector(x))
  {stop('x must be a vector')}

  sl<--1/slope(a,b);
 int<-p[2]+sl *(0-p[1]);
  perpln<-p[2]+sl *(x-p[1])

  if (abs(sl)==Inf)
  {
   int<-perpln<-NA
  } else
  {
   int<-p[2]+sl *(0-p[1]);
    perpln<-p[2]+sl *(x-p[1])
  }

  names(sl)<-"slope"
  names(int)<-"intercept"

  aname <-deparse(substitute(a))
  bname <-deparse(substitute(b))
  pname <-deparse(substitute(p))

  line.desc<-paste("Line Passing thru Point ",pname, " Perpendicular to Line Segment [",aname,bname,"]",sep="")
  main.txt<-paste("Line Passing thru Point ",pname, "\n Perpendicular to Line Segment [",aname,bname,"]",sep="")

  pts<-rbind(a,b,p)
  row.names(pts)<-c(aname,bname,pname)

  if (abs(sl)==Inf)
  { vert<-p[1]
  eqn<-reqn<-paste("x =",vert)
  } else
  {
    vert<-NA
    eqn<-ifelse(sl==0 & int==0,"y=0",ifelse(sl!=0 & int==0,paste("y=",sl,"x",sep=""),
                                            ifelse(sl==0 & int!=0,paste("y=",int,sep=""),ifelse(sl==1,
                                                                                                ifelse(sign(int)<0,paste("y=x",int,sep=""),paste("y=x+",int,sep="")),
                                                                                                ifelse(sign(int)<0,paste("y=",sl,"x",int,sep=""),paste("y=",sl,"x+",int,sep="")))) ))
    rsl<-round(sl,2)
    rint<-round(int,2)
    reqn<-ifelse(rsl==0 & rint==0,"y=0",ifelse(rsl!=0 & rint==0,paste("y=",rsl,"x",sep=""),
                                               ifelse(rsl==0 & rint!=0,paste("y=",rint,sep=""),ifelse(rsl==1,
                                                                                                      ifelse(sign(rint)<0,paste("y=x",rint,sep=""),paste("y=x+",rint,sep="")),
                                                                                                      ifelse(sign(rint)<0,paste("y=",rsl,"x",rint,sep=""),paste("y=",rsl,"x+",rint,sep="")))) ))
  }

  res<-list(
    desc=line.desc, mtitle=main.txt,
    points=pts, vert=vert,
    x=x,
    y=perpln,
    slope=sl,
   intercept=int,
    equation=eqn,
    eqnlabel=reqn
  )

  class(res)<-"Lines"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

# funsRankOrderTe
#'
#' @title Returns the ranks and orders of points in decreasing distance to the edges of the triangle
#'
#' @description
#' Two functions: \code{rank.d2e.Te} and \code{order.d2e.Te}.
#'
#' \code{rank.d2e.Te} finds the ranks of the distances of points in data, \code{Dt}, to the edges of the standard equilateral triangle
#' \eqn{T_e=T((0,0),(1,0),(1/2,\sqrt{3}/2))}
#'
#' \code{dec} is a logical argument, default is \code{TRUE}, so the ranks are for decreasing distances, if \code{FALSE} it will be
#' in increasing distances.
#'
#' \code{order.d2e.Te} finds the orders of the distances of points in data, \code{Dt}, to the edges of \eqn{T_e}. The arguments are
#' as in \code{rank.d2e.Te}.
#'
#' @param Dt A set of 2D points representing the data set in which ranking in terms of the distance
#' to the edges of \eqn{T_e} is performed.
#' @param dec A logical argument indicating the how the ranking will be performed. If \code{TRUE},
#' the ranks are for decreasing distances,and if \code{FALSE} they will be in increasing distances,
#' default is \code{TRUE}.
#'
#' @return A \code{list} with two elements
#' \item{distances}{Distances from data points to the edges of \eqn{T_e}}
#' \item{dist.rank}{The ranks of the data points in decreasing distances to the edges of \eqn{T_e}}
#'
#' @name funsRankOrderTe
NULL
#'
#' @rdname funsRankOrderTe
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' #Examples for rank.d2e.Te
#' n<-20
#' set.seed(1)
#' dat<-runifTe(n)$gen.points
#'
#' dec.dist<-rank.d2e.Te(dat)
#' dec.dist
#' dec.dist.rank<-dec.dist[[2]] #the rank of distances to the edges in decreasing order
#' dec.dist.rank
#' dist<-dec.dist[[1]] #distances to the edges of the std eq. triangle
#' dist
#'
#' A<-c(0,0); B<-c(1,0); C<-c(1/2,sqrt(3)/2);
#' Te<-rbind(A,B,C);
#'
#' Xlim<-range(Te[,1])
#' Ylim<-range(Te[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(A,pch=".",xlab="",ylab="",xlim=Xlim+xd*c(-.0,.01),ylim=Ylim+yd*c(-.01,.01))
#' polygon(Te)
#' points(dat,pch=".")
#' text(dat,labels = factor(dec.dist.rank) )
#'
#' inc.dist<-rank.d2e.Te(dat,dec = FALSE)
#' inc.dist
#' inc.dist.rank<-inc.dist[[2]] #the rank of distances to the edges in increasing order
#' inc.dist.rank
#' dist<-inc.dist[[1]] #distances to the edges of the std eq. triangle
#' dist
#'
#' plot(A,pch=".",xlab="",ylab="",xlim=Xlim,ylim=Ylim)
#' polygon(Te)
#' points(dat,pch=".",xlab="",ylab="", main="",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' text(dat,labels = factor(inc.dist.rank) )
#'
#' rank.d2e.Te(rbind(dat,dat))
#'
#' dat.fr<-data.frame(a=dat)
#' rank.d2e.Te(dat.fr)
#'
#' @export
rank.d2e.Te <- function(Dt,dec=TRUE)
{
  if (!is.numeric(as.matrix(Dt)))
  {stop('Dt must be numeric')}

  if (is.point(Dt))
  { Dt<-matrix(Dt,ncol=2)
  } else
  {Dt<-as.matrix(Dt)
  if (ncol(Dt)!=2 )
  {stop('Dt must be of dimension nx2')}
  }

  A<-c(0,0); B<-c(1,0); C<-c(0.5,sqrt(3)/2); Te<-rbind(A,B,C)

  n<-nrow(Dt)
  dist.edge<-rep(NA,n)
  for (i in 1:n)
  { pnt<-as.vector(Dt[i,])
  if (in.triangle(pnt,Te,boundary = TRUE)$in.tri)
  {
    x<-pnt[1]; y<-pnt[2];
    dist.edge[i]<-min(y,-0.5*y+0.866025404*x,-0.5*y+0.8660254040-0.866025404*x)
  }
  }

  ifelse(dec==TRUE,ranks<-rank(-dist.edge),ranks<-rank(dist.edge))
  ranks[is.na(dist.edge)]<-NA

  list(distances=dist.edge,
       dist.rank=ranks)
} #end of the function
#'
#' @rdname funsRankOrderTe
#'
#' @examples
#' #Examples for order.d2e.Te
#' n<-20
#' set.seed(1)
#' dat<-runifTe(n)$gen.points  #try also dat<-cbind(runif(n),runif(n))
#'
#' dec.dist<-order.d2e.Te(dat)
#' dec.dist
#' dec.dist.order<-dec.dist[[2]] #the order of distances to the edges in decreasing order
#' dec.dist.order
#' dist<-dec.dist[[1]] #distances to the edges of the std eq. triangle
#' dist
#' dist[dec.dist.order] #distances in decreasing order
#' dat[dec.dist.order,] #data in decreasing order
#'
#' A<-c(0,0); B<-c(1,0); C<-c(1/2,sqrt(3)/2);
#' Te<-rbind(A,B,C);
#'
#' Xlim<-range(Te[,1])
#' Ylim<-range(Te[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(A,pch=".",xlab="",ylab="",xlim=Xlim+xd*c(-.01,.01),ylim=Ylim+yd*c(-.01,.01))
#' polygon(Te)
#' points(dat,pch=".")
#' text(dat[dec.dist.order,],labels = factor(1:n) )
#'
#' inc.dist<-order.d2e.Te(dat,dec = FALSE)
#' inc.dist
#' inc.dist.order<-inc.dist[[2]] #the order of distances to the edges in increasing order
#' inc.dist.order
#' dist<-inc.dist[[1]] #distances to the edges of the std eq. triangle
#' dist
#' dist[inc.dist.order] #distances in increasing order
#'
#' plot(A,pch=".",xlab="",ylab="",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Te)
#' points(dat,pch=".")
#' text(dat[inc.dist.order,],labels = factor(1:n) )
#'
#' order.d2e.Te(rbind(dat,dat))
#'
#' dat.fr<-data.frame(a=dat)
#' order.d2e.Te(dat.fr)
#'
#' @export
order.d2e.Te <- function(Dt,dec=TRUE)
{
  if (!is.numeric(as.matrix(Dt)))
  {stop('Dt must be numeric')}

  if (is.point(Dt))
  { Dt<-matrix(Dt,ncol=2)
  } else
  {Dt<-as.matrix(Dt)
  if (ncol(Dt)!=2 )
  {stop('Dt must be of dimension nx2')}
  }

  A<-c(0,0); B<-c(1,0); C<-c(0.5,sqrt(3)/2); Te<-rbind(A,B,C)

  n<-nrow(Dt)
  dist.edge<-rep(NA,n)
  for (i in 1:n)
  {  pnt<-as.vector(Dt[i,])
  if (in.triangle(pnt,Te,boundary = TRUE)$in.tri)
  {
    x<-pnt[1]; y<-pnt[2];
    dist.edge[i]<-min(y,-0.5*y+0.866025404*x,-0.5*y+0.8660254040-0.866025404*x)
  }
  }
  ifelse(dec==TRUE,orders<-order(dist.edge,decreasing=dec),orders<-order(dist.edge))
  nint<-sum(!is.na(dist.edge))
  orders[-(1:nint)]<-NA

  list(distances=dist.edge,dist.order=orders)
} #end of the function
#'

#################################################################

#' @title Check whether a triangle is a standard equilateral triangle
#'
#' @description Checks whether the triangle, \code{tri}, is the standard equilateral triangle \eqn{T_e=T((0,0),(1,0),(1/2,\sqrt{3}/2))}
#' or not.
#'
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#'
#' @return \code{TRUE} if \code{tri} is a standard equilateral triangle, else \code{FALSE}.
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(0,0); B<-c(1,0); C<-c(1/2,sqrt(3)/2);
#' Te<-rbind(A,B,C)  #try adding +10^(-16) to each vertex
#' isStdEqTri(Te)
#'
#' isStdEqTri(Te)
#'
#' Te<-rbind(B,C,A)
#' isStdEqTri(Te)
#'
#' Tr<-rbind(A,B,-C)
#' isStdEqTri(Tr)
#'
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#' isStdEqTri(Tr)
#'
#' A<-c(0,0); B<-c(1,0); C<-c(.5,0.8660254);
#' Te<-rbind(A,B,C)
#' isStdEqTri(Te)
#'
#' isStdEqTri(rbind(A,A,B))
#'
#' dat.fr<-data.frame(a=Te)
#' isStdEqTri(dat.fr)
#'
#' @export isStdEqTri
isStdEqTri <- function(tri)
{
  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('the argument must be numeric and of dimension 3x2')}

  p1<-as.numeric(tri[1,]); p2<-as.numeric(tri[2,]); p3<-as.numeric(tri[3,]);
  A<-c(0,0); B<-c(1,0); C<-c(1/2,sqrt(3)/2)

  checkA<-isTRUE(all.equal(p1,A)) + isTRUE(all.equal(p2,A)) + isTRUE(all.equal(p3,A))
  checkB<-isTRUE(all.equal(p1,B)) + isTRUE(all.equal(p2,B)) + isTRUE(all.equal(p3,B))
  checkC<-isTRUE(all.equal(p1,C)) + isTRUE(all.equal(p2,C)) + isTRUE(all.equal(p3,C))

  eq.tri<-sum(checkA+checkB+checkC)==3 & min(checkA,checkB,checkC)==1
  eq.tri
} #end of the function
#'

#################################################################

#' @title The radii of points from one class with respect to points from the other class
#'
#' @description Returns the radii of the balls centered at \code{x} points where radius of an \code{x} point equals to the minimum distance
#' to \code{y} points (i.e., distance to the closest \code{y} point). That is, for each \code{x} point \eqn{radius= \min_{y \in Y}(d(x,y))}.
#' \code{x} and \code{y} points must be of the same dimension.
#'
#' @param x A set of \eqn{d}-dimensional points for which the radii are computed. Radius of an \code{x} point equals to the
#' distance to the closest \code{y} point.
#' @param y A set of \eqn{d}-dimensional points representing the reference points for the balls. That is, radius
#' of an \code{x} point is defined as the minimum distance to the \code{y} points.
#'
#' @return A \code{list} with three elements
#' \item{rad}{A \code{vector} whose entries are the radius values for the \code{x} points. Radius of an \code{x} point equals to
#' the distance to the closest \code{y} point}
#' \item{index.of.clYp}{A \code{vector} of indices of the closest \code{y} points to the \code{x} points. The \eqn{i}-th entry in this
#' \code{vector} is the index of the closest \code{y} point to \eqn{i}-th \code{x} point.}
#' \item{closest.Yp}{A \code{vector} of the closest \code{y} points to the \code{x} points. The \eqn{i}-th entry in this
#' \code{vector} or \eqn{i}-th row in the matrix is the closest \code{y} point to \eqn{i}-th \code{x} point.}
#'
#' @seealso \code{\link{radius}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' nx<-10
#' ny<-5
#' X<-cbind(runif(nx),runif(nx))
#' Y<-cbind(runif(ny),runif(ny))
#' Rad<-radii(X,Y)
#' Rad
#' rd<-Rad$rad
#'
#' Xlim<-range(X[,1]-rd,X[,1]+rd,Y[,1])
#' Ylim<-range(X[,2]-rd,X[,2]+rd,Y[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(rbind(Y),asp=1,pch=16,col=2,xlab="",ylab="",axes=TRUE,
#' xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' points(rbind(X))
#' interp::circles(X[,1],X[,2],Rad$rad,lty=1,lwd=1,col=4)
#'
#' nx<-5
#' ny<-1
#' X<-cbind(runif(nx),runif(nx))
#' Y<-matrix(c(runif(ny),runif(ny)),ncol=2)
#' Rad<-radii(X,Y)
#' Rad
#' radii(Y,X)
#'
#' rd<-Rad$rad
#'
#' Xlim<-range(X[,1]-rd,X[,1]+rd,Y[,1])
#' Ylim<-range(X[,2]-rd,X[,2]+rd,Y[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(rbind(Y),asp=1,pch=16,col=2,xlab="",ylab="",axes=TRUE,
#' xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' points(rbind(X))
#' interp::circles(X[,1],X[,2],Rad$rad,lty=1,lwd=1,col=4)
#'
#' radii(c(1,2),c(2,3))
#'
#' nx<-10
#' ny<-5
#' X<-as.matrix(X)
#' Y<-as.matrix(Y)
#' radii(X,Y)  #this works as X and Y are treated as 1D data sets
#' #but will give error if X<-runif(nx) and  Y<-runif(ny) is used
#' #as X and Y are treated as vectors (i.e., points)
#'
#' nx<-10
#' ny<-5
#' X<-cbind(runif(nx),runif(nx),runif(nx))
#' Y<-cbind(runif(ny),runif(ny),runif(ny))
#' radii(X,Y)
#'
#' dat.fr<-data.frame(a=X)
#' radii(dat.fr,Y)
#' radii(Y,dat.fr)
#'
#' @export radii
radii <- function(x,y)
{
  if (!is.numeric(as.matrix(x)) || !is.numeric(as.matrix(y)) )
  {stop('both arguments must be numeric')}

  ifelse(is.point(x,length(x)), x<-matrix(x,nrow=1),x<-as.matrix(x))
  ifelse(is.point(y,length(y)), y<-matrix(y,nrow=1),y<-as.matrix(y))

  dimx<-dimension(x)
  dimy<-dimension(y)

  if (dimx != dimy)
  {stop('the arguments are not of the same dimension')}

  nx<-nrow(x)
  ny<-nrow(y)

  rad<-rep(0,nx)
  cl.ind<-vector()
  for (i in 1:nx)
  { dis<-rep(0,ny)
  for (j in 1:ny)
  {
    dis[j]<-Dist(x[i,],y[j,])
  }
  rad[i]<-min(dis)
  cl.ind<-c(cl.ind,which(dis==rad[i]))
  }
  list(
    rad=rad,
   index.of.clYp=cl.ind,
    closest.Yp=y[cl.ind,]
  )
} #end of the function
#'

#################################################################

#' @title The radius of a point from one class with respect to points from the other class
#'
#' @description Returns the radius for the ball centered at point \code{pt} with radius=min distance to \code{Y} points.
#' That is, for the point \code{pt} \eqn{radius= \min_{y \in Y}d(pt,y)} (i.e., distance from \code{pt} to the closest \code{Y} point).
#' The point \code{pt} and \code{Y} points must be of same dimension.
#'
#' @param pt A \eqn{d}-dimensional point for which radius is computed. Radius of \code{pt} equals to the
#' distance to the closest \code{Y} point to \code{pt}.
#' @param Y A set of \eqn{d}-dimensional points representing the reference points for the balls. That is, radius
#' of the point \code{pt} is defined as the minimum distance to the \code{Y} points.
#'
#' @return A \code{list} with three elements
#' \item{rad}{Radius value for the point, \code{pt} defined as \eqn{\min_{y in Y} d(pt,y)}}
#' \item{index.of.clYpnt}{Index of the closest \code{Y} points to the point \code{pt}}
#' \item{closest.Ypnt}{The closest \code{Y} point to the point \code{pt}}
#'
#' @seealso \code{\link{radii}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#'
#' ny<-10
#' Y<-cbind(runif(ny),runif(ny))
#' radius(A,Y)
#' radius(B,Y)
#' radius(C,Y)
#'
#' radius(B,C)
#'
#' nx<-10
#' X<-cbind(runif(nx),runif(nx))
#' rad<-rep(0,nx)
#' for (i in 1:nx)
#' rad[i]<-radius(X[i,],Y)$rad
#'
#' Xlim<-range(X[,1]-rad,X[,1]+rad,Y[,1])
#' Ylim<-range(X[,2]-rad,X[,2]+rad,Y[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(rbind(Y),asp=1,pch=16,col=2,xlab="",ylab="",axes=TRUE,
#' xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' points(rbind(X))
#' interp::circles(X[,1],X[,2],rad,lty=1,lwd=1,col=4)
#'
#' dat.fr<-data.frame(a=Y)
#' radii(A,dat.fr)
#'
#' @export radius
radius <- function(pt,Y)
{
  if (!is.point(pt))
  {stop('pt must be a numeric 2D point')}

  if (!is.numeric(as.matrix(Y)) )
  {stop('Y must be numeric')}

  ifelse(is.point(Y,length(Y)), Y<-matrix(Y,nrow=1),Y<-as.matrix(Y))

  ny<-nrow(Y)

  dis<-rep(0,ny)
  for (j in 1:ny)
    dis[j]<-Dist(pt,Y[j,]);

  min.dis<-min(dis)
  cl.ind<-which(dis==min.dis)
  cl.Ypnt<-Y[cl.ind,]
  list(
    rad=min.dis,
   index.of.clYpnt=cl.ind,
    closest.Ypnt=cl.Ypnt
  )
} #end of the function
#'

#################################################################

#' @title The index of the \eqn{CM}-vertex region in a triangle that contains a given point
#'
#' @description Returns the index of the vertex whose region contains point \code{p} in
#' the triangle \code{tri}\eqn{=(y_1,y_2,y_3)} with vertex regions are constructed with center of mass \eqn{CM=(y_1+y_2+y_3)/3}
#' (see the plots in the example for illustrations).
#'
#' The vertices of triangle, \code{tri}, are labeled as \eqn{1,2,3}
#' according to the row number the vertex is recorded in \code{tri}. If the point, \code{p}, is not inside \code{tri}, then the
#' function yields \code{NA} as output. The corresponding vertex region is the polygon with the vertex, \eqn{CM}, and
#' midpoints of the edges adjacent to the vertex.
#'
#' See (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:mcap2012;textual}{pcds})
#'
#' @param p A 2D point for which \eqn{CM}-vertex region it resides in is to be determined in the triangle \code{tri}.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#'
#' @return A \code{list} with two elements
#' \item{rv}{Index of the \eqn{CM}-vertex region that contains point, \code{p} in the triangle \code{tri}.}
#' \item{tri}{The vertices of the triangle, where row number corresponds to the vertex index in \code{rv}.}
#'
#' @seealso \code{\link{rv.tri.cent}}, \code{\link{rv.triCC}}, \code{\link{rv.bastriCM}},
#' \code{\link{rv.bastriCC}}, \code{\link{rv.bastri.cent}}, and \code{\link{rvTeCM}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.6,2);
#' Tr<-rbind(A,B,C);
#' P<-c(1.4,1.2)
#' rv.triCM(P,Tr)
#'
#' P<-c(.8,.2)
#' rv.triCM(P,Tr)
#'
#' P<-c(1.5,1.6)
#' rv.triCM(P,Tr)
#'
#' n<-10  #try also n<-20
#' dat<-runif.tri(n,Tr)$g
#'
#' Rv<-vector()
#' for (i in 1:n)
#'   Rv<-c(Rv,rv.triCM(dat[i,],Tr)$rv)
#' Rv
#'
#' CM<-(A+B+C)/3
#' D1<-(B+C)/2; D2<-(A+C)/2; D3<-(A+B)/2;
#' Ds<-rbind(D1,D2,D3)
#'
#' Xlim<-range(Tr[,1],dat[,1])
#' Ylim<-range(Tr[,2],dat[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(Tr,xlab="",ylab="",axes=TRUE,pch=".",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tr)
#' points(dat,pch=".")
#' L<-Ds; R<-matrix(rep(CM,3),ncol=2,byrow=TRUE)
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#' text(dat,labels=factor(Rv))
#'
#' txt<-rbind(Tr,CM,D1,D2,D3)
#' xc<-txt[,1]+c(-.02,.02,.02,-.02,.02,-.01,-.01)
#' yc<-txt[,2]+c(-.02,-.04,.06,-.02,.02,.06,-.06)
#' txt.str<-c("rv=1","rv=2","rv=3","CM","D1","D2","D3")
#' text(xc,yc,txt.str)
#'
#' rv.triCM(A,Tr)
#' rv.triCM(CM,Tr)
#' rv.triCM(D1,Tr)
#' rv.triCM(D2,Tr)
#' rv.triCM(D3,Tr)
#'
#' #right triangle
#' A<-c(1,1); B<-c(1,2); C<-c(1.5,2);
#' T3<-rbind(A,B,C);
#' P<-c(1.1,1.5)
#' rv.triCM(P,T3)
#'
#' #isosceles triangle
#' A<-c(1,1); B<-c(2,1); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#' P<-c(1.5,1.1)
#' rv.triCM(P,Tr)
#'
#' dat.fr<-data.frame(a=Tr)
#' rv.triCM(P,dat.fr)
#'
#' @export rv.triCM
rv.triCM <- function(p,tri)
{
  if (!is.point(p))
  {stop('p must be a numeric 2D point')}

  tri<-as.matrix(tri)
  if (!is.numeric(tri) )
  {stop('tri must be numeric')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  if (in.triangle(p,tri,boundary=TRUE)$in.tri==F)
  {rv<-NA
  } else
  {
    y1<-tri[1,]; y2<-tri[2,]; y3<-tri[3,];
    a1<-y1[1]; a2<-y1[2]; b1<-y2[1]; b2<-y2[2]; c1<-y3[1]; c2<-y3[2];

    CM<-(y1+y2+y3)/3

    D1<-(y2+y3)/2; D2<-(y1+y3)/2; D3<-(y1+y2)/2;
    x<-p[1]; y<-p[2];

    if (in.triangle(p,rbind(y1,D3,CM),boundary=TRUE)$in.tri | in.triangle(p,rbind(y1,CM,D2),boundary=TRUE)$in.tri)
    {rv<-1}
    else
    {
      if (in.triangle(p,rbind(D3,y2,CM),boundary=TRUE)$in.tri | in.triangle(p,rbind(y2,D1,CM),boundary=TRUE)$in.tri)
      {rv<-2}
      else
      {rv<-3}
    }
  }
  row.names(tri)<-c("vertex 1","vertex 2","vertex 3")  #vertex labeling

  list(rv=rv, #relative vertex
       tri=tri #vertex labeling
  )
} #end of the function
#'

#################################################################

#' @title The labels of the vertices of a triangle in the (unscaled) basic triangle form
#'
#' @description Labels the vertices of triangle, \code{tri}, as \eqn{ABC} so that  \eqn{AB} is the longest edge,  \eqn{BC} is the second longest
#' and  \eqn{AC} is the shortest edge (the order is as in the basic triangle). The new triangle \eqn{T(A,B,C)} is unscaled,
#' i.e., the longest edge \eqn{AB} may not be of unit length.
#'
#' The basic triangle is \eqn{T_b=T((0,0),(1,0),(c_1,c_2))} where \eqn{c_1} is in \eqn{[0,1/2]},
#' \eqn{c_2>0} and \eqn{(1-c_1)^2+c_2^2 \le 1}.
#'
#' Any given triangle can be mapped to the basic triangle by a combination of rigid body motions
#' (i.e., translation, rotation and reflection) and scaling, preserving uniformity of the points in the
#' original triangle. Hence basic triangle is useful for simulation
#' studies under the uniformity hypothesis.
#'
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#'
#' @return A \code{list} with two elements
#' \item{tri}{The vertices of the triangle stacked row-wise and labeled row-wise as \eqn{A}, \eqn{B}, \eqn{C}.}
#' \item{desc}{Description of the edges based on the vertices, i.e.,
#'  \code{"Edges (in decreasing length are) AB, BC, and AC"}.}
#' \item{orig.order}{Row order of the input triangle, \code{tri},
#' when converted to the scaled version of the basic triangle}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' c1<-.4; c2<-.6
#' A<-c(0,0); B<-c(1,0); C<-c(c1,c2);
#'
#' as.bastri(rbind(A,B,C))
#'
#' as.bastri(rbind(B,C,A))
#'
#' as.bastri(rbind(B,A,C))
#' as.bastri(rbind(A,C,B))
#'
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' as.bastri(rbind(A,B,C))
#' as.bastri(rbind(A,C,B))
#' as.bastri(rbind(B,A,C))
#'
#' A<-runif(2); B<-runif(2); C<-runif(2)
#' as.bastri(rbind(A,B,C))
#'
#' dat.fr<-data.frame(a=rbind(A,B,C))
#' as.bastri(dat.fr)
#'
#' @export
as.bastri <- function(tri)
{
  tri<-as.matrix(tri)#
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('the argument must be numeric and of dimension 3x2')}

  v1<-tri[1,];v2<-tri[2,];v3<-tri[3,];
  de1<-Dist(v1,v2); de2<-Dist(v2,v3); de3<-Dist(v1,v3);
  ord<-order(c(de1,de2,de3),decreasing = TRUE)
  vord<-cbind(c(1,2,1),c(2,3,3))
  vord<-vord[ord,]
  C<-tri[setdiff(1:3,vord[1,]),]
  A<-tri[setdiff(1:3,vord[2,]),]
  B<-tri[setdiff(1:3,vord[3,]),]

  orig.ord<-c(setdiff(1:3,vord[2,]),setdiff(1:3,vord[3,]),setdiff(1:3,vord[1,]))
  Tr<-rbind(A,B,C)
  row.names(Tr)<-c("A","B","C")  #vertex labeling
  edge.desc<-"Edges (in decreasing length are) AB, BC, and AC"

  list(tri=Tr, #vertex labeling
       desc=edge.desc,
       orig.order=orig.ord #order of vertices in argument tri
  )
} #end of the function
#'

#################################################################

# funsCartBary
#'
#' @title Converts of a point in Cartesian coordinates to Barycentric coordinates and vice versa
#'
#' @description
#' Two functions: \code{cart2bary} and \code{bary2cart}.
#'
#' \code{cart2bary} converts Cartesian coordinates of a given point \code{P}\eqn{=(x,y)} to barycentric coordinates
#' (in the normalized form) with respect to the triangle \code{tri}\eqn{=(v_1,v_2,v_3)}
#' with vertex labeling done row-wise in \code{tri} (i.e., row \eqn{i} corresponds to vertex \eqn{v_i} for \eqn{i=1,2,3}).
#'
#' \code{bary2cart} converts barycentric coordinates of the point \code{P}\eqn{=(t_1,t_2,t_3)} (not necessarily normalized) to
#' Cartesian coordinates according to the coordinates of the triangle, \code{tri}.
#' For information on barycentric coordinates, see (\insertCite{weisstein-barycentric;textual}{pcds}).
#'
#' @param P A 2D point for \code{cart2bary}, and a \code{vector} of three \code{numeric} entries for \code{bary2cart}.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#'
#' @return \code{cart2bary} returns the barycentric coordinates of a given point \code{P}\eqn{=(x,y)} and
#' \code{bary2cart} returns the Cartesian coordinates of the point \code{P}\eqn{=(t_1,t_2,t_3)} (not necessarily normalized)
#'
#' @name funsCartBary
NULL
#'
#' @rdname funsCartBary
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' #Examples for cart2bary
#' c1<-.4; c2<-.6
#' A<-c(0,0); B<-c(1,0); C<-c(c1,c2);
#' Tr<-rbind(A,B,C)
#'
#' cart2bary(A,Tr)
#' cart2bary(c(.3,.2),Tr)
#' cart2bary(c(.4,.2),Tr)
#' cart2bary(c(.5,.2),Tr)
#' cart2bary(c(.6,.2),Tr)
#'
#' P<-c(.8,.2)
#' round(cart2bary(P,Tr),2)
#'
#' P<-c(.5,.61)
#' cart2bary(P,Tr)
#'
#' CM<-(A+B+C)/3
#' cart2bary(CM,Tr)
#'
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#' cart2bary(c(1.4,1.2),Tr)
#'
#' cart2bary(c(.8,.2),Tr)
#'
#' cart2bary(c(1.5,1.61),Tr)
#'
#' dat.fr<-data.frame(a=Tr)
#' cart2bary(c(.8,.2),dat.fr)
#'
#' @export cart2bary
cart2bary <- function(P,tri)
{
  if (!is.point(P))
  {stop('P must be a numeric 2D point')}

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  v1<-tri[1,];v2<-tri[2,];v3<-tri[3,];
  T<-cbind(v1,v2)-cbind(v3,v3)
  lam<-solve(T) %*% (matrix(P-v3))
  c(lam,1-sum(lam))
} #end of the function
#'
#' @rdname funsCartBary
#'
#' @examples
#' #Examples for bary2cart
#' c1<-.4; c2<-.6
#' A<-c(0,0); B<-c(1,0); C<-c(c1,c2);
#' Tr<-rbind(A,B,C)
#'
#' bary2cart(c(.3,.2,.5),Tr)
#' bary2cart(c(.4,.2,.4),Tr)
#' bary2cart(c(.5,.2,.3),Tr)
#' bary2cart(c(6,2,4),Tr)
#'
#' P<-c(.8,.2,.3)
#' bary2cart(P,Tr)
#'
#' P<-c(-.5,.4,.2)
#' bary2cart(P,Tr)
#'
#' CM<-(A+B+C)/3; CM
#' bary2cart(c(1,1,1),Tr)
#'
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#' bary2cart(c(1.4,1.2,1),Tr)
#'
#' bary2cart(c(.8,.2,.6),Tr)
#'
#' bary2cart(c(1,2,3),Tr)
#'
#' dat.fr<-data.frame(a=Tr)
#' bary2cart(c(.8,.2,.3),dat.fr)
#'
#' @references
#' \insertAllCited{}
#'
#' @export bary2cart
bary2cart <- function(P,tri)
{
  if (!is.point(P,3))
  {stop('P must be a numeric 3D point')}

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  P<-P/sum(P)  #normalized barycentric coordinates
  R<-t(tri)
  cc<-R %*% matrix(P)  #Cartesian coordinates
  as.vector(cc)
} #end of the function
#'

#################################################################

#' @title The index of the \eqn{CM}-vertex region in a basic triangle that contains a point
#'
#' @description Returns the index of the vertex whose region contains point \code{p} in
#' the basic triangle \eqn{T_b=T((0,0),(1,0),(c_1,c_2))} where \eqn{c_1} is in \eqn{[0,1/2]}, \eqn{c_2>0}
#' and \eqn{(1-c_1)^2+c_2^2 \le 1} and vertex regions are based on the center of mass CM=((1+c1)/3,c2/3) of \eqn{T_b}.
#' (see the plots in the example for illustrations).
#'
#' The vertices of the basic triangle \eqn{T_b} are labeled as
#' \eqn{1=(0,0)}, \eqn{2=(1,0)},and \eqn{3=(c_1,c_2)} also according to the row number the vertex is recorded in \eqn{T_b}.
#' If the point, \code{p}, is not inside \eqn{T_b}, then the function yields \code{NA} as output.
#' The corresponding vertex region is the polygon with the vertex, \eqn{CM}, and
#' midpoints of the edges adjacent to the vertex.
#'
#' Any given triangle can be mapped to the basic triangle
#' by a combination of rigid body motions (i.e., translation, rotation and reflection) and scaling,
#' preserving uniformity of the points in the original triangle. Hence basic triangle is useful for simulation
#' studies under the uniformity hypothesis.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:mcap2012,ceyhan:arc-density-PE;textual}{pcds})
#'
#' @param p A 2D point for which \eqn{CM}-vertex region it resides in is to be determined in the
#' basic triangle \eqn{T_b}.
#' @param c1,c2 Positive real numbers which constitute the upper vertex of the basic triangle (i.e., the vertex
#' adjacent to the shorter edges of \eqn{T_b}); \eqn{c_1} must be in \eqn{[0,1/2]}, \eqn{c_2>0} and
#' \eqn{(1-c_1)^2+c_2^2 \le 1}.
#'
#' @return A \code{list} with two elements
#' \item{rv}{Index of the \eqn{CM}-vertex region that contains point, \code{p} in the basic triangle \eqn{T_b}}
#' \item{tri}{The vertices of the triangle, where row number corresponds to the vertex index in \code{rv}
#' with row \eqn{1=(0,0)}, row \eqn{2=(1,0)}, and row \eqn{3=(c_1,c_2)}.}
#'
#' @seealso \code{\link{rv.triCM}}, \code{\link{rv.tri.cent}}, \code{\link{rv.triCC}},
#' \code{\link{rv.bastriCC}}, \code{\link{rv.bastri.cent}}, and \code{\link{rvTeCM}}
#'
#' @references
#' \insertAllCited{}
#'
#' #' @author Elvan Ceyhan
#'
#' @examples
#' c1<-.4; c2<-.6
#' P<-c(.4,.2)
#' rv.bastriCM(P,c1,c2)
#'
#' c1<-.5; c2<-.8
#' P<-c(.4,.2)
#' rv.bastriCM(P,c1,c2)
#'
#' P<-c(.8,.2)
#' rv.bastriCM(P,c1,c2)
#'
#' P<-c(1,.2)
#' rv.bastriCM(P,c1,c2)
#'
#' c1<-.4; c2<-.6
#' A<-c(0,0);B<-c(1,0);C<-c(c1,c2);
#' Tb<-rbind(A,B,C)
#' CM<-(A+B+C)/3
#' D1<-(B+C)/2; D2<-(A+C)/2; D3<-(A+B)/2;
#' Ds<-rbind(D1,D2,D3)
#'
#' rv.bastriCM(A,c1,c2)
#' rv.bastriCM(CM,c1,c2)
#' rv.bastriCM(D1,c1,c2)
#' rv.bastriCM(D2,c1,c2)
#' rv.bastriCM(D3,c1,c2)
#'
#' n<-10  #try also n<-20
#' dat<-runif.bastri(n,c1,c2)$g
#'
#' Rv<-vector()
#' for (i in 1:n)
#'   Rv<-c(Rv,rv.bastriCM(dat[i,],c1,c2)$rv)
#' Rv
#'
#' Xlim<-range(Tb[,1],dat[,1])
#' Ylim<-range(Tb[,2],dat[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(Tb,xlab="",ylab="",axes="T",pch=".",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' points(dat,pch=".")
#' polygon(Tb)
#' L<-Ds; R<-matrix(rep(CM,3),ncol=2,byrow=TRUE)
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#' text(dat,labels=factor(Rv))
#'
#' txt<-rbind(Tb,CM,Ds)
#' xc<-txt[,1]+c(-.03,.03,.02,-.01,.06,-.05,.0)
#' yc<-txt[,2]+c(.02,.02,.02,.04,.02,.02,-.03)
#' txt.str<-c("A","B","C","CM","D1","D2","D3")
#' text(xc,yc,txt.str)
#'
#' plot(Tb,xlab="",ylab="",axes="T",pch=".",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tb)
#' L<-Ds; R<-matrix(rep(CM,3),ncol=2,byrow=TRUE)
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#'
#' RV1<-(A+D3+CM+D2)/4
#' RV2<-(B+D3+CM+D1)/4
#' RV3<-(C+D2+CM+D1)/4
#'
#' txt<-rbind(RV1,RV2,RV3)
#' xc<-txt[,1]
#' yc<-txt[,2]
#' txt.str<-c("rv=1","rv=2","rv=3")
#' text(xc,yc,txt.str)
#'
#' txt<-rbind(Tb,CM,Ds)
#' xc<-txt[,1]+c(-.03,.03,.02,-.01,.04,-.03,.0)
#' yc<-txt[,2]+c(.02,.02,.02,.04,.02,.02,-.03)
#' txt.str<-c("A","B","C","CM","D1","D2","D3")
#' text(xc,yc,txt.str)
#'
#' @export rv.bastriCM
rv.bastriCM <- function(p,c1,c2)
{
  if (!is.point(c1,1) || !is.point(c2,1))
  {stop('c1 and c2 must be scalars')}

  if (c1<0 || c1>1/2 || c2<=0 || (1-c1)^2+c2^2 >1)
  {stop('c1 must be in [0,1/2], c2>0 and (1-c1)^2+c2^2 <=1')}

  p1<-c(0,0); p2<-c(1,0); p3<-c(c1,c2);
  Tb<-rbind(p1,p2,p3)

  if (in.triangle(p,Tb,boundary = TRUE)$in.tri==F)
  {rv<-NA
  } else
  {
    x<-p[1]; y<-p[2];

    if (c1<1/2)
    {
      if ( y<=c2*x/(1+c1) && y>=c2 * (-1 + 2 * x) / (2 * c1 - 1))
      {rv<-2}
      else
      {
        if (y>c2*x/(1+c1) && y>=c2 * (-1 + x) / (c1 - 2))
        {rv<-3}
        else
        {rv<-1}
      }
    }
    else
    {
      if (c1 == 1/2)
      {
        if ( y<=c2*x/(1+c1) && x>=c1)
        {rv<-2}
        else
        {
          if (y>c2*x/(1+c1) && y>=c2 * (-1 + x) / (c1 - 2))
          {rv<-3}
          else
          {rv<-1}
        }
      }
      else
      {if ( y<=c2*x/(1+c1) && y<=c2 * (-1 + 2 * x) / (2 * c1 - 1))
      {rv<-2}
        else
        {
          if (y>c2*x/(1+c1) && y>=c2 * (-1 + x) / (c1 - 2))
          {rv<-3}
          else
          {rv<-1}
        }
      }
    }
  }
  row.names(Tb)<-c("vertex 1","vertex 2","vertex 3")  #vertex labeling

  list(rv=rv, #relative vertex
       tri=Tb #vertex labeling
  )
} #end of the function
#'

#################################################################

#' @title The area of a polygon in \eqn{R^2}
#'
#' @description Returns the area of the polygon, \code{h}, in the real plane \eqn{R^{2}}; the vertices of the polygon \code{h}
#' must be provided in clockwise or counter-clockwise order, otherwise the function does not yield
#' the area of the polygon. Also, the polygon could be convex or non-convex.
#' See (\insertCite{weisstein-area-poly;textual}{pcds}).
#'
#' @param h A \code{vector} of \eqn{n} 2D points, stacked row-wise, each row representing a vertex of the polygon,
#' where \eqn{n} is the number of vertices of the polygon.
#'
#' @return area of the polygon \code{h}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(0,0); B<-c(1,0); C<-c(0.5,.8);
#' Tr<-rbind(A,B,C);
#' area.polygon(Tr)
#'
#' A<-c(0,0); B<-c(1,0); C<-c(.7,.6); D<-c(0.3,.8);
#' h1<-rbind(A,B,C,D);  #try also h1<-rbind(A,B,D,C) or h1<-rbind(A,C,B,D) or h1<-rbind(A,D,C,B);
#' area.polygon(h1)
#'
#' Xlim<-range(h1[,1])
#' Ylim<-range(h1[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(h1,xlab="",ylab="",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(h1)
#' xc<-rbind(A,B,C,D)[,1]+c(-.03,.03,.02,-.01)
#' yc<-rbind(A,B,C,D)[,2]+c(.02,.02,.02,.04)
#' txt.str<-c("A","B","C","D")
#' text(xc,yc,txt.str)
#'
#' dat.fr<-data.frame(a=Tr)
#' area.polygon(dat.fr)
#'
#' #when the triangle is degenerate, it gives zero area
#' B<-A+2*(C-A);
#' T2<-rbind(A,B,C)
#' area.polygon(T2)
#'
#' @export area.polygon
area.polygon <- function(h)
{
  h<-as.matrix(h)

  if (!is.numeric(h) || ncol(h)!=2)
  {stop('the argument must be numeric and of dimension nx2')}

  n<-nrow(h);
  area<-0;
  for (i in 1:(n-1) )
  { a1<-h[i,1]; b1<-h[i,2]; a2<-h[i+1,1]; b2<-h[i+1,2];
  area<-area + 1/2*(a1*b2-a2*b1);
  }
  a1<-h[1,1]; b1<-h[1,2];

  abs(as.numeric(area + 1/2*(a2*b1-a1*b2)));
} #end of the function
#'

#################################################################

#' @title The index of the \eqn{CM}-edge region in a triangle that contains the point
#'
#' @description Returns the index of the edge whose region contains point, \code{pt}, in
#' the triangle \code{tri}\eqn{=T(A,B,C)} with edge regions based on center of mass \eqn{CM=(A+B+C)/3}.
#'
#' Edges are labeled as 3 for edge \eqn{AB}, 1 for edge \eqn{BC}, and 2 for edge \eqn{AC}.
#' If the point, \code{pt}, is not inside \code{tri}, then the function yields \code{NA} as output.
#' Edge region 1 is the triangle \eqn{T(B,C,CM)}, edge region 2 is \eqn{T(A,C,CM)}, and
#' edge region 3 is \eqn{T(A,B,CM)}.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:mcap2012,ceyhan:arc-density-CS;textual}{pcds}).
#'
#' @param pt A 2D point for which \eqn{CM}-edge region it resides in is to be determined in the triangle
#' \code{tri}.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#'
#' @return A \code{list} with three elements
#' \item{re}{Index of the \eqn{CM}-edge region that contains point, \code{pt} in the triangle \code{tri}.}
#' \item{tri}{The vertices of the triangle, where row labels are \eqn{A}, \eqn{B}, and \eqn{C}
#' with edges are labeled as 3 for edge \eqn{AB}, 1 for edge \eqn{BC}, and 2 for edge \eqn{AC}.}
#' \item{desc}{Description of the edge labels}
#'
#' @seealso \code{\link{re.tri.cent}}, \code{\link{re.bastriCM}}, \code{\link{re.bastri.cent}},
#' \code{\link{reTeCM}}, and \code{\link{redge.triCM}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#' P<-c(1.4,1.2)
#' re.triCM(P,Tr)
#'
#' P<-c(.8,.2)
#' re.triCM(P,Tr)
#'
#' P<-c(1.5,1.61)
#' re.triCM(P,Tr)
#'
#' CM<-(A+B+C)/3
#'
#' re.triCM(A,Tr)
#' re.triCM(B,Tr)
#' re.triCM(C,Tr)
#' re.triCM(CM,Tr)
#'
#' n<-10  #try also n<-20
#' dat<-runif.tri(n,Tr)$g
#'
#' re<-vector()
#' for (i in 1:n)
#'   re<-c(re,re.triCM(dat[i,],Tr)$re)
#' re
#'
#' Xlim<-range(Tr[,1],dat[,1])
#' Ylim<-range(Tr[,2],dat[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(Tr,xlab="",ylab="",axes=TRUE,pch=".",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' points(dat,pch=".")
#' polygon(Tr)
#' L<-Tr; R<-matrix(rep(CM,3),ncol=2,byrow=TRUE)
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#' text(dat,labels=factor(re))
#'
#' txt<-rbind(Tr,CM)
#' xc<-txt[,1]
#' yc<-txt[,2]
#' txt.str<-c("A","B","C","CM")
#' text(xc,yc,txt.str)
#'
#' p1<-(A+B+CM)/3
#' p2<-(B+C+CM)/3
#' p3<-(A+C+CM)/3
#'
#' plot(Tr,xlab="",ylab="",axes=TRUE,pch=".",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tr)
#' L<-Tr; R<-matrix(rep(CM,3),ncol=2,byrow=TRUE)
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#'
#' txt<-rbind(Tr,CM,p1,p2,p3)
#' xc<-txt[,1]+c(-.02,.02,.02,.02)
#' yc<-txt[,2]+c(.02,.02,.04,.05)
#' txt.str<-c("A","B","C","CM","re=3","re=1","re=2")
#' text(xc,yc,txt.str)
#'
#' A<-c(0,0); B<-c(1,0); C<-c(0.5,.8);
#' Tr<-rbind(A,B,C);
#' P<-c(.4,.2)
#' re.triCM(P,Tr)
#'
#' dat.fr<-data.frame(a=Tr)
#' re.triCM(P,dat.fr)
#'
#' @export re.triCM
re.triCM <- function(pt,tri)
{
  if (!is.point(pt))
  {stop('pt must be a numeric 2D point')}

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  if (in.triangle(pt,tri,boundary=TRUE)$in.tri==F)
  {reled<-NA
  } else
  {
    y1<-tri[1,];y2<-tri[2,]; y3<-tri[3,];
    CM<-(1/3)*(y1+y2+y3);

    if (in.triangle(pt,rbind(y1,y2,CM),boundary = TRUE)$in.tri)
    {reled<-3
    } else if (in.triangle(pt,rbind(y2,y3,CM),boundary = TRUE)$in.tri)
    {reled<-1
    } else
    {reled<-2}
  }
  row.names(tri)<-c("A","B","C")  #vertex labeling
  edge.desc<-"Edge labels are AB=3, BC=1, and AC=2"

  list(re=reled, #relative edge
       tri=tri, #vertex labeling
       desc=edge.desc
  )
} #end of the function
#'

#################################################################

#' @title The index of the edge region in a triangle that contains the point
#'
#' @description Returns the index of the edge whose region contains point, \code{pt}, in
#' the triangle \code{tri}\eqn{=T(A,B,C)} with edge regions based on center \eqn{M=(m_1,m_2)} in Cartesian coordinates or
#' \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the interior of the triangle \code{tri}.
#'
#' Edges are labeled as 3 for edge \eqn{AB}, 1 for edge \eqn{BC}, and 2 for edge \eqn{AC}.
#' If the point, \code{pt}, is not inside \code{tri}, then the function yields \code{NA} as output.
#' Edge region 1 is the triangle \eqn{T(B,C,M)}, edge region 2 is \eqn{T(A,C,M)}, and edge region 3 is \eqn{T(A,B,M)}.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:mcap2012,ceyhan:arc-density-CS;textual}{pcds}).
#'
#' @param pt A 2D point for which \code{M}-edge region it resides in is to be determined in the triangle
#' \code{tri}.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param M A 2D point in Cartesian coordinates or a 3D point in barycentric coordinates
#' which serves as a center in the interior of the triangle \code{tri}.
#'
#' @return A \code{list} with three elements
#' \item{re}{Index of the \code{M}-edge region that contains point, \code{pt} in the triangle \code{tri}.}
#' \item{tri}{The vertices of the triangle, where row labels are \eqn{A}, \eqn{B}, and \eqn{C}
#' with edges are labeled as 3 for edge \eqn{AB}, 1 for edge \eqn{BC}, and 2 for edge \eqn{AC}.}
#' \item{desc}{Description of the edge labels}
#'
#' @seealso \code{\link{re.triCM}}, \code{\link{re.bastriCM}},
#' \code{\link{re.bastri.cent}}, \code{\link{reTeCM}}, and \code{\link{redge.triCM}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#'
#' P<-c(1.4,1.2)
#' M<-as.numeric(runif.tri(1,Tr)$g)  #try also M<-c(1.6,1.2)
#'
#' re.tri.cent(P,Tr,M)
#'
#' P<-c(.8,.2)
#' re.tri.cent(P,Tr,M)
#'
#' P<-c(1.5,1.61)
#' re.tri.cent(P,Tr,M)
#'
#' re.tri.cent(A,Tr,M)
#' re.tri.cent(B,Tr,M)
#' re.tri.cent(C,Tr,M)
#' re.tri.cent(M,Tr,M)
#'
#' n<-10  #try also n<-20
#' dat<-runif.tri(n,Tr)$g
#'
#' re<-vector()
#' for (i in 1:n)
#'   re<-c(re,re.tri.cent(dat[i,],Tr,M)$re)
#' re
#'
#' Xlim<-range(Tr[,1],dat[,1])
#' Ylim<-range(Tr[,2],dat[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' if (dimension(M)==3) {M<-bary2cart(M,Tr)}
#'
#' plot(Tr,xlab="",ylab="",axes=TRUE,pch=".",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tr)
#' points(dat,pch=".")
#' L<-Tr; R<-rbind(M,M,M)
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#' text(dat,labels=factor(re))
#'
#' txt<-rbind(Tr,M)
#' xc<-txt[,1]
#' yc<-txt[,2]
#' txt.str<-c("A","B","C","M")
#' text(xc,yc,txt.str)
#'
#' p1<-(A+B+M)/3
#' p2<-(B+C+M)/3
#' p3<-(A+C+M)/3
#'
#' plot(Tr,xlab="",ylab="",axes=TRUE,pch=".",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tr)
#' L<-Tr; R<-rbind(M,M,M)
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#'
#' txt<-rbind(Tr,M,p1,p2,p3)
#' xc<-txt[,1]+c(-.02,.02,.02,.02)
#' yc<-txt[,2]+c(.02,.02,.04,.05)
#' txt.str<-c("A","B","C","M","re=3","re=1","re=2")
#' text(xc,yc,txt.str)
#'
#' dat.fr<-data.frame(a=Tr)
#' re.tri.cent(P,dat.fr,M)
#'
#' @export re.tri.cent
re.tri.cent <- function(pt,tri,M)
{
  if (!is.point(pt))
  {stop('pt must be a numeric 2D point')}

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  if (!is.point(M) && !is.point(M,3) )
  {stop('M must be a numeric 2D point for Cartesian coordinates or 3D point for barycentric coordinates')}

  if (dimension(M)==3)
  {
    M<-bary2cart(M,tri)
  }
  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  if (in.triangle(M,tri,boundary=FALSE)$in.tri==F)
  {stop('center is not in the interior of the triangle')}

  if (in.triangle(pt,tri,boundary=TRUE)$in.tri==F)
  {reled<-NA
  } else
  {
    y1<-tri[1,];y2<-tri[2,]; y3<-tri[3,];

    if (in.triangle(pt,rbind(y1,y2,M),boundary = TRUE)$in.tri)
      reled<-3
    else
    {
      if (in.triangle(pt,rbind(y2,y3,M),boundary = TRUE)$in.tri)
        reled<-1
      else reled<-2
    }
  }
  row.names(tri)<-c("A","B","C")  #vertex labeling
  edge.desc<-"Edge labels are AB=3, BC=1, and AC=2"

  list(re=reled, #relative edge
       tri=tri, #vertex labeling
       desc=edge.desc
  )
} #end of the function
#'

#################################################################

#' @title The index of the \eqn{CM}-edge region in a basic triangle that contains a point
#'
#' @description Returns the index of the edge whose region contains point, \code{pt}, in the
#' basic triangle \eqn{T_b=T(A=(0,0),B=(1,0),C=(c_1,c_2)} where \eqn{c_1} is in \eqn{[0,1/2]}, \eqn{c_2>0} and \eqn{(1-c_1)^2+c_2^2 \le 1} with
#' edge regions based on center of mass \eqn{CM=(A+B+C)/3}.
#'
#' Edges are labeled as 3 for edge \eqn{AB}, 1 for edge \eqn{BC}, and 2 for edge \eqn{AC}.
#' If the point, \code{pt}, is not inside \code{tri}, then the function yields \code{NA} as output.
#' Edge region 1 is the triangle \eqn{T(B,C,CM)}, edge region 2 is \eqn{T(A,C,CM)}, and edge region 3 is \eqn{T(A,B,CM)}.
#'
#' Any given triangle can be mapped to the basic triangle
#' by a combination of rigid body motions (i.e., translation, rotation and reflection) and scaling,
#' preserving uniformity of the points in the original triangle. Hence basic triangle is useful for simulation
#' studies under the uniformity hypothesis.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:mcap2012,ceyhan:arc-density-CS;textual}{pcds}).
#'
#' @param pt A 2D point for which \eqn{CM}-edge region it resides in is to be determined in the
#' basic triangle \eqn{T_b}.
#' @param c1,c2 Positive real numbers which constitute the upper vertex of the basic triangle (i.e., the vertex
#' adjacent to the shorter edges of \eqn{T_b}); \eqn{c_1} must be in \eqn{[0,1/2]}, \eqn{c_2>0} and
#' \eqn{(1-c_1)^2+c_2^2 \le 1}.
#'
#' @return A \code{list} with three elements
#'
#' \item{re}{Index of the \eqn{CM}-edge region that contains point, \code{pt} in the basic triangle \eqn{T_b}}
#' \item{tri}{The vertices of the triangle, where row labels are \eqn{A=(0,0)}, \eqn{B=(1,0)}, and \eqn{C=(c_1,c_2)}
#' with edges are labeled as 3 for edge \eqn{AB}, 1 for edge \eqn{BC}, and 2 for edge \eqn{AC}.}
#' \item{desc}{Description of the edge labels}
#'
#' @seealso \code{\link{re.triCM}}, \code{\link{re.tri.cent}},
#' \code{\link{re.bastri.cent}}, \code{\link{reTeCM}}, and \code{\link{redge.triCM}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' c1<-.4; c2<-.6
#' P<-c(.4,.2)
#' re.bastriCM(P,c1,c2)
#'
#' c1<-.5; c2<-.8
#' P<-c(.4,.2)
#' re.bastriCM(P,c1,c2)
#'
#' P<-c(.8,.2)
#' re.bastriCM(P,c1,c2)
#'
#' P<-c(1.8,.2)
#' re.bastriCM(P,c1,c2)
#'
#' c1<-.4; c2<-.6
#' A<-c(0,0);B<-c(1,0);C<-c(c1,c2);
#' Tb<-rbind(A,B,C)
#' CM<-(A+B+C)/3
#'
#' re.bastriCM(A,c1,c2)
#' re.bastriCM(B,c1,c2)
#' re.bastriCM(C,c1,c2)
#' re.bastriCM(CM,c1,c2)
#'
#' n<-10  #try also n<-20
#' dat<-runif.bastri(n,c1,c2)$g
#'
#' re<-vector()
#' for (i in 1:n)
#'   re<-c(re,re.bastriCM(dat[i,],c1,c2)$re)
#' re
#'
#' Xlim<-range(Tb[,1],dat[,1])
#' Ylim<-range(Tb[,2],dat[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(Tb,xlab="",ylab="",axes=TRUE,pch=".",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' points(dat,pch=".")
#' polygon(Tb)
#' L<-Tb; R<-matrix(rep(CM,3),ncol=2,byrow=TRUE)
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#' text(dat,labels=factor(re))
#'
#' txt<-rbind(Tb,CM)
#' xc<-txt[,1]+c(-.03,.03,.02,0)
#' yc<-txt[,2]+c(.02,.02,.02,-.04)
#' txt.str<-c("A","B","C","CM")
#' text(xc,yc,txt.str)
#'
#' @export re.bastriCM
re.bastriCM <- function(pt,c1,c2)
{
  if (!is.point(pt))
  {stop('pt must be a numeric 2D point')}

  if (!is.point(c1,1) || !is.point(c2,1))
  {stop('c1 and c2 must be scalars')}

  if (c1<0 || c1>1/2 || c2<=0 || (1-c1)^2+c2^2 >1)
  {stop('c1 must be in [0,1/2], c2>0 and (1-c1)^2+c2^2 <=1')}

  y1<-c(0,0); y2<-c(1,0); y3<-c(c1,c2);
  Tb<-rbind(y1,y2,y3)

  if (in.triangle(pt,Tb,boundary = TRUE)$in.tri==F)
  {reled<-NA
  } else
  {
    CM<-(1/3)*(y1+y2+y3);

    if (in.triangle(pt,rbind(y1,y2,CM),boundary = TRUE)$in.tri)
      reled<-3
    else
    {
      if (in.triangle(pt,rbind(y2,y3,CM),boundary = TRUE)$in.tri)
        reled<-1
      else reled<-2
    }
  }
  row.names(Tb)<-c("A","B","C")  #vertex labeling
  edge.desc<-"Edge labels are AB=3, BC=1, and AC=2"

  list(re=reled, #relative edge
       tri=Tb, #vertex labeling
       desc=edge.desc #description of the edge labels
  )
} #end of the function
#'

#################################################################

#' @title The index of the edge region in a basic triangle that contains a point
#'
#' @description Returns the index of the edge whose region contains point, \code{pt}, in
#' the basic triangle \eqn{T_b=T(A=(0,0),B=(1,0),C=(c_1,c_2))} and edge regions based on center
#' \eqn{M=(m_1,m_2)} in Cartesian coordinates or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates
#' in the interior of the basic triangle \eqn{T_b}.
#'
#' Edges are labeled as 3 for edge \eqn{AB}, 1 for edge \eqn{BC}, and 2 for edge \eqn{AC}.
#' If the point, \code{pt}, is not inside \code{tri}, then the function yields \code{NA} as output.
#' Edge region 1 is the triangle \eqn{T(B,C,M)}, edge region 2 is \eqn{T(A,C,M)}, and edge region 3 is \eqn{T(A,B,M)}.
#' In the basic triangle \eqn{T_b} \eqn{c_1} is in \eqn{[0,1/2]}, \eqn{c_2>0} and \eqn{(1-c_1)^2+c_2^2 \le 1}.
#'
#' Any given triangle can be mapped to the basic triangle by a combination of rigid body motions
#' (i.e., translation, rotation and reflection) and scaling, preserving uniformity of the points in the
#' original triangle. Hence basic triangle is useful for simulation
#' studies under the uniformity hypothesis.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:mcap2012,ceyhan:arc-density-CS;textual}{pcds}).
#'
#' @param pt A 2D point for which \code{M}-edge region it resides in is to be determined in the
#' basic triangle \eqn{T_b}.
#' @param c1,c2 Positive real numbers which constitute the upper vertex of the basic triangle (i.e., the vertex
#' adjacent to the shorter edges of \eqn{T_b}); \eqn{c_1} must be in \eqn{[0,1/2]}, \eqn{c_2>0} and
#' \eqn{(1-c_1)^2+c_2^2 \le 1}.
#' @param M A 2D point in Cartesian coordinates or a 3D point in barycentric coordinates
#' which serves as a center in the interior of the basic triangle \eqn{T_b}.
#'
#' @return A \code{list} with three elements
#'
#' \item{re}{Index of the \code{M}-edge region that contains point, \code{pt} in the basic triangle \eqn{T_b}.}
#' \item{tri}{The vertices of the triangle, where row labels are \eqn{A}, \eqn{B}, and \eqn{C}
#' with edges are labeled as 3 for edge \eqn{AB}, 1 for edge \eqn{BC}, and 2 for edge \eqn{AC}.}
#' \item{desc}{Description of the edge labels}
#'
#' @seealso \code{\link{re.triCM}}, \code{\link{re.tri.cent}},
#' \code{\link{re.bastri.cent}}, \code{\link{reTeCM}}, and \code{\link{redge.triCM}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' c1<-.4; c2<-.6
#' A<-c(0,0); B<-c(1,0); C<-c(c1,c2);
#' Tb<-rbind(A,B,C);
#' M<-c(.6,.2)
#'
#' P<-c(.4,.2)
#' re.bastri.cent(P,c1,c2,M)
#'
#' P<-c(1.4,.2)
#' re.bastri.cent(P,c1,c2,M)
#'
#' c1<-.5; c2<-.8
#' P<-c(.4,.2)
#' re.bastri.cent(P,c1,c2,M)
#'
#' P<-c(.8,.2)
#' re.bastri.cent(P,c1,c2,M)
#'
#' c1<-.4; c2<-.6
#' A<-c(0,0);B<-c(1,0);C<-c(c1,c2);
#' Tb<-rbind(A,B,C)
#'
#' re.bastri.cent(A,c1,c2,M)
#' re.bastri.cent(B,c1,c2,M)
#' re.bastri.cent(C,c1,c2,M)
#' re.bastri.cent(M,c1,c2,M)
#'
#' n<-10  #try also n<-20
#' dat<-runif.bastri(n,c1,c2)$g
#'
#' M<-as.numeric(runif.bastri(1,c1,c2)$g)  #try also M<-c(.6,.2)
#'
#' re<-vector()
#' for (i in 1:n)
#'   re<-c(re,re.bastri.cent(dat[i,],c1,c2,M)$re)
#' re
#'
#' Xlim<-range(Tb[,1],dat[,1])
#' Ylim<-range(Tb[,2],dat[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(Tb,xlab="",ylab="",axes=TRUE,pch=".",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' points(dat,pch=".")
#' polygon(Tb)
#' L<-Tb; R<-rbind(M,M,M)
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#' text(dat,labels=factor(re))
#'
#' txt<-rbind(Tb,M)
#' xc<-txt[,1]+c(-.03,.03,.02,0)
#' yc<-txt[,2]+c(.02,.02,.02,-.03)
#' txt.str<-c("A","B","C","M")
#' text(xc,yc,txt.str)
#'
#' @export re.bastri.cent
re.bastri.cent <- function(pt,c1,c2,M)
{
  if (!is.point(pt))
  {stop('pt must be a numeric 2D point')}

  if (!is.point(c1,1) || !is.point(c2,1))
  {stop('c1 and c2 arguments must be scalars')}

  if (c1<0 || c1>1/2 || c2<=0 || (1-c1)^2+c2^2 >1)
  {stop('c1 must be in [0,1/2], c2>0 and (1-c1)^2+c2^2 <=1')}

  if (!is.point(M) && !is.point(M,3) )
  {stop('M must be a numeric 2D point for Cartesian coordinates or 3D point for barycentric coordinates')}

  y1<-c(0,0); y2<-c(1,0); y3<-c(c1,c2);
  Tb<-rbind(y1,y2,y3)

  if (dimension(M)==3)
  {
    M<-bary2cart(M,Tb)
  }

  if (in.triangle(M,Tb,boundary=FALSE)$in.tri==F)
  {stop('center is not in the interior of the triangle')}

  if (in.triangle(pt,Tb,boundary = TRUE)$in.tri==F)
  {reled<-NA
  } else
  {

    if (in.triangle(pt,rbind(y1,y2,M),boundary = TRUE)$in.tri)
      reled<-3
    else
    {
      if (in.triangle(pt,rbind(y2,y3,M),boundary = TRUE)$in.tri)
        reled<-1
      else reled<-2
    }
  }
  row.names(Tb)<-c("A","B","C")  #vertex labeling
  edge.desc<-"Edge labels are AB=3, BC=1, and AC=2"

  list(re=reled, #relative edge
       tri=Tb, #vertex labeling
       desc=edge.desc
  )
} #end of the function
#'

#################################################################

#' @title The index of the edge region in the standard equilateral triangle that contains a point
#'
#' @description Returns the index of the edge whose region contains point, \code{pt}, in
#' the standard equilateral triangle \eqn{T_e=T(A=(0,0),B=(1,0),C=(1/2,\sqrt{3}/2))} with edge regions based on
#' center of mass \eqn{CM=(A+B+C)/3}.
#'
#' Edges are labeled as 3 for edge \eqn{AB}, 1 for edge \eqn{BC}, and 2 for edge \eqn{AC}.
#' If the point, \code{pt}, is not inside \code{tri}, then the function yields \code{NA} as output.
#' Edge region 1 is the triangle \eqn{T(B,C,M)}, edge region 2 is \eqn{T(A,C,M)}, and edge region 3 is \eqn{T(A,B,M)}.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:mcap2012,ceyhan:arc-density-CS;textual}{pcds}).
#'
#' @param pt A 2D point for which \eqn{CM}-edge region it resides in is to be determined in the
#' the standard equilateral triangle \eqn{T_e}.
#'
#' @return A \code{list} with three elements
#' \item{re}{Index of the \eqn{CM}-edge region that contains point, \code{pt} in the standard equilateral triangle \eqn{T_e}}
#' \item{tri}{The vertices of the standard equilateral triangle \eqn{T_e}, where row labels are \eqn{A}, \eqn{B}, and \eqn{C}
#' with edges are labeled as 3 for edge \eqn{AB}, 1 for edge \eqn{BC}, and 2 for edge \eqn{AC}.}
#' \item{desc}{Description of the edge labels}
#'
#' @seealso \code{\link{re.triCM}}, \code{\link{re.tri.cent}},
#' \code{\link{re.bastriCM}}, \code{\link{re.bastri.cent}}, and \code{\link{redge.triCM}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' P<-c(.4,.2)
#' reTeCM(P)
#'
#' P<-c(.8,.2)
#' reTeCM(P)
#'
#' P<-c(.8,.8)
#' reTeCM(P)
#'
#' P<-c(.5,.61)
#' reTeCM(P)
#'
#' A<-c(0,0); B<-c(1,0); C<-c(0.5,sqrt(3)/2);
#' Te<-rbind(A,B,C)
#' D1<-(B+C)/2; D2<-(A+C)/2; D3<-(A+B)/2;
#' reTeCM(A)
#' reTeCM(B)
#' reTeCM(C)
#' reTeCM(D1)
#' reTeCM(D2)
#' reTeCM(D3)
#'
#' CM<-(A+B+C)/3
#' reTeCM(CM)
#'
#' n<-10  #try also n<-20
#' dat<-runifTe(n)$gen.points
#'
#' re<-vector()
#' for (i in 1:n)
#'   re<-c(re,reTeCM(dat[i,])$re)
#' re
#'
#' Xlim<-range(Te[,1],dat[,1])
#' Ylim<-range(Te[,2],dat[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(Te,asp=1,xlab="",ylab="",axes=TRUE,pch=".",xlim=Xlim+xd*c(-.01,.01),ylim=Ylim+yd*c(-.01,.01))
#' points(dat,pch=".")
#' polygon(Te)
#' L<-Te; R<-matrix(rep(CM,3),ncol=2,byrow=TRUE)
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#' text(dat,labels=factor(re))
#'
#' txt<-rbind(Te,CM)
#' xc<-txt[,1]+c(-.03,.03,.03,-.06)
#' yc<-txt[,2]+c(.02,.02,.02,.03)
#' txt.str<-c("A","B","C","CM")
#' text(xc,yc,txt.str)
#'
#' p1<-(A+B+CM)/3
#' p2<-(B+C+CM)/3
#' p3<-(A+C+CM)/3
#'
#' plot(Te,xlab="",ylab="",axes=TRUE,pch=".",xlim=Xlim+xd*c(-.01,.01),ylim=Ylim+yd*c(-.01,.01))
#' polygon(Te)
#' L<-Te; R<-matrix(rep(CM,3),ncol=2,byrow=TRUE)
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#'
#' txt<-rbind(Te,CM,p1,p2,p3)
#' xc<-txt[,1]+c(-.03,.03,.03,-.06,0,0,0)
#' yc<-txt[,2]+c(.02,.02,.02,.03,0,0,0)
#' txt.str<-c("A","B","C","CM","re=3","re=1","re=2")
#' text(xc,yc,txt.str)
#'
#' @export reTeCM
reTeCM <- function(pt)
{
  if (!is.point(pt))
  {stop('the argument must be a numeric 2D point')}

  A<-c(0,0); B<-c(1,0); C<-c(0.5,sqrt(3)/2);
  Te<-rbind(A,B,C)
  if (in.triangle(pt,Te,boundary=TRUE)$in.tri==F)
  {reled<-NA
  } else
  {
    if (pt[2]<= .5773502693*pt[1] && pt[2]<= .5773502693-.5773502693*pt[1])
      reled<-3
    else
    {
      reled<-ifelse (pt[1]>=1/2, 1, 2)
    }
  }
  row.names(Te)<-c("A","B","C")  #vertex labeling
  edge.desc<-"Edge labels are AB=3, BC=1, and AC=2"

  list(re=reled, #relative edge
       tri=Te, #vertex labeling
       desc=edge.desc
  )
} #end of the function
#'

#################################################################

#' @title The furthest points in a data set from edges in each \eqn{CM}-edge region in the standard
#' equilateral triangle
#'
#' @description
#' An object of class \code{"Extrema"}.
#' Returns the furthest data points among the data set, \code{Dt}, in each \eqn{CM}-edge region from the edge in the
#' standard equilateral triangle \eqn{T_e=T(A=(0,0),B=(1,0),C=(1/2,\sqrt{3}/2))}.
#'
#' \code{ch.all.intri} is for checking whether all data points are inside \eqn{T_e} (default is \code{FALSE}).
#'
#' See also (\insertCite{ceyhan:Phd-thesis;textual}{pcds}).
#'
#' @param Dt A set of 2D points, some could be inside and some could be outside standard equilateral triangle
#' \eqn{T_e}.
#' @param ch.all.intri A logical argument used for checking whether all data points are inside \eqn{T_e}
#' (default is \code{FALSE}).
#'
#' @return A \code{list} with the elements
#' \item{txt1}{Edge labels as \eqn{AB=3}, \eqn{BC=1}, and \eqn{AC=2} for \eqn{T_e} (corresponds to row number in Extremum Points).}
#' \item{txt2}{A short description of the distances as \code{"Distances to Edges"}.}
#' \item{type}{Type of the extrema points}
#' \item{desc}{A short description of the extrema points}
#' \item{mtitle}{The \code{"main"} title for the plot of the extrema}
#' \item{ext}{The extrema points, here, furthest points from edges in each edge region.}
#' \item{X}{The input data, \code{Dt}, can be a \code{matrix} or \code{data frame}}
#' \item{num.points}{The number of data points, i.e., size of \code{Dt}}
#' \item{supp}{Support of the data points, here, it is \eqn{T_e}.}
#' \item{cent}{The center point used for construction of edge regions.}
#' \item{ncent}{Name of the center, \code{cent}, it is center of mass \code{"CM"} for this function.}
#' \item{regions}{Edge regions inside the triangle, \eqn{T_e}, provided as a list.}
#' \item{region.names}{Names of the edge regions as \code{"er=1"}, \code{"er=2"}, and \code{"er=3"}.}
#' \item{region.centers}{Centers of mass of the edge regions inside \eqn{T_e}.}
#' \item{dist2ref}{Distances from furthest points in each edge region to the corresponding edge.}
#'
#' @seealso \code{\link{fr2vTbVRCC}}, \code{\link{fr2vVRCC}}, \code{\link{Kfr2vTbVRCC}},
#' \code{\link{Kfr2vVRCC}}, and \code{\link{cl2edgesTe}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' n<-20
#' dat<-runifTe(n)$gen.points
#'
#' Ext<-fr2eTeER(dat)
#' Ext
#' summary(Ext)
#' plot(Ext,asp=1)
#'
#' fr2eTeER(dat[1,])
#' ed.far<-fr2eTeER(dat)
#'
#' dat2<-rbind(dat,c(.8,.8))
#' fr2eTeER(dat2)
#' fr2eTeER(dat2,ch.all.intri = FALSE) #gives error if ch.all.intri = TRUE
#'
#' A<-c(0,0); B<-c(1,0); C<-c(0.5,sqrt(3)/2);
#' Te<-rbind(A,B,C)
#' CM<-(A+B+C)/3
#' p1<-(A+B)/2
#' p2<-(B+C)/2
#' p3<-(A+C)/2
#'
#' Xlim<-range(Te[,1],dat[,1])
#' Ylim<-range(Te[,2],dat[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(A,pch=".",xlab="",ylab="",axes=TRUE,xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Te)
#' L<-Te; R<-matrix(rep(CM,3),ncol=2,byrow=TRUE)
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#' points(dat,xlab="",ylab="")
#' points(ed.far$ext,pty=2,pch=4,col="red")
#'
#' txt<-rbind(Te,CM,p1,p2,p3)
#' xc<-txt[,1]+c(-.03,.03,.03,-.06,0,0,0)
#' yc<-txt[,2]+c(.02,.02,.02,.02,0,0,0)
#' txt.str<-c("A","B","C","CM","re=2","re=3","re=1")
#' text(xc,yc,txt.str)
#'
#' @export
fr2eTeER <- function(Dt,ch.all.intri=FALSE)
{
  if (!is.numeric(as.matrix(Dt)))
  {stop('Dt must be numeric')}

  if (is.point(Dt))
  { Dt<-matrix(Dt,ncol=2)
  } else
  {Dt<-as.matrix(Dt)
  if (ncol(Dt)!=2 )
  {stop('Dt must be of dimension nx2')}
  }

  A<-c(0,0); B<-c(1,0); C<-c(0.5,sqrt(3)/2);
  Te<-rbind(A,B,C)
  Cent<-(A+B+C)/3; Cname<-"CM"

  if (ch.all.intri==TRUE)
  {
    if (!inTriAll(Dt,Te,boundary=TRUE))
    {stop('not all points in the data set are in the standard equilateral triangle Te=T((0,0),(1,0),(1/2,sqrt(3)/2))')}
  }

  n<-nrow(Dt)

  D<-rep(0,3)
  xf<-matrix(NA,nrow=3,ncol=2)
  for (i in 1:n)
  {
    if (in.triangle(Dt[i,],Te,boundary=TRUE)$in.tri)
    {
      re<-reTeCM(Dt[i,])$re
      dis<-c((-.5*Dt[i,2]+.8660254040-.8660254040*Dt[i,1]),(-.5*Dt[i,2]+.8660254040*Dt[i,1]),Dt[i,2])
      if ( dis[re] > D[re])
      {
        D[re]<-dis[re]; xf[re,]<-Dt[i,]
      }
    }
  }

  row.names(Te)<-c("A","B","C")  #vertex labeling

  typ<-"Furthest Points from Edges among Data Points, Dt, in CM-Edge Regions in the Standard Equilateral Triangle T=(A,B,C) with A=(0,0), B=(1,0), and C=(1/2,sqrt(3)/2)"
  txt1<-"Edge Labels are AB=3, BC=1, and AC=2 (corresponds to row number in Extremum Points)"
  txt2<-"Distances from furthest points to edges in each edge region\n (Row i corresponds to edge i for i=1,2,3)"
  description<-"Furthest Points from Edges in corresponding CM-Edge Regions \n (Row i corresponds to edge i for i=1,2,3)  "
  main.txt<-"Furthest Points from Edges\n in CM-Edge Regions"

  Dis<-rbind(ifelse(!is.na(xf[1,1]),D[1],NA),ifelse(!is.na(xf[2,1]),D[2],NA),ifelse(!is.na(xf[3,1]),D[3],NA))
  #distances of the furthest points to the edges in corresponding edge regions

  Regs<-list(r1=rbind(A,B,Cent), #regions inside the triangles
             r2=rbind(B,C,Cent),
             r3=rbind(C,A,Cent)
  )
  Reg.Cent<-vector()
  for (i in 1:length(Regs))
  { Reg.Cent<-rbind(Reg.Cent,apply(Regs[[i]],2,mean))}

  Reg.names<-c("er=1","er=2","er=3")  #regions names

  res<-list(
    txt1=txt1, txt2=txt2,
    type=typ, desc=description,
    mtitle=main.txt,
    ext=xf, #furthest points from edges in each edge region
    X=Dt, num.points=n, #data points and its size
    ROI=Te, # region of interest for X points
    cent=Cent, ncent=Cname, #center and center name
    regions=Regs, region.names=Reg.names, region.centers=Reg.Cent,
    dist2ref=Dis #distances of furthest points to edges in each edge region
  )
  class(res)<-"Extrema"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

#' @title The closest points in a data set to edges in the standard equilateral triangle
#'
#' @description
#' An object of class \code{"Extrema"}.
#' Returns the closest points from the 2D data set, \code{Dt}, to the edges in the
#' standard equilateral triangle \eqn{T_e=T(A=(0,0),B=(1,0),C=(1/2,\sqrt{3}/2))}.
#'
#' \code{ch.all.intri} is for checking whether all data points are inside \eqn{T_e} (default is \code{FALSE}).
#'
#' If some of the data points are not inside \eqn{T_e} and \code{ch.all.intri=TRUE}, then the function yields
#' an error message.
#' If some of the data points are not inside \eqn{T_e} and \code{ch.all.intri=FALSE}, then the function yields
#' the closest points to edges among the data points inside \eqn{T_e} (yields \code{NA} if there are no data points
#' inside \eqn{T_e}).
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:arc-density-PE,ceyhan:masa-2007;textual}{pcds}).
#'
#' @param Dt A set of 2D points representing the set of data points.
#' @param ch.all.intri A logical argument (default=\code{FALSE}) to check whether all data points are inside
#' the standard equilateral triangle \eqn{T_e}. So, if it is \code{TRUE},
#' the function checks if all data points are inside the closure of the triangle (i.e., interior and boundary
#' combined) else it does not.
#'
#' @return A \code{list} with the elements
#' \item{txt1}{Edge labels as \eqn{AB=3}, \eqn{BC=1}, and \eqn{AC=2} for \eqn{T_e} (corresponds to row number in Extremum Points).}
#' \item{txt2}{A short description of the distances as \code{"Distances to Edges ..."}.}
#' \item{type}{Type of the extrema points}
#' \item{desc}{A short description of the extrema points}
#' \item{mtitle}{The \code{"main"} title for the plot of the extrema}
#' \item{ext}{The extrema points, i.e., closest points to edges}
#' \item{X}{The input data, \code{Dt}, which can be a \code{matrix} or \code{data frame}}
#' \item{num.points}{The number of data points, i.e., size of \code{Dt}}
#' \item{supp}{Support of the data points, i.e., the standard equilateral triangle \eqn{T_e}}
#' \item{cent}{The center point used for construction of edge regions, not required for this extrema,
#' hence it is \code{NULL} for this function}
#' \item{ncent}{Name of the center, \code{cent}, not required for this extrema, hence it is \code{NULL} for this function}
#' \item{regions}{Edge regions inside the triangle, \eqn{T_e}, not required for this extrema, hence it is \code{NULL}
#' for this function}
#' \item{region.names}{Names of the edge regions, not required for this extrema, hence it is \code{NULL}
#' for this function}
#' \item{region.centers}{Centers of mass of the edge regions inside \eqn{T_e}, not required for this extrema,
#' hence it is \code{NULL} for this function}
#' \item{dist2ref}{Distances from closest points in each edge region to the corresponding edge}
#'
#' @seealso \code{\link{cl2eTbVRcent}}, \code{\link{cl2eVRcent}}, \code{\link{cl2eVRCM}} and \code{\link{fr2eTeER}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' n<-10  #try also n<-20
#' dat<-runifTe(n)$gen.points
#'
#' Ext<-cl2edgesTe(dat)
#' Ext
#' summary(Ext)
#' plot(Ext,asp=1)
#'
#' cl2edgesTe(dat[1,])
#' cl2edgesTe(c(10,10))
#'
#' ed.clo<-cl2edgesTe(dat)
#'
#' dat2<-rbind(dat,c(.8,.8))
#' cl2edgesTe(dat2)
#'
#' A<-c(0,0); B<-c(1,0); C<-c(0.5,sqrt(3)/2);
#' Te<-rbind(A,B,C)
#' CM<-(A+B+C)/3
#' p1<-(A+B)/2
#' p2<-(B+C)/2
#' p3<-(A+C)/2
#'
#' Xlim<-range(Te[,1],dat[,1])
#' Ylim<-range(Te[,2],dat[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(A,pch=".",xlab="",ylab="",axes=TRUE,xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Te)
#' points(dat,xlab="",ylab="")
#' points(ed.clo$ext,pty=2,pch=4,col="red")
#'
#' txt<-rbind(Te,p1,p2,p3)
#' xc<-txt[,1]+c(-.03,.03,.03,0,0,0)
#' yc<-txt[,2]+c(.02,.02,.02,0,0,0)
#' txt.str<-c("A","B","C","re=1","re=2","re=3")
#' text(xc,yc,txt.str)
#'
#' dat.fr<-data.frame(a=dat)
#' cl2edgesTe(dat.fr)
#'
#' @export cl2edgesTe
cl2edgesTe <- function(Dt,ch.all.intri=FALSE)
{

  if (!is.numeric(as.matrix(Dt)))
  {stop('Dt must be numeric')}

  if (is.point(Dt))
  { Dt<-matrix(Dt,ncol=2)
  } else
  {Dt<-as.matrix(Dt)
  if (ncol(Dt)!=2 )
  {stop('Dt must be of dimension nx2')}
  }

  A<-c(0,0); B<-c(1,0); C<-c(0.5,sqrt(3)/2);
  Te<-rbind(A,B,C)
  Cent<-c()
  Cname<-NULL

  if (ch.all.intri==TRUE)
  {
    if (!inTriAll(Dt,Te,boundary=TRUE))
    {stop('not all points in the data set are in the standard equilateral triangle Te=T((0,0),(1,0),(1/2,sqrt(3)/2))')}
  }

  n<-nrow(Dt)

  D<-rep(0.8660254,3); #distance from a vertex to the opposite edge in Te
  xc<-matrix(NA,nrow=3,ncol=2)
  for (i in 1:n)
  {
    if (in.triangle(Dt[i,],Te,boundary=TRUE)$in.tri)
    {
      dis<-c((-.5*Dt[i,2]+.8660254040-.8660254040*Dt[i,1]),(-.5*Dt[i,2]+.8660254040*Dt[i,1]),Dt[i,2])
      for (j in 1:3)
      {
        if (dis[j]<D[j])
        {D[j]<-dis[j]; xc[j,]<-Dt[i,]}
      }
    }
  }

  row.names(Te)<-c("A","B","C")  #vertex labeling
  typ<-"Closest Points to Edges in the Standard Equilateral Triangle Te=T(A,B,C) with Vertices A=(0,0), B=(1,0), and C=(1/2,sqrt(3)/2)"
  txt1<-"Edge labels are AB=3, BC=1, and AC=2 (corresponds to row number in Extremum Points)"
  txt2<-"Distances to Edges in the Standard Equilateral Triangle \n (Row i corresponds to edge i for i=1,2,3)"
  description<-"Closest Points to Edges in the Standard Equilateral Triangle \n (Row i corresponds to edge i for i=1,2,3)  "
  main.txt<-"Closest Points to Edges in the\n Standard Equilateral Triangle"

  Dis<-rbind(ifelse(!is.na(xc[1,1]),D[1],NA),ifelse(!is.na(xc[2,1]),D[2],NA),ifelse(!is.na(xc[3,1]),D[3],NA))
  #distances of the closest points to the edges in \eqn{T_e}

  Regs<-Reg.Cent<-Reg.names<-c()

  res<-list(
    txt1=txt1, txt2=txt2,
    type=typ, desc=description,
    mtitle=main.txt,
    ext=xc, #closest points to edges in the std eq triangle
    X=Dt, num.points=n, #data points
    ROI=Te, # region of interest for X points
    cent=Cent, ncent=Cname, #center and center name
    regions=Regs, region.names=Reg.names, region.centers=Reg.Cent,
    dist2ref=Dis #distances of closest points to edges in each edge region
  )

  class(res)<-"Extrema"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

#' @title Region index inside the Gamma-1 region
#'
#' @description Returns the region index of the point \code{Pt} for the 6 regions in standard equilateral triangle
#' \eqn{T_e=T((0,0),(1,0),(1/2,\sqrt{3}/2))}, starting with 1 on the first one-sixth of the triangle,
#' and numbering follows the counter-clockwise direction (see the plot in the examples).
#' These regions are in the inner hexagon which is the Gamma-1 region for CS-PCD with \eqn{t=1}
#' if \code{Pt} is not in any of the 6 regions the function returns \code{NA}.
#'
#' @param Pt A 2D point whose index for the 6 regions in standard equilateral triangle \eqn{T_e} is determined.
#'
#' @return rel An integer between 1-6 (inclusive) or \code{NA}
#'
#' @seealso \code{\link{runifTe.onesixth}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' P<-c(.4,.2)
#' rel.six.Te(P)
#'
#' P<-c(.4,.3)
#' rel.six.Te(P)
#'
#' P<-c(.8,.8)
#' rel.six.Te(P)
#'
#' P<-c(.5,.61)
#' rel.six.Te(P)
#'
#' A<-c(0,0); B<-c(1,0); C<-c(0.5,sqrt(3)/2);
#' Te<-rbind(A,B,C)
#' CM<-(A+B+C)/3
#' D1<-(B+C)/2; D2<-(A+C)/2; D3<-(A+B)/2;
#' Ds<-rbind(D1,D2,D3)
#'
#' h1<-c(1/2,sqrt(3)/18); h2<-c(2/3, sqrt(3)/9); h3<-c(2/3, 2*sqrt(3)/9);
#' h4<-c(1/2, 5*sqrt(3)/18); h5<-c(1/3, 2*sqrt(3)/9); h6<-c(1/3, sqrt(3)/9);
#'
#' r1<-(h1+h6+CM)/3;r2<-(h1+h2+CM)/3;r3<-(h2+h3+CM)/3;
#' r4<-(h3+h4+CM)/3;r5<-(h4+h5+CM)/3;r6<-(h5+h6+CM)/3;
#'
#' Xlim<-range(Te[,1])
#' Ylim<-range(Te[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(A,pch=".",xlab="",ylab="",axes=TRUE,xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Te)
#' L<-Te; R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#' polygon(rbind(h1,h2,h3,h4,h5,h6))
#'
#' txt<-rbind(h1,h2,h3,h4,h5,h6)
#' xc<-txt[,1]+c(-.02,.02,.02,0,0,0)
#' yc<-txt[,2]+c(.02,.02,.02,0,0,0)
#' txt.str<-c("h1","h2","h3","h4","h5","h6")
#' text(xc,yc,txt.str)
#'
#' txt<-rbind(Te,CM,r1,r2,r3,r4,r5,r6)
#' xc<-txt[,1]+c(-.02,.02,.02,0,0,0,0,0,0,0)
#' yc<-txt[,2]+c(.02,.02,.02,0,0,0,0,0,0,0)
#' txt.str<-c("A","B","C","CM","1","2","3","4","5","6")
#' text(xc,yc,txt.str)
#'
#' n<-10  #try also n<-40
#' dat<-runifTe(n)$gen.points
#'
#' Xlim<-range(Te[,1],dat[,1])
#' Ylim<-range(Te[,2],dat[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' rsix<-vector()
#' for (i in 1:n)
#'   rsix<-c(rsix,rel.six.Te(dat[i,]))
#' rsix
#'
#' plot(A,pch=".",xlab="",ylab="",axes=TRUE,xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Te)
#' points(dat,pch=".")
#' L<-Te; R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#' polygon(rbind(h1,h2,h3,h4,h5,h6))
#' text(dat,labels=factor(rsix))
#'
#' txt<-rbind(Te,CM)
#' xc<-txt[,1]+c(-.02,.02,.02,0)
#' yc<-txt[,2]+c(.02,.02,.02,-.05)
#' txt.str<-c("A","B","C","CM")
#' text(xc,yc,txt.str)
#'
#' @export
rel.six.Te <- function(Pt)
{
  if (!is.point(Pt))
  {stop('the argument must be a numeric 2D point')}

  A<-c(0,0); B<-c(1,0); C<-c(0.5,sqrt(3)/2); Te<-rbind(A,B,C)
  if (in.triangle(Pt,Te,boundary = TRUE)$in.tri==F)
  {rel<-NA; return(rel); stop}

  rel<-NA
  if (Pt[1] <= 1/2 && Pt[2] <= .5773502693*Pt[1] && Pt[2] >= 0.3849001795 - 0.5773502693*Pt[1])
    rel<-1
  else
  {
    if (Pt[1] > 1/2 && Pt[2] <= .5773502693-.5773502693*Pt[1] && Pt[2] >= -.1924500898+.5773502693*Pt[1])
      rel<-2
    else
    {
      if (Pt[1] <= 2/3 && Pt[2] > .5773502693-.5773502693*Pt[1] && Pt[2] <= .5773502693*Pt[1])
        rel<-3
      else
      {
        if (Pt[1] >= 1/2 && Pt[2] <= .7698003590-.5773502693*Pt[1] && Pt[2] > .5773502693*Pt[1])
          rel<-4
        else
        {
          if (Pt[1] < 1/2 && Pt[2] <= 0.1924500898 + 0.5773502693*Pt[1] && Pt[2] >= 0.5773502693 - 0.5773502693*Pt[1])
            rel<-5
          else
          {
            if (Pt[1] >= 1/3 && Pt[2] > .5773502693*Pt[1] && Pt[2] < 0.5773502693 - 0.5773502693*Pt[1])
              rel<-6
          }
        }
      }
    }
  }
  rel
} #end of the function
#'

#################################################################

#' @title The closest points among a data set in the standard equilateral triangle
#' to the median lines in the six  half edge regions
#'
#' @description
#' An object of class \code{"Extrema"}.
#' Returns the six closest points among the data set, \code{Dt}, in the standard equilateral triangle
#' \eqn{T_e=T(A=(0,0),B=(1,0),C=(1/2,\sqrt{3}/2))} in half edge regions. In particular,
#' in regions \eqn{r_1} and \eqn{r_6}, it finds the closest point in each region to the line segment \eqn{[A,CM]}
#' in regions \eqn{r_2} and \eqn{r_3}, it finds the closest point in each region to the line segment \eqn{[B,CM]} and
#' in regions \eqn{r_4} and \eqn{r_5}, it finds the closest point in each region to the line segment \eqn{[C,CM]}
#' where \eqn{CM=(A+B+C)/3} is the center of mass.
#'
#' See the example for this function or example for
#' \code{rel.six.Te} function. If there is no data point in region \eqn{r_i}, then it returns "\code{NA} \code{NA}" for \eqn{i}-th row in the extrema.
#' \code{ch.all.intri} is for checking whether all data points are in \eqn{T_e} (default is \code{FALSE}).
#'
#' @param Dt A set of 2D points among which the closest points in the standard equilateral triangle
#' to the median lines in 6 half edge regions.
#' @param ch.all.intri A logical argument for checking whether all data points are in \eqn{T_e}
#' (default is \code{FALSE}).
#'
#' @return A \code{list} with the elements
#' \item{txt1}{Region labels as r1-r6 (corresponds to row number in Extremum Points).}
#' \item{txt2}{A short description of the distances as \code{"Distances to Line Segments (A,CM), (B,CM), and (C,CM) in the six regions r1-r6"}.}
#' \item{type}{Type of the extrema points}
#' \item{mtitle}{The \code{"main"} title for the plot of the extrema}
#' \item{ext}{The extrema points, here, closest points in each of regions \code{r1-r6} to the line segments
#' joining vertices to the center of mass, \eqn{CM}.}
#' \item{X}{The input data, \code{Dt}, can be a \code{matrix} or \code{data frame}}
#' \item{num.points}{The number of data points, i.e., size of \code{Dt}}
#' \item{supp}{Support of the data points, here, it is \eqn{T_e}.}
#' \item{cent}{The center point used for construction of edge regions.}
#' \item{ncent}{Name of the center, \code{cent}, it is center of mass \code{"CM"} for this function.}
#' \item{regions}{The six regions, \code{r1-r6} and edge regions inside the triangle, \eqn{T_e}, provided as a list.}
#' \item{region.names}{Names of the regions as \code{"r1"}-\code{"r6"} and names of the edge regions as \code{"er=1"}, \code{"er=2"}, and \code{"er=3"}.}
#' \item{region.centers}{Centers of mass of the regions \code{r1-r6} and of edge regions inside \eqn{T_e}.}
#' \item{dist2ref}{Distances from closest points in each of regions \code{r1-r6} to the line segments
#' joining vertices to the center of mass, \eqn{CM}.}
#'
#' @seealso \code{\link{rel.six.Te}} and \code{\link{cl2edgesTe}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' n<-10  #try also n<-100
#' dat<-runifTe(n)$gen.points
#'
#' Ext<-six.ext(dat)
#' Ext
#' summary(Ext)
#' plot(Ext)
#'
#' six.ext(dat[1:5,])$ext
#' sixt<-six.ext(dat)
#'
#' A<-c(0,0); B<-c(1,0); C<-c(0.5,sqrt(3)/2);
#' Te<-rbind(A,B,C)
#' CM<-(A+B+C)/3
#' D1<-(B+C)/2; D2<-(A+C)/2; D3<-(A+B)/2;
#' Ds<-rbind(D1,D2,D3)
#'
#' h1<-c(1/2,sqrt(3)/18); h2<-c(2/3, sqrt(3)/9); h3<-c(2/3, 2*sqrt(3)/9);
#' h4<-c(1/2, 5*sqrt(3)/18); h5<-c(1/3, 2*sqrt(3)/9); h6<-c(1/3, sqrt(3)/9);
#'
#' r1<-(h1+h6+CM)/3;r2<-(h1+h2+CM)/3;r3<-(h2+h3+CM)/3;
#' r4<-(h3+h4+CM)/3;r5<-(h4+h5+CM)/3;r6<-(h5+h6+CM)/3;
#'
#' Xlim<-range(Te[,1],dat[,1])
#' Ylim<-range(Te[,2],dat[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(A,pch=".",xlab="",ylab="",axes=TRUE,xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Te)
#' L<-Te; R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#' polygon(rbind(h1,h2,h3,h4,h5,h6))
#' points(dat)
#' points(sixt$ext,pty=2,pch=4,col="red")
#'
#' txt<-rbind(Te,r1,r2,r3,r4,r5,r6)
#' xc<-txt[,1]+c(-.02,.02,.02,0,0,0,0,0,0)
#' yc<-txt[,2]+c(.02,.02,.03,0,0,0,0,0,0)
#' txt.str<-c("A","B","C","1","2","3","4","5","6")
#' text(xc,yc,txt.str)
#'
#' dat.fr<-data.frame(a=dat)
#' six.ext(dat.fr)
#'
#' dat2<-rbind(dat,c(.8,.8))
#' six.ext(dat2)
#'
#' @export
six.ext <- function(Dt,ch.all.intri=FALSE)
{
  A<-c(0,0); B<-c(1,0); C<-c(0.5,sqrt(3)/2); Te<-rbind(A,B,C)
  Cent<-(A+B+C)/3; Cname<-"CM"
  D1<-(B+C)/2; D2<-(A+C)/2; D3<-(A+B)/2;
  Ds<-rbind(D1,D2,D3)

  if (ch.all.intri==TRUE)
  {
    if (!inTriAll(Dt,Te,boundary=TRUE))
    {stop('not all points in the data set are in the standard equilateral triangle Te=T((0,0),(1,0),(1/2,sqrt(3)/2))')}
  }

  if (is.point(Dt))
  { Dt<-matrix(Dt,ncol=2)
  } else
  {Dt<-as.matrix(Dt)
  if (ncol(Dt)!=2 )
  {stop('Dt must be of dimension nx2')}
  }

  n<-nrow(Dt)

  D<-rep(0.5773503,6); #distance from CM to each of the vertices in CM
  xc<-matrix(NA,nrow=6,ncol=2)
  for (i in 1:n)
  {rel<-rel.six.Te(Dt[i,])
  if (!is.na(rel))
  {x<-Dt[i,1]; y<-Dt[i,2];
  dis<-c((-0.8660254042*y + 0.5*x),(-.8660254042*y+.5-.5*x),(0.8660254042*y - 0.5+ 0.5*x),(x-.5),(.5-x),
         (.8660254042*y-.5000000003*x))
  if ( dis[rel] < D[rel])
  {
    D[rel]<-dis[rel]; xc[rel,]<-Dt[i,]
  }
  }
  }

  row.names(Te)<-c("A","B","C")  #vertex labeling
  row.names(xc)<-c("closest to line segment (A,CM) in region r1:",
                   "closest to line segment (B,CM) in region r2:",
                   "closest to line segment (B,CM) in region r3:",
                   "closest to line segment (C,CM) in region r4:",
                   "closest to line segment (C,CM) in region r5:",
                   "closest to line segment (A,CM) in region r6:")  #extrema labeling
  typ<-"Closest Points to Line Segments (A,CM), (B,CM), and (C,CM), in the six regions r1-r6 in the Standard Equilateral Triangle with Vertices A=(0,0), B=(1,0), and C=(1/2,sqrt(3)/2)"
  txt1<-"Region labels are r1-r6 (corresponding to row number in Extremum Points)"
  txt2<-"Distances to Line Segments (A,CM), (B,CM), and (C,CM) in the six regions r1-r6"
  description<-"Closest Points to Line Segments (A,CM), (B,CM), and (C,CM) in the Six Regions r1-r6 in the Standard Equilateral Triangle"
  main.txt<-paste("Closest Points to Line Segments (A,CM), (B,CM),\n and (C,CM) in the Six Regions r1-r6\n in the Standard Equilateral Triangle")

  h1<-c(1/2,sqrt(3)/18); h2<-c(2/3, sqrt(3)/9); h3<-c(2/3, 2*sqrt(3)/9);
  h4<-c(1/2, 5*sqrt(3)/18); h5<-c(1/3, 2*sqrt(3)/9); h6<-c(1/3, sqrt(3)/9);
  Regs<-list(r1=rbind(h6,h1,Cent), #regions inside the triangles
             r2=rbind(h1,h2,Cent),
             r3=rbind(h2,h3,Cent),
             r4=rbind(h3,h4,Cent),
             r5=rbind(h4,h5,Cent),
             r6=rbind(h5,h6,Cent),
             reg1=rbind(A,B,Cent),
             reg2=rbind(A,C,Cent),
             reg3=rbind(B,C,Cent)
  )
  Reg.Cent<-vector()
  for (i in 1:length(Regs))
  { Reg.Cent<-rbind(Reg.Cent,apply(Regs[[i]],2,mean))}

  Reg.names<-c("r1","r2","r3","r4","r5","r6"," "," ")  #regions names

  res<-list(
    txt1=txt1, txt2=txt2,
    type=typ, desc=description,
    mtitle=main.txt, #main label in the plot
    ext=xc, #closest points to line segments joining vertices to CM in each of regions r1-r6.
    X=Dt, num.points=n, #data points and its size
    ROI=Te, # region of interest for X points
    cent=Cent, ncent=Cname, #center and center name
    regions=Regs, region.names=Reg.names, region.centers=Reg.Cent,
    dist2ref=D #distances of closest points to line segments joining vertices to CM in each of regions r1-r6.
  )

  class(res)<-"Extrema"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

#functions generated for general triangles

#' @title Generation of Uniform Points in the Convex Hull of Points
#'
#' @description
#' An object of class \code{"Uniform"}.
#' Generates \code{n} points uniformly in the Convex Hull of set of points, \code{Yp}. That is, generates
#' uniformly in each of the triangles in the Delaunay triangulation of \code{Yp}, i.e., in the multiple
#' triangles partitioning the convex hull of \code{Yp}.
#'
#' If \code{Yp} consists only of 3 points, then the function behaves like the
#' function \code{\link{runif.tri}}.
#'
#' \code{DTmesh} is the Delaunay triangulation of \code{Yp}, default is \code{DTmesh=NULL}.
#' \code{DTmesh} yields triangulation nodes with neighbours (result of \code{\link[interp]{tri.mesh}} function from \code{interp} package).
#'
#' See (\insertCite{okabe:2000,ceyhan:comp-geo-2010,sinclair:2016;textual}{pcds}) for more on Delaunay triangulation and the corresponding algorithm.
#'
#' @param n A positive integer representing the number of uniform points to be generated in the convex hull
#' of the point set \code{Yp}.
#' @param Yp A set of 2D points whose convex hull is the support of the uniform points to be generated.
#' @param DTmesh Triangulation nodes with neighbours (result of \code{\link[interp]{tri.mesh}} function from
#' \code{interp} package).
#'
#' @return A \code{list} with the elements
#' \item{type}{The type of the pattern from which points are to be generated}
#' \item{mtitle}{The \code{"main"} title for the plot of the point pattern}
#' \item{tess.points}{The points which constitute the vertices of the triangulation and
#' whose convex hull determines the support of the generated points.}
#' \item{gen.points}{The output set of generated points uniformly in the convex hull of \code{Yp}}
#' \item{out.region}{The outer region which contains the support region, \code{NULL} for this function.}
#' \item{desc.pat}{Description of the point pattern from which points are to be generated}
#' \item{num.points}{The \code{vector} of two numbers, which are the number of generated points and
#' the number of vertices in the triangulation (i.e., size of \code{Yp}) points.}
#' \item{txt4pnts}{Description of the two numbers in \code{num.points}}
#' \item{xlimit,ylimit}{The ranges of the \eqn{x}- and \eqn{y}-coordinates of the points in Yp}
#'
#' @seealso \code{\link{runif.tri}}, \code{\link{runifTe}}, and \code{\link{runif.bastri}},
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' #nx is number of X points (target) and ny is number of Y points (nontarget)
#' nx<-20; ny<-4;  #try also nx<-1000; ny<-10;
#' set.seed(1)
#' Yp<-cbind(runif(ny,0,10),runif(ny,0,10))
#'
#' Xdt<-runifMT(nx,Yp)  #data under CSR in the convex hull of Ypoints
#' Xdt
#' summary(Xdt)
#' plot(Xdt)
#'
#' dat<-runifMT(nx,Yp)$g #data under CSR in the convex hull of Ypoints
#' #or use
#' DTY<-interp::tri.mesh(Yp[,1],Yp[,2],duplicate="remove")  #Delaunay triangulation based on Y points
#' dat<-runifMT(nx,Yp,DTY)$g #data under CSR in the convex hull of Ypoints
#'
#' Xlim<-range(Yp[,1])
#' Ylim<-range(Yp[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' #plot of the data in the convex hull of Y points together with the Delaunay triangulation
#' plot(dat,main=" ", xlab=" ", ylab=" ",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05),type="n")
#' interp::plot.triSht(DTY, add=TRUE, do.points = TRUE,pch=16,col="blue")
#' points(dat,pch=".",cex=3)
#'
#' Yp<-rbind(c(.3,.2),c(.4,.5),c(.14,.15))
#' runifMT(nx,Yp)
#'
#' dat.fr<-data.frame(a=Yp)
#' runifMT(nx,dat.fr)
#'
#' @export runifMT
runifMT <- function(n,Yp,DTmesh=NULL)
{
  Yp<-as.matrix(Yp)
  if (!is.numeric(Yp) || ncol(Yp)!=2 || nrow(Yp)<3)
  {stop('Yp must be numeric and of dimension kx2 with k>=3')}

  ny<-nrow(Yp)
  if (ny==3)
  {
    res<-runif.tri(n,Yp)
  } else
  {
  if (is.null(DTmesh))
  {DTmesh<-interp::tri.mesh(Yp[,1],Yp[,2],duplicate="remove")} #Delaunay triangulation

  cnt<-1
  dat<-NULL
  while (cnt <= n)
  {
    lx<-min(Yp[,1]); rx<-max(Yp[,1])
    ly<-min(Yp[,2]); ry<-max(Yp[,2])
    x1<-runif(1,lx,rx); y1<-runif(1,ly,ry)
    if (interp::in.convex.hull(DTmesh,x1,y1))
    {
      dat<-rbind(dat,c(x1,y1));
      cnt<-cnt+1
    }
  }
  typ<-"Uniform distribution in the union of multiple triangles that partition the convex hull of Yp"
  main.txt<-"Uniform Points in Convex Hull of Yp"
  txt<-paste(n, "uniform points in the union of the Delaunay triangles based on Yp  (or in convex hull of Yp)")

  ny<-nrow(Yp)
  npts<-c(n,ny)
  names(npts)<-c("nx","ny")
  txt4pnts<-paste(names(npts)[1], "is the number of Uniform points \n",names(npts)[2], "is the number of points whose convex hull determines the support")

  res<-list(
    type=typ,
    mtitle=main.txt,
    gen.points=dat, #uniformly generated points in the triangle, tri
    tess.points=Yp, #The points that determine the support of the uniform points
    desc.pat=txt, #description of the pattern
    out.region=NULL, #outer region for convex hull
    num.points=npts,  txt4pnts=txt4pnts,
    xlimit=range(Yp[,1]),
    ylimit=range(Yp[,2])
  )

  class(res)<-"Uniform"
  res$call <-match.call()
  }

  res
} #end of the function
#'

#################################################################

# funsIndDelTri
#'
#' @title Functions provide the indices of the Delaunay triangles where the points reside
#'
#' @description
#' Two functions: \code{ind.Del.tri} and \code{indices.Del.tri}.
#'
#' \code{ind.Del.tri} finds the index of the Delaunay triangle in which the given point, \code{pt}, resides.
#'
#' \code{indices.Del.tri} finds the indices of triangles for all the points in data set, \code{dat}, as a vector.
#'
#' Delaunay triangulation is based on \code{Yp} and \code{DTmesh} are the Delaunay triangles with default \code{NULL}.
#' The function returns \code{NA} for a point not inside the convex hull of \code{Yp}.
#' Number of \code{Yp} points (i.e., size of \code{Yp}) should be at least three and the points
#' should be in general position so that Delaunay triangulation is (uniquely) defined.
#'
#' If the number of \code{Yp} points is 3, then there is only one Delaunay triangle and the indices of all
#' the points inside this triangle are all 1.
#'
#' See (\insertCite{okabe:2000,ceyhan:comp-geo-2010,sinclair:2016;textual}{pcds}) for more on Delaunay
#' triangulation and the corresponding algorithm.
#'
#' @param pt A 2D point; the index of the Delaunay triangle in which \code{pt} resides is to be
#' determined. It is an argument for \code{ind.Del.tri}.
#' @param dat A set of 2D points representing the set of data points for which the indices of the Delaunay
#' triangles they reside is to be determined. It is an argument for \code{indices.Del.tri}.
#' @param Yp A set of 2D points from which Delaunay triangulation is constructed.
#' @param DTmesh Delaunay triangles based on \code{Yp}, default is \code{NULL},
#' which is computed via \code{\link[interp]{tri.mesh}} function
#' in \code{interp} package. \code{\link[interp]{triangles}} function yields a triangulation data structure from the triangulation object
#' created by \code{tri.mesh}.
#'
#' @return \code{ind.Del.tri} returns the index of the Delaunay triangle in which the given point, \code{pt}, resides
#' and \code{indices.Del.tri} returns the \code{vector} of indices of the Delaunay triangles in which points in the data
#' set, \code{dat}, reside.
#'
#' @name funsIndDelTri
NULL
#'
#' @rdname funsIndDelTri
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' #Examples for ind.Del.tri
#' nx<-100 #number of X points (target)
#' ny<-10 #number of Y points (nontarget)
#'
#' Yp<-cbind(runif(ny),runif(ny))
#'
#' dat<-runifMT(nx,Yp)$g #data under CSR in the convex hull of Ypoints
#' #try also dat<-cbind(runif(nx),runif(nx))
#'
#' ind.Del.tri(dat[10,],Yp)
#'
#' #or use
#' DTY<-interp::tri.mesh(Yp[,1],Yp[,2],duplicate="remove")  #Delaunay triangulation
#' TRY<-interp::triangles(DTY)[,1:3];
#' ind.Del.tri(dat[10,],Yp,DTY)
#'
#' ind.Del.tri(c(.5,.5),Yp)
#'
#' ind.Del.tri(c(1.5,.5),Yp)
#'
#' ind.DT<-vector()
#' for (i in 1:nx)
#'  ind.DT<-c(ind.DT,ind.Del.tri(dat[i,],Yp))
#' ind.DT
#'
#' Xlim<-range(Yp[,1],dat[,1])
#' Ylim<-range(Yp[,2],dat[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' DTY<-interp::tri.mesh(Yp[,1],Yp[,2],duplicate="remove")  #Delaunay triangulation based on Y points
#'
#' #plot of the data in the convex hull of Y points together with the Delaunay triangulation
#' plot(dat,main=" ", xlab=" ", ylab=" ",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05),type="n")
#' interp::plot.triSht(DTY, add=TRUE, do.points = TRUE,pch=16,col="blue")
#' points(dat,pch=".",cex=3)
#' text(dat,labels = factor(ind.DT) )
#'
#' Yp<-rbind(c(.3,.2),c(.4,.5),c(.14,.15))
#' ind.Del.tri(c(.25,.25),Yp)
#'
#' dat.fr<-data.frame(a=Yp)
#' ind.Del.tri(c(.25,.25),dat.fr)
#'
#' @export
ind.Del.tri <- function(pt,Yp,DTmesh=NULL)
{
  if (!is.point(pt))
  {stop('pt must be a numeric 2D point')}

 ind<-NA;
  if (nrow(Yp)==3)
  {
    if (in.triangle(pt,Yp)$in.tri)
    {ind<-1}
  } else
  {
  Yp<-as.matrix(Yp)
  if (!is.numeric(Yp) || ncol(Yp)!=2 || nrow(Yp)<3)
  {stop('Yp must be numeric and of dimension kx2 with k>=3')}

  if (is.null(DTmesh))
  {DTmesh<-interp::tri.mesh(Yp[,1],Yp[,2],duplicate="remove")  #Delaunay triangulation
  }
  DTr<-matrix(interp::triangles(DTmesh)[,1:3],ncol=3); #the Delaunay triangles

  nt<-nrow(DTr)  #number of Delaunay triangles;

  for (i in 1:nt)
  {
    tri<-Yp[DTr[i,],];
    if (in.triangle(pt,tri)$in.tri)
    {ind<-i}
  }
  }
 ind
} #end of the function
#'
#' @rdname funsIndDelTri
#'
#' @examples
#' #Examples for indices.Del.tri
#' #nx is number of X points (target) and ny is number of Y points (nontarget)
#' nx<-20; ny<-4;  #try also nx<-40; ny<-10 or nx<-1000; ny<-10;
#'
#' set.seed(1)
#' Yp<-cbind(runif(ny),runif(ny))
#' dat<-runifMT(nx,Yp)$g #data under CSR in the convex hull of Ypoints
#' #try also dat<-cbind(runif(nx),runif(nx))
#'
#' tr.ind<-indices.Del.tri(dat,Yp)  #indices of the Delaunay triangles
#' tr.ind
#'
#' #or use
#' DTY<-interp::tri.mesh(Yp[,1],Yp[,2],duplicate="remove")  #Delaunay triangulation based on Y points
#' tr.ind<-indices.Del.tri(dat,Yp,DTY)  #indices of the Delaunay triangles
#' tr.ind
#'
#' Xlim<-range(Yp[,1],dat[,1])
#' Ylim<-range(Yp[,2],dat[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' #plot of the data in the convex hull of Y points together with the Delaunay triangulation
#'
#' oldpar <- par(no.readonly = TRUE)
#' par(pty="s")
#' plot(dat,main=" ", xlab=" ", ylab=" ",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05),pch=".")
#' interp::plot.triSht(DTY, add=TRUE, do.points = TRUE,pch=16,col="blue")
#' text(dat,labels = factor(tr.ind) )
#' par(oldpar)
#'
#' Yp<-rbind(c(.3,.2),c(.4,.5),c(.14,.15))
#' indices.Del.tri(c(.25,.25),Yp)
#'
#' dat<-rbind(c(.25,.25),c(.15,.25))
#' indices.Del.tri(dat,Yp)
#'
#' dat.fr<-data.frame(a=dat)
#' indices.Del.tri(dat.fr,Yp)
#'
#' dat.fr<-data.frame(a=Yp)
#' indices.Del.tri(c(.25,.25),dat.fr)
#'
#' @export
indices.Del.tri <- function(dat,Yp,DTmesh=NULL)
{
  if (!is.numeric(as.matrix(dat)))
  {stop('dat must be numeric')}

  if (is.point(dat))
  { dat<-matrix(dat,ncol=2)
  } else
  {dat<-as.matrix(dat)
  if (ncol(dat)!=2 )
  {stop('dat must be of dimension nx2')}
  }

  nt<-nrow(dat)

  Yp<-as.matrix(Yp)
  if (!is.numeric(Yp) || ncol(Yp)!=2 || nrow(Yp)<3)
  {stop('Yp must be numeric and of dimension kx2 with k>=3')}

 ind.set<-vector()
  for (i in 1:nt)
  {
   ind.set<-c(ind.set, ind.Del.tri(dat[i,],Yp,DTmesh) )
  }
 ind.set
} #end of the function
#'

#################################################################

#' @title The auxiliary triangle to define the support of type I segregation
#'
#' @description Returns the triangle whose intersection with a general triangle gives the support for
#' type I segregation given the \code{delta} (i.e., \eqn{\delta 100} \% area of a triangle around the
#' vertices is chopped off). See the plot in the examples.
#'
#' Caveat: the vertices of this triangle may be outside the triangle, \code{tri}, depending on the value of
#' \code{delta} (i.e., for small values of \code{delta}).
#'
#' @param delta A positive real number between 0 and 1 that determines the percentage of area of the triangle
#' around the vertices forbidden for point generation.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#'
#' @return the vertices of the triangle (stacked row-wise) whose intersection with a general triangle
#' gives the support for type I segregation for the given \code{delta}
#'
#' @seealso \code{\link{rsegTe}} and \code{\link{rsegMT}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' #the standard equilateral triangle
#' A<-c(0,0); B<-c(1,0); C<-c(1/2,sqrt(3)/2);
#' Te<-rbind(A,B,C);
#' delta<-.3  #try also .5,.75,.85
#' seg.tri.supp(delta,Te)
#'
#' Tseg<-seg.tri.supp(delta,Te)
#'
#' Xlim<-range(Te[,1],Tseg[,1])
#' Ylim<-range(Te[,2],Tseg[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' oldpar <- par(no.readonly = TRUE)
#' par(pty="s")
#' plot(Te,pch=".",xlab="",ylab="",
#' main="segregation support is the intersection\n of these two triangles"
#'      ,axes=TRUE,xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Te)
#' polygon(Tseg,lty=2)
#'
#' txt<-rbind(Te,Tseg)
#' xc<-txt[,1]+c(-.03,.03,.03,.05,.04,-.04)
#' yc<-txt[,2]+c(.02,.02,.04,-.03,0,0)
#' txt.str<-c("A","B","C","T1","T2","T3")
#' text(xc,yc,txt.str)
#' par(oldpar)
#'
#' #for a general triangle
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#' delta<-.3  #try also .5,.75,.85
#' Tseg<-seg.tri.supp(delta,Tr)
#'
#' Xlim<-range(Tr[,1],Tseg[,1])
#' Ylim<-range(Tr[,2],Tseg[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' oldpar <- par(no.readonly = TRUE)
#' par(pty="s")
#' plot(Tr,pch=".",xlab="",ylab="",
#' main="segregation support is the intersection\n of these two triangles"
#'      ,axes=TRUE,xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tr)
#' polygon(Tseg,lty=2)
#'
#' txt<-rbind(Tr,Tseg)
#' xc<-txt[,1]+c(-.03,.03,.03,.06,.04,-.04)
#' yc<-txt[,2]+c(.02,.02,.04,-.03,0,0)
#' txt.str<-c("A","B","C","T1","T2","T3")
#' text(xc,yc,txt.str)
#' par(oldpar)
#'
#' dat.fr<-data.frame(a=Tr)
#' seg.tri.supp(delta,dat.fr)
#'
#' @export
seg.tri.supp <- function(delta,tri)
{
  if (!is.point(delta,1) || delta<=0 || delta>=1)
  {stop('delta must be a scalar in (0,1)')}

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  if (delta <= 3/4)
  {k<-sqrt(delta/3);}
  else
  {k<-(2-sqrt(1-delta))/3}
  A<-tri[1,]; a1<-A[1]; a2<-A[2];
  B<-tri[2,]; b1<-B[1]; b2<-B[2];
  C<-tri[3,]; c1<-C[1]; c2<-C[2];

  T1<-c(2*k*c1-c1+a1-b1*k+b1-a1*k, -b2*k-a2*k+a2+2*c2*k-c2+b2);
  T2<-c(-k*c1+c1-a1+2*a1*k-b1*k+b1, 2*a2*k-b2*k+b2-c2*k+c2-a2);
  T3<-c(-k*c1+c1-b1-a1*k+a1+2*b1*k, -a2*k+2*b2*k+a2-c2*k+c2-b2);

  TRI<-rbind(T1,T2,T3)
  row.names(TRI)<-c()
  TRI
} #end of the function
#'

#################################################################

#' @title Generation of points segregated (in a Type I fashion) from a given set of points
#'
#' @description
#' An object of class \code{"Patterns"}.
#' Generates \code{n} points uniformly in the support for Type I segregation in the convex hull of
#' set of points, \code{Yp}.
#'
#' \code{delta} is the parameter of segregation (that is, \eqn{\delta 100} \% of the area around each vertex
#' in each Delaunay
#' triangle is forbidden for point generation). \code{delta} corresponds to \code{eps} in the standard equilateral triangle
#' \eqn{T_e} as \eqn{delta=4 eps ^2/3} (see \code{rsegTe}  function).
#'
#' If \code{Yp} consists only of 3 points, then the function behaves like the
#' function \code{\link{rseg.tri}}.
#'
#' \code{DTmesh} must be the Delaunay triangulation of \code{Yp}
#' and \code{DTr} must be the corresponding Delaunay triangles (both \code{DTmesh} and \code{DTr} are \code{NULL} by default).
#' If \code{NULL}, \code{DTmesh} is computed via \code{\link[interp]{tri.mesh}} and \code{DTr} is computed via \code{\link[interp]{triangles}} function in \code{interp} package.
#'
#' \code{\link[interp]{tri.mesh}} function yields the triangulation nodes with their neighbours, and creates a triangulation object,
#' and \code{\link[interp]{triangles}} function yields a triangulation data structure from the triangulation object created
#' by \code{\link[interp]{tri.mesh}} (the first three columns are the vertex indices of the Delaunay triangles.)
#'
#' See (\insertCite{ceyhan:arc-density-PE,ceyhan:arc-density-CS,ceyhan:dom-num-NPE-Spat2011;textual}{pcds}) for more on
#' the segregation pattern.
#' Also see (\insertCite{okabe:2000,ceyhan:comp-geo-2010,sinclair:2016;textual}{pcds}) for more on Delaunay triangulation and the corresponding algorithm.
#'
#' @param n A positive integer representing the number of points to be generated.
#' @param Yp A set of 2D points from which Delaunay triangulation is constructed.
#' @param delta A positive real number in \eqn{(0,4/9)}. \code{delta} is the parameter of segregation (that is, \eqn{\delta 100} %
#' area around each vertex in each Delaunay triangle is forbidden for point generation).
#' @param DTmesh Delaunay triangulation of \code{Yp}, default is \code{NULL}, which is computed via \code{\link[interp]{tri.mesh}} function
#' in \code{interp} package. \code{\link[interp]{tri.mesh}} function yields the triangulation nodes with their neighbours, and
#' creates a triangulation object.
#' @param DTr Delaunay triangles based on \code{Yp}, default is \code{NULL}, which is computed via \code{\link[interp]{tri.mesh}} function
#' in \code{interp} package. \code{\link[interp]{triangles}} function yields a triangulation data structure from the triangulation object created
#' by \code{\link[interp]{tri.mesh}}.
#'
#' @return A \code{list} with the elements
#' \item{type}{The type of the pattern from which points are to be generated}
#' \item{mtitle}{The \code{"main"} title for the plot of the point pattern}
#' \item{parameters}{Exclusion parameter, \code{delta}, of the Type I segregation pattern. \code{delta} is in \eqn{(0,4/9)}
#' \eqn{\delta 100} \% area around each vertex in each Delaunay triangle is forbidden for point generation.}
#' \item{ref.points}{The input set of points \code{Yp};
#' reference points, i.e., points from which generated points are segregated.}
#' \item{gen.points}{The output set of generated points segregated from \code{Yp} points.}
#' \item{tri.Y}{Logical output, \code{TRUE} if triangulation based on \code{Yp} points should be implemented.}
#' \item{desc.pat}{Description of the point pattern}
#' \item{num.points}{The \code{vector} of two numbers, which are the number of generated points
#' and the number of reference (i.e., \code{Yp}) points.}
#' \item{xlimit,ylimit}{The ranges of the \eqn{x}- and \eqn{y}-coordinates of the reference points, which are the
#' \code{Yp} points}
#'
#' @seealso \code{\link{rseg.disc}}, \code{\link{rsegTe}}, \code{\link{rsegIITe}}, and \code{\link{rascMT}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' #nx is number of X points (target) and ny is number of Y points (nontarget)
#' nx<-20; ny<-4;  #try also nx<-1000; ny<-10;
#'
#' set.seed(1)
#' Yp<-cbind(runif(ny),runif(ny))
#' del<-.4
#'
#' Xdt<-rsegMT(nx,Yp,del)
#' Xdt
#' summary(Xdt)
#' plot(Xdt)
#'
#' Yp<-cbind(runif(ny),runif(ny))
#' del<-.3 #try .5, .75, .85
#' dat<-rsegMT(nx,Yp,del)  #data under CSR in the convex hull of Ypoints
#'
#' #or use
#' DTY<-interp::tri.mesh(Yp[,1],Yp[,2],duplicate="remove")  #Delaunay triangulation based on Y points
#' dat<-rsegMT(nx,Yp,del,DTY)  #data under CSR in the convex hull of Ypoints
#'
#' #or use
#' DTY<-interp::tri.mesh(Yp[,1],Yp[,2],duplicate="remove")  #Delaunay triangulation based on Y points
#' TRY<-interp::triangles(DTY)[,1:3];
#' dat<-rsegMT(nx,Yp,del,DTr=TRY)  #data under CSR in the convex hull of Ypoints
#'
#' #or use
#' DTY<-interp::tri.mesh(Yp[,1],Yp[,2],duplicate="remove")  #Delaunay triangulation based on Y points
#' TRY<-interp::triangles(DTY)[,1:3];
#' dat<-rsegMT(nx,Yp,del,DTY,TRY)$gen.points #data under CSR in the convex hull of Ypoints
#'
#' Xlim<-range(Yp[,1])
#' Ylim<-range(Yp[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' #plot of the data in the convex hull of Y points together with the Delaunay triangulation
#' DTY<-interp::tri.mesh(Yp[,1],Yp[,2],duplicate="remove")  #Delaunay triangulation based on Y points
#'
#' oldpar <- par(no.readonly = TRUE)
#' par(pty="s")
#' plot(dat,main=" ", xlab=" ", ylab=" ",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05),type="n")
#' interp::plot.triSht(DTY, add=TRUE, do.points=TRUE,col="blue")
#' points(dat,pch=".",cex=3)
#' par(oldpar)
#'
#' Yp<-rbind(c(.3,.2),c(.4,.5),c(.14,.15))
#' rsegMT(nx,Yp,del)
#'
#' dat.fr<-data.frame(a=Yp)
#' rsegMT(nx,dat.fr,del)
#'
#' @export rsegMT
rsegMT <- function(n,Yp,delta,DTmesh=NULL,DTr=NULL)
{
  Yp<-as.matrix(Yp)
  if (!is.numeric(Yp) || ncol(Yp)!=2 || nrow(Yp)<3)
  {stop('Yp must be numeric and of dimension kx2 with k>=3')}

  if (!is.point(delta,1) || delta<=0 || delta>=1)
  {stop('delta must be a scalar in (0,1)')}

  ny<-nrow(Yp)
  if (ny==3)
  {
    res<-rseg.tri(n,Yp,delta)
  } else
  {
    if (is.null(DTmesh))
    {DTmesh<-interp::tri.mesh(Yp[,1],Yp[,2],duplicate="remove")  #Delaunay triangulation
    }

    if (is.null(DTr))
    {DTr<-matrix(interp::triangles(DTmesh)[,1:3],ncol=3); #the Delaunay triangles
    }

    DTr<-matrix(DTr,ncol=3)

    cnt<-1;ind<-0
    dat<-NULL
    while (cnt <= n)
    {
      pnt<-as.vector(runifMT(1,Yp,DTmesh)$g)
     ind<-ind.Del.tri(pnt,Yp,DTmesh)
      nodes<-as.numeric(DTr[ind,])
      tri<-Yp[nodes,]
      seg.tri<-seg.tri.supp(delta,tri)
      if (in.triangle(pnt,seg.tri,boundary=TRUE)$in.tri)
      {
        dat<-rbind(dat,pnt);
        cnt<-cnt+1
      }
    }
    row.names(dat)<-c()
    dname <-deparse(substitute(delta))

    param<-delta
    names(param)<-"exclusion parameter"
    typ<-paste("Type I Segregation of ",n, " points from ",ny, " Y points with exclusion parameter ",dname, "=",delta,sep="")

    npts<-c(n,ny)
    names(npts)<-c("nx","ny")

    txt<-"Type I Segregation of One Class from Another"
    main.txt<-paste("Type I Segregation in the Multi-Triangle Case \n with  Exclusion Parameter ",dname,"=",delta,sep="")

    res<-list(
      type=typ,
      parameters=param,
      gen.points=dat, #generated points segregated from Y points (vertices of Y triangles)
      ref.points=Yp, #reference points, i.e., points from which generated points are segregated (vertices of Y triangles)
      desc.pat=txt, #description of the pattern
      mtitle=main.txt,
      tri.Y=TRUE,
      num.points=npts,
      xlimit=range(Yp[,1]),
      ylimit=range(Yp[,2])
    )

    class(res)<-"Patterns"
    res$call <-match.call()
  }

  res
} #end of the function
#'

#################################################################

#' @title Generation of points associated (in a Type I fashion) with a given set of points
#'
#' @description
#' An object of class \code{"Patterns"}.
#' Generates \code{n} points uniformly in the support for Type I association in the convex hull of set of points, \code{Yp}.
#' \code{delta} is the parameter of association (that is, only \eqn{\delta 100} \% area around each vertex in each Delaunay
#' triangle is allowed for point generation).
#'
#' \code{delta} corresponds to \code{eps} in the standard equilateral triangle
#' \eqn{T_e} as \eqn{delta=4eps^2/3} (see \code{rsegTe}  function).
#'
#' If \code{Yp} consists only of 3 points, then the function behaves like the
#' function \code{\link{rasc.tri}}.
#'
#' \code{DTmesh} must be the Delaunay triangulation of \code{Yp}
#' and \code{DTr} must be the corresponding Delaunay triangles (both \code{DTmesh} and \code{DTr} are \code{NULL} by default).
#' If \code{NULL}, \code{DTmesh} is computed via \code{\link[interp]{tri.mesh}} and \code{DTr} is computed via \code{\link[interp]{triangles}} function in \code{interp} package.
#'
#' \code{\link[interp]{tri.mesh}} function yields the triangulation nodes with their neighbours, and creates a triangulation object,
#' and \code{\link[interp]{triangles}} function yields a triangulation data structure from the triangulation object created
#' by \code{\link[interp]{tri.mesh}} (the first three columns are the vertex indices of the Delaunay triangles).
#'
#' See (\insertCite{ceyhan:arc-density-PE,ceyhan:arc-density-CS,ceyhan:dom-num-NPE-Spat2011;textual}{pcds}) for more on the
#' association pattern.
#' Also see (\insertCite{okabe:2000,ceyhan:comp-geo-2010,sinclair:2016;textual}{pcds}) for more on Delaunay triangulation and the corresponding algorithm.
#'
#' @param n A positive integer representing the number of points to be generated.
#' @param Yp A set of 2D points from which Delaunay triangulation is constructed.
#' @param delta A positive real number in \eqn{(0,4/9)}. \code{delta} is the parameter of association (that is, only
#' \eqn{\delta 100} \% area around each vertex in each Delaunay triangle is allowed for point generation).
#' @param DTmesh Delaunay triangulation of \code{Yp}, default is \code{NULL}, which is computed via \code{\link[interp]{tri.mesh}} function
#' in \code{interp} package. \code{\link[interp]{tri.mesh}} function yields the triangulation nodes with their neighbours, and
#' creates a triangulation object.
#' @param DTr Delaunay triangles based on \code{Yp}, default is \code{NULL}, which is computed via \code{\link[interp]{tri.mesh}} function
#' in \code{interp} package. \code{\link[interp]{triangles}} function yields a triangulation data structure from the triangulation object created
#' by \code{\link[interp]{tri.mesh}}.
#'
#' @return A \code{list} with the elements
#' \item{type}{The type of the pattern from which points are to be generated}
#' \item{mtitle}{The \code{"main"} title for the plot of the point pattern}
#' \item{parameters}{Attraction parameter, \code{delta}, of the Type I association pattern. \code{delta} is in \eqn{(0,4/9)}
#' only \eqn{\delta 100} \% of the area around each vertex in each Delaunay triangle is allowed for point generation.}
#' \item{ref.points}{The input set of points \code{Yp};
#' reference points, i.e., points with which generated points are associated.}
#' \item{gen.points}{The output set of generated points associated with \code{Yp} points.}
#' \item{tri.Y}{Logical output, \code{TRUE} if triangulation based on \code{Yp} points should be implemented.}
#' \item{desc.pat}{Description of the point pattern}
#' \item{num.points}{The \code{vector} of two numbers, which are the number of generated points
#' and the number of reference (i.e., \code{Yp}) points.}
#' \item{xlimit,ylimit}{The ranges of the \eqn{x}- and \eqn{y}-coordinates of the reference points, which are the
#' \code{Yp} points}
#'
#' @seealso \code{\link{rasc.disc}}, \code{\link{rascTe}}, \code{\link{rascIITe}}, and \code{\link{rsegMT}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' \donttest{
#' #nx is number of X points (target) and ny is number of Y points (nontarget)
#' nx<-20; ny<-4;  #try also nx<-40; ny<-10 or nx<-1000; ny<-10;
#'
#' set.seed(1)
#' Yp<-cbind(runif(ny),runif(ny))
#' del<-.4
#'
#' Xdt<-rascMT(nx,Yp,del)
#' Xdt
#' summary(Xdt)
#' plot(Xdt)
#'
#' Yp<-cbind(runif(ny),runif(ny))
#' del<-.3 #try .5, .75, .85
#' dat<-rascMT(nx,Yp,del)  #data under CSR in the convex hull of Ypoints
#'
#' #or use
#' DTY<-interp::tri.mesh(Yp[,1],Yp[,2],duplicate="remove")  #Delaunay triangulation based on Y points
#' dat<-rascMT(nx,Yp,del,DTY)  #data under CSR in the convex hull of Ypoints
#'
#' #or use
#' DTY<-interp::tri.mesh(Yp[,1],Yp[,2],duplicate="remove")  #Delaunay triangulation based on Y points
#' TRY<-interp::triangles(DTY)[,1:3];
#' dat<-rascMT(nx,Yp,del,DTr=TRY)  #data under CSR in the convex hull of Ypoints
#'
#' #or use
#' DTY<-interp::tri.mesh(Yp[,1],Yp[,2],duplicate="remove")  #Delaunay triangulation based on Y points
#' TRY<-interp::triangles(DTY)[,1:3];
#' dat<-rascMT(nx,Yp,del,DTY,TRY)$g #data under CSR in the convex hull of Ypoints
#'
#' Xlim<-range(Yp[,1])
#' Ylim<-range(Yp[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' #plot of the data in the convex hull of Y points together with the Delaunay triangulation
#' DTY<-interp::tri.mesh(Yp[,1],Yp[,2],duplicate="remove")  #Delaunay triangulation based on Y points
#'
#' plot(dat,main=" ", xlab=" ", ylab=" ",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05),type="n")
#' interp::plot.triSht(DTY, add=TRUE, do.points=TRUE,col="blue")
#' points(dat,pch=".",cex=3)
#'
#' Yp<-rbind(c(.3,.2),c(.4,.5),c(.14,.15))
#' rascMT(nx,Yp,del)
#'
#' dat.fr<-data.frame(a=Yp)
#' rascMT(nx,dat.fr,del)
#' }
#'
#' @export rascMT
rascMT <- function(n,Yp,delta,DTmesh=NULL,DTr=NULL)
{
  Yp<-as.matrix(Yp)
  if (!is.numeric(Yp) || ncol(Yp)!=2 || nrow(Yp)<3)
  {stop('Yp must be numeric and of dimension kx2 with k>=3')}

  if (!is.point(delta,1) || delta<=0 || delta>=1)
  {stop('delta must be a scalar in (0,1)')}

  ny<-nrow(Yp)
  if (ny==3)
  {
    res<-rasc.tri(n,Yp,delta)
  } else
  {
    if (is.null(DTmesh))
    {DTmesh<-interp::tri.mesh(Yp[,1],Yp[,2],duplicate="remove")  #Delaunay triangulation
    }

    if (is.null(DTr))
    {DTr<-matrix(interp::triangles(DTmesh)[,1:3],ncol=3); #the Delaunay triangles
    }

    DTr<-matrix(DTr,ncol=3)

    cnt<-1;ind<-0
    dat<-NULL
    while (cnt <= n)
    {
      pnt<-as.vector(runifMT(1,Yp,DTmesh)$g)
     ind<-ind.Del.tri(pnt,Yp,DTmesh)
      nodes<-as.numeric(DTr[ind,])
      tri<-Yp[nodes,]
      seg.tri<-seg.tri.supp(delta,tri)
      if (!in.triangle(pnt,seg.tri)$in.tri)
      {
        dat<-rbind(dat,pnt);
        cnt<-cnt+1
      }
    }
    row.names(dat)<-c()

    dname <-deparse(substitute(delta))

    param<-delta
    names(param)<-"attraction parameter"
    typ<-paste("Type I Association of ",n, " points with ",ny, " Y points with attraction parameter ",dname, "=",delta,sep="")

    ny<-nrow(Yp)
    npts<-c(n,ny)
    names(npts)<-c("nx","ny")

    txt<-"Type I Association of One Class with Y points"
    main.txt<-paste("Type I Association in the \n Multi-Triangle Case with \n Attraction Parameter ",dname,"=",delta,sep="")

    res<-list(
      type=typ,
      parameters=param,
      gen.points=dat, #generated points associated Y points (vertices of Y triangles)
      ref.points=Yp, #reference points, i.e., points with which generated points are associated (vertices of Y triangles)
      desc.pat=txt, #description of the pattern
      mtitle=main.txt,
      tri.Y=TRUE,
      num.points=npts,
      xlimit=range(Yp[,1]),
      ylimit=range(Yp[,2])
    )

    class(res)<-"Patterns"
    res$call <-match.call()
  }

  res
} #end of the function
#'

#################################################################

#' @title Approximate domination number and approximate dominating set by the greedy algorithm
#'
#' @description Returns the (approximate) domination number and the indices (i.e., row numbers) for the corresponding
#' (approximate) minimum dominating set based on the incidence matrix \code{Inc.Mat} of a graph or a digraph
#' by using the greedy algorithm (\insertCite{chvatal:1979;textual}{pcds}).
#' Here the row number in the incidence matrix corresponds to the index of the vertex (i.e., index of the
#' data point). The function works whether loops are allowed or not (i.e., whether the first diagonal
#' is all 1 or all 0). This function may yield the actual domination number or overestimates it.
#'
#' @param Inc.Mat A square matrix consisting of 0's and 1's which represents the incidence matrix of
#' a graph or digraph.
#'
#' @return A \code{list} with two elements
#' \item{dom.num}{The cardinality of the (approximate) minimum dominating set found by the greedy algorithm.
#' i.e., (approximate) domination number of the graph or digraph whose incidence matrix \code{Inc.Mat} is given
#' as input.}
#' \item{ind.domset}{Indices of the rows in the incidence matrix \code{Inc.Mat} for the ((approximate) minimum
#' dominating set). The row numbers in the incidence matrix correspond to the indices of the vertices
#' (i.e., indices of the data points).}
#'
#' @seealso \code{\link{dom.exact}}, \code{\link{PEdom1D}}, \code{\link{PEdomtri}}, \code{\link{PEdomMTnd}},
#' and \code{\link{IndCSdomUBtri}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' n<-10
#' M<-matrix(sample(c(0,1),n^2,replace=TRUE),nrow=n)
#' diag(M)<-1
#'
#' dom.greedy(M)
#'
#' @export dom.greedy
dom.greedy <- function(Inc.Mat)
{
 inc.mat<-as.matrix(Inc.Mat)
  nr<-nrow(inc.mat); nc<-ncol(inc.mat)

  if (nr!=nc)
  {stop('Inc.Mat must a square matrix')}

  if (sum(inc.mat!=0 & inc.mat!=1)!=0 )
  {stop('Inc.Mat must have entries equal to 0 or 1')}

  if (!(all(diag(inc.mat)==0) || all(diag(inc.mat)==1) ))
  {stop('Inc.Mat must have all zeroes (when loops are not allowed) or all ones (when loops are allowed) in the diagonal.')}

  cnt<-nr
  dom.ind<-c()
  vert.ind<-1:nr
  while (cnt>0)
  {
    rsums<-apply(inc.mat,1,sum)
    max.ind<-which(rsums==max(rsums))
    sel.ind<-ifelse(length(max.ind)>1,sample(max.ind,1),max.ind)
    dom.ind<-c(dom.ind,vert.ind[sel.ind])
    nghd<-which(inc.mat[sel.ind,]==1)  #indices in the nghd of the max dominating vertex
    vert.ind<-vert.ind[-nghd]
   inc.mat1<-inc.mat[-nghd,]
    if (length(inc.mat1)>0)
    {ifelse(ncol(as.matrix(inc.mat1))==1,inc.mat2<-matrix(inc.mat1,ncol=length(inc.mat1)),inc.mat2<-as.matrix(inc.mat1))
     inc.mat<-as.matrix(inc.mat2[,-nghd])
      cnt<-nrow(inc.mat)
    } else
    {cnt<-0}
  }
  list(
    dom.num=length(dom.ind), #greedy domination number
   ind.domset=dom.ind #indices of vertices or data points for the greedy domination number
  )
} #end of the function
#'

#################################################################

#' @title Indicator for an upper bound for the domination number by the exact algorithm
#'
#' @description Returns 1 if the domination number is less than or equal to the prespecified value \code{k} and also the indices
#' (i.e., row numbers) of a dominating set of size \code{k} based on the incidence matrix \code{Inc.Mat} of a graph or
#' a digraph. Here the row number in the incidence matrix corresponds to the index of the vertex
#' (i.e., index of the data point). The function works whether loops are allowed or not
#' (i.e., whether the first diagonal is all 1 or all 0).
#' It takes a rather long time for large number of vertices (i.e., large number of row numbers).
#'
#' @param Inc.Mat A square matrix consisting of 0's and 1's which represents the incidence matrix of
#' a graph or digraph.
#' @param k A positive integer for the upper bound (to be checked) for the domination number.
#'
#' @return A \code{list} with two elements
#' \item{domUB}{The upper bound (to be checked) for the domination number. It is prespecified as \code{k}
#' in the function arguments.}
#' \item{IndUBdom}{The indicator for the upper bound for domination number of the graph or digraph being
#' the specified value \code{k} or not. It returns 1 if the upper bound is \code{k}, and 0 otherwise based on the incidence
#' matrix \code{Inc.Mat} of the graph or digraph.}
#' \item{ind.domset}{Indices of the rows in the incidence matrix \code{Inc.Mat} that correspond to the vertices in the
#' dominating set of size \code{k} if it exists, otherwise it yields \code{NULL}.}
#'
#' @seealso \code{\link{dom.exact}} and \code{\link{dom.greedy}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' n<-10
#' M<-matrix(sample(c(0,1),n^2,replace=TRUE),nrow=n)
#' diag(M)<-1
#'
#' dom.greedy(M)
#' IndUBdom(M,2)
#'
#' for (k in 1:n)
#' print(c(k,IndUBdom(M,k)))
#'
#' @export IndUBdom
IndUBdom <- function(Inc.Mat,k)
{
 inc.mat<-as.matrix(Inc.Mat)
  nr<-nrow(inc.mat); nc<-ncol(inc.mat)

  if (nr!=nc)
  {stop('Inc.Mat must a square matrix')}

  if (sum(inc.mat!=0 & inc.mat!=1)!=0 )
  {stop('Inc.Mat must have entries equal to 0 or 1')}

  if (!(all(diag(inc.mat)==0) || all(diag(inc.mat)==1) ))
  {stop('Inc.Mat must have all zeroes (when loops are not allowed) or all ones (when loops are allowed) in the diagonal.')}

  if (all(diag(inc.mat)==0))
  {diag(inc.mat)<-1}

  xc<-combinat::combn(1:nr,k); N1<-choose(nr,k);
  xc<-matrix(xc,ncol=N1)  #required when N1=1
  dom<-0; j<-1; ind.domset<-c();
  while (j<=N1 && dom==0)
  { ifelse(nrow(xc)==1,dom.check<-inc.mat[xc[,j],],dom.check<-apply(inc.mat[xc[,j],],2,sum))

    if (all(dom.check>=1))  #this is where domination is checked
    {dom<-1; ind.domset<-xc[,j]}
    j<-j+1;
  }

  list(domUB=k, #upper bound for the domination number
       IndUBdom=dom, #indicator that domination number <=k
      ind.domset=ind.domset #indices of the vertices in a dominating set of size k (if exists)
  )
} #end of the function
#'

#################################################################

#' @title Exact domination number (i.e., domination number by the exact algorithm)
#'
#' @description Returns the (exact) domination number based on the incidence matrix \code{Inc.Mat} of a graph or a digraph
#' and the indices (i.e., row numbers of \code{Inc.Mat}) for the corresponding (exact) minimum dominating set.
#' Here the row number in the incidence matrix corresponds to the index of the vertex (i.e., index of the data
#' point). The function works whether loops are allowed or not (i.e., whether the first diagonal is all 1 or
#' all 0). It takes a rather long time for large number of vertices (i.e., large number of row numbers).
#'
#' @inheritParams dom.greedy
#'
#' @return A \code{list} with two elements
#' \item{dom.num}{The cardinality of the (exact) minimum dominating set i.e., (exact) domination number of the
#' graph or digraph whose incidence matrix \code{Inc.Mat} is given as input.}
#' \item{ind.mds}{Indices of the rows in the incidence matrix \code{Inc.Mat} for the (exact) minimum
#' dominating set. The row numbers in the incidence matrix correspond to the indices of the vertices
#' (i.e., indices of the data points).}
#'
#' @seealso \code{\link{dom.greedy}}, \code{\link{PEdom1D}}, \code{\link{PEdomtri}}, \code{\link{PEdomMTnd}},
#' and \code{\link{IndCSdomUBtri}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' n<-10
#' M<-matrix(sample(c(0,1),n^2,replace=TRUE),nrow=n)
#' diag(M)<-1
#'
#' dom.greedy(M)
#' IndUBdom(M,2)
#' dom.exact(M)
#'
#' @export dom.exact
dom.exact <- function(Inc.Mat)
{
 inc.mat<-as.matrix(Inc.Mat)
  nr<-nrow(inc.mat); nc<-ncol(inc.mat)

  if (nr!=nc)
  {stop('Inc.Mat must a square matrix')}

  if (sum(inc.mat!=0 & inc.mat!=1)!=0 )
  {stop('Inc.Mat must have entries equal to 0 or 1')}

  if (!(all(diag(inc.mat)==0) || all(diag(inc.mat)==1) ))
  {stop('Inc.Mat must have all zeroes (when loops are not allowed) or all ones (when loops are allowed) in the diagonal.')}

  if (all(diag(inc.mat)==0))
  {diag(inc.mat)<-1}

  crit<-0; i<-1; ind.mds<-c();
  while (i<=nr && crit==0)
  { DN<-IndUBdom(inc.mat,i)
  if (DN$IndUBdom==1)  #this is where domination is checked
  {dom.num<-i; ind.mds<-DN$ind.domset;
  crit<-1}
  i<-i+1;
  }

  list(dom.num=dom.num, #domination number
      ind.mds=ind.mds #indices of the vertices in a minimum dominating set
  )
} #end of the function
#'

##################################################################################################################
###############################ARCSLICE or NAS Functions################################################################
##################################################################################################################
#AS_Functions.r;
#Contains the functions used in PCD calculations, such as generation of data in a given a triangle and
#tetrahedron estimation of gamma, arc density etc.
#Functions for NAS in R^2
#################################################################

#' @title Circumcenter of a basic triangle
#'
#' @description Returns the circumcenter of a basic triangle \eqn{T_b=T((0,0),(1,0),(c_1,c_2))} given \eqn{c_1}, \eqn{c_2}
#' where \eqn{c_1} is in \eqn{[0,1/2]}, \eqn{c_2>0} and \eqn{(1-c_1)^2+c_2^2 \le 1}.
#'
#' Any given triangle can be mapped to the basic triangle by a combination of rigid body motions
#' (i.e., translation, rotation and reflection) and scaling, preserving uniformity of the points in the
#' original triangle. Hence basic triangle is useful for simulation
#' studies under the uniformity hypothesis.
#'
#' See (\insertCite{weisstein-tri-centers,ceyhan:comp-geo-2010;textual}{pcds}) for triangle centers and
#' (\insertCite{ceyhan:arc-density-PE,ceyhan:arc-density-CS,ceyhan:dom-num-NPE-Spat2011;textual}{pcds}) for the
#' basic triangle.
#'
#' @param c1,c2 Positive real numbers representing the top vertex in basic triangle \eqn{T_b=T((0,0),(1,0),(c_1,c_2))},
#' \eqn{c_1} must be in \eqn{[0,1/2]}, \eqn{c_2>0} and \eqn{(1-c_1)^2+c_2^2 \le 1}.
#'
#' @return circumcenter of the basic triangle \eqn{T_b=T((0,0),(1,0),(c_1,c_2))} given \eqn{c_1}, \eqn{c_2} as the arguments of the
#' function
#'
#' @seealso \code{\link{circ.cent.tri}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' c1<-.4; c2<-.6;
#' A<-c(0,0); B<-c(1,0); C<-c(c1,c2);  #the vertices of the basic triangle Tb
#' Tb<-rbind(A,B,C)
#' CC<-circ.cent.bastri(c1,c2)  #the circumcenter
#' CC
#'
#' D1<-(B+C)/2; D2<-(A+C)/2; D3<-(A+B)/2; #midpoints of the edges
#' Ds<-rbind(D1,D2,D3)
#'
#' Xlim<-range(Tb[,1])
#' Ylim<-range(Tb[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' oldpar <- par(no.readonly = TRUE)
#' par(pty="s")
#' plot(A,pch=".",asp=1,xlab="",ylab="",axes=TRUE,xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tb)
#' points(rbind(CC))
#' L<-matrix(rep(CC,3),ncol=2,byrow=TRUE); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#'
#' txt<-rbind(Tb,CC,D1,D2,D3)
#' xc<-txt[,1]+c(-.03,.04,.03,.06,.06,-.03,0)
#' yc<-txt[,2]+c(.02,.02,.03,-.03,.02,.04,-.03)
#' txt.str<-c("A","B","C","CC","D1","D2","D3")
#' text(xc,yc,txt.str)
#' par(oldpar)
#'
#' #for an obtuse triangle
#' c1<-.4; c2<-.3;
#' A<-c(0,0); B<-c(1,0); C<-c(c1,c2);  #the vertices of the basic triangle Tb
#' Tb<-rbind(A,B,C)
#' CC<-circ.cent.bastri(c1,c2)  #the circumcenter
#' CC
#'
#' D1<-(B+C)/2; D2<-(A+C)/2; D3<-(A+B)/2; #midpoints of the edges
#' Ds<-rbind(D1,D2,D3)
#'
#' Xlim<-range(Tb[,1],CC[1])
#' Ylim<-range(Tb[,2],CC[2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' oldpar <- par(no.readonly = TRUE)
#' par(pty="s")
#' plot(A,pch=".",asp=1,xlab="",ylab="",axes=TRUE,xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tb)
#' points(rbind(CC))
#' L<-matrix(rep(CC,3),ncol=2,byrow=TRUE); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#'
#' txt<-rbind(Tb,CC,D1,D2,D3)
#' xc<-txt[,1]+c(-.03,.03,.03,.07,.07,-.05,0)
#' yc<-txt[,2]+c(.02,.02,.04,-.03,.03,.04,.06)
#' txt.str<-c("A","B","C","CC","D1","D2","D3")
#' text(xc,yc,txt.str)
#' par(oldpar)
#'
#' @export circ.cent.bastri
circ.cent.bastri <- function(c1,c2)
{
  if (!is.point(c1,1) || !is.point(c2,1))
  {stop('both arguments must be scalars')}

  if (c1<0 || c1>1/2 || c2<=0 || (1-c1)^2+c2^2 >1)
  {stop('c1 must be in [0,1/2], c2>0 and (1-c1)^2+c2^2 <=1')}

  c(1/2, 1/2*(c2^2-c1+c1^2)/c2)
} #end of the function
#'

#################################################################

#' @title Circumcenter of a general triangle
#'
#' @description Returns the circumcenter a given triangle, \code{tri}, with vertices stacked row-wise.
#' See (\insertCite{weisstein-tri-centers,ceyhan:comp-geo-2010;textual}{pcds}) for triangle centers.
#'
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#'
#' @return circumcenter of the triangle \code{tri}
#'
#' @seealso \code{\link{circ.cent.bastri}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);  #the vertices of the triangle Tr
#'
#' CC<-circ.cent.tri(Tr)  #the circumcenter
#' CC
#'
#' D1<-(B+C)/2; D2<-(A+C)/2; D3<-(A+B)/2; #midpoints of the edges
#' Ds<-rbind(D1,D2,D3)
#'
#' Xlim<-range(Tr[,1],CC[1])
#' Ylim<-range(Tr[,2],CC[2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' oldpar <- par(no.readonly = TRUE)
#' par(pty="s")
#' plot(A,asp=1,pch=".",xlab="",ylab="",axes=TRUE,xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tr)
#' points(rbind(CC))
#' L<-matrix(rep(CC,3),ncol=2,byrow=TRUE); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#'
#' txt<-rbind(Tr,CC,Ds)
#' xc<-txt[,1]+c(-.08,.08,.08,.12,-.09,-.1,-.09)
#' yc<-txt[,2]+c(.02,-.02,.03,-.06,.02,.06,-.04)
#' txt.str<-c("A","B","C","CC","D1","D2","D3")
#' text(xc,yc,txt.str)
#' par(oldpar)
#'
#' A<-c(0,0); B<-c(1,0); C<-c(1/2,sqrt(3)/2);
#' Te<-rbind(A,B,C);  #the vertices of the equilateral triangle Te
#' circ.cent.tri(Te)  #the circumcenter
#'
#' A<-c(0,0); B<-c(0,1); C<-c(2,0);
#' Tr<-rbind(A,B,C);  #the vertices of the triangle T
#' circ.cent.tri(Tr)  #the circumcenter
#'
#' dat.fr<-data.frame(a=Tr)
#' circ.cent.tri(dat.fr)
#'
#' @export circ.cent.tri
circ.cent.tri <- function(tri)
{
  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('the argument must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  A<-tri[1,]; B<-tri[2,]; C<-tri[3,];
  a1<-A[1]; a2<-A[2];
  b1<-B[1]; b2<-B[2];
  c1<-C[1]; c2<-C[2];

  Dx<--det(matrix(cbind(c(sum(A^2),sum(B^2),sum(C^2)),tri[,2],vec1),ncol=3))
  Dy<-det(matrix(cbind(c(sum(A^2),sum(B^2),sum(C^2)),tri[,1],vec1),ncol=3))

  cc1<--Dx/(2*D0); cc2<--Dy/(2*D0);
  c(cc1,cc2)
} #end of the function
#'

#################################################################

# funsTbMid2CC
#'
#' @title Two functions \code{lD1CCinTb} and \code{lD2CCinTb} which are of class \code{"TriLines"} ---
#'
#' The lines joining the midpoints of edges to the circumcenter (eqn{CC}) in the basic triangle.
#'
#' @description Returns the \code{equation, slope, intercept}, and \eqn{y}-coordinates of the lines
#' joining \eqn{D_1} and \eqn{CC} and also \eqn{D_2} and \eqn{CC}, in the basic triangle \eqn{T_b=T(A=(0,0),B=(1,0),C=(c_1,c_2))}
#' where \eqn{c_1} is in \eqn{[0,1/2]}, \eqn{c_2>0} and \eqn{(1-c_1)^2+c_2^2 \le 1}
#' and \eqn{D_1=(B+C)/2} and \eqn{D_2=(A+C)/2} are the midpoints of edges \eqn{BC} and \eqn{AC}.
#'
#' Any given triangle can be mapped to the basic triangle by a combination of rigid body motions
#' (i.e., translation, rotation and reflection) and scaling, preserving uniformity of the points in the
#' original triangle. Hence basic triangle is useful for simulation
#' studies under the uniformity hypothesis.
#' \eqn{x}-coordinates are provided in \code{vector} \code{x}.
#'
#' @param x A single scalar or a \code{vector} of scalars.
#' @param c1,c2 Positive real numbers which constitute the vertex of the basic triangle
#' adjacent to the shorter edges; \eqn{c_1} must be in \eqn{[0,1/2]}, \eqn{c_2>0} and \eqn{(1-c_1)^2+c_2^2 \le 1}.
#'
#' @return A \code{list} with the elements
#' \item{txt1}{Longer description of the line.}
#' \item{txt2}{Shorter description of the line (to be inserted over the line in the plot).}
#' \item{mtitle}{The \code{"main"} title for the plot of the line.}
#' \item{cent}{The center chosen inside the standard equilateral triangle.}
#' \item{cent.name}{The name of the center inside the basic triangle.
#' It is \code{"CC"} for these two functions.}
#' \item{tri}{The triangle (it is the basic triangle for this function).}
#' \item{x}{The input vector, can be a scalar or a \code{vector} of scalars,
#' which constitute the \eqn{x}-coordinates of the point(s) of interest on the line.}
#' \item{y}{The output vector, will be a scalar if \code{x} is a scalar or a \code{vector} of scalars if \code{x} is a \code{vector} of scalar,
#' constitutes the \eqn{y}-coordinates of the point(s) of interest on the line.}
#' \item{slope}{Slope of the line.}
#' \item{intercept}{Intercept of the line.}
#' \item{equation}{Equation of the line.}
#'
#' @name funsTbMid2CC
NULL
#'
#' @seealso \code{\link{lA_CM.Te}}, \code{\link{lB_CM.Te}}, \code{\link{lA_M.Te}}, \code{\link{lB_M.Te}},
#' and \code{\link{lC_M.Te}}
#'
#' @rdname funsTbMid2CC
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' #Examples for lD1CCinTb
#' c1<-.4; c2<-.6;
#' A<-c(0,0); B<-c(1,0); C<-c(c1,c2);  #the vertices of the basic triangle Tb
#'
#' Tb<-rbind(A,B,C)
#'
#' xfence<-abs(A[1]-B[1])*.25 #how far to go at the lower and upper ends in the x-coordinate
#' x<-seq(min(A[1],B[1])-xfence,max(A[1],B[1])+xfence,by=.1)  #try also by=.01
#'
#' lnD1CC<-lD1CCinTb(x,c1,c2)
#' lnD1CC
#' summary(lnD1CC)
#' plot(lnD1CC)
#'
#' CC<-circ.cent.bastri(c1,c2)  #the circumcenter
#' CC
#' D1<-(B+C)/2; D2<-(A+C)/2; D3<-(A+B)/2; #midpoints of the edges
#' Ds<-rbind(D1,D2,D3)
#'
#' x1<-seq(0,1,by=.1)  #try also by=.01
#' y1<-lD1CCinTb(x1,c1,c2)$y
#'
#' Xlim<-range(Tb[,1],x1)
#' Ylim<-range(Tb[,2],y1)
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(A,pch=".",asp=1,xlab="",ylab="",axes=TRUE,xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tb)
#' L<-matrix(rep(CC,3),ncol=2,byrow=TRUE); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#'
#' txt<-rbind(Tb,CC,D1,D2,D3)
#' xc<-txt[,1]+c(-.03,.04,.03,.02,.09,-.08,0)
#' yc<-txt[,2]+c(.02,.02,.04,.08,.03,.03,-.05)
#' txt.str<-c("A","B","C","CC","D1","D2","D3")
#' text(xc,yc,txt.str)
#'
#' lines(x1,y1,type="l",lty=2)
#' text(.8,.5,"lD1CCinTb")
#'
#' c1<-.4; c2<-.6;
#' x1<-seq(0,1,by=.1)  #try also by=.01
#' lD1CCinTb(x1,c1,c2)
#'
#' @export
lD1CCinTb <- function(x,c1,c2)
{
  if (!is.vector(x))
  {stop('x must be a vector')}

  if (!is.point(c1,1) || !is.point(c2,1))
  {stop('c1 and c2 must be scalars')}

  if (c1<0 || c1>1/2 || c2<=0 || (1-c1)^2+c2^2 >1)
  {stop('c1 must be in [0,1/2], c2>0 and (1-c1)^2+c2^2 <=1')}

  A<-c(0,0); B<-c(1,0); C<-c(c1,c2);  #the vertices of the basic triangle
  Tr<-rbind(A,B,C)
  M<-circ.cent.tri(Tr)

  sl<-(1-c1)/c2
 int<--1/2*(-c2^2-c1^2-2*0+2*c1*0+1)/c2
  ln<-int+sl*x

  names(sl)<-"slope"
  names(int)<-"intercept"

  dname <-deparse(substitute(x))
  cname <-"CC"

  txt1<-"Line joining midpoint of the edge BC, D1, to CC (circumcenter) in the basic triangle Tb=T(A,B,C)=T((0,0),(1,0),(c1,c2))"
  txt2<-paste("lD1CCinTb(",dname,")",sep="")

  res<-list(
    txt1=txt1, txt2=txt2,
    cent=M, cent.name=cname,
    tri=Tr,
    x=x,
    y=ln,
    slope=sl,
   intercept=int,
    equation=paste("y=",sl,"x+",int,sep="")
  )

  class(res)<-"TriLines"
  res$call <-match.call()
  res
} #end of the function
#'
#' @rdname funsTbMid2CC
#'
#' @examples
#' #Examples for lD2CCinTb
#' c1<-.4; c2<-.6;
#' A<-c(0,0); B<-c(1,0); C<-c(c1,c2);  #the vertices of the basic triangle Tb
#'
#' Tb<-rbind(A,B,C)
#'
#' xfence<-abs(A[1]-B[1])*.25 #how far to go at the lower and upper ends in the x-coordinate
#' x<-seq(min(A[1],B[1])-xfence,max(A[1],B[1])+xfence,by=.1)  #try also by=.01
#'
#' lnD2CC<-lD2CCinTb(x,c1,c2)
#' lnD2CC
#' summary(lnD2CC)
#' plot(lnD2CC)
#'
#' CC<-circ.cent.bastri(c1,c2)  #the circumcenter
#' CC
#' D1<-(B+C)/2; D2<-(A+C)/2; D3<-(A+B)/2; #midpoints of the edges
#' Ds<-rbind(D1,D2,D3)
#'
#' x2<-seq(0,1,by=.1)  #try also by=.01
#' y2<-lD2CCinTb(x2,c1,c2)$y
#'
#' Xlim<-range(Tb[,1],x1)
#' Ylim<-range(Tb[,2],y2)
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(A,pch=".",asp=1,xlab="",ylab="",axes=TRUE,xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tb)
#' L<-matrix(rep(CC,3),ncol=2,byrow=TRUE); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#'
#' txt<-rbind(Tb,CC,D1,D2,D3)
#' xc<-txt[,1]+c(-.03,.04,.03,.02,.09,-.08,0)
#' yc<-txt[,2]+c(.02,.02,.04,.08,.03,.03,-.05)
#' txt.str<-c("A","B","C","CC","D1","D2","D3")
#' text(xc,yc,txt.str)
#'
#' lines(x2,y2,type="l",lty=2)
#' text(0,.5,"lD2CCinTb")
#'
#' @export
lD2CCinTb <- function(x,c1,c2)
{
  if (!is.vector(x))
  {stop('x must be a vector')}

  if (!is.point(c1,1) || !is.point(c2,1))
  {stop('c1 and c2 must be scalars')}

  if (c1<0 || c1>1/2 || c2<=0 || (1-c1)^2+c2^2 >1)
  {stop('c1 must be in [0,1/2], c2>0 and (1-c1)^2+c2^2 <=1')}

  A<-c(0,0); B<-c(1,0); C<-c(c1,c2);  #the vertices of the basic triangle
  Tr<-rbind(A,B,C)
  M<-circ.cent.tri(Tr)

  sl<--c1/c2
 int<--1/2*(-c2^2-c1^2+2*c1*0)/c2
  ln<-int+sl*x

  names(sl)<-"slope"
  names(int)<-"intercept"

  dname <-deparse(substitute(x))
  cname <-"CC"

  txt1<-"Line joining midpoint of the edge AC, D2, to CC (circumcenter) in the basic triangle Tb=T(A,B,C)=T((0,0),(1,0),(c1,c2))"
  txt2<-paste("lD2CCinTb(",dname,")",sep="")

  res<-list(
    txt1=txt1, txt2=txt2,
    cent=M, cent.name=cname,
    tri=Tr,
    x=x,
    y=ln,
    slope=sl,
   intercept=int,
    equation=paste("y=",sl,"x+",int,sep="")
  )

  class(res)<-"TriLines"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

#' @title The index of the \eqn{CC}-vertex region in a basic triangle that contains a point
#'
#' @description Returns the index of the vertex whose region contains point \code{p} in
#' the basic triangle \eqn{T_b=T((0,0),(1,0),(c_1,c_2))} where \eqn{c_1} is in \eqn{[0,1/2]}, \eqn{c_2>0} and \eqn{(1-c_1)^2+c_2^2 \le 1}
#' and vertex regions are based on the circumcenter \eqn{CC} of \eqn{T_b}.
#' (see the plots in the example for illustrations).
#'
#' The vertices of the basic triangle \eqn{T_b} are labeled as
#' \eqn{1=(0,0)}, \eqn{2=(1,0)},and \eqn{3=(c_1,c_2)} also according to the row number the vertex is recorded in \eqn{T_b}.
#' If the point, \code{p}, is not inside \eqn{T_b}, then the function yields \code{NA} as output.
#' The corresponding vertex region is the polygon whose interior points are closest to that vertex.
#'
#' Any given triangle can be mapped to the basic triangle
#' by a combination of rigid body motions (i.e., translation, rotation and reflection) and scaling,
#' preserving uniformity of the points in the original triangle. Hence basic triangle is useful for simulation
#' studies under the uniformity hypothesis.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:mcap2012;textual}{pcds}).
#'
#' @param p A 2D point for which \eqn{CC}-vertex region it resides in is to be determined in the
#' basic triangle \eqn{T_b}.
#' @param c1,c2 Positive real numbers which constitute the upper vertex of the basic triangle (i.e., the vertex
#' adjacent to the shorter edges of \eqn{T_b}); \eqn{c_1} must be in \eqn{[0,1/2]},
#' \eqn{c_2>0} and \eqn{(1-c_1)^2+c_2^2 \le 1}.
#'
#' @return A \code{list} with two elements
#' \item{rv}{Index of the \eqn{CC}-vertex region that contains point, \code{p} in the basic triangle \eqn{T_b}}
#' \item{tri}{The vertices of the triangle, where row number corresponds to the vertex index in \code{rv}
#' with row \eqn{1=(0,0)}, row \eqn{2=(1,0)}, and row \eqn{3=(c_1,c_2)}.}
#'
#' @seealso \code{\link{rv.triCM}}, \code{\link{rv.tri.cent}}, \code{\link{rv.triCC}},
#' \code{\link{rv.bastriCM}}, \code{\link{rv.bastri.cent}}, and \code{\link{rvTeCM}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' c1<-.4; c2<-.6;  #try also c1<-.5; c2<-.5;
#'
#' P<-c(.3,.2)
#' rv.bastriCC(P,c1,c2)
#'
#' P<-c(.6,.2)
#' rv.bastriCC(P,c1,c2)
#'
#' P<-c(.5,.4)
#' rv.bastriCC(P,c1,c2)
#'
#' P<-c(1.5,.4)
#' rv.bastriCC(P,c1,c2)
#'
#' A<-c(0,0);B<-c(1,0);C<-c(c1,c2);
#' Tb<-rbind(A,B,C)
#' CC<-circ.cent.bastri(c1,c2)  #the circumcenter
#' D1<-(B+C)/2; D2<-(A+C)/2; D3<-(A+B)/2;
#' Ds<-rbind(D1,D2,D3)
#'
#' Xlim<-range(Tb[,1])
#' Ylim<-range(Tb[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(Tb,asp=1,xlab="",ylab="",axes=TRUE,pch=".",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tb)
#' L<-matrix(rep(CC,3),ncol=2,byrow=TRUE); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#'
#' txt<-rbind(Tb,CC,Ds)
#' xc<-txt[,1]+c(-.03,.03,0.02,-.01,.05,-.05,.01)
#' yc<-txt[,2]+c(.02,.02,.03,.06,.03,.03,-.03)
#' txt.str<-c("A","B","C","CC","D1","D2","D3")
#' text(xc,yc,txt.str)
#'
#' RV1<-(A+D3+CC+D2)/4
#' RV2<-(B+D3+CC+D1)/4
#' RV3<-(C+D2+CC+D1)/4
#'
#' txt<-rbind(RV1,RV2,RV3)
#' xc<-txt[,1]
#' yc<-txt[,2]
#' txt.str<-c("rv=1","rv=2","rv=3")
#' text(xc,yc,txt.str)
#'
#' n<-10  #try also n<-20
#' dat<-runif.bastri(n,c1,c2)$g
#'
#' Rv<-vector()
#' for (i in 1:n)
#'   Rv<-c(Rv,rv.bastriCC(dat[i,],c1,c2)$rv)
#' Rv
#'
#' Xlim<-range(Tb[,1],dat[,1])
#' Ylim<-range(Tb[,2],dat[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(Tb,asp=1,xlab="",pch=".",ylab="",axes=TRUE,xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' points(dat,pch=".")
#' polygon(Tb)
#' L<-matrix(rep(CC,3),ncol=2,byrow=TRUE); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#' text(dat,labels=factor(Rv))
#'
#' txt<-rbind(Tb,CC,Ds)
#' xc<-txt[,1]+c(-.03,.03,0.02,-.01,.05,-.05,.01)
#' yc<-txt[,2]+c(.02,.02,.03,.06,.03,.03,-.04)
#' txt.str<-c("A","B","C","CC","D1","D2","D3")
#' text(xc,yc,txt.str)
#'
#' P<-c(.3,.2)
#' rv.bastriCC(P,c1,c2)
#'
#' @export rv.bastriCC
rv.bastriCC <- function(p,c1,c2)
{
  if (!is.point(p))
  {stop('p must be a numeric point of dimension 2')}

  if (!is.point(c1,1) || !is.point(c2,1))
  {stop('c1 and c2 must be scalars')}

  if (c1<0 || c1>1/2 || c2<=0 || (1-c1)^2+c2^2 >1)
  {stop('c1 must be in [0,1/2], c2>0 and (1-c1)^2+c2^2 <=1')}

  p1<-c(0,0); p2<-c(1,0); p3<-c(c1,c2);
  Tb<-rbind(p1,p2,p3)
  if (in.triangle(p,Tb,boundary=TRUE)$in.tri==F)
  {rv<-NA
  } else
  {
    x<-p[1]; y<-p[2];
    if ( y<= lD2CCinTb(x,c1,c2)$y && x<=1/2)
    {rv<-1}
    else
    {
      if (y<= lD1CCinTb(x,c1,c2)$y && x>=1/2)
      {rv<-2}
      else
      {rv<-3}
    }
  }
  row.names(Tb)<-c("vertex 1","vertex 2","vertex 3")  #vertex labeling

  list(rv=rv, #relative vertex
       tri=Tb #vertex labeling
  )
} #end of the function
#'

#################################################################

#' @title The index of the \eqn{CC}-vertex region in a triangle that contains a point
#'
#' @description Returns the index of the vertex whose region contains point \code{p} in
#' a triangle \code{tri}\eqn{=(A,B,C)} and vertex regions are based on the circumcenter \eqn{CC} of \code{tri}.
#' (see the plots in the example for illustrations).
#'
#' The vertices of the  triangle \code{tri} are labeled as
#' \eqn{1=A}, \eqn{2=B}, and \eqn{3=C} also according to the row number the vertex is recorded in \code{tri}.
#' If the point, \code{p}, is not inside \code{tri}, then the function yields \code{NA} as output.
#' The corresponding vertex region is the polygon whose interior points are closest to that vertex.
#' If \code{tri} is equilateral triangle, then \eqn{CC} and \eqn{CM} (center of mass) coincide.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:mcap2012;textual}{pcds}).
#'
#' @param p A 2D point for which \eqn{CC}-vertex region it resides in is to be determined in the
#' triangle \code{tri}.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#'
#' @return A \code{list} with two elements
#' \item{rv}{Index of the \eqn{CC}-vertex region that contains point, \code{p} in the triangle \code{tri}}
#' \item{tri}{The vertices of the triangle, where row number corresponds to the vertex index in \code{rv}.}
#'
#' @seealso \code{\link{rv.tri.cent}}, \code{\link{rv.triCM}}, \code{\link{rv.bastriCM}},
#' \code{\link{rv.bastriCC}}, \code{\link{rv.bastri.cent}}, and \code{\link{rvTeCM}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#'
#' P<-c(1.3,1.2)
#' rv.triCC(P,Tr)
#'
#' P<-c(1.8,.5)
#' rv.triCC(P,Tr)
#'
#' P<-c(1.6,1.4)
#' rv.triCC(P,Tr)
#'
#' P<-c(.5,.8)
#' rv.triCC(P,Tr)
#'
#' CC<-circ.cent.tri(Tr)  #the circumcenter
#' D1<-(B+C)/2; D2<-(A+C)/2; D3<-(A+B)/2;
#' Ds<-rbind(D1,D2,D3)
#'
#' Xlim<-range(Tr[,1],CC[1])
#' Ylim<-range(Tr[,2],CC[2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(Tr,asp=1,xlab="",ylab="",pch=".",axes=TRUE,xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tr)
#' L<-matrix(rep(CC,3),ncol=2,byrow=TRUE); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#'
#' txt<-rbind(Tr,CC,Ds)
#' xc<-txt[,1]+c(-.07,.08,.06,.12,-.1,-.1,-.09)
#' yc<-txt[,2]+c(.02,-.02,.03,.0,.02,.06,-.04)
#' txt.str<-c("A","B","C","CC","D1","D2","D3")
#' text(xc,yc,txt.str)
#'
#' RV1<-(A+.5*(D3-A)+A+.5*(D2-A))/2
#' RV2<-(B+.5*(D3-B)+B+.5*(D1-B))/2
#' RV3<-(C+.5*(D2-C)+C+.5*(D1-C))/2
#'
#' txt<-rbind(RV1,RV2,RV3)
#' xc<-txt[,1]
#' yc<-txt[,2]
#' txt.str<-c("rv=1","rv=2","rv=3")
#' text(xc,yc,txt.str)
#'
#' n<-10  #try also n<-20
#' dat<-runif.tri(n,Tr)$g
#'
#' Rv<-vector()
#' for (i in 1:n)
#'   Rv<-c(Rv,rv.triCC(dat[i,],Tr)$rv)
#' Rv
#'
#' Xlim<-range(Tr[,1],dat[,1])
#' Ylim<-range(Tr[,2],dat[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(Tr,asp=1,xlab="",ylab="",pch=".",axes=TRUE,xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tr)
#' points(dat,pch=".")
#' L<-matrix(rep(CC,3),ncol=2,byrow=TRUE); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#' text(dat,labels=factor(Rv))
#'
#' txt<-rbind(Tr,CC,Ds)
#' xc<-txt[,1]+c(-.07,.08,.06,.12,-.1,-.1,-.09)
#' yc<-txt[,2]+c(.02,-.02,.03,.0,.02,.06,-.04)
#' txt.str<-c("A","B","C","CC","D1","D2","D3")
#' text(xc,yc,txt.str)
#'
#' P<-c(1.3,1.2)
#' rv.triCC(P,Tr)
#'
#' dat.fr<-data.frame(a=Tr)
#' rv.triCC(P,dat.fr)
#'
#' @export rv.triCC
rv.triCC <- function(p,tri)
{
  if (!is.point(p))
  {stop('p must be a numeric point of dimension 2')}

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  p1<-tri[1,]; p2<-tri[2,]; p3<-tri[3,]
  if (in.triangle(p,tri)$in.tri==F)
  {rv<-NA
  } else
  {  mdt<-max(Dist(p1,p2),Dist(p1,p3),Dist(p2,p3)); #max edge length
  for (i in 1:3)
  {
    d1<-Dist(p,tri[i,]);
    if (d1<mdt)
    {mdt<-d1; rv<-i }
  }
  }
  row.names(tri)<-c("vertex 1","vertex 2","vertex 3")  #vertex labeling

  list(rv=rv, #relative vertex
       tri=tri #vertex labeling
  )
} #end of the function
#'

#################################################################

#' @title The indices of the \eqn{CC}-vertex regions in a triangle that contains the points in a give data set.
#'
#' @description Returns the indices of the vertices whose regions contain the points in data set \code{Dt} in
#' a triangle \code{tri}\eqn{=(A,B,C)} and vertex regions are based on the circumcenter \eqn{CC} of \code{tri}.
#' (see the plots in the example for illustrations).
#'
#' The vertices of the  triangle \code{tri} are labeled as
#' \eqn{1=A}, \eqn{2=B}, and \eqn{3=C} also according to the row number the vertex is recorded in \code{tri}.
#' If a point in \code{Dt} is not inside \code{tri}, then the function yields \code{NA} as output.
#' The corresponding vertex region is the polygon whose interior points are closest to that vertex.
#' If \code{tri} is equilateral triangle, then \eqn{CC} and \eqn{CM} (center of mass) coincide.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:mcap2012;textual}{pcds}).
#'
#' @param Dt A set of 2D points representing the set of data points for which indices of the vertex regions
#' containing them are to be determined.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#'
#' @return A \code{list} with two elements
#' \item{rv}{Indices (i.e., a \code{vector} of indices) of the vertices whose region contains points in \code{Dt}
#' in the triangle \code{tri}}
#' \item{tri}{The vertices of the triangle, where row number corresponds to the vertex index in \code{rv}.}
#'
#' @seealso \code{\link{rverts.triCM}}, \code{\link{rverts.tri.cent}}, and \code{\link{rverts.tri.nd}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#'
#' P<-c(.4,.2)
#' rverts.triCC(P,Tr)
#'
#' P<-c(1.8,.5)
#' rverts.triCC(P,Tr)
#'
#' P<-c(10.5,1.6)
#' rverts.triCC(P,Tr)
#'
#' n<-10  #try also n<-20
#' set.seed(1)
#' dat<-runif.tri(n,Tr)$g
#'
#' rverts.triCC(dat,Tr)
#' rverts.triCC(rbind(dat,c(2,2)),Tr)
#'
#' rv<-rverts.triCC(dat,Tr)
#' rv
#'
#' CC<-circ.cent.tri(Tr)
#' D1<-(B+C)/2; D2<-(A+C)/2; D3<-(A+B)/2;
#' Ds<-rbind(D1,D2,D3)
#'
#' Xlim<-range(Tr[,1],dat[,1],CC[1])
#' Ylim<-range(Tr[,2],dat[,2],CC[2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(Tr,pch=".",asp=1,xlab="",ylab="",axes=TRUE,xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tr)
#' points(dat,pch=".",col=1)
#' L<-matrix(rep(CC,3),ncol=2,byrow=TRUE); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#'
#' xc<-Tr[,1]
#' yc<-Tr[,2]
#' txt.str<-c("rv=1","rv=2","rv=3")
#' text(xc,yc,txt.str)
#'
#' txt<-rbind(CC,Ds)
#' xc<-txt[,1]+c(.04,.04,-.03,0)
#' yc<-txt[,2]+c(-.07,.04,.06,-.08)
#' txt.str<-c("CC","D1","D2","D3")
#' text(xc,yc,txt.str)
#'
#' text(dat,labels=factor(rv$rv))
#'
#' P<-c(1.8,.5)
#' rverts.triCC(P,Tr)
#'
#' dat.fr<-data.frame(a=dat)
#' rverts.triCC(dat.fr,Tr)
#'
#' @export rverts.triCC
rverts.triCC <- function(Dt,tri)
{
  if (!is.numeric(as.matrix(Dt)))
  {stop('Dt must be numeric')}

  if (is.point(Dt))
  { Dt<-matrix(Dt,ncol=2)
  } else
  {Dt<-as.matrix(Dt)
  if (ncol(Dt)!=2 )
  {stop('Dt must be of dimension nx2')}
  }

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  nt<-nrow(Dt)
 ind.set<-c()
  if (nt==0)
  {ind.set<-NA}
  else
  {
    for (i in 1:nt)
    {
     ind.set<-c(ind.set,rv.triCC(Dt[i,],tri)$rv)
    }
  }

  row.names(tri)<-c("vertex 1","vertex 2","vertex 3")  #vertex labeling

  list(rv=ind.set, #relative vertices
       tri=tri #vertex labeling
  )
} #end of the function
#'

#################################################################

#' @title The angle between two line segments
#'
#' @description Returns the angle in radians or degrees between two vectors or line segments at the point of
#' intersection. The line segments are \eqn{[ba]} and \eqn{[bc]} when the arguments of the function are given as \code{a,b,c}.
#' \code{radian} is a logical argument (default=\code{TRUE}) which yields the angle in radians if \code{TRUE}, and in degrees if \code{FALSE}.
#' The smaller of the angle between the line segments is provided by the function.
#'
#' @param a,b,c Three 2D points which represent the intersecting line segments \eqn{[ba]} and \eqn{[bc]}.
#' The smaller angle between these line segments is to be computed.
#' @param radian A logical argument (default=\code{TRUE}). If \code{TRUE}, the (smaller) angle between the line segments
#' \eqn{[ba]} and \eqn{[bc]} is provided in radians, else it is provided in degrees.
#'
#' @return angle in radians or degrees between the line segments \eqn{[ba]} and \eqn{[bc]}
#'
#' @seealso \code{\link{angle.str2end}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(.3,.2); B<-c(.6,.3); C<-c(1,1)
#' pts<-rbind(A,B,C)
#'
#' angle3pnts(A,B,C)
#'
#' angle3pnts(A,B,A)
#' round(angle3pnts(A,B,A),7)
#'
#' angle3pnts(A,B,C,radian=FALSE)
#'
#' #plot of the line segments
#' Xlim<-range(pts[,1])
#' Ylim<-range(pts[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' ang1<-angle3pnts(A,B,C,radian=FALSE)
#' ang2<-angle3pnts(B+c(1,0),B,C,radian=FALSE)
#'
#' sa<-angle.str2end(A,B,C,radian=FALSE)$s #small arc angles
#' ang1<-sa[1]
#' ang2<-sa[2]
#'
#' plot(pts,asp=1,pch=1,xlab="x",ylab="y",
#' xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' L<-rbind(B,B); R<-rbind(A,C)
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#' plotrix::draw.arc(B[1],B[2],radius=xd*.1,deg1=ang1,deg2=ang2)
#' txt<-rbind(A,B,C)
#' text(txt+cbind(rep(xd*.05,nrow(txt)),rep(-xd*.02,nrow(txt))),c("A","B","C"))
#'
#' text(rbind(B)+.15*xd*c(cos(pi*(ang2+ang1)/360),sin(pi*(ang2+ang1)/360)),
#' paste(round(abs(ang1-ang2),2)," degrees"))
#'
#' @export angle3pnts
angle3pnts <- function(a,b,c,radian=TRUE)
{
  if (!is.point(a) || !is.point(b) || !is.point(c) )
  {stop('all entries must be all numeric 2D points')}

  u<-a-b; v<-c-b
  rhs<-sum(u*v)/(sqrt(sum(u*u))*sqrt(sum(v*v)))

  if  (radian==T)
  {angle<-acos(rhs)  #in radians
  } else
  {
    angle<-acos(rhs)*180/pi # in degrees
  }
  angle
} #end of the function
#'

#################################################################

#' @title The angles to draw arcs between two line segments
#'
#' @description
#' Gives the two pairs of angles in radians or degrees to draw arcs between two vectors or line segments
#' for the \code{\link[plotrix]{draw.arc}} function in the \code{plotrix} package.
#' The angles are provided with respect to the \eqn{x}-axis in the coordinate system.
#' The line segments are \eqn{[ba]} and \eqn{[bc]} when the argument is given as \code{a,b,c} in the function.
#'
#' \code{radian} is a logical argument (default=\code{TRUE}) which yields the angle in radians if \code{TRUE}, and in degrees if \code{FALSE}.
#' The first pair of angles is for drawing arcs in the smaller angle between \eqn{[ba]} and \eqn{[bc]}
#' and the second pair of angles is for drawing arcs in the counter-clockwise order from \eqn{[ba]} to \eqn{[bc]}.
#'
#' @param a,b,c Three 2D points which represent the intersecting line segments \eqn{[ba]} and \eqn{[bc]}.
#' @param radian A logical argument (default=\code{TRUE}).
#' If \code{TRUE}, the smaller angle or counter-clockwise angle
#' between the line segments \eqn{[ba]} and \eqn{[bc]} is provided in radians, else it is provided in degrees.
#'
#' @return A \code{list} with two elements
#' \item{small.arc.angles}{Angles of \eqn{[ba]} and \eqn{[bc]} with the \eqn{x}-axis so that difference between them
#' is the smaller angle between \eqn{[ba]} and \eqn{[bc]} }
#' \item{ccw.arc.angles}{Angles of \eqn{[ba]} and \eqn{[bc]} with the \eqn{x}-axis so that difference between them
#' is the counter-clockwise angle between \eqn{[ba]} and \eqn{[bc]}}
#'
#' @seealso \code{\link{angle3pnts}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(.3,.2); B<-c(.6,.3); C<-c(1,1)
#'
#' pts<-rbind(A,B,C)
#'
#' Xp<-c(B[1]+max(abs(C[1]-B[1]),abs(A[1]-B[1])),0)
#'
#' angle.str2end(A,B,C)
#' angle.str2end(A,B,A)
#'
#' angle.str2end(A,B,C,radian=FALSE)
#'
#' #plot of the line segments
#' ang.rad<-angle.str2end(A,B,C,radian=TRUE); ang.rad
#' ang.deg<-angle.str2end(A,B,C,radian=FALSE); ang.deg
#' ang.deg1<-ang.deg$s; ang.deg1
#' ang.deg2<-ang.deg$c; ang.deg2
#'
#' rad<-min(Dist(A,B),Dist(B,C))
#'
#' Xlim<-range(pts[,1],Xp[1],B+Xp,B[1]+c(+rad,-rad))
#' Ylim<-range(pts[,2],B[2]+c(+rad,-rad))
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' #plot for the smaller arc
#' plot(pts,pch=1,asp=1,xlab="x",ylab="y",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' L<-rbind(B,B,B); R<-rbind(A,C,B+Xp)
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#' plotrix::draw.arc(B[1],B[2],radius=.3*rad,angle1=ang.rad$s[1],angle2=ang.rad$s[2])
#' plotrix::draw.arc(B[1],B[2],radius=.6*rad,angle1=0, angle2=ang.rad$s[1],lty=2,col=2)
#' plotrix::draw.arc(B[1],B[2],radius=.9*rad,angle1=0,angle2=ang.rad$s[2],col=3)
#' txt<-rbind(A,B,C)
#' text(txt+cbind(rep(xd*.02,nrow(txt)),rep(-xd*.02,nrow(txt))),c("A","B","C"))
#'
#' text(rbind(B)+.5*rad*c(cos(mean(ang.rad$s)),sin(mean(ang.rad$s))),
#'      paste(abs(round(ang.deg1[2]-ang.deg1[1],2))," degrees",sep=""))
#' text(rbind(B)+.6*rad*c(cos(ang.rad$s[1]/2),sin(ang.rad$s[1]/2)),paste(round(ang.deg1[1],2)),col=2)
#' text(rbind(B)+.9*rad*c(cos(ang.rad$s[2]/2),sin(ang.rad$s[2]/2)),paste(round(ang.deg1[2],2)),col=3)
#'
#' #plot for the counter-clockwise arc
#' plot(pts,pch=1,asp=1,xlab="x",ylab="y",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' L<-rbind(B,B,B); R<-rbind(A,C,B+Xp)
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#' plotrix::draw.arc(B[1],B[2],radius=.3*rad,angle1=ang.rad$c[1],angle2=ang.rad$c[2])
#' plotrix::draw.arc(B[1],B[2],radius=.6*rad,angle1=0, angle2=ang.rad$s[1],lty=2,col=2)
#' plotrix::draw.arc(B[1],B[2],radius=.9*rad,angle1=0,angle2=ang.rad$s[2],col=3)
#' txt<-pts
#' text(txt+cbind(rep(xd*.02,nrow(txt)),rep(-xd*.02,nrow(txt))),c("A","B","C"))
#'
#' text(rbind(B)+.5*rad*c(cos(mean(ang.rad$c)),sin(mean(ang.rad$c))),
#'      paste(abs(round(ang.deg2[2]-ang.deg2[1],2))," degrees",sep=""))
#' text(rbind(B)+.6*rad*c(cos(ang.rad$s[1]/2),sin(ang.rad$s[1]/2)),paste(round(ang.deg1[1],2)),col=2)
#' text(rbind(B)+.9*rad*c(cos(ang.rad$s[2]/2),sin(ang.rad$s[2]/2)),paste(round(ang.deg1[2],2)),col=3)
#'
#' @export angle.str2end
angle.str2end <- function(a,b,c,radian=TRUE)
{
  if (!is.point(a) || !is.point(b) || !is.point(c) )
  {stop('all entries must be all numeric 2D points')}

  if (isTRUE(all.equal(a,b))==T || isTRUE(all.equal(c,b))==T)
  {stop('The angle is undefined')}

  u<-a-b; v<-c-b
  a1<-atan2(u[2],u[1])
  a2<-atan2(v[2],v[1])

  if (sign(a1)!=sign(a2))
  {
    if (a2<0) {A1<-a1; A2<-2*pi+a2} else {A1<-a1; A2<-a2}
    if (abs(a1)+abs(a2)<pi)
    { st.ang<-min(a1,a2);
    end.ang<-max(a1,a2)
    } else {
      B1<-max(a1,a2)
      B2<-2*pi+min(a1,a2)
      st.ang<-min(B1,B2);
      end.ang<-max(B1,B2)
    }
  } else
  {
    if (a1>a2) {A1<-a1; A2<-a2+2*pi} else {A1<-a1; A2<-a2}
    st.ang<-min(a1,a2);
    end.ang<-max(a1,a2)
  }

  if  (radian==T)
  {angles<-c(st.ang,end.ang)  #in radians
  } else
  {
    angles<-c(st.ang,end.ang)*180/pi # in degrees
    A1<-A1*180/pi; A2<-A2*180/pi;
  }
  list(
    small.arc.angles=angles, #angles are given so that arc between [ba] and [bc] is plotted in the smaller angle
    ccw.arc.angles=c(A1,A2)  #angles are given so that arc between [ba] and [bc] is plotted in counter-clockwise order,
  )
} #end of the function
#'

#################################################################

#' @title The points of intersection of a line and a circle
#'
#' @description Returns the intersection point(s) of a line and a circle. The line is determined by the two points
#' \code{pt1} and \code{pt2} and the circle is centered at point \code{cent} and has radius \code{rad}.
#' If the circle does not intersect the line, the function yields \code{NULL};
#' if the circle intersects at only one point, it yields only that point;
#' otherwise it yields both intersection points as output.
#' When there are two intersection points, they are listed in the order of the \eqn{x}-coordinates of \code{pt1} and \code{pt2};
#' and if the \eqn{x}-coordinates of \code{pt1} and \code{pt2} are equal, intersection points are listed in the order of
#' \eqn{y}-coordinates of \code{pt1} and \code{pt2}.
#'
#' @param pt1,pt2 2D points that determine the straight line (i.e., through which the straight line passes).
#' @param cent A 2D point representing the center of the circle.
#' @param rad A positive real number representing the radius of the circle.
#'
#' @return point(s) of intersection between the circle and the line (if they do not intersect, the function
#' yields \code{NULL} as the output)
#'
#' @seealso \code{\link{int.2lines}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' P1<-c(.3,.2)*100
#' P2<-c(.6,.3)*100
#' cent<-c(1.1,1.1)*100
#' rad<-2*100
#'
#' int.circ.line(P1,P2,cent,rad)
#' int.circ.line(P2,P1,cent,rad)
#' int.circ.line(P1,P1+c(0,1),cent,rad)
#' int.circ.line(P1+c(0,1),P1,cent,rad)
#'
#' dp2l(cent,P1,P2)
#' rad2<-dp2l(cent,P1,P2)$d
#' int.circ.line(P1,P2,cent,rad2)
#' int.circ.line(P1,P2,cent,rad=.8)
#' int.circ.line(P1,P2,cent,rad=.78)
#'
#' #plot of the line and the circle
#' A<-c(.3,.2); B<-c(.6,.3); cent<-c(1,1); rad<-2 #check dp2l(cent,A,B)$dis, .3
#'
#' IPs<-int.circ.line(A,B,cent,rad)
#'
#' xr<-range(A[1],B[1],cent[1])
#' xf<-(xr[2]-xr[1])*.1 #how far to go at the lower and upper ends in the x-coordinate
#' x<-seq(xr[1]-rad-xf,xr[2]+rad+xf,l=20)  #try also l=100
#' lnAB<-Line(A,B,x)
#' y<-lnAB$y
#'
#' Xlim<-range(x,cent[1])
#' Ylim<-range(y,A[2],B[2],cent[2]-rad,cent[2]+rad)
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(rbind(A,B,cent),pch=1,asp=1,xlab="x",ylab="y",
#' xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' lines(x,y,lty=1)
#' interp::circles(cent[1],cent[2],rad)
#' IP.txt<-c()
#' if (!is.null(IPs))
#' {
#'   for (i in 1:(length(IPs)/2))
#'     IP.txt<-c(IP.txt,paste("I",i, sep = ""))
#' }
#' txt<-rbind(A,B,cent,IPs)
#' text(txt+cbind(rep(xd*.03,nrow(txt)),rep(-yd*.03,nrow(txt))),c("A","B","M",IP.txt))
#'
#' @export int.circ.line
int.circ.line <- function(pt1,pt2,cent,rad)
{
  if (!is.point(pt1) || !is.point(pt2) || !is.point(cent) )
  {stop('pt1 and pt2 and cent must be all numeric 2D points')}

  if (!is.point(rad,1) || rad<0)
  {stop('rad must be a positive scalar')}

  a<-cent[1]; b<-cent[2];r<-rad; c<-pt1[1]

  if (pt1[1]!=pt2[1])
  {
    ln<-Line(pt1,pt2,0)
    m<-as.numeric(ln$s); d<-as.numeric(ln$int)
    delta<-round(r^2*(1+m^2)-(b-m*a-d)^2,7)
    if (delta<0)
    {int.pts<-NULL
    } else if (delta ==0)
    {
      x<-(a+b*m-d*m)/(1+m^2); y<-(d+a*m+b*m^2)/(1+m^2);
     int.pts<-c(x,y)
    } else
    {
      x1<-(a+b*m-d*m+sqrt(delta))/(1+m^2);
      x2<-(a+b*m-d*m-sqrt(delta))/(1+m^2);
      y1<-(d+a*m+b*m^2+m*sqrt(delta))/(1+m^2);
      y2<-(d+a*m+b*m^2-m*sqrt(delta))/(1+m^2);
     int.pts<-rbind(c(x1,y1),c(x2,y2))

      ord1<-order(int.pts[,1])

      if (pt1[1]<pt2[1])
      {int.pts<-int.pts[ord1,]
      } else
      {int.pts<-int.pts[rev(ord1),]
      }
    }
  } else
  {
    delta<-round(r^2-(c-a)^2,7)
    if (delta<0)
    {int.pts<-NULL
    } else if (delta ==0)
    {
      x<-c; y<-b;
     int.pts<-c(x,y)
    } else
    {
      x1<-x2<-c;
      y1<-b-sqrt(delta);
      y2<-b+sqrt(delta);
     int.pts<-rbind(c(x1,y1),c(x2,y2))

      ord2<-order(int.pts[,2])

      if (pt1[2]<pt2[2])
      {int.pts<-int.pts[ord2,]
      } else
      {int.pts<-int.pts[rev(ord2),]
      }

    }
  }
 int.pts
} #end of the function
#'

#################################################################

#' @title The vertices of the Arc Slice (AS) Proximity Region in the basic triangle
#'
#' @description Returns the end points of the line segments and arc-slices that constitute the boundary of
#' AS proximity region for a point in the basic triangle \eqn{T_b=T((0,0),(1,0),(c_1,c_2))}
#' where \eqn{c_1} is in \eqn{[0,1/2]}, \eqn{c_2>0} and \eqn{(1-c_1)^2+c_2^2 \le 1}.
#'
#' Vertex regions are based on the center \code{M="CC"} for circumcenter of \eqn{T_b}; or \eqn{M=(m_1,m_2)} in Cartesian
#' coordinates or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the interior of \eqn{T_b};
#' default is \code{M="CC"} the circumcenter of \eqn{T_b}.
#' \code{rv} is the index of the vertex region \code{pt} resides, with default=\code{NULL}.
#'
#' If \code{pt} is outside \eqn{T_b}, it returns \code{NULL} for the proximity region.
#' \code{dec} is the number of decimals (default is 4) to round the barycentric coordinates when checking whether
#' the end points fall on the boundary of the triangle \eqn{T_b} or not (so as not to miss the intersection points
#' due to precision in the decimals).
#'
#' Any given triangle can be mapped to the basic triangle
#' by a combination of rigid body motions (i.e., translation, rotation and reflection) and scaling,
#' preserving uniformity of the points in the original triangle. Hence basic triangle is useful for simulation
#' studies under the uniformity hypothesis.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:mcap2012;textual}{pcds}).
#'
#' @param pt A 2D point whose AS proximity region is to be computed.
#' @param c1,c2 Positive real numbers representing the top vertex in basic triangle \eqn{T_b=T((0,0),(1,0),(c_1,c_2))},
#' \eqn{c_1} must be in \eqn{[0,1/2]}, \eqn{c_2>0} and \eqn{(1-c_1)^2+c_2^2 \le 1}.
#' @param M The center of the triangle. \code{"CC"} stands for circumcenter of the triangle \eqn{T_b} or a 2D point in Cartesian coordinates or
#' a 3D point in barycentric coordinates which serves as a center in the interior of the triangle \eqn{T_b};
#' default is \code{M="CC"} i.e., the circumcenter of \eqn{T_b}.
#' @param rv The index of the \code{M}-vertex region containing the point, either \code{1,2,3} or \code{NULL}
#' (default is \code{NULL}).
#' @param dec a positive integer the number of decimals (default is 4) to round the barycentric coordinates when checking whether
#' the end points fall on the boundary of the triangle \eqn{T_b} or not.
#'
#' @return A \code{list} with the elements
#' \item{L,R}{The end points of the line segments on the boundary of the AS proximity region.
#'  Each row in \code{L} and \code{R} constitute a line segment on the boundary.}
#' \item{Arc.Slices}{The end points of the arc-slices on the circular parts of the AS proximity region.
#' Here points in row 1 and row 2 constitute the end points of one arc-slice, points on row 3 and row 4
#' constitute the end points for the next arc-slice and so on.}
#'
#' @seealso \code{\link{NAStri}} and \code{\link{IndNASbastri}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' c1<-.4; c2<-.6  #try also c1<-.2; c2<-.2;
#' A<-c(0,0); B<-c(1,0); C<-c(c1,c2);
#' Tb<-rbind(A,B,C)
#'
#' M<-as.numeric(runif.bastri(1,c1,c2)$g)  #try also M<-c(.6,.2)
#'
#' P1<-as.numeric(runif.bastri(1,c1,c2)$g);  #try also P1<-c(.3,.2)
#' NASbastri(P1,c1,c2)  #default with M="CC"
#' NASbastri(P1,c1,c2,M)
#'
#' #or try
#' Rv<-rv.bastriCC(P1,c1,c2)$rv
#' NASbastri(P1,c1,c2,M,Rv)
#'
#' NASbastri(c(3,5),c1,c2,M)
#'
#' P2<-c(.5,.4)
#' NASbastri(P2,c1,c2,M)
#'
#' P3<-c(1.5,.4)
#' NASbastri(P3,c1,c2,M)
#'
#' if (dimension(M)==3) {M<-bary2cart(M,Tr)}
#' #need to run this when M is given in barycentric coordinates
#'
#' #plot of the NAS region
#' P1<-as.numeric(runif.bastri(1,c1,c2)$g);
#' CC<-circ.cent.bastri(c1,c2)
#'
#' if (isTRUE(all.equal(M,CC)) || identical(M,"CC"))
#' {cent<-CC
#' D1<-(B+C)/2; D2<-(A+C)/2; D3<-(A+B)/2;
#' Ds<-rbind(D1,D2,D3)
#' cent.name<-"CC"
#' rv<-rv.bastriCC(P1,c1,c2)$rv
#' } else
#' {cent<-M
#' cent.name<-"M"
#' Ds<-cp2e.bastri(c1,c2,M)
#' D1<-Ds[1,]; D2<-Ds[2,]; D3<-Ds[3,]
#' rv<-rv.bastri.cent(P1,c1,c2,M)$rv
#' }
#' RV<-Tb[rv,]
#' rad<-Dist(P1,RV)
#'
#' Int.Pts<-NASbastri(P1,c1,c2,M)
#'
#' Xlim<-range(Tb[,1],P1[1]+rad,P1[1]-rad)
#' Ylim<-range(Tb[,2],P1[2]+rad,P1[2]-rad)
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(A,pch=".",asp=1,xlab="",ylab="",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tb)
#' points(rbind(Tb,P1,rbind(Int.Pts$L,Int.Pts$R)))
#' L<-rbind(cent,cent,cent); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#' interp::circles(P1[1],P1[2],rad,lty=2)
#' L<-Int.Pts$L; R<-Int.Pts$R
#' segments(L[,1], L[,2], R[,1], R[,2], lty=1,col=2)
#' Arcs<-Int.Pts$A;
#' if (!is.null(Arcs))
#' {
#'   K<-nrow(Arcs)/2
#'   for (i in 1:K)
#'   {A1<-Arcs[2*i-1,]; A2<-Arcs[2*i,];
#'   angles<-angle.str2end(A1,P1,A2)$c
#'
#'   plotrix::draw.arc(P1[1],P1[2],rad,angle1=angles[1],angle2=angles[2],col=2)
#'   }
#' }
#'
#' #proximity region with the triangle (i.e., for labeling the vertices of the NAS)
#' IP.txt<-intpts<-c()
#' if (!is.null(Int.Pts$A))
#' {
#'  intpts<-unique(round(Int.Pts$A,7))
#'   #this part is for labeling the intersection points of the spherical
#'   for (i in 1:(length(intpts)/2))
#'     IP.txt<-c(IP.txt,paste("I",i+1, sep = ""))
#' }
#' txt<-rbind(Tb,P1,cent,intpts)
#' txt.str<-c("A","B","C","P1",cent.name,IP.txt)
#' text(txt+cbind(rep(xd*.02,nrow(txt)),rep(-xd*.03,nrow(txt))),txt.str)
#'
#' c1<-.4; c2<-.6;
#' P1<-c(.3,.2)
#' NASbastri(P1,c1,c2,M)
#'
#' @export NASbastri
NASbastri <- function(pt,c1,c2,M="CC",rv=NULL,dec=4)
{
  if (!is.point(pt))
  {stop('pt must be a numeric 2D point')}

  if (!is.point(c1,1) || !is.point(c2,1))
  {stop('c1 and c2 must be scalars')}

  if (c1<0 || c1>1/2 || c2<=0 || (1-c1)^2+c2^2 >1)
  {stop('c1 must be in [0,1/2], c2>0 and (1-c1)^2+c2^2 <=1')}

  y1<-c(0,0); y2<-c(1,0); y3<-c(c1,c2); Tb<-rbind(y1,y2,y3)

  if (identical(M,"CC"))
  {M<-circ.cent.tri(Tb)
  } else
  {
    if ( !is.point(M) && !is.point(M,3) )
    {stop('M must be the circumcenter "CC" or a numeric 2D point for Cartesian coordinates or
          3D point for barycentric coordinates')}

    if (dimension(M)==3)
    {
      M<-bary2cart(M,Tb)
    }

    if (isTRUE(all.equal(M,circ.cent.tri(Tb)))==F & in.triangle(M,Tb,boundary=FALSE)$in.tri==F)
    {stop('center is not the circumcenter or not in the interior of the triangle')}
  }

  if (is.null(rv))
  { rv<-ifelse(isTRUE(all.equal(M,circ.cent.tri(Tb)))==T,rv.bastriCC(pt,c1,c2)$rv,rv.bastri.cent(pt,c1,c2,M)$rv)  #vertex region for pt
  } else
  {  if (!is.numeric(rv) || sum(rv==c(1,2,3))!=1)
  {stop('vertex index, rv, must be 1, 2 or 3')}}

  if (!in.triangle(pt,Tb,boundary=TRUE)$in.tri)
  {reg<-list(L=NULL, R=NULL, Arc.Slices=NULL); return(reg); stop}

  Seg.LPts<-Seg.RPts<-Arc.Pts<-NULL; #segment and arc end points

  if (rv==1)
  {
    rad<-Dist(pt,y1)
    if (in.circle(y1,pt,rad,boundary = T) && in.circle(y2,pt,rad,boundary = T) && in.circle(y3,pt,rad,boundary = T))
    {
      Seg.LPts<-Tb; Seg.RPts<-rbind(y2,y3,y1)
      Arc.Pts<-NULL
    } else
    {
      pts1<-int.circ.line(y1,y2,pt,rad); pts2<-int.circ.line(y2,y3,pt,rad); pts3<-int.circ.line(y1,y3,pt,rad)

      if (length(pts1)/2>1)
      {p1<-y1; p2<-pts1[2,]
      cond<-all(round(in.triangle(p2,Tb,boundary = T)$b,dec)>=0);
      if (cond)
      {Seg.LPts<-rbind(Seg.LPts,p1); Seg.RPts<-rbind(Seg.RPts,p2)
      Arc.Pts<-rbind(Arc.Pts,p2)
      } else
      {Seg.LPts<-rbind(Seg.LPts,p1); Seg.RPts<-rbind(Seg.RPts,y2)
      }
      } else
      {Arc.Pts<-rbind(Arc.Pts,pts1,pts1)}

      if (length(pts2)/2>1)
      {p1<-pts2[1,]; p2<-pts2[2,];
      cond1<-all(round(in.triangle(p1,Tb,boundary = T)$b,dec)>=0);
      cond2<-all(round(in.triangle(p2,Tb,boundary = T)$b,dec)>=0)
      if (cond1 && cond2)
      {Seg.LPts<-rbind(Seg.LPts,p1); Seg.RPts<-rbind(Seg.RPts,p2)
      Arc.Pts<-rbind(Arc.Pts,p1,p2)
      } else
      { if (cond1) {Seg.LPts<-rbind(Seg.LPts,p1); Seg.RPts<-rbind(Seg.RPts,y3); Arc.Pts<-rbind(Arc.Pts,p1)}
        if (cond2) {Seg.LPts<-rbind(Seg.LPts,p2); Seg.RPts<-rbind(Seg.RPts,y2);Arc.Pts<-rbind(Arc.Pts,p2)}
      }
      } else
      {Arc.Pts<-rbind(Arc.Pts,pts2,pts2)}

      if (length(pts3)/2>1)
      {p1<-y1; p2<-pts3[2,]
      cond<-all(round(in.triangle(p2,Tb,boundary = T)$b,dec)>=0);
      if (cond)
      {Seg.LPts<-rbind(Seg.LPts,p1); Seg.RPts<-rbind(Seg.RPts,p2)
      Arc.Pts<-rbind(Arc.Pts,p2)
      } else
      {Seg.LPts<-rbind(Seg.LPts,p1); Seg.RPts<-rbind(Seg.RPts,y3)
      }
      } else
      {Arc.Pts<-rbind(Arc.Pts,pts3,pts3)}
    }
  } else
    if (rv==2)
    {
      rad<-Dist(pt,y2)
      if (in.circle(y1,pt,rad,boundary = T) && in.circle(y2,pt,rad,boundary = T) && in.circle(y3,pt,rad,boundary = T))
      {
        Seg.LPts<-Tb; Seg.RPts<-rbind(y2,y3,y1)
        Arc.Pts<-NULL
      } else
      {
        pts1<-int.circ.line(y2,y3,pt,rad); pts2<-int.circ.line(y3,y1,pt,rad); pts3<-int.circ.line(y2,y1,pt,rad)

        if (length(pts1)/2>1)
        {
          p1<-y2; p2<-pts1[2,]
          cond<-all(round(in.triangle(p2,Tb,boundary = T)$b,dec)>=0);
          if (cond)
          {Seg.LPts<-rbind(Seg.LPts,p1); Seg.RPts<-rbind(Seg.RPts,p2)
          Arc.Pts<-rbind(Arc.Pts,p2)
          } else
          {Seg.LPts<-rbind(Seg.LPts,p1); Seg.RPts<-rbind(Seg.RPts,y3)
          }
        } else
        {Arc.Pts<-rbind(Arc.Pts,pts1,pts1)}

        if (length(pts2)/2>1)
        { p1<-pts2[1,]; p2<-pts2[2,];
        cond1<-all(round(in.triangle(p1,Tb,boundary = T)$b,dec)>=0);
        cond2<-all(round(in.triangle(p2,Tb,boundary = T)$b,dec)>=0)
        if (cond1 && cond2)
        {Seg.LPts<-rbind(Seg.LPts,p1); Seg.RPts<-rbind(Seg.RPts,p2)
        Arc.Pts<-rbind(Arc.Pts,p1,p2)
        } else
        {if (cond1) {Seg.LPts<-rbind(Seg.LPts,p1); Seg.RPts<-rbind(Seg.RPts,y1); Arc.Pts<-rbind(Arc.Pts,p1)}
          if (cond2) {Seg.LPts<-rbind(Seg.LPts,p2); Seg.RPts<-rbind(Seg.RPts,y3); Arc.Pts<-rbind(Arc.Pts,p2)}
        }
        } else
        {Arc.Pts<-rbind(Arc.Pts,pts2,pts2)}

        if (length(pts3)/2>1)
        {p1<-y2; p2<-pts3[2,]
        cond<-all(round(in.triangle(p2,Tb,boundary = T)$b,dec)>=0);
        if (cond)
        {Seg.LPts<-rbind(Seg.LPts,p1); Seg.RPts<-rbind(Seg.RPts,p2)
        Arc.Pts<-rbind(Arc.Pts,p2)
        } else
        { Seg.LPts<-rbind(Seg.LPts,p1); Seg.RPts<-rbind(Seg.RPts,y1)
        }
        } else
        {Arc.Pts<-rbind(Arc.Pts,pts3,pts3)}
      }
    } else {
      rad<-Dist(pt,y3)
      if (in.circle(y1,pt,rad,boundary = T) && in.circle(y2,pt,rad,boundary = T) && in.circle(y3,pt,rad,boundary = T))
      {
        Seg.LPts<-Tb; Seg.RPts<-rbind(y2,y3,y1)
        Arc.Pts<-NULL
      } else
      {
        pts1<-int.circ.line(y3,y1,pt,rad); pts2<-int.circ.line(y1,y2,pt,rad); pts3<-int.circ.line(y3,y2,pt,rad)

        if (length(pts1)/2>1)
        { p1<-y3; p2<-pts1[2,]
        cond<-all(round(in.triangle(p2,Tb,boundary = T)$b,dec)>=0);
        if (cond)
        {Seg.LPts<-rbind(Seg.LPts,p1); Seg.RPts<-rbind(Seg.RPts,p2)
        Arc.Pts<-rbind(Arc.Pts,p2)
        } else
        {
          Seg.LPts<-rbind(Seg.LPts,p1); Seg.RPts<-rbind(Seg.RPts,y1)
        }
        } else
        {Arc.Pts<-rbind(Arc.Pts,pts1,pts1)}

        if (length(pts2)/2>1)
        {p1<-pts2[1,]; p2<-pts2[2,];
        cond1<-all(round(in.triangle(p1,Tb,boundary = T)$b,dec)>=0);
        cond2<-all(round(in.triangle(p2,Tb,boundary = T)$b,dec)>=0)
        if (cond1 && cond2)
        {Seg.LPts<-rbind(Seg.LPts,p1); Seg.RPts<-rbind(Seg.RPts,p2)
        Arc.Pts<-rbind(Arc.Pts,p1,p2)
        } else
        {if (cond1) {Seg.LPts<-rbind(Seg.LPts,p1); Seg.RPts<-rbind(Seg.RPts,y2); Arc.Pts<-rbind(Arc.Pts,p1)}
          if (cond2) {Seg.LPts<-rbind(Seg.LPts,y1); Seg.RPts<-rbind(Seg.RPts,p2); Arc.Pts<-rbind(Arc.Pts,p2)}
        }
        } else
        {Arc.Pts<-rbind(Arc.Pts,pts2,pts2)}

        if (length(pts3)/2>1)
        {p1<-y3; p2<-pts3[2,]
        cond<-all(round(in.triangle(p2,Tb,boundary = T)$b,dec)>=0);
        if (cond)
        {Seg.LPts<-rbind(Seg.LPts,p1); Seg.RPts<-rbind(Seg.RPts,p2)
        Arc.Pts<-rbind(Arc.Pts,p2)
        } else
        {
          Seg.LPts<-rbind(Seg.LPts,p1); Seg.RPts<-rbind(Seg.RPts,y2)
        }
        } else
        {Arc.Pts<-rbind(Arc.Pts,pts3,pts3)}
      }
    }
  list(L=Seg.LPts,
       R=Seg.RPts,
       Arc.Slices=Arc.Pts)
} #end of the function
#'

#################################################################

#' @title The index of the vertex region in a basic triangle that contains a given point
#'
#' @description Returns the index of the related vertex in the basic triangle whose region contains point \code{p}.
#' The basic triangle is \eqn{T_b=T((0,0),(1,0),(c_1,c_2))} where \eqn{c_1} is in \eqn{[0,1/2]}, \eqn{c_2>0} and \eqn{(1-c_1)^2+c_2^2 \le 1}..
#'
#' Vertex regions are based on the general center \eqn{M=(m_1,m_2)} in Cartesian coordinates or
#' \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the interior of the basic triangle \eqn{T_b}.
#' Vertices of the basic triangle \eqn{T_b} are labeled according to the row number the
#' vertex is recorded, i.e., as 1=(0,0), 2=(1,0),and \eqn{3=(c_1,c_2)}.
#'
#' If the point, \code{p}, is not inside \eqn{T_b}, then the function yields \code{NA} as output.
#' The corresponding vertex region is the polygon with the vertex, \code{M}, and projections from \code{M}
#' to the edges on the lines joining vertices and \code{M}.
#' That is, \code{rv=1} has vertices \eqn{(0,0),D_3,M,D_2}; \code{rv=2} has vertices \eqn{(1,0),D_1,M,D_3}; and
#' \eqn{rv=3} has vertices \eqn{(c_1,c_2),D_2,M,D_1} (see the illustration in the examples).
#'
#' Any given triangle can be mapped to the basic triangle
#' by a combination of rigid body motions (i.e., translation, rotation and reflection) and scaling,
#' preserving uniformity of the points in the original triangle. Hence basic triangle is useful for simulation
#' studies under the uniformity hypothesis.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:mcap2012;textual}{pcds}).
#'
#' @param p A 2D point for which \code{M}-vertex region it resides in is to be determined in the
#' basic triangle \eqn{T_b}.
#' @param c1,c2 Positive real numbers which constitute the vertex of the basic triangle
#' adjacent to the shorter edges; \eqn{c_1} must be in \eqn{[0,1/2]}, \eqn{c_2>0} and \eqn{(1-c_1)^2+c_2^2 \le 1}.
#' @param M A 2D point in Cartesian coordinates or a 3D point in barycentric coordinates
#' which serves as a center in the interior of the basic triangle.
#'
#' @return A \code{list} with two elements
#' \item{rv}{Index of the vertex whose region contains point, \code{p}; index of the vertex is the same as the row
#' number in the basic triangle, \eqn{T_b}}
#' \item{tri}{The vertices of the basic triangle, \eqn{T_b}, where row number corresponds to the vertex index \code{rv}
#' with \code{rv=1} is row \eqn{1=(0,0)}, \code{rv=2} is row \eqn{2=(1,0)}, and \eqn{rv=3} is row \eqn{3=(c_1,c_2)}.}
#'
#' @seealso \code{\link{rv.bastriCM}}, \code{\link{rv.tri.cent}}, \code{\link{rv.triCC}},
#' \code{\link{rv.bastriCC}}, \code{\link{rv.triCM}}, and \code{\link{rvTeCM}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' c1<-.4; c2<-.6
#' A<-c(0,0); B<-c(1,0); C<-c(c1,c2);
#' Tb<-rbind(A,B,C);
#' M<-c(.6,.2)
#'
#' P<-c(.4,.2)
#' rv.bastri.cent(P,c1,c2,M)
#'
#' P<-c(1.8,.5)
#' rv.bastri.cent(P,c1,c2,M)
#'
#' P<-c(.5,.26)
#' rv.bastri.cent(P,c1,c2,M)
#'
#' n<-10  #try also n<-20
#' set.seed(1)
#' dat<-runif.bastri(n,c1,c2)$g
#'
#' M<-as.numeric(runif.bastri(1,c1,c2)$g)  #try also M<-c(.6,.2)
#'
#' Rv<-vector()
#' for (i in 1:n)
#' { Rv<-c(Rv,rv.bastri.cent(dat[i,],c1,c2,M)$rv)}
#' Rv
#'
#' Ds<-cp2e.bastri(c1,c2,M)
#'
#' Xlim<-range(Tb[,1],dat[,1])
#' Ylim<-range(Tb[,2],dat[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' if (dimension(M)==3) {M<-bary2cart(M,Tb)}
#' #need to run this when M is given in barycentric coordinates
#'
#' plot(Tb,pch=".",xlab="",ylab="",axes=TRUE,
#' xlim=Xlim+xd*c(-.1,.1),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tb)
#' points(dat,pch=".",col=1)
#' L<-rbind(M,M,M); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#'
#' xc<-Tb[,1]+c(-.04,.05,.04)
#' yc<-Tb[,2]+c(.02,.02,.03)
#' txt.str<-c("rv=1","rv=2","rv=3")
#' text(xc,yc,txt.str)
#'
#' txt<-rbind(M,Ds)
#' xc<-txt[,1]+c(-.02,.04,-.03,0)
#' yc<-txt[,2]+c(-.02,.02,.02,-.03)
#' txt.str<-c("M","D1","D2","D3")
#' text(xc,yc,txt.str)
#'
#' text(dat,labels=factor(Rv))
#'
#' P<-c(.4,.2)
#' rv.bastri.cent(P,c1,c2,M)
#'
#' @export rv.bastri.cent
rv.bastri.cent <- function(p,c1,c2,M)
{
  if (!is.point(p))
  {stop('p must be a numeric 2D point')}

  if (!is.point(c1,1) || !is.point(c2,1))
  {stop('c1 and c2 must be scalars')}

  if (c1<0 || c1>1/2 || c2<=0 || (1-c1)^2+c2^2 >1)
  {stop('c1 must be in [0,1/2], c2>0 and (1-c1)^2+c2^2 <=1')}

  if (!is.point(M) && !is.point(M,3) )
  {stop('M must be a numeric 2D point for Cartesian coordinates or 3D point for barycentric coordinates')}

  y1<-c(0,0); y2<-c(1,0); y3<-c(c1,c2); Tb<-rbind(y1,y2,y3)

  if (dimension(M)==3)
  {
    M<-bary2cart(M,Tb)
  }

  if (in.triangle(M,Tb,boundary=FALSE)$in.tri==F)
  {stop('center is not in the interior of the basic triangle')}

  a1<-y1[1]; a2<-y1[2]; b1<-y2[1]; b2<-y2[2]; c1<-y3[1]; c2<-y3[2];

  if (in.triangle(p,Tb)$in.tri==F)
  {rv<-NA
  } else
  {
    Ms<-cp2e.bastri(c1,c2,M)
    M1<-Ms[1,]; M2<-Ms[2,]; M3<-Ms[3,];

    if (in.triangle(p,rbind(y1,M3,M),boundary=TRUE)$in.tri | in.triangle(p,rbind(y1,M,M2),boundary=TRUE)$in.tri)
    {rv<-1}
    else
    {
      if (in.triangle(p,rbind(M3,y2,M),boundary=TRUE)$in.tri | in.triangle(p,rbind(y2,M1,M),boundary=TRUE)$in.tri)
      {rv<-2}
      else
      {rv<-3}
    }
  }
  row.names(Tb)<-c("vertex 1","vertex 2","vertex 3")  #vertex labeling

  list(rv=rv, #relative vertex
       tri=Tb #vertex labeling
  )
} #end of the function
#'

#################################################################

#' @title Projections of a point inside the basic triangle to its edges
#'
#' @description Returns the projections from a general center \eqn{M=(m_1,m_2)} in Cartesian coordinates or \eqn{M=(\alpha,\beta,\gamma)} in
#' barycentric coordinates in the interior of the basic triangle \eqn{T_b=T((0,0),(1,0),(c_1,c_2))}
#' to the edges on the extension of the lines joining \code{M} to the vertices (see the examples for an illustration).
#' In the basic triangle \eqn{T_b}, \eqn{c_1} is in \eqn{[0,1/2]}, \eqn{c_2>0} and \eqn{(1-c_1)^2+c_2^2 \le 1}.
#'
#' Any given triangle can be mapped to the basic triangle by a combination of rigid body motions
#' (i.e., translation, rotation and reflection) and scaling, preserving uniformity of the points in the
#' original triangle. Hence basic triangle is useful for simulation studies under the uniformity hypothesis.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010;textual}{pcds}).
#'
#' @param c1,c2 Positive real numbers which constitute the vertex of the basic triangle
#' adjacent to the shorter edges; \eqn{c_1} must be in \eqn{[0,1/2]}, \eqn{c_2>0} and \eqn{(1-c_1)^2+c_2^2 \le 1}.
#' @param M A 2D point in Cartesian coordinates or a 3D point in barycentric coordinates
#' which serves as a center in the interior of the basic triangle.
#'
#' @return Three projection points (stacked row-wise) from a general center \eqn{M=(m_1,m_2)} in Cartesian coordinates
#' or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the interior of a basic triangle to the edges on
#' the extension of the lines joining \code{M} to the vertices;
#' row \eqn{i} is the projection  point into edge \eqn{i}, for \eqn{i=1,2,3}.
#'
#' @seealso \code{\link{cp2e.tri}} and \code{\link{cp2edges.nd}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' c1<-.4; c2<-.6
#' A<-c(0,0); B<-c(1,0); C<-c(c1,c2);
#' Tb<-rbind(A,B,C);
#'
#' M<-as.numeric(runif.bastri(1,c1,c2)$g)  #try also M<-c(.6,.2)
#'
#' cp2e.bastri(c1,c2,M)
#'
#' Ds<-cp2e.bastri(c1,c2,M)
#'
#' Xlim<-range(Tb[,1])
#' Ylim<-range(Tb[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' if (dimension(M)==3) {M<-bary2cart(M,Tb)}
#' #need to run this when M is given in barycentric coordinates
#'
#' plot(Tb,pch=".",xlab="",ylab="",axes=TRUE,
#' xlim=Xlim+xd*c(-.1,.1),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tb)
#' L<-rbind(M,M,M); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#' L<-rbind(M,M,M); R<-Tb
#' segments(L[,1], L[,2], R[,1], R[,2], lty=3,col=2)
#'
#' xc<-Tb[,1]+c(-.04,.05,.04)
#' yc<-Tb[,2]+c(.02,.02,.03)
#' txt.str<-c("rv=1","rv=2","rv=3")
#' text(xc,yc,txt.str)
#'
#' txt<-rbind(M,Ds)
#' xc<-txt[,1]+c(-.02,.03,-.03,0)
#' yc<-txt[,2]+c(-.02,.02,.02,-.03)
#' txt.str<-c("M","D1","D2","D3")
#' text(xc,yc,txt.str)
#'
#' cp2e.bastri(c1,c2,M)
#'
#' @export cp2e.bastri
cp2e.bastri <- function(c1,c2,M)
{
  if (!is.point(c1,1) || !is.point(c2,1))
  {stop('c1 and c2 must be scalars')}

  if (c1<0 || c1>1/2 || c2<=0 || (1-c1)^2+c2^2 >1)
  {stop('c1 must be in [0,1/2], c2>0 and (1-c1)^2+c2^2 <=1')}

  if (!is.point(M) && !is.point(M,3) )
  {stop('M must be a numeric 2D point for Cartesian coordinates or 3D point for barycentric coordinates')}

  y1<-c(0,0); y2<-c(1,0); y3<-c(c1,c2); Tb<-rbind(y1,y2,y3)

  if (dimension(M)==3)
  {
    M<-bary2cart(M,Tb)
  }

  if (in.triangle(M,Tb,boundary=FALSE)$in.tri==F)
  {stop('center, M, is not in the interior of the basic triangle')}

  m1<-M[1]; m2<-M[2]

  M11<--c2/(m2*c1-c2*m1-m2)*m1;
  M12<--m2*c2/(m2*c1-c2*m1-m2);
  M1<-c(M11,M12)
  M21 <-m2/(m2*c1-c2*m1+c2)*c1;
  M22 <-m2*c2/(m2*c1-c2*m1+c2);
  M2<-c(M21,M22)
  M31 <- -(m2*c1-c2*m1)/(-m2+c2);
  M32 <- 0;
  M3<-c(M31,M32)
  prj.pnts<-rbind(M1,M2,M3)
  row.names(prj.pnts)<-c()
  prj.pnts
} #end of the function
#'

#################################################################

#' @title The indicator for the presence of an arc from a point to another for Arc Slice Proximity Catch Digraphs
#' (AS-PCDs) - basic triangle case
#'
#' @description Returns \eqn{I(pt2 \in N_{AS}(pt1))} for points \code{pt1} and \code{pt2}, that is, returns 1 if \eqn{pt2} is in \eqn{N_{AS}(pt1)}, returns 0
#' otherwise, where \eqn{N_{AS}(x)} is the AS proximity region for point \eqn{x}.
#'
#' AS proximity region is constructed in the basic triangle \eqn{T_b=T((0,0),(1,0),(c_1,c_2))}
#' where \eqn{c_1} is in \eqn{[0,1/2]}, \eqn{c_2>0} and \eqn{(1-c_1)^2+c_2^2 \le 1}.
#'
#' Vertex regions are based on the center \code{M="CC"} for circumcenter of \eqn{T_b};
#' or \eqn{M=(m_1,m_2)} in Cartesian coordinates or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the
#' interior of \eqn{T_b}; default is \code{M="CC"} i.e., circumcenter of \eqn{T_b}.
#' \code{rv} is the index of the vertex region \code{pt1} resides, with default=\code{NULL}.
#'
#' If \code{pt1} and \code{pt2} are distinct and either of them are outside \eqn{T_b}, the function returns 0,
#' but if they are identical, then it returns 1 regardless of their locations (i.e., it allows loops).
#'
#' Any given triangle can be mapped to the basic triangle
#' by a combination of rigid body motions (i.e., translation, rotation and reflection) and scaling,
#' preserving uniformity of the points in the original triangle. Hence basic triangle is useful for simulation
#' studies under the uniformity hypothesis.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:mcap2012;textual}{pcds}).
#'
#' @param pt1 A 2D point whose AS proximity region is constructed.
#' @param pt2 A 2D point. The function determines whether \code{pt2} is inside the AS proximity region of
#' \code{pt1} or not.
#' @param c1,c2 Positive real numbers representing the top vertex in basic triangle \eqn{T_b=T((0,0),(1,0),(c_1,c_2))},
#' \eqn{c_1} must be in \eqn{[0,1/2]}, \eqn{c_2>0} and \eqn{(1-c_1)^2+c_2^2 \le 1}.
#' @param M The center of the triangle. \code{"CC"} stands for circumcenter or a 2D point in Cartesian coordinates or a 3D point in
#' barycentric coordinates which serves as a center in the interior of the triangle \eqn{T_b};
#' default is \code{M="CC"} i.e., the circumcenter of \eqn{T_b}.
#' @param rv The index of the \code{M}-vertex region in \eqn{T_b} containing the point, either \code{1,2,3} or \code{NULL}
#' (default is \code{NULL}).
#'
#' @return \eqn{I(pt2 \in N_{AS}(pt1))} for points \code{pt1} and \code{pt2}, that is, returns 1 if \eqn{pt2} is in \eqn{N_{AS}(pt1)}
#' (i.e., if there is an arc from \code{pt1} to \code{pt2}), returns 0 otherwise.
#'
#' @seealso \code{\link{IndNAStri}} and \code{\link{NAStri}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' c1<-.4; c2<-.6;
#' A<-c(0,0); B<-c(1,0); C<-c(c1,c2);
#' Tb<-rbind(A,B,C)
#'
#' M<-as.numeric(runif.bastri(1,c1,c2)$g)  #try also M<-c(.6,.2)
#'
#' P1<-as.numeric(runif.bastri(1,c1,c2)$g)
#' P2<-as.numeric(runif.bastri(1,c1,c2)$g)
#' IndNASbastri(P1,P2,c1,c2,M)
#'
#' P1<-c(.3,.2)
#' P2<-c(.6,.2)
#' IndNASbastri(P1,P2,c1,c2,M)
#'
#' #or try
#' Rv<-rv.bastriCC(P1,c1,c2)$rv
#' IndNASbastri(P1,P2,c1,c2,M,Rv)
#'
#' P1<-c(.3,.2)
#' P2<-c(.8,.2)
#' IndNASbastri(P1,P2,c1,c2,M)
#'
#' P3<-c(.5,.4)
#' IndNASbastri(P1,P3,c1,c2,M)
#'
#' P4<-c(1.5,.4)
#' IndNASbastri(P1,P4,c1,c2,M)
#' IndNASbastri(P4,P4,c1,c2,M)
#'
#' c1<-.4; c2<-.6;
#' P1<-c(.3,.2)
#' P2<-c(.6,.2)
#' IndNASbastri(P1,P2,c1,c2,M)
#'
#' @export IndNASbastri
IndNASbastri <- function(pt1,pt2,c1,c2,M="CC",rv=NULL)
{
  if (!is.point(pt1) || !is.point(pt2) )
  {stop('pt1 and pt2 must both be numeric 2D points')}

  if (!is.point(c1,1) || !is.point(c2,1))
  {stop('c1 and c2 must be scalars')}

  if (c1<0 || c1>1/2 || c2<=0 || (1-c1)^2+c2^2 >1)
  {stop('c1 must be in [0,1/2], c2>0 and (1-c1)^2+c2^2 <=1')}

  y1<-c(0,0); y2<-c(1,0); y3<-c(c1,c2); Tb<-rbind(y1,y2,y3)

  if (identical(M,"CC"))
  {M<-circ.cent.tri(Tb)
  } else
  {
    if ( !is.point(M) && !is.point(M,3) )
    {stop('M must be the circumcenter "CC" or a numeric 2D point for Cartesian coordinates or
          3D point for barycentric coordinates')}

    if (dimension(M)==3)
    {
      M<-bary2cart(M,Tb)
    }

    if (isTRUE(all.equal(M,circ.cent.tri(Tb)))==F & in.triangle(M,Tb,boundary=FALSE)$in.tri==F)
    {stop('center is not the circumcenter or not in the interior of the triangle')}
  }

  if (identical(pt1,pt2))
  {arc<-1; return(arc); stop}

  if (!in.triangle(pt1,Tb,boundary=TRUE)$in.tri || !in.triangle(pt2,Tb,boundary=TRUE)$in.tri)
  {arc<-0; return(arc); stop}

  if (is.null(rv))
  { rv<-ifelse(isTRUE(all.equal(M,circ.cent.tri(Tb)))==T,rv.bastriCC(pt1,c1,c2)$rv,rv.bastri.cent(pt1,c1,c2,M)$rv)  #vertex region for pt
  } else
  {  if (!is.numeric(rv) || sum(rv==c(1,2,3))!=1)
  {stop('vertex index, rv, must be 1, 2 or 3')}}

  X1<-pt1[1]; Y1<-pt1[2];
  X2<-pt2[1]; Y2<-pt2[2];
  arc<-0;
  if (rv==1)
  {
    if ( (Y2-Y1)^2+(X2-X1)^2 <= X1^2+Y1^2 ) {arc <-1}
  } else {
    if (rv==2)
    {
      if ( (Y2-Y1)^2+(X2-X1)^2 <= (X1-1)^2+Y1^2 ) {arc <-1}
    } else {
      if ( (Y2-Y1)^2+(X2-X1)^2 <= (X1-c1)^2+(Y1-c2)^2 ) {arc<-1}
    }}
  arc
} #end of the function
#'

#################################################################

#' @title The vertices of the Arc Slice (AS) Proximity Region in a general triangle
#'
#' @description Returns the end points of the line segments and arc-slices that constitute the boundary of AS proximity
#' region for a point in the triangle \code{tri}\eqn{=T(A,B,C)=}\code{(rv=1,rv=2,rv=3)}.
#'
#' Vertex regions are based on the center \code{M="CC"} for circumcenter of \code{tri}; or \eqn{M=(m_1,m_2)} in Cartesian coordinates
#' or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the interior of the triangle \code{tri};
#' default is \code{M="CC"} the circumcenter of \code{tri}. \code{rv} is the index of the vertex region \code{pt1} resides,
#' with default=\code{NULL}.
#'
#' If \code{pt} is outside of \code{tri}, it returns \code{NULL} for the proximity region.
#' \code{dec} is the number of decimals (default is 4) to round the barycentric coordinates when checking the points
#' fall on the boundary of the triangle \code{tri} or not (so as not to miss the intersection points due to precision
#' in the decimals).
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:mcap2012;textual}{pcds}).
#'
#' @param pt A 2D point whose AS proximity region is to be computed.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param M The center of the triangle. \code{"CC"} stands for circumcenter of the triangle \code{tri} or a 2D point in Cartesian coordinates or a 3D point in
#' barycentric coordinates which serves as a center in the interior of the triangle \code{tri};
#' default is \code{M="CC"} i.e., the circumcenter of \code{tri}.
#' @param rv Index of the \code{M}-vertex region containing the point \code{pt}, either \code{1,2,3} or
#' \code{NULL} (default is \code{NULL}).
#' @param dec a positive integer the number of decimals (default is 4) to round the
#' barycentric coordinates when checking whether
#' the end points fall on the boundary of the triangle \code{tri} or not.
#'
#' @return A \code{list} with the elements
#' \item{L,R}{End points of the line segments on the boundary of the AS proximity region.
#'  Each row in \code{L} and \code{R} constitute a line segment on the boundary.}
#' \item{Arc.Slices}{The end points of the arc-slices on the circular parts of the AS proximity region.
#' Here points in row 1 and row 2 constitute the end points of one arc-slice, points on row 3 and row 4
#' constitute the end points for the next arc-slice and so on.}
#'
#' @seealso \code{\link{NASbastri}}, \code{\link{NPEtri}}, \code{\link{NCStri}} and \code{\link{IndNAStri}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#'
#' M<-as.numeric(runif.tri(1,Tr)$g)  #try also M<-c(.6,.2)
#'
#' P1<-as.numeric(runif.tri(1,Tr)$g)  #try also P1<-c(1.3,1.2)
#' NAStri(P1,Tr,M)
#'
#' #or try
#' Rv<-rv.triCC(P1,Tr)$rv
#' NAStri(P1,Tr,M,Rv)
#'
#' NAStri(c(3,5),Tr,M)
#'
#' P2<-c(1.5,1.4)
#' NAStri(P2,Tr,M)
#'
#' P3<-c(1.5,.4)
#' NAStri(P3,Tr,M)
#'
#' if (dimension(M)==3) {M<-bary2cart(M,Tr)}
#' #need to run this when M is given in barycentric coordinates
#'
#' CC<-circ.cent.tri(Tr)  #the circumcenter
#'
#' if (isTRUE(all.equal(M,CC)) || identical(M,"CC"))
#' {cent<-CC
#' D1<-(B+C)/2; D2<-(A+C)/2; D3<-(A+B)/2;
#' Ds<-rbind(D1,D2,D3)
#' cent.name<-"CC"
#' rv<-rv.triCC(P1,Tr)$rv
#' } else
#' {cent<-M
#' cent.name<-"M"
#' Ds<-cp2e.tri(Tr,M)
#' D1<-Ds[1,]; D2<-Ds[2,]; D3<-Ds[3,]
#' rv<-rv.tri.cent(P1,Tr,M)$rv
#' }
#' RV<-Tr[rv,]
#' rad<-Dist(P1,RV)
#'
#' Int.Pts<-NAStri(P1,Tr,M)
#'
#' #plot of the NAS region
#' Xlim<-range(Tr[,1],P1[1]+rad,P1[1]-rad)
#' Ylim<-range(Tr[,2],P1[2]+rad,P1[2]-rad)
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(A,pch=".",asp=1,xlab="",ylab="",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' #asp=1 must be the case to have the arc properly placed in the figure
#' polygon(Tr)
#' points(rbind(Tr,P1,rbind(Int.Pts$L,Int.Pts$R)))
#' L<-rbind(cent,cent,cent); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#' interp::circles(P1[1],P1[2],rad,lty=2)
#' L<-Int.Pts$L; R<-Int.Pts$R
#' segments(L[,1], L[,2], R[,1], R[,2], lty=1,col=2)
#' Arcs<-Int.Pts$A;
#' if (!is.null(Arcs))
#' {
#'   K<-nrow(Arcs)/2
#'   for (i in 1:K)
#'   {A1<-Int.Pts$Arc[2*i-1,]; A2<-Int.Pts$Arc[2*i,];
#'   angles<-angle.str2end(A1,P1,A2)$c
#'
#'   test.ang1<-angles[1]+(.01)*(angles[2]-angles[1])
#'   test.Pnt<-P1+rad*c(cos(test.ang1),sin(test.ang1))
#'   if (!in.triangle(test.Pnt,Tr,boundary = T)$i) {angles<-c(min(angles),max(angles)-2*pi)}
#'   plotrix::draw.arc(P1[1],P1[2],rad,angle1=angles[1],angle2=angles[2],col=2)
#'   }
#' }
#'
#' #proximity region with the triangle (i.e., for labeling the vertices of the NAS)
#' IP.txt<-intpts<-c()
#' if (!is.null(Int.Pts$A))
#' {
#'  intpts<-unique(round(Int.Pts$A,7))
#'   #this part is for labeling the intersection points of the spherical
#'   for (i in 1:(length(intpts)/2))
#'     IP.txt<-c(IP.txt,paste("I",i+1, sep = ""))
#' }
#' txt<-rbind(Tr,P1,cent,intpts)
#' txt.str<-c("A","B","C","P1",cent.name,IP.txt)
#' text(txt+cbind(rep(xd*.02,nrow(txt)),rep(-xd*.03,nrow(txt))),txt.str)
#'
#' P1<-c(.3,.2)
#' NAStri(P1,Tr,M)
#'
#' @export NAStri
NAStri <- function(pt,tri,M="CC",rv=NULL,dec=4)
{
  if (!is.point(pt))
  {stop('pt must be a numeric 2D point')}

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  if (identical(M,"CC"))
  {M<-circ.cent.tri(tri)
  } else
  {
    if ( !is.point(M) && !is.point(M,3) )
    {stop('M must be the circumcenter "CC" or a numeric 2D point for Cartesian coordinates or
          3D point for barycentric coordinates')}

    if (dimension(M)==3)
    {
      M<-bary2cart(M,tri)
    }

    if (isTRUE(all.equal(M,circ.cent.tri(tri)))==F & in.triangle(M,tri,boundary=FALSE)$in.tri==F)
    {stop('center is not the circumcenter or not in the interior of the triangle')}
  }

  if (!in.triangle(pt,tri,boundary=TRUE)$in.tri)
  {reg<-list(L=NULL, R=NULL, Arc.Slices=NULL); return(reg); stop}

  Tr<-tri[order(tri[,1]),] #order the vertices according to their x axis, so that angles for the
  ifelse(Tr[2,2]>Tr[3,2],tri<-Tr[c(1,3,2),],tri<-Tr)   #arcs are provided in counter-clockwise

  if (is.null(rv))
  { rv<-ifelse(isTRUE(all.equal(M,circ.cent.tri(tri)))==T,rv.triCC(pt,tri)$rv,rv.tri.cent(pt,tri,M)$rv)  #vertex region for pt
  } else
  {  if (!is.numeric(rv) || sum(rv==c(1,2,3))!=1)
  {stop('vertex index, rv, must be 1, 2 or 3')}}

  y1<-tri[1,]; y2<-tri[2,]; y3<-tri[3,];

  Seg.LPts<-Seg.RPts<-Arc.Pts<-NULL; #segment and arc end points

  if (rv==1)
  {
    rad<-Dist(pt,y1)
    if (in.circle(y1,pt,rad,boundary = T) && in.circle(y2,pt,rad,boundary = T) && in.circle(y3,pt,rad,boundary = T))
    {
      Seg.LPts<-tri; Seg.RPts<-rbind(y2,y3,y1)
      Arc.Pts<-NULL
    } else
    {
      pts1<-int.circ.line(y1,y2,pt,rad); pts2<-int.circ.line(y2,y3,pt,rad); pts3<-int.circ.line(y1,y3,pt,rad)

      if (length(pts1)/2>1)
      {p1<-y1; p2<-pts1[2,]
      cond<-all(round(in.triangle(p2,tri,boundary = T)$b,dec)>=0);
      if (cond)
      {Seg.LPts<-rbind(Seg.LPts,p1); Seg.RPts<-rbind(Seg.RPts,p2)
      Arc.Pts<-rbind(Arc.Pts,p2)
      } else
      {Seg.LPts<-rbind(Seg.LPts,p1); Seg.RPts<-rbind(Seg.RPts,y2)
      }
      } else
      {Arc.Pts<-rbind(Arc.Pts,pts1,pts1)}

      if (length(pts2)/2>1)
      {p1<-pts2[1,]; p2<-pts2[2,];
      cond1<-all(round(in.triangle(p1,tri,boundary = T)$b,dec)>=0);
      cond2<-all(round(in.triangle(p2,tri,boundary = T)$b,dec)>=0)
      if (cond1 && cond2)
      {Seg.LPts<-rbind(Seg.LPts,p1); Seg.RPts<-rbind(Seg.RPts,p2)
      Arc.Pts<-rbind(Arc.Pts,p1,p2)
      } else
      { if (cond1) {Seg.LPts<-rbind(Seg.LPts,p1); Seg.RPts<-rbind(Seg.RPts,y3); Arc.Pts<-rbind(Arc.Pts,p1)}
        if (cond2) {Seg.LPts<-rbind(Seg.LPts,p2); Seg.RPts<-rbind(Seg.RPts,y2); Arc.Pts<-rbind(Arc.Pts,p2)}
      }
      } else
      {Arc.Pts<-rbind(Arc.Pts,pts2,pts2)}

      if (length(pts3)/2>1)
      {p1<-y1; p2<-pts3[2,]
      cond<-all(round(in.triangle(p2,tri,boundary = T)$b,dec)>=0);
      if (cond)
      {Seg.LPts<-rbind(Seg.LPts,p1); Seg.RPts<-rbind(Seg.RPts,p2)
      Arc.Pts<-rbind(Arc.Pts,p2)
      } else
      {Seg.LPts<-rbind(Seg.LPts,p1); Seg.RPts<-rbind(Seg.RPts,y3)
      }
      } else
      {Arc.Pts<-rbind(Arc.Pts,pts3,pts3)}
    }
  } else
    if (rv==2)
    {
      rad<-Dist(pt,y2)
      if (in.circle(y1,pt,rad,boundary = T) && in.circle(y2,pt,rad,boundary = T) && in.circle(y3,pt,rad,boundary = T))
      {
        Seg.LPts<-tri; Seg.RPts<-rbind(y2,y3,y1)
        Arc.Pts<-NULL
      } else
      {
        pts1<-int.circ.line(y2,y3,pt,rad); pts2<-int.circ.line(y3,y1,pt,rad); pts3<-int.circ.line(y2,y1,pt,rad)

        if (length(pts1)/2>1)
        {
          p1<-y2; p2<-pts1[2,]
          cond<-all(round(in.triangle(p2,tri,boundary = T)$b,dec)>=0);
          if (cond)
          {Seg.LPts<-rbind(Seg.LPts,p1); Seg.RPts<-rbind(Seg.RPts,p2)
          Arc.Pts<-rbind(Arc.Pts,p2)
          } else
          {Seg.LPts<-rbind(Seg.LPts,p1); Seg.RPts<-rbind(Seg.RPts,y3)
          }
        } else
        {Arc.Pts<-rbind(Arc.Pts,pts1,pts1)}

        if (length(pts2)/2>1)
        { p1<-pts2[1,]; p2<-pts2[2,];
        cond1<-all(round(in.triangle(p1,tri,boundary = T)$b,dec)>=0);
        cond2<-all(round(in.triangle(p2,tri,boundary = T)$b,dec)>=0)
        if (cond1 && cond2)
        {Seg.LPts<-rbind(Seg.LPts,p1); Seg.RPts<-rbind(Seg.RPts,p2)
        Arc.Pts<-rbind(Arc.Pts,p1,p2)
        } else
        {if (cond1) {Seg.LPts<-rbind(Seg.LPts,p1); Seg.RPts<-rbind(Seg.RPts,y1); Arc.Pts<-rbind(Arc.Pts,p1)}
          if (cond2) {Seg.LPts<-rbind(Seg.LPts,p2); Seg.RPts<-rbind(Seg.RPts,y3); Arc.Pts<-rbind(Arc.Pts,p2)}
        }
        } else
        {Arc.Pts<-rbind(Arc.Pts,pts2,pts2)}

        if (length(pts3)/2>1)
        {p1<-y2; p2<-pts3[2,]
        cond<-all(round(in.triangle(p2,tri,boundary = T)$b,dec)>=0);
        if (cond)
        {Seg.LPts<-rbind(Seg.LPts,p1); Seg.RPts<-rbind(Seg.RPts,p2)
        Arc.Pts<-rbind(Arc.Pts,p2)
        } else
        { Seg.LPts<-rbind(Seg.LPts,p1); Seg.RPts<-rbind(Seg.RPts,y1)
        }
        } else
        {Arc.Pts<-rbind(Arc.Pts,pts3,pts3)}
      }
    } else {
      rad<-Dist(pt,y3)
      if (in.circle(y1,pt,rad,boundary = T) && in.circle(y2,pt,rad,boundary = T) && in.circle(y3,pt,rad,boundary = T))
      {
        Seg.LPts<-tri; Seg.RPts<-rbind(y2,y3,y1)
        Arc.Pts<-NULL
      } else
      {
        pts1<-int.circ.line(y3,y1,pt,rad); pts2<-int.circ.line(y1,y2,pt,rad); pts3<-int.circ.line(y3,y2,pt,rad)

        if (length(pts1)/2>1)
        { p1<-y3; p2<-pts1[2,]
        cond<-all(round(in.triangle(p2,tri,boundary = T)$b,dec)>=0);
        if (cond)
        {Seg.LPts<-rbind(Seg.LPts,p1); Seg.RPts<-rbind(Seg.RPts,p2)
        Arc.Pts<-rbind(Arc.Pts,p2)
        } else
        {
          Seg.LPts<-rbind(Seg.LPts,p1); Seg.RPts<-rbind(Seg.RPts,y1)
        }
        } else
        {Arc.Pts<-rbind(Arc.Pts,pts1,pts1)}

        if (length(pts2)/2>1)
        {p1<-pts2[1,]; p2<-pts2[2,];
        cond1<-all(round(in.triangle(p1,tri,boundary = T)$b,dec)>=0);
        cond2<-all(round(in.triangle(p2,tri,boundary = T)$b,dec)>=0)
        if (cond1 && cond2)
        {Seg.LPts<-rbind(Seg.LPts,p1); Seg.RPts<-rbind(Seg.RPts,p2)
        Arc.Pts<-rbind(Arc.Pts,p1,p2)
        } else
        {if (cond1) {Seg.LPts<-rbind(Seg.LPts,p1); Seg.RPts<-rbind(Seg.RPts,y2); Arc.Pts<-rbind(Arc.Pts,p1)}
          if (cond2) {Seg.LPts<-rbind(Seg.LPts,y1); Seg.RPts<-rbind(Seg.RPts,p2); Arc.Pts<-rbind(Arc.Pts,p2)}
        }
        } else
        {Arc.Pts<-rbind(Arc.Pts,pts2,pts2)}

        if (length(pts3)/2>1)
        {p1<-y3; p2<-pts3[2,]
        cond<-all(round(in.triangle(p2,tri,boundary = T)$b,dec)>=0);
        if (cond)
        {Seg.LPts<-rbind(Seg.LPts,p1); Seg.RPts<-rbind(Seg.RPts,p2)
        Arc.Pts<-rbind(Arc.Pts,p2)
        } else
        {
          Seg.LPts<-rbind(Seg.LPts,p1); Seg.RPts<-rbind(Seg.RPts,y2)
        }
        } else
        {Arc.Pts<-rbind(Arc.Pts,pts3,pts3)}
      }
    }
  list(L=Seg.LPts,
       R=Seg.RPts,
       Arc.Slices=Arc.Pts)
} #end of the function
#'

#################################################################

#' @title The indicator for the presence of an arc from a point to another for Arc Slice Proximity Catch Digraphs
#' (AS-PCDs) - one triangle case
#'
#' @description Returns \eqn{I(pt2 \in N_{AS}(pt1))} for points \code{pt1} and \code{pt2}, that is, returns 1 if \eqn{pt2} is in \eqn{N_{AS}(pt1)}, returns 0
#' otherwise, where \eqn{N_{AS}(x)} is the AS proximity region for point \eqn{x}.
#'
#' AS proximity regions are constructed with respect to the triangle, \code{tri}\eqn{=T(A,B,C)=}\code{(rv=1,rv=2,rv=3)},
#' and vertex regions are based on the center \code{M="CC"} for circumcenter of \code{tri};
#' or \eqn{M=(m_1,m_2)} in Cartesian coordinates or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the
#' interior of the triangle \code{tri}; default is \code{M="CC"} i.e., circumcenter of \code{tri}.
#' \code{rv} is the index of the vertex region \code{pt1} resides, with default=\code{NULL}.
#'
#' If \code{pt1} and \code{pt2} are distinct and either of them are outside \code{tri}, the function returns 0,
#' but if they are identical, then it returns 1 regardless of their locations (i.e., it allows loops).
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:mcap2012;textual}{pcds}).
#'
#' @param pt1 A 2D point whose AS proximity region is constructed.
#' @param pt2 A 2D point. The function determines whether \code{pt2} is inside the AS proximity region of
#' \code{pt1} or not.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param M The center of the triangle. \code{"CC"} stands for circumcenter of the triangle \code{tri} or a 2D point in Cartesian coordinates or
#' a 3D point in barycentric coordinates which serves as a center in the interior of \code{tri};
#' default is \code{M="CC"} i.e., the circumcenter of \code{tri}.
#' @param rv The index of the \code{M}-vertex region in \code{tri} containing the point, either \code{1,2,3} or \code{NULL}
#' (default is \code{NULL}).
#'
#' @return \eqn{I(pt2 \in N_{AS}(pt1))} for \code{pt1}, that is, returns 1 if \eqn{pt2} is in \eqn{N_{AS}(pt1)}, returns 0 otherwise
#'
#' @seealso \code{\link{IndNASbastri}}, \code{\link{IndNPEtri}}, and \code{\link{IndNCStri}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#'
#' Tr<-rbind(A,B,C);
#'
#' M<-as.numeric(runif.tri(1,Tr)$g)  #try also M<-c(1.6,1.2)
#'
#' P1<-as.numeric(runif.tri(1,Tr)$g)
#' P2<-as.numeric(runif.tri(1,Tr)$g)
#' IndNAStri(P1,P2,Tr,M)
#'
#' P1<-c(1.3,1.2)
#' P2<-c(1.8,.5)
#' IndNAStri(P1,P2,Tr,M)
#' IndNAStri(P1,P1,Tr,M)
#'
#' #or try
#' Rv<-rv.triCC(P1,Tr)$rv
#' IndNAStri(P1,P2,Tr,M,Rv)
#'
#' P3<-c(1.6,1.4)
#' IndNAStri(P1,P3,Tr,M)
#'
#' P4<-c(1.5,1.0)
#' IndNAStri(P1,P4,Tr,M)
#'
#' P5<-c(.5,1.0)
#' IndNAStri(P1,P5,Tr,M)
#' IndNAStri(P5,P5,Tr,M)
#'
#' #or try
#' Rv<-rv.triCC(P5,Tr)$rv
#' IndNAStri(P5,P5,Tr,M,Rv)
#'
#' dat.fr<-data.frame(a=Tr)
#' IndNAStri(P1,P2,dat.fr,M)
#'
#' @export IndNAStri
IndNAStri <- function(pt1,pt2,tri,M="CC",rv=NULL)
{
  if (!is.point(pt1) || !is.point(pt2) )
  {stop('pt1 and pt2 must both be numeric 2D points')}

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  if (identical(M,"CC"))
  {M<-circ.cent.tri(tri)
  } else
  {
    if ( !is.point(M) && !is.point(M,3) )
    {stop('M must be the circumcenter "CC" or a numeric 2D point for Cartesian coordinates or
          3D point for barycentric coordinates')}

    if (dimension(M)==3)
    {
      M<-bary2cart(M,tri)
    }

    if (isTRUE(all.equal(M,circ.cent.tri(tri)))==F & in.triangle(M,tri,boundary=FALSE)$in.tri==F)
    {stop('center is not the circumcenter or not in the interior of the triangle')}
  }

  if (identical(pt1,pt2))
  {arc<-1; return(arc); stop}

  if (!in.triangle(pt1,tri,boundary=TRUE)$in.tri || !in.triangle(pt2,tri,boundary=TRUE)$in.tri)
  {arc<-0; return(arc); stop}

  if (is.null(rv))
  { rv<-ifelse(isTRUE(all.equal(M,circ.cent.tri(tri)))==T,rv.triCC(pt1,tri)$rv,rv.tri.cent(pt1,tri,M)$rv)  #vertex region for pt
  } else
  {  if (!is.numeric(rv) || sum(rv==c(1,2,3))!=1)
  {stop('vertex index, rv, must be 1, 2 or 3')}}

  X1<-pt1[1]; Y1<-pt1[2]; X2<-pt2[1]; Y2<-pt2[2];
  a1<-tri[1,1]; a2<-tri[1,2]; b1<-tri[2,1]; b2<-tri[2,2]; c1<-tri[3,1]; c2<-tri[3,2];
  arc<-0;
  if (rv==1)
  {
    if ( (Y2-Y1)^2+(X2-X1)^2 < (X1-a1)^2+(Y1-a2)^2 ) {arc <-1}
  } else {
    if (rv==2)
    {
      if ( (Y2-Y1)^2+(X2-X1)^2 < (X1-b1)^2+(Y1-b2)^2 ) {arc <-1}
    } else {
      if ( (Y2-Y1)^2+(X2-X1)^2 < (X1-c1)^2+(Y1-c2)^2 ) {arc<-1}
    }}
  arc
} #end of the function
#'

#################################################################

#' @title The index of the vertex region in a triangle that contains a given point
#'
#' @description Returns the index of the related vertex in the triangle, \code{tri}, whose region contains point \code{p}.
#'
#' Vertex regions are based on the general center \eqn{M=(m_1,m_2)} in Cartesian coordinates or
#' \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the interior of the triangle \code{tri}.
#' Vertices of the triangle \code{tri} are labeled according to the row number the vertex is recorded.
#'
#' If the point, \code{p}, is not inside \code{tri}, then the function yields \code{NA} as output.
#' The corresponding vertex region is the polygon with the vertex, \code{M}, and projections from \code{M}
#' to the edges on the lines joining vertices and \code{M}. (see the illustration in the examples).
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:mcap2012;textual}{pcds}).
#'
#' @param p A 2D point for which \code{M}-vertex region it resides in is to be determined in the
#' triangle \code{tri}.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param M A 2D point in Cartesian coordinates or a 3D point in barycentric coordinates
#' which serves as a center in the interior of the triangle \code{tri}.
#'
#' @return A \code{list} with two elements
#' \item{rv}{Index of the vertex whose region contains point, \code{p}; index of the vertex is the same as the row
#' number in the triangle, \code{tri}}
#' \item{tri}{The vertices of the triangle, \code{tri}, where row number corresponds to the vertex index \code{rv}
#' with \code{rv=1} is row 1, \code{rv=2} is row 2, and \eqn{rv=3} is is row 3.}
#'
#' @seealso \code{\link{rv.triCM}}, \code{\link{rv.triCC}}, \code{\link{rv.bastriCC}},
#' \code{\link{rv.bastriCM}}, \code{\link{rv.bastri.cent}}, and \code{\link{rvTeCM}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#' M<-c(1.6,1.0)
#'
#' P<-c(.4,.2)
#' rv.tri.cent(P,Tr,M)
#'
#' P<-c(1.8,.5)
#' rv.tri.cent(P,Tr,M)
#'
#' P<-c(1.5,1.6)
#' rv.tri.cent(P,Tr,M)
#' #try also rv.tri.cent(P,Tr,M=c(2,2))  #center is not in the interior of the triangle
#'
#' n<-10  #try also n<-20
#' set.seed(1)
#' dat<-runif.tri(n,Tr)$g
#'
#' M<-as.numeric(runif.tri(1,Tr)$g)  #try also M<-c(1.6,1.0)
#'
#' Rv<-vector()
#' for (i in 1:n)
#' {Rv<-c(Rv,rv.tri.cent(dat[i,],Tr,M)$rv)}
#' Rv
#'
#' Ds<-cp2e.tri(Tr,M)
#'
#' Xlim<-range(Tr[,1],dat[,1])
#' Ylim<-range(Tr[,2],dat[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' if (dimension(M)==3) {M<-bary2cart(M,Tr)}
#' #need to run this when M is given in barycentric coordinates
#'
#' plot(Tr,pch=".",xlab="",ylab="",axes=TRUE,
#' xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tr)
#' points(dat,pch=".",col=1)
#' L<-rbind(M,M,M); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#'
#' xc<-Tr[,1]
#' yc<-Tr[,2]
#' txt.str<-c("rv=1","rv=2","rv=3")
#' text(xc,yc,txt.str)
#'
#' txt<-rbind(M,Ds)
#' xc<-txt[,1]+c(-.02,.04,-.04,0)
#' yc<-txt[,2]+c(-.02,.04,.05,-.08)
#' txt.str<-c("M","D1","D2","D3")
#' text(xc,yc,txt.str)
#'
#' text(dat,labels=factor(Rv))
#'
#' P<-c(1.8,.5)
#' rv.tri.cent(P,Tr,M)
#'
#' dat.fr<-data.frame(a=Tr)
#' rv.tri.cent(P,dat.fr,M)
#'
#' @export rv.tri.cent
rv.tri.cent <- function(p,tri,M)
{
  if (!is.point(p))
  {stop('p must be a numeric 2D point')}

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  if (!is.point(M) && !is.point(M,3) )
  {stop('M must be a numeric 2D point for Cartesian coordinates or 3D point for barycentric coordinates')}

  if (dimension(M)==3)
  {
    M<-bary2cart(M,tri)
  }

  y1<-tri[1,]; y2<-tri[2,]; y3<-tri[3,];
  a1<-y1[1]; a2<-y1[2]; b1<-y2[1]; b2<-y2[2]; c1<-y3[1]; c2<-y3[2];

  if (in.triangle(p,tri)$in.tri==F)
  {rv<-NA
  } else
  {
    if (in.triangle(M,tri,boundary=FALSE)$in.tri==F)
    {stop('center is not in the interior of the triangle')}

    Ds<-cp2e.tri(tri,M)
    D1<-Ds[1,]; D2<-Ds[2,]; D3<-Ds[3,];

    if (in.triangle(p,rbind(y1,D3,M),boundary=TRUE)$in.tri | in.triangle(p,rbind(y1,M,D2),boundary=TRUE)$in.tri)
    {rv<-1}
    else
    {
      if (in.triangle(p,rbind(D3,y2,M),boundary=TRUE)$in.tri | in.triangle(p,rbind(y2,D1,M),boundary=TRUE)$in.tri)
      {rv<-2}
      else
      {rv<-3}
    }
  }
  row.names(tri)<-c("vertex 1","vertex 2","vertex 3")  #vertex labeling

  list(rv=rv, #relative vertex
       tri=tri #vertex labeling
  )
} #end of the function
#'

#################################################################

#' @title Projections of a point inside a triangle to its edges
#'
#' @description Returns the projections from a general center \eqn{M=(m_1,m_2)} in Cartesian coordinates or \eqn{M=(\alpha,\beta,\gamma)} in
#' barycentric coordinates in the interior of a triangle to the edges on the extension of the lines joining
#' \code{M} to the vertices (see the examples for an illustration).
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:mcap2012;textual}{pcds}).
#'
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param M A 2D point in Cartesian coordinates or a 3D point in barycentric coordinates
#' which serves as a center in the interior of the triangle \code{tri}.
#'
#' @return Three projection points (stacked row-wise) from a general center \eqn{M=(m_1,m_2)} in Cartesian coordinates
#' or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the interior of a triangle to the edges on
#' the extension of the lines joining \code{M} to the vertices;
#' row \eqn{i} is the projection  point into edge \eqn{i}, for \eqn{i=1,2,3}.
#'
#' @seealso \code{\link{cp2e.bastri}} and \code{\link{cp2edges.nd}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#'
#' M<-as.numeric(runif.tri(1,Tr)$g)  #try also M<-c(1.6,1.0)
#'
#' cp2e.tri(Tr,M)  #try also cp2e.tri(Tr,M=c(1,1))
#'
#' Ds<-cp2e.tri(Tr,M)
#'
#' Xlim<-range(Tr[,1])
#' Ylim<-range(Tr[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' if (dimension(M)==3) {M<-bary2cart(M,Tr)}
#' #need to run this when M is given in barycentric coordinates
#'
#' plot(Tr,pch=".",xlab="",ylab="",axes=TRUE,
#' xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tr)
#' L<-rbind(M,M,M); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#'
#' xc<-Tr[,1]
#' yc<-Tr[,2]
#' txt.str<-c("rv=1","rv=2","rv=3")
#' text(xc,yc,txt.str)
#'
#' txt<-rbind(M,Ds)
#' xc<-txt[,1]+c(-.02,.04,-.04,-.02)
#' yc<-txt[,2]+c(-.02,.04,.04,-.06)
#' txt.str<-c("M","D1","D2","D3")
#' text(xc,yc,txt.str)
#'
#' cp2e.tri(Tr,M)
#'
#' dat.fr<-data.frame(a=Tr)
#' cp2e.tri(dat.fr,M)
#'
#' @export cp2e.tri
cp2e.tri <- function(tri,M)
{
  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  if (!is.point(M) && !is.point(M,3) )
  {stop('M must be a numeric 2D point for Cartesian coordinates or 3D point for barycentric coordinates')}

  if (dimension(M)==3)
  {
    M<-bary2cart(M,tri)
  }

  y1<-tri[1,]; y2<-tri[2,]; y3<-tri[3,];
  a1<-y1[1]; a2<-y1[2]; b1<-y2[1]; b2<-y2[2]; c1<-y3[1]; c2<-y3[2];

  if (in.triangle(M,tri,boundary=FALSE)$in.tri==F)
  {stop('center is not in the interior of the triangle')}

  m1<-M[1]; m2<-M[2]

  M11<--(a1*b1*c2-a1*b1*m2-a1*b2*c1+a1*c1*m2+a2*b1*m1-a2*c1*m1-b1*c2*m1+b2*c1*m1)/(b2*a1-c2*a1-a2*b1+a2*c1+m2*b1-b2*m1-m2*c1+c2*m1);
  M12<-(a1*b2*m2-a1*c2*m2-a2*b1*c2+a2*b2*c1-a2*b2*m1+a2*c2*m1+b1*c2*m2-b2*c1*m2)/(b2*a1-c2*a1-a2*b1+a2*c1+m2*b1-b2*m1-m2*c1+c2*m1);
  D1<-c(M11,M12)
  M21 <-(a1*b1*c2-a1*b1*m2+a1*b2*m1-a1*c2*m1-a2*b1*c1+a2*c1*m1+b1*c1*m2-b2*c1*m1)/(b2*a1-m2*a1-a2*b1+m1*a2+c2*b1-c1*b2+m2*c1-c2*m1);
  M22 <-(a1*b2*c2-a1*c2*m2-a2*b1*m2-a2*b2*c1+a2*b2*m1+a2*c1*m2+b1*c2*m2-b2*c2*m1)/(b2*a1-m2*a1-a2*b1+m1*a2+c2*b1-c1*b2+m2*c1-c2*m1);
  D2<-c(M21,M22)
  M31 <- (a1*b2*c1-a1*b2*m1-a1*c1*m2+a1*c2*m1-a2*b1*c1+a2*b1*m1+b1*c1*m2-b1*c2*m1)/(c2*a1-m2*a1-a2*c1+m1*a2-c2*b1+m2*b1+c1*b2-b2*m1);
  M32 <- (a1*b2*c2-a1*b2*m2-a2*b1*c2+a2*b1*m2-a2*c1*m2+a2*c2*m1+b2*c1*m2-b2*c2*m1)/(c2*a1-m2*a1-a2*c1+m1*a2-c2*b1+m2*b1+c1*b2-b2*m1);
  D3<-c(M31,M32)

  Ds<-rbind(D1,D2,D3)
  row.names(Ds)<-c()

  Ds
} #end of the function
#'

#################################################################

#' @title Number of arcs of Arc Slice Proximity Catch Digraphs (AS-PCDs) - one triangle case
#'
#' @description Returns the number of arcs of Arc Slice Proximity Catch Digraphs (AS-PCDs)
#' whose vertices are the 2D data set, \code{dat}. The data points could be inside or outside a general
#' triangle \code{tri}\eqn{=T(A,B,C)=}\code{(rv=1,rv=2,rv=3)}, with vertices of \code{tri} stacked row-wise.
#'
#' AS proximity regions are defined with respect to the triangle \code{tri} and vertex regions are
#' based on the center \code{M="CC"} for circumcenter of \code{tri};
#' or \eqn{M=(m_1,m_2)} in Cartesian coordinates or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the
#' interior of the triangle \code{tri}; default is \code{M="CC"} i.e., circumcenter of \code{tri}.
#' For the number of arcs, loops are not allowed,
#' so arcs are only possible for points inside the triangle, \code{tri}.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:mcap2012;textual}{pcds}).
#'
#' @param dat A set of 2D points which constitute the vertices of the digraph (i.e., AS-PCD).
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param M The center of the triangle. \code{"CC"} stands for circumcenter of the triangle \code{tri} or a 2D point in Cartesian coordinates or
#' a 3D point in barycentric coordinates which serves as a center in the interior of \code{tri};
#' default is \code{M="CC"} i.e., the circumcenter of \code{tri}.
#'
#' @return The number of arcs of Arc Slice Proximity Catch Digraphs (AS-PCDs)
#'
#' @seealso \code{\link{NumArcsASMT}}, \code{\link{NumArcsPEtri}}, and \code{\link{NumArcsCStri}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#'
#' n<-10  #try also n<-20
#' set.seed(1)
#' dat<-runif.tri(n,Tr)$g
#'
#' M<-as.numeric(runif.tri(1,Tr)$g)  #try also M<-c(1.6,1.2)
#'
#' NumArcsAStri(dat,Tr)  #with default M="CC"
#' NumArcsAStri(dat,Tr,M)
#'
#' NumArcsAStri(rbind(dat,c(0,2)),Tr,M)
#'
#' dat.fr<-data.frame(a=dat)
#' NumArcsAStri(dat.fr,Tr,M)
#'
#' @export NumArcsAStri
NumArcsAStri <- function(dat,tri,M="CC")
{
  if (!is.numeric(as.matrix(dat)))
  {stop('dat must be numeric')}

  if (is.point(dat))
  { dat<-matrix(dat,ncol=2)
  } else
  {dat<-as.matrix(dat)
  if (ncol(dat)!=2 )
  {stop('dat must be of dimension nx2')}
  }

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  if (identical(M,"CC"))
  {M<-circ.cent.tri(tri)
  } else
  {
    if ( !is.point(M) && !is.point(M,3) )
    {stop('M must be the circumcenter "CC" or a numeric 2D point for Cartesian coordinates or
          3D point for barycentric coordinates')}

    if (dimension(M)==3)
    {
      M<-bary2cart(M,tri)
    }

    if (isTRUE(all.equal(M,circ.cent.tri(tri)))==F & in.triangle(M,tri,boundary=FALSE)$in.tri==F)
    {stop('center is not the circumcenter or not in the interior of the triangle')}
  }

  n<-nrow(dat)
  arcs<-0
  if (n<=1)
  {
    arcs<-0
  } else
  {
    for (i in 1:n)
    { pt1<-dat[i,]
    if (in.triangle(pt1,tri,boundary=TRUE)$in.tri)
    { vert<-ifelse(isTRUE(all.equal(M,circ.cent.tri(tri)))==T,rv.triCC(pt1,tri)$rv,rv.tri.cent(pt1,tri,M)$rv)  #vertex region for pt
    for (j in (1:n)[-i])  #to avoid loops
    {
      arcs<-arcs+IndNAStri(pt1,dat[j,],tri,M,rv=vert)
    }
    }
    }
  }
  arcs
} #end of the function
#'

#################################################################

#' @title Arc density of Arc Slice Proximity Catch Digraphs (AS-PCDs) - one triangle case
#'
#' @description Returns the arc density of AS-PCD whose vertex set is the given 2D numerical data set, \code{Xp},
#' (some of its members are) in the triangle \code{tri}.
#'
#' AS proximity regions is defined with respect to \code{tri}
#' and vertex regions are defined with the center \code{M="CC"} for circumcenter of \code{tri};
#' or \eqn{M=(m_1,m_2)} in Cartesian coordinates or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the
#' interior of the triangle \code{tri}; default is \code{M="CC"} i.e., circumcenter of \code{tri}.
#' For the number of arcs, loops are not allowed so arcs are only possible for points inside \code{tri}
#' for this function.
#'
#' \code{tri.cor} is a logical argument for triangle correction (default is \code{TRUE}),
#' if \code{TRUE}, only the points inside the triangle are considered (i.e., digraph induced by these vertices
#' are considered) in computing the arc density, otherwise all points are considered
#' (for the number of vertices in the denominator of arc density).
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:mcap2012;textual}{pcds}).
#'
#' @param Xp A set of 2D points which constitute the vertices of the AS-PCD.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param M The center of the triangle. \code{"CC"} stands for circumcenter of the triangle \code{tri} or a 2D point in Cartesian coordinates or
#' a 3D point in barycentric coordinates which serves as a center in the interior of \code{tri};
#' default is \code{M="CC"} i.e., the circumcenter of \code{tri}.
#' @param tri.cor A logical argument for computing the arc density for only the points inside the triangle, \code{tri}
#' (default=\code{TRUE}), i.e., if \code{TRUE} only the induced digraph with the vertices inside \code{tri} are considered in the
#' computation of arc density.
#'
#' @return Arc density of AS-PCD whose vertices are the 2D numerical data set, \code{Xp};
#' AS proximity regions are defined with respect to the triangle \code{tri} and \eqn{CC}-vertex regions.
#'
#' @seealso \code{\link{ASarcdens.tri}}, \code{\link{CSarcdens.tri}}, and \code{\link{NumArcsAStri}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#'
#' set.seed(1)
#' n<-10  #try also n<-20
#'
#' dat<-runif.tri(n,Tr)$g
#'
#' M<-as.numeric(runif.tri(1,Tr)$g)  #try also M<-c(1.6,1.2)
#'
#' NumArcsAStri(dat,Tr,M)
#' ASarcdens.tri(dat,Tr,M)
#' ASarcdens.tri(dat,Tr,M,tri.cor = FALSE)
#'
#' ASarcdens.tri(dat,Tr,M)
#'
#' dat.fr<-data.frame(a=dat)
#' ASarcdens.tri(dat.fr,Tr,M)
#'
#' dat.fr<-data.frame(a=Tr)
#' ASarcdens.tri(dat,dat.fr,M)
#'
#' @export ASarcdens.tri
ASarcdens.tri <- function(Xp,tri,M="CC",tri.cor=TRUE)
{
  if (!is.numeric(as.matrix(Xp)) )
  {stop('Xp must be numeric')}

  if (is.point(Xp))
  { Xp<-matrix(Xp,ncol=2)
  } else
  {Xp<-as.matrix(Xp)
  if (ncol(Xp)!=2 )
  {stop('Xp must be of dimension nx2')}
  }

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  if (identical(M,"CC"))
  {M<-circ.cent.tri(tri)
  } else
  {
    if ( !is.point(M) && !is.point(M,3) )
    {stop('M must be the circumcenter "CC" or a numeric 2D point for Cartesian coordinates or
          3D point for barycentric coordinates')}

    if (dimension(M)==3)
    {
      M<-bary2cart(M,tri)
    }

    if (isTRUE(all.equal(M,circ.cent.tri(tri)))==F & in.triangle(M,tri,boundary=FALSE)$in.tri==F)
    {stop('center is not the circumcenter or not in the interior of the triangle')}
  }

  nx<-nrow(Xp)
  narcs<-NumArcsAStri(Xp,tri,M)

  if (tri.cor==TRUE)
  {
   ind.it<-c()
    for (i in 1:nx)
    {
     ind.it<-c(ind.it,in.triangle(Xp[i,],tri)$in.tri)
    }
    Xp.it<-Xp[ind.it,] #Xp points inside the triangle
    NinTri<-nrow(Xp.it)
    if (NinTri<=1)
    {stop('not enough points in the triangle to compute the arc density')}
    n<-NinTri
  } else
  {
    n<-nx
  }
  rho<-narcs/(n*(n-1))
rho
} #end of the function
#'

#################################################################

#' @title The indicator for the presence of an arc from a point in set \code{S} to the point \code{pt} for
#' Arc Slice Proximity Catch Digraphs (AS-PCDs) - one triangle case
#'
#' @description Returns I(\eqn{pt \in N_{AS}(x)} for some \eqn{x \in S}), that is, returns 1 if \eqn{pt} is in \eqn{\cup_{x \in S}N_{AS}(x)},
#' returns 0 otherwise, where \eqn{N_{AS}(x)} is the AS proximity region for point \eqn{x}.
#'
#' AS proximity regions are constructed with respect to the triangle, \code{tri}\eqn{=T(A,B,C)=}\code{(rv=1,rv=2,rv=3)},
#' and vertices of \code{tri} are also labeled as 1,2, and 3, respectively.
#'
#' Vertex regions are based on the center \code{M="CC"} for circumcenter of \code{tri};
#' or \eqn{M=(m_1,m_2)} in Cartesian coordinates or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the
#' interior of the triangle \code{tri}; default is \code{M="CC"} i.e., circumcenter of \code{tri}.
#'
#' If \code{pt} is not in \code{S} and either \code{pt} or all points in \code{S} are outside \code{tri}, it returns 0,
#' but if \code{pt} is in \code{S}, then it always returns 1 (i.e., loops are allowed).
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:mcap2012;textual}{pcds}).
#'
#' @param S A set of 2D points whose AS proximity regions are considered.
#' @param pt A 2D point. The function determines whether \code{pt} is inside the union of AS proximity
#' regions of points in \code{S} or not.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param M The center of the triangle. \code{"CC"} stands for circumcenter of the triangle \code{tri} or a 2D point in Cartesian coordinates or
#' a 3D point in barycentric coordinates which serves as a center in the interior of \code{tri};
#' default is \code{M="CC"} i.e., the circumcenter of \code{tri}.
#'
#' @return \eqn{I(pt \in \cup_{x in S}N_{AS}(x,r))}, that is, returns 1 if \code{pt} is in \code{S} or inside \eqn{N_{AS}(x)} for at least
#' one \eqn{x} in \code{S}, returns 0 otherwise, where AS proximity region is constructed in \code{tri}
#'
#' @seealso \code{\link{IndNAStri}}, \code{\link{IndNAStriSet}}, and \code{\link{IndNCStriSet}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#' n<-10
#'
#' set.seed(1)
#' dat<-runif.tri(n,Tr)$gen.points
#'
#' S<-rbind(dat[1,],dat[2,])  #try also S<-c(1.5,1)
#'
#' M<-as.numeric(runif.tri(1,Tr)$g)  #try also M<-c(1.6,1.2)
#'
#' IndNAStriSet(S,dat[3,],Tr,M)
#'
#' S<-rbind(dat[1,],dat[2,],dat[3,],dat[5,])
#' IndNAStriSet(S,dat[3,],Tr,M)
#'
#' IndNAStriSet(S,dat[6,],Tr,M)
#'
#' S<-rbind(c(.1,.1),c(.3,.4),c(.5,.3))
#' IndNAStriSet(S,dat[3,],Tr,M)
#'
#' IndNAStriSet(c(.2,.5),dat[2,],Tr,M)
#' IndNAStriSet(dat,c(.2,.5),Tr,M)
#' IndNAStriSet(dat,dat[2,],Tr,M)
#' IndNAStriSet(c(.2,.5),c(.2,.5),Tr,M)
#' IndNAStriSet(dat[5,],dat[2,],Tr,M)
#'
#' S<-rbind(dat[1,],dat[2,],dat[3,],dat[5,],c(.2,.5))
#' IndNAStriSet(S,dat[3,],Tr,M)
#'
#' P<-c(.4,.2)
#' S<-dat[c(1,3,4),]
#' IndNAStriSet(dat,P,Tr,M)
#' IndNAStriSet(S,P,Tr,M)
#'
#' IndNAStriSet(rbind(S,S),P,Tr,M)
#'
#' dat.fr<-data.frame(a=S)
#' IndNAStriSet(dat.fr,P,Tr,M)
#'
#' @export IndNAStriSet
IndNAStriSet <- function(S,pt,tri,M="CC")
{
  if (!is.point(pt))
  {stop('pt must be a numeric 2D point')}

  if (!is.numeric(as.matrix(S)))
  {stop('S must be a matrix of numeric values')}

  if (is.point(S))
  { S<-matrix(S,ncol=2)
  } else
  {S<-as.matrix(S)
  if (ncol(S)!=2 )
  {stop('S must be of dimension nx2')}
  }

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  if (identical(M,"CC"))
  {M<-circ.cent.tri(tri)
  } else
  {
    if ( !is.point(M) && !is.point(M,3) )
    {stop('M must be the circumcenter "CC" or a numeric 2D point for Cartesian coordinates or
          3D point for barycentric coordinates')}

    if (dimension(M)==3)
    {
      M<-bary2cart(M,tri)
    }

    if (isTRUE(all.equal(M,circ.cent.tri(tri)))==F & in.triangle(M,tri,boundary=FALSE)$in.tri==F)
    {stop('center is not the circumcenter or not in the interior of the triangle')}
  }

  k<-nrow(S);
  dom<-0; i<-1;
  while (dom ==0 && i<= k)
  {
    if (IndNAStri(S[i,],pt,tri,M)==1)
    {dom<-1};
    i<-i+1;
  }
  dom
} #end of the function
#'

#################################################################

#' @title The indicator for the set of points \code{S} being a dominating set or not for Arc Slice Proximity
#' Catch Digraphs (AS-PCDs) - one triangle case
#'
#' @description Returns \eqn{I(}\code{S} a dominating set of AS-PCD\eqn{)}, that is, returns 1 if \code{S} is a dominating set of AS-PCD,
#' returns 0 otherwise.
#'
#' AS-PCD has vertex set \code{Dt} and AS proximity region is constructed with vertex
#' regions based on the center \code{M="CC"} for circumcenter of \code{tri};
#' or \eqn{M=(m_1,m_2)} in Cartesian coordinates or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the
#' interior of the triangle \code{tri}; default is \code{M="CC"} i.e., circumcenter of \code{tri}
#' whose vertices are also labeled as edges 1, 2, and 3, respectively.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:mcap2012;textual}{pcds}).
#'
#' @param S A set of 2D points which is to be tested for being a dominating set for the AS-PCDs.
#' @param Dt A set of 2D points which constitute the vertices of the AS-PCD.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param M The center of the triangle. \code{"CC"} stands for circumcenter of the triangle \code{tri} or a 2D point in Cartesian coordinates or
#' a 3D point in barycentric coordinates which serves as a center in the interior of \code{tri};
#' default is \code{M="CC"} i.e., the circumcenter of \code{tri}.
#'
#' @return \eqn{I(}\code{S} a dominating set of AS-PCD\eqn{)}, that is, returns 1 if \code{S} is a dominating set of AS-PCD whose
#' vertices are the data points in \code{Dt}; returns 0 otherwise, where AS proximity region is constructed in
#' the triangle \code{tri}.
#'
#' @seealso \code{\link{IndNAStriSet}}, \code{\link{IndNPEtri.domset}} and \code{\link{IndNCStri.domset}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#'
#' Tr<-rbind(A,B,C);
#' n<-10
#'
#' set.seed(1)
#' dat<-runif.tri(n,Tr)$gen.points
#'
#' M<-as.numeric(runif.tri(1,Tr)$g)  #try also M<-c(1.6,1.2)
#'
#' S<-rbind(dat[1,],dat[2,])
#' IndNAStri.domset(S,dat,Tr,M)
#'
#' S<-rbind(dat[1,],dat[2,],dat[3,],dat[5,])
#' IndNAStri.domset(S,dat,Tr,M)
#'
#' S<-rbind(c(.1,.1),c(.3,.4),c(.5,.3))
#' IndNAStri.domset(S,dat,Tr,M)
#'
#' IndNAStri.domset(c(.2,.5),dat,Tr,M)
#' IndNAStri.domset(c(.2,.5),c(.2,.5),Tr,M)
#' IndNAStri.domset(dat[5,],dat[2,],Tr,M)
#'
#' S<-rbind(dat[1,],dat[2,],dat[3,],dat[5,],c(.2,.5))
#' IndNAStri.domset(S,dat[3,],Tr,M)
#'
#' IndNAStri.domset(dat,dat,Tr,M)
#'
#' P<-c(.4,.2)
#' S<-dat[c(1,3,4),]
#' IndNAStri.domset(dat,P,Tr,M)
#' IndNAStri.domset(S,P,Tr,M)
#' IndNAStri.domset(S,dat,Tr,M)
#'
#' IndNAStri.domset(rbind(S,S),dat,Tr,M)
#'
#' dat.fr<-data.frame(a=dat)
#' IndNAStri.domset(S,dat.fr,Tr,M)
#'
#' @export IndNAStri.domset
IndNAStri.domset <- function(S,Dt,tri,M="CC")
{
  if (!is.numeric(as.matrix(S)) || !is.numeric(as.matrix(Dt)))
  {stop('S and Dt must be numeric')}

  if (is.point(S))
  { S<-matrix(S,ncol=2)
  } else
  {S<-as.matrix(S)
  if (ncol(S)!=2 )
  {stop('S must be of dimension nx2')}
  }

  if (is.point(Dt))
  { Dt<-matrix(Dt,ncol=2)
  } else
  {Dt<-as.matrix(Dt)
  if (ncol(Dt)!=2 )
  {stop('Dt must be of dimension nx2')}
  }

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  if (identical(M,"CC"))
  {M<-circ.cent.tri(tri)
  } else
  {
    if ( !is.point(M) && !is.point(M,3) )
    {stop('M must be the circumcenter "CC" or a numeric 2D point for Cartesian coordinates or
          3D point for barycentric coordinates')}

    if (dimension(M)==3)
    {
      M<-bary2cart(M,tri)
    }

    if (isTRUE(all.equal(M,circ.cent.tri(tri)))==F & in.triangle(M,tri,boundary=FALSE)$in.tri==F)
    {stop('center is not the circumcenter or not in the interior of the triangle')}
  }

  k<-nrow(S);
  n<-nrow(Dt);

  dom<-1; i<-1;
  while (dom ==1 && i<= n)
  {
    if (IndNAStriSet(S,Dt[i,],tri,M)==0)  #this is where tri is used
    {dom<-0};
    i<-i+1;
  }
  dom
} #end of the function
#'

#################################################################

#' @title Indicator for an upper bound for the domination number of Arc Slice Proximity Catch Digraph
#' (AS-PCD) by the exact algorithm - one triangle case
#'
#' @description Returns \eqn{I(}domination number of AS-PCD whose vertices are the data points \code{Dt} is less than or equal to \code{k}\eqn{)},
#' that is, returns 1 if the domination number of AS-PCD is less than the prespecified value \code{k}, returns 0
#' otherwise. It also provides the vertices (i.e., data points) in a dominating set of size \code{k} of AS-PCD.
#'
#' AS proximity regions are constructed with respect to the triangle \code{tri} and
#' vertex regions are based on the center \code{M="CC"} for circumcenter of \code{tri};
#' or \eqn{M=(m_1,m_2)} in Cartesian coordinates or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the
#' interior of the triangle \code{tri}; default is \code{M="CC"} i.e., circumcenter of \code{tri}.
#'
#' The vertices of \code{tri}angle, \code{tri},
#' are labeled as \eqn{1,2,3} according to the row number the vertex is recorded in \code{tri}.
#' Loops are allowed in the digraph.
#' It takes a long time for large number of vertices (i.e., large number of row numbers).
#'
#' @param Dt A set of 2D points which constitute the vertices of the AS-PCD.
#' @param k A positive integer to be tested for an upper bound for the domination number of AS-PCDs.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param M The center of the triangle. \code{"CC"} stands for circumcenter of the triangle \code{tri} or a 2D point in Cartesian coordinates or
#' a 3D point in barycentric coordinates which serves as a center in the interior of \code{tri};
#' default is \code{M="CC"} i.e., the circumcenter of \code{tri}.
#'
#' @return A \code{list} with the elements
#' \item{domUB}{The suggested upper bound (to be checked) for the domination number of AS-PCD.
#' It is prespecified as \code{k} in the function arguments.}
#' \item{IndUBdom}{The indicator for the upper bound for domination number of AS-PCD being the
#' specified value \code{k} or not. It returns 1 if the upper bound is \code{k}, and 0 otherwise.}
#' \item{ind.domset}{The vertices (i.e., data points) in the dominating set of size \code{k} if it exists,
#' otherwise it yields \code{NULL}.}
#'
#' @seealso \code{\link{IndCSdomUBtri}}, \code{\link{IndCSdomUBTe}}, \code{\link{IndUBdom}},
#' and \code{\link{dom.exact}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#'
#' Tr<-rbind(A,B,C);
#' n<-10
#'
#' set.seed(1)
#' dat<-runif.tri(n,Tr)$gen.points
#'
#' M<-as.numeric(runif.tri(1,Tr)$g)  #try also M<-c(1.6,1.2)
#'
#' IndASdomUBtri(dat,1,Tr)
#'
#' for (k in 1:n)
#'   print(c(k,IndASdomUBtri(dat,k,Tr,M)))
#'
#' IndASdomUBtri(dat,k=4,Tr,M)
#'
#' P<-c(.4,.2)
#' IndASdomUBtri(P,1,Tr,M)
#'
#' IndASdomUBtri(rbind(dat,dat),k=2,Tr,M)
#'
#' dat.fr<-data.frame(a=dat)
#' IndASdomUBtri(dat.fr,1,Tr,M)
#'
#' @export IndASdomUBtri
IndASdomUBtri <- function(Dt,k,tri,M="CC")
{
  if (!is.numeric(as.matrix(Dt)))
  {stop('Dt must be numeric')}

  if (is.point(Dt))
  { Dt<-matrix(Dt,ncol=2)
  } else
  {Dt<-as.matrix(Dt)
  if (ncol(Dt)!=2 )
  {stop('Dt must be of dimension nx2')}
  }

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  if (identical(M,"CC"))
  {M<-circ.cent.tri(tri)
  } else
  {
    if ( !is.point(M) && !is.point(M,3) )
    {stop('M must be the circumcenter "CC" or a numeric 2D point for Cartesian coordinates or
          3D point for barycentric coordinates')}

    if (dimension(M)==3)
    {
      M<-bary2cart(M,tri)
    }

    if (isTRUE(all.equal(M,circ.cent.tri(tri)))==F & in.triangle(M,tri,boundary=FALSE)$in.tri==F)
    {stop('center is not the circumcenter or not in the interior of the triangle')}
  }

  n<-nrow(Dt);
  xc<-combinat::combn(1:n,k); N1<-choose(n,k);
  xc<-matrix(xc,ncol=N1)
  dom<-0; j<-1; domset<-c();
  while (j<=N1 && dom==0)
  {
    if (IndNAStri.domset(Dt[xc[,j],],Dt,tri,M)==1)  #this is where triangle tri is used
    {dom<-1; domset<-Dt[xc[,j],];}
    j<-j+1;
  }

  list(domUB=k, #upper bound for the domination number of AS-PCD
       IndUBdom=dom, #indicator that domination number <=k
       domset=domset #a dominating set of size k (if exists)
  )
} #end of the function
#'

#################################################################

#' @title Number of arcs of Arc Slice Proximity Catch Digraphs (AS-PCDs) - multiple triangle case
#'
#' @description Returns the number of arcs of Arc Slice Proximity Catch Digraph (AS-PCD) whose vertices are
#' the data points in \code{Xp} in the multiple triangle case.
#'
#' AS proximity regions are defined with respect to the
#' Delaunay triangles based on \code{Yp} points and vertex regions in each triangle are based on the center \code{M="CC"}
#' for circumcenter of each Delaunay triangle or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the
#' interior of each Delaunay triangle; default is \code{M="CC"} i.e., circumcenter of each triangle.
#'
#' Convex hull of \code{Yp} is partitioned by the Delaunay triangles based on \code{Yp} points
#' (i.e., multiple triangles are the set of these Delaunay triangles whose union constitutes
#' the convex hull of \code{Yp} points).
#'
#' See (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:mcap2012;textual}{pcds}) for more on AS-PCDs.
#' Also see (\insertCite{okabe:2000,ceyhan:comp-geo-2010,sinclair:2016;textual}{pcds}) for more on Delaunay triangulation
#' and the corresponding algorithm.
#'
#' @param Xp A set of 2D points which constitute the vertices of the AS-PCD.
#' @param Yp A set of 2D points which constitute the vertices of the Delaunay triangles.
#' @param M The center of the triangle. \code{"CC"} stands for circumcenter of each Delaunay triangle \code{tri} or 3D point in barycentric
#' coordinates which serves as a center in the interior of each Delaunay triangle;
#' default is \code{M="CC"} i.e., the circumcenter of each triangle.
#'
#' @return A \code{list} with the elements
#' \item{num.pts.in.CH}{Number of \code{Xp} points in the convex hull of \code{Yp} points}
#' \item{num.arcs}{Number of arcs in the AS-PCD for the \code{Xp} points inside the convex hull of \code{Yp} points}
#' \item{weight.vec}{The \code{vector} of areas of the Delaunay triangles that have at least on \code{Xp} points in them}
#'
#' @seealso \code{\link{NumArcsAStri}}, \code{\link{NumArcsPEMT}}, and \code{\link{NumArcsCSMT}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' nx<-20; ny<-4;  #try also nx<-40; ny<-10 or nx<-1000; ny<-10;
#'
#' set.seed(1)
#' Xp<-cbind(runif(nx),runif(nx))
#' Yp<-cbind(runif(ny),runif(ny))
#'
#' oldpar <- par(no.readonly = TRUE)
#' plotDeltri(Xp,Yp,xlab="",ylab="")
#' par(oldpar)
#'
#' M<-"CC"  #try also M<-c(1,1,1)
#'
#' NumArcsASMT(Xp,Yp,M)
#' NumArcsASMT(Xp,Yp[1:3,],M)
#'
#' NumArcsASMT(c(.4,.2),Yp,M)
#'
#' NumArcsASMT(Xp,rbind(Yp,Yp),M)
#'
#' dat.fr<-data.frame(a=Xp)
#' NumArcsASMT(dat.fr,Yp,M)
#'
#' @export NumArcsASMT
NumArcsASMT <- function(Xp,Yp,M="CC")
{
  if (!is.numeric(as.matrix(Xp)) || !is.numeric(as.matrix(Yp)))
  {stop('Xp and Yp must be numeric')}

  if (is.point(Xp))
  { Xp<-matrix(Xp,ncol=2)
  } else
  {Xp<-as.matrix(Xp)
  if (ncol(Xp)!=2 )
  {stop('Xp must be of dimension nx2')}
  }

  Yp<-as.matrix(Yp)
  if (ncol(Yp)!=2 || nrow(Yp)<3)
  {stop('Yp must be of dimension kx2 with k>=3')}

  if (nrow(Yp)==3)
  {
    Tri.Ind<-indices.Del.tri(Xp,Yp)  #returns 1's if the points Xp[i,]'s are inside triangle based on Yp, NA otherwise

   inTri<-which(Tri.Ind==1)
    NinTri<-length(inTri)  #number of points in the triangle

    if (NinTri==0)
    {Tot.Arcs<-0;
    ListW<-vector()
    } else
    {
      Xdt<-matrix(Xp[inTri,],ncol=2)
      tri<-as.bastri(Yp)$tri #convert the triangle Yp into an unscaled basic triangle, see as.bastri help page
      ListW<-area.polygon(tri)
      Tot.Arcs<-NumArcsAStri(Xdt,tri,M)  #number of arcs in the triangle Yp
    }
    res<-list(num.arcs=Tot.Arcs,
              num.in.conv.hull=NinTri,
              weight.vec=ListW)
  } else
  {
    if ((!is.point(M,3) && M!="CC") || !all(M>0))
    {stop('M must be a numeric 3D point with positive barycentric coordinates or "CC" for circumcenter')}

    #Delaunay triangulation of Yp points
    Ytrimesh<-interp::tri.mesh(Yp[,1],Yp[,2],duplicate="remove")
    Ytri<-matrix(interp::triangles(Ytrimesh)[,1:3],ncol=3); #the vertices of the Delaunay triangles
    nt<-nrow(Ytri)  #number of Delaunay triangles

   inCH<-interp::in.convex.hull(Ytrimesh,Xp[,1],Xp[,2])
    Ninch<-sum(inCH)  #number of points in the convex hull
    if (Ninch==0)
    {Tot.Arcs<-0;
    ListW<-NULL
    } else
    {
      Xdt<-matrix(Xp[inCH==TRUE,],ncol=2)
      Tri.Ind<-indices.Del.tri(Xdt,Yp,Ytrimesh)
      #indices of triangles in which the points in the data fall

      #calculation of the total number of arcs
      List.W<-ni<-arcs<-vector()
      for (i in 1:nt)
      {
        dati<-Xdt[Tri.Ind==i,] #points in ith Delaunay triangle
        Tri<-Yp[Ytri[i,],]
        tri<-as.bastri(Tri)$tri #convert the triangle Tri into an unscaled basic triangle, see as.bastri help page
        List.W<-c(List.W,area.polygon(tri))
        ni<-c(ni,length(dati)/2)  #number of points in ith Delaunay triangle

        num.arcs<-NumArcsAStri(dati,tri,M)  #number of arcs in ith triangle
        arcs<-c(arcs,num.arcs)  #number of arcs in all triangles as a vector
      }

      Tot.Arcs<-sum(arcs)  #the total number of arcs in all triangles
      ListW<-List.W[ni >= 1] #adjusted for triangles with one or more points in them
    }
    res<-list(num.pts.in.CH=Ninch,
              num.arcs=Tot.Arcs,
              weight.vec=ListW)
  }
  res
} #end of the function
#'

#################################################################

#' @title The scatterplot of points from one class and plot of the Delaunay triangulation of the other class
#'
#' @description Plots the scatter plot of \code{Xp} points together with the Delaunay triangles based on the \code{Yp} points.
#' Both sets of points are of 2D.
#'
#' See (\insertCite{okabe:2000,ceyhan:comp-geo-2010,sinclair:2016;textual}{pcds}) for more on Delaunay triangulation and the corresponding algorithm.
#'
#' @param Xp A set of 2D points whose scatterplot is to be plotted.
#' @param Yp A set of 2D points which constitute the vertices of the Delaunay triangles.
#' @param main An overall title for the plot (default=\code{""}).
#' @param xlab,ylab Titles for the \eqn{x} and \eqn{y} axes, respectively (default=\code{""} for both).
#' @param xlim,ylim Two \code{numeric} vectors of length 2, giving the \eqn{x}- and \eqn{y}-coordinate ranges (default=\code{NULL} for both)
#' @param \dots Additional \code{plot} parameters.
#'
#' @return A scatterplot of \code{Xp} points and the Delaunay triangulation of \code{Yp} points.
#'
#' @seealso \code{\link[interp]{plot.triSht}} in \code{interp} package
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' nx<-20; ny<-4;  #try also nx<-40; ny<-10 or nx<-1000; ny<-10;
#'
#' set.seed(1)
#' Xp<-cbind(runif(nx),runif(nx))
#' Yp<-cbind(runif(ny),runif(ny))
#'
#' oldpar <- par(no.readonly = TRUE)
#' plotDeltri(Xp,Yp,xlab="",ylab="",main="X points and Delaunay Triangulation of Y points")
#'
#' P<-c(.6,.4)
#' plotDeltri(P,Yp,xlab="",ylab="",main="X points and Delaunay Triangulation of Y points")
#'
#' plotDeltri(Xp,Yp,xlab="",ylab="")
#' plotDeltri(Xp,Yp[1:3,],xlab="",ylab="")
#'
#' plotDeltri(Xp,rbind(Yp,Yp),xlab="",ylab="")
#'
#' dat.fr<-data.frame(a=Xp)
#' plotDeltri(dat.fr,Yp,xlab="",ylab="")
#'
#' dat.fr<-data.frame(a=Yp)
#' plotDeltri(Xp,dat.fr,xlab="",ylab="")
#' par(oldpar)
#'
#' @export plotDeltri
plotDeltri <- function(Xp,Yp,main="",xlab="",ylab="",xlim=NULL,ylim=NULL, ...)
{
  if (!is.numeric(as.matrix(Xp)) || !is.numeric(as.matrix(Yp)))
  {stop('Xp and Yp must be numeric')}

  if (is.point(Xp))
  { Xp<-matrix(Xp,ncol=2)
  } else
  {Xp<-as.matrix(Xp)
  if (ncol(Xp)!=2 )
  {stop('Xp must be of dimension nx2')}
  }

  Yp<-as.matrix(Yp)
  if (ncol(Yp)!=2 || nrow(Yp)<3)
  {stop('Yp must be of dimension kx2 with k>=3')}

  if (is.null(xlim))
  {xlim<-range(Xp[,1],Yp[,1])}
  if (is.null(ylim))
  {ylim<-range(Xp[,2],Yp[,2])}

  oldpar <- par(no.readonly = TRUE)    # default par options
  on.exit(par(oldpar))            # sets default par options when the function is exited

  if (nrow(Yp)==3)
  {
    par(mfrow=c(1,1),mar=c(5,5,4,2))
    plot(Xp[,1],Xp[,2],main=main, xlab=xlab, ylab=ylab,xlim=xlim,ylim=ylim,pch=".",cex=4, ...)
    polygon(Yp,lty=2)
    #par(mfrow=c(1,1),mar=c(5.1, 4.1, 4.1, 2.1))
  } else
  {
    Ytrimesh<-interp::tri.mesh(Yp[,1],Yp[,2],duplicate="remove")  #Delaunay triangulation
    par(mfrow=c(1,1),mar=c(5,5,4,2))
    plot(Xp[,1],Xp[,2],main=main, xlab=xlab, ylab=ylab,xlim=xlim,ylim=ylim,pch=".",cex=4, ...)
   interp::plot.triSht(Ytrimesh, add=TRUE, do.points = TRUE, ...)
    #par(mfrow=c(1,1),mar=c(5.1, 4.1, 4.1, 2.1))
  }
} #end of the function
#'

#################################################################

#' @title Incidence matrix for Arc Slice Proximity Catch Digraphs (AS-PCDs) - one triangle case
#'
#' @description Returns the incidence matrix for the AS-PCD whose vertices are the given 2D numerical data set, \code{dat}.
#'
#' AS proximity regions are defined with respect to the triangle \code{tri}\eqn{=T(v=1,v=2,v=3)} and
#' vertex regions based on the center \code{M="CC"} for circumcenter of \code{tri};
#' or \eqn{M=(m_1,m_2)} in Cartesian coordinates or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the
#' interior of the triangle \code{tri}; default is \code{M="CC"} i.e., circumcenter of \code{tri}.
#' Loops are allowed, so the diagonal entries are all equal to 1.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:mcap2012;textual}{pcds}).
#'
#' @param dat A set of 2D points which constitute the vertices of AS-PCD.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param M The center of the triangle. \code{"CC"} stands for circumcenter of the triangle \code{tri} or a 2D point in Cartesian coordinates or
#' a 3D point in barycentric coordinates which serves as a center in the interior of \code{tri};
#' default is \code{M="CC"} i.e., the circumcenter of \code{tri}.
#'
#' @return Incidence matrix for the AS-PCD whose vertices are 2D data set, \code{dat},
#' and AS proximity regions are defined with respect to the triangle \code{tri} and
#' vertex regions based on circumcenter.
#'
#' @seealso \code{\link{IncMatASMT}}, \code{\link{IncMatPEtri}}, and \code{\link{IncMatCStri}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#' n<-10
#'
#' set.seed(1)
#' dat<-runif.tri(n,Tr)$g
#'
#' M<-as.numeric(runif.tri(1,Tr)$g)  #try also M<-c(1.6,1.2)
#'
#' IM<-IncMatAStri(dat,Tr,M)
#' IM
#'
#' dom.greedy(IM)
#' dom.exact(IM)
#'
#' dat.fr<-data.frame(a=dat)
#' IncMatAStri(dat.fr,Tr,M)
#'
#' @export IncMatAStri
IncMatAStri <- function(dat,tri,M="CC")
{
  if (!is.numeric(as.matrix(dat)))
  {stop('dat must be numeric')}

  if (is.point(dat))
  { dat<-matrix(dat,ncol=2)
  } else
  {dat<-as.matrix(dat)
  if (ncol(dat)!=2 )
  {stop('dat must be of dimension nx2')}
  }

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  if (identical(M,"CC"))
  {M<-circ.cent.tri(tri)
  } else
  {
    if ( !is.point(M) && !is.point(M,3) )
    {stop('M must be the circumcenter "CC" or a numeric 2D point for Cartesian coordinates or
          3D point for barycentric coordinates')}

    if (dimension(M)==3)
    {
      M<-bary2cart(M,tri)
    }

    if (isTRUE(all.equal(M,circ.cent.tri(tri)))==F & in.triangle(M,tri,boundary=FALSE)$in.tri==F)
    {stop('center is not the circumcenter or not in the interior of the triangle')}
  }

  n<-nrow(dat)
 inc.mat<-matrix(0, nrow=n, ncol=n)
  if (n>=1)
  {
    for (i in 1:n)
    {pt1<-dat[i,]
    vert<-ifelse(isTRUE(all.equal(M,circ.cent.tri(tri)))==T,rv.triCC(pt1,tri)$rv,rv.tri.cent(pt1,tri,M)$rv)  #vertex region for pt
    for (j in ((1:n)) )
    {pt2<-dat[j,]
   inc.mat[i,j]<-IndNAStri(pt1,pt2,tri,M,rv=vert)
    }
    }
  }
 inc.mat
} #end of the function
#'

#################################################################

#' @title Incidence matrix for Arc Slice Proximity Catch Digraphs (AS-PCDs) - multiple triangle case
#'
#' @description Returns the incidence matrix for the AS-PCD whose vertices are a given 2D numerical data set, \code{Xp},
#' in the convex hull of \code{Yp} which is partitioned by the Delaunay triangles based on \code{Yp} points.
#'
#' AS proximity regions are defined with respect to the Delaunay triangles based on \code{Yp} points and vertex
#' regions are based on the center \code{M="CC"}
#' for circumcenter of each Delaunay triangle or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the
#' interior of each Delaunay triangle; default is \code{M="CC"} i.e., circumcenter of each triangle.
#' Loops are allowed, so the diagonal entries are all equal to 1.
#'
#' See (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:mcap2012;textual}{pcds}) for more on AS-PCDs.
#' Also see (\insertCite{okabe:2000,ceyhan:comp-geo-2010,sinclair:2016;textual}{pcds}) for more on Delaunay triangulation
#' and the corresponding algorithm.
#'
#' @param Xp A set of 2D points which constitute the vertices of the AS-PCD.
#' @param Yp A set of 2D points which constitute the vertices of the Delaunay triangles.
#' @param M The center of the triangle. \code{"CC"} stands for circumcenter of each Delaunay triangle \code{tri} or 3D point in barycentric
#' coordinates which serves as a center in the interior of each Delaunay triangle;
#' default is \code{M="CC"} i.e., the circumcenter of each triangle.
#'
#' @return Incidence matrix for the AS-PCD whose vertices are the 2D data set, \code{Xp},
#' and AS proximity regions are defined in the Delaunay triangles based on \code{Yp} points.
#'
#' @seealso \code{\link{IncMatAStri}}, \code{\link{IncMatPEMT}}, and \code{\link{IncMatCSMT}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' #nx is number of X points (target) and ny is number of Y points (nontarget)
#' nx<-20; ny<-4;  #try also nx<-40; ny<-10 or nx<-1000; ny<-10;
#'
#' set.seed(1)
#' Xp<-cbind(runif(nx,0,1),runif(nx,0,1))
#' Yp<-cbind(runif(ny,0,1),runif(ny,0,1))
#'
#' M<-"CC"  #try also M<-c(1,1,1)
#'
#' IM<-IncMatASMT(Xp,Yp,M)
#' IM
#' dom.greedy(IM)  #try also dom.exact(IM)  #this might take a long time for large  nx
#'
#' IM<-IncMatASMT(Xp,Yp[1:3,],M)
#'
#' IncMatASMT(Xp,rbind(Yp,Yp))
#'
#' dat.fr<-data.frame(a=Xp)
#' IncMatASMT(dat.fr,Yp,M)
#'
#' @export IncMatASMT
IncMatASMT <- function(Xp,Yp,M="CC")
{
  if (!is.numeric(as.matrix(Xp)) || !is.numeric(as.matrix(Yp)))
  {stop('Xp and Yp must be numeric')}

  if (is.point(Xp))
  { Xp<-matrix(Xp,ncol=2)
  } else
  {Xp<-as.matrix(Xp)
  if (ncol(Xp)!=2 )
  {stop('Xp must be of dimension nx2')}
  }

  Yp<-as.matrix(Yp)
  if (ncol(Yp)!=2 || nrow(Yp)<3)
  {stop('Yp must be of dimension kx2 with k>=3')}

  if (nrow(Yp)==3)
  {
   inc.mat<-IncMatAStri(Xp,Yp,M)
  } else
  {
    if ((!is.point(M,3) && M!="CC") || !all(M>0))
    {stop('M must be a numeric 3D point with positive barycentric coordinates or "CC" for circumcenter')}

    DTmesh<-interp::tri.mesh(Yp[,1],Yp[,2],duplicate="remove")

    nx<-nrow(Xp)
    ch<-rep(0,nx)
    for (i in 1:nx)
      ch[i]<-interp::in.convex.hull(DTmesh,Xp[i,1],Xp[i,2])

   inc.mat<-matrix(0, nrow=nx, ncol=nx)

    DTr<-matrix(interp::triangles(DTmesh)[,1:3],ncol=3)
    nt<-nrow(DTr)  #number of Delaunay triangles

    if (nx>1)
    {
      i.tr<-rep(0,nx)  #the \code{vector} of indices for the triangles that contain the Xp points
      for (i in 1:nx)
        for (j in 1:nt)
        {
          tri<-Yp[DTr[j,],]
          if (in.triangle(Xp[i,],tri,boundary=TRUE)$in.tri )
            i.tr[i]<-j
        }

      for (i in 1:nx)
      {pt1<-Xp[i,]
      if (i.tr[i]!=0)
      {
        Yi.Tri<-Yp[DTr[i.tr[i],],]
        Yi.tri<-as.bastri(Yi.Tri)$tri #convert the triangle Yi.Tri into an unscaled basic triangle, see as.bastri help page
        vert<-ifelse(identical(M,"CC"),rv.triCC(pt1,Yi.tri)$rv,rv.tri.cent(pt1,Yi.tri,M)$rv)  #vertex region for pt
        for (j in 1:nx )
        {pt2<-Xp[j,]
       inc.mat[i,j]<-IndNAStri(pt1,pt2,Yi.tri,M,rv=vert)
        }
      }
      }
    }

    diag(inc.mat)<-1
  }
 inc.mat
} #end of the function
#'

#################################################################

#' @title The furthest points from vertices in each \eqn{CC}-vertex region in a basic triangle
#'
#' @description
#' An object of class \code{"Extrema"}.
#' Returns the furthest data points among the data set, \code{Dt}, in each \eqn{CC}-vertex region from the
#' corresponding vertex in the basic triangle \eqn{T_b=T(A=(0,0),B=(1,0),C=(c_1,c_2))}.
#'
#' Any given triangle can be mapped to the basic triangle by a combination of rigid body motions
#' (i.e., translation, rotation and reflection) and scaling, preserving uniformity of the points in the
#' original triangle. Hence basic triangle is useful for simulation
#' studies under the uniformity hypothesis.
#'
#' \code{ch.all.intri} is for checking whether all data points are inside \eqn{T_b} (default is \code{FALSE}).
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:mcap2012;textual}{pcds}).
#'
#' @param Dt A set of 2D points.
#' @param c1,c2 Positive real numbers which constitute the vertex of the basic triangle.
#' adjacent to the shorter edges; \eqn{c_1} must be in \eqn{[0,1/2]}, \eqn{c_2>0} and \eqn{(1-c_1)^2+c_2^2 \le 1}
#' @param ch.all.intri A logical argument for checking whether all data points are inside \eqn{T_b}
#' (default is \code{FALSE}).
#'
#' @return A \code{list} with the elements
#' \item{txt1}{Vertex labels are \eqn{A=1}, \eqn{B=2}, and \eqn{C=3} (corresponds to row number in Extremum Points).}
#' \item{txt2}{A short description of the distances as \code{"Distances from furthest points to ..."}.}
#' \item{type}{Type of the extrema points}
#' \item{desc}{A short description of the extrema points}
#' \item{mtitle}{The \code{"main"} title for the plot of the extrema}
#' \item{ext}{The extrema points, here, furthest points from vertices in each vertex region.}
#' \item{X}{The input data, \code{Dt}, can be a \code{matrix} or \code{data frame}}
#' \item{num.points}{The number of data points, i.e., size of \code{Dt}}
#' \item{supp}{Support of the data points, here, it is \eqn{T_b}.}
#' \item{cent}{The center point used for construction of edge regions.}
#' \item{ncent}{Name of the center, \code{cent}, it is circumcenter \code{"CC"} for this function.}
#' \item{regions}{Vertex regions inside the triangle, \eqn{T_b}, provided as a list.}
#' \item{region.names}{Names of the vertex regions as \code{"vr=1"}, \code{"vr=2"}, and \code{"vr=3"}}
#' \item{region.centers}{Centers of mass of the vertex regions inside \eqn{T_b}.}
#' \item{dist2ref}{Distances from furthest points in each vertex region to the corresponding vertex.}
#'
#' @seealso \code{\link{fr2vVRCC}}, \code{\link{fr2eTeER}}, \code{\link{Kfr2vTbVRCC}} and \code{\link{Kfr2vVRCC}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' c1<-.4; c2<-.6;
#' A<-c(0,0); B<-c(1,0); C<-c(c1,c2);
#' Tb<-rbind(A,B,C)
#' n<-20
#'
#' set.seed(1)
#' dat<-runif.bastri(n,c1,c2)$g
#'
#' Ext<-fr2vTbVRCC(dat,c1,c2)
#' Ext
#' summary(Ext)
#' plot(Ext)
#'
#' fr2vTbVRCC(dat[1,],c1,c2)
#'
#' dat2<-rbind(dat,c(.2,.4))
#' fr2vTbVRCC(dat2,c1,c2)
#'
#' f2v<-fr2vTbVRCC(dat,c1,c2)
#'
#' CC<-circ.cent.bastri(c1,c2)  #the circumcenter
#' D1<-(B+C)/2; D2<-(A+C)/2; D3<-(A+B)/2;
#' Ds<-rbind(D1,D2,D3)
#'
#' Xlim<-range(Tb[,1],dat[,1])
#' Ylim<-range(Tb[,2],dat[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(A,pch=".",asp=1,xlab="",ylab="",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tb)
#' L<-matrix(rep(CC,3),ncol=2,byrow=TRUE); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#' points(dat)
#' points(rbind(f2v$ext),pch=4,col=2)
#'
#' txt<-rbind(Tb,CC,D1,D2,D3)
#' xc<-txt[,1]+c(-.03,.03,0.02,.07,.06,-.05,.01)
#' yc<-txt[,2]+c(.02,.02,.03,.01,.02,.02,-.04)
#' txt.str<-c("A","B","C","CC","D1","D2","D3")
#' text(xc,yc,txt.str)
#'
#' set.seed(1)
#' dat<-runif.bastri(n,c1,c2)$g
#' fr2vTbVRCC(dat,c1,c2)
#'
#' fr2vTbVRCC(c(.4,.2),c1,c2)
#'
#' dat.fr<-data.frame(a=dat)
#' fr2vTbVRCC(dat.fr,c1,c2)
#'
#' @export
fr2vTbVRCC <- function(Dt,c1,c2,ch.all.intri=FALSE)
{
  if (!is.numeric(as.matrix(Dt)))
  {stop('Dt must be numeric')}

  if (is.point(Dt))
  { Dt<-matrix(Dt,ncol=2)
  } else
  {Dt<-as.matrix(Dt)
  if (ncol(Dt)!=2 )
  {stop('Dt must be of dimension nx2')}
  }

  if (!is.point(c1,1) || !is.point(c2,1))
  {stop('c1 and c2 must be scalars')}

  if (c1<0 || c1>1/2 || c2<=0 || (1-c1)^2+c2^2 >1)
  {stop('c1 must be in [0,1/2], c2>0 and (1-c1)^2+c2^2 <=1')}

  y1<-c(0,0); y2<-c(1,0); y3<-c(c1,c2); tri<-rbind(y1,y2,y3)
  CC<-circ.cent.tri(tri)
  D1<-(y2+y3)/2; D2<-(y1+y3)/2; D3<-(y1+y2)/2;
  Ds<-rbind(D1,D2,D3)
  L<-matrix(rep(CC,3),ncol=3,byrow=TRUE); R<-Ds

  if (ch.all.intri==TRUE)
  {
    if (!inTriAll(Dt,tri,boundary=TRUE))
    {stop('not all points in the data set are in the basic triangle')}
  }

  n<-nrow(Dt)
  mdt<-rep(0,3); U<-matrix(NA,nrow=3,ncol=2);
  for (i in 1:n)
  {
    if (in.triangle(Dt[i,],tri,boundary = TRUE)$in.tri)
    {
      rv<-rv.bastriCC(Dt[i,],c1,c2)$rv;
      if (rv==1)
      {d1<-Dist(Dt[i,],y1);
      if (d1>=mdt[1]) {mdt[1]<-d1; U[1,]<-Dt[i,]};
      } else {
        if (rv==2)
        {d2<-Dist(Dt[i,],y2);
        if (d2>=mdt[2]) {mdt[2]<-d2; U[2,]<-Dt[i,]}
        } else {
          d3<-Dist(Dt[i,],y3);
          if (d3>=mdt[3]) {mdt[3]<-d3; U[3,]<-Dt[i,]}
        }}
    }
  }
  row.names(tri)<-c("A","B","C")  #vertex labeling
  typ<-paste("Furthest Points from Vertices in CC-Vertex Regions in the Basic Triangle with Vertices A=(0,0), B=(1,0), and C=(",c1,",",c2,")",sep="")
  description<-"Furthest Points from Vertices in corresponding CC-Vertex Regions \n (Row i corresponds to vertex i for i=1,2,3)"
  txt1<-"Vertex labels are A=1, B=2, and C=3 (correspond to row number in Extremum Points)"
  txt2<-"Distances from furthest points to vertices in each vertex region\n (Row i corresponds to region of vertex i for i=1,2,3)"
  main.txt<-"Furthest Points from Vertices\n in CC-Vertex Regions"

  Dis<-rbind(ifelse(!is.na(U[1,1]),mdt[1],NA),ifelse(!is.na(U[2,1]),mdt[2],NA),ifelse(!is.na(U[3,1]),mdt[3],NA))
  #distances of the furthest points to the vertices
  Regs<-list(vr1=rbind(y1,D3,CC,D2), #regions inside the triangles
             vr2=rbind(y2,D1,CC,D3),
             vr3=rbind(y3,D2,CC,D1)
  )
  Reg.Cent<-vector()
  for (i in 1:length(Regs))
  { Reg.Cent<-rbind(Reg.Cent,apply(Regs[[i]],2,mean))}

  Reg.names<-c("vr=1","vr=2","vr=3")  #regions names

  res<-list(
    txt1=txt1, txt2=txt2,
    type=typ, desc=description,
    mtitle=main.txt, #main label in the plot
    ext=U, #furthest points from vertices in each vertex region
    X=Dt, num.points=n, #data points and its size
    ROI=tri, # region of interest for X points
    cent=CC, ncent="CC", #center and center name
    regions=Regs, region.names=Reg.names, region.centers=Reg.Cent,
    dist2ref=Dis #distances of furthest points to vertices in each vertex region
  )

  class(res)<-"Extrema"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

#' @title The furthest points in a data set from vertices in each \eqn{CC}-vertex region in a triangle
#'
#' @description
#' An object of class \code{"Extrema"}.
#' Returns the furthest data points among the data set, \code{Dt}, in each \eqn{CC}-vertex region from the vertex in the
#' triangle, \code{tri}\eqn{=T(A,B,C)}. Vertex region labels/numbers correspond to the row number of the vertex in \code{tri}.
#' \code{ch.all.intri} is for checking whether all data points are inside \code{tri} (default is \code{FALSE}).
#'
#' If some of the data points are not inside \code{tri} and \code{ch.all.intri=TRUE}, then the function yields
#' an error message.
#' If some of the data points are not inside \code{tri} and \code{ch.all.intri=FALSE}, then the function yields
#' the closest points to edges among the data points inside \code{tri} (yields \code{NA} if there are no data points
#' inside \code{tri}).
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:mcap2012;textual}{pcds}).
#'
#' @param Dt A set of 2D points representing the set of data points.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param ch.all.intri A logical argument (default=\code{FALSE}) to check whether all data points are inside
#' the triangle \code{tri}. So, if it is \code{TRUE},
#' the function checks if all data points are inside the closure of the triangle (i.e., interior and boundary
#' combined) else it does not.
#'
#' @return A \code{list} with the elements
#' \item{txt1}{Vertex labels are \eqn{A=1}, \eqn{B=2}, and \eqn{C=3} (corresponds to row number in Extremum Points).}
#' \item{txt2}{A short description of the distances as \code{"Distances from furthest points to ..."}.}
#' \item{type}{Type of the extrema points}
#' \item{desc}{A short description of the extrema points}
#' \item{mtitle}{The \code{"main"} title for the plot of the extrema}
#' \item{ext}{The extrema points, here, furthest points from vertices in each \eqn{CC}-vertex region in the
#' triangle \code{tri}.}
#' \item{X}{The input data, \code{Dt}, can be a \code{matrix} or \code{data frame}}
#' \item{num.points}{The number of data points, i.e., size of \code{Dt}}
#' \item{supp}{Support of the data points, here, it is the triangle \code{tri} for this function.}
#' \item{cent}{The center point used for construction of edge regions.}
#' \item{ncent}{Name of the center, \code{cent}, it is circumcenter \code{"CC"} for this function}
#' \item{regions}{CC-Vertex regions inside the triangle, \code{tri}, provided as a list}
#' \item{region.names}{Names of the vertex regions as \code{"vr=1"}, \code{"vr=2"}, and \code{"vr=3"}}
#' \item{region.centers}{Centers of mass of the vertex regions inside \code{tri}}
#' \item{dist2ref}{Distances from furthest points in each vertex region to the corresponding vertex}
#'
#' @seealso \code{\link{fr2vTbVRCC}}, \code{\link{fr2eTeER}}, \code{\link{Kfr2vTbVRCC}} and \code{\link{Kfr2vVRCC}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#' n<-10  #try also n<-20
#'
#' set.seed(1)
#' dat<-runif.tri(n,Tr)$g
#'
#' Ext<-fr2vVRCC(dat,Tr)
#' Ext
#' summary(Ext)
#' plot(Ext)
#'
#' fr2vVRCC(dat[1,],Tr)
#' f2v<-fr2vVRCC(dat,Tr)
#'
#' CC<-circ.cent.tri(Tr)  #the circumcenter
#' D1<-(B+C)/2; D2<-(A+C)/2; D3<-(A+B)/2;
#' Ds<-rbind(D1,D2,D3)
#'
#' Xlim<-range(Tr[,1],dat[,1])
#' Ylim<-range(Tr[,2],dat[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(Tr,xlab="",asp=1,ylab="",pch=".",axes=TRUE,xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tr)
#' L<-matrix(rep(CC,3),ncol=2,byrow=TRUE); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#' points(dat)
#' points(rbind(f2v$ext),pch=4,col=2)
#'
#' txt<-rbind(Tr,CC,Ds)
#' xc<-txt[,1]+c(-.06,.08,.05,.12,-.1,-.1,-.09)
#' yc<-txt[,2]+c(.02,-.02,.05,.0,.02,.06,-.04)
#' txt.str<-c("A","B","C","CC","D1","D2","D3")
#' text(xc,yc,txt.str)
#'
#' fr2vVRCC(dat,Tr)
#'
#' fr2vVRCC(c(1.4,1.2),Tr)
#'
#' dat.fr<-data.frame(a=dat)
#' fr2vVRCC(dat.fr,Tr)
#'
#' dat.fr<-data.frame(a=Tr)
#' fr2vVRCC(dat,dat.fr)
#'
#' dat2<-rbind(dat,c(.2,.4))
#' fr2vVRCC(dat2,Tr)
#'
#' fr2vVRCC(dat2,Tr,ch.all.intri = FALSE)
#' #gives an error message if ch.all.intri = TRUE
#' #since not all points in the data set are in the triangle
#'
#'
#' @export fr2vVRCC
fr2vVRCC <- function(Dt,tri,ch.all.intri=FALSE)
{
  if (!is.numeric(as.matrix(Dt)))
  {stop('Dt must be numeric')}

  if (is.point(Dt))
  { Dt<-matrix(Dt,ncol=2)
  } else
  {Dt<-as.matrix(Dt)
  if (ncol(Dt)!=2 )
  {stop('Dt must be of dimension nx2')}
  }

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  y1<-tri[1,]; y2<-tri[2,]; y3<-tri[3,];
  CC<-circ.cent.tri(tri)
  D1<-(y2+y3)/2; D2<-(y1+y3)/2; D3<-(y1+y2)/2;
  Ds<-rbind(D1,D2,D3)
  L<-matrix(rep(CC,3),ncol=2,byrow=TRUE); R<-Ds
  mdt<-rep(0,3); U<-matrix(NA,nrow=3,ncol=2);

  n<-nrow(Dt)

  if (ch.all.intri==TRUE)
  {
    if (!inTriAll(Dt,tri,boundary=TRUE))
    {stop('not all points in the data set are in the triangle')}
  }

  for (i in 1:n)
  {
    if (in.triangle(Dt[i,],tri,boundary = TRUE)$in.tri)
    {rv<-rv.triCC(Dt[i,],tri)$rv;
    if (rv==1)
    {d1<-Dist(Dt[i,],y1);
    if (d1>=mdt[1]) {mdt[1]<-d1; U[1,]<-Dt[i,]};
    } else {
      if (rv==2)
      {d2<-Dist(Dt[i,],y2);
      if (d2>=mdt[2]) {mdt[2]<-d2; U[2,]<-Dt[i,]}
      } else {
        d3<-Dist(Dt[i,],y3);
        if (d3>=mdt[3]) {mdt[3]<-d3; U[3,]<-Dt[i,]}
      }}
    }
  }

  row.names(tri)<-c("A","B","C")  #vertex labeling
  typ<-"Furthest Points from Vertices in CC-Vertex Regions in the Triangle with Vertices A, B, and C "
  description<-"Furthest Points from Vertices in corresponding CC-Vertex Regions \n (Row i corresponds to edge i for i=1,2,3)  "
  txt1<-"Vertex labels are A=1, B=2, and C=3 (corresponds to row number in Extremum Points)"
  txt2<-"Distances to Vertices"
  txt2<-"Distances from furthest points to vertices in each vertex region\n (Row i corresponds to vertex i for i=1,2,3)"
  main.txt<-"Furthest Points from Vertices\n in CC-Vertex Regions"

  Dis<-rbind(ifelse(!is.na(U[1,1]),mdt[1],NA),ifelse(!is.na(U[2,1]),mdt[2],NA),ifelse(!is.na(U[3,1]),mdt[3],NA))
  #distances of the furthest points to the vertices
  Regs<-list(vr1=rbind(y1,D3,CC,D2), #regions inside the triangles
             vr2=rbind(y2,D1,CC,D3),
             vr3=rbind(y3,D2,CC,D1)
  )
  Reg.Cent<-vector()
  for (i in 1:length(Regs))
  { Reg.Cent<-rbind(Reg.Cent,apply(Regs[[i]],2,mean))}

  Reg.names<-c("vr=1","vr=2","vr=3")  #regions names

  res<-list(
    txt1=txt1, txt2=txt2,
    type=typ, desc=description,
    mtitle=main.txt, #main label in the plot
    ext=U, #furthest points from vertices in each vertex region
    X=Dt, num.points=n, #data points and its size
    ROI=tri, # region of interest for X points
    cent=CC, ncent="CC", #center and center name
    regions=Regs, region.names=Reg.names, region.centers=Reg.Cent,
    dist2ref=Dis #distances of furthest points to vertices in each vertex region
  )

  class(res)<-"Extrema"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

#' @title The \code{k} furthest points from vertices in each \eqn{CC}-vertex region in a basic triangle
#'
#' @description
#' An object of class \code{"Extrema"}.
#' Returns the \code{k} furthest data points among the data set, \code{Dt}, in each \eqn{CC}-vertex region from the vertex in the
#' basic triangle \eqn{T_b=T(A=(0,0),B=(1,0),C=(c_1,c_2))}.
#'
#' Any given triangle can be mapped to the basic triangle by a combination of rigid body motions
#' (i.e., translation, rotation and reflection) and scaling, preserving uniformity of the points in the
#' original triangle. Hence basic triangle is useful for simulation
#' studies under the uniformity hypothesis.
#'
#' \code{ch.all.intri} is for checking whether all data points are inside \eqn{T_b} (default is \code{FALSE}).
#' In the extrema, \eqn{ext}, in the output, the first \code{k} entries are the \code{k} furthest points from vertex 1,
#' second \code{k} entries are \code{k} furthest points are from vertex 2, and
#' last \code{k} entries are the \code{k} furthest points from vertex 3
#'  If data size does not allow, \code{NA}'s are inserted for some or all of the \code{k} furthest points for each vertex.
#'
#' @param Dt A set of 2D points representing the set of data points.
#' @param c1,c2 Positive real numbers which constitute the vertex of the basic triangle.
#' adjacent to the shorter edges; \eqn{c_1} must be in \eqn{[0,1/2]}, \eqn{c_2>0} and \eqn{(1-c_1)^2+c_2^2 \le 1}
#' @param k A positive integer. \code{k} furthest data points in each \eqn{CC}-vertex region are to be found if exists, else
#' \code{NA} are provided for (some of) the \code{k} furthest points.
#' @param ch.all.intri A logical argument for checking whether all data points are inside \eqn{T_b}
#' (default is \code{FALSE}).
#'
#' @return A \code{list} with the elements
#' \item{txt1}{Vertex labels are \eqn{A=1}, \eqn{B=2}, and \eqn{C=3} (corresponds to row number in Extremum Points).}
#' \item{txt2}{A shorter description of the distances as \code{"Distances of k furthest points in the vertex regions to Vertices"}.}
#' \item{type}{Type of the extrema points}
#' \item{desc}{A short description of the extrema points}
#' \item{mtitle}{The \code{"main"} title for the plot of the extrema}
#' \item{ext}{The extrema points, here, \code{k} furthest points from vertices in each vertex region.}
#' \item{X}{The input data, \code{Dt}, can be a \code{matrix} or \code{data frame}}
#' \item{num.points}{The number of data points, i.e., size of \code{Dt}}
#' \item{supp}{Support of the data points, here, it is \eqn{T_b}.}
#' \item{cent}{The center point used for construction of edge regions.}
#' \item{ncent}{Name of the center, \code{cent}, it is circumcenter \code{"CC"} for this function.}
#' \item{regions}{Vertex regions inside the triangle, \eqn{T_b}, provided as a list.}
#' \item{region.names}{Names of the vertex regions as \code{"vr=1"}, \code{"vr=2"}, and \code{"vr=3"}}
#' \item{region.centers}{Centers of mass of the vertex regions inside \eqn{T_b}.}
#' \item{dist2ref}{Distances from \code{k} furthest points in each vertex region to the corresponding vertex
#' (each row representing a vertex).}
#'
#' @seealso \code{\link{fr2vTbVRCC}}, \code{\link{fr2vVRCC}}, \code{\link{fr2eTeER}}, and \code{\link{Kfr2vVRCC}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' c1<-.4; c2<-.6;
#' A<-c(0,0); B<-c(1,0); C<-c(c1,c2);
#' Tb<-rbind(A,B,C)
#' n<-20
#' k<-3
#'
#' set.seed(1)
#' dat<-runif.bastri(n,c1,c2)$g
#'
#' Ext<-Kfr2vTbVRCC(dat,c1,c2,k)
#' Ext
#' summary(Ext)
#' plot(Ext)
#'
#' Kfr2vTbVRCC(dat[1:k,],c1,c2,k)
#' Kfr2vTbVRCC(dat[1,],c1,c2,k)
#'
#' kf2v<-Kfr2vTbVRCC(dat,c1,c2,k)
#'
#' CC<-circ.cent.bastri(c1,c2)  #the circumcenter
#' D1<-(B+C)/2; D2<-(A+C)/2; D3<-(A+B)/2;
#' Ds<-rbind(D1,D2,D3)
#'
#' Xlim<-range(Tb[,1],dat[,1])
#' Ylim<-range(Tb[,2],dat[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(A,pch=".",asp=1,xlab="",ylab="",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tb)
#' L<-matrix(rep(CC,3),ncol=2,byrow=TRUE); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#' points(dat)
#' points(kf2v$ext,pch=4,col=2)
#'
#' txt<-rbind(Tb,CC,Ds)
#' xc<-txt[,1]+c(-.03,.03,.02,.07,.06,-.05,.01)
#' yc<-txt[,2]+c(.02,.02,.03,-.02,.02,.03,-.04)
#' txt.str<-c("A","B","C","CC","D1","D2","D3")
#' text(xc,yc,txt.str)
#'
#' Kfr2vTbVRCC(dat,c1,c2,k)
#'
#' Kfr2vTbVRCC(c(.4,.2),c1,c2,k)
#'
#' dat.fr<-data.frame(a=dat)
#' Kfr2vTbVRCC(dat.fr,c1,c2,k)
#'
#' dat2<-rbind(dat,c(.2,.4))
#' Kfr2vTbVRCC(dat2,c1,c2,k)
#'
#' @export
Kfr2vTbVRCC <- function(Dt,c1,c2,k,ch.all.intri=FALSE)
{
  if (!is.numeric(as.matrix(Dt)))
  {stop('Dt must be numeric')}

  if (is.point(Dt))
  { Dt<-matrix(Dt,ncol=2)
  } else
  {Dt<-as.matrix(Dt)
  if (ncol(Dt)!=2 )
  {stop('Dt must be of dimension nx2')}
  }

  if (!is.point(c1,1) || !is.point(c2,1))
  {stop('c1 and c2 must be scalars')}

  if (c1<0 || c1>1/2 || c2<=0 || (1-c1)^2+c2^2 >1)
  {stop('c1 must be in [0,1/2], c2>0 and (1-c1)^2+c2^2 <=1')}

  y1<-c(0,0); y2<-c(1,0); y3<-c(c1,c2); Tb<-rbind(y1,y2,y3)
  CC<-circ.cent.tri(Tb)
  D1<-(y2+y3)/2; D2<-(y1+y3)/2; D3<-(y1+y2)/2;
  Ds<-rbind(D1,D2,D3)
  L<-matrix(rep(CC,3),ncol=2,byrow=TRUE); R<-Ds

  if (ch.all.intri==TRUE)
  {
    if (!inTriAll(Dt,Tb,boundary=TRUE))
    {stop('not all points in the data set are in the basic triangle')}
  }

  U1<-U2<-U3<-matrix(NA,nrow=k,ncol=2)
  Dis1<-Dis2<-Dis3<-rep(NA,k)

  n<-nrow(Dt);
  rv<-rep(0,n);
  for (i in 1:n)
  {rv[i]<-rv.bastriCC(Dt[i,],c1,c2)$rv};

  Dt1<-matrix(Dt[rv==1,],ncol=2);
  Dt2<-matrix(Dt[rv==2,],ncol=2);
  Dt3<-matrix(Dt[rv==3,],ncol=2);

  n1<-nrow(Dt1); n2<-nrow(Dt2); n3<-nrow(Dt3);

  if (n1>0)
  {
    Dis1<-rep(0,n1)
    for (i in 1:n1)
    { Dis1[i]<-Dist(Dt1[i,],y1) }

    ord1<-order(-Dis1)
    K1<-min(k,n1)
    U1[1:K1,]<-Dt1[ord1[1:K1],]
  }

  if (n2>0)
  {
    Dis2<-rep(0,n2)
    for (i in 1:n2)
    { Dis2[i]<-Dist(Dt2[i,],y2) }

    ord2<-order(-Dis2)
    K2<-min(k,n2)
    U2[1:K2,]<-Dt2[ord2[1:K2],]
  }

  if (n3>0)
  {
    Dis3<-rep(0,n3)
    for (i in 1:n3)
    { Dis3[i]<-Dist(Dt3[i,],y3) }
    ord3<-order(-Dis3)
    K3<-min(k,n3)
    U3[1:K3,]<-Dt3[ord3[1:K3],]
  }
  U<-rbind(U1,U2,U3)

  row.names(Tb)<-c("A","B","C")  #vertex labeling
  rn1<-rn2<-rn3<-vector()
  for (i in 1:k) {rn1<-c(rn1,paste(i,". furthest from vertex 1",sep=""));
  rn2<-c(rn2,paste(i,". furthest from vertex 2",sep=""));
  rn3<-c(rn3,paste(i,". furthest from vertex 3",sep=""))}
  row.names(U)<-c(rn1,rn2,rn3)  #extrema labeling

  typ<-paste(k, "Furthest Points from Vertices in CC-Vertex Regions in the Basic Triangle with Vertices A=(0,0), B=(1,0), and C=(",c1,",",c2,")",sep="")
  description<-paste(k, "Furthest Points from Vertices in Corresponding CC-Vertex Regions \n (Row i corresponds to vertex i for i=1,2,3)")
  txt1<-"Vertex labels are A=1, B=2, and C=3 (corresponds to row number in Extremum Points)"
  txt2<-paste("Distances from ",k,"furthest points to vertices in each vertex region\n (Row i corresponds to region of vertex i for i=1,2,3)",sep="")
  main.txt<-paste(k, "Furthest Points from Vertices\n in CC-Vertex Regions")

  Dis<-rbind(Dis1[1:k],Dis2[1:k],Dis3[1:k])
  #distances of the furthest points to the vertices
  Regs<-list(vr1=rbind(y1,D3,CC,D2), #regions inside the triangles
             vr2=rbind(y2,D1,CC,D3),
             vr3=rbind(y3,D2,CC,D1)
  )
  Reg.Cent<-vector()
  for (i in 1:length(Regs))
  { Reg.Cent<-rbind(Reg.Cent,apply(Regs[[i]],2,mean))}

  Reg.names<-c("vr=1","vr=2","vr=3")  #regions names

  res<-list(
    txt1=txt1, txt2=txt2,
    type=typ, desc=description,
    mtitle=main.txt, #main label in the plot
    ext=U, #k furthest points from vertices in each vertex region
    X=Dt, num.points=n, #data points and its size
    ROI=Tb, # region of interest for X points
    cent=CC, ncent="CC", #center and center name
    regions=Regs, region.names=Reg.names, region.centers=Reg.Cent,
    dist2ref=Dis #distances of k furthest points to vertices in each vertex region (earh row corresponds to a vertex)
  )

  class(res)<-"Extrema"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

#' @title The \code{k} furthest points in a data set from vertices in each \eqn{CC}-vertex region in a triangle
#'
#' @description
#' An object of class \code{"Extrema"}.
#' Returns the \code{k} furthest data points among the data set, \code{Dt}, in each \eqn{CC}-vertex region from the vertex in the
#' triangle, \code{tri}\eqn{=T(A,B,C)}, vertices are stacked row-wise. Vertex region labels/numbers correspond to the
#' row number of the vertex in \code{tri}.
#'
#' \code{ch.all.intri} is for checking whether all data points are inside \code{tri} (default is \code{FALSE}).
#' If some of the data points are not inside \code{tri} and \code{ch.all.intri=TRUE}, then the function yields
#' an error message.
#' If some of the data points are not inside \code{tri} and \code{ch.all.intri=FALSE}, then the function yields
#' the closest points to edges among the data points inside \code{tri} (yields \code{NA} if there are no data points
#' inside \code{tri}).
#'
#' In the extrema, \eqn{ext}, in the output, the first \code{k} entries are the \code{k} furthest points from vertex 1,
#' second \code{k} entries are \code{k} furthest points are from vertex 2, and
#' last \code{k} entries are the \code{k} furthest points from vertex 3.
#' If data size does not allow, \code{NA}'s are inserted for some or all of the \code{k} furthest points for each vertex.
#'
#' @param Dt A set of 2D points representing the set of data points.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param k A positive integer. \code{k} furthest data points in each \eqn{CC}-vertex region are to be found if exists, else
#' \code{NA} are provided for (some of) the \code{k} furthest points.
#' @param ch.all.intri A logical argument (default=\code{FALSE}) to check whether all data points are inside
#' the triangle \code{tri}. So, if it is \code{TRUE},
#' the function checks if all data points are inside the closure of the triangle (i.e., interior and boundary
#' combined) else it does not.
#'
#' @return A \code{list} with the elements
#' \item{txt1}{Vertex labels are \eqn{A=1}, \eqn{B=2}, and \eqn{C=3} (corresponds to row number in Extremum Points).}
#' \item{txt2}{A shorter description of the distances as \code{"Distances of k furthest points in the vertex regions
#'  to Vertices"}.}
#' \item{type}{Type of the extrema points}
#' \item{desc}{A short description of the extrema points}
#' \item{mtitle}{The \code{"main"} title for the plot of the extrema}
#' \item{ext}{The extrema points, here, \code{k} furthest points from vertices in each \eqn{CC}-vertex region in
#' the triangle \code{tri}.}
#' \item{X}{The input data, \code{Dt}, can be a \code{matrix} or \code{data frame}}
#' \item{num.points}{The number of data points, i.e., size of \code{Dt}}
#' \item{supp}{Support of the data points, it is \code{tri} for this function.}
#' \item{cent}{The center point used for construction of vertex regions}
#' \item{ncent}{Name of the center, \code{cent}, it is circumcenter \code{"CC"} for this function.}
#' \item{regions}{Vertex regions inside the triangle, \code{tri}, provided as a list}
#' \item{region.names}{Names of the vertex regions as \code{"vr=1"}, \code{"vr=2"}, and \code{"vr=3"}}
#' \item{region.centers}{Centers of mass of the vertex regions inside \eqn{T_b}.}
#' \item{dist2ref}{Distances from \code{k} furthest points in each vertex region to the corresponding vertex
#' (each row representing a vertex in \code{tri}). Among the distances the first \code{k} entries are the distances
#' from the \code{k} furthest points from vertex 1 to vertex 1, second \code{k} entries are distances from the \code{k} furthest
#' points from vertex 2 to vertex 2, and the last \code{k} entries are the distances from the \code{k} furthest points
#' from vertex 3 to vertex 3.}
#'
#' @seealso \code{\link{Kfr2vTbVRCC}}, \code{\link{fr2vTbVRCC}}, \code{\link{fr2vVRCC}}, and \code{\link{fr2eTeER}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#' n<-10  #try also n<-20
#' k<-3
#'
#' set.seed(1)
#' dat<-runif.tri(n,Tr)$g
#'
#' Ext<-Kfr2vVRCC(dat,Tr,k)
#' Ext
#' summary(Ext)
#' plot(Ext)
#'
#' Kfr2vVRCC(dat[1:k,],Tr,k)
#' Kfr2vVRCC(dat[1,],Tr,k)
#'
#' dat2<-rbind(dat,c(.2,.4))
#' Kfr2vVRCC(dat2,Tr,k)  #try also Kfr2vVRCC(dat2,Tr,k,ch.all.intri = TRUE)
#'
#' kf2v<-Kfr2vVRCC(dat,Tr,k)
#'
#' CC<-circ.cent.tri(Tr)  #the circumcenter
#' D1<-(B+C)/2; D2<-(A+C)/2; D3<-(A+B)/2;
#' Ds<-rbind(D1,D2,D3)
#'
#' Xlim<-range(Tr[,1],dat[,1])
#' Ylim<-range(Tr[,2],dat[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(A,pch=".",asp=1,xlab="",ylab="",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tr)
#' L<-matrix(rep(CC,3),ncol=2,byrow=TRUE); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#' points(dat)
#' points(kf2v$ext,pch=4,col=2)
#'
#' txt<-rbind(Tr,CC,Ds)
#' xc<-txt[,1]+c(-.06,.08,.05,.12,-.1,-.1,-.09)
#' yc<-txt[,2]+c(.02,-.02,.04,.0,.02,.06,-.04)
#' txt.str<-c("A","B","C","CC","D1","D2","D3")
#' text(xc,yc,txt.str)
#'
#' Kfr2vVRCC(dat,Tr,k)
#'
#' dat.fr<-data.frame(a=dat)
#' Kfr2vVRCC(dat.fr,Tr,k)
#'
#' dat.fr<-data.frame(a=Tr)
#' fr2vVRCC(dat,dat.fr)
#'
#' @export Kfr2vVRCC
Kfr2vVRCC <- function(Dt,tri,k,ch.all.intri=FALSE)
{
  if (!is.numeric(as.matrix(Dt)))
  {stop('Dt must be numeric')}

  if (is.point(Dt))
  { Dt<-matrix(Dt,ncol=2)
  } else
  {Dt<-as.matrix(Dt)
  if (ncol(Dt)!=2 )
  {stop('Dt must be of dimension nx2')}
  }

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  y1<-tri[1,]; y2<-tri[2,]; y3<-tri[3,];
  CC<-circ.cent.tri(tri)
  D1<-(y2+y3)/2; D2<-(y1+y3)/2; D3<-(y1+y2)/2;
  Ds<-rbind(D1,D2,D3)
  L<-matrix(rep(CC,3),ncol=2,byrow=TRUE); R<-Ds

  if (ch.all.intri==TRUE)
  {
    if (!inTriAll(Dt,tri,boundary=TRUE))
    {stop('not all points in the data set are in the triangle')}
  }

  U1<-U2<-U3<-matrix(NA,nrow=k,ncol=2)
  Dis1<-Dis2<-Dis3<-rep(NA,k)

  n<-nrow(Dt)
  rv<-rep(0,n);

  for (i in 1:n)
  {rv[i]<-rv.triCC(Dt[i,],tri)$rv};

  Dt1<-matrix(Dt[rv==1,],ncol=2);
  Dt2<-matrix(Dt[rv==2,],ncol=2);
  Dt3<-matrix(Dt[rv==3,],ncol=2);

  n1<-nrow(Dt1); n2<-nrow(Dt2); n3<-nrow(Dt3);

  if (n1>0)
  {
    Dis1<-rep(0,n1)
    for (i in 1:n1)
    { Dis1[i]<-Dist(Dt1[i,],y1) }

    ord1<-order(-Dis1)
    K1<-min(k,n1)
    U1[1:K1,]<-Dt1[ord1[1:K1],]
  }

  if (n2>0)
  {
    Dis2<-rep(0,n2)
    for (i in 1:n2)
    { Dis2[i]<-Dist(Dt2[i,],y2) }

    ord2<-order(-Dis2)
    K2<-min(k,n2)
    U2[1:K2,]<-Dt2[ord2[1:K2],]
  }

  if (n3>0)
  {
    Dis3<-rep(0,n3)
    for (i in 1:n3)
    { Dis3[i]<-Dist(Dt3[i,],y3) }
    ord3<-order(-Dis3)
    K3<-min(k,n3)
    U3[1:K3,]<-Dt3[ord3[1:K3],]
  }
  U<-rbind(U1,U2,U3)

  row.names(tri)<-c("A","B","C")  #vertex labeling
  rn1<-rn2<-rn3<-vector()
  for (i in 1:k) {rn1<-c(rn1,paste(i,". furthest from vertex 1",sep=""));
  rn2<-c(rn2,paste(i,". furthest from vertex 2",sep=""));
  rn3<-c(rn3,paste(i,". furthest from vertex 3",sep=""))}
  row.names(U)<-c(rn1,rn2,rn3)  #extrema labeling

  typ<-paste(k, "Furthest Points from Vertices in CC-Vertex Regions in the Triangle with Vertices A, B, and C")
  description<-paste(k, "Furthest Points from Vertices in Respective CC-Vertex Regions \n (Row i corresponds to vertex i for i=1,2,3)")
  txt1<-"Vertex labels are A=1, B=2, and C=3 (corresponds to row number in Extremum Points)"
  txt2<-paste("Distances from ",k," furthest points to vertices in each vertex region\n (Row i corresponds to region of vertex i for i=1,2,3)",sep="")
  main.txt<-paste(k, "Furthest Points from Vertices\n in CC-Vertex Regions")

  Dis<-rbind(Dis1[1:k],Dis2[1:k],Dis3[1:k])
  #distances of the furthest points to the vertices
  Regs<-list(vr1=rbind(y1,D3,CC,D2), #regions inside the triangles
             vr2=rbind(y2,D1,CC,D3),
             vr3=rbind(y3,D2,CC,D1)
  )
  Reg.Cent<-vector()
  for (i in 1:length(Regs))
  { Reg.Cent<-rbind(Reg.Cent,apply(Regs[[i]],2,mean))}

  Reg.names<-c("vr=1","vr=2","vr=3")  #regions names

  res<-list(
    txt1=txt1, txt2=txt2,
    type=typ, desc=description,
    mtitle=main.txt, #main label in the plot
    ext=U, #k furthest points from vertices in each vertex region
    X=Dt, num.points=n, #data points and its size
    ROI=tri, # region of interest for X points
    cent=CC, ncent="CC", #center and center name
    regions=Regs, region.names=Reg.names, region.centers=Reg.Cent,
    dist2ref=Dis #distances of k furthest points to vertices in each vertex region (each row corresponds to a vertex)
  )

  class(res)<-"Extrema"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

#' @title The closest points to circumcenter in each \eqn{CC}-vertex region in a basic triangle
#'
#' @description
#' An object of class \code{"Extrema"}.
#' Returns the closest data points among the data set, \code{Dt}, to circumcenter, \eqn{CC}, in each \eqn{CC}-vertex region
#' in the basic triangle \eqn{T_b=T(A=(0,0),B=(1,0),C=(c_1,c_2))=}(vertex 1,vertex 2,vertex 3).
#' \code{ch.all.intri} is for checking whether all data points are inside \eqn{T_b} (default is \code{FALSE}).
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:mcap2012;textual}{pcds}).
#'
#' @param Dt A set of 2D points representing the set of data points.
#' @param c1,c2 Positive real numbers which constitute the vertex of the basic triangle.
#' adjacent to the shorter edges; \eqn{c_1} must be in \eqn{[0,1/2]}, \eqn{c_2>0} and \eqn{(1-c_1)^2+c_2^2 \le 1}
#' @param ch.all.intri A logical argument for checking whether all data points are inside \eqn{T_b}
#' (default is \code{FALSE}).
#'
#' @return A \code{list} with the elements
#' \item{txt1}{Vertex labels are \eqn{A=1}, \eqn{B=2}, and \eqn{C=3} (corresponds to row number in Extremum Points).}
#' \item{txt2}{A short description of the distances as \code{"Distances from closest points to ..."}.}
#' \item{type}{Type of the extrema points}
#' \item{mtitle}{The \code{"main"} title for the plot of the extrema}
#' \item{ext}{The extrema points, here, closest points to \eqn{CC} in each vertex region.}
#' \item{X}{The input data, \code{Dt}, can be a \code{matrix} or \code{data frame}}
#' \item{num.points}{The number of data points, i.e., size of \code{Dt}}
#' \item{supp}{Support of the data points, here, it is \eqn{T_b}.}
#' \item{cent}{The center point used for construction of vertex regions}
#' \item{ncent}{Name of the center, \code{cent}, it is \code{"CC"} for this function.}
#' \item{regions}{Vertex regions inside the triangle, \eqn{T_b}, provided as a list.}
#' \item{region.names}{Names of the vertex regions as \code{"vr=1"}, \code{"vr=2"}, and \code{"vr=3"}}
#' \item{region.centers}{Centers of mass of the vertes regions inside \eqn{T_b}.}
#' \item{dist2ref}{Distances from closest points in each vertex region to CC.}
#'
#' @seealso \code{\link{cl2CC.VR}}, \code{\link{cl2eTbVRcent}}, \code{\link{cl2eVRcent}}, \code{\link{cl2eVRCM}}
#' and \code{\link{fr2eTeER}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' c1<-.4; c2<-.6;
#' A<-c(0,0); B<-c(1,0); C<-c(c1,c2);
#' Tb<-rbind(A,B,C)
#' n<-20
#'
#' set.seed(1)
#' dat<-runif.bastri(n,c1,c2)$g
#'
#' Ext<-cl2CC.TbVR(dat,c1,c2)
#' Ext
#' summary(Ext)
#' plot(Ext)
#'
#' cl2CC.TbVR(dat[1,],c1,c2)
#'
#' c2CC<-cl2CC.TbVR(dat,c1,c2)
#'
#' CC<-circ.cent.bastri(c1,c2)  #the circumcenter
#' D1<-(B+C)/2; D2<-(A+C)/2; D3<-(A+B)/2;
#' Ds<-rbind(D1,D2,D3)
#'
#' Xlim<-range(Tb[,1],dat[,1])
#' Ylim<-range(Tb[,2],dat[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(A,pch=".",asp=1,xlab="",ylab="",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tb)
#' L<-matrix(rep(CC,3),ncol=2,byrow=TRUE); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#' points(dat)
#' points(c2CC$ext,pch=4,col=2)
#'
#' txt<-rbind(Tb,CC,Ds)
#' xc<-txt[,1]+c(-.03,.03,.02,.07,.06,-.05,.01)
#' yc<-txt[,2]+c(.02,.02,.03,-.01,.03,.03,-.04)
#' txt.str<-c("A","B","C","CC","D1","D2","D3")
#' text(xc,yc,txt.str)
#'
#' cl2CC.TbVR(dat,c1,c2)
#'
#' cl2CC.TbVR(c(.4,.2),c1,c2)
#'
#' dat.fr<-data.frame(a=dat)
#' cl2CC.TbVR(dat.fr,c1,c2)
#'
#' dat2<-rbind(dat,c(.2,.4))
#' cl2CC.TbVR(dat2,c1,c2)
#'
#' cl2CC.TbVR(dat2,c1,c2,ch.all.intri = FALSE)
#' #gives an error message if ch.all.intri = TRUE
#' #since not all points are in the basic triangle
#'
#' @export
cl2CC.TbVR <- function(Dt,c1,c2,ch.all.intri=FALSE)
{
  if (!is.numeric(as.matrix(Dt)))
  {stop('Dt must be numeric')}

  if (is.point(Dt))
  { Dt<-matrix(Dt,ncol=2)
  } else
  {Dt<-as.matrix(Dt)
  if (ncol(Dt)!=2 )
  {stop('Dt must be of dimension nx2')}
  }

  if (!is.point(c1,1) || !is.point(c2,1))
  {stop('c1 and c2 must be scalars')}

  if (c1<0 || c1>1/2 || c2<=0 || (1-c1)^2+c2^2 >1)
  {stop('c1 must be in [0,1/2], c2>0 and (1-c1)^2+c2^2 <=1')}

  y1<-c(0,0); y2<-c(1,0); y3<-c(c1,c2); Tb<-rbind(y1,y2,y3)
  CC<-circ.cent.bastri(c1,c2)
  D1<-(y2+y3)/2; D2<-(y1+y3)/2; D3<-(y1+y2)/2;
  Ds<-rbind(D1,D2,D3)
  L<-matrix(rep(CC,3),ncol=2,byrow=TRUE); R<-Ds

  if (ch.all.intri==TRUE)
  {
    if (!inTriAll(Dt,Tb,boundary=TRUE))
    {stop('not all points in the data set are in the basic triangle')}
  }

  mdt<-c(Dist(y1,CC),Dist(y2,CC),Dist(y3,CC)); #distances from the vertices to CC
  U<-matrix(NA,nrow=3,ncol=2);

  n<-nrow(Dt)
  for (i in 1:n)
  {
    if (in.triangle(Dt[i,],Tb,boundary = TRUE)$in.tri)
    {
      rv<-rv.bastriCC(Dt[i,],c1,c2)$rv;
      if (rv==1)
      {d1<-Dist(Dt[i,],CC);
      if (d1<=mdt[1]) {mdt[1]<-d1; U[1,]<-Dt[i,]};
      } else {
        if (rv==2)
        {d2<-Dist(Dt[i,],CC);
        if (d2<=mdt[2]) {mdt[2]<-d2; U[2,]<-Dt[i,]}
        } else {
          d3<-Dist(Dt[i,],CC);
          if (d3<=mdt[3]) {mdt[3]<-d3; U[3,]<-Dt[i,]}
        }}
    }
  }

  row.names(Tb)<-c("A","B","C")  #vertex labeling
  typ<-paste("Closest Points to CC in CC-Vertex Regions in the Basic Triangle with Vertices A=(0,0), B=(1,0), and C=(",c1,",",c2,")",sep="")
  description<-"Closest Points to CC in CC-Vertex Regions \n (Row i corresponds to vertex i for i=1,2,3)"
  txt1<-"Vertex labels are A=1, B=2, and C=3 (corresponds to row number in Extremum Points)"
  txt2<-"Distances from closest points to circumcenter in each vertex region\n (Row i corresponds to vertex i for i=1,2,3)"
  main.txt<-"Closest Points to Circumcenter \n in CC-Vertex Regions"

  Dis<-rbind(ifelse(!is.na(U[1,1]),mdt[1],NA),ifelse(!is.na(U[2,1]),mdt[2],NA),ifelse(!is.na(U[3,1]),mdt[3],NA))
  #distances of the closest points to CC
  Regs<-list(vr1=rbind(y1,D3,CC,D2), #regions inside the triangles
             vr2=rbind(y2,D1,CC,D3),
             vr3=rbind(y3,D2,CC,D1)
  )
  Reg.Cent<-vector()
  for (i in 1:length(Regs))
  { Reg.Cent<-rbind(Reg.Cent,apply(Regs[[i]],2,mean))}

  Reg.names<-c("vr=1","vr=2","vr=3")  #regions names

  res<-list(
    txt1=txt1, txt2=txt2,
    type=typ, desc=description,
    mtitle=main.txt, #main label in the plot
    ext=U, #closest points to CC in each vertex region
    X=Dt, num.points=n, #data points and its size
    ROI=Tb, # region of interest for X points
    cent=CC, ncent="CC", #center and center name
    regions=Regs, region.names=Reg.names, region.centers=Reg.Cent,
    dist2ref=Dis #distances of closest points to CC in each vertex region
  )

  class(res)<-"Extrema"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

#' @title The closest points to circumcenter in each \eqn{CC}-vertex region in a triangle
#'
#' @description
#' An object of class \code{"Extrema"}.
#' Returns the closest data points among the data set, \code{Dt}, to circumcenter, \eqn{CC}, in each \eqn{CC}-vertex region
#' in the triangle \code{tri} \eqn{=T(A,B,C)=}(vertex 1,vertex 2,vertex 3).
#'
#' \code{ch.all.intri} is for checking whether all data points are inside \code{tri} (default is \code{FALSE}).
#' If some of the data points are not inside \code{tri} and \code{ch.all.intri=TRUE}, then the function yields
#' an error message.
#' If some of the data points are not inside \code{tri} and \code{ch.all.intri=FALSE}, then the function yields
#' the closest points to \eqn{CC} among the data points in each \eqn{CC}-vertex region of \code{tri} (yields \code{NA} if
#' there are no data points inside \code{tri}).
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:mcap2012;textual}{pcds}).
#'
#' @param Dt A set of 2D points representing the set of data points.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param ch.all.intri A logical argument (default=\code{FALSE}) to check whether all data points are inside
#' the triangle \code{tri}. So, if it is \code{TRUE},
#' the function checks if all data points are inside the closure of the triangle (i.e., interior and boundary
#' combined) else it does not.
#'
#' @return A \code{list} with the elements
#' \item{txt1}{Vertex labels are \eqn{A=1}, \eqn{B=2}, and \eqn{C=3} (corresponds to row number in Extremum Points).}
#' \item{txt2}{A short description of the distances as \code{"Distances from closest points to CC ..."}}
#' \item{type}{Type of the extrema points}
#' \item{mtitle}{The \code{"main"} title for the plot of the extrema}
#' \item{ext}{The extrema points, here, closest points to \eqn{CC} in each \eqn{CC}-vertex region}
#' \item{X}{The input data, \code{Dt}, can be a \code{matrix} or \code{data frame}}
#' \item{num.points}{The number of data points, i.e., size of \code{Dt}}
#' \item{supp}{Support of the data points, here, it is \code{tri}}
#' \item{cent}{The center point used for construction of vertex regions}
#' \item{ncent}{Name of the center, \code{cent}, it is \code{"CC"} for this function}
#' \item{regions}{Vertex regions inside the triangle, \code{tri}, provided as a list}
#' \item{region.names}{Names of the vertex regions as \code{"vr=1"}, \code{"vr=2"}, and \code{"vr=3"}}
#' \item{region.centers}{Centers of mass of the vertex regions inside \code{tri}}
#' \item{dist2ref}{Distances from closest points in each \eqn{CC}-vertex region to CC.}
#'
#' @seealso \code{\link{cl2CC.TbVR}}, \code{\link{cl2eTbVRcent}}, \code{\link{cl2eVRcent}},
#' \code{\link{cl2eVRCM}} and  \code{\link{fr2eTeER}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#' n<-10  #try also n<-20
#'
#' set.seed(1)
#' dat<-runif.tri(n,Tr)$g
#'
#' Ext<-cl2CC.VR(dat,Tr)
#' Ext
#' summary(Ext)
#' plot(Ext)
#'
#' cl2CC.VR(dat[1,],Tr)
#'
#' c2CC<-cl2CC.VR(dat,Tr)
#'
#' CC<-circ.cent.tri(Tr)  #the circumcenter
#' D1<-(B+C)/2; D2<-(A+C)/2; D3<-(A+B)/2;
#' Ds<-rbind(D1,D2,D3)
#'
#' Xlim<-range(Tr[,1],dat[,1])
#' Ylim<-range(Tr[,2],dat[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(A,pch=".",asp=1,xlab="",ylab="",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tr)
#' points(dat)
#' L<-matrix(rep(CC,3),ncol=2,byrow=TRUE); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#' points(c2CC$ext,pch=4,col=2)
#'
#' txt<-rbind(Tr,CC,Ds)
#' xc<-txt[,1]+c(-.07,.08,.06,.12,-.1,-.1,-.09)
#' yc<-txt[,2]+c(.02,-.02,.03,.0,.02,.06,-.04)
#' txt.str<-c("A","B","C","CC","D1","D2","D3")
#' text(xc,yc,txt.str)
#'
#' cl2CC.VR(dat,Tr)
#'
#' cl2CC.VR(c(1.4,1.2),Tr)
#'
#' dat.fr<-data.frame(a=dat)
#' cl2CC.VR(dat.fr,Tr)
#'
#' dat.fr<-data.frame(a=Tr)
#' cl2CC.VR(dat,dat.fr)
#'
#' dat2<-rbind(dat,c(.2,.4))
#' cl2CC.VR(dat2,Tr)
#'
#' cl2CC.VR(dat2,Tr,ch.all.intri = FALSE)
#' #gives an error message if ch.all.intri = TRUE since not all points are in the triangle
#'
#'
#' @export
cl2CC.VR <- function(Dt,tri,ch.all.intri=FALSE)
{
  if (!is.numeric(as.matrix(Dt)))
  {stop('Dt must be numeric')}

  if (is.point(Dt))
  { Dt<-matrix(Dt,ncol=2)
  } else
  {Dt<-as.matrix(Dt)
  if (ncol(Dt)!=2 )
  {stop('Dt must be of dimension nx2')}
  }

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  y1<-tri[1,]; y2<-tri[2,]; y3<-tri[3,];
  CC<-circ.cent.tri(tri)
  D1<-(y2+y3)/2; D2<-(y1+y3)/2; D3<-(y1+y2)/2;
  Ds<-rbind(D1,D2,D3)
  L<-matrix(rep(CC,3),ncol=2,byrow=TRUE); R<-Ds

  if (ch.all.intri==TRUE)
  {
    if (!inTriAll(Dt,tri,boundary=TRUE))
    {stop('not all points in the data set are in the triangle')}
  }

  mdt<-c(Dist(y1,CC),Dist(y2,CC),Dist(y3,CC)); #distances from the vertices to CC
  U<-matrix(NA,nrow=3,ncol=2);

  n<-nrow(Dt)
  for (i in 1:n)
  {
    if (in.triangle(Dt[i,],tri,boundary = TRUE)$in.tri)
    {rv<-rv.triCC(Dt[i,],tri)$rv;
    if (rv==1)
    {d1<-Dist(Dt[i,],CC);
    if (d1<=mdt[1]) {mdt[1]<-d1; U[1,]<-Dt[i,]};
    } else {
      if (rv==2)
      {d2<-Dist(Dt[i,],CC);
      if (d2<=mdt[2]) {mdt[2]<-d2; U[2,]<-Dt[i,]}
      } else {
        d3<-Dist(Dt[i,],CC);
        if (d3<=mdt[3]) {mdt[3]<-d3; U[3,]<-Dt[i,]}
      }}
    }
  }

  Mdt<-c(ifelse(!is.na(U[1,1]),mdt[1],NA),ifelse(!is.na(U[2,1]),mdt[2],NA),ifelse(!is.na(U[3,1]),mdt[3],NA))

  row.names(tri)<-c("A","B","C")  #vertex labeling
  typ<-"Closest Points to CC in CC-Vertex Regions in the Triangle with Vertices A, B, and C"
  description<-"Closest Points to CC in CC-Vertex Regions \n (Row i corresponds to vertex i for i=1,2,3)"
  txt1<-"Vertex labels are A=1, B=2, and C=3 (corresponds to row number in Extremum Points)"
  txt2<-"Distances from closest points to CC in each CC-vertex region\n (Row i corresponds to vertex i for i=1,2,3)"
  main.txt<-"Closest Points to Circumcenter\n in CC-Vertex Regions"

  Dis<-rbind(ifelse(!is.na(U[1,1]),mdt[1],NA),ifelse(!is.na(U[2,1]),mdt[2],NA),ifelse(!is.na(U[3,1]),mdt[3],NA))
  #distances of the closest points to CC
  Regs<-list(vr1=rbind(y1,D3,CC,D2), #regions inside the triangles
             vr2=rbind(y2,D1,CC,D3),
             vr3=rbind(y3,D2,CC,D1)
  )
  Reg.Cent<-vector()
  for (i in 1:length(Regs))
  { Reg.Cent<-rbind(Reg.Cent,apply(Regs[[i]],2,mean))}

  Reg.names<-c("vr=1","vr=2","vr=3")  #regions names

  res<-list(
    txt1=txt1, txt2=txt2,
    type=typ, desc=description,
    mtitle=main.txt, #main label in the plot
    ext=U, #closest points to CC in each vertex region
    X=Dt, num.points=n, #data points and its size
    ROI=tri, # region of interest for X points
    cent=CC, ncent="CC", #center and center name
    regions=Regs, region.names=Reg.names, region.centers=Reg.Cent,
    dist2ref=Dis #distances of closest points to CC in each vertex region
  )

  class(res)<-"Extrema"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

#' @title The indicator for a point being a dominating point for Arc Slice Proximity Catch Digraphs
#' (AS-PCDs) - basic triangle case
#'
#' @description Returns I(\code{p} is a dominating point of the AS-PCD) where the vertices of the AS-PCD are the 2D data set \code{Dt}, that is, returns 1 if \code{p} is a dominating
#' point of AS-PCD, returns 0 otherwise. AS proximity regions are defined with respect to the basic triangle,
#' \eqn{T_b}, \eqn{c_1} is in \eqn{[0,1/2]}, \eqn{c_2>0} and \eqn{(1-c_1)^2+c_2^2 \le 1}.
#'
#' Any given triangle can be mapped to the basic triangle by a combination of rigid body motions
#' (i.e., translation, rotation and reflection) and scaling, preserving uniformity of the points in the
#' original triangle. Hence basic triangle is useful for simulation
#' studies under the uniformity hypothesis.
#'
#' Vertex regions are based on the center \code{M="CC"} for circumcenter
#' of \eqn{T_b}; or \eqn{M=(m_1,m_2)} in Cartesian coordinates or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the
#' interior of \eqn{T_b}; default is \code{M="CC"}.
#' Point, \code{p}, is in the vertex region of vertex \code{rv} (default is \code{NULL}); vertices are labeled as \eqn{1,2,3}
#' in the order they are stacked row-wise.
#'
#' \code{ch.data.pnt} is for checking whether point \code{p} is a data point in \code{Dt} or not (default is \code{FALSE}),
#' so by default this function checks whether the point \code{p} would be a dominating point
#' if it actually were in the data set.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:mcap2012;textual}{pcds}).
#'
#' @param p A 2D point that is to be tested for being a dominating point or not of the AS-PCD.
#' @param Dt A set of 2D points which constitutes the vertices of the AS-PCD.
#' @param c1,c2 Positive real numbers which constitute the vertex of the basic triangle
#' adjacent to the shorter edges; \eqn{c_1} must be in \eqn{[0,1/2]}, \eqn{c_2>0} and \eqn{(1-c_1)^2+c_2^2 \le 1}.
#' @param M The center of the triangle. \code{"CC"} stands for circumcenter of the triangle \eqn{T_b} or a 2D point in Cartesian coordinates or
#' a 3D point in barycentric coordinates which serves as a center in the interior of the triangle \eqn{T_b};
#' default is \code{M="CC"} i.e., the circumcenter of \eqn{T_b}.
#' @param rv Index of the vertex whose region contains point \code{p}, \code{rv} takes the vertex labels as \eqn{1,2,3} as
#' in the row order of the vertices in \eqn{T_b}.
#' @param ch.data.pnt A logical argument for checking whether point \code{p} is a data point in \code{Dt} or not
#' (default is \code{FALSE}).
#'
#' @return I(\code{p} is a dominating point of the AS-PCD) where the vertices of the AS-PCD are the 2D data set \code{Dt},
#' that is, returns 1 if \code{p} is a dominating point, returns 0 otherwise
#'
#' @seealso \code{\link{Gam1AStri}} and \code{\link{Gam1PEbastri}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' c1<-.4; c2<-.6;
#' A<-c(0,0); B<-c(1,0); C<-c(c1,c2);
#' Tb<-rbind(A,B,C)
#' n<-20
#'
#' set.seed(1)
#' dat<-runif.bastri(n,c1,c2)$g
#'
#' M<-as.numeric(runif.bastri(1,c1,c2)$g)  #try also M<-c(.6,.2)
#'
#' Gam1ASbastri(dat[1,],dat,c1,c2,M)
#'
#' gam.vec<-vector()
#' for (i in 1:n)
#' {gam.vec<-c(gam.vec,Gam1ASbastri(dat[i,],dat,c1,c2,M))}
#'
#' ind.gam1<-which(gam.vec==1)
#' ind.gam1
#'
#' #or try
#' Rv<-rv.bastriCC(dat[1,],c1,c2)$rv
#' Gam1ASbastri(dat[1,],dat,c1,c2,M,Rv)
#'
#' Gam1ASbastri(c(.2,.4),dat,c1,c2,M)
#' Gam1ASbastri(c(.2,.4),c(.2,.4),c1,c2,M)
#'
#' dat2<-rbind(dat,c(.2,.4))
#' Gam1ASbastri(dat[1,],dat2,c1,c2,M)
#'
#' CC<-circ.cent.bastri(c1,c2)  #the circumcenter
#'
#' if (dimension(M)==3) {M<-bary2cart(M,Tb)}
#' #need to run this when M is given in barycentric coordinates
#'
#' if (isTRUE(all.equal(M,CC)) || identical(M,"CC"))
#' {cent<-CC
#' D1<-(B+C)/2; D2<-(A+C)/2; D3<-(A+B)/2;
#' Ds<-rbind(D1,D2,D3)
#' cent.name<-"CC"
#' } else
#' {cent<-M
#' cent.name<-"M"
#' Ds<-cp2e.bastri(c1,c2,M)
#' D1<-Ds[1,]; D2<-Ds[2,]; D3<-Ds[3,]
#' }
#'
#' Xlim<-range(Tb[,1],dat[,1])
#' Ylim<-range(Tb[,2],dat[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(A,pch=".",xlab="",ylab="",
#' xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tb)
#' L<-rbind(cent,cent,cent); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#' points(dat)
#' points(rbind(dat[ind.gam1,]),pch=4,col=2)
#'
#' txt<-rbind(Tb,cent,D1,D2,D3)
#' xc<-txt[,1]+c(-.03,.03,.02,.06,.06,-0.05,.01)
#' yc<-txt[,2]+c(.02,.02,.03,.0,.03,.03,-.03)
#' txt.str<-c("A","B","C",cent.name,"D1","D2","D3")
#' text(xc,yc,txt.str)
#'
#' Gam1ASbastri(c(.4,.2),dat,c1,c2,M)
#'
#' dat.fr<-data.frame(a=dat)
#' Gam1ASbastri(c(.4,.2),dat.fr,c1,c2,M)
#'
#' Gam1ASbastri(c(.5,.11),dat,c1,c2,M)
#'
#' Gam1ASbastri(c(.5,.11),dat,c1,c2,M,ch.data.pnt=FALSE)
#' #gives an error message if ch.data.pnt=TRUE since the point is not in the basic triangle
#'
#' @export
Gam1ASbastri <- function(p,Dt,c1,c2,M="CC",rv=NULL,ch.data.pnt=FALSE)
{
  if (!is.point(p))
  {stop('p must be a numeric point of dimension 2')}

  if (!is.numeric(as.matrix(Dt)))
  {stop('Dt must be numeric')}

  if (is.point(Dt))
  { Dt<-matrix(Dt,ncol=2)
  } else
  {Dt<-as.matrix(Dt)
  if (ncol(Dt)!=2 )
  {stop('Dt must be of dimension nx2')}
  }

  if (isTRUE(all.equal(matrix(p,ncol=2),Dt)))
  {dom<-1; return(dom); stop}

  if (!is.point(c1,1) || !is.point(c2,1))
  {stop('c1 and c2 must be scalars')}

  if (c1<0 || c1>1/2 || c2<=0 || (1-c1)^2+c2^2 >1)
  {stop('c1 must be in [0,1/2], c2>0 and (1-c1)^2+c2^2 <=1')}

  if (identical(M,"CC"))
  {M<-circ.cent.tri(Tb)
  } else
  {
    if ( !is.point(M) && !is.point(M,3) )
    {stop('M must be the circumcenter "CC" or a numeric 2D point for Cartesian coordinates or
          3D point for barycentric coordinates')}

    if (dimension(M)==3)
    {
      M<-bary2cart(M,Tb)
    }

    if (isTRUE(all.equal(M,circ.cent.tri(Tb)))==F & in.triangle(M,Tb,boundary=FALSE)$in.tri==F)
    {stop('center is not the circumcenter or not in the interior of the triangle')}
  }

  if (ch.data.pnt==TRUE)
  {
    if (!is.in.data(p,Dt))
    {stop('p is not a data point in Dt')}
  }

  A<-c(0,0); B<-c(1,0); C<-c(c1,c2); Tb<-rbind(A,B,C)
  if (in.triangle(p,Tb,boundary = TRUE)$in.tri==F)
  {dom<-0; return(dom); stop}

  if (is.null(rv))
  { rv<-ifelse(isTRUE(all.equal(M,circ.cent.tri(Tb)))==T,rv.bastriCC(p,c1,c2)$rv,rv.bastri.cent(p,c1,c2,M)$rv)  #vertex region for pt
  } else
  {  if (!is.numeric(rv) || sum(rv==c(1,2,3))!=1)
  {stop('vertex index, rv, must be 1, 2 or 3')}}

  n<-nrow(Dt)
  dom<-1; i<-1;

  while (i <= n & dom==1)
  {if (IndNASbastri(p,Dt[i,],c1,c2,M,rv)==0)
    dom<-0;
  i<-i+1;
  }
  dom
} #end of the function
#'

#################################################################

#' @title The indicator for a point being a dominating point for Arc Slice Proximity Catch Digraphs
#' (AS-PCDs) - one triangle case
#'
#' @description Returns I(\code{p} is a dominating point of the AS-PCD whose vertices are the 2D data set \code{Dt}),
#' that is, returns 1 if \code{p} is a dominating point of AS-PCD, returns 0 otherwise.
#' Point, \code{p}, is in the region of vertex \code{rv} (default is \code{NULL}); vertices are labeled as \eqn{1,2,3}
#' in the order they are stacked row-wise in \code{tri}.
#'
#' AS proximity regions are defined with respect to the
#' triangle \code{tri} and vertex regions are based on the center \code{M="CC"} for circumcenter of \code{tri}; or \eqn{M=(m_1,m_2)} in Cartesian coordinates
#' or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the interior of the triangle \code{tri};
#' default is \code{M="CC"} the circumcenter of \code{tri}.
#'
#' \code{ch.data.pnt} is for checking whether point \code{p} is a data point in \code{Dt} or not (default is \code{FALSE}),
#' so by default this function checks whether the point \code{p} would be a dominating point
#' if it actually were in the data set.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:mcap2012;textual}{pcds}).
#'
#' @param p A 2D point that is to be tested for being a dominating point or not of the AS-PCD.
#' @param Dt A set of 2D points which constitutes the vertices of the AS-PCD.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param M The center of the triangle. \code{"CC"} stands for circumcenter of the triangle \code{tri} or a 2D point in Cartesian coordinates or
#' a 3D point in  barycentric coordinates which serves as a center in the interior of the triangle \eqn{T_b};
#' default is \code{M="CC"} i.e., the circumcenter of \code{tri}.
#' @param rv Index of the vertex whose region contains point \code{p}, \code{rv} takes the vertex labels as \eqn{1,2,3} as
#' in the row order of the vertices in \code{tri}.
#' @param ch.data.pnt A logical argument for checking whether point \code{p} is a data point in \code{Dt} or not
#' (default is \code{FALSE}).
#'
#' @return I(\code{p} is a dominating point of the AS-PCD whose vertices are the 2D data set \code{Dt}),
#' that is, returns 1 if \code{p} is a dominating point of the AS-PCD, returns 0 otherwise
#'
#' @seealso \code{\link{Gam1ASbastri}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#' n<-10
#'
#' set.seed(1)
#' dat<-runif.tri(n,Tr)$g
#'
#' M<-as.numeric(runif.tri(1,Tr)$g)  #try also M<-c(1.6,1.2)
#'
#' Gam1AStri(dat[1,],dat,Tr,M)
#' Gam1AStri(dat[1,],dat[1,],Tr,M)
#' Gam1AStri(c(1.5,1.5),c(1.6,1),Tr,M)
#' Gam1AStri(c(1.6,1),c(1.5,1.5),Tr,M)
#'
#' gam.vec<-vector()
#' for (i in 1:n)
#' {gam.vec<-c(gam.vec,Gam1AStri(dat[i,],dat,Tr,M))}
#'
#' ind.gam1<-which(gam.vec==1)
#' ind.gam1
#'
#' #or try
#' Rv<-rv.triCC(dat[1,],Tr)$rv
#' Gam1AStri(dat[1,],dat,Tr,M,Rv)
#'
#' Gam1AStri(c(.2,.4),dat,Tr,M)
#' Gam1AStri(c(.2,.4),c(.2,.4),Tr,M)
#'
#' dat2<-rbind(dat,c(.2,.4))
#' Gam1AStri(dat[1,],dat2,Tr,M)
#'
#' if (dimension(M)==3) {M<-bary2cart(M,Tr)}
#' #need to run this when M is given in barycentric coordinates
#'
#' CC<-circ.cent.tri(Tr)  #the circumcenter
#'
#' if (isTRUE(all.equal(M,CC)) || identical(M,"CC"))
#' {cent<-CC
#' D1<-(B+C)/2; D2<-(A+C)/2; D3<-(A+B)/2;
#' Ds<-rbind(D1,D2,D3)
#' cent.name<-"CC"
#' } else
#' {cent<-M
#' cent.name<-"M"
#' Ds<-cp2e.tri(Tr,M)
#' D1<-Ds[1,]; D2<-Ds[2,]; D3<-Ds[3,]
#' }
#'
#' Xlim<-range(Tr[,1],dat[,1])
#' Ylim<-range(Tr[,2],dat[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(A,pch=".",xlab="",ylab="",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tr)
#' points(dat)
#' L<-rbind(cent,cent,cent); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#' points(rbind(dat[ind.gam1,]),pch=4,col=2)
#'
#' txt<-rbind(Tr,cent,Ds)
#' xc<-txt[,1]
#' yc<-txt[,2]
#' txt.str<-c("A","B","C",cent.name,"D1","D2","D3")
#' text(xc,yc,txt.str)
#'
#' Gam1AStri(c(1.5,1.1),dat,Tr,M)
#'
#' dat.fr<-data.frame(a=dat)
#' Gam1AStri(c(1.5,1.1),dat.fr,Tr,M)
#'
#' dat.fr<-data.frame(a=Tr)
#' Gam1AStri(c(1.5,1.1),dat,dat.fr,M)
#'
#' Gam1AStri(c(1.5,1.1),dat,Tr,M)
#'
#' Gam1AStri(c(1.5,1.1),dat,Tr,M,ch.data.pnt=FALSE)
#' #gives an error message if ch.data.pnt=TRUE since point p is not a data point in Dt
#'
#' @export
Gam1AStri <- function(p,Dt,tri,M="CC",rv=NULL,ch.data.pnt=FALSE)
{
  if (!is.point(p))
  {stop('p must be a numeric point of dimension 2')}

  if (!is.numeric(as.matrix(Dt)))
  {stop('p must be numeric')}

  if (is.point(Dt))
  { Dt<-matrix(Dt,ncol=2)
  } else
  {Dt<-as.matrix(Dt)
  if (ncol(Dt)!=2 )
  {stop('Dt must be of dimension nx2')}
  }

  if (isTRUE(all.equal(matrix(p,ncol=2),Dt)))
  {dom<-1; return(dom); stop}

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  if (identical(M,"CC"))
  {M<-circ.cent.tri(tri)
  } else
  {
    if ( !is.point(M) && !is.point(M,3) )
    {stop('M must be the circumcenter "CC" or a numeric 2D point for Cartesian coordinates or
          3D point for barycentric coordinates')}

    if (dimension(M)==3)
    {
      M<-bary2cart(M,tri)
    }

    if (isTRUE(all.equal(M,circ.cent.tri(tri)))==F & in.triangle(M,tri,boundary=FALSE)$in.tri==F)
    {stop('center is not the circumcenter or not in the interior of the triangle')}
  }

  if (ch.data.pnt==TRUE)
  {
    if (!is.in.data(p,Dt))
    {stop('p is not a data point in Dt')}
  }

  if (in.triangle(p,tri,boundary = TRUE)$in.tri==F)
  {dom<-0; return(dom); stop}

  if (is.null(rv))
  { rv<-ifelse(isTRUE(all.equal(M,circ.cent.tri(tri)))==T,rv.triCC(p,tri)$rv,rv.tri.cent(p,tri,M)$rv)  #vertex region for pt
  } else
  {  if (!is.numeric(rv) || sum(rv==c(1,2,3))!=1)
  {stop('vertex index, rv, must be 1, 2 or 3')}}

  n<-nrow(Dt)
  dom<-1; i<-1;
  while (i <= n & dom==1)
  {if (IndNAStri(p,Dt[i,],tri,M,rv)==0)
    dom<-0;
  i<-i+1;
  }
  dom
} #end of the function
#'

#################################################################

#' @title Check a point belong to a given data set
#'
#' @description returns \code{TRUE} if the point \code{p} of any dimension is inside the data set \code{Dt} of the same dimension as \code{p};
#' otherwise returns \code{FALSE}.
#'
#' @param p A 2D point for which the function checks membership to the data set \code{Dt}.
#' @param Dt A set of 2D points representing the set of data points.
#'
#' @return \code{TRUE} if \code{p} belongs to the data set \code{Dt}.
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' n<-10
#' dat<-cbind(runif(n),runif(n))
#'
#' P<-dat[7,]
#' is.in.data(P,dat)
#' is.in.data(P,dat[7,])
#'
#' P<-dat[7,]+10^(-7)
#' is.in.data(P,dat)
#'
#' P<-dat[7,]+10^(-9)
#' is.in.data(P,dat)
#'
#' is.in.data(P,P)
#'
#' is.in.data(c(2,2),c(2,2))
#'
#' #for 1D data
#' n<-10
#' dat<-runif(n)
#'
#' P<-dat[7]
#' is.in.data(P,dat[7])  #this works because both entries are treated as 1D vectors but
#' #is.in.data(P,dat) does not work since entries are treated as vectors of different dimensions
#'
#' dat<-as.matrix(dat)
#' is.in.data(P,dat)
#' #this works, because P is a 1D point, and dat is treated as a set of 10 1D points
#'
#' P<-dat[7]+10^(-7)
#' is.in.data(P,dat)
#'
#' P<-dat[7]+10^(-9)
#' is.in.data(P,dat)
#'
#' is.in.data(P,P)
#'
#' #for 3D data
#' n<-10
#' dat<-cbind(runif(n),runif(n),runif(n))
#'
#' P<-dat[7,]
#' is.in.data(P,dat)
#' is.in.data(P,dat[7,])
#'
#' P<-dat[7,]+10^(-7)
#' is.in.data(P,dat)
#'
#' P<-dat[7,]+10^(-9)
#' is.in.data(P,dat)
#'
#' is.in.data(P,P)
#'
#' n<-10
#' dat<-cbind(runif(n),runif(n))
#' P<-dat[7,]
#' is.in.data(P,dat)
#'
#' dat.fr<-data.frame(a=dat)
#' is.in.data(P,dat.fr)
#'
#' @export is.in.data
is.in.data <- function(p,Dt)
{
  if (!is.numeric(as.matrix(p)) || !is.numeric(as.matrix(Dt)) )
  {stop ('p and Dt must be numeric')}

  if (!is.vector(p))
  {stop('p must be a vector')}

  dimp<-length(p)

 ins<-FALSE
  if (is.vector(Dt))
  {dimDt<-length(Dt);
  if (dimp != dimDt )
  {stop('Both arguments must be of the same dimension')
  } else
  {
    if (isTRUE(all.equal(p,Dt)))
     ins<-TRUE }
  } else
  {
    if (!is.matrix(Dt) && !is.data.frame(Dt))
    {stop('Dt must be a matrix or a data frame, each row representing a point')}
    Dt<-as.matrix(Dt)
    dimDt<-ncol(Dt);
    if (dimp != dimDt)
    {stop('p and Dt must be of the same dimension')}
    nDt<-nrow(Dt)
    i<-1
    cnt<-0

    while (i <= nDt & cnt==0)
    {
      if (isTRUE(all.equal(p,Dt[i,])))
      {ins<-TRUE; cnt<-1}
      else
      {i<-i+1}
    }
  }
 ins
} #end of the function
#'

#################################################################

#' @title The indicator for two points being a dominating set for Arc Slice Proximity Catch Digraphs
#' (AS-PCDs) - basic triangle case
#'
#' @description Returns \eqn{I(}\{\code{pt1,pt2}\} is a dominating set of AS-PCD\eqn{)} where vertices of AS-PCD are the 2D
#' data set \code{Dt}), that is, returns 1 if \{\code{pt1,pt2}\} is a dominating set of AS-PCD, returns 0 otherwise.
#'
#' AS proximity regions are defined with respect to the basic triangle \eqn{T_b=T(c(0,0),c(1,0),c(c1,c2))},
#' In the basic triangle, \eqn{T_b}, \eqn{c_1} is in \eqn{[0,1/2]}, \eqn{c_2>0} and \eqn{(1-c_1)^2+c_2^2 \le 1}.
#'
#' Any given triangle can be mapped to the basic triangle by a combination of rigid body motions
#' (i.e., translation, rotation and reflection) and scaling, preserving uniformity of the points in the
#' original triangle. Hence basic triangle is useful for simulation
#' studies under the uniformity hypothesis.
#'
#' Point, \code{pt1}, is in the vertex region of vertex \code{rv1} (default is \code{NULL})
#' and point, \code{pt2}, is in the vertex region of vertex \code{rv2} (default is \code{NULL}); vertices are labeled as \eqn{1,2,3}
#' in the order they are stacked row-wise.
#'
#' Vertex regions are based on the center \code{M="CC"} for circumcenter
#' of \eqn{T_b}; or \eqn{M=(m_1,m_2)} in Cartesian coordinates or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the
#' interior of \eqn{T_b}; default is \code{M="CC"}.
#'
#' \code{ch.data.pnts} is for checking whether points \code{pt1} and \code{pt2} are data points in \code{Dt} or not
#' (default is \code{FALSE}), so by default this function checks whether the points \code{pt1} and \code{pt2} would be a
#' dominating set if they actually were in the data set.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:mcap2012;textual}{pcds}).
#'
#' @param pt1,pt2 Two 2D points to be tested for constituting a dominating set of the AS-PCD.
#' @param Dt A set of 2D points which constitutes the vertices of the AS-PCD.
#' @param c1,c2 Positive real numbers which constitute the vertex of the basic triangle
#' adjacent to the shorter edges; \eqn{c_1} must be in \eqn{[0,1/2]}, \eqn{c_2>0} and \eqn{(1-c_1)^2+c_2^2 \le 1}.
#' @param M The center of the triangle. \code{"CC"} stands for circumcenter of the triangle \eqn{T_b} or a 2D point in Cartesian coordinates or
#' a 3D point in barycentric coordinates which serves as a center in the interior of the triangle \eqn{T_b};
#' default is \code{M="CC"} i.e., the circumcenter of \eqn{T_b}.
#' @param rv1,rv2 The indices of the vertices whose regions contains \code{pt1} and \code{pt2}, respectively.
#' They take the vertex labels as \eqn{1,2,3} as in the row order of the vertices in \eqn{T_b}
#' (default is \code{NULL} for both).
#' @param ch.data.pnts A logical argument for checking whether points \code{pt1} and \code{pt2} are data points in \code{Dt} or not
#' (default is \code{FALSE}).
#'
#' @return \eqn{I(}\{\code{pt1,pt2}\} is a dominating set of the AS-PCD\eqn{)} where the vertices of AS-PCD are the 2D data set \code{Dt}),
#' that is, returns 1 if \{\code{pt1,pt2}\} is a dominating set of AS-PCD, returns 0 otherwise
#'
#' @seealso \code{\link{Gam2AStri}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' c1<-.4; c2<-.6;
#' A<-c(0,0); B<-c(1,0); C<-c(c1,c2);
#' Tb<-rbind(A,B,C)
#' n<-20
#'
#' set.seed(1)
#' dat<-runif.bastri(n,c1,c2)$g
#'
#' M<-as.numeric(runif.bastri(1,c1,c2)$g)  #try also M<-c(.6,.2)
#'
#' Gam2ASbastri(dat[1,],dat[2,],dat,c1,c2,M)
#' Gam2ASbastri(dat[1,],dat[1,],dat,c1,c2,M)  #one point can not a dominating set of size two
#'
#' Gam2ASbastri(c(.2,.4),c(.2,.5),rbind(c(.2,.4),c(.2,.5)),c1,c2,M)
#'
#' ind.gam2<-vector()
#' for (i in 1:(n-1))
#'   for (j in (i+1):n)
#'   {if (Gam2ASbastri(dat[i,],dat[j,],dat,c1,c2,M)==1)
#'    ind.gam2<-rbind(ind.gam2,c(i,j))}
#' ind.gam2
#'
#' #or try
#' rv1<-rv.bastriCC(dat[1,],c1,c2)$rv
#' rv2<-rv.bastriCC(dat[2,],c1,c2)$rv
#' Gam2ASbastri(dat[1,],dat[2,],dat,c1,c2,M,rv1,rv2)
#' Gam2ASbastri(c(.2,.4),dat[2,],dat,c1,c2,M,rv1,rv2)
#'
#' #or try
#' rv1<-rv.bastriCC(dat[1,],c1,c2)$rv
#' Gam2ASbastri(dat[1,],dat[2,],dat,c1,c2,M,rv1)
#'
#' #or try
#' Rv2<-rv.bastriCC(dat[2,],c1,c2)$rv
#' Gam2ASbastri(dat[1,],dat[2,],dat,c1,c2,M,rv2=Rv2)
#'
#' Gam2ASbastri(c(.3,.2),c(.35,.25),dat,c1,c2,M)
#'
#' dat.fr<-data.frame(a=dat)
#' Gam2ASbastri(c(.3,.2),c(.35,.25),dat.fr,c1,c2,M)
#'
#' @export
Gam2ASbastri <- function(pt1,pt2,Dt,c1,c2,M="CC",rv1=NULL,rv2=NULL,ch.data.pnts=FALSE)
{
  if (!is.point(pt1) || !is.point(pt2) )
  {stop('pt1 and pt2 must both be numeric 2D points')}

  if (!is.numeric(as.matrix(Dt)))
  {stop('Dt must be numeric')}

  if (is.point(Dt))
  { Dt<-matrix(Dt,ncol=2)
  } else
  {Dt<-as.matrix(Dt)
  if (ncol(Dt)!=2 )
  {stop('Dt must be of dimension nx2')}
  }

  if (isTRUE(all.equal(matrix(rbind(pt1,pt2),ncol=2),Dt)))
  {dom<-1; return(dom); stop}

  if (!is.point(c1,1) || !is.point(c2,1))
  {stop('c1 and c2 must be scalars')}

  if (c1<0 || c1>1/2 || c2<=0 || (1-c1)^2+c2^2 >1)
  {stop('c1 must be in [0,1/2], c2>0 and (1-c1)^2+c2^2 <=1')}

  A<-c(0,0); B<-c(1,0); C<-c(c1,c2); Tb<-rbind(A,B,C);

  if (identical(M,"CC"))
  {M<-circ.cent.tri(Tb)
  } else
  {
    if ( !is.point(M) && !is.point(M,3) )
    {stop('M must be the circumcenter "CC" or a numeric 2D point for Cartesian coordinates or
          3D point for barycentric coordinates')}

    if (dimension(M)==3)
    {
      M<-bary2cart(M,Tb)
    }

    if (isTRUE(all.equal(M,circ.cent.tri(Tb)))==F & in.triangle(M,Tb,boundary=FALSE)$in.tri==F)
    {stop('center is not the circumcenter or not in the interior of the triangle')}
  }

  if (ch.data.pnts==TRUE)
  {
    if (!is.in.data(pt1,Dt) || !is.in.data(pt2,Dt))
    {stop('not both points, pt1 and pt2, are data points in Dt')}
  }

  if (identical(pt1,pt2))
  {dom<-0; return(dom); stop}

  if (is.null(rv1))
  { rv1<-ifelse(isTRUE(all.equal(M,circ.cent.tri(Tb)))==T,rv.bastriCC(pt1,c1,c2)$rv,rv.bastri.cent(pt1,c1,c2,M)$rv)  #vertex region for pt
  } else
  {  if (!is.numeric(rv1) || sum(rv1==c(1,2,3))!=1)
  {stop('vertex index, rv1, must be 1, 2 or 3')}}

  if (is.null(rv2))
  { rv2<-ifelse(isTRUE(all.equal(M,circ.cent.tri(Tb)))==T,rv.bastriCC(pt2,c1,c2)$rv,rv.bastri.cent(pt2,c1,c2,M)$rv)  #vertex region for pt
  } else
  {  if (!is.numeric(rv2) || sum(rv2==c(1,2,3))!=1)
  {stop('vertex index, rv2, must be 1, 2 or 3')}}

  n<-nrow(Dt)
  dom<-1; i<-1;
  while (i <= n & dom==1)
  {if (max(IndNASbastri(pt1,Dt[i,],c1,c2,M,rv=rv1),IndNASbastri(pt2,Dt[i,],c1,c2,M,rv=rv2))==0)
    dom<-0;
  i<-i+1;
  }
  dom
} #end of the function
#'

#################################################################

#' @title The indicator for two points constituting a dominating set for Arc Slice Proximity Catch Digraphs
#' (AS-PCDs) - one triangle case
#'
#' @description Returns \eqn{I(}\{\code{pt1,pt2}\} is a dominating set of the AS-PCD\eqn{)} where vertices of the AS-PCD are
#' the 2D data set \code{Dt}), that is, returns 1 if \{\code{pt1,pt2}\} is a dominating set of AS-PCD, returns 0 otherwise.
#'
#' AS proximity regions are defined with respect to the triangle \code{tri}.
#' Point, \code{pt1}, is in the region of vertex \code{rv1} (default is \code{NULL})
#' and point, \code{pt2}, is in the region of vertex \code{rv2} (default is \code{NULL}); vertices (and hence \code{rv1} and \code{rv2})
#' are labeled as \eqn{1,2,3} in the order they are stacked row-wise in \code{tri}.
#'
#' Vertex regions are based on
#' the center \code{M="CC"} for circumcenter of \code{tri}; or \eqn{M=(m_1,m_2)} in Cartesian coordinates
#' or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the interior of the triangle \code{tri};
#' default is \code{M="CC"} the circumcenter of \code{tri}.
#'
#' \code{ch.data.pnts} is for checking whether points \code{pt1} and \code{pt2} are data
#' points in \code{Dt} or not (default is \code{FALSE}), so by default this function checks whether the points \code{pt1} and \code{pt2}
#' would constitute dominating set if they actually were in the data set.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:mcap2012;textual}{pcds}).
#'
#' @param pt1,pt2 Two 2D points to be tested for constituting a dominating set of the AS-PCD.
#' @param Dt A set of 2D points which constitutes the vertices of the AS-PCD.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param M The center of the triangle. \code{"CC"} stands for circumcenter of the triangle \code{tri} or a 2D point in Cartesian coordinates or
#' a 3D point in  barycentric coordinates which serves as a center in the interior of the triangle \eqn{T_b};
#' default is \code{M="CC"} i.e., the circumcenter of \code{tri}.
#' @param rv1,rv2 The indices of the vertices whose regions contains \code{pt1} and \code{pt2}, respectively.
#' They take the vertex labels as \eqn{1,2,3} as in the row order of the vertices in \code{tri}
#' (default is \code{NULL} for both).
#' @param ch.data.pnts A logical argument for checking whether points \code{pt1} and \code{pt2} are data points in \code{Dt} or not
#' (default is \code{FALSE}).
#'
#' @return \eqn{I(}\{\code{pt1,pt2}\} is a dominating set of the AS-PCD\eqn{)} where vertices of the AS-PCD are the 2D data set \code{Dt}),
#' that is, returns 1 if \{\code{pt1,pt2}\} is a dominating set of AS-PCD, returns 0 otherwise
#'
#' @seealso \code{\link{Gam2ASbastri}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#' n<-10
#'
#' set.seed(1)
#' dat<-runif.tri(n,Tr)$g
#'
#' M<-as.numeric(runif.tri(1,Tr)$g)  #try also M<-c(1.6,1.2)
#'
#' Gam2AStri(dat[1,],dat[2,],dat,Tr,M)
#' Gam2AStri(dat[1,],dat[1,],dat,Tr,M)  #same two points cannot be a dominating set of size 2
#'
#' Gam2AStri(c(.2,.4),dat[2,],dat,Tr,M)
#' Gam2AStri(c(.2,.4),c(.2,.5),dat,Tr,M)
#' Gam2AStri(c(.2,.4),c(.2,.5),rbind(c(.2,.4),c(.2,.5)),Tr,M)
#'
#' #or try
#' rv1<-rv.triCC(c(.2,.4),Tr)$rv
#' rv2<-rv.triCC(c(.2,.5),Tr)$rv
#' Gam2AStri(c(.2,.4),c(.2,.5),rbind(c(.2,.4),c(.2,.5)),Tr,M,rv1,rv2)
#'
#' ind.gam2<-vector()
#' for (i in 1:(n-1))
#'   for (j in (i+1):n)
#'   {if (Gam2AStri(dat[i,],dat[j,],dat,Tr,M)==1)
#'    ind.gam2<-rbind(ind.gam2,c(i,j))}
#' ind.gam2
#'
#' #or try
#' rv1<-rv.triCC(dat[1,],Tr)$rv
#' rv2<-rv.triCC(dat[2,],Tr)$rv
#' Gam2AStri(dat[1,],dat[2,],dat,Tr,M,rv1,rv2)
#'
#' #or try
#' rv1<-rv.triCC(dat[1,],Tr)$rv
#' Gam2AStri(dat[1,],dat[2,],dat,Tr,M,rv1)
#'
#' #or try
#' Rv2<-rv.triCC(dat[2,],Tr)$rv
#' Gam2AStri(dat[1,],dat[2,],dat,Tr,M,rv2=Rv2)
#'
#' Gam2AStri(c(1.3,1.2),c(1.35,1.25),dat,Tr,M)
#'
#' dat.fr<-data.frame(a=dat)
#' Gam2AStri(c(.3,.2),c(.35,.25),dat.fr,Tr,M)
#'
#' dat.fr<-data.frame(a=Tr)
#' Gam2AStri(c(.3,.2),c(.35,.25),dat,dat.fr,M)
#'
#' @export Gam2AStri
Gam2AStri <- function(pt1,pt2,Dt,tri,M="CC",rv1=NULL,rv2=NULL,ch.data.pnts=FALSE)
{
  if (!is.point(pt1) || !is.point(pt2) )
  {stop('pt1 and pt2 must both be numeric 2D points')}

  if (!is.numeric(as.matrix(Dt)))
  {stop('Dt must be numeric')}

  if (is.point(Dt))
  { Dt<-matrix(Dt,ncol=2)
  } else
  {Dt<-as.matrix(Dt)
  if (ncol(Dt)!=2 )
  {stop('Dt must be of dimension nx2')}
  }

  if (isTRUE(all.equal(matrix(rbind(pt1,pt2),ncol=2),Dt)))
  {dom<-1; return(dom); stop}

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  if (ch.data.pnts==TRUE)
  {
    if (!is.in.data(pt1,Dt) || !is.in.data(pt2,Dt))
    {stop('not both points, pt1 and pt2, are data points in Dt')}
  }

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  if (identical(M,"CC"))
  {M<-circ.cent.tri(tri)
  } else
  {
    if ( !is.point(M) && !is.point(M,3) )
    {stop('M must be the circumcenter "CC" or a numeric 2D point for Cartesian coordinates or
          3D point for barycentric coordinates')}

    if (dimension(M)==3)
    {
      M<-bary2cart(M,tri)
    }

    if (isTRUE(all.equal(M,circ.cent.tri(tri)))==F & in.triangle(M,tri,boundary=FALSE)$in.tri==F)
    {stop('center is not the circumcenter or not in the interior of the triangle')}
  }

  if (identical(pt1,pt2))
  {dom<-0; return(dom); stop}

  if (is.null(rv1))
  { rv1<-ifelse(isTRUE(all.equal(M,circ.cent.tri(tri)))==T,rv.triCC(pt1,tri)$rv,rv.tri.cent(pt1,tri,M)$rv)  #vertex region for pt
  } else
  {  if (!is.numeric(rv1) || sum(rv1==c(1,2,3))!=1)
  {stop('vertex index, rv1, must be 1, 2 or 3')}}

  if (is.null(rv2))
  { rv2<-ifelse(isTRUE(all.equal(M,circ.cent.tri(tri)))==T,rv.triCC(pt2,tri)$rv,rv.tri.cent(pt2,tri,M)$rv)  #vertex region for pt
  } else
  {  if (!is.numeric(rv2) || sum(rv2==c(1,2,3))!=1)
  {stop('vertex index, rv2, must be 1, 2 or 3')}}

  n<-nrow(Dt)
  dom<-1; i<-1;
  while (i <= n & dom==1)
  {if (max(IndNAStri(pt1,Dt[i,],tri,M,rv=rv1),IndNAStri(pt2,Dt[i,],tri,M,rv=rv2))==0)
    dom<-0;
  i<-i+1;
  }
  dom
} #end of the function
#'

#################################################################

#' @title The arcs of Arc Slice Proximity Catch Digraph (AS-PCD) for 2D data - one triangle case
#'
#' @description
#' An object of class \code{"PCDs"}.
#' Returns arcs as tails (or sources) and heads (or arrow ends) for data set \code{Xp} as the vertices
#' of AS-PCD.
#'
#' AS proximity regions are constructed with respect to the triangle \code{tri}, i.e.,
#' arcs may exist for points only inside \code{tri}.
#' It also provides various descriptions and quantities about the arcs of the AS-PCD
#' such as number of arcs, arc density, etc.
#'
#' Vertex regions are based on the center \code{M="CC"} for
#' circumcenter of \code{tri}; or \eqn{M=(m_1,m_2)} in Cartesian coordinates or \eqn{M=(\alpha,\beta,\gamma)} in barycentric
#' coordinates in the interior of the triangle \code{tri}; default is \code{M="CC"} the circumcenter of \code{tri}.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:mcap2012;textual}{pcds}).
#'
#' @param Xp A set of 2D points which constitute the vertices of the AS-PCD.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param M The center of the triangle. \code{"CC"} stands for circumcenter of the triangle \code{tri} or a 2D point in Cartesian coordinates or a 3D point in
#' barycentric coordinates which serves as a center in the interior of the triangle \eqn{T_b};
#' default is \code{M="CC"} i.e., the circumcenter of \code{tri}.
#'
#' @return A \code{list} with the elements
#' \item{type}{A description of the type of the digraph}
#' \item{parameters}{Parameters of the digraph, here, it is the center used to construct the vertex regions
#' i.e., circumcenter}
#' \item{tess.points}{Points on which the tessellation of the study region is performed, here, tessellation
#' is the support triangle.}
#' \item{tess.name}{Name of data set used in tessellation (i.e., vertices of the triangle).}
#' \item{vertices}{Vertices of the digraph, \code{Xp}.}
#' \item{vert.name}{Name of the data set which constitute the vertices of the digraph}
#' \item{S}{Tails (or sources) of the arcs of AS-PCD for 2D data set \code{Xp} as vertices of the digraph}
#' \item{E}{Heads (or arrow ends) of the arcs of AS-PCD for 2D data set \code{Xp} as vertices of the digraph}
#' \item{mtitle}{Text for \code{"main"} title in the plot of the digraph}
#' \item{quant}{Various quantities for the digraph: number of vertices, number of partition points,
#' number of intervals, number of arcs, and arc density.}
#'
#' @seealso \code{\link{ArcsASMT}}, \code{\link{ArcsPEtri}}, \code{\link{ArcsCStri}}, \code{\link{ArcsPEMT}},
#' and \code{\link{ArcsCSMT}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#'
#' Tr<-rbind(A,B,C);
#' n<-10
#'
#' set.seed(1)
#' dat<-runif.tri(n,Tr)$g
#'
#' M<-as.numeric(runif.tri(1,Tr)$g)  #try also  #M<-c(1.6,1.2)
#'
#' ArcsAStri(dat,Tr,M)
#'
#' Arcs<-ArcsAStri(dat,Tr,M)
#' Arcs
#' summary(Arcs)
#' plot(Arcs)
#'
#' S<-Arcs$S
#' E<-Arcs$E
#'
#' Xlim<-range(Tr[,1],dat[,1])
#' Ylim<-range(Tr[,2],dat[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' if (dimension(M)==3) {M<-bary2cart(M,Tr)}
#' #need to run this when M is given in barycentric coordinates
#'
#' CC<-circ.cent.tri(Tr)
#' if (isTRUE(all.equal(M,CC)) || identical(M,"CC"))
#' {cent<-CC
#' D1<-(B+C)/2; D2<-(A+C)/2; D3<-(A+B)/2;
#' Ds<-rbind(D1,D2,D3)
#' cent.name<-"CC"
#' plot(Tr,pch=".",asp=1,xlab="",ylab="",axes=TRUE,
#' xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' } else
#' {cent<-M
#' cent.name<-"M"
#' Ds<-cp2e.tri(Tr,M)
#' D1<-Ds[1,]; D2<-Ds[2,]; D3<-Ds[3,]
#' plot(Tr,pch=".",xlab="",ylab="",axes=TRUE,
#' xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' }
#' polygon(Tr)
#' points(dat,pch=1,col=1)
#' L<-rbind(cent,cent,cent); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#' arrows(S[,1], S[,2], E[,1], E[,2], length = 0.1, col= 4)
#'
#' txt<-rbind(Tr,cent,Ds)
#' xc<-txt[,1]+c(-.02,.03,.02,.03,.04,-.03,-.01)
#' yc<-txt[,2]+c(.02,.02,.03,.06,.04,.05,-.07)
#' txt.str<-c("A","B","C",cent.name,"D1","D2","D3")
#' text(xc,yc,txt.str)
#'
#' dat.fr<-data.frame(a=dat)
#' ArcsAStri(dat.fr,Tr,M)
#'
#' @export ArcsAStri
ArcsAStri <- function(Xp,tri,M="CC")
{
  if (!is.numeric(as.matrix(Xp)) )
  {stop('Xp must be numeric')}

  if (is.point(Xp))
  { Xp<-matrix(Xp,ncol=2)
  } else
  {Xp<-as.matrix(Xp)
  if (ncol(Xp)!=2 )
  {stop('Xp must be of dimension nx2')}
  }

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  if (identical(M,"CC"))
  {M<-circ.cent.tri(tri)
  } else
  {
    if ( !is.point(M) && !is.point(M,3) )
    {stop('M must be the circumcenter "CC" or a numeric 2D point for Cartesian coordinates or
          3D point for barycentric coordinates')}

    if (dimension(M)==3)
    {
      M<-bary2cart(M,tri)
    }

    if (isTRUE(all.equal(M,circ.cent.tri(tri)))==F & in.triangle(M,tri,boundary=FALSE)$in.tri==F)
    {stop('center is not the circumcenter or not in the interior of the triangle')}
  }
  n<-nrow(Xp)
 in.tri<-rep(0,n)
  for (i in 1:n)
   in.tri[i]<-in.triangle(Xp[i,],tri,boundary=TRUE)$in.tri #indices of the Xp points inside the triangle

  Xtri<-Xp[in.tri==1,] #the Xp points inside the triangle
  n2<-length(Xtri)/2

  #the arcs of AS-PCDs
  S<-E<-NULL #S is for source and E is for end points for the arcs
  if (n2>1)
  {
    for (j in 1:n2)
    {
      pt1<-Xtri[j,];
      RV1<-ifelse(isTRUE(all.equal(M,circ.cent.tri(tri)))==T,rv.triCC(pt1,tri)$rv,rv.tri.cent(pt1,tri,M)$rv)  #vertex region for pt
      for (k in (1:n2)[-j])  #to avoid loops
      {
        pt2<-Xtri[k,];
        if (IndNAStri(pt1,pt2,tri,M,RV1)==1)
        {
          S <-rbind(S,Xtri[j,]); E <-rbind(E,Xtri[k,]);
        }
      }
    }
  }

  xname <-deparse(substitute(Xp))
  yname <-deparse(substitute(tri))
  cname <-ifelse(isTRUE(all.equal(M,circ.cent.tri(tri)))==T,"CC","M")

  param<-M<-circ.cent.tri(tri)
  names(param)<-c("circumcenter","")
  typ<-paste("Arc Slice Proximity Catch Digraph (AS-PCD) for 2D Points in the Triangle with Center ", cname,"=(",round(M[1],2),",",round(M[2],2),")",sep="")

  main.txt<-paste("Arcs of AS-PCD for Points in one Triangle \n with Center ", cname,"=(",round(M[1],2),",",round(M[2],2),")",sep="")

  nvert<-n2; ny<-3; ntri<-1; narcs=ifelse(!is.null(S),nrow(S),0);
  arc.dens<-ifelse(nvert>1,narcs/(nvert*(nvert-1)),NA)

  quantities<-c(nvert,ny,ntri,narcs,arc.dens)
  names(quantities)<-c("number of vertices", "number of partition points",
                       "number of triangles","number of arcs", "arc density")

  res<-list(
    type=typ,
    parameters=param,
    tess.points=tri, tess.name=yname, #tessellation points
    vertices=Xp, vert.name=xname, #vertices of the digraph
    S=S,
    E=E,
    mtitle=main.txt,
    quant=quantities
  )

  class(res)<-"PCDs"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

#' @title The plot of the arcs of Arc Slice Proximity Catch Digraph (AS-PCD) for a 2D data set - one triangle case
#'
#' @description Plots the arcs of AS-PCD whose vertices are the data points, \code{Xp} and the triangle \code{tri}. AS proximity regions
#' are constructed with respect to the triangle \code{tri}, i.e., only for \code{Xp} points inside the triangle \code{tri}.
#'
#' Vertex regions are based on the center \code{M="CC"} for
#' circumcenter of \code{tri}; or \eqn{M=(m_1,m_2)} in Cartesian coordinates or \eqn{M=(\alpha,\beta,\gamma)} in barycentric
#' coordinates in the interior of the triangle \code{tri}; default is \code{M="CC"} the circumcenter of \code{tri}.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:mcap2012;textual}{pcds}).
#'
#' @param Xp A set of 2D points which constitute the vertices of the AS-PCD.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param M The center of the triangle. \code{"CC"} stands for circumcenter of the triangle \code{tri} or a 2D point in Cartesian coordinates or a 3D point in
#' barycentric coordinates which serves as a center in the interior of the triangle \eqn{T_b};
#' default is \code{M="CC"} i.e., the circumcenter of \code{tri}.
#' @param asp A \code{numeric} value, giving the aspect ratio for \eqn{y} axis to \eqn{x}-axis \eqn{y/x} (default is \code{NA}),
#' see the official help page for \code{asp} by typing "\code{? asp}".
#' @param main An overall title for the plot (default=\code{""}).
#' @param xlab,ylab Titles for the \eqn{x} and \eqn{y} axes, respectively (default=\code{""} for both).
#' @param xlim,ylim Two \code{numeric} vectors of length 2, giving the \eqn{x}- and \eqn{y}-coordinate ranges
#' (default=\code{NULL} for both).
#' @param \dots Additional \code{plot} parameters.
#'
#' @return A plot of the arcs of the AS-PCD for a 2D data set \code{Xp} where AS proximity regions
#' are defined with respect to the triangle \code{tri}; also plots the triangle \code{tri}
#'
#' @seealso \code{\link{plotASarcsMT}}, \code{\link{plotPEarcsTri}}, \code{\link{plotPEarcsMT}},
#' \code{\link{plotCSarcsTri}}, and \code{\link{plotCSarcsMT}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#' n<-10
#'
#' set.seed(1)
#' dat<-runif.tri(n,Tr)$g  #try also dat<-cbind(runif(n,1,2),runif(n,0,2))
#'
#' M<-as.numeric(runif.tri(1,Tr)$g)  #try also  #M<-c(1.6,1.2)
#'
#' if (dimension(M)==3) {M<-bary2cart(M,Tr)}
#' #need to run this when M is given in barycentric coordinates
#'
#' CC<-circ.cent.tri(Tr)
#' if (isTRUE(all.equal(M,CC)) || identical(M,"CC"))
#' {cent<-CC
#' D1<-(B+C)/2; D2<-(A+C)/2; D3<-(A+B)/2;
#' Ds<-rbind(D1,D2,D3)
#' cent.name<-"CC"
#' } else
#' {cent<-M
#' cent.name<-"M"
#' Ds<-cp2e.tri(Tr,M)
#' D1<-Ds[1,]; D2<-Ds[2,]; D3<-Ds[3,]
#' }
#'
#' Xlim<-range(Tr[,1],dat[,1])
#' Ylim<-range(Tr[,2],dat[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plotASarcsTri(dat,Tr,M,main="arcs of AS-PCD",xlab="",ylab="",
#' xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' L<-rbind(cent,cent,cent); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#'
#' txt<-rbind(Tr,cent,Ds)
#' xc<-txt[,1]+c(-.02,.03,.03,.03,.05,-0.03,-.01)
#' yc<-txt[,2]+c(.02,.02,.02,.07,.02,.05,-.06)
#' txt.str<-c("A","B","C",cent.name,"D1","D2","D3")
#' text(xc,yc,txt.str)
#'
#' @export plotASarcsTri
plotASarcsTri <- function(Xp,tri,M="CC",asp=NA,main="",xlab="",ylab="",xlim=NULL,ylim=NULL, ...)
{
  ArcsAS<-ArcsAStri(Xp,tri,M)
  S<-ArcsAS$S
  E<-ArcsAS$E

  if (is.null(xlim))
  {xlim<-range(tri[,1],Xp[,1])}
  if (is.null(ylim))
  {ylim<-range(tri[,2],Xp[,2])}

  plot(Xp,main=main, asp=asp, xlab=xlab, ylab=ylab,xlim=xlim,ylim=ylim,pch=".",cex=3, ...)
  polygon(tri, ...)
  if (!is.null(S)) {arrows(S[,1], S[,2], E[,1], E[,2], length = 0.1, col= 4)}
} #end of the function
#'

#################################################################

#' @title The plot of the Arc Slice (AS) Proximity Regions for a 2D data set - one triangle case
#'
#' @description Plots the points in and outside of the triangle \code{tri} and also the AS proximity regions
#' for points in data set \code{Xp}.
#'
#' AS proximity regions are defined with respect to the triangle \code{tri},
#' so AS proximity regions are defined only for points inside the triangle \code{tri} and
#' vertex regions are based on the center \code{M="CC"} for
#' circumcenter of \code{tri}; or \eqn{M=(m_1,m_2)} in Cartesian coordinates or \eqn{M=(\alpha,\beta,\gamma)} in barycentric
#' coordinates in the interior of the triangle \code{tri}; default is \code{M="CC"} the circumcenter of \code{tri}.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:mcap2012;textual}{pcds}).
#'
#' @param Xp A set of 2D points for which AS proximity regions are constructed.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param M The center of the triangle. \code{"CC"} stands for circumcenter of the triangle \code{tri} or a 2D point in Cartesian coordinates or a 3D point in
#' barycentric coordinates which serves as a center in the interior of the triangle \eqn{T_b};
#' default is \code{M="CC"} i.e., the circumcenter of \code{tri}.
#' @param main An overall title for the plot (default=\code{""}).
#' @param xlab,ylab Titles for the \eqn{x} and \eqn{y} axes, respectively (default=\code{""} for both).
#' @param xlim,ylim Two \code{numeric} vectors of length 2, giving the \eqn{x}- and \eqn{y}-coordinate ranges
#' (default=\code{NULL} for both).
#' @param \dots Additional \code{plot} parameters.
#'
#' @return Plot of the AS proximity regions for points inside the triangle \code{tri} (and only the points outside \code{tri})
#'
#' @seealso \code{\link{plotASregsMT}}, \code{\link{plotPEregsTri}}, \code{\link{plotPEregsMT}},
#' \code{\link{plotCSregsTri}}, and \code{\link{plotCSregsMT}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#' n<-1
#'
#' set.seed(1)
#' dat<-runif.tri(n,Tr)$g
#'
#' M<-as.numeric(runif.tri(1,Tr)$g)  #try also  #M<-c(1.6,1.2);
#'
#' dat<-matrix(dat,ncol=2)
#' Xlim<-range(Tr[,1],dat[,1])
#' Ylim<-range(Tr[,2],dat[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plotASregsTri(dat,Tr,M,main="Proximity Regions for AS-PCD",
#' xlab="",ylab="",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#'
#' plotASarcsTri(dat,Tr,main="arcs of AS-PCD", xlab="",ylab="",xlim=Xlim+c(-.05,.05),ylim=Ylim)
#'
#' if (dimension(M)==3) {M<-bary2cart(M,Tr)}
#' #need to run this when M is given in barycentric coordinates
#'
#' CC<-circ.cent.tri(Tr)
#' if (isTRUE(all.equal(M,CC)) || identical(M,"CC"))
#' {cent<-CC
#' D1<-(B+C)/2; D2<-(A+C)/2; D3<-(A+B)/2;
#' Ds<-rbind(D1,D2,D3)
#' cent.name<-"CC"
#' } else
#' {cent<-M
#' cent.name<-"M"
#' Ds<-cp2e.tri(Tr,M)
#' D1<-Ds[1,]; D2<-Ds[2,]; D3<-Ds[3,]
#' }
#'
#' plotASregsTri(dat,Tr,M,main="Proximity Regions for AS-PCD",xlab="",ylab="")
#' L<-rbind(cent,cent,cent); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#'
#' txt<-rbind(Tr,cent,Ds)
#' xc<-txt[,1]+c(-.02,.03,.03,.03,.05,-0.03,-.01)
#' yc<-txt[,2]+c(.02,.02,.02,.07,.02,.05,-.06)
#' txt.str<-c("A","B","C",cent.name,"D1","D2","D3")
#' text(xc,yc,txt.str)
#'
#' @export plotASregsTri
plotASregsTri <- function(Xp,tri,M="CC",main="",xlab="",ylab="",xlim=NULL,ylim=NULL, ...)
{
  if (!is.numeric(as.matrix(Xp)) )
  {stop('Xp must be numeric')}

  if (is.point(Xp))
  { Xp<-matrix(Xp,ncol=2)
  } else
  {Xp<-as.matrix(Xp)
  if (ncol(Xp)!=2 )
  {stop('Xp must be of dimension nx2')}
  }

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  if (identical(M,"CC"))
  {M<-circ.cent.tri(tri)
  } else
  {
    if ( !is.point(M) && !is.point(M,3) )
    {stop('M must be the circumcenter "CC" or a numeric 2D point for Cartesian coordinates or
          3D point for barycentric coordinates')}

    if (dimension(M)==3)
    {
      M<-bary2cart(M,tri)
    }

    if (isTRUE(all.equal(M,circ.cent.tri(tri)))==F & in.triangle(M,tri,boundary=FALSE)$in.tri==F)
    {stop('center is not the circumcenter or not in the interior of the triangle')}
  }

  nx<-nrow(Xp)
 in.tri<-rep(0,nx)
  for (i in 1:nx)
   in.tri[i]<-in.triangle(Xp[i,],tri,boundary=TRUE)$in.tri #indices of the Xp points inside the triangle

  Xtri<-matrix(Xp[in.tri==1,],ncol=2)  #the Xp points inside the triangle
  nx2<-nrow(Xtri)

  if (is.null(xlim))
  {xlim<-range(tri[,1],Xp[,1])}
  if (is.null(ylim))
  {ylim<-range(tri[,2],Xp[,2])}

  xr<-xlim[2]-xlim[1]
  yr<-ylim[2]-ylim[1]

  plot(Xp,main=main, asp=1, xlab=xlab, ylab=ylab,xlim=xlim+xr*c(-.05,.05),
       ylim=ylim+yr*c(-.05,.05),pch=".",cex=3, ...)
  polygon(tri,lty=2)
  if (nx2>=1)
  {
    for (i in 1:nx2)
    {
      P1<-Xtri[i,]
      rv<-ifelse(isTRUE(all.equal(M,circ.cent.tri(tri)))==T,rv.triCC(P1,tri)$rv,rv.tri.cent(P1,tri,M)$rv)  #vertex region for pt
      RV<-tri[rv,]
      rad<-Dist(P1,RV)

      Int.Pts<-NAStri(P1,tri,M)
      L<-Int.Pts$L; R<-Int.Pts$R
      segments(L[,1], L[,2], R[,1], R[,2], lty=1,col=2)
      Arcs<-Int.Pts$A;
      if (!is.null(Arcs))
      {
        K<-nrow(Arcs)/2
        for (j in 1:K)
        {A1<-Arcs[2*j-1,]; A2<-Arcs[2*j,];
        angles<-angle.str2end(A1,P1,A2)$c

        test.ang1<-angles[1]+(.01)*(angles[2]-angles[1])
        test.Pnt<-P1+rad*c(cos(test.ang1),sin(test.ang1))
        if (!in.triangle(test.Pnt,tri,boundary = T)$i) {angles<-c(min(angles),max(angles)-2*pi)}
        plotrix::draw.arc(P1[1],P1[2],rad,angle1=angles[1],angle2=angles[2],col=2)
        }
      }
    }
  }
} #end of the function
#'

#################################################################

#' @title The arcs of Arc Slice Proximity Catch Digraph (AS-PCD) for a 2D data set - multiple triangle case
#'
#' @description
#' An object of class \code{"PCDs"}.
#' Returns arcs as tails (or sources) and heads (or arrow ends) of AS-PCD whose vertices are the data set \code{Xp}.
#'
#' AS proximity regions are defined with respect to the Delaunay triangles based on
#' \code{Yp} points, i.e., AS proximity regions are defined only for \code{Xp} points inside the convex hull of \code{Yp} points.
#' That is, arcs may exist for points only inside the convex hull of \code{Yp} points.
#' It also provides various descriptions and quantities about the arcs of the AS-PCD
#' such as number of arcs, arc density, etc.
#'
#' Vertex regions are based on the center \code{M="CC"}
#' for circumcenter of each Delaunay triangle or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the
#' interior of each Delaunay triangle; default is \code{M="CC"} i.e., circumcenter of each triangle.
#'
#' See (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:mcap2012;textual}{pcds}) for more on AS PCDs.
#' Also see (\insertCite{okabe:2000,ceyhan:comp-geo-2010,sinclair:2016;textual}{pcds}) for more on Delaunay triangulation and the corresponding algorithm.
#'
#' @param Xp A set of 2D points which constitute the vertices of the AS-PCD.
#' @param Yp A set of 2D points which constitute the vertices of the Delaunay triangulation. The Delaunay
#' triangles partition the convex hull of \code{Yp} points.
#' @param M The center of the triangle. \code{"CC"} represents the circumcenter of each Delaunay triangle \code{tri}
#' or 3D point in barycentric coordinates which serves as a center in the interior of each Delaunay triangle;
#' default is \code{M="CC"} i.e., the circumcenter of each triangle.
#'
#' @return A \code{list} with the elements
#' \item{type}{A description of the type of the digraph}
#' \item{parameters}{Parameters of the digraph, here, it is the center used to construct the vertex regions, i.e.
#' circumcenter.}
#' \item{tess.points}{Points on which the tessellation of the study region is performed, here, tessellation
#' is the Delaunay triangulation based on \code{Yp} points.}
#' \item{tess.name}{Name of data set used in tessellation, i.e., \code{Yp}}
#' \item{vertices}{Vertices of the digraph, \code{Xp}.}
#' \item{vert.name}{Name of the data set which constitute the vertices of the digraph}
#' \item{S}{Tails (or sources) of the arcs of AS-PCD for 2D data set \code{Xp} in the multiple triangle case
#' as the vertices of the digraph}
#' \item{E}{Heads (or arrow ends) of the arcs of AS-PCD for 2D data set \code{Xp} in the multiple triangle case
#' as the vertices of the digraph}
#' \item{mtitle}{Text for \code{"main"} title in the plot of the digraph}
#' \item{quant}{Various quantities for the digraph: number of vertices, number of partition points,
#' number of intervals, number of arcs, and arc density.}
#'
#' @seealso \code{\link{ArcsAStri}}, \code{\link{ArcsPEtri}}, \code{\link{ArcsCStri}}, \code{\link{ArcsPEMT}},
#' and \code{\link{ArcsCSMT}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' #nx is number of X points (target) and ny is number of Y points (nontarget)
#' nx<-20; ny<-4;  #try also nx<-40; ny<-10 or nx<-1000; ny<-10;
#'
#' set.seed(1)
#' Xp<-cbind(runif(nx,0,1),runif(nx,0,1))
#' Yp<-cbind(runif(ny,0,1),runif(ny,0,1))
#'
#' M<-c(1,1,1)  #try also M<-c(1,2,3)
#'
#' ArcsASMT(Xp,Yp,M)
#'
#' Arcs<-ArcsASMT(Xp,Yp,M)
#' Arcs
#' summary(Arcs)
#' plot(Arcs)
#'
#' S<-Arcs$S
#' E<-Arcs$E
#' DT<-interp::tri.mesh(Yp[,1],Yp[,2],duplicate="remove")
#'
#' Xlim<-range(Xp[,1],Yp[,1])
#' Ylim<-range(Xp[,2],Yp[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(Xp,main=" ", xlab=" ", ylab=" ",xlim=Xlim+xd*c(-.05,.05),
#' ylim=Ylim+yd*c(-.05,.05),pch=".",cex=3)
#' interp::plot.triSht(DT, add=TRUE, do.points = TRUE)
#' arrows(S[,1], S[,2], E[,1], E[,2], length = 0.1, col= 4)
#'
#' ArcsASMT(Xp,Yp[1:3,],M)
#'
#' ArcsASMT(Xp,rbind(Yp,Yp),M)
#'
#' dat.fr<-data.frame(a=Xp)
#' ArcsASMT(dat.fr,Yp,M)
#'
#' @export ArcsASMT
ArcsASMT <- function(Xp,Yp,M="CC")
{
  if (!is.numeric(as.matrix(Xp)) || !is.numeric(as.matrix(Yp)))
  {stop('Xp and Yp must be numeric')}

  if (is.point(Xp))
  { Xp<-matrix(Xp,ncol=2)
  } else
  {Xp<-as.matrix(Xp)
  if (ncol(Xp)!=2 )
  {stop('Xp must be of dimension nx2')}
  }

  Yp<-as.matrix(Yp)
  if (ncol(Yp)!=2 || nrow(Yp)<3)
  {stop('Yp must be of dimension kx2 with k>=3')}

  if (nrow(Yp)==3)
  {
    res<-ArcsAStri(Xp,Yp,M)
  } else
  {
    if ((!is.point(M,3) && M!="CC") || !all(M>0))
    {stop('M must be a numeric 3D point with positive barycentric coordinates or "CC" for circumcenter')}

    DTmesh<-interp::tri.mesh(Yp[,1],Yp[,2],duplicate="remove")

    nx<-nrow(Xp)
    ch<-rep(0,nx)
    for (i in 1:nx)
      ch[i]<-interp::in.convex.hull(DTmesh,Xp[i,1],Xp[i,2])

    Xch<-matrix(Xp[ch==1,],ncol=2)  #the Xp points inside the convex hull of Yp points

    DTr<-matrix(interp::triangles(DTmesh)[,1:3],ncol=3)
    nt<-nrow(DTr)
    nx2<-nrow(Xch)  #number of Xp points inside the convex hull of Yp points

    S<-E<-NULL #S is for source and E is for end points for the arcs
    if (nx2>1)
    {
      i.tr<-rep(0,nx2)  #the vector of indices for the triangles that contain the Xp points
      for (i in 1:nx2)
        for (j in 1:nt)
        {
          tri<-Yp[DTr[j,],]
          if (in.triangle(Xch[i,],tri,boundary=TRUE)$in.tri )
            i.tr[i]<-j
        }

      for (i in 1:nt)
      {
        Xl<-matrix(Xch[i.tr==i,],ncol=2)
        if (nrow(Xl)>1)
        {
          Yi.Tri<-Yp[DTr[i,],] #vertices of the ith triangle
          Yi.tri<-as.bastri(Yi.Tri)$tri #convert the triangle Tri into an unscaled basic triangle, see as.bastri help page
          nl<-nrow(Xl)
          ifelse(identical(M,"CC"), rv.ind<-rverts.triCC(Xl,tri=Yi.tri)$rv,
                 rv.ind<-rverts.tri.cent(Xl,tri=Yi.tri,M)$rv)  #vertex region for pt
          for (j in 1:nl)
          {RV<-rv.ind[j]
          for (k in (1:nl)[-j])  # to avoid loops
            if (IndNAStri(Xl[j,],Xl[k,],Yi.tri,M,rv=RV)==1 )
            {
              S <-rbind(S,Xl[j,]); E <-rbind(E,Xl[k,]);
            }
          }
        }
      }
    }

    xname <-deparse(substitute(Xp))
    yname <-deparse(substitute(Yp))
    cname <-ifelse(identical(M,"CC"),"CC","M")

    param<-ifelse(identical(M,"CC"),"Circumcenter","Center M")
    names(param)<-NULL

    typ<-paste("Arc Slice Proximity Catch Digraph (AS-PCD) for 2D Points in Multiple Triangles with Center ", cname,sep="")

    main.txt<-paste("Arcs of AS-PCD for Points in One Triangle \n with Center ", cname,sep="")


    nvert<-nx2; ny<-nrow(Yp); ntri<-nt; narcs=ifelse(!is.null(S),nrow(S),0);
    arc.dens<-ifelse(nvert>1,narcs/(nvert*(nvert-1)),NA)

    quantities<-c(nvert,ny,ntri,narcs,arc.dens)
    names(quantities)<-c("number of vertices", "number of partition points",
                         "number of triangles","number of arcs", "arc density")

    res<-list(
      type=typ,
      parameters=param,
      tess.points=Yp, tess.name=yname, #tessellation points
      vertices=Xp, vert.name=xname, #vertices of the digraph
      S=S,
      E=E,
      mtitle=main.txt,
      quant=quantities
    )

    class(res)<-"PCDs"
    res$call <-match.call()
  }
  res
} #end of the function
#'

#################################################################

#' @title The indices of the vertex regions in a triangle that contains the points in a give data set
#'
#' @description Returns the indices of the vertices whose regions contain the points in data set \code{Dt} in
#' a triangle \code{tri}\eqn{=T(A,B,C)}.
#'
#' Vertex regions are based on center \eqn{M=(m_1,m_2)} in Cartesian
#' coordinates or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the interior of the triangle
#' to the edges on the extension of the lines joining \code{M} to the vertices or based on the circumcenter of \code{tri}.
#' Vertices of triangle \code{tri} are labeled as \eqn{1,2,3} according to the row number the vertex is recorded.
#'
#' If a point in \code{Dt} is not inside \code{tri}, then the function yields \code{NA} as output for that entry.
#' The corresponding vertex region is the polygon with the vertex, \code{M}, and
#' projection points from \code{M} to the edges crossing the vertex (as the output of \code{cp2e.tri(Tr,M)})
#' or \eqn{CC}-vertex region. (see the examples for an illustration).
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:dom-num-NPE-Spat2011,ceyhan:comp-geo-2010,ceyhan:mcap2012;textual}{pcds}).
#'
#' @param Dt A set of 2D points representing the set of data points for which indices of the vertex regions
#' containing them are to be determined.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param M A 2D point in Cartesian coordinates or a 3D point in barycentric coordinates
#' which serves as a center in the interior of the triangle \code{tri} or the circumcenter of \code{tri}.
#'
#' @return A \code{list} with two elements
#' \item{rv}{Indices of the vertices whose regions contains points in \code{Dt}.}
#' \item{tri}{The vertices of the triangle, where row number corresponds to the vertex index in \code{rv}.}
#'
#' @seealso \code{\link{rverts.triCM}}, \code{\link{rverts.triCC}} and \code{\link{rverts.tri.nd}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#' M<-c(1.6,1.0)
#'
#' P<-c(.4,.2)
#' rverts.tri.cent(P,Tr,M)
#'
#' P<-c(1.8,.5)
#' rverts.tri.cent(P,Tr,M)
#'
#' P<-c(1.5,1.6)
#' rverts.tri.cent(P,Tr,M)
#'
#' n<-10  #try also n<-20
#' set.seed(1)
#' dat<-runif.tri(n,Tr)$g
#'
#' M<-as.numeric(runif.tri(1,Tr)$g)  #try also  #M<-c(1.6,1.0)
#'
#' rverts.tri.cent(dat,Tr,M)
#' rverts.tri.cent(rbind(dat,c(2,2)),Tr,M)
#'
#' rv<-rverts.tri.cent(dat,Tr,M)
#' rv
#'
#' ifelse(identical(M,circ.cent.tri(Tr)),
#' Ds<-rbind((B+C)/2,(A+C)/2,(A+B)/2),Ds<-cp2e.tri(Tr,M))
#'
#' Xlim<-range(Tr[,1],M[1],dat[,1])
#' Ylim<-range(Tr[,2],M[2],dat[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' if (dimension(M)==3) {M<-bary2cart(M,Tr)}
#' #need to run this when M is given in barycentric coordinates
#'
#' plot(Tr,pch=".",xlab="",ylab="",axes=TRUE,
#' xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tr)
#' points(dat,pch=".",col=1)
#' L<-rbind(M,M,M); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#'
#' xc<-Tr[,1]
#' yc<-Tr[,2]
#' txt.str<-c("rv=1","rv=2","rv=3")
#' text(xc,yc,txt.str)
#'
#' txt<-rbind(M,Ds)
#' xc<-txt[,1]+c(.02,.04,-.03,0)
#' yc<-txt[,2]+c(.07,.04,.05,-.07)
#' txt.str<-c("M","D1","D2","D3")
#' text(xc,yc,txt.str)
#'
#' text(dat,labels=factor(rv$rv))
#'
#' P<-c(1.4,1.0)
#' rverts.tri.cent(P,Tr,M)
#' rverts.tri.cent(dat,Tr,M)
#'
#' rverts.tri.cent(rbind(dat,dat),Tr,M)
#'
#' dat.fr<-data.frame(a=dat)
#' rverts.tri.cent(dat.fr,Tr,M)
#'
#' dat.fr<-data.frame(a=Tr)
#' rverts.tri.cent(dat,dat.fr,M)
#'
#' @export rverts.tri.cent
rverts.tri.cent <- function(Dt,tri,M)
{
  if (!is.numeric(as.matrix(Dt)))
  {stop('Dt must be numeric')}

  if (is.point(Dt))
  { Dt<-matrix(Dt,ncol=2)
  } else
  {Dt<-as.matrix(Dt)
  if (ncol(Dt)!=2 )
  {stop('Dt must be of dimension nx2')}
  }

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  if (!is.point(M) && !is.point(M,3) )
  {stop('M must be a numeric 2D point for Cartesian coordinates or 3D point for barycentric coordinates')}

  if (dimension(M)==3)
  {
    M<-bary2cart(M,tri)
  }

  if (isTRUE(all.equal(M,circ.cent.tri(tri)))==F & in.triangle(M,tri,boundary=FALSE)$in.tri==F)
  {stop('center is not the circumcenter or not in the interior of the triangle')}

  nt<-nrow(Dt)

  if (nt==0)
  {ind.set<-NA}
  else
  {
    if (isTRUE(all.equal(M,circ.cent.tri(tri)))==T)
    {
     ind.set<-rverts.triCC(Dt,tri)$rv
    } else
    {
      A<-tri[1,]; a1<-A[1]; a2<-A[2];
      B<-tri[2,]; b1<-B[1]; b2<-B[2];
      C<-tri[3,]; c1<-C[1]; c2<-C[2];

      m1<-M[1]; m2<-M[2]

      M11<--(a1*b1*c2-a1*b1*m2-a1*b2*c1+a1*c1*m2+a2*b1*m1-a2*c1*m1-b1*c2*m1+b2*c1*m1)/(b2*a1-c2*a1-a2*b1+a2*c1+m2*b1-b2*m1-m2*c1+c2*m1);
      M12<-(a1*b2*m2-a1*c2*m2-a2*b1*c2+a2*b2*c1-a2*b2*m1+a2*c2*m1+b1*c2*m2-b2*c1*m2)/(b2*a1-c2*a1-a2*b1+a2*c1+m2*b1-b2*m1-m2*c1+c2*m1);
      D1<-c(M11,M12)
      M21 <-(a1*b1*c2-a1*b1*m2+a1*b2*m1-a1*c2*m1-a2*b1*c1+a2*c1*m1+b1*c1*m2-b2*c1*m1)/(b2*a1-m2*a1-a2*b1+m1*a2+c2*b1-c1*b2+m2*c1-c2*m1);
      M22 <-(a1*b2*c2-a1*c2*m2-a2*b1*m2-a2*b2*c1+a2*b2*m1+a2*c1*m2+b1*c2*m2-b2*c2*m1)/(b2*a1-m2*a1-a2*b1+m1*a2+c2*b1-c1*b2+m2*c1-c2*m1);
      D2<-c(M21,M22)
      M31 <- (a1*b2*c1-a1*b2*m1-a1*c1*m2+a1*c2*m1-a2*b1*c1+a2*b1*m1+b1*c1*m2-b1*c2*m1)/(c2*a1-m2*a1-a2*c1+m1*a2-c2*b1+m2*b1+c1*b2-b2*m1);
      M32 <- (a1*b2*c2-a1*b2*m2-a2*b1*c2+a2*b1*m2-a2*c1*m2+a2*c2*m1+b2*c1*m2-b2*c2*m1)/(c2*a1-m2*a1-a2*c1+m1*a2-c2*b1+m2*b1+c1*b2-b2*m1);
      D3<-c(M31,M32)

      t.mA<-interp::tri.mesh(c(A[1],D3[1],M[1],D2[1]),c(A[2],D3[2],M[2],D2[2]),duplicate="remove")
      t.mB<-interp::tri.mesh(c(B[1],D1[1],M[1],D3[1]),c(B[2],D1[2],M[2],D3[2]),duplicate="remove")
      t.mC<-interp::tri.mesh(c(C[1],D2[1],M[1],D1[1]),c(C[2],D2[2],M[2],D1[2]),duplicate="remove")

     ind.set<-rep(NA,nt)
     ind.vA<-interp::in.convex.hull(t.mA,Dt[,1],Dt[,2])
     ind.vB<-interp::in.convex.hull(t.mB,Dt[,1],Dt[,2])
     ind.vC<-interp::in.convex.hull(t.mC,Dt[,1],Dt[,2])

     ind.set[ind.vA==TRUE]<-1
     ind.set[ind.vB==TRUE]<-2
     ind.set[ind.vC==TRUE]<-3
    }
  }
  row.names(tri)<-c("vertex 1","vertex 2","vertex 3")  #vertex labeling

  list(rv=ind.set, #relative vertices
       tri=tri #vertex labeling
  )
} #end of the function
#'

#################################################################

#' @title The plot of the arcs of Arc Slice Proximity Catch Digraph (AS-PCD) for a 2D data set -
#' multiple triangle case
#'
#' @description Plots the arcs of AS-PCD whose vertices are the data points in \code{Xp} and Delaunay triangles based on \code{Yp} points.
#'
#' AS proximity regions are constructed with respect to the Delaunay triangles based on \code{Yp} points, i.e.,
#' AS proximity regions are defined only for \code{Xp} points inside the convex hull of \code{Yp} points.
#' That is, arcs may exist for \code{Xp} points only inside the convex hull of \code{Yp} points.
#'
#' Vertex regions are based on the center \code{M="CC"}
#' for circumcenter of each Delaunay triangle or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the
#' interior of each Delaunay triangle; default is \code{M="CC"} i.e., circumcenter of each triangle.
#'
#' See (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:mcap2012;textual}{pcds}) for more on AS-PCDs.
#' Also see (\insertCite{okabe:2000,ceyhan:comp-geo-2010,sinclair:2016;textual}{pcds}) for more on Delaunay triangulation and the corresponding algorithm.
#'
#' @param Xp A set of 2D points which constitute the vertices of the AS-PCD.
#' @param Yp A set of 2D points which constitute the vertices of the Delaunay triangulation. The Delaunay
#' triangles partition the convex hull of \code{Yp} points.
#' @param M The center of the triangle. \code{"CC"} stands for circumcenter of each Delaunay triangle \code{tri}
#' or 3D point in barycentric coordinates which serves as a center in the interior of each Delaunay triangle;
#' default is \code{M="CC"} i.e., the circumcenter of each triangle.
#' @param asp A \code{numeric} value, giving the aspect ratio for \eqn{y} axis to \eqn{x}-axis \eqn{y/x} (default is \code{NA}),
#' see the official help page for \code{asp} by typing "\code{? asp}".
#' @param main An overall title for the plot (default=\code{""}).
#' @param xlab,ylab Titles for the \eqn{x} and \eqn{y} axes, respectively (default=\code{""} for both).
#' @param xlim,ylim Two \code{numeric} vectors of length 2, giving the \eqn{x}- and \eqn{y}-coordinate ranges
#' (default=\code{NULL} for both).
#' @param \dots Additional \code{plot} parameters.
#'
#' @return A plot of the arcs of the AS-PCD for a 2D data set \code{Xp} where AS proximity regions
#' are defined with respect to the Delaunay triangles based on \code{Yp} points; also plots the Delaunay triangles
#' based on \code{Yp} points.
#'
#' @seealso \code{\link{plotASarcsTri}}, \code{\link{plotPEarcsTri}}, \code{\link{plotPEarcsMT}},
#' \code{\link{plotCSarcsTri}}, and \code{\link{plotCSarcsMT}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' #nx is number of X points (target) and ny is number of Y points (nontarget)
#' nx<-20; ny<-4;  #try also nx<-40; ny<-10 or nx<-1000; ny<-10;
#'
#' set.seed(1)
#' Xp<-cbind(runif(nx,0,1),runif(nx,0,1))
#' Yp<-cbind(runif(ny,0,1),runif(ny,0,1))
#'
#' M<-c(1,1,1)  #try also M<-c(1,2,3)
#'
#' plotASarcsMT(Xp,Yp,M,xlab="",ylab="")
#' plotASarcsMT(Xp,Yp,M,asp=1,xlab="",ylab="")
#'
#' plotASarcsMT(Xp,Yp[1:3,],M,xlab="",ylab="")
#'
#' Xlim<-range(Xp[,1],Yp[,1])
#' Ylim<-range(Xp[,2],Yp[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plotASarcsMT(Xp,Yp,M,xlab="",ylab="",xlim=Xlim+xd*c(-.01,.01),ylim=Ylim+yd*c(-.01,.01))
#' plotASarcsMT(Xp,Yp,M,asp=1,xlab="",ylab="",xlim=Xlim+xd*c(-.01,.01),ylim=Ylim+yd*c(-.01,.01))
#'
#' @export plotASarcsMT
plotASarcsMT <- function(Xp,Yp,M="CC",asp=NA,main="",xlab="",ylab="",xlim=NULL,ylim=NULL, ...)
{
  Yp<-as.matrix(Yp)
  if (ncol(Yp)!=2 || nrow(Yp)<3)
  {stop('Yp must be of dimension kx2 with k>=3')}

  if (nrow(Yp)==3)
  {
    plotASarcsTri(Xp,Yp,M,asp,main,xlab,ylab,xlim,ylim)
  } else
  {
    ArcsAS<-ArcsASMT(Xp,Yp,M)
    S<-ArcsAS$S
    E<-ArcsAS$E

    DTmesh<-interp::tri.mesh(Yp[,1],Yp[,2],duplicate="remove")

    if (is.null(xlim))
    {xlim<-range(Yp[,1],Xp[,1])}
    if (is.null(ylim))
    {ylim<-range(Yp[,2],Xp[,2])}

    xr<-xlim[2]-xlim[1]
    yr<-ylim[2]-ylim[1]

    plot(rbind(Xp),asp=asp,main=main, xlab=xlab, ylab=ylab,xlim=xlim+xr*c(-.05,.05),
         ylim=ylim+yr*c(-.05,.05),pch=".",cex=3, ...)
   interp::plot.triSht(DTmesh, add=TRUE, do.points = TRUE, ...)
    if (!is.null(S)) {arrows(S[,1], S[,2], E[,1], E[,2], length = 0.1, col= 4)}
  }
} #end of the function
#'

#################################################################

#' @title The plot of the Arc Slice (AS) Proximity Regions for a 2D data set - multiple triangle case
#'
#' @description Plots the \code{Xp} points in and outside of the convex hull of \code{Yp} points and also plots the AS proximity regions
#' for \code{Xp} points and Delaunay triangles based on \code{Yp} points.
#'
#' AS proximity regions are constructed with respect
#' to the Delaunay triangles based on \code{Yp} points (these triangles partition the convex hull of \code{Yp} points),
#' i.e., AS proximity regions are only defined for \code{Xp} points inside the convex hull of \code{Yp} points.
#'
#' Vertex regions are based on the center \code{M="CC"}
#' for circumcenter of each Delaunay triangle or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the
#' interior of each Delaunay triangle; default is \code{M="CC"} i.e., circumcenter of each triangle.
#'
#' See (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:mcap2012;textual}{pcds}) for more on AS-PCDs.
#' Also see (\insertCite{okabe:2000,ceyhan:comp-geo-2010,sinclair:2016;textual}{pcds}) for more on Delaunay triangulation and the corresponding algorithm.
#'
#' @param Xp A set of 2D points for which AS proximity regions are constructed.
#' @param Yp A set of 2D points which constitute the vertices of the Delaunay triangulation. The Delaunay
#' triangles partition the convex hull of \code{Yp} points.
#' @param M The center of the triangle. \code{"CC"} stands for circumcenter of each Delaunay triangle \code{tri} or 3D point in barycentric
#' coordinates which serves as a center in the interior of each Delaunay triangle;
#' default is \code{M="CC"} i.e., the circumcenter of each triangle.
#' @param main An overall title for the plot (default=\code{""}).
#' @param xlab,ylab Titles for the \eqn{x} and \eqn{y} axes, respectively (default=\code{""} for both).
#' @param xlim,ylim Two \code{numeric} vectors of length 2, giving the \eqn{x}- and \eqn{y}-coordinate ranges
#' (default=\code{NULL} for both).
#' @param \dots Additional \code{plot} parameters.
#'
#' @return Plot of the \code{Xp} points, Delaunay triangles based on \code{Yp} and also the AS proximity regions
#' for \code{Xp} points inside the convex hull of \code{Yp} points
#'
#' @seealso \code{\link{plotASregsTri}}, \code{\link{plotPEregsTri}}, \code{\link{plotPEregsMT}},
#' \code{\link{plotCSregsTri}}, and \code{\link{plotCSregsMT}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' nx<-10 ; ny<-10
#'
#' set.seed(1)
#' Xp<-cbind(runif(nx,0,1),runif(nx,0,1))
#' Yp<-cbind(runif(ny,0,1),runif(ny,0,1))
#'
#' M<-c(1,1,1)  #try also M<-c(1,2,3)
#'
#' plotASregsMT(Xp,Yp,M,xlab="",ylab="")
#'
#' Xlim<-range(Xp[,1],Yp[,1])
#' Ylim<-range(Xp[,2],Yp[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plotASregsMT(Xp,Yp,M,xlab="",ylab="",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' plotASregsMT(Xp,Yp[1:3,],M,xlab="",ylab="",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#'
#' Xp<-c(.5,.5)
#' Xlim<-Ylim<-range(Xp,Yp)
#' plotASregsMT(Xp,Yp,M,xlab="",ylab="",xlim=Xlim,ylim=Ylim)
#'
#' @export plotASregsMT
plotASregsMT <- function(Xp,Yp,M="CC",main="",xlab="",ylab="",xlim=NULL,ylim=NULL, ...)
{
  if (!is.numeric(as.matrix(Xp)) || !is.numeric(as.matrix(Yp)))
  {stop('Xp and Yp must be numeric')}

  if (is.point(Xp))
  { Xp<-matrix(Xp,ncol=2)
  } else
  {Xp<-as.matrix(Xp)
  if (ncol(Xp)!=2 )
  {stop('Xp must be of dimension nx2')}
  }

  Yp<-as.matrix(Yp)
  if (ncol(Yp)!=2 || nrow(Yp)<3)
  {stop('Yp must be of dimension kx2 with k>=3')}

  if (nrow(Yp)==3)
  {
    plotASregsTri(Xp,Yp,M,main,xlab,ylab,xlim,ylim)
  } else
  {

    if ((!is.point(M,3) && M!="CC") || !all(M>0))
    {stop('M must be a numeric 3D point with positive barycentric coordinates or "CC" for circumcenter')}

    DTmesh<-interp::tri.mesh(Yp[,1],Yp[,2],duplicate="remove")

    nx<-nrow(Xp)
    ch<-rep(0,nx)
    for (i in 1:nx)
      ch[i]<-interp::in.convex.hull(DTmesh,Xp[i,1],Xp[i,2])

    Xch<-matrix(Xp[ch==1,],ncol=2)  #the Xp points inside the convex hull of Yp points

    DTr<-matrix(interp::triangles(DTmesh)[,1:3],ncol=3)
    nt<-nrow(DTr)  #number of Delaunay triangles
    nx2<-nrow(Xch)  #number of Xp points inside the convex hull of Yp points

    if (is.null(xlim))
    {xlim<-range(Yp[,1],Xp[,1])
    xd<-xlim[2]-xlim[1]
    xlim<-xlim+xd*c(-.05,.05)}
    if (is.null(ylim))
    {ylim<-range(Yp[,2],Xp[,2])
    yd<-ylim[2]-ylim[1]
    ylim<-ylim+yd*c(-.05,.05)}

    plot(rbind(Xp),asp=1,main=main, xlab=xlab, ylab=ylab,
         xlim=xlim,ylim=ylim,pch=".",cex=3, ...)

    if (nx2==0)
    {
      for (i in 1:nt)
      {
        tri<-Yp[DTr[i,],]  #vertices of the ith triangle
        polygon(tri,lty=2)
      }
    } else
    {
      i.tr<-rep(0,nx2)  #the vector of indices for the triangles that contain the Xp points
      for (i1 in 1:nx2)
        for (j1 in 1:nt)
        {
          Tri<-Yp[DTr[j1,],]
          if (in.triangle(Xch[i1,],Tri,boundary=TRUE)$in.tri )
            i.tr[i1]<-j1
        }


      for (i in 1:nt)
      {
        Tri<-Yp[DTr[i,],] #vertices of the ith triangle
        tri<-as.bastri(Tri)$tri #convert the triangle Tri into an unscaled basic triangle, see as.bastri help page

        polygon(tri,lty=2)
        Xtri<-matrix(Xch[i.tr==i,],ncol=2)  #Xp points inside triangle i
        ni<-nrow(Xtri)
        if (ni>=1)
        {
          ################
          for (j in 1:ni)
          {
            P1<-Xtri[j,]
            rv<-ifelse(identical(M,"CC"), rv.triCC(P1,tri)$rv, rv.tri.cent(P1,tri,M)$rv)  #vertex region for P1
            RV<-tri[rv,]
            rad<-Dist(P1,RV)

            Int.Pts<-NAStri(P1,tri,M)
            L<-Int.Pts$L; R<-Int.Pts$R
            segments(L[,1], L[,2], R[,1], R[,2], lty=1,col=2)
            Arcs<-Int.Pts$A;
            if (!is.null(Arcs))
            {
              K<-nrow(Arcs)/2
              for (k in 1:K)
              {A1<-Arcs[2*k-1,]; A2<-Arcs[2*k,];
              angles<-angle.str2end(A1,P1,A2)$c

              test.ang1<-angles[1]+(.01)*(angles[2]-angles[1])
              test.Pnt<-P1+rad*c(cos(test.ang1),sin(test.ang1))
              if (!in.triangle(test.Pnt,tri,boundary = T)$i) {angles<-c(min(angles),max(angles)-2*pi)}
              plotrix::draw.arc(P1[1],P1[2],rad,angle1=angles[1],angle2=angles[2],col=2)
              }
            }
          }
          ################
        }
      }
    }
  }
} #end of the function
#'

##################################################################################################################
################################PROPORTIONAL EDGE FUNCTIONS#######################################################
##################################################################################################################
#NPE Functions.r;
#################################################################
#Functions for NPE in R^1
#################################################################

#' @title Parameterized center of an interval
#'
#' @description Returns the (parameterized) center, \eqn{M_c}, of the interval, \code{int}\eqn{=(a,b)}, parameterized by \eqn{c \in (0,1)}
#' so that \eqn{100c} \% of the length of interval is to the left of \eqn{M_c} and \eqn{100(1-c)} \% of the length of the interval
#' is to the right of \eqn{M_c}. That is, for the interval, \code{int}\eqn{=(a,b)}, the parameterized center is \eqn{M_c=a+c(b-a)}.
#'
#' See also (\insertCite{ceyhan:metrika-2012,ceyhan:revstat-2016;textual}{pcds}).
#'
#' @param int A \code{vector} with two entries representing an interval.
#' @param c A positive real number in \eqn{(0,1)} parameterizing the center inside \code{int}\eqn{=(a,b)}.
#' For the interval, \code{int}\eqn{=(a,b)}, the parameterized center is \eqn{M_c=a+c(b-a)}.
#'
#' @return (parameterized) center inside \code{int}
#'
#' @seealso \code{\link{centersMc}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' c<-.4
#' a<-0; b<-10; int<-c(a,b)
#' int
#' centMc(int,c)
#'
#' c<-.3
#' a<-2; b<-4; int<-c(a,b)
#' centMc(int,c)
#'
#' @export centMc
centMc <- function(int,c)
{
  if (!is.point(int))
  {stop('int must a numeric vector of length 2')}

  y1<-int[1]; y2<-int[2];
  if (y1>=y2)
  {stop('interval is degenerate or void, left end must be smaller than right end')}

  if (!is.point(c,1) || c<=0 || c>=1)
  {stop('c must be a scalar in (0,1)')}

  a<-int[1]; b<-int[2]
  Mc<-a+c*(b-a)
  Mc
} #end of the function
#'

#################################################################

#' @title Parameterized centers of intervals
#'
#' @description Returns the centers of the intervals based on 1D points in \eqn{x} parameterized by \eqn{c \in (0,1)}
#' so that \eqn{100c} \% of the length of interval is to the left of \eqn{M_c} and \eqn{100(1-c)} \% of the length of the interval
#' is to the right of \eqn{M_c}. That is, for an interval \eqn{(a,b)}, the parameterized center is \eqn{M_c=a+c(b-a)}
#' \code{x} is a \code{vector} of 1D points, not necessarily sorted.
#'
#' See also (\insertCite{ceyhan:metrika-2012,ceyhan:revstat-2016;textual}{pcds}).
#'
#' @param x A \code{vector} real numbers that constitute the end points of intervals.
#' @param c A positive real number in \eqn{(0,1)} parameterizing the centers inside the intervals.
#' For the interval, \code{int}\eqn{=(a,b)}, the parameterized center is \eqn{M_c=a+c(b-a)}.
#'
#' @return (parameterized) centers of the intervals based on \code{x} points as a vector
#'
#' @seealso \code{\link{centMc}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' n<-10
#' c<-.4  #try also c<-runif(1)
#' x<-runif(n)
#' centersMc(x,c)
#'
#' n<-10
#' c<-.3  #try also c<-runif(1)
#' x<-runif(n,0,10)
#' centersMc(x,c)
#'
#' @export centersMc
centersMc <- function(x,c)
{
  if (!is.point(x,length(x)))
  {stop('x must be a 1D vector of numerical entries')}

  if (!is.point(c,1) || c<=0 || c>=1 )
  {stop('c must be a scalar in (0,1)')}

  x<-sort(x)
  n<-length(x)
  Mcvec<-vector()
  for  (i in 1:(n-1))
  {int<-c(x[i],x[i+1])
  Mcvec<-c(Mcvec,centMc(int,c))
  }
  Mcvec
} #end of the function
#'

#################################################################

#' @title The index of the vertex region in a middle interval that contains a given point
#'
#' @description Returns the index of the vertex whose region contains point \code{pt} in
#' the interval \code{int}\eqn{=(a,b)=}(vertex 1,vertex 2) with (parameterized) center \eqn{M_c} associated with
#' the centrality parameter \eqn{c \in (0,1)}; vertices of interval are labeled as 1 and 2 according to their
#' order in the interval \code{int}. If the point, \code{pt}, is not inside \code{int}, then the function yields \code{NA} as output.
#' The corresponding vertex region is the interval \eqn{(a,b)} as \eqn{(a,M_c)} and \eqn{(M_c,b)} where \eqn{M_c=a+c(b-a)}.
#'
#' See also (\insertCite{ceyhan:metrika-2012,ceyhan:revstat-2016;textual}{pcds}).
#'
#' @param pt A 1D point. The vertex region \code{pt} resides is to be found.
#' @param c A positive real number in \eqn{(0,1)} parameterizing the center inside \code{int}\eqn{=(a,b)}.
#' For the interval, \code{int}\eqn{=(a,b)}, the parameterized center is \eqn{M_c=a+c(b-a)}.
#' @param int A \code{vector} of two real numbers representing an interval.
#'
#' @return A \code{list} with two elements
#' \item{rv}{Index of the vertex in the interval \code{int} whose region contains point, \code{pt}.}
#' \item{int}{The vertices of the interval as a \code{vector} where position of the vertex corresponds to
#' the vertex index as \code{int=(rv=1,rv=2)}.}
#'
#' @seealso \code{\link{rv.end.int}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' c<-.4
#' a<-0; b<-10; int<-c(a,b)
#'
#' Mc<-centMc(int,c)
#'
#' rv.mid.int(6,c,int)
#'
#' rv.mid.int(3,c,int)
#'
#' rv.mid.int(13,c,int)
#'
#' rv.mid.int(4,c,int)
#' rv.mid.int(0,c,int)
#' rv.mid.int(-3,c,int)
#'
#' n<-10  #try also n<-20
#' xr<-range(a,b,Mc)
#' xf<-(xr[2]-xr[1])*.5
#' dat<-runif(n,a,b)
#'
#' Rv<-vector()
#' for (i in 1:n)
#'   Rv<-c(Rv,rv.mid.int(dat[i],c,int)$rv)
#' Rv
#'
#' jit<-.1
#' yjit<-runif(n,-jit,jit)
#'
#' Xlim<-range(a,b,dat)
#' xd<-Xlim[2]-Xlim[1]
#'
#' plot(cbind(Mc,0),main="vertex region indices for the points", xlab=" ", ylab=" ",
#'      xlim=Xlim+xd*c(-.05,.05),ylim=3*range(yjit),pch=".",cex=3)
#' abline(h=0)
#' points(dat,yjit)
#' abline(v=c(a,b,Mc),lty=2,col=c(1,1,2))
#' text(dat,yjit,labels=factor(Rv))
#' text(cbind(c(a,b,Mc),.02),c("rv=1","rv=2","Mc"))
#'
#' rv.mid.int(6,c,int)
#'
#' @export rv.mid.int
rv.mid.int <- function(pt,c,int)
{
  if (!is.point(pt,1))
  {stop('pt must be a scalar')}

  if (!is.point(c,1) || c<=0 || c>=1)
  {stop('c must be a scalar in (0,1)')}

  if (!is.point(int))
  {stop('int must a numeric vector of length 2')}

  y1<-int[1]; y2<-int[2];
  if (y1>=y2)
  {stop('interval is degenerate or void, left end must be smaller than right end')}

  if (pt<y1 || pt>y2)
  {rv<-NA
  } else
  {
    Mc<-y1+c*(y2-y1)
    rv<-1;
    if (pt > Mc)
    {rv<-2}
  }
  names(int)<-c("vertex 1","vertex 2")  #vertex labeling

  list(rv=rv, #relative vertex
      int=int #vertex labeling
  )
} #end of the function
#'

#################################################################

#' @title The indicator for the presence of an arc from a point to another for Proportional Edge
#' Proximity Catch Digraphs (PE-PCDs) - middle interval case
#'
#' @description Returns \eqn{I(x_2 \in N_{PE}(x_1,r,c))} for points \eqn{x_1} and \eqn{x_2}, that is, returns 1 if \eqn{x_2} is in \eqn{N_{PE}(x_1,r,c)}, returns 0
#' otherwise, where \eqn{N_{PE}(x,r,c)} is the PE proximity region for point \eqn{x} and is constructed with expansion
#' parameter \eqn{r \ge 1} and centrality parameter \eqn{c \in (0,1)} for the interval \eqn{(a,b)}.
#'
#' PE proximity regions are defined with respect to the middle interval \code{int} and vertex regions are based
#' on the center associated with the centrality parameter \eqn{c \in (0,1)}. For the interval, \code{int}\eqn{=(a,b)}, the
#' parameterized center is \eqn{M_c=a+c(b-a)}. \code{rv} is the index of the vertex region \eqn{x_1} resides, with default=\code{NULL}.
#' If \eqn{x_1} and \eqn{x_2} are distinct and either of them are outside interval \code{int}, it returns 0,
#' but if they are identical, then it returns 1 regardless of their locations
#' (i.e., loops are allowed in the digraph).
#'
#' See also (\insertCite{ceyhan:metrika-2012,ceyhan:revstat-2016;textual}{pcds}).
#'
#' @param x1,x2 1D points; \eqn{x_1} is the point for which the proximity region, \eqn{N_{PE}(x_1,r,c)} is
#' constructed and \eqn{x_2} is the point which the function is checking whether its inside
#' \eqn{N_{PE}(x_1,r,c)} or not.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param c A positive real number in \eqn{(0,1)} parameterizing the center inside \code{int}\eqn{=(a,b)}.
#' For the interval, \code{int}\eqn{=(a,b)}, the parameterized center is \eqn{M_c=a+c(b-a)}.
#' @param int A \code{vector} of two real numbers representing an interval.
#' @param rv The index of the vertex region \eqn{x_1} resides, with default=\code{NULL}.
#'
#' @return \eqn{I(x_2 \in N_{PE}(x_1,r,c))} for points \eqn{x_1} and \eqn{x_2} that is, returns 1 if \eqn{x_2} is in \eqn{N_{PE}(x_1,r,c)},
#' returns 0 otherwise
#'
#' @seealso \code{\link{IndNPEend1D}}, \code{\link{IndNCSmid1D}}, and \code{\link{IndNCSend1D}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' c<-.4
#' r<-2
#' a<-0; b<-10; int<-c(a,b)
#'
#' IndNPEmid1D(7,5,r,c,int)
#' IndNPEmid1D(17,17,r,c,int)
#' IndNPEmid1D(1,3,r,c,int)
#'
#' IndNPEmid1D(3,5,r,c,int)
#' IndNPEmid1D(3,3,r,c,int)
#' IndNPEmid1D(4,5,r,c,int)
#' IndNPEmid1D(a,5,r,c,int)
#'
#' c<-.4
#' r<-2
#' a<-0; b<-10; int<-c(a,b)
#' IndNPEmid1D(7,5,r,c,int)
#'
#' @export IndNPEmid1D
IndNPEmid1D <- function(x1,x2,r,c,int,rv=NULL)
{
  if (!is.point(x1,1) || !is.point(x2,1) )
  {stop('x1 and x2 must be scalars')}

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if (!is.point(c,1) || c<=0 || c>=1)
  {stop('c must be a scalar in (0,1)')}

  if (!is.point(int))
  {stop('int must a numeric vector of length 2')}

  y1<-int[1]; y2<-int[2];
  if (y1>=y2)
  {stop('interval is degenerate or void, left end must be smaller than right end')}

  if (x1==x2 )
  {arc<-1; return(arc); stop}

  y1<-int[1]; y2<-int[2];
  if (x1<y1 || x1>y2 || x2<y1 || x2>y2 )
  {arc<-0; return(arc); stop}

  if (is.null(rv))
  {rv<-rv.mid.int(x1,c,int)$rv #determines the vertex region for 1D point x1
  } else
  {  if (!is.numeric(rv) || sum(rv==c(1,2))!=1)
  {stop('vertex index, rv, must be 1 or 2')}}

  arc<-0;
  if (rv==1)
  {
    if ( x2 < y1+r*(x1-y1) ) {arc <-1}
  } else {
    if ( x2 > y2-r*(y2-x1) ) {arc<-1}
  }
  arc
} #end of the function
#'

#################################################################

#' @title Number of Arcs for Proportional Edge Proximity Catch Digraphs (PE-PCDs) - middle interval case
#'
#' @description Returns the number of arcs of Proportional Edge Proximity Catch Digraphs (PE-PCDs) whose vertices are the
#' given 1D numerical data set, \code{dat}. PE proximity region \eqn{N_{PE}(x,r,c)} is defined with respect to the interval
#' \code{int}\eqn{=(a,b)} for this function.
#'
#' PE proximity region is constructed with expansion parameter \eqn{r \ge 1} and
#' centrality parameter \eqn{c \in (0,1)}.
#'
#' Vertex regions are based on the center associated with the centrality parameter \eqn{c \in (0,1)}.
#' For the interval, \code{int}\eqn{=(a,b)}, the parameterized center is \eqn{M_c=a+c(b-a)} and for the number of arcs,
#' loops are not allowed so arcs are only possible for points inside the middle interval \code{int} for this function.
#'
#' See also (\insertCite{ceyhan:metrika-2012;textual}{pcds}).
#'
#' @param dat A set or \code{vector} of 1D points which constitute the vertices of PE-PCD.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param c A positive real number in \eqn{(0,1)} parameterizing the center inside \code{int}\eqn{=(a,b)}.
#' For the interval, \code{int}\eqn{=(a,b)}, the parameterized center is \eqn{M_c=a+c(b-a)}.
#' @param int A \code{vector} of two real numbers representing an interval.
#'
#' @return Number of arcs for the PE-PCD whose vertices are the 1D data set, \code{dat},
#' with expansion parameter, \eqn{r \ge 1}, and centrality parameter, \eqn{c \in (0,1)}. PE proximity regions are defined only
#' for \code{dat} points inside the interval \code{int}, i.e., arcs are possible for such points only.
#'
#' @seealso \code{\link{NumArcsPEend1D}}, \code{\link{NumArcsCSmid1D}}, and \code{\link{NumArcsCSend1D}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' c<-.4
#' r<-2
#' a<-0; b<-10; int<-c(a,b)
#'
#' n<-10
#' dat<-runif(n,a,b)
#' NumArcsPEmid1D(dat,r,c,int)
#'
#' NumArcsPEmid1D(3,r,c,int)
#'
#' NumArcsPEmid1D(dat,r,c=.3,int)
#'
#' NumArcsPEmid1D(dat,r=1.5,c,int)
#'
#' n<-10  #try also n<-20
#' dat<-runif(n,a-5,b+5)
#' NumArcsPEmid1D(dat,r,c,int)
#'
#' dat<-runif(n,a+10,b+10)
#' NumArcsPEmid1D(dat,r,c,int)
#'
#' n<-10
#' dat<-runif(n,a,b)
#' NumArcsPEmid1D(dat,r,c,int)
#'
#' @export NumArcsPEmid1D
NumArcsPEmid1D <- function(dat,r,c,int)
{
  if (!is.point(dat,length(dat)))
  {stop('dat must be a 1D vector of numerical entries')}

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if (!is.point(c,1) || c<=0 || c>=1)
  {stop('c must be a scalar in (0,1)')}

  if (!is.point(int))
  {stop('int must a numeric vector of length 2')}

  y1<-int[1]; y2<-int[2];
  if (y1>=y2)
  {stop('interval is degenerate or void, left end must be smaller than right end')}

  dat<-dat[dat>=y1 & dat<=y2] #data points inside the interval int
  n<-length(dat)
  if (n>0)
  {
    arcs<-0
    for (i in 1:n)
    {x1<-dat[i]
    v<-rv.mid.int(x1,c,int)$rv
    if (v==1)
    {
      xR<-y1+r*(x1-y1)
      arcs<-arcs+sum(dat < xR )-1 #minus 1 is for the loop at dat.int[i]
    } else {
      xL <-y2-r*(y2-x1)
      arcs<-arcs+sum(dat > xL)-1 #minus 1 is for the loop at dat.int[i]
    }
    }
  } else
  {arcs<-0}
  arcs
} #end of the function
#'

#################################################################

#' @title The closest points to center in each vertex region in an interval
#'
#' @description
#' An object of class \code{"Extrema"}.
#' Returns the closest data points among the data set, \code{Dt}, in each \eqn{M_c}-vertex region
#' i.e., finds the closest points from right and left to \eqn{M_c} among points of the 1D data set \code{Dt} which reside in
#' in the interval \code{int}\eqn{=(a,b)}.
#'
#' \eqn{M_c} is based on the centrality parameter \eqn{c \in (0,1)},
#' so that \eqn{100c} \% of the length of interval is to the left of \eqn{M_c} and \eqn{100(1-c)} \% of the length of the interval
#' is to the right of \eqn{M_c}. That is, for the interval \eqn{(a,b)}, \eqn{M_c=a+c(b-a)}. If there are no points from \code{Dt} to
#' the left of \eqn{M_c} in the interval, then it yields \code{NA}, and likewise for the right of \eqn{M_c} in the interval.
#'
#' See also (\insertCite{ceyhan:metrika-2012;textual}{pcds}).
#'
#' @param Dt A set or \code{vector} of 1D points from which closest points to \eqn{M_c} are found
#' in the interval \code{int}.
#' @param int A \code{vector} of two real numbers representing an interval.
#' @param c A positive real number in \eqn{(0,1)} parameterizing the center inside \code{int}\eqn{=(a,b)}.
#' For the interval, \code{int}\eqn{=(a,b)}, the parameterized center is \eqn{M_c=a+c(b-a)}.
#'
#' @return A \code{list} with the elements
#' \item{txt1}{Vertex Labels are \eqn{a=1} and \eqn{b=2} for the interval \eqn{(a,b)}.}
#' \item{txt2}{A short description of the distances as \code{"Distances from ..."}}
#' \item{type}{Type of the extrema points}
#' \item{desc}{A short description of the extrema points}
#' \item{mtitle}{The \code{"main"} title for the plot of the extrema}
#' \item{ext}{The extrema points, here, closest points to \eqn{M_c} in each vertex region}
#' \item{X}{The input data vector, \code{Dt}.}
#' \item{num.points}{The number of data points, i.e., size of \code{Dt}}
#' \item{supp}{Support of the data points, here, it is \code{int}.}
#' \item{cent}{The (parameterized) center point used for construction of vertex regions.}
#' \item{ncent}{Name of the (parameterized) center, \code{cent}, it is \code{"Mc"} for this function.}
#' \item{regions}{Vertex regions inside the interval, \code{int}, provided as a list.}
#' \item{region.names}{Names of the vertex regions as \code{"vr=1"}, \code{"vr=2"}}
#' \item{region.centers}{Centers of mass of the vertex regions inside \code{int}.}
#' \item{dist2ref}{Distances from closest points in each vertex region to \eqn{M_c}.}
#'
#' @seealso \code{\link{cl2CC.TbVR}} and \code{\link{cl2CC.VR}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' c<-.4
#' a<-0; b<-10; int<-c(a,b)
#'
#' Mc<-centMc(int,c)
#'
#' nx<-10
#' xr<-range(a,b,Mc)
#' xf<-(xr[2]-xr[1])*.5
#'
#' dat<-runif(nx,a,b)
#'
#' Ext<-cl2Mc.int(dat,int,c)
#' Ext
#' summary(Ext)
#' plot(Ext)
#'
#' cl2Mc.int(dat[1],int,c)
#' cMc<-cl2Mc.int(dat,int,c)
#'
#' Xlim<-range(a,b,dat)
#' xd<-Xlim[2]-Xlim[1]
#'
#' plot(cbind(a,0),xlab="",pch=".",xlim=Xlim+xd*c(-.05,.05))
#' abline(h=0)
#' abline(v=c(a,b,Mc),col=c(1,1,2),lty=2)
#' points(cbind(dat,0))
#' points(cbind(c(cMc$ext),0),pch=4,col=2)
#' text(cbind(c(a,b,Mc),-0.1),c("a","b","Mc"))
#'
#' n<-10  #try also n<-20
#' dat<-runif(n,a-5,b+5)
#' cl2Mc.int(dat,c(a,b),c)
#'
#' dat<-runif(n,a+b,b+10)
#' cl2Mc.int(dat,int,c)
#'
#' c<-.4
#' a<-0; b<-10; int<-c(a,b)
#' n<-10
#' dat<-runif(n,a,b)
#' cl2Mc.int(dat,int,c)
#'
#' @export cl2Mc.int
cl2Mc.int <- function(Dt,int,c)
{
  if (!is.point(Dt,length(Dt)))
  {stop('Dt must be a 1D vector of numerical entries')}

  if (!is.point(int))
  {stop('int must a numeric vector of length 2')}

  if (!is.point(c,1) || c<=0 || c>=1)
  {stop('c must be a scalar in (0,1)')}

  y1<-int[1]; y2<-int[2];
  if (y1>=y2)
  {stop('interval is degenerate or void, left end must be smaller than right end')}

  Mc<-y1+c*(y2-y1)

  dat<-Dt[(Dt>=y1 & Dt<=y2)]

 ind<-dat<=Mc

  U<-rep(NA,2)
  if (sum(ind)>0)
  {U[1]<-max(dat[ind])};

  if (sum(!ind)>0)
  {U[2]<-min(dat[!ind])};

  names(int)<-c("a","b")  #vertex labeling
  typ<-"Closest Points to Mc in Mc-Vertex Regions in the Interval (a,b)"
  txt1<-"Vertex Labels are a=1 and b=2 for the interval (a,b)"
  txt2<-"Distances from closest points to Mc in each vertex region\n (Row i corresponds to vertex i for i=1,2)"
  description<-"Closest Points to Mc in the Mc-Vertex Regions \n (Row i corresponds to vertex i for i=1,2)  "
  main.txt<-"Closest Points to Mc in Mc-Vertex Regions"

  Dis<-rbind(Mc-U[1],U[2]-Mc)  #distances of the closest points to Mc
  Regs<-list(vr1=c(int[1],Mc), #regions inside the interval
             vr2=c(Mc,int[2]))

  Reg.Cent<-vector()
  for (i in 1:length(Regs))
  { Reg.Cent<-c(Reg.Cent,mean(Regs[[i]]))}

  Reg.names<-c("vr=1","vr=2")  #regions names

  res<-list(
    txt1=txt1, txt2=txt2,
    type=typ, desc=description,
    mtitle=main.txt, #main label in the plot
    ext=U, #closest points to Mc in each vertex region
    X=Dt, num.points=length(Dt), #data points and its size
    ROI=int, # region of interest for X points
    cent=Mc, ncent="Mc", #center and center name
    regions=Regs, region.names=Reg.names, region.centers=Reg.Cent,
    dist2ref=Dis #distances of closest points to Mc in each vertex region
  )

  class(res)<-"Extrema"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

# funsMuVarPE1D
#'
#' @title Returns the mean and (asymptotic) variance of arc density of Proportional Edge Proximity
#' Catch Digraph (PE-PCD) for 1D data - middle interval case
#'
#' @description
#' The functions \code{muPE1D} and \code{asyvarPE1D} and their auxiliary functions.
#'
#' \code{muPE1D} returns the mean of the (arc) density of PE-PCD
#' and \code{asyvarPE1D} returns the (asymptotic) variance of the arc density of PE-PCD
#' for a given centrality parameter \eqn{c \in (0,1)} and an expansion parameter \eqn{r \ge 1} and for 1D uniform data in a
#' finite interval \eqn{(a,b)}, i.e., data from \eqn{U(a,b)} distribution.
#'
#' \code{muPE1D} uses auxiliary (internal) function \code{mu1PE1D} which yields mean (i.e., expected value)
#' of the arc density of PE-PCD for a given \eqn{c \in (0,1/2)} and \eqn{r \ge 1}.
#'
#' \code{asyvarPE1D} uses auxiliary (internal) functions \code{fvar1} which yields asymptotic variance
#' of the arc density of PE-PCD for \eqn{c \in (1/4,1/2)} and \eqn{r \ge 1}; and \code{fvar2} which yields asymptotic variance
#' of the arc density of PE-PCD for \eqn{c \in (0,1/4)} and \eqn{r \ge 1}.
#'
#' See also (\insertCite{ceyhan:metrika-2012;textual}{pcds}).
#'
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param c A positive real number in \eqn{(0,1)} parameterizing the center inside \code{int}\eqn{=(a,b)}.
#' For the interval, \code{int}\eqn{=(a,b)}, the parameterized center is \eqn{M_c=a+c(b-a)}.
#'
#' @return \code{muPE1D} returns the mean and \code{asyvarPE1D} returns the asymptotic variance of the
#' arc density of PE-PCD for \eqn{U(a,b)} data
#'
#' @name funsMuVarPE1D
NULL
#'
#' @seealso \code{\link{muCS1D}} and \code{\link{asyvarCS1D}}
#'
#' @rdname funsMuVarPE1D
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
mu1PE1D <- function(r,c)
{
  mean<-0;
  if (r<1/(1-c))
  {
    mean<-r*c^2+1/2*r-r*c;
  } else {
    if (r<1/c)
    {
      mean<-(1/(2*r))*(c^2*r^2+2*r-1-2*c*r);
    } else {
      mean<-(r-1)/r;
    }}
  mean
} #end of the function
#'
#' @rdname funsMuVarPE1D
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' #Examples for muPE1D
#' muPE1D(1.2,.4)
#' muPE1D(1.2,.6)
#'
#' rseq<-seq(1.01,5,by=.05)
#' cseq<-seq(0.01,.99,by=.05)
#'
#' lrseq<-length(rseq)
#' lcseq<-length(cseq)
#'
#' mu.grid<-matrix(0,nrow=lrseq,ncol=lcseq)
#' for (i in 1:lrseq)
#'   for (j in 1:lcseq)
#'   {
#'     mu.grid[i,j]<-muPE1D(rseq[i],cseq[j])
#'   }
#'
#' persp(rseq,cseq,mu.grid, xlab="r", ylab="c", zlab="mu(r,c)", theta = -30, phi = 30,
#' expand = 0.5, col = "lightblue", ltheta = 120, shade = 0.05, ticktype = "detailed")
#'
#' @export muPE1D
muPE1D <- function(r,c)
{
  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if (!is.point(c,1) || c<=0 || c>=1)
  {stop('c must be a scalar in (0,1)')}

  mean<-0;
  if (c <= 1/2)
  {
    mean<-mu1PE1D(r,c);
  } else {
    mean<-mu1PE1D(r,1-c);
  }
  mean
} #end of the function
#'
#' @rdname funsMuVarPE1D
#'
fvar1 <- function(r,c)
{
  asyvar<-0;
  if (r<1/(1-c))
  {
    asyvar<--1/3*(12*c^4*r^4-24*c^3*r^4+3*c^2*r^5+15*c^2*r^4-3*c*r^5-9*c^2*r^3-3*c*r^4+r^5+6*c^2*r^2+9*c*r^3-r^4-6*c*r^2-2*r^3+4*r^2-3*r+1)/r^2;
  } else {
    if (r<1/c)
    {
      asyvar<--1/3*(3*c^4*r^4+c^3*r^5-c^3*r^4-11*c^3*r^3-3*c^2*r^4+6*c^2*r^3+9*c^2*r^2+3*c*r^3-9*c*r^2+3*c*r-r^2+2*r-1)/r^2;
    } else {
      asyvar<-1/3*(2*r-3)/r^2;
    }}
  asyvar
} #end of the function
#'
#' @rdname funsMuVarPE1D
#'
fvar2 <- function(r,c)
{
  asyvar<-0;
  if (r<1/(1-c))
  {
    asyvar<--1/3*(12*c^4*r^4-24*c^3*r^4+3*c^2*r^5+15*c^2*r^4-3*c*r^5-9*c^2*r^3-3*c*r^4+r^5+6*c^2*r^2+9*c*r^3-r^4-6*c*r^2-2*r^3+4*r^2-3*r+1)/r^2;
  } else {
    if (r<(1-sqrt(1-4*c))/(2*c))
    {
      asyvar<--1/3*(3*c^4*r^4+c^3*r^5-c^3*r^4-11*c^3*r^3-3*c^2*r^4+6*c^2*r^3+9*c^2*r^2+3*c*r^3-9*c*r^2+3*c*r-r^2+2*r-1)/r^2;
    } else {
      if (r<(1+sqrt(1-4*c))/(2*c))
      {
        asyvar<--1/3*(3*c^4*r^5-c^3*r^5-11*c^3*r^4+3*c^2*r^4+9*c^2*r^3-3*c*r^3-r^2+2*r-1)/r^3;
      } else {
        if (r<1/c)
        {
          asyvar<--1/3*(3*c^4*r^4+c^3*r^5-c^3*r^4-11*c^3*r^3-3*c^2*r^4+6*c^2*r^3+9*c^2*r^2+3*c*r^3-9*c*r^2+3*c*r-r^2+2*r-1)/r^2;
        } else {
          asyvar<-1/3*(2*r-3)/r^2;
        }}}}
  asyvar
} #end of the function
#'
#' @rdname funsMuVarPE1D
#'
#' @examples
#' #Examples for asyvarPE1D
#' asyvarPE1D(1.2,.8)
#'
#' rseq<-seq(1.01,5,by=.05)
#' cseq<-seq(0.01,.99,by=.05)
#'
#' lrseq<-length(rseq)
#' lcseq<-length(cseq)
#'
#' var.grid<-matrix(0,nrow=lrseq,ncol=lcseq)
#' for (i in 1:lrseq)
#'   for (j in 1:lcseq)
#'   {
#'     var.grid[i,j]<-asyvarPE1D(rseq[i],cseq[j])
#'   }
#'
#' persp(rseq,cseq,var.grid, xlab="r", ylab="c", zlab="var(r,c)", theta = -30, phi = 30,
#' expand = 0.5, col = "lightblue", ltheta = 120, shade = 0.05, ticktype = "detailed")
#'
#' @export asyvarPE1D
asyvarPE1D <- function(r,c)
{
  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if (!is.point(c,1) || c<=0 || c>=1)
  {stop('c must be a scalar in (0,1)')}

  asyvar<-0;
  if (c<1/4)
  {
    asyvar<-fvar2(r,c);
  } else {
    if (c<1/2)
    {
      asyvar<-fvar1(r,c);
    } else {
      if (c<3/4)
      {
        asyvar<-fvar1(r,1-c);
      } else {
        asyvar<-fvar2(r,1-c);
      }}}
  asyvar
} #end of the function
#'

#################################################################

#' @title A test of uniformity of 1D data in a given interval based on Proportional Edge Proximity Catch Digraph
#' (PE-PCD)
#'
#' @description
#' An object of class \code{"htest"}.
#' This is an \code{"htest"} (i.e., hypothesis test) function which performs a hypothesis test of uniformity of 1D data
#' in one interval based on the normal approximation of the arc density of the PE-PCD with expansion parameter \eqn{r \ge 1}
#' and centrality parameter \eqn{c \in (0,1)}.
#'
#' The function yields the test statistic, \eqn{p}-value for the
#' corresponding \code{alternative}, the confidence interval, estimate and null value for the parameter of interest
#' (which is the arc density), and method and name of the data set used.
#'
#' The null hypothesis is that data is
#' uniform in a finite interval (i.e., arc density of PE-PCD equals to its expected value under uniform
#' distribution) and \code{alternative} could be two-sided, or left-sided (i.e., data is accumulated around the end
#' points) or right-sided (i.e., data is accumulated around the mid point or center \eqn{M_c}).
#'
#' See also (\insertCite{ceyhan:metrika-2012,ceyhan:revstat-2016;textual}{pcds}).
#'
#' @param dat A set or \code{vector} of 1D points which constitute the vertices of PE-PCD.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param c A positive real number in \eqn{(0,1)} parameterizing the center inside \code{int}\eqn{=(a,b)}.
#' For the interval, \code{int}\eqn{=(a,b)}, the parameterized center is \eqn{M_c=a+c(b-a)}.
#' @param int A \code{vector} of two real numbers representing an interval.
#' @param alternative Type of the alternative hypothesis in the test, one of \code{"two.sided"}, \code{"less"}, \code{"greater"}.
#' @param conf.level Level of the confidence interval, default is \code{0.95}, for the arc density of PE-PCD based on
#' the 1D data set \code{dat}.
#'
#' @return A \code{list} with the elements
#' \item{statistic}{Test statistic}
#' \item{p.value}{The \eqn{p}-value for the hypothesis test for the corresponding \code{alternative}}
#' \item{conf.int}{Confidence interval for the arc density at the given confidence level \code{conf.level} and
#' depends on the type of \code{alternative}.}
#' \item{estimate}{Estimate of the parameter, i.e., arc density}
#' \item{null.value}{Hypothesized value for the parameter, i.e., the null arc density, which is usually the
#' mean arc density under uniform distribution.}
#' \item{alternative}{Type of the alternative hypothesis in the test, one of \code{"two.sided"}, \code{"less"}, \code{"greater"}}
#' \item{method}{Description of the hypothesis test}
#' \item{data.name}{Name of the data set}
#'
#' @seealso \code{\link{TSArcDensCS1D}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' \donttest{
#' c<-.4
#' r<-2
#' a<-0; b<-10; int<-c(a,b)
#'
#' n<-10  #try also n<-20
#' dat<-runif(n,a,b)
#'
#' NumArcsPEint(dat,r,c,int)
#' TSArcDensPE1D(dat,r,c,int)
#' TSArcDensPE1D(dat,r,c,int,alt="g")
#' TSArcDensPE1D(dat,r,c,int,alt="l")
#'
#' NumArcsPEint(dat,r,c=.3,int)
#' TSArcDensPE1D(dat,r,c=.3,int)
#'
#' NumArcsPEint(dat,r=1.5,c,int)
#' TSArcDensPE1D(dat,r=1.5,c,int)
#'
#' dat<-runif(n,a-1,b+1)
#' NumArcsPEint(dat,r,c,int)
#' TSArcDensPE1D(dat,r,c,int)
#'
#' c<-.4
#' r<-2
#' a<-0; b<-10; int<-c(a,b)
#' n<-10  #try also n<-20
#' dat<-runif(n,a,b)
#' TSArcDensPE1D(dat,r,c,int)
#' }
#'
#' @export TSArcDensPE1D
TSArcDensPE1D <- function(dat,r,c,int,alternative=c("two.sided", "less", "greater"),conf.level = 0.95)
{
  alternative <-match.arg(alternative)
  if (length(alternative) > 1 || is.na(alternative))
    stop("alternative must be one \"greater\", \"less\", \"two.sided\"")

  if (!is.point(dat,length(dat)))
  {stop('dat must be a 1D vector of numerical entries')}

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if (!is.point(c,1) || c<=0 || c>=1)
  {stop('c must be a scalar in (0,1)')}

  if (!is.point(int))
  {stop('int must a numeric vector of length 2')}

  y1<-int[1]; y2<-int[2];
  if (y1>=y2)
  {stop('interval is degenerate or void, left end must be smaller than right end')}

  if (!missing(conf.level))
    if (length(conf.level) != 1 || is.na(conf.level) || conf.level < 0 || conf.level > 1)
      stop("conf.level must be a number between 0 and 1")

  dat<-dat[dat>=y1 & dat<=y2] #data points inside the interval int
  n<-length(dat)
  if (n<=1)
  {stop('not enough points in the interval to compute arc density')}

  num.arcs<-NumArcsPEint(dat,r,c,int)
  arc.dens<-num.arcs/(n*(n-1))
  estimate1<-arc.dens
  mn<-muPE1D(r,c)
  asy.var<-asyvarPE1D(r,c)
  TS<-sqrt(n) *(arc.dens-mn)/sqrt(asy.var)
  method <-c("Large Sample z-Test Based on Arc Density for 1D data")
  names(estimate1) <-c("arc density")

  null.dens<-mn
  names(null.dens) <-"(expected) arc density"
  names(TS) <-"standardized arc density (i.e., Z)"

  if (alternative == "less") {
    pval <-pnorm(TS)
    cint <-arc.dens+c(-Inf, qnorm(conf.level))*sqrt(asy.var/n)
  }
  else if (alternative == "greater") {
    pval <-pnorm(TS, lower.tail = FALSE)
    cint <-arc.dens+c(-qnorm(conf.level),Inf)*sqrt(asy.var/n)
  }
  else {
    pval <-2 * pnorm(-abs(TS))
    alpha <-1 - conf.level
    cint <-qnorm(1 - alpha/2)
    cint <-arc.dens+c(-cint, cint)*sqrt(asy.var/n)
  }
  attr(cint, "conf.level") <-conf.level

  dname <-deparse(substitute(dat))

  rval <-list(
    statistic=TS,
    p.value=pval,
    conf.int = cint,
    estimate = estimate1,
    null.value = null.dens,
    alternative = alternative,
    method = method,
    data.name = dname
  )

  attr(rval, "class") <-"htest"
  return(rval)
} #end for the function
#'

#################################################################

#' @title The arcs of Proportional Edge Proximity Catch Digraph (PE-PCD) for 1D data - middle intervals case
#'
#' @description
#' An object of class \code{"PCDs"}.
#' Returns arcs as tails (or sources) and heads (or arrow ends) for 1D data set \code{Xp} as the vertices
#' of PE-PCD.
#'
#' For this function, PE proximity regions are constructed with respect to the intervals
#' based on \code{Yp} points with expansion parameter \eqn{r \ge 1} and centrality parameter \eqn{c \in (0,1)}. That is, for this function,
#' arcs may exist for points only inside the intervals.
#' It also provides various descriptions and quantities about the arcs of the PE-PCD
#' such as number of arcs, arc density, etc.
#'
#' Vertex regions are based on center \eqn{M_c} of each middle interval.
#'
#' See also (\insertCite{ceyhan:metrika-2012;textual}{pcds}).
#'
#' @param Xp A set or \code{vector} of 1D points which constitute the vertices of the PE-PCD.
#' @param Yp A set or \code{vector} of 1D points which constitute the end points of the intervals.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param c A positive real number in \eqn{(0,1)} parameterizing the center inside \code{int}\eqn{=(a,b)}.
#' For the interval, \code{int}\eqn{=(a,b)}, the parameterized center is \eqn{M_c=a+c(b-a)}.
#'
#' @return A \code{list} with the elements
#' \item{type}{A description of the type of the digraph}
#' \item{parameters}{Parameters of the digraph, here, they are expansion and centrality parameters.}
#' \item{tess.points}{Points on which the tessellation of the study region is performed, here, tessellation
#' is the intervalization based on \code{Yp} points.}
#' \item{tess.name}{Name of data set used in tessellation, it is \code{Yp} for this function}
#' \item{vertices}{Vertices of the digraph, i.e., \code{Xp} points}
#' \item{vert.name}{Name of the data set which constitute the vertices of the digraph}
#' \item{S}{Tails (or sources) of the arcs of PE-PCD for 1D data in the middle intervals}
#' \item{E}{Heads (or arrow ends) of the arcs of PE-PCD for 1D data in the middle intervals}
#' \item{mtitle}{Text for \code{"main"} title in the plot of the digraph}
#' \item{quant}{Various quantities for the digraph: number of vertices, number of partition points,
#' number of intervals, number of arcs, and arc density.}
#'
#' @seealso \code{\link{ArcsPEend1D}}, \code{\link{ArcsPE1D}}, \code{\link{ArcsCSmid1D}},
#'  \code{\link{ArcsCSend1D}} and \code{\link{ArcsCS1D}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' r<-2
#' c<-.4
#' a<-0; b<-10;
#'
#' #nx is number of X points (target) and ny is number of Y points (nontarget)
#' nx<-20; ny<-4;  #try also nx<-40; ny<-10 or nx<-1000; ny<-10;
#'
#' set.seed(1)
#' Xp<-runif(nx,a,b)
#' Yp<-runif(ny,a,b)
#'
#' Arcs<-ArcsPEmid1D(Xp,Yp,r,c)
#' Arcs
#' summary(Arcs)
#' plot(Arcs)
#'
#' S<-Arcs$S
#' E<-Arcs$E
#'
#' ArcsPEmid1D(Xp,Yp,r,c)
#' ArcsPEmid1D(Xp,Yp+10,r,c)
#'
#' jit<-.1
#' yjit<-runif(nx,-jit,jit)
#'
#' Xlim<-range(Xp,Yp)
#' xd<-Xlim[2]-Xlim[1]
#'
#' plot(cbind(a,0),
#' main="arcs of PE-PCD for points (jittered along y-axis)\n in middle intervals ",
#' xlab=" ", ylab=" ", xlim=Xlim+xd*c(-.05,.05),ylim=3*c(-jit,jit),pch=".")
#' abline(h=0,lty=1)
#' points(Xp, yjit,pch=".",cex=3)
#' abline(v=Yp,lty=2)
#' arrows(S, yjit, E, yjit, length = .05, col= 4)
#'
#' r<-2
#' c<-.4
#' a<-0; b<-10;
#' nx<-20; ny<-4;  #try also nx<-40; ny<-10 or nx<-1000; ny<-10;
#' Xp<-runif(nx,a,b)
#' Yp<-runif(ny,a,b)
#' ArcsPEmid1D(Xp,Yp,r,c)
#'
#' @export ArcsPEmid1D
ArcsPEmid1D <- function(Xp,Yp,r,c)
{
  if (!is.point(Xp,length(Xp)) || !is.point(Yp,length(Yp)) )
  {stop('Xp and Yp must be 1D vectors of numerical entries')}

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if (!is.point(c,1) || c<=0 || c>=1)
  {stop('c must be a scalar in (0,1)')}

  nx<-length(Xp); ny<-length(Yp)

  if (ny<=1 | nx<=1)
  {
    S<-E<-vector(); nx2<-0
  } else
  {
    Xs<-sort(Xp); Ys<-sort(Yp)  #sorted data points from classes X and Y
    ymin<-Ys[1]; ymax<-Ys[ny];

   int<-rep(0,nx)
    for (i in 1:nx)
     int[i]<-(Xs[i]>ymin & Xs[i] < ymax )  #indices of X points in the middle intervals, i.e., inside min(Yp) and max (Yp)

    Xint<-Xs[int==1] # X points inside  min(Yp) and max (Yp)
    XLe<-Xs[Xs<ymin] # X points in the left end interval of Yp points
    XRe<-Xs[Xs>ymax] # X points in the right end interval of Yp points

    nt<-ny-1 #number of Yp middle intervals
    nx2<-length(Xint)  #number of Xp points inside the middle intervals

    if (nx2==0)
    {S<-E<-NA
    } else
    {
      i.int<-rep(0,nx2)
      for (i in 1:nx2)
        for (j in 1:nt)
        {
          if (Xint[i]>=Ys[j] & Xint[i] < Ys[j+1] )
            i.int[i]<-j #indices of the Yp intervals in which X points reside
        }

      #the arcs of PE-PCDs for parameters r and c
      S<-E<-vector()  #S is for source and E is for end points for the arcs for middle intervals
      for (i in 1:nt)
      {
        Xi<-Xint[i.int==i] #X points in the ith Yp mid interval
        ni<-length(Xi)
        if (ni>1 )
        {
          y1<-Ys[i]; y2<-Ys[i+1]; int<-c(y1,y2)
          for (j in 1:ni)
          {x1 <-Xi[j] ; Xinl<-Xi[-j] #to avoid loops
          v<-rv.mid.int(x1,c,int)$rv
          if (v==1)
          {
            xR<-y1+r*(x1-y1)
           ind.tails<-((Xinl < min(xR,y2)) & (Xinl > y1))
            st<-sum(ind.tails)  #sum of tails of the arcs with head Xi[j]
            S<-c(S,rep(x1,st)); E<-c(E,Xinl[ind.tails])
          } else {
            xL <-y2-r*(y2-x1)
           ind.tails<-((Xinl < y2) & (Xinl > max(xL,y1)))
            st<-sum(ind.tails)  #sum of tails of the arcs with head Xi[j]
            S<-c(S,rep(x1,st)); E<-c(E,Xinl[ind.tails])
          }
          }
        }

      }
    }
  }
  if (length(S)==0)
  {S<-E<-NA}

  xname <-deparse(substitute(Xp))
  yname <-deparse(substitute(Yp))
  cname <-deparse(substitute(c))
  rname <-deparse(substitute(r))

  param<-c(r,c)
  names(param)<-c("expansion parameter","centrality parameter")
  typ<-paste("Proportional Edge Proximity Catch Digraph (PE-PCD) for 1D Points in the Middle Intervals with Expansion Parameter ",rname, "=",r," and Centrality Parameter ", cname,"=",c,sep="")

  main.txt<-paste("Arcs of PE-PCD for Points (jittered\n along y-axis) in Middle Intervals with ",rname, "=",round(r,2)," and ", cname,"=",round(c,2),sep="")

  nvert<-nx2; nint<-ny-1; narcs<-ifelse(sum(is.na(S))==0,length(S),0);
  arc.dens<-ifelse(nvert>1,narcs/(nvert*(nvert-1)),NA)

  quantities<-c(nvert,ny,nint,narcs,arc.dens)
  names(quantities)<-c("number of vertices", "number of partition points",
                       "number of middle intervals","number of arcs", "arc density")

  res<-list(
    type=typ,
    parameters=param,
    tess.points=Yp, tess.name=yname, #tessellation points
    vertices=Xp, vert.name=xname, #vertices of the digraph
    S=S,
    E=E,
    mtitle=main.txt,
    quant=quantities
  )

  class(res)<-"PCDs"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################
#The case of end intervals
#################################################################

#' @title The index of the vertex region in an end-interval that contains a given point
#'
#' @description Returns the index of the vertex in the interval, \code{int}, whose end interval contains the 1D point \code{pt},
#' that is, it finds the index of the vertex for the point, \code{pt}, outside
#' the interval \code{int}\eqn{=(a,b)=}(vertex 1,vertex 2);
#' vertices of interval are labeled as 1 and 2 according to their order in the interval.
#'
#' If the point, \code{pt}, is inside \code{int}, then the function yields \code{NA} as output.
#' The corresponding vertex region is an interval as \eqn{(-\infty,a)} or \eqn{(b,\infty)} for the interval \eqn{(a,b)}.
#' Then if \eqn{pt<a}, then \code{rv=1} and if \eqn{pt>b}, then \code{rv=2}. Unlike \code{\link{rv.mid.int}}, centrality parameter (i.e., center
#' of the interval is not relevant for \code{rv.end.int}.)
#'
#' See also (\insertCite{ceyhan:metrika-2012,ceyhan:revstat-2016;textual}{pcds}).
#'
#' @param pt A 1D point whose end interval region is provided by the function.
#' @param int A \code{vector} of two real numbers representing an interval.
#'
#' @return A \code{list} with two elements
#' \item{rv}{Index of the end vertex whose region contains point, \code{pt}.}
#' \item{int}{The vertices of the interval as a \code{vector} where position of the vertex corresponds to
#' the vertex index as \code{int=(rv=1,rv=2)}.}
#'
#' @seealso \code{\link{rv.mid.int}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' a<-0; b<-10; int<-c(a,b)
#'
#' rv.end.int(-6,int)
#' rv.end.int(16,int)
#'
#' n<-5
#' xr<-range(a,b)
#' xf<-(xr[2]-xr[1])*.5
#' datL<-runif(n,a-xf,a)
#' datR<-runif(n,b,b+xf)
#' dat<-c(datL,datR)
#' rv.end.int(dat[1],int)
#'
#' Rv<-vector()
#' for (i in 1:length(dat))
#'   Rv<-c(Rv,rv.end.int(dat[i],int)$rv)
#' Rv
#'
#' Xlim<-range(a,b,dat)
#' xd<-Xlim[2]-Xlim[1]
#'
#' plot(cbind(a,0),xlab="",pch=".",xlim=Xlim+xd*c(-.05,.05))
#' abline(h=0)
#' abline(v=c(a,b),col=1,lty=2)
#' points(cbind(dat,0))
#' text(cbind(dat,0.1),labels=factor(Rv))
#' text(cbind(c(a,b),-0.1),c("rv=1","rv=2"))
#'
#' jit<-.1
#' yjit<-runif(length(dat),-jit,jit)
#'
#' Xlim<-range(a,b,dat)
#' xd<-Xlim[2]-Xlim[1]
#'
#' plot(cbind(a,0),main="vertex region indices for the points\n in the end intervals",
#'      xlab=" ", ylab=" ",pch=".",xlim=Xlim+xd*c(-.05,.05),ylim=3*range(yjit))
#' points(dat, yjit,xlim=Xlim+xd*c(-.05,.05),pch=".",cex=3)
#' abline(h=0)
#' abline(v=c(a,b),lty=2)
#' text(dat,yjit,labels=factor(Rv))
#' text(cbind(c(a,b),-.01),c("rv=1","rv=2"))
#'
#' @export rv.end.int
rv.end.int <- function(pt,int)
{
  if (!is.point(pt,1))
  {stop('pt must be a scalar')}

  if (!is.point(int))
  {stop('int must a numeric vector of length 2')}

  y1<-int[1]; y2<-int[2];
  if (y1>=y2)
  {stop('interval is degenerate or void, left end must be smaller than right end')}

  if (pt>y1 & pt<y2)
  {stop('point must be outside the interval')}

  rv<-1;
  if (pt >y2)
  {rv<-2}

  names(int)<-c("vertex 1","vertex 2")  #vertex labeling

  list(rv=rv, #relative vertex
      int=int #vertex labeling
  )
} #end of the function
#'

#################################################################

#' @title The indicator for the presence of an arc from a point to another for
#' Proportional Edge Proximity Catch Digraphs (PE-PCDs) - end interval case
#'
#' @description Returns \eqn{I(x_2 \in N_{PE}(x_1,r))} for points \eqn{x_1} and \eqn{x_2}, that is, returns 1 if \eqn{x_2} is in \eqn{N_{PE}(x_1,r)}, returns 0
#' otherwise, where \eqn{N_{PE}(x,r)} is the PE proximity region for point \eqn{x} with expansion parameter \eqn{r \ge 1}
#' for the region outside the interval \eqn{(a,b)}.
#'
#' \code{rv} is the index of the end vertex region \eqn{x_1} resides, with default=\code{NULL},
#' and \code{rv=1} for left end interval and \code{rv=2} for the right end interval.
#' If \eqn{x_1} and \eqn{x_2} are distinct and either of them are inside interval \code{int}, it returns 0,
#' but if they are identical, then it returns 1 regardless of their locations (i.e., it allows loops).
#'
#' See also (\insertCite{ceyhan:metrika-2012;textual}{pcds}).
#'
#' @param x1 A 1D point whose PE proximity region is constructed.
#' @param x2 A 1D point. The function determines whether \eqn{x_2} is inside the PE proximity region of
#' \eqn{x_1} or not.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param int A \code{vector} of two real numbers representing an interval.
#' @param rv Index of the end interval containing the point, either \code{1,2} or \code{NULL} (default is \code{NULL}).
#'
#' @return \eqn{I(x_2 \in N_{PE}(x_1,r))} for points \eqn{x_1} and \eqn{x_2}, that is, returns 1 if \eqn{x_2} is in \eqn{N_{PE}(x_1,r)}
#' (i.e., if there is an arc from \eqn{x_1} to \eqn{x_2}), returns 0 otherwise
#'
#' @seealso \code{\link{IndNPEmid1D}}, \code{\link{IndNCSmid1D}}, and \code{\link{IndNCSend1D}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' a<-0; b<-10; int<-c(a,b)
#' r<-2
#'
#' IndNPEend1D(15,17,r,int)
#' IndNPEend1D(15,15,r,int)
#'
#' IndNPEend1D(1.5,17,r,int)
#'
#' IndNPEend1D(-15,17,r,int)
#'
#' IndNPEend1D(a,17,r,int)
#' IndNPEend1D(15,17,r=1.1,int)
#'
#' a<-0; b<-10; int<-c(a,b)
#' r<-2
#' IndNPEend1D(15,17,r,int)
#'
#' @export IndNPEend1D
IndNPEend1D <- function(x1,x2,r,int,rv=NULL)
{
  if (!is.point(x1,1) || !is.point(x2,1) )
  {stop('x1 and x2 must be scalars')}

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if (!is.point(int))
  {stop('int must a numeric vector of length 2')}

  y1<-int[1]; y2<-int[2];
  if (y1>=y2)
  {stop('interval is degenerate or void, left end must be smaller than right end')}

  if (x1==x2 )
  {arc<-1; return(arc); stop}

  if ((x1>y1 & x1<y2) || (x2>y1 & x2<y2))
  {arc<-0; return(arc); stop}

  if (is.null(rv))
  {rv<-rv.end.int(x1,int)$rv #determines the vertex for the end interval for 1D point x1
  } else
  {  if (!is.numeric(rv) || sum(rv==c(1,2))!=1)
  {stop('vertex index, rv, must be 1 or 2')}}

  arc<-0;
  if (rv==1)
  {
    if ( x2 > y1-r*(y1-x1) & x2< y1 ) {arc <-1}
  } else
  {
    if ( x2 < y2+r*(x1-y2) & x2>y2 ) {arc<-1}
  }
  arc
} #end of the function
#'

#################################################################

#' @title Number of arcs of Proportional Edge Proximity Catch Digraphs (PE-PCDs) - end interval case
#'
#' @description Returns the number of arcs of Proportional Edge Proximity Catch Digraphs (PE-PCDs) whose
#' vertices are a 1D numerical data set, \code{dat}, outside the interval \code{int}\eqn{=(a,b)}.
#'
#' PE proximity region is constructed only with expansion parameter \eqn{r \ge 1} for points outside the interval \eqn{(a,b)}.
#' End vertex regions are based on the end points of the interval,
#' i.e., the corresponding vertex region is an interval as \eqn{(-\infty,a)} or \eqn{(b,\infty)} for the interval \eqn{(a,b)}.
#' For the number of arcs, loops are not allowed, so arcs are only possible for points outside
#' the interval, \code{int}, for this function.
#'
#' See also (\insertCite{ceyhan:metrika-2012;textual}{pcds}).
#'
#' @param dat A \code{vector} of 1D points which constitute the vertices of the digraph.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param int A \code{vector} of two real numbers representing an interval.
#'
#' @return Number of arcs for the PE-PCD with vertices being 1D data set, \code{dat},
#' expansion parameter, \eqn{r \ge 1}, for the end intervals.
#'
#' @seealso \code{\link{NumArcsPEmid1D}}, \code{\link{NumArcsCSmid1D}}, and \code{\link{NumArcsCSend1D}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' a<-0; b<-10; int<-c(a,b)
#'
#' n<-5
#' datL<-runif(n,a-5,a)
#' datR<-runif(n,b,b+5)
#' dat<-c(datL,datR)
#'
#' r<-2
#' NumArcsPEend1D(dat,r,int)
#'
#' NumArcsPEend1D(dat,r=1.2,int)
#' NumArcsPEend1D(dat,r=4,int)
#'
#' n<-10  #try also n<-20
#' dat2<-runif(n,a-5,b+5)
#' NumArcsPEend1D(dat2,r,int)
#'
#' NumArcsPEend1D(dat,r,int)
#'
#' @export NumArcsPEend1D
NumArcsPEend1D <- function(dat,r,int)
{
  if (!is.point(dat,length(dat)))
  {stop('dat must be a 1D vector of numerical entries')}

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if (!is.point(int))
  {stop('int must a numeric vector of length 2')}

  y1<-int[1]; y2<-int[2];
  if (y1>=y2)
  {stop('interval is degenerate or void, left end must be smaller than right end')}

  dat<-dat[dat<y1 | dat>y2]
  n<-length(dat)
  if (n<=1)
  {arcs<-0
  } else
  {
    arcs<-0
    for (i in 1:n)
    {pt1<-dat[i]; rv<-rv.end.int(pt1,int)$rv
    for (j in ((1:n)[-i]) )
    {pt2<-dat[j]
    arcs<-arcs+IndNPEend1D(pt1,pt2,r,int,rv)
    }
    }
  }
  arcs
} #end of the function
#'

#################################################################

# funsMuVarPEend1D
#'
#' @title Returns the mean and (asymptotic) variance of arc density of Proportional Edge Proximity Catch Digraph
#' (PE-PCD) for 1D data - end interval case
#'
#' @description
#' Two functions: \code{muPEend1D} and \code{asyvarPEend1D}.
#'
#' \code{muPEend1D} returns the mean of the arc density of PE-PCD
#' and \code{asyvarPEend1D} returns the asymptotic variance of the arc density of PE-PCD
#' for a given expansion parameter \eqn{r \ge 1} for 1D uniform data in the left and right end intervals
#' for the interval \eqn{(a,b)}.
#'
#' See also (\insertCite{ceyhan:metrika-2012;textual}{pcds}).
#'
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#'
#' @return \code{muPEend1D} returns the mean and \code{asyvarPEend1D} returns the asymptotic variance of the
#' arc density of PE-PCD for uniform data in end intervals
#'
#' @name funsMuVarPEend1D
NULL
#'
#' @seealso \code{\link{muCSend1D}} and \code{\link{asyvarCSend1D}}
#'
#' @rdname funsMuVarPEend1D
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' #Examples for muPEend1D
#' muPEend1D(1.2)
#'
#' rseq<-seq(1.01,5,by=.05)
#' lrseq<-length(rseq)
#'
#' mu.end<-vector()
#' for (i in 1:lrseq)
#' {
#'   mu.end<-c(mu.end,muPEend1D(rseq[i]))
#' }
#'
#' plot(rseq, mu.end,type="l",
#' ylab=expression(paste(mu,"(r)")),xlab="r",lty=1,xlim=range(rseq),ylim=c(0,1))
#'
#' @export muPEend1D
muPEend1D <- function(r)
{
  if (!is.point(r,1) || r<1)
  {stop('the argument must be a scalar greater than 1')}

  1-1/(2*r);
} #end of the function
#'
#' @rdname funsMuVarPEend1D
#'
#' @examples
#' #Examples for asyvarPEend1D
#' asyvarPEend1D(1.2)
#'
#' rseq<-seq(1.01,5,by=.05)
#' lrseq<-length(rseq)
#'
#' var.end<-vector()
#' for (i in 1:lrseq)
#' {
#'   var.end<-c(var.end,asyvarPEend1D(rseq[i]))
#' }
#'
#' oldpar <- par(no.readonly = TRUE)
#' par(mar=c(5,5,4,2))
#' plot(rseq, var.end,type="l",
#' xlab="r",ylab=expression(paste(sigma^2,"(r)")),lty=1,xlim=range(rseq))
#' par(oldpar)
#'
#' @export asyvarPEend1D
asyvarPEend1D <- function(r)
{
  if (!is.point(r,1) || r<1)
  {stop('the argument must be a scalar greater than 1')}

  (r-1)^2/(3*r^3);
} #end of the function
#'

#################################################################

#' @title The arcs of Proportional Edge Proximity Catch Digraph (PE-PCD) for 1D data - end interval case
#'
#' @description
#' An object of class \code{"PCDs"}.
#' Returns arcs as tails (or sources) and heads (or arrow ends) for 1D data set \code{Xp} as the vertices
#' of PE-PCD.  \code{Yp} determines the end points of the end intervals.
#'
#' For this function, PE proximity regions are constructed data points outside the intervals based on
#' \code{Yp} points with expansion parameter \eqn{r \ge 1}. That is, for this function,
#' arcs may exist for points only inside end intervals.
#' It also provides various descriptions and quantities about the arcs of the PE-PCD
#' such as number of arcs, arc density, etc.
#'
#' See also (\insertCite{ceyhan:metrika-2012;textual}{pcds}).
#'
#' @param Xp A set or \code{vector} of 1D points which constitute the vertices of the PE-PCD.
#' @param Yp A set or \code{vector} of 1D points which constitute the end points of the intervals.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#'
#' @return A \code{list} with the elements
#' \item{type}{A description of the type of the digraph}
#' \item{parameters}{Parameters of the digraph, here, it is the expansion parameter.}
#' \item{tess.points}{Points on which the tessellation of the study region is performed, here, tessellation
#' is the intervalization based on \code{Yp}.}
#' \item{tess.name}{Name of data set used in tessellation, it is \code{Yp} for this function}
#' \item{vertices}{Vertices of the digraph, \code{Xp} points}
#' \item{vert.name}{Name of the data set which constitutes the vertices of the digraph}
#' \item{S}{Tails (or sources) of the arcs of PE-PCD for 1D data in the end intervals}
#' \item{E}{Heads (or arrow ends) of the arcs of PE-PCD for 1D data in the end intervals}
#' \item{mtitle}{Text for \code{"main"} title in the plot of the digraph}
#' \item{quant}{Various quantities for the digraph: number of vertices, number of partition points,
#' number of intervals (which is 2 for end intervals), number of arcs, and arc density.}
#'
#' @seealso \code{\link{ArcsPEmid1D}}, \code{\link{ArcsPE1D}} , \code{\link{ArcsCSmid1D}},
#' \code{\link{ArcsCSend1D}}  and \code{\link{ArcsCS1D}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' r<-2
#' a<-0; b<-10;
#'
#' #nx is number of X points (target) and ny is number of Y points (nontarget)
#' nx<-20; ny<-4;  #try also nx<-40; ny<-10 or nx<-1000; ny<-10;
#'
#' set.seed(1)
#' xr<-range(a,b)
#' xf<-(xr[2]-xr[1])*.5
#'
#' Xp<-runif(nx,a-xf,b+xf)
#' Yp<-runif(ny,a,b)  #try also Yp<-runif(ny,a,b)+c(-10,10)
#'
#' Arcs<-ArcsPEend1D(Xp,Yp,r)
#' Arcs
#' summary(Arcs)
#' plot(Arcs)
#'
#' S<-Arcs$S
#' E<-Arcs$E
#'
#' jit<-.1
#' yjit<-runif(nx,-jit,jit)
#'
#' Xlim<-range(a,b,Xp,Yp)
#' xd<-Xlim[2]-Xlim[1]
#'
#' plot(cbind(a,0),pch=".",
#' main="arcs of PE-PCDs for points (jittered along y-axis)\n in end intervals ",
#' xlab=" ", ylab=" ", xlim=Xlim+xd*c(-.05,.05),ylim=3*c(-jit,jit))
#' abline(h=0,lty=1)
#' points(Xp, yjit,pch=".",cex=3)
#' abline(v=Yp,lty=2)
#' arrows(S, yjit, E, yjit, length = .05, col= 4)
#'
#' ArcsPEend1D(Xp,Yp,r)
#'
#' @export ArcsPEend1D
ArcsPEend1D <- function(Xp,Yp,r)
{
  if (!is.point(Xp,length(Xp)) || !is.point(Yp,length(Yp)) )
  {stop('Xp and Yp must be 1D vectors of numerical entries')}

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  Xs<-sort(Xp); Ys<-sort(Yp)  #sorted data points
  ymin<-Ys[1]; ymax<-max(Yp)

  XLe<-Xs[Xs<ymin]; XRe<-Xs[Xs>ymax] #X points in the left and right end intervals respectively

  #the arcs of PE-PCDs for parameters r and c
  S<-E<-vector()  #S is for source and E is for end points for the arcs

  #for end intervals
  #left end interval
  nle<-length(XLe)
  if (nle>1 )
  {
    for (j in 1:nle)
    {x1 <-XLe[j];  xLe<-XLe[-j] #to avoid loops
    xL<-ymin-r*(ymin-x1)
   ind.tails<-((xLe < ymin) & (xLe > xL))
    st<-sum(ind.tails)  #sum of tails of the arcs with head XLe[j]
    S<-c(S,rep(x1,st)); E<-c(E,xLe[ind.tails])
    }
  }

  #right end interval
  nre<-length(XRe)
  if (nre>1 )
  {

    for (j in 1:nre)
    {x1 <-XRe[j]; xRe<-XRe[-j]
    xR<-ymax+r*(x1-ymax)
   ind.tails<-((xRe < xR) & xRe > ymax )
    st<-sum(ind.tails)  #sum of tails of the arcs with head XRe[j]
    S<-c(S,rep(x1,st)); E<-c(E,xRe[ind.tails])
    }
  }

  if (length(S)==0)
  {S<-E<-NA}

  xname <-deparse(substitute(Xp))
  yname <-deparse(substitute(Yp))
  rname <-deparse(substitute(r))

  param<-r
  names(param)<-"expansion parameter"
  typ<-paste("Proportional Edge Proximity Catch Digraph (PE-PCD) for 1D Points in the End Intervals with Expansion Parameter ",rname, "=",r,sep="")

  main.txt<-paste("Arcs of PE-PCD for Points (jittered\n along y-axis) in End Intervals with ",rname, "=",round(r,2),sep="")

  nvert<-nle+nre; ny<-length(Yp); nint<-2; narcs<-ifelse(sum(is.na(S))==0,length(S),0);
  arc.dens<-ifelse(nvert>1,narcs/(nvert*(nvert-1)),NA)

  quantities<-c(nvert,ny,nint,narcs,arc.dens)
  names(quantities)<-c("number of vertices", "number of partition points",
                       "number of end intervals","number of arcs", "arc density")
  res<-list(
    type=typ,
    parameters=param,
    tess.points=Yp, tess.name=yname, #tessellation points
    vertices=Xp, vert.name=xname, #vertices of the digraph
    S=S,
    E=E,
    mtitle=main.txt,
    quant=quantities
  )

  class(res)<-"PCDs"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

#' @title The arcs of Proportional Edge Proximity Catch Digraph (PE-PCD) for 1D data - multiple interval case
#'
#' @description
#' An object of class \code{"PCDs"}.
#' Returns arcs as tails (or sources) and heads (or arrow ends) for 1D data set \code{Xp} as the vertices
#' of PE-PCD.  \code{Yp} determines the end points of the intervals.
#'
#' For this function, PE proximity regions are constructed data points inside or outside the intervals based
#' on \code{Yp} points with expansion parameter \eqn{r \ge 1} and centrality parameter \eqn{c \in (0,1)}. That is, for this function,
#' arcs may exist for points in the middle or end intervals.
#' It also provides various descriptions and quantities about the arcs of the PE-PCD
#' such as number of arcs, arc density, etc.
#'
#' Equivalent to function \code{\link{ArcsPEMI}}.
#'
#' See also (\insertCite{ceyhan:metrika-2012;textual}{pcds}).
#'
#' @param Xp A set or \code{vector} of 1D points which constitute the vertices of the PE-PCD.
#' @param Yp A set or \code{vector} of 1D points which constitute the end points of the intervals.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param c A positive real number in \eqn{(0,1)} parameterizing the center inside \code{int}\eqn{=(a,b)}.
#' For the interval, \code{int}\eqn{=(a,b)}, the parameterized center is \eqn{M_c=a+c(b-a)}.
#'
#' @return A \code{list} with the elements
#' \item{type}{A description of the type of the digraph}
#' \item{parameters}{Parameters of the digraph, here, they are expansion and centrality parameters.}
#' \item{tess.points}{Points on which the tessellation of the study region is performed, here, tessellation
#' is the intervalization of the real line based on \code{Yp} points.}
#' \item{tess.name}{Name of data set used in tessellation, it is \code{Yp} for this function}
#' \item{vertices}{Vertices of the digraph, \code{Xp} points}
#' \item{vert.name}{Name of the data set which constitute the vertices of the digraph}
#' \item{S}{Tails (or sources) of the arcs of PE-PCD for 1D data}
#' \item{E}{Heads (or arrow ends) of the arcs of PE-PCD for 1D data}
#' \item{mtitle}{Text for \code{"main"} title in the plot of the digraph}
#' \item{quant}{Various quantities for the digraph: number of vertices, number of partition points,
#' number of intervals, number of arcs, and arc density.}
#'
#' @seealso \code{\link{ArcsPEmid1D}}, \code{\link{ArcsPEend1D}}, \code{\link{ArcsPEMI}},
#' \code{\link{ArcsCSmid1D}}, \code{\link{ArcsCSend1D}} and \code{\link{ArcsCS1D}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' r<-2
#' c<-.4
#' a<-0; b<-10; int<-c(a,b)
#'
#' #nx is number of X points (target) and ny is number of Y points (nontarget)
#' nx<-20; ny<-4;  #try also nx<-40; ny<-10 or nx<-1000; ny<-10;
#'
#' set.seed(1)
#' xr<-range(a,b)
#' xf<-(xr[2]-xr[1])*.1
#'
#' Xp<-runif(nx,a-xf,b+xf)
#' Yp<-runif(ny,a,b)
#'
#' Arcs<-ArcsPE1D(Xp,Yp,r,c)
#' Arcs
#' summary(Arcs)
#' plot(Arcs)
#'
#' S<-Arcs$S
#' E<-Arcs$E
#'
#' ArcsPE1D(Xp,Yp,r,c)
#'
#' jit<-.1
#' yjit<-runif(nx,-jit,jit)
#'
#' Xlim<-range(a,b,Xp,Yp)
#' xd<-Xlim[2]-Xlim[1]
#'
#' plot(cbind(a,0),pch=".",main="arcs for points in mid and end intervals ", xlab=" ", ylab=" ",
#'      xlim=Xlim+xd*c(-.05,.05),ylim=3*c(-jit,jit))
#' abline(h=0,lty=1)
#' points(Xp, yjit,pch=".",cex=3)
#' abline(v=Yp,lty=2)
#' arrows(S, yjit, E, yjit, length = .05, col= 4)
#'
#' @export ArcsPE1D
ArcsPE1D <- function(Xp,Yp,r,c)
{
  arcs.mid<-ArcsPEmid1D(Xp,Yp,r,c)
  arcs.end<-ArcsPEend1D(Xp,Yp,r)
  S<-c(arcs.mid$S, arcs.end$S)
  E<-c(arcs.mid$E, arcs.end$E)

  xname <-deparse(substitute(Xp))
  yname <-deparse(substitute(Yp))
  cname <-deparse(substitute(c))
  rname <-deparse(substitute(r))

  param<-c(r,c)
  names(param)<-c("expansion parameter","centrality parameter")
  typ<-paste("Proportional Edge Proximity Catch Digraph (PE-PCD) for 1D Points with Expansion Parameter ",rname, "=",r, " and Centrality Parameter ", cname,"=",c,sep="")

  main.txt<-paste("Arcs of PE-PCD for 1D Points (jittered\n along the y-axis) with ",rname, "=",round(r,2)," and ", cname,"=",round(c,2),sep="")

  nvert<-length(Xp); ny<-length(Yp); nint<-ny+1; narcs<-ifelse(sum(is.na(S))==0,length(S),0);
  arc.dens<-ifelse(nvert>1,narcs/(nvert*(nvert-1)),NA)

  quantities<-c(nvert,ny,nint,narcs,arc.dens)
  names(quantities)<-c("number of vertices", "number of partition points",
                       "number of intervals","number of arcs", "arc density")

  res<-list(
    type=typ,
    parameters=param,
    tess.points=Yp, tess.name=yname, #tessellation points
    vertices=Xp, vert.name=xname, #vertices of the digraph
    S=S,
    E=E,
    mtitle=main.txt,
    quant=quantities
  )

  class(res)<-"PCDs"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

#' @title The plot of the arcs of Proportional Edge Proximity Catch Digraphs (PE-PCDs) for 1D data
#' (vertices jittered along \eqn{y}-coordinate) - multiple interval case
#'
#' @description Plots the arcs of PE-PCD whose vertices are the 1D points, \code{Xp}. PE proximity regions are constructed with
#' expansion parameter \eqn{r \ge 1} and centrality parameter \eqn{c \in (0,1)} and the intervals are based on \code{Yp} points (i.e.
#' the intervalization is based on \code{Yp} points). That is, data set \code{Xp}
#' constitutes the vertices of the digraph and \code{Yp} determines the end points of the intervals.
#'
#' For better visualization, a uniform jitter from \eqn{U(-Jit,Jit)} (default for \eqn{Jit=.1}) is added to
#' the \eqn{y}-direction where \code{Jit} equals to the range of \code{Xp} and \code{Yp} multiplied by \code{Jit} with default for \eqn{Jit=.1}).
#' \code{centers} is a logical argument, if \code{TRUE}, plot includes the centers of the intervals
#' as vertical lines in the plot, else centers of the intervals are not plotted.
#'
#' See also (\insertCite{ceyhan:metrika-2012;textual}{pcds}).
#'
#' @param Xp A \code{vector} of 1D points constituting the vertices of the PE-PCD.
#' @param Yp A \code{vector} of 1D points constituting the end points of the intervals.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param c A positive real number in \eqn{(0,1)} parameterizing the center inside \code{int}\eqn{=(a,b)}.
#' For the interval, \code{int}\eqn{=(a,b)}, the parameterized center is \eqn{M_c=a+c(b-a)}.
#' @param Jit A positive real number that determines the amount of jitter along the \eqn{y}-axis, default=\code{0.1} and
#' \code{Xp} points are jittered according to \eqn{U(-Jit,Jit)} distribution along the \eqn{y}-axis where \code{Jit} equals to the range of \code{Xp} and \code{Yp} multiplied by
#' \code{Jit}).
#' @param main Title of the main heading of the plot.
#' @param xlab,ylab Titles of the \eqn{x} and \eqn{y} axes in the plot (default=\code{""} for both).
#' @param xlim,ylim Two \code{numeric} vectors of length 2, giving the \eqn{x}- and \eqn{y}-coordinate ranges
#' (default=\code{NULL} for both).
#' @param centers A logical argument, if \code{TRUE}, plot includes the centers of the intervals
#' as vertical lines in the plot, else centers of the intervals are not plotted.
#' @param \dots Additional \code{plot} parameters.
#'
#' @return A plot of the arcs of PE-PCD whose vertices are the 1D data set \code{Xp} in which vertices are jittered
#' along \eqn{y}-axis for better visualization.
#'
#' @seealso \code{\link{plotCSarcs1D}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' r<-2
#' c<-.4
#' a<-0; b<-10; int<-c(a,b)
#'
#' #nx is number of X points (target) and ny is number of Y points (nontarget)
#' nx<-20; ny<-4;  #try also nx<-40; ny<-10 or nx<-1000; ny<-10;
#'
#' set.seed(1)
#' xr<-range(a,b)
#' xf<-(xr[2]-xr[1])*.1
#'
#' Xp<-runif(nx,a-xf,b+xf)
#' Yp<-runif(ny,a,b)
#'
#' Xlim=range(Xp,Yp)
#' Ylim=c(-.2,.2)
#'
#' jit<-.1
#'
#' plotPEarcs1D(Xp,Yp,r,c,jit,xlab="",ylab="",xlim=Xlim,ylim=Ylim)
#'
#' set.seed(1)
#' plotPEarcs1D(Xp,Yp,r=1.5,c=.3,jit,main="r=1.5, c=.3",xlab="",ylab="",centers=TRUE)
#' set.seed(1)
#' plotPEarcs1D(Xp,Yp,r=2,c=.3,jit,main="r=2, c=.3",xlab="",ylab="",centers=TRUE)
#' set.seed(1)
#' plotPEarcs1D(Xp,Yp,r=1.5,c=.5,jit,main="r=1.5, c=.5",xlab="",ylab="",centers=TRUE)
#' set.seed(1)
#' plotPEarcs1D(Xp,Yp,r=2,c=.5,jit,main="r=2, c=.5",xlab="",ylab="",centers=TRUE)
#'
#' @export plotPEarcs1D
plotPEarcs1D <- function(Xp,Yp,r,c,Jit=.1,main="",xlab="",ylab="",xlim=NULL,ylim=NULL,centers=FALSE, ...)
{
  arcs<-ArcsPE1D(Xp,Yp,r,c)
  S<-arcs$S
  E<-arcs$E

  if (is.null(xlim))
  {xlim<-range(Xp,Yp)}

  jit<-(xlim[2]-xlim[1])*Jit
  ns<-length(S)
  yjit<-runif(ns,-jit,jit)

  if (is.null(ylim))
  {ylim<-3*c(-jit,jit)}

  nx<-length(Xp)
  plot(Xp, rep(0,nx),main=main, xlab=xlab, ylab=ylab,xlim=xlim,ylim=ylim,pch=".",cex=3, ...)
  if (centers==TRUE)
  {cents<-centersMc(Yp,c)
  abline(v=cents,lty=3)}
  abline(v=Yp,lty=1)
  abline(h=0,lty=2)
  if (!is.null(S)) {arrows(S, yjit, E, yjit, length = .05, col= 4)}
} #end of the function
#'

#################################################################
#NPE Functions that work for both middle and end intervals
#################################################################

#' @title The end points of the Proportional Edge (PE) Proximity Region for a point - one interval case
#'
#' @description Returns the end points of the interval which constitutes the PE proximity region for a point in the
#' interval \code{int}\eqn{=(a,b)=}\code{(rv=1,rv=2)}. PE proximity region is constructed with respect to the interval \code{int}
#' with expansion parameter \eqn{r \ge 1} and centrality parameter \eqn{c \in (0,1)}.
#'
#' Vertex regions are based on the (parameterized) center, \eqn{M_c},
#' which is \eqn{M_c=a+c(b-a)} for the interval, \code{int}\eqn{=(a,b)}.
#' The PE proximity region is constructed whether \code{x} is inside or outside the interval \code{int}.
#'
#' See also (\insertCite{ceyhan:metrika-2012;textual}{pcds}).
#'
#' @param x A 1D point for which PE proximity region is constructed.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param c A positive real number in \eqn{(0,1)} parameterizing the center inside \code{int}\eqn{=(a,b)}.
#' For the interval, \code{int}\eqn{=(a,b)}, the parameterized center is \eqn{M_c=a+c(b-a)}, and default=\code{0.5}
#' @param int A \code{vector} of two real numbers representing an interval.
#'
#' @return The interval which constitutes the PE proximity region for the point \code{x}
#'
#' @seealso \code{\link{NCSint}}, \code{\link{NPEtri}} and \code{\link{NPEtetra}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' c<-.4
#' r<-2
#' a<-0; b<-10; int<-c(a,b)
#'
#' NPEint(7,r,c,int)
#' NPEint(17,r,c,int)
#' NPEint(1,r,c,int)
#' NPEint(-1,r,c,int)
#'
#' NPEint(3,r,c,int)
#' NPEint(4,r,c,int)
#' NPEint(a,r,c,int)
#'
#' @export NPEint
NPEint <- function(x,r,c=.5,int)
{
  if (!is.point(x,1) )
  {stop('x must be a scalar')}

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if (!is.point(c,1) || c<=0 || c>=1)
  {stop('c must be a scalar in (0,1)')}

  if (!is.point(int))
  {stop('int must a numeric vector of length 2')}

  y1<-int[1]; y2<-int[2];
  if (y1>y2)
  {stop('interval is degenerate or void, left end must be smaller than or equal to right end')}

  if (x<y1 || x>y2)
  {
    ifelse(x<y1,reg<-c(y1-r*(y1-x), y1),reg<-c(y2,y2+r*(x-y2)))
  } else
  {
    Mc<-y1+c*(y2-y1)
    if (x<=Mc)
    {
      reg <-c(y1,min(y1+r*(x-y1),y2) )
    } else
    {
      reg<-c(max(y1,y2-r*(y2-x)),y2 )
    }
  }
  reg #proximity region interval
} #end of the function
#'

#################################################################

#' @title The indicator for the presence of an arc from a point to another for
#' Proportional Edge Proximity Catch Digraphs (PE-PCDs) - one interval case
#'
#' @description Returns \eqn{I(x_2 \in N_{PE}(x_1,r,c))} for points \eqn{x_1} and \eqn{x_2}, that is, returns 1 if \eqn{x_2} is in \eqn{N_{PE}(x_1,r,c)},
#' returns 0 otherwise, where \eqn{N_{PE}(x,r,c)} is the PE proximity region for point \eqn{x} with expansion parameter \eqn{r \ge 1}
#' and centrality parameter \eqn{c \in (0,1)}.
#'
#' PE proximity region is constructed with respect to the
#' interval \eqn{(a,b)}. This function works whether \eqn{x_1} and \eqn{x_2} are inside or outside the interval \code{int}.
#'
#' Vertex regions for middle intervals are based on the center associated with the centrality parameter \eqn{c \in (0,1)}.
#' If \eqn{x_1} and \eqn{x_2} are identical, then it returns 1 regardless of their locations
#' (i.e., loops are allowed in the digraph).
#'
#' See also (\insertCite{ceyhan:metrika-2012;textual}{pcds}).
#'
#' @param x1 A 1D point for which the proximity region is constructed.
#' @param x2 A 1D point for which it is checked whether it resides in the proximity region
#' of \eqn{x_1} or not.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region
#'  must be \eqn{\ge 1}.
#' @param c A positive real number in \eqn{(0,1)} parameterizing the center inside \code{int}\eqn{=(a,b)}.
#' For the interval, \code{int}\eqn{=(a,b)}, the parameterized center is \eqn{M_c=a+c(b-a)}, and default=\code{0.5}
#' @param int A \code{vector} of two real numbers representing an interval.
#'
#' @return \eqn{I(x_2 \in N_{PE}(x_1,r,c))} for x2, that is, returns 1 if \eqn{x_2} in \eqn{N_{PE}(x_1,r,c)}, returns 0 otherwise
#'
#' @seealso \code{\link{IndNPEmid1D}}, \code{\link{IndNPEend1D}} and \code{\link{IndNCSint}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' c<-.4
#' r<-2
#' a<-0; b<-10; int<-c(a,b)
#'
#' IndNPEint(7,5,r,c,int)
#' IndNPEint(17,17,r,c,int)
#' IndNPEint(15,17,r,c,int)
#' IndNPEint(1,3,r,c,int)
#'
#' IndNPEint(-17,17,r,c,int)
#'
#' IndNPEint(3,5,r,c,int)
#' IndNPEint(3,3,r,c,int)
#' IndNPEint(4,5,r,c,int)
#' IndNPEint(a,5,r,c,int)
#'
#' c<-.4
#' r<-2
#' a<-0; b<-10; int<-c(a,b)
#' IndNPEint(7,5,r,c,int)
#'
#' @export IndNPEint
IndNPEint <- function(x1,x2,r,c=.5,int)
{
  if (!is.point(x2,1) )
  {stop('x2 must be a scalar')}

  arc<-0
  pr<-NPEint(x1,r,c,int)  #proximity region as interval
  if (x2>=pr[1] && x2<=pr[2])
  {arc<-1}
  arc
} #end of the function
#'

#################################################################

#' @title The plot of the Proportional Edge (PE) Proximity Regions for a general interval
#' (vertices jittered along \eqn{y}-coordinate) - one interval case
#'
#' @description Plots the points in and outside of the interval \code{int} and also the PE proximity regions (which are also intervals).
#' PE proximity regions are constructed with expansion parameter \eqn{r \ge 1} and centrality parameter \eqn{c \in (0,1)}.
#'
#' For better visualization, a uniform jitter from \eqn{U(-Jit,Jit)} (default is \eqn{Jit=.1}) times range of proximity
#' regions and \code{dat}) is added to the \eqn{y}-direction.
#' \code{centers} is a logical argument, if \code{TRUE}, plot includes the
#' centers of the intervals as vertical lines in the plot, else centers of the intervals are not plotted.
#'
#' See also (\insertCite{ceyhan:metrika-2012;textual}{pcds}).
#'
#' @param dat A set of 1D points for which PE proximity regions are to be constructed.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param c A positive real number in \eqn{(0,1)} parameterizing the center inside \code{int}\eqn{=(a,b)}.
#' For the interval, \code{int}\eqn{=(a,b)}, the parameterized center is \eqn{M_c=a+c(b-a)}; default=\code{0.5}
#' @param int A \code{vector} of two real numbers representing an interval.
#' @param Jit A positive real number that determines the amount of jitter along the \eqn{y}-axis, default=\code{0.1} and
#' \code{dat} points are jittered according to \eqn{U(-Jit,Jit)} distribution along the \eqn{y}-axis where \code{Jit} equals to the range of \code{dat} and proximity region
#' intervals multiplied by \code{Jit}).
#' @param main Title of the main heading of the plot.
#' @param xlab,ylab Titles for the \eqn{x} and \eqn{y} axes, respectively (default=\code{""} for both).
#' @param xlim,ylim Two \code{numeric} vectors of length 2, giving the \eqn{x}- and \eqn{y}-coordinate ranges.
#' @param centers A logical argument, if \code{TRUE}, plot includes the centers of the intervals
#' as vertical lines in the plot, else centers of the intervals are not plotted.
#' @param \dots	Additional \code{plot} parameters.
#'
#' @return Plot of the PE proximity regions for 1D points in or outside the interval \code{int}
#'
#' @seealso \code{\link{plotPEregsMI}}, \code{\link{plotCSregsInt}}, and \code{\link{plotCSregsMI}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' c<-.4
#' r<-2
#' a<-0; b<-10; int<-c(a,b)
#'
#' n<-10
#' xr<-range(a,b)
#' xf<-(xr[2]-xr[1])*.1
#'
#' dat<-runif(n,a-xf,b+xf)  #try also dat<-runif(n,a-5,b+5)
#'
#' plotPEregsInt(7,r,c,int)
#'
#' plotPEregsInt(dat,r,c,int)
#'
#' plotPEregsInt(17,r,c,int)
#' plotPEregsInt(1,r,c,int)
#' plotPEregsInt(4,r,c,int)
#'
#' plotPEregsInt(-7,r,c,int)
#'
#' @export plotPEregsInt
plotPEregsInt <- function(dat,r,c=.5,int,Jit=.1,main="",xlab="",ylab="",xlim=NULL,ylim=NULL,centers=FALSE, ...)
{
  if (!is.point(dat,length(dat)))
  {stop('dat must be a 1D vector of numerical entries')}
  n<-length(dat)
  pr<-c()
  for (i in 1:n)
  {x1<-dat[i]
  pr<-rbind(pr,NPEint(x1,r,c,int))
  }

  if (is.null(xlim))
  {xlim<-range(dat,int,pr)}

  xr<-xlim[2]-xlim[1]
  jit<-xr*Jit
  ifelse(n==1,yjit<-rep(0,n),yjit<-runif(n,-jit,jit))

  if (is.null(ylim))
  {ylim<-3*c(-jit,jit)}

  plot(dat, yjit,main=main, xlab=xlab, ylab=ylab,xlim=xlim+.05*xr*c(-1,1),ylim=ylim,pch=".",cex=3, ...)
  if (centers==TRUE)
  {cents<-centersMc(dat,c)
  abline(v=cents,lty=3)}
  abline(v=int,lty=2)
  abline(h=0,lty=2)
  for (i in 1:n)
  {
    plotrix::draw.arc(pr[i,1]+xr*.05,yjit[i],xr*.05, deg1=150,deg2 = 210, col = "blue")
    plotrix::draw.arc(pr[i,2]-xr*.05, yjit[i],xr*.05, deg1=-30,deg2 = 30, col = "blue")
    segments(pr[i,1], yjit[i], pr[i,2], yjit[i], col= "blue")
  }
} #end of the function
#'

#################################################################

#' @title Number of arcs of Proportional Edge Proximity Catch Digraphs (PE-PCDs) - one interval case
#'
#' @description Returns the number of arcs of Proportional Edge Proximity Catch Digraphs (PE-PCDs)
#' whose vertices are the 1D data set \code{dat} in the one-interval case.
#'
#' The data points could be inside or outside the interval is \code{int}\eqn{=(a,b)}. PE proximity region is constructed
#' with an expansion parameter \eqn{r \ge 1} and a centrality parameter \eqn{c \in (0,1)}.
#'
#' The PE proximity region is constructed for both points inside and outside the interval, hence
#' the arcs may exist for all points inside or outside the interval.
#'
#' See also (\insertCite{ceyhan:metrika-2012;textual}{pcds}).
#'
#' @param dat A set of 1D points which constitute the vertices of PE-PCD.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param c A positive real number in \eqn{(0,1)} parameterizing the center inside \code{int}\eqn{=(a,b)}.
#' For the interval, \code{int}\eqn{=(a,b)}, the parameterized center is \eqn{M_c=a+c(b-a)}; default=\code{0.5}
#' @param int A \code{vector} of two real numbers representing an interval.
#'
#' @return Number of arcs for the PE-PCD whose vertices are the 1D data set, \code{dat},
#' with expansion parameter, \eqn{r \ge 1}, and centrality parameter, \eqn{c \in (0,1)}.
#'
#' @seealso \code{\link{NumArcsPEmid1D}}, \code{\link{NumArcsPEend1D}}, and \code{\link{NumArcsCSint}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' c<-.4
#' r<-2
#' a<-0; b<-10; int<-c(a,b)
#'
#' n<-10
#' dat<-runif(n,a,b)
#' NumArcsPEint(dat,r,c,int)
#'
#' NumArcsPEint(3,r,c,int)
#'
#' NumArcsPEint(dat,r,c=.3,int)
#'
#' NumArcsPEint(dat,r=1.5,c,int)
#'
#' n<-10  #try also n<-20
#' dat<-runif(n,a,b)
#'
#' NumArcsPEint(dat,r,c,int)
#'
#' dat<-runif(n,a+10,b+10)
#' NumArcsPEint(dat,r,c,int)
#'
#' n<-10
#' dat<-runif(n,a,b)
#' NumArcsPEint(dat,r,c,int)
#'
#' @export NumArcsPEint
NumArcsPEint <- function(dat,r,c=.5,int)
{
  if (!is.point(dat,length(dat)))
  {stop('dat must be a 1D vector of numerical entries')}
  n<-length(dat)
  if (n>0)
  {
    arcs<-0
    for (i in 1:n)
    {x1<-dat[i]
    pr<-NPEint(x1,r,c,int)
    arcs<-arcs+sum(dat>=pr[1] & dat<=pr[2] )-1 #minus 1 is for the loop at dat.int[i]
    }
  } else
  {arcs<-0}
  arcs
} #end of the function
#'

#################################################################

#' @title The arcs of Proportional Edge Proximity Catch Digraph (PE-PCD) for 1D data - multiple interval case
#'
#' @description
#' An object of class \code{"PCDs"}.
#' Returns arcs as tails (or sources) and heads (or arrow ends) for 1D data set \code{Xp} as the vertices
#' of PE-PCD.  \code{Yp} determines the end points of the intervals.
#'
#' For this function, PE proximity regions are constructed data points inside or outside the intervals based
#' on \code{Yp} points with expansion parameter \eqn{r \ge 1} and centrality parameter \eqn{c \in (0,1)}. That is, for this function,
#' arcs may exist for points in the middle or end intervals.
#' It also provides various descriptions and quantities about the arcs of the PE-PCD
#' such as number of arcs, arc density, etc.
#'
#' Equivalent to function  \code{\link{ArcsPE1D}}.
#'
#' See also (\insertCite{ceyhan:metrika-2012;textual}{pcds}).
#'
#' @param Xp A set or \code{vector} of 1D points which constitute the vertices of the PE-PCD.
#' @param Yp A set or \code{vector} of 1D points which constitute the end points of the intervals.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param c A positive real number in \eqn{(0,1)} parameterizing the center inside \code{int}\eqn{=(a,b)}.
#' For the interval, \code{int}\eqn{=(a,b)}, the parameterized center is \eqn{M_c=a+c(b-a)}.
#'
#' @return A \code{list} with the elements
#' \item{type}{A description of the type of the digraph}
#' \item{parameters}{Parameters of the digraph, here, they are expansion and centrality parameters.}
#' \item{tess.points}{Points on which the tessellation of the study region is performed, here, tessellation
#' is the intervalization of the real line based on \code{Yp} points.}
#' \item{tess.name}{Name of data set used in tessellation, it is \code{Yp} for this function}
#' \item{vertices}{Vertices of the digraph, \code{Xp} points}
#' \item{vert.name}{Name of the data set which constitute the vertices of the digraph}
#' \item{S}{Tails (or sources) of the arcs of PE-PCD for 1D data}
#' \item{E}{Heads (or arrow ends) of the arcs of PE-PCD for 1D data}
#' \item{mtitle}{Text for \code{"main"} title in the plot of the digraph}
#' \item{quant}{Various quantities for the digraph: number of vertices, number of partition points,
#' number of intervals, number of arcs, and arc density.}
#'
#' @seealso \code{\link{ArcsPE1D}}, \code{\link{ArcsPEmid1D}}, \code{\link{ArcsPEend1D}}, and \code{\link{ArcsCSMI}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' r<-2
#' c<-.4
#' a<-0; b<-10;
#'
#' #nx is number of X points (target) and ny is number of Y points (nontarget)
#' nx<-20; ny<-4;  #try also nx<-40; ny<-10 or nx<-1000; ny<-10;
#'
#' set.seed(1)
#' xr<-range(a,b)
#' xf<-(xr[2]-xr[1])*.1
#'
#' Xp<-runif(nx,a-xf,b+xf)
#' Yp<-runif(ny,a,b)
#'
#' Arcs<-ArcsPEMI(Xp,Yp,r,c)
#' Arcs
#' summary(Arcs)
#' plot(Arcs)
#'
#' S<-Arcs$S
#' E<-Arcs$E
#'
#' ArcsPEMI(Xp,Yp,r,c)
#' ArcsPEMI(Xp,Yp+10,r,c)
#'
#' jit<-.1
#' yjit<-runif(nx,-jit,jit)
#'
#' Xlim<-range(a,b,Xp,Yp)
#' xd<-Xlim[2]-Xlim[1]
#'
#' plot(cbind(a,0),
#' main="arcs of PE-PCD for points (jittered along y-axis)\n in middle intervals ",
#' xlab=" ", ylab=" ", xlim=Xlim+xd*c(-.05,.05),ylim=3*c(-jit,jit),pch=".")
#' abline(h=0,lty=1)
#' points(Xp, yjit,pch=".",cex=3)
#' abline(v=Yp,lty=2)
#' arrows(S, yjit, E, yjit, length = .05, col= 4)
#'
#' r<-2
#' c<-.4
#' a<-0; b<-10;
#' nx<-20; ny<-4;  #try also nx<-40; ny<-10 or nx<-1000; ny<-10;
#' Xp<-runif(nx,a,b)
#' Yp<-runif(ny,a,b)
#' ArcsPEMI(Xp,Yp,r,c)
#'
#' @export ArcsPEMI
ArcsPEMI <- function(Xp,Yp,r,c)
{
  if (!is.point(Xp,length(Xp)) || !is.point(Yp,length(Yp)) )
  {stop('Xp and Yp must be 1D vectors of numerical entries')}

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if (!is.point(c,1) || c<=0 || c>=1)
  {stop('c must be a scalar in (0,1)')}

  nx<-length(Xp); ny<-length(Yp)
  S<-E<-vector()  #S is for source and E is for end points for the arcs
  if (nx==0 || ny==0)
  {stop('Not enough points to construct PE-PCD')}

  if (nx>1)
  {
    Ys<-sort(Yp)  #sorted data points from classes X and Y
    ymin<-Ys[1]; ymax<-Ys[ny];

   int<-rep(0,nx)
    for (i in 1:nx)
     int[i]<-(Xp[i]>ymin & Xp[i] < ymax )  #indices of X points in the middle intervals, i.e., inside min(Yp) and max (Yp)

    Xint<-Xp[int==1] # X points inside  min(Yp) and max (Yp)
    XLe<-Xp[Xp<ymin] # X points in the left end interval of Yp points
    XRe<-Xp[Xp>ymax] # X points in the right end interval of Yp points

    #for left end interval
    nle<-length(XLe)
    if (nle>1 )
    {
      for (j in 1:nle)
      {x1 <-XLe[j];  xLe<-XLe[-j] #to avoid loops
      xL<-ymin-r*(ymin-x1)
     ind.tails<-((xLe < ymin) & (xLe > xL))
      st<-sum(ind.tails)  #sum of tails of the arcs with head XLe[j]
      S<-c(S,rep(x1,st)); E<-c(E,xLe[ind.tails])
      }
    }

    #for middle intervals
    nt<-ny-1 #number of Yp middle intervals
    nx2<-length(Xint)  #number of Xp points inside the middle intervals

    if (nx2>1)
    {
      i.int<-rep(0,nx2)
      for (i in 1:nx2)
        for (j in 1:nt)
        {
          if (Xint[i]>=Ys[j] & Xint[i] < Ys[j+1] )
            i.int[i]<-j #indices of the Yp intervals in which X points reside
        }

      for (i in 1:nt)
      {
        Xi<-Xint[i.int==i] #X points in the ith Yp mid interval
        ni<-length(Xi)
        if (ni>1 )
        {
          y1<-Ys[i]; y2<-Ys[i+1]; int<-c(y1,y2)
          for (j in 1:ni)
          {x1 <-Xi[j] ; Xinl<-Xi[-j] #to avoid loops
          v<-rv.mid.int(x1,c,int)$rv
          if (v==1)
          {
            xR<-y1+r*(x1-y1)
           ind.tails<-((Xinl < min(xR,y2)) & (Xinl > y1))
            st<-sum(ind.tails)  #sum of tails of the arcs with head Xi[j]
            S<-c(S,rep(x1,st)); E<-c(E,Xinl[ind.tails])
          } else {
            xL <-y2-r*(y2-x1)
           ind.tails<-((Xinl < y2) & (Xinl > max(xL,y1)))
            st<-sum(ind.tails)  #sum of tails of the arcs with head Xi[j]
            S<-c(S,rep(x1,st)); E<-c(E,Xinl[ind.tails])
          }
          }
        }
      }
    }

    #for right end interval
    nre<-length(XRe)
    if (nre>1 )
    {

      for (j in 1:nre)
      {x1 <-XRe[j]; xRe<-XRe[-j]
      xR<-ymax+r*(x1-ymax)
     ind.tails<-((xRe < xR) & xRe > ymax )
      st<-sum(ind.tails)  #sum of tails of the arcs with head XRe[j]
      S<-c(S,rep(x1,st)); E<-c(E,xRe[ind.tails])
      }
    }
  }
  if (length(S)==0)
  {S<-E<-NA}

  xname <-deparse(substitute(Xp))
  yname <-deparse(substitute(Yp))
  cname <-deparse(substitute(c))
  rname <-deparse(substitute(r))

  param<-c(r,c)
  names(param)<-c("expansion parameter","centrality parameter")
  typ<-paste("Proportional Edge Proximity Catch Digraph (PE-PCD) for 1D Points in Multiple Intervals with Expansion Parameter ",rname, "=",r," and Centrality Parameter ", cname,"=",c,sep="")

  main.txt<-"Arcs of PE-PCD for Points (jittered\n along the y-axis) in Multiple Intervals"

  nvert<-nx; nint<-ny+1; narcs<-ifelse(sum(is.na(S))==0,length(S),0);
  arc.dens<-ifelse(nvert>1,narcs/(nvert*(nvert-1)),NA)

  quantities<-c(nvert,ny,nint,narcs,arc.dens)
  names(quantities)<-c("number of vertices", "number of partition points",
                       "number of intervals","number of arcs", "arc density")

  res<-list(
    type=typ,
    parameters=param,
    tess.points=Yp, tess.name=yname, #tessellation points
    vertices=Xp, vert.name=xname, #vertices of the digraph
    S=S,
    E=E,
    mtitle=main.txt,
    quant=quantities
  )

  class(res)<-"PCDs"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

#' @title Incidence matrix for Proportional-Edge Proximity Catch Digraphs (PE-PCDs)
#' for 1D data - multiple interval case
#'
#' @description Returns the incidence matrix for the PE-PCD for a given 1D numerical data set, \code{Xp},
#' as the vertices of the digraph and \code{Yp} determines the end points of the intervals (in the multi-interval case).
#' Loops are allowed, so the diagonal entries are all equal to 1.
#'
#' PE proximity region is constructed
#' with an expansion parameter \eqn{r \ge 1} and a centrality parameter \eqn{c \in (0,1)}.
#'
#' See also (\insertCite{ceyhan:metrika-2012;textual}{pcds}).
#'
#' @param Xp a set of 1D points which constitutes the vertices of the digraph.
#' @param Yp a set of 1D points which constitutes the end points of the intervals
#' that partition the real line.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param c A positive real number in \eqn{(0,1)} parameterizing the center inside \code{int}\eqn{=(a,b)}.
#' For the interval, \code{int}\eqn{=(a,b)}, the parameterized center is \eqn{M_c=a+c(b-a)}.
#'
#' @return Incidence matrix for the PE-PCD with vertices being 1D data set, \code{Xp},
#' and \code{Yp} determines the end points of the intervals (in the multi-interval case)
#'
#' @seealso \code{\link{IncMatCS1D}}, \code{\link{IncMatPEtri}}, and \code{\link{IncMatPEMT}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' r<-2
#' c<-.4
#' a<-0; b<-10;
#' nx<-10; ny<-4
#'
#' set.seed(1)
#' Xp<-runif(nx,a,b)
#' Yp<-runif(ny,a,b)
#'
#' IM<-IncMatPE1D(Xp,Yp,r,c)
#' IM
#'
#' dom.greedy(IM)
#' IndUBdom(IM,6)
#' dom.exact(IM)
#'
#' Arcs<-ArcsPEMI(Xp,Yp,r,c)
#' Arcs
#' summary(Arcs)
#' plot(Arcs)
#'
#' IncMatPE1D(Xp,Yp+10,r,c)
#'
#' r<-2
#' c<-.4
#' a<-0; b<-10;
#'
#' #nx is number of X points (target) and ny is number of Y points (nontarget)
#' nx<-20; ny<-4;  #try also nx<-40; ny<-10 or nx<-1000; ny<-10;
#' Xp<-runif(nx,a,b)
#' Yp<-runif(ny,a,b)
#' IncMatPE1D(Xp,Yp,r,c)
#'
#' @export IncMatPE1D
IncMatPE1D <- function(Xp,Yp,r,c)
{
  if (!is.point(Xp,length(Xp)) || !is.point(Yp,length(Yp)) )
  {stop('Xp and Yp must be 1D vectors of numerical entries')}

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if (!is.point(c,1) || c<=0 || c>=1)
  {stop('c must be a scalar in (0,1)')}

  nx<-length(Xp); ny<-length(Yp)
  nt<-ny-1 #number of Yp middle intervals
  if (nx==0 || ny==0)
  {stop('Not enough points to construct PE-PCD')}

  if (nx>=1)
  {
    Ys<-sort(Yp)  #sorted data points from classes X and Y
    ymin<-Ys[1]; ymax<-Ys[ny];

    pr<-c()
    for (i in 1:nx)
    { x1<-Xp[i]
    if (x1<ymin || x1>=ymax)
    {int<-c(ymin,ymax)
    pr<-rbind(pr,NPEint(x1,r,c,int))
    }
    if (nt>=1)
    {
      for (j in 1:nt)
      {
        if (x1>=Ys[j] & x1 < Ys[j+1] )
        {  y1<-Ys[j]; y2<-Ys[j+1]; int<-c(y1,y2)
        pr<-rbind(pr,NPEint(x1,r,c,int))
        }
      }
    }
    }

   inc.mat<-matrix(0, nrow=nx, ncol=nx)
    for (i in 1:nx)
    { reg<-pr[i,]
    for (j in 1:nx)
    {
     inc.mat[i,j]<-sum(Xp[j]>=reg[1] & Xp[j]<reg[2])
    }
    }
  }
 inc.mat
} #end of the function
#'

#################################################################

#' @title The plot of the subintervals based on \code{Yp} points together with \code{Xp} points
#'
#' @description Plots the \code{Xp} points and the intervals based on \code{Yp} points
#' points.
#'
#' @param Xp A set of 1D points whose scatter-plot is provided.
#' @param Yp A set of 1D points which constitute the end points of the intervals which
#' partition the real line.
#' @param main An overall title for the plot (default=\code{""}).
#' @param xlab,ylab Titles for the \eqn{x} and \eqn{y} axes, respectively (default=\code{""} for both).
#' @param xlim,ylim Two \code{numeric} vectors of length 2, giving the \eqn{x}- and \eqn{y}-coordinate ranges
#' (default=\code{NULL} for both).
#' @param \dots Additional \code{plot} parameters.
#'
#' @return Plot of the intervals based on \code{Yp} points and also scatter plot of \code{Xp} points
#'
#' @seealso \code{\link{plotPEregsMI}} and \code{\link{plotDeltri}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' a<-0; b<-10;
#'
#' #nx is number of X points (target) and ny is number of Y points (nontarget)
#' nx<-20; ny<-4;  #try also nx<-40; ny<-10 or nx<-1000; ny<-10;
#'
#' set.seed(1)
#' Xp<-runif(nx,a,b)
#' Yp<-runif(ny,a,b)
#'
#' plotIntervals(Xp,Yp,xlab="",ylab="")
#' plotIntervals(Xp,Yp+10,xlab="",ylab="")
#'
#' @export plotIntervals
plotIntervals <- function(Xp,Yp,main="",xlab="",ylab="",xlim=NULL,ylim=NULL, ...)
{
  if (!is.point(Xp,length(Xp)) || !is.point(Yp,length(Yp)) )
  {stop('Xp and Yp must be 1D vectors of numerical entries')}

  nx<-length(Xp); ny<-length(Yp)

  if (ny<1 || nx<1)
  {stop('Both Xp and Yp points must be nonempty to construct intervals')}

  Ys<-sort(Yp)
  LE<-Ys[1:(ny-1)]
  RE<-Ys[2:ny]

  if (is.null(xlim))
  {xl<-range(Xp,Yp)
  xr<-xl[2]-xl[1]
  xlim<-xl+.05*xr*c(-1,1)
  } else
  {
    xr<-xlim[2]-xlim[1]
    xlim<-xlim+.05*xr*c(-1,1)
  }

  #xr<-xlim[2]-xlim[1]

  if (is.null(ylim))
  {yl<-xr*.05*sin(30*pi/180)
    ylim<-yl*c(-1,1)
  }

  plot(cbind(Xp, 0),main=main, xlab=xlab, ylab=ylab,xlim=xlim,ylim=ylim,pch=".",cex=3, ...)
  points(cbind(Yp,0), col=2, ...)
  abline(h=0,lty=1)
  for (i in 1:ny)
  {
    plotrix::draw.arc(LE[i]+xr*.05, 0,xr*.05, deg1=150,deg2 = 210, col = "blue")
    plotrix::draw.arc(RE[i]-xr*.05, 0, xr*.05, deg1=-30,deg2 = 30, col = "blue")
  }
} #end of the function
#'

#################################################################

#' @title The plot of the Proportional Edge (PE) Proximity Regions (vertices jittered along \eqn{y}-coordinate)
#' - multiple interval case
#'
#' @description Plots the points in and outside of the intervals based on \code{Yp} points and also the PE proximity regions
#' (i.e., intervals). PE proximity region is constructed with expansion parameter \eqn{r \ge 1} and
#' centrality parameter \eqn{c \in (0,1)}.
#'
#' For better visualization, a uniform jitter from \eqn{U(-Jit,Jit)}
#' (default is \eqn{Jit=.1}) times range of \code{Xp} and \code{Yp} and the proximity regions (intervals)) is added to the
#' \eqn{y}-direction.
#'
#' \code{centers} is a logical argument, if \code{TRUE},
#' plot includes the centers of the intervals as vertical lines in the plot,
#' else centers of the intervals are not plotted.
#'
#' See also (\insertCite{ceyhan:metrika-2012;textual}{pcds}).
#'
#' @param Xp A set of 1D points for which PE proximity regions are plotted.
#' @param Yp A set of 1D points which constitute the end points of the intervals which
#' partition the real line.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param c A positive real number in \eqn{(0,1)} parameterizing the center inside \code{int}\eqn{=(a,b)}.
#' For the interval, \code{int}\eqn{=(a,b)}, the parameterized center is \eqn{M_c=a+c(b-a)}; default=\code{0.5}.
#' @param Jit A positive real number that determines the amount of jitter along the \eqn{y}-axis, default=\code{0.1} and
#' \code{Xp} points are jittered according to \eqn{U(-Jit,Jit)} distribution along the \eqn{y}-axis where \code{Jit} equals to the range of \code{Xp} and \code{Yp} and the
#' proximity regions (intervals) multiplied by \code{Jit}).
#' @param main An overall title for the plot (default=\code{""}).
#' @param xlab,ylab Titles for the \eqn{x} and \eqn{y} axes, respectively (default=\code{""} for both).
#' @param xlim,ylim Two \code{numeric} vectors of length 2, giving the \eqn{x}- and \eqn{y}-coordinate ranges
#' (default=\code{NULL} for both).
#' @param centers A logical argument, if \code{TRUE}, plot includes the centers of the intervals
#' as vertical lines in the plot, else centers of the intervals are not plotted (default is \code{FALSE}).
#' @param \dots Additional \code{plot} parameters.
#'
#' @return Plot of the PE proximity regions for 1D points located in the middle or end intervals
#' based on \code{Yp} points
#'
#' @seealso \code{\link{plotPEregsMI}}, \code{\link{plotCSregsInt}}, and \code{\link{plotCSregsMI}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' r<-2
#' c<-.4
#' a<-0; b<-10;
#'
#' #nx is number of X points (target) and ny is number of Y points (nontarget)
#' nx<-20; ny<-4;  #try also nx<-40; ny<-10 or nx<-1000; ny<-10;
#'
#' set.seed(1)
#' xr<-range(a,b)
#' xf<-(xr[2]-xr[1])*.1
#'
#' Xp<-runif(nx,a-xf,b+xf)
#' Yp<-runif(ny,a,b)
#'
#' plotPEregsMI(Xp,Yp,r,c,xlab="",ylab="")
#'
#' plotPEregsMI(Xp,Yp+10,r,c,xlab="",ylab="")
#'
#' @export
plotPEregsMI <- function(Xp,Yp,r,c,Jit=.1,main="",xlab="",ylab="",xlim=NULL,ylim=NULL,centers=FALSE, ...)
{
  if (!is.point(Xp,length(Xp)) || !is.point(Yp,length(Yp)) )
  {stop('Xp and Yp must be 1D vectors of numerical entries')}

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if (!is.point(c,1) || c<=0 || c>=1)
  {stop('c must be a scalar in (0,1)')}

  nx<-length(Xp); ny<-length(Yp)

  if (ny<1 || nx<1)
  {stop('Both Xp and Yp points must be nonempty to construct PE-PCD')}

  LE<-RE<-vector()
  if (nx>=1)
  { Xp<-sort(Xp)
  Ys<-sort(Yp)  #sorted data points from classes X and Y
  ymin<-Ys[1]; ymax<-Ys[ny];

 in.int<-rep(0,nx)
  for (i in 1:nx)
   in.int[i]<-(Xp[i]>ymin & Xp[i] < ymax )  #indices of X points in the middle intervals, i.e., inside min(Yp) and max (Yp)

  Xint<-Xp[in.int==1] # X points inside  min(Yp) and max (Yp)
  XLe<-Xp[Xp<ymin] # X points in the left end interval of Yp points
  XRe<-Xp[Xp>ymax] # X points in the right end interval of Yp points

  #for left end interval
  nle<-length(XLe)
  if (nle>=1 )
  {
    for (j in 1:nle)
    {x1 <-XLe[j]; int<-c(ymin,ymax)
    pr<-NPEint(x1,r,c,int)
    LE<-c(LE,pr[1]); RE<-c(RE,pr[2])
    }
  }

  #for middle intervals
  nt<-ny-1 #number of Yp middle intervals
  nx2<-length(Xint)  #number of Xp points inside the middle intervals

  if (nx2>=1)
  {
    i.int<-rep(0,nx2)
    for (i in 1:nx2)
      for (j in 1:nt)
      {
        if (Xint[i]>=Ys[j] & Xint[i] < Ys[j+1] )
          i.int[i]<-j #indices of the Yp intervals in which X points reside
      }

    for (i in 1:nt)
    {
      Xi<-Xint[i.int==i] #X points in the ith Yp mid interval
      ni<-length(Xi)
      if (ni>=1 )
      {
        y1<-Ys[i]; y2<-Ys[i+1]; int<-c(y1,y2)
        for (j in 1:ni)
        {x1 <-Xi[j] ;
        pr<-NPEint(x1,r,c,int)
        LE<-c(LE,pr[1]); RE<-c(RE,pr[2])
        }
      }

    }
  }

  #for right end interval
  nre<-length(XRe)
  if (nre>=1 )
  {

    for (j in 1:nre)
    {x1 <-XRe[j]; int<-c(ymin,ymax)
    pr<-NPEint(x1,r,c,int)
    LE<-c(LE,pr[1]); RE<-c(RE,pr[2])
    }
  }
  }

  if (is.null(xlim))
  {xlim<-range(Xp,Yp,LE,RE)}

  xr<-xlim[2]-xlim[1]
  jit<-xr*Jit
  yjit<-runif(nx,-jit,jit)

  if (is.null(ylim))
  {ylim<-3*c(-jit,jit)}

  plot(Xp, yjit,main=main, xlab=xlab, ylab=ylab,xlim=xlim+.05*xr*c(-1,1),ylim=ylim,pch=".",cex=3, ...)
  if (centers==TRUE)
  {cents<-centersMc(Yp,c)
  abline(v=cents,lty=3)}
  abline(v=Yp,lty=2)
  abline(h=0,lty=2)
  for (i in 1:nx)
  {
    plotrix::draw.arc(LE[i]+xr*.05, yjit[i],xr*.05, deg1=150,deg2 = 210, col = "blue")
    plotrix::draw.arc(RE[i]-xr*.05,  yjit[i],xr*.05, deg1=-30,deg2 = 30, col = "blue")
    segments(LE[i], yjit[i], RE[i], yjit[i], col= "blue")
  }
} #end of the function
#'

#################################################################

#' @title The indicator for a point being a dominating point for Proportional Edge
#' Proximity Catch Digraphs (PE-PCDs) for an interval
#'
#' @description Returns \eqn{I(}\code{p} is a dominating point of the PE-PCD\eqn{)} where the vertices of the PE-PCD are the 1D data set \code{Dt}.
#'
#' PE proximity region is defined with respect to the interval \code{int} with an expansion parameter, \eqn{r \ge 1},
#' and a centrality parameter, \eqn{c \in (0,1)}, so arcs may exist for \code{Dt} points inside the interval \code{int}\eqn{=(a,b)}.
#'
#' Vertex regions are based on the center associated with the centrality parameter \eqn{c \in (0,1)}.
#' \code{rv} is the index of the vertex region \code{p} resides, with default=\code{NULL}.
#'
#' \code{ch.data.pnt} is for checking whether point \code{p} is a data point in \code{Dt} or not (default is \code{FALSE}),
#' so by default this function checks whether the point \code{p} would be a dominating point
#' if it actually were in the data set.
#'
#' @param p A 1D point that is to be tested for being a dominating point or not of the PE-PCD.
#' @param Dt A set of 1D points which constitutes the vertices of the PE-PCD.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param c A positive real number in \eqn{(0,1)} parameterizing the center inside \code{int}\eqn{=(a,b)}.
#' For the interval, \code{int}\eqn{=(a,b)}, the parameterized center is \eqn{M_c=a+c(b-a)}; default \code{c=.5}.
#' @param int A \code{vector} of two real numbers representing an interval.
#' @param rv Index of the vertex region in which the point resides, either \code{1,2} or \code{NULL}
#' (default is \code{NULL}).
#' @param ch.data.pnt A logical argument for checking whether point \code{p} is a data point
#' in \code{Dt} or not (default is \code{FALSE}).
#'
#' @return \eqn{I(}\code{p} is a dominating point of the PE-PCD\eqn{)} where the vertices of the PE-PCD are the 1D data set \code{Dt},
#' that is, returns 1 if \code{p} is a dominating point, returns 0 otherwise
#'
#' @seealso \code{\link{Gam1PEtri}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' r<-2
#' c<-.4
#' a<-0; b<-10; int<-c(a,b)
#'
#' Mc<-centMc(int,c)
#'
#' n<-10
#'
#' set.seed(1)
#' dat<-runif(n,a,b)
#'
#' Gam1PE1D(dat[5],dat,r,c,int)
#'
#' gam.vec<-vector()
#' for (i in 1:n)
#' {gam.vec<-c(gam.vec,Gam1PE1D(dat[i],dat,r,c,int))}
#'
#' ind.gam1<-which(gam.vec==1)
#' ind.gam1
#'
#' domset<-dat[ind.gam1]
#' if (length(ind.gam1)==0)
#' {domset<-NA}
#'
#' #or try
#' Rv<-rv.mid.int(dat[5],c,int)$rv
#' Gam1PE1D(dat[5],dat,r,c,int,Rv)
#'
#' Xlim<-range(a,b,dat)
#' xd<-Xlim[2]-Xlim[1]
#'
#' plot(cbind(a,0),xlab="",pch=".",xlim=Xlim+xd*c(-.05,.05))
#' abline(h=0)
#' points(cbind(dat,0))
#' abline(v=c(a,b,Mc),col=c(1,1,2),lty=2)
#' points(cbind(domset,0),pch=4,col=2)
#' text(cbind(c(a,b,Mc),-0.1),c("a","b","Mc"))
#'
#' Gam1PE1D(dat[5],dat,r,c,int)
#'
#' n<-10
#' dat2<-runif(n,a+b,b+10)
#' Gam1PE1D(5,dat2,r,c,int)
#'
#' Gam1PE1D(2,dat,r,c,int,ch.data.pnt = FALSE)
#' #gives an error message if ch.data.pnt = TRUE since point p is not a data point in Dt
#'
#' @export Gam1PE1D
Gam1PE1D <- function(p,Dt,r,c,int,rv=NULL,ch.data.pnt=FALSE)
{
  if (!is.point(p,1) )
  {stop('p must be a scalar')}

  if (!is.point(Dt,length(Dt)))
  {stop('Dt must be a 1D vector of numerical entries')}

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if (!is.point(c,1) || c<=0 || c>=1)
  {stop('c must be a scalar in (0,1)')}

  if (!is.point(int))
  {stop('int must a numeric vector of length 2')}

  if (ch.data.pnt==TRUE)
  {
    if (!is.in.data(p,as.matrix(Dt)))
    {stop('p is not a data point in Dt')}
  }

  y1<-int[1]; y2<-int[2];
  if (y1>=y2)
  {stop('interval is degenerate or void, left end must be smaller than right end')}

  if (is.null(rv))
  {rv<-rv.mid.int(p,c,int)$rv #determines the vertex region for 1D point p
  } else
  {  if (!is.numeric(rv) || sum(rv==c(1,2,3))!=1)
  {stop('vertex index, rv, must be 1, 2 or 3')}}

  Dt<-Dt[(Dt>=y1 & Dt<=y2)]
  n<-length(Dt)
  dom<-1; i<-1;
  while (i <= n & dom==1)
  {if (IndNPEint(p,Dt[i],r,c,int)==0)
  {dom<-0;}
    i<-i+1;
  }
  dom
} #end of the function
#'

#################################################################

# funsPG2PE1D
#'
#' @title The functions for probability of domination number \eqn{= 2} for Proportional Edge Proximity Catch Digraphs
#' (PE-PCDs) - middle interval case
#'
#' @description
#' The function \code{PG2PE1D} and its auxiliary functions.
#'
#' Returns \eqn{P(\gamma=2)} for PE-PCD whose vertices are a uniform data set of size \code{n} in a finite interval
#' \eqn{(a,b)} where \eqn{\gamma} stands for the domination number.
#'
#' The PE proximity region \eqn{N_{PE}(x,r,c)} is defined with respect to \eqn{(a,b)} with centrality parameter \eqn{c \in (0,1)}
#' and expansion parameter \eqn{r \ge 1}.
#'
#' To compute the probability \eqn{P(\gamma=2)} for PE-PCD in the 1D case,
#' we partition the domain \eqn{(r,c)=(1,\infty) \times (0,1)}, and compute the probability for each partition
#' set. The sample size (i.e., number of vertices or data points) is a positive integer, \code{n}.
#'
#' @section Auxiliary Functions for \code{PG2PE1D}:
#' The auxiliary functions are \code{PG2AI, PG2AII, PG2AIII, PG2AIV, PG2A, PG2Asym, PG2BIII, PG2B, PG2B,
#' PG2Bsym, PG2CIV, PG2C}, and \code{PG2Csym}, each corresponding to a partition of the domain of
#' \code{r} and \code{c}. In particular, the domain partition is handled in 3 cases as
#'
#' CASE A: \eqn{c \in ((3-\sqrt{5})/2, 1/2)}
#'
#' CASE B: \eqn{c \in (1/4,(3-\sqrt{5})/2)} and
#'
#' CASE C: \eqn{c \in (0,1/4)}.
#'
#'
#' @section Case A - \eqn{c \in ((3-\sqrt{5})/2, 1/2)}:
#' In Case A, we compute \eqn{P(\gamma=2)} with
#'
#' \code{PG2AIV(r,c,n)} if \eqn{1 < r < (1-c)/c};
#'
#' \code{PG2AIII(r,c,n)} if \eqn{(1-c)/c< r < 1/(1-c)};
#'
#' \code{PG2AII(r,c,n)} if \eqn{1/(1-c)< r < 1/c};
#'
#' and \code{PG2AI(r,c,n)} otherwise.
#'
#' \code{PG2A(r,c,n)} combines these functions in Case A: \eqn{c \in ((3-\sqrt{5})/2,1/2)}.
#' Due to the symmetry in the PE proximity regions, we use \code{PG2Asym(r,c,n)} for \eqn{c} in
#' \eqn{(1/2,(\sqrt{5}-1)/2)} with the same auxiliary functions
#'
#' \code{PG2AIV(r,1-c,n)} if \eqn{1 < r < c/(1-c)};
#'
#' \code{PG2AIII(r,1-c,n)} if \eqn{(c/(1-c) < r < 1/c};
#'
#' \code{PG2AII(r,1-c,n)} if \eqn{1/c < r < 1/(1-c)};
#'
#' and \code{PG2AI(r,1-c,n)} otherwise.
#'
#' @section Case B - \eqn{c \in (1/4,(3-\sqrt{5})/2)}:
#'
#' In Case B, we compute \eqn{P(\gamma=2)} with
#'
#' \code{PG2AIV(r,c,n)} if \eqn{1 < r < 1/(1-c)};
#'
#' \code{PG2BIII(r,c,n)} if \eqn{1/(1-c) < r < (1-c)/c};
#'
#' \code{PG2AII(r,c,n)} if \eqn{(1-c)/c < r < 1/c};
#'
#' and \code{PG2AI(r,c,n)} otherwise.
#'
#' PG2B(r,c,n) combines these functions in Case B: \eqn{c \in (1/4,(3-\sqrt{5})/2)}.
#' Due to the symmetry in the PE proximity regions, we use \code{PG2Bsym(r,c,n)} for \code{c} in
#' \eqn{((\sqrt{5}-1)/2,3/4)} with the same auxiliary functions
#'
#' \code{PG2AIV(r,1-c,n)} if \eqn{ 1< r < 1/c};
#'
#' \code{PG2BIII(r,1-c,n)} if \eqn{1/c < r < c/(1-c)};
#'
#' \code{PG2AII(r,1-c,n)} if \eqn{c/(1-c) < r < 1/(1-c)};
#'
#' and \code{PG2AI(r,1-c,n)} otherwise.
#'
#' @section Case C - \eqn{c \in (0,1/4)}:
#'
#' In Case C, we compute \eqn{P(\gamma=2)} with
#'
#' \code{PG2AIV(r,c,n)} if \eqn{1< r < 1/(1-c)};
#'
#' \code{PG2BIII(r,c,n)} if \eqn{1/(1-c) < r < (1-\sqrt{1-4 c})/(2 c)};
#'
#' \code{PG2CIV(r,c,n)} if \eqn{(1-\sqrt{1-4 c})/(2 c) < r < (1+\sqrt{1-4 c})/(2 c)};
#'
#' \code{PG2BIII(r,c,n)} if \eqn{(1+\sqrt{1-4 c})/(2 c) < r <1/(1-c)};
#'
#' \code{PG2AII(r,c,n)} if \eqn{1/(1-c) < r < 1/c};
#'
#' and \code{PG2AI(r,c,n)} otherwise.
#'
#' \code{PG2C(r,c,n)} combines these functions in Case C: \eqn{c \in (0,1/4)}.
#' Due to the symmetry in the PE proximity regions, we use \code{PG2Csym(r,c,n)} for \eqn{c \in (3/4,1)}
#' with the same auxiliary functions
#'
#' \code{PG2AIV(r,1-c,n)} if \eqn{1< r < 1/c};
#'
#' \code{PG2BIII(r,1-c,n)} if \eqn{1/c < r < (1-\sqrt{1-4(1-c)})/(2(1-c))};
#'
#' \code{PG2CIV(r,1-c,n)} if \eqn{(1-\sqrt{1-4(1-c)})/(2(1-c)) < r < (1+\sqrt{1-4(1-c)})/(2(1-c))};
#'
#' \code{PG2BIII(r,1-c,n)} if \eqn{(1+\sqrt{1-4(1-c)})/(2(1-c)) < r < c/(1-c)};
#'
#' \code{PG2AII(r,1-c,n)} if \eqn{c/(1-c)< r < 1/(1-c)};
#'
#' and \code{PG2AI(r,1-c,n)} otherwise.
#'
#' Combining Cases A, B, and C, we get our main function \code{PG2PE1D} which computes \eqn{P(\gamma=2)}
#' for any (\code{r,c}) in its domain.
#'
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param c A positive real number in \eqn{(0,1)} parameterizing the center inside \code{int}\eqn{=(a,b)}.
#' For the interval, \code{int}\eqn{=(a,b)}, the parameterized center is \eqn{M_c=a+c(b-a)}.
#' @param n A positive integer representing the size of the uniform data set.
#'
#' @return \eqn{P(}domination number\eqn{=2)} for PE-PCD whose vertices are a uniform data set of size \code{n} in a finite
#' interval \eqn{(a,b)}
#'
#' @name funsPG2PE1D
NULL
#'
#' @seealso \code{\link{PG2PEtri}} and  \code{\link{PG2PE1D.asy}}
#'
#' @author Elvan Ceyhan
#'
#' @rdname funsPG2PE1D
#'
PG2AI <- function(r,c,n)
{
  r^2*(2^n*(1/r)^n*r-2^n*(1/r)^n-2*((r-1)/r^2)^n*r)/((r-1)*(r+1)^2);
} #end of the function
#'
#' @rdname funsPG2PE1D
#'
PG2AII <- function(r,c,n)
{
  -1/((r-1)*(r+1)^2)*r*(((r-1)/r^2)^n*r^2-((c*r+1)/r)^n*r^2+(-(c-1)/r)^n*r^2+((c*r^2+c*r-r+1)/r)^n*r+((r-1)*(c*r+c-1)/r)^n+((c*r+1)/r)^n-((c*r^2+c*r-r+1)/r)^n-(-(c-1)/r)^n);
} #end of the function
#'
#' @rdname funsPG2PE1D
#'
PG2AIII <- function(r,c,n)
{
  -1/(r-1)/(r+1)^2*((-(c-1)/r)^n*r^3+(c/r)^n*r^3+(r-1)^n*r^3-(r-1)^(1+n)*r^2+(-(c-1)*r)^n*r^2+(c*r)^n*r^2-(r-1)^n*r^2-r^3+((r-1)*(c*r+c-1)/r)^n*r+(-(r-1)/r*(c*r+c-r))^n*r-r*(-(c-1)/r)^n-r*(c/r)^n-(r-1)^n*r-r^2-(-(c-1)*r)^n-(c*r)^n+(r-1)^n+r+1);
} #end of the function
#'
#' @rdname funsPG2PE1D
#'
PG2AIV <- function(r,c,n)
{
  1/(r-1)/(r+1)^2*(-(-(c-1)/r)^n*r^3-(c/r)^n*r^3-(r-1)^n*r^3+(r-1)^(1+n)*r^2-(-(c-1)*r)^n*r^2-(c*r)^n*r^2+(r-1)^n*r^2+r^3-(-(r-1)/r*(c*r+c-r))^n*r+r*(-(c-1)/r)^n+r*(c/r)^n+(r-1)^n*r+r^2+(-(r-1)*(c*r+c-1))^n+(-(c-1)*r)^n+(c*r)^n-(r-1)^n-r-1);
} #end of the function
#'
#' @rdname funsPG2PE1D
#'
PG2A <- function(r,c,n)
{
  if (r<1)
  {pg2<-0;
  } else {
    if (r<(1-c)/c)
    {
      pg2<-PG2AIV(r,c,n);
    } else {
      if (r<1/(1-c))
      {
        pg2<-PG2AIII(r,c,n);
      } else {
        if (r<1/c)
        {
          pg2<-PG2AII(r,c,n);
        } else {
          pg2<-PG2AI(r,c,n);
        }}}}
  pg2
} #end of the function
#'
#' @rdname funsPG2PE1D
#'
PG2Asym <- function(r,c,n)
{
  if (r<1)
  {pg2<-0;
  } else {
    if (r<c/(1-c))
    {
      pg2<-PG2AIV(r,1-c,n);
    } else {
      if (r<1/c)
      {
        pg2<-PG2AIII(r,1-c,n);
      } else {
        if (r<1/(1-c))
        {
          pg2<-PG2AII(r,1-c,n);
        } else {
          pg2<-PG2AI(r,1-c,n);
        }}}}
  pg2
} #end of the function
#'
#' @rdname funsPG2PE1D
#'
PG2BIII <- function(r,c,n)
{
  -1/(r-1)/(r+1)^2*(r^3*((r-1)/r^2)^n-((c*r+1)/r)^n*r^3+(-(c-1)/r)^n*r^3+((c*r^2+c*r-r+1)/r)^n*r^2+r*((c*r+1)/r)^n-((c*r^2+c*r-r+1)/r)^n*r-r*(-(c-1)/r)^n-(-(r-1)*(c*r+c-1))^n);
} #end of the function
#'
#' @rdname funsPG2PE1D
#'
PG2B <- function(r,c,n)
{
  if (r<1)
  { pg2<-0;
  } else {
    if (r<1/(1-c))
    {
      pg2<-PG2AIV(r,c,n);
    } else {
      if (r<(1-c)/c)
      {
        pg2<-PG2BIII(r,c,n);
      } else {
        if (r<1/c)
        {
          pg2<-PG2AII(r,c,n);
        } else {
          pg2<-PG2AI(r,c,n);
        }}}}
  pg2
} #end of the function
#'
#' @rdname funsPG2PE1D
#'
PG2Bsym <- function(r,c,n)
{
  if (r<1)
  {pg2<-0;
  } else {
    if (r<1/c)
    {
      pg2<-PG2AIV(r,1-c,n);
    } else {
      if (r<c/(1-c))
      {
        pg2<-PG2BIII(r,1-c,n);
      } else {
        if (r<1/(1-c))
        {
          pg2<-PG2AII(r,1-c,n);
        } else {
          pg2<-PG2AI(r,1-c,n);
        }}}}
  pg2
} #end of the function
#'
#' @rdname funsPG2PE1D
#'
PG2CIV <- function(r,c,n)
{
  1/(r+1)*(-r*(-(c-1)/r)^n-c^n*r-c^n+r*((c*r+1)/r)^n);
} #end of the function
#'
#' @rdname funsPG2PE1D
#'
PG2C <- function(r,c,n)
{
  if (r<1)
  { pg2<-0;
  } else {
    if (r<1/(1-c))
    {
      pg2<-PG2AIV(r,c,n);
    } else {
      if (r<(1-sqrt(1-4*c))/(2*c))
      {
        pg2<-PG2BIII(r,c,n);
      } else {
        if (r<(1+sqrt(1-4*c))/(2*c))
        {
          pg2<-PG2CIV(r,c,n);
        } else {
          if (r<(1-c)/c)
          {
            pg2<-PG2BIII(r,c,n);
          } else {
            if (r<1/c)
            {
              pg2<-PG2AII(r,c,n);
            } else {
              pg2<-PG2AI(r,c,n);
            }}}}}}
  pg2
} #end of the function
#'
#' @rdname funsPG2PE1D
#'
PG2Csym <- function(r,c,n)
{
  if (r<1)
  { pg2<-0;
  } else {
    if (r<1/c)
    {
      pg2<-PG2AIV(r,1-c,n);
    } else {
      if (r<(1-sqrt(1-4*(1-c)))/(2*(1-c)))
      {
        pg2<-PG2BIII(r,1-c,n);
      } else {
        if (r<(1+sqrt(1-4*(1-c)))/(2*(1-c)))
        {
          pg2<-PG2CIV(r,1-c,n);
        } else {
          if (r<c/(1-c))
          {
            pg2<-PG2BIII(r,1-c,n);
          } else {
            if (r<1/(1-c))
            {
              pg2<-PG2AII(r,1-c,n);
            } else {
              pg2<-PG2AI(r,1-c,n);
            }}}}}}
  pg2
} #end of the function
#'
#' @rdname funsPG2PE1D
#'
#' @examples
#' #Examples for the main function PG2PE1D
#' r<-2
#' c<-.5
#' n<-10
#'
#' PG2PE1D(r,c,n)
#'
#' PG2PE1D(r=1.5,c=1/1.5,n=100)
#'
#' @export
PG2PE1D <- function(r,c,n)
{
  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if (!is.point(c,1) || c<=0 || c>=1)
  {stop('c must be a scalar in (0,1)')}

  if (c<=0 | c>=1)
  { pg2<-0;
  } else {
    if (c < 1/4)
    {
      pg2<-PG2C(r,c,n);
    } else {
      if (c < (3-sqrt(5))/2)
      {
        pg2<-PG2B(r,c,n);
      } else {
        if (c < 1/2)
        {
          pg2<-PG2A(r,c,n);
        } else {
          if (c < (sqrt(5)-1)/2)
          {
            pg2<-PG2Asym(r,c,n);
          } else {
            if (c < 3/4)
            {
              pg2<-PG2Bsym(r,c,n);
            } else {
              pg2<-PG2Csym(r,c,n);
            }}}}}}
  pg2
} #end of the function
#'

#################################################################
#'
#' @title The asymptotic probability of domination number \eqn{= 2} for Proportional Edge Proximity Catch Digraphs (PE-PCDs)
#' - middle interval case
#'
#' @description Returns the asymptotic \eqn{P(}domination number\eqn{=2)} for PE-PCD whose vertices are a uniform data set in a finite
#' interval \eqn{(a,b)}.
#'
#' The PE proximity region \eqn{N_{PE}(x,r,c)} is defined with respect to \eqn{(a,b)} with centrality parameter \code{c}
#' in \eqn{(0,1)} and expansion parameter \eqn{r=1/\max(c,1-c)}.
#'
#' @param c A positive real number in \eqn{(0,1)} parameterizing the center inside \code{int}\eqn{=(a,b)}.
#' For the interval, \code{int}\eqn{=(a,b)}, the parameterized center is \eqn{M_c=a+c(b-a)}.
#'
#' @return The asymptotic \eqn{P(}domination number\eqn{=2)} for PE-PCD whose vertices are a uniform data set in a finite
#' interval \eqn{(a,b)}
#'
#' @seealso \code{\link{PG2PE1D}} and \code{\link{PG2PEtri}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' c<-.5
#'
#' PG2PE1D.asy(c)
#'
#' PG2PE1D.asy(c=1/1.5)
#' PG2PE1D(r=1.5,c=1/1.5,n=10)
#' PG2PE1D(r=1.5,c=1/1.5,n=100)
#'
#' @export PG2PE1D.asy
PG2PE1D.asy <- function(c)
{
  if (!is.point(c,1) || c<=0 || c>=1)
  {stop('c must be a scalar in (0,1)')}

  rstar<-1/max(c,1-c)  #r value for the non-degenerate asymptotic distribution

  if (c<=0 | c>=1)
  { pg2<-0;
  } else {
    if (c != 1/2)
    {
      pg2<-rstar/(1+rstar);
    } else {
      pg2<-4/9
    }}
  pg2
} #end of the function
#'

#################################################################

#' @title Indices of the intervals where the 1D point(s) reside
#'
#' @description Returns the indices of intervals for all the points in 1D data set, \code{dat}, as a vector.
#'
#' Intervals are based on \code{Yp} and left end interval is labeled as 0, the next interval as 1, and so on.
#'
#' @param dat A set of 1D points for which the indices of intervals are to be determined.
#' @param Yp A set of 1D points from which intervals are constructed.
#'
#' @return The \code{vector} of indices of the intervals in which points in the 1D data set, \code{dat}, reside
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' a<-0; b<-10; int<-c(a,b)
#'
#' #nx is number of X points (target) and ny is number of Y points (nontarget)
#' nx<-20; ny<-4;  #try also nx<-40; ny<-10 or nx<-1000; ny<-10;
#'
#' set.seed(1)
#' xr<-range(a,b)
#' xf<-(xr[2]-xr[1])*.1
#' Xp<-runif(nx,a-xf,b+xf)
#' Yp<-runif(ny,a,b)  #try also Yp<-runif(ny,a+1,b-1)
#'
#' ind<-ind.int.set(Xp,Yp)
#' ind
#'
#' jit<-.1
#' yjit<-runif(nx,-jit,jit)
#'
#' Xlim<-range(a,b,Xp,Yp)
#' xd<-Xlim[2]-Xlim[1]
#'
#' plot(cbind(a,0), xlab=" ", ylab=" ",xlim=Xlim+xd*c(-.05,.05),ylim=3*c(-jit,jit),pch=".")
#' abline(h=0)
#' points(Xp, yjit,pch=".",cex=3)
#' abline(v=Yp,lty=2)
#' text(Xp,yjit,labels=factor(ind))
#'
#' ind.int.set(3,5)
#' ind.int.set(6,5)
#'
#' ind.int.set(Xp,Yp)
#'
#' @export ind.int.set
ind.int.set <- function(dat,Yp)
{
  if (!is.point(dat,length(dat)) || !is.point(Yp,length(Yp)))
  {stop('Both arguments must be 1D vectors of numerical entries')}

  nt<-length(dat)
  ny<-length(Yp)
 ind.set<-rep(0,nt)
  Ys<-sort(Yp)
 ind.set[dat<Ys[1]]<-0; ind.set[dat>Ys[ny]]<-ny;
  for (i in 1:(ny-1))
  {
   ind<-(dat>=Ys[i] & dat<=Ys[i+1] )
   ind.set[ind]<-i
  }
 ind.set
} #end of the function
#'

#################################################################

#' @title The domination number of Proportional Edge Proximity Catch Digraph (PE-PCD) for 1D data
#'
#' @description Returns the domination number of PE-PCD whose vertices are the 1D data set \code{Xp}.
#'
#' \code{Yp} determines the end points of the intervals (i.e., partition the real line via intervalization).
#'
#' PE proximity region is constructed with expansion parameter \eqn{r \ge 1} and centrality parameter \eqn{c \in (0,1)}.
#'
#' @param Xp A set of 1D points which constitute the vertices of the PE-PCD.
#' @param Yp A set of 1D points which constitute the end points of the intervals which
#' partition the real line.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param c A positive real number in \eqn{(0,1)} parameterizing the center inside \code{int} (default \code{c=.5}).
#'
#' @return A \code{list} with two elements
#' \item{dom.num}{Domination number of PE-PCD with vertex set = \code{Xp} and expansion parameter \eqn{r \ge 1} and
#' centrality parameter \eqn{c \in (0,1)}.}
#' \item{mds}{A minimum dominating set of PE-PCD with vertex set = \code{Xp} and expansion parameter \eqn{r \ge 1} and
#' centrality parameter \eqn{c \in (0,1)}.}
#'
#' @seealso \code{\link{PEdomMTnd}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' a<-0; b<-10; int<-c(a,b)
#' c<-.4
#' r<-2
#'
#' #nx is number of X points (target) and ny is number of Y points (nontarget)
#' nx<-20; ny<-4;  #try also nx<-40; ny<-10 or nx<-1000; ny<-10;
#'
#' set.seed(1)
#' Xp<-runif(nx,a,b)
#' Yp<-runif(ny,a,b)
#'
#' plotIntervals(Xp,Yp,main="Xp Points and Intervals Based on Yp Points")
#' plotPEregsMI(Xp,Yp,r,c,main="PE Proximity Regions for Xp points - Intervalization is by Yp Points")
#'
#' PEdom1D(Xp,Yp,r,c)
#'
#' PEdom1D(Xp,Yp,r,c=.25)
#' PEdom1D(Xp,Yp,r,c=.01)
#' PEdom1D(Xp,Yp,r=1.25,c)
#'
#' @export PEdom1D
PEdom1D <- function(Xp,Yp,r,c=.5)
{
  if (!is.point(Xp,length(Xp)) || !is.point(Yp,length(Yp)))
  {stop('Xp and Yp must be 1D vectors of numerical entries')}

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if (!is.point(c,1) || c<=0 || c>=1)
  {stop('c must be a scalar in (0,1)')}

  nx<-length(Xp)  #number of Xp points
  ny<-length(Yp)  #number of Yp points

  Ys<-sort(Yp)  #sorted Yp points (ends of the subintervals)
  nint<-ny-1

  if (nint==0)
  {
    gam<-0; mds<-NULL
  } else
  {
    Int.Ind<-ind.int.set(Xp,Ys)  #indices of intervals in which Xp points in the data fall

    #calculation of the domination number
    gam<-rep(0,nint);
    mds<-c()
    for (i in 1:nint)
    {
      dati<-Xp[Int.Ind==i] #points in ith interval
      ni<-length(dati)  #number of points in ith interval
      if (ni==0)
      {
        gam[i]<-0
      } else
      {
       int<-c(Ys[i],Ys[i+1])  #end points of the ith interval
        Clvert<-as.numeric(cl2Mc.int(dati,int,c)$ext)

        #Gamma=1 piece
        cnt<-0; j<-1;
        while (j<=2 & cnt==0)
        {
          if ( !is.na(Clvert[j]) && Gam1PE1D(Clvert[j],dati,r,c,int,rv=j)==1)
          {gam[i]<-1; cnt<-1; mds<-c(mds,Clvert[j])
          }
          else
          {j<-j+1}
        }

        #Gamma=2 piece
        if (cnt==0)
        {gam[i]<-2; mds<-c(mds,Clvert)}
      }
    }
  }
  Gam<-sum(gam)  #domination number for the entire digraph in middle intervals
  Dom.Num<-Gam+sum(sum(Xp<Ys[1])>0)+sum(sum(Xp>Ys[ny])>0)  #adding the domination number in the end intervals
  if (sum(Xp<Ys[1])>0) {ledp<-min(Xp[Xp<Ys[1]])} else {ledp<-NULL} #left end interval dominating point
  if (sum(Xp>Ys[ny])>0) {redp<-max(Xp[Xp>Ys[ny]])} else {redp<-NULL} #right end interval dominating point
  MDS<-c(ledp,mds,redp)  #a minimum dominating set
  list(dom.num=Dom.Num,  #domination number
       mds=MDS #a minimum dominating set
  )
} #end of the function
#'

#################################################################

#' @title A test of segregation/association based on domination number of Proportional Edge Proximity Catch Digraph
#' (PE-PCD) for 1D data - Binomial Approximation
#'
#' @description
#' An object of class \code{"htest"} (i.e., hypothesis test) function which performs a hypothesis test of complete spatial
#' randomness (CSR) or uniformity of \code{Xp} points within the subintervals based on \code{Yp} points (both residing in the
#' interval \eqn{(a,b)}).
#'
#' If \code{Yp=NULL} the support interval \eqn{(a,b)} is partitioned as \code{Yp=(b-a)*(0:nint)/nint}
#' where \code{nint=round(sqrt(nx),0)} and \code{nx} is number of \code{Xp} points, and the test is for testing the uniformity of \code{Xp}
#' points in the interval \eqn{(a,b)}. If \code{Yp} points are given, the test is for testing the spatial interaction between
#' \code{Xp} and \code{Yp} points.
#'
#' In either case, the null hypothesis is uniformity of \code{Xp} points on \eqn{(a,b)}.
#' \code{Yp} determines the end points of the intervals (i.e., partition the real line via intervalization)
#' where end points are the order statistics of \code{Yp} points.
#'
#' The alternatives are segregation (where \code{Xp} points cluster away from \code{Yp} points i.e., cluster around the centers of the subintervals)
#' and association (where \code{Xp} points cluster around \code{Yp} points). The test is based on the (asymptotic) binomial
#' distribution of the domination number of PE-PCD for uniform 1D data in the subintervals based on \code{Yp} points.
#'
#' The function yields the test statistic, \eqn{p}-value for the corresponding
#' alternative, the confidence interval, estimate and null value for the parameter of interest (which is
#' \eqn{Pr(}domination number\eqn{=2)}), and method and name of the data set used.
#'
#' Under the null hypothesis of uniformity of \code{Xp} points in the interval based on \code{Yp} points, probability of success
#' (i.e., \eqn{Pr(}domination number\eqn{=2)}) equals to its expected value under the uniform distribution) and
#' \code{alternative} could be two-sided, or left-sided (i.e., data is accumulated around the \code{Yp} points, or association)
#' or right-sided (i.e., data is accumulated around the centers of the subintervals, or segregation).
#'
#' PE proximity region is constructed with the expansion parameter \eqn{r \ge 1} and centrality parameter \code{c} which yields
#' \eqn{M}-vertex regions. More precisely \eqn{M=a+c(b-a)} for the centrality parameter \code{c} and for a given \eqn{c \in (0,1)}, the
#' expansion parameter r is taken to be \eqn{1/\max(c,1-c)} which yields non-degenerate asymptotic distribution of the
#' domination number.
#'
#' The test statistic is based on the binomial distribution, when domination number is scaled
#' to have value 0 and 1 in the one interval case (i.e., Domination Number minus 1 for the one interval case).
#' That is, the test statistic is based on the domination number for \code{Xp} points inside the interval based on \code{Yp} points
#' for the PE-PCD . For this approach to work, \code{Xp} must be large for each subinterval, but 5 or more per subinterval
#' seems to work fine in practice. Probability of success is chosen in the following way for various parameter choices.
#'
#' \code{asy.bin} is a logical argument for the use of asymptotic probability of success for the binomial distribution,
#' default is \code{asy.bin=FALSE}. It is an option only when \code{Yp} is not provided. When \code{Yp} is provided or when \code{Yp} is not provided
#' but \code{asy.bin=TRUE}, asymptotic probability of success for the binomial distribution is used. When \code{Yp} is not provided
#' and \code{asy.bin=FALSE}, the finite sample asymptotic probability of success for the binomial distribution is used with number
#' of trials equals to expected number of \code{Xp} points per subinterval.
#'
#' @param Xp A set of 1D points which constitute the vertices of the PE-PCD.
#' @param Yp A set of 1D points which constitute the end points of the subintervals, default is \code{NULL}.
#' When \code{Yp=NULL}, the support interval \eqn{(a,b)} is partitioned as \code{Yp=(b-a)*(0:nint)/nint}
#' where \code{nint=round(sqrt(nx),0)} and \code{nx} is the number of \code{Xp} points.
#' @param int Support interval \eqn{(a,b)} with \eqn{a<b}. uniformity of \code{Xp} points in this interval
#' is tested.
#' @param c A positive real number which serves as the centrality parameter in PE proximity region;
#' must be in \eqn{(0,1)} (default \code{c=.5}).
#' @param asy.bin A logical argument for the use of asymptotic probability of success for the binomial distribution,
#' default \code{asy.bin=FALSE}. It is an option only when \code{Yp} is not provided. When \code{Yp} is provided or when \code{Yp} is not provided
#' but \code{asy.bin=TRUE}, asymptotic probability of success for the binomial distribution is used. When \code{Yp} is not provided
#' and \code{asy.bin=FALSE}, the finite sample asymptotic probability of success for the binomial distribution is used with number
#' of trials equals to expected number of \code{Xp} points per subinterval.
#' @param end.int.cor A logical argument for end interval correction, default is \code{FALSE},
#' recommended when both \code{Xp} and \code{Yp} have the same interval support.
#' @param alternative Type of the alternative hypothesis in the test, one of \code{"two.sided"}, \code{"less"}, \code{"greater"}.
#' @param conf.level Level of the confidence interval, default is \code{0.95}, for the probability of success
#' (i.e., \eqn{Pr(}domination number\eqn{=2)} for PE-PCD whose vertices are the 1D data set \code{Xp}.
#'
#' @return A \code{list} with the elements
#' \item{statistic}{Test statistic}
#' \item{p.value}{The \eqn{p}-value for the hypothesis test for the corresponding \code{alternative}}
#' \item{conf.int}{Confidence interval for \eqn{Pr(}domination number\eqn{=2)} at the given level \code{conf.level} and
#' depends on the type of \code{alternative}.}
#' \item{estimate}{A \code{vector} with two entries: first is is the estimate of the parameter, i.e.,
#' \eqn{Pr(}domination number\eqn{=2)} and second is the domination number}
#' \item{null.value}{Hypothesized value for the parameter, i.e., the null value for \eqn{Pr(}domination number\eqn{=2)}}
#' \item{alternative}{Type of the alternative hypothesis in the test, one of \code{"two.sided"}, \code{"less"}, \code{"greater"}}
#' \item{method}{Description of the hypothesis test}
#' \item{data.name}{Name of the data set}
#'
#' @seealso \code{\link{TSDomPEBin}} and \code{\link{PEdom1D}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' a<-0; b<-10; int<-c(a,b)
#' c<-.4
#'
#' r<-1/max(c,1-c)
#'
#' #nx is number of X points (target) and ny is number of Y points (nontarget)
#' nx<-20; ny<-4;  #try also nx<-40; ny<-10 or nx<-1000; ny<-10;
#'
#' set.seed(1)
#' Xp<-runif(nx,a,b)
#' Yp<-runif(ny,a,b)
#' PEdom1D(Xp,Yp,r,c)
#'
#' plotIntervals(Xp,Yp,xlab="",ylab="")
#' plotPEregsMI(Xp,Yp,r,c)
#'
#' TSDomPEBin1D(Xp,Yp,int,c,alt="t")
#' TSDomPEBin1D(Xp,int=int,c=c,alt="t")
#'
#' TSDomPEBin1D(Xp,Yp,int,c,alt="l")
#' TSDomPEBin1D(Xp,Yp,int,c,alt="g")
#' TSDomPEBin1D(Xp,Yp,int,c,end=TRUE)
#' TSDomPEBin1D(Xp,Yp,int,c=.25)
#'
#' @export TSDomPEBin1D
TSDomPEBin1D <- function(Xp,Yp=NULL,int,c=.5,asy.bin=FALSE,end.int.cor=FALSE,
                       alternative=c("two.sided", "less", "greater"),conf.level = 0.95)
{
  alternative <-match.arg(alternative)
  if (length(alternative) > 1 || is.na(alternative))
    stop("alternative must be one \"greater\", \"less\", \"two.sided\"")

  dname <-deparse(substitute(Xp))

  if ((!is.point(Xp,length(Xp)) || !is.point(Yp,length(Yp))) && !is.null(Yp))
  {stop('Xp and Yp must be 1D vectors of numerical entries, if Yp is provided')}

  if (!is.point(int) || int[2]<=int[1])
  {stop('int must be an interval as (a,b) with a<b')}

  if (!is.point(c,1) || c<=0 || c>=1)
  {stop('c must be a scalar in (0,1)')}

  rstar<-1/max(c,1-c)  #r value for the non-degenerate asymptotic distribution

  nx<-length(Xp)  #number of Xp points
  if (is.null(Yp))
  { nint<-round(sqrt(nx),0)
  Yp<-(int[2]-int[1])*(0:nint)/nint #Y points (ends of the subintervals)
  if (asy.bin==T)
  {p<-PG2PE1D.asy(c)
  } else
  {
    Enx<-nx/nint
    p<-PG2PE1D(rstar,c,Enx)  #p: prob of success; on average n/nint X points fall on each interval
  }
  } else
  {
    p<-PG2PE1D.asy(c)  #asymptotic probability of success, used when Yp is provided or when asy.bin=F
  }

  if (length(Yp)<2)
  {stop('Yp must be of length >2')}

  if (!missing(conf.level))
    if (length(conf.level) != 1 || is.na(conf.level) || conf.level < 0 || conf.level > 1)
      stop("conf.level must be a number between 0 and 1")

  ny<-length(Yp)  #number of Yp points
  nint<-ny-1
  Ys<-sort(Yp)  #sorted Yp points (ends of the subintervals)

  Gam.all<-PEdom1D(Xp,Yp,rstar,c)$d #domination number (with the end intervals counted)
  Gam<-Gam.all-sum(sum(Xp<Ys[1])>0)-sum(sum(Xp>Ys[ny])>0)  #removing the domination number in the end intervals
  estimate2<-Gam
  Bm<-Gam-nint; #the binomial test statistic

  if (Bm<0)
    warning('The adjusted binomial test statistic is negative! So, 0 is taken as its value')

  Bm<-max(0,Bm)  # to avoid negative Bm values

  method <-c("Exact Binomial Test for the Domination Number for Testing uniformity of 1D Data \n without End Interval Correction")

  if (end.int.cor==T)  #the part for the end interval correction
  {
    out.int<-sum(Xp<Ys[1])+sum(Xp>Ys[ny])
    prop.out<-out.int/nx #observed proportion of points in the end intervals
    exp.prop.out<-2/(ny+1)  #expected proportion of points in the end intervals

    Bm<-Bm*(1-(prop.out-exp.prop.out))
    method <-c("Exact Binomial Test for the Domination Number for Testing uniformity of 1D Data \n with End Interval Correction")
  }

  x<-round(Bm)
  pval <-switch(alternative, less = pbinom(x, nint, p),
                greater = pbinom(x - 1, nint, p, lower.tail = FALSE),
                two.sided = {if (p == 0) (x == 0) else if (p == 1) (x == nint)
                  else { relErr <-1 + 1e-07
                  d <-dbinom(x, nint, p)
                  m <-nint * p
                  if (x == m) 1 else if (x < m)
                  {i <-seq.int(from = ceiling(m), to = nint)
                  y <-sum(dbinom(i, nint, p) <= d * relErr)
                  pbinom(x, nint, p) + pbinom(nint - y, nint, p, lower.tail = FALSE)
                  } else {
                    i <-seq.int(from = 0, to = floor(m))
                    y <-sum(dbinom(i, nint, p) <= d * relErr)
                    pbinom(y - 1, nint, p) + pbinom(x - 1, nint, p, lower.tail = FALSE)
                  }
                  }
                })

  p.L <- function(x, alpha) {
    if (x == 0)
      0
    else qbeta(alpha, x, nint - x + 1)
  }
  p.U <- function(x, alpha) {
    if (x == nint)
      1
    else qbeta(1 - alpha, x + 1, nint - x)
  }
  cint <-switch(alternative, less = c(0, p.U(x, 1 - conf.level)),
                greater = c(p.L(x, 1 - conf.level), 1), two.sided = {
                  alpha <-(1 - conf.level)/2
                  c(p.L(x, alpha), p.U(x, alpha))
                })
  attr(cint, "conf.level") <-conf.level

  estimate1 <-x/nint
  names(x) <-"domination number - number of subintervals"
  names(nint) <-"number of subintervals based on Yp"
  names(p) <-"Pr(Domination Number=2)"
  names(estimate1) <-c(" Pr(domination number = 2)")
  names(estimate2) <-c("|| domination number")
  structure(
    list(statistic = x,
         p.value = pval,
         conf.int = cint,
         estimate = c(estimate1,estimate2),
         null.value = p,
         alternative = alternative,
         method = method,
         data.name = dname
    ),
    class = "htest")

} #end of the function
#'

################################################
#Functions for NPE in R^2 and R^3
################################################
#Contains the functions used in PCD calculations, such as generation of data in a given a triangle and
#tetrahedron estimation of gamma, arc density etc.

#################################################################

#' @title The vertices of the Proportional Edge (PE) Proximity Region in a basic triangle
#'
#' @description Returns the vertices of the PE proximity region (which is itself a triangle) for a point in the
#' basic triangle \eqn{T_b=T((0,0),(1,0),(c_1,c_2))=}\code{(rv=1,rv=2,rv=3)}.
#'
#' PE proximity region is defined with respect
#' to the basic triangle \eqn{T_b} with expansion parameter \eqn{r \ge 1} and vertex regions based on center \eqn{M=(m_1,m_2)} in
#' Cartesian coordinates or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the interior of the basic
#' triangle \eqn{T_b} or based on the circumcenter of \eqn{T_b}; default is \eqn{M=(1,1,1)} i.e., the center of mass of \eqn{T_b}.
#'
#' Vertex regions are labeled as \eqn{1,2,3} rowwise for the vertices
#' of the triangle \eqn{T_b}. \code{rv} is the index of the vertex region \code{pt} resides, with default=\code{NULL}.
#' If \code{pt} is outside of \code{tri}, it returns \code{NULL} for the proximity region.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:mcap2012;textual}{pcds}).
#'
#' @param pt A 2D point whose PE proximity region is to be computed.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param c1,c2 Positive real numbers representing the top vertex in basic triangle \eqn{T_b=T((0,0),(1,0),(c_1,c_2))},
#' \eqn{c_1} must be in \eqn{[0,1/2]}, \eqn{c_2>0} and \eqn{(1-c_1)^2+c_2^2 \le 1}.
#' @param M A 2D point in Cartesian coordinates or a 3D point in barycentric coordinates
#' which serves as a center in the interior of the basic triangle \eqn{T_b} or the circumcenter of \eqn{T_b};
#' default is \eqn{M=(1,1,1)} i.e., the center of mass of \eqn{T_b}.
#' @param rv Index of the \code{M}-vertex region containing the point \code{pt}, either \code{1,2,3} or \code{NULL}
#' (default is \code{NULL}).
#'
#' @return Vertices of the triangular region which constitutes the PE proximity region with expansion parameter
#' r and center \code{M} for a point \code{pt}
#'
#' @seealso \code{\link{NPEtri}}, \code{\link{NAStri}}, \code{\link{NCStri}} and \code{\link{IndNPEbastri}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' c1<-.4; c2<-.6
#' A<-c(0,0); B<-c(1,0); C<-c(c1,c2);
#' Tb<-rbind(A,B,C);
#'
#' M<-as.numeric(runif.bastri(1,c1,c2)$g)  #try also M<-c(.6,.2)
#'
#' r<-2
#'
#' P1<-as.numeric(runif.bastri(1,c1,c2)$g)  #try also P1<-c(.4,.2)
#' NPEbastri(P1,r,c1,c2,M)
#'
#' #or try
#' Rv<-rv.bastri.cent(P1,c1,c2,M)$rv
#' NPEbastri(P1,r,c1,c2,M,Rv)
#'
#' P2<-c(1.8,.5)
#' NPEbastri(P2,r,c1,c2,M)
#'
#' P3<-c(1.7,.6)
#' NPEbastri(P3,r,c1,c2,M)
#'
#' r<-2
#' P1<-c(1.4,1.2)
#' P2<-c(1.5,1.26)
#' NPEbastri(P1,r,c1,c2,M) #gives an error if M=c(1.3,1.3)
#' #since center is not the circumcenter or not in the interior of the triangle
#' NPEbastri(P2,r,c1,c2,M) #gives an error  due to same reason as above
#'
#' @export NPEbastri
NPEbastri <- function(pt,r,c1,c2,M=c(1,1,1),rv=NULL)
{
  if (!is.point(pt) )
  {stop('must be a numeric 2D point')}

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if (c1<0 || c1>1/2 || c2<=0 || (1-c1)^2+c2^2 >1)
  {stop('c1 must be in [0,1/2], c2>0 and (1-c1)^2+c2^2 <=1')}

  if (!is.point(M) && !is.point(M,3) )
  {stop('M must be a numeric 2D point for Cartesian coordinates or 3D point for barycentric coordinates')}

  A<-c(0,0); B<-c(1,0); C<-c(c1,c2); Tb<-rbind(A,B,C)  #basic triangle

  if (dimension(M)==3)
  {
    M<-bary2cart(M,Tb)
  }

  if (isTRUE(all.equal(M,circ.cent.tri(Tb)))==F & in.triangle(M,Tb,boundary=FALSE)$in.tri==F)
  {stop('center is not the circumcenter or not in the interior of the triangle')}

  if (!in.triangle(pt,Tb,boundary=TRUE)$in.tri)
  {reg<-NULL; return(reg); stop}

  if (is.null(rv))
  { rv<-ifelse(isTRUE(all.equal(M,circ.cent.tri(Tb)))==T,rv.triCC(pt,Tb)$rv,rv.tri.cent(pt,Tb,M)$rv)  #vertex region for pt
  } else
  {  if (!is.numeric(rv) || sum(rv==c(1,2,3))!=1)
  {stop('vertex index, rv, must be 1, 2 or 3')}}


  x1<-pt[1]; y1<-pt[2]
  if (rv==1)
  {
    A1<-c(-y1*c1*r/c2+x1*r+y1*r/c2, 0)
    A2<-c(-c1^2*y1*r/c2+c1*x1*r+y1*c1*r/c2, -c1*r*y1+c2*r*x1+r*y1)
    reg<-rbind(A,A1,A2)
  } else if (rv==2)
  {
    B1<-c(-r+c1^2*y1*r/c2-c1*x1*r+c1*r-y1*c1*r/c2+x1*r+1, c1*r*y1-c2*r*x1+c2*r)
    B2<-c(-r-y1*c1*r/c2+x1*r+1, 0)
    reg<-rbind(B,B1,B2)
  } else
  {
    C1<-c(-c1*r+y1*c1*r/c2+c1, -c2*r+r*y1+c2)
    C2<-c(y1*c1*r/c2+r-y1*r/c2-c1*r+c1, -c2*r+r*y1+c2)
    reg<-rbind(C,C1,C2)
  }
  if (abs(area.polygon(reg))>abs(area.polygon(Tb)))
  {reg<-Tb}
  row.names(reg)<-c()
  reg
} #end of the function
#'

#################################################################

#' @title The indicator for the presence of an arc from a point to another for Proportional Edge Proximity Catch
#' Digraphs (PE-PCDs) - basic triangle case
#'
#' @description Returns \eqn{I(}\code{pt2} is in \eqn{N_{PE}(pt1,r))} for points \code{pt1} and \code{pt2}, that is, returns 1 if \code{pt2} is in \eqn{N_{PE}(pt1,r)},
#' returns 0 otherwise, where \eqn{N_{PE}(x,r)} is the PE proximity region for point \eqn{x} with expansion parameter \eqn{r \ge 1}.
#'
#' PE proximity region is defined with respect to the basic triangle \eqn{T_b=T((0,0),(1,0),(c_1,c_2))}
#' where \eqn{c_1} is in \eqn{[0,1/2]}, \eqn{c_2>0} and \eqn{(1-c_1)^2+c_2^2 \le 1}.
#'
#' Vertex regions are based on the center, \eqn{M=(m_1,m_2)} in Cartesian coordinates or \eqn{M=(\alpha,\beta,\gamma)} in
#' barycentric coordinates in the interior of the basic triangle \eqn{T_b} or based on circumcenter of \eqn{T_b};
#' default is \eqn{M=(1,1,1)} i.e., the center of mass of \eqn{T_b}.
#' \code{rv} is the index of the vertex region \code{pt1} resides, with default=\code{NULL}.
#'
#' If \code{pt1} and \code{pt2} are distinct and either of them are outside \eqn{T_b}, it returns 0,
#' but if they are identical, then it returns 1 regardless of their locations (i.e., it allows loops).
#'
#' Any given triangle can be mapped to the basic triangle
#' by a combination of rigid body motions (i.e., translation, rotation and reflection) and scaling,
#' preserving uniformity of the points in the original triangle. Hence basic triangle is useful for simulation
#' studies under the uniformity hypothesis.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:arc-density-PE;textual}{pcds}).
#'
#' @param pt1 A 2D point whose PE proximity region is constructed.
#' @param pt2 A 2D point. The function determines whether \code{pt2} is inside the PE proximity region of
#' \code{pt1} or not.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region; must be \eqn{\ge 1}
#' @param c1,c2 Positive real numbers which constitute the vertex of the basic triangle
#' adjacent to the shorter edges; \eqn{c_1} must be in \eqn{[0,1/2]}, \eqn{c_2>0} and \eqn{(1-c_1)^2+c_2^2 \le 1}.
#' @param M A 2D point in Cartesian coordinates or a 3D point in barycentric coordinates
#' which serves as a center in the interior of the basic triangle or circumcenter of \eqn{T_b};
#' default is \eqn{M=(1,1,1)} i.e., the center of mass of \eqn{T_b}.
#' @param rv The index of the vertex region in \eqn{T_b} containing the point, either \code{1,2,3} or \code{NULL}
#' (default is \code{NULL}).
#'
#' @return \eqn{I(}\code{pt2} is in \eqn{N_{PE}(pt1,r))} for points \code{pt1} and \code{pt2}, that is, returns 1 if \code{pt2} is in \eqn{N_{PE}(pt1,r)},
#' returns 0 otherwise
#'
#' @seealso \code{\link{IndNPEtri}} and \code{\link{IndNPETe}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' c1<-.4; c2<-.6
#' A<-c(0,0); B<-c(1,0); C<-c(c1,c2);
#' Tb<-rbind(A,B,C);
#'
#' M<-as.numeric(runif.bastri(1,c1,c2)$g)
#'
#' r<-2
#'
#' P1<-as.numeric(runif.bastri(1,c1,c2)$g)
#' P2<-as.numeric(runif.bastri(1,c1,c2)$g)
#' IndNPEbastri(P1,P2,r,c1,c2,M)
#'
#' P1<-c(.4,.2)
#' P2<-c(.5,.26)
#' IndNPEbastri(P1,P2,r,c1,c2,M)
#' IndNPEbastri(P2,P1,r,c1,c2,M)
#' IndNPEbastri(P1,P1,r,c1,c2,M)
#'
#' #or try
#' Rv<-rv.bastri.cent(P1,c1,c2,M)$rv
#' IndNPEbastri(P1,P2,r,c1,c2,M,Rv)
#'
#' P1<-c(1.4,.2)
#' P2<-c(.5,.26)
#' IndNPEbastri(P1,P2,r,c1,c2,M)
#' IndNPEbastri(P1,P1,r,c1,c2,M)
#'
#' #or try
#' Rv<-rv.bastri.cent(P1,c1,c2,M)$rv
#' IndNPEbastri(P1,P2,r,c1,c2,M,Rv)
#' IndNPEbastri(P1,P1,r,c1,c2,M,Rv)
#'
#' P1<-c(.4,.2)
#' P2<-c(1.5,.26)
#' IndNPEbastri(P1,P2,r,c1,c2,M)
#'
#' P1<-c(.4,.2)
#' P2<-c(.5,.26)
#' IndNPEbastri(P1,P2,r,c1,c2,M)
#'
#' @export IndNPEbastri
IndNPEbastri <- function(pt1,pt2,r,c1,c2,M=c(1,1,1),rv=NULL)
{
  if (!is.point(pt1) || !is.point(pt2))
  {stop('pt1 and pt2 must be numeric 2D points')}

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if (!is.point(c1,1) || !is.point(c2,1))
  {stop('c1 and c2 must be scalars')}

  if (c1<0 || c1>1/2 || c2<=0 || (1-c1)^2+c2^2 >1)
  {stop('c1 must be in [0,1/2], c2>0 and (1-c1)^2+c2^2 <=1')}

  if (!is.point(M) && !is.point(M,3) )
  {stop('M must be a numeric 2D point for Cartesian coordinates or 3D point for barycentric coordinates')}

  y1<-c(0,0); y2<-c(1,0); y3<-c(c1,c2); Tb<-rbind(y1,y2,y3)

  if (dimension(M)==3)
  {
    M<-bary2cart(M,Tb)
  }

  if (in.triangle(M,Tb,boundary=FALSE)$in.tri==F)
  {stop('center is not in the interior of the triangle')}

  if (identical(pt1,pt2))
  {arc<-1; return(arc); stop}

  if (!in.triangle(pt1,Tb,boundary=TRUE)$in.tri || !in.triangle(pt2,Tb,boundary=TRUE)$in.tri)
  {arc<-0; return(arc); stop}

  if (is.null(rv))
  { rv<-ifelse(isTRUE(all.equal(M,circ.cent.tri(Tb)))==T,rv.triCC(pt1,Tb)$rv,rv.tri.cent(pt1,Tb,M)$rv)  #vertex region for pt1
  } else
  {  if (!is.numeric(rv) || sum(rv==c(1,2,3))!=1)
  {stop('vertex index, rv, must be 1, 2 or 3')}}

  X1<-pt1[1]; Y1<-pt1[2];
  X2<-pt2[1]; Y2<-pt2[2];
  arc<-0;
  if (rv==1)
  {
    x1n<-X1*r; y1n<-Y1*r;
    if ( Y2 < paraline(c(x1n,y1n),y2,y3,X2)$y ) {arc <-1}
  } else {
    if (rv==2)
    {
      x1n<-1+(X1-1)*r; y1n<-Y1*r;
      if ( Y2 < paraline(c(x1n,y1n),y1,y3,X2)$y ) {arc <-1}
    } else {
      y1n<-y3[2]+(Y1-y3[2])*r;
      if ( Y2 > y1n ) {arc<-1}
    }}
  arc
} #end of the function
#'

#################################################################

#' @title The index of the \eqn{CM}-vertex region in the standard equilateral triangle that contains a given point
#'
#' @description Returns the index of the vertex whose region contains point \code{p} in standard equilateral triangle
#' \eqn{T_e=T((0,0),(1,0),(1/2,\sqrt{3}/2))} with vertex regions are constructed with center of mass CM
#' (see the plots in the example for illustrations).
#'
#' The vertices of triangle, \eqn{T_e}, are labeled as \eqn{1,2,3}
#' according to the row number the vertex is recorded in \eqn{T_e}. If the point, \code{p}, is not inside \eqn{T_e}, then the
#' function yields \code{NA} as output. The corresponding vertex region is the polygon with the vertex, \eqn{CM}, and
#' midpoints of the edges adjacent to the vertex.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:mcap2012;textual}{pcds}).
#'
#' @param pt A 2D point for which \eqn{CM}-vertex region it resides in is to be determined in the
#' standard equilateral triangle \eqn{T_e}.
#'
#' @return A \code{list} with two elements
#' \item{rv}{Index of the vertex whose region contains point, \code{p}.}
#' \item{tri}{The vertices of the triangle, \eqn{T_e}, where row number corresponds to the vertex index in \code{rv}
#' with row \eqn{1=(0,0)}, row \eqn{2=(1,0)}, and row \eqn{3=(1/2,\sqrt{3}/2)}.}
#'
#' @seealso \code{\link{rv.bastriCM}}, \code{\link{rv.tri.cent}}, \code{\link{rv.triCC}},
#' \code{\link{rv.bastriCC}}, \code{\link{rv.triCM}}, and \code{\link{rv.bastri.cent}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(0,0); B<-c(1,0); C<-c(1/2,sqrt(3)/2);
#' Te<-rbind(A,B,C)
#'
#' n<-10  #try also n<-20
#'
#' set.seed(1)
#' dat<-runifTe(n)$gen.points
#'
#' rvTeCM(dat[1,])
#' rvTeCM(c(.7,.2))
#' rvTeCM(c(0,1))
#'
#' Rv<-vector()
#' for (i in 1:n)
#'   Rv<-c(Rv,rvTeCM(dat[i,])$rv)
#' Rv
#'
#' CM<-(A+B+C)/3
#' D1<-(B+C)/2; D2<-(A+C)/2; D3<-(A+B)/2;
#' Ds<-rbind(D1,D2,D3)
#'
#' Xlim<-range(Te[,1],dat[,1])
#' Ylim<-range(Te[,2],dat[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(Te,asp=1,pch=".",xlab="",ylab="",axes=TRUE,xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Te)
#' points(dat,pch=".",col=1)
#' L<-matrix(rep(CM,3),ncol=2,byrow=TRUE); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#'
#' txt<-rbind(Te,CM)
#' xc<-txt[,1]+c(-.02,.03,.02,0)
#' yc<-txt[,2]+c(.02,.02,.03,.05)
#' txt.str<-c("A","B","C","CM")
#' text(xc,yc,txt.str)
#'
#' text(dat,labels=factor(Rv))
#'
#' rvTeCM(c(.7,.2))
#'
#' @export rvTeCM
rvTeCM <- function(pt)
{
  if (!is.point(pt))
  {stop('the argument must be a numeric 2D point')}

  y1<-c(0,0); y2<-c(1,0); y3<-c(1/2,sqrt(3)/2);
  tri<-rbind(y1,y2,y3);
  if (in.triangle(pt,tri,boundary = TRUE)$in.tri==F)
  {rv<-NA
  } else
  {
    mdt<-1; #maximum distance from a point inside Te to a vertex
    for (i in 1:3)
    {
      d1<-Dist(pt,tri[i,]);
      if (d1<mdt)
      {mdt<-d1; rv<-i }
    }
  }
  row.names(tri)<-c("vertex 1","vertex 2","vertex 3")  #vertex labeling

  list(rv=rv, #relative vertex
       tri=tri #vertex labeling
  )
} #end of the function
#'

#################################################################

#' @title The index of the vertex region in the standard equilateral triangle that contains a given point
#'
#' @description Returns the index of the vertex whose region contains point \code{pt} in standard equilateral triangle
#' \eqn{T_e=T((0,0),(1,0),(1/2,\sqrt{3}/2))} with vertex regions are constructed with center \eqn{M=(m_1,m_2)}
#' in Cartesian coordinates or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the interior of \eqn{T_e}.
#' (see the plots in the example for illustrations).
#'
#' The vertices of triangle, \eqn{T_e}, are labeled as \eqn{1,2,3}
#' according to the row number the vertex is recorded in \eqn{T_e}. If the point, \code{pt}, is not inside \eqn{T_e}, then the
#' function yields \code{NA} as output. The corresponding vertex region is the polygon with the vertex, \code{M}, and
#' projections from \code{M} to the edges on the lines joining vertices and \code{M}.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:mcap2012;textual}{pcds}).
#'
#' @param pt A 2D point for which \code{M}-vertex region it resides in is to be determined in the
#' standard equilateral triangle \eqn{T_e}.
#' @param M A 2D point in Cartesian coordinates or a 3D point in barycentric coordinates
#' which serves as a center in the interior of the standard equilateral triangle \eqn{T_e}.
#'
#' @return A \code{list} with two elements
#' \item{rv}{Index of the vertex whose region contains point, \code{pt}.}
#' \item{tri}{The vertices of the triangle, \eqn{T_e}, where row number corresponds to the vertex index in \code{rv}
#' with row \eqn{1=(0,0)}, row \eqn{2=(1,0)}, and row \eqn{3=(1/2,\sqrt{3}/2)}.}
#'
#' @seealso \code{\link{rvTeCM}}, \code{\link{rv.tri.cent}}, \code{\link{rv.triCC}},
#' \code{\link{rv.bastriCC}}, \code{\link{rv.triCM}}, and \code{\link{rv.bastri.cent}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(0,0); B<-c(1,0); C<-c(1/2,sqrt(3)/2);
#' Te<-rbind(A,B,C)
#' n<-10  #try also n<-20
#'
#' set.seed(1)
#' dat<-runifTe(n)$gen.points
#'
#' M<-as.numeric(runifTe(1)$g)  #try also M<-c(.6,.2)
#'
#' rvTe.cent(dat[1,],M)
#' rvTe.cent(c(.7,.2),M)
#' rvTe.cent(c(0,1),M)
#'
#' Rv<-vector()
#' for (i in 1:n)
#'   Rv<-c(Rv,rvTe.cent(dat[i,],M)$rv)
#' Rv
#'
#' Ds<-cp2e.tri(Te,M)
#'
#' Xlim<-range(Te[,1],dat[,1])
#' Ylim<-range(Te[,2],dat[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' if (dimension(M)==3) {M<-bary2cart(M,Te)}
#' #need to run this when M is given in barycentric coordinates
#'
#' plot(Te,asp=1,pch=".",xlab="",ylab="",axes=TRUE,
#' xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Te)
#' points(dat,pch=".",col=1)
#' L<-rbind(M,M,M); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#'
#' txt<-rbind(Te,M)
#' xc<-txt[,1]+c(-.02,.03,.02,0)
#' yc<-txt[,2]+c(.02,.02,.03,.05)
#' txt.str<-c("A","B","C","M")
#' text(xc,yc,txt.str)
#'
#' text(dat,labels=factor(Rv))
#'
#' rvTe.cent(c(.7,.2),M)
#'
#' @export rvTe.cent
rvTe.cent <- function(pt,M)
{
  if (!is.point(pt))
  {stop('the argument must be a numeric 2D point')}

  y1<-c(0,0); y2<-c(1,0); y3<-c(1/2,sqrt(3)/2); Te<-rbind(y1,y2,y3);

  if (!is.point(M) && !is.point(M,3) )
  {stop('M must be a numeric 2D point for Cartesian coordinates or 3D point for barycentric coordinates')}

  if (dimension(M)==3)
  {
    M<-bary2cart(M,Te)
  }

  if (in.triangle(pt,Te)$in.tri==F)
  {rv<-NA
  } else
  {
    if (in.triangle(M,Te,boundary=FALSE)$in.tri==F)
    {stop('center is not in the interior of the triangle')}

    Ds<-cp2e.tri(Te,M)
    D1<-Ds[1,]; D2<-Ds[2,]; D3<-Ds[3,];

    if (in.triangle(pt,rbind(y1,D3,M),boundary=TRUE)$in.tri | in.triangle(pt,rbind(y1,M,D2),boundary=TRUE)$in.tri)
    {rv<-1}
    else
    {
      if (in.triangle(pt,rbind(D3,y2,M),boundary=TRUE)$in.tri | in.triangle(pt,rbind(y2,D1,M),boundary=TRUE)$in.tri)
      {rv<-2}
      else
      {rv<-3}
    }
  }
  row.names(Te)<-c("vertex 1","vertex 2","vertex 3")  #vertex labeling

  list(rv=rv, #relative vertex
       tri=Te #vertex labeling
  )
} #end of the function
#'

#################################################################

#' @title The closest points among a data set in the vertex regions to the corresponding edges in a basic triangle
#'
#' @description
#' An object of class \code{"Extrema"}.
#' Returns the closest data points among the data set, \code{Dt}, to edge \eqn{i} in \code{M}-vertex region \eqn{i} for \eqn{i=1,2,3}
#' in the basic triangle \eqn{T_b=T(A=(0,0),B=(1,0),C=(c_1,c_2))} where \eqn{c_1} is in \eqn{[0,1/2]}, \eqn{c_2>0} and \eqn{(1-c_1)^2+c_2^2 \le 1}.
#' Vertex labels are \eqn{A=1}, \eqn{B=2}, and \eqn{C=3}, and corresponding edge labels are \eqn{BC=1}, \eqn{AC=2}, and \eqn{AB=3}.
#'
#' Vertex regions are based on center \eqn{M=(m_1,m_2)} in Cartesian coordinates or
#' \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the interior of the basic triangle \eqn{T_b}
#' or based on the circumcenter of \eqn{T_b}.
#'
#' Any given triangle can be mapped to the basic triangle by a combination of rigid body motions
#' (i.e., translation, rotation and reflection) and scaling, preserving uniformity of the points in the
#' original triangle. Hence basic triangle is useful for simulation
#' studies under the uniformity hypothesis.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:dom-num-NPE-Spat2011;textual}{pcds}).
#'
#' @param Dt A set of 2D points representing the set of data points.
#' @param c1,c2 Positive real numbers which constitute the vertex of the basic triangle
#' adjacent to the shorter edges; \eqn{c_1} must be in \eqn{[0,1/2]}, \eqn{c_2>0} and \eqn{(1-c_1)^2+c_2^2 \le 1}.
#' @param M A 2D point in Cartesian coordinates or a 3D point in barycentric coordinates
#' which serves as a center in the interior of the basic triangle \eqn{T_b} or the circumcenter of \eqn{T_b}.
#'
#' @return A \code{list} with the elements
#' \item{txt1}{Vertex labels are \eqn{A=1}, \eqn{B=2}, and \eqn{C=3} (corresponds to row number in Extremum Points).}
#' \item{txt2}{A short description of the distances as \code{"Distances to Edges in the Respective \eqn{M}-Vertex Regions"}.}
#' \item{type}{Type of the extrema points}
#' \item{desc}{A short description of the extrema points}
#' \item{mtitle}{The \code{"main"} title for the plot of the extrema}
#' \item{ext}{The extrema points, here, closest points to edges in the corresponding vertex region.}
#' \item{X}{The input data, \code{Dt}, can be a \code{matrix} or \code{data frame}}
#' \item{num.points}{The number of data points, i.e., size of \code{Dt}}
#' \item{supp}{Support of the data points, here, it is \eqn{T_b}.}
#' \item{cent}{The center point used for construction of vertex regions}
#' \item{ncent}{Name of the center, \code{cent}, it is \code{"M"} or \code{"CC"} for this function}
#' \item{regions}{Vertex regions inside the triangle, \eqn{T_b}.}
#' \item{region.names}{Names of the vertex regions as \code{"vr=1"}, \code{"vr=2"}, and \code{"vr=3"}}
#' \item{region.centers}{Centers of mass of the vertex regions inside \eqn{T_b}.}
#' \item{dist2ref}{Distances of closest points in the vertex regions to corresponding edges.}
#'
#' @seealso \code{\link{cl2eVRCM}}, \code{\link{cl2eVRcent}}, and \code{\link{cl2edgesTe}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' c1<-.4;  c2<-.6
#' A<-c(0,0); B<-c(1,0); C<-c(c1,c2);
#' Tb<-rbind(A,B,C);
#'
#' set.seed(1)
#' n<-20
#' dat<-runif.bastri(n,c1,c2)$g
#'
#' M<-as.numeric(runif.bastri(1,c1,c2)$g)  #try also M<-c(.6,.3)
#'
#' Ext<-cl2eTbVRcent(dat,c1,c2,M)
#' Ext
#' summary(Ext)
#' plot(Ext)
#'
#' cl2eTbVRcent(dat[1,],c1,c2,M)
#' cl2eTbVRcent(c(1,2),c1,c2,M)
#'
#' cl2e<-cl2eTbVRcent(dat,c1,c2,M)
#' cl2e
#'
#' Ds<-cp2e.bastri(c1,c2,M)
#'
#' Xlim<-range(Tb[,1],dat[,1])
#' Ylim<-range(Tb[,2],dat[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(Tb,pch=".",xlab="",ylab="",axes=TRUE,xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tb)
#' points(dat,pch=1,col=1)
#' L<-rbind(M,M,M); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#' points(cl2e$ext,pch=3,col=2)
#'
#' xc<-Tb[,1]+c(-.02,.02,0.02)
#' yc<-Tb[,2]+c(.02,.02,.02)
#' txt.str<-c("A","B","C")
#' text(xc,yc,txt.str)
#'
#' txt<-rbind(M,Ds)
#' xc<-txt[,1]+c(-.02,.04,-.03,0)
#' yc<-txt[,2]+c(-.02,.02,.02,-.03)
#' txt.str<-c("M","D1","D2","D3")
#' text(xc,yc,txt.str)
#'
#' cl2eTbVRcent(dat,c1,c2,M)
#'
#' dat.fr<-data.frame(a=dat)
#' cl2eTbVRcent(dat.fr,c1,c2,M)
#'
#' @export
cl2eTbVRcent <- function(Dt,c1,c2,M)
{
  if (!is.numeric(as.matrix(Dt)))
  {stop('Dt must be numeric')}

  if (is.point(Dt))
  { Dt<-matrix(Dt,ncol=2)
  } else
  {Dt<-as.matrix(Dt)
  if (ncol(Dt)!=2 )
  {stop('Dt must be of dimension nx2')}
  }

  if (!is.point(c1,1) || !is.point(c2,1))
  {stop('c1 and c2 must be scalars')}

  if (c1<0 || c1>1/2 || c2<=0 || (1-c1)^2+c2^2 >1)
  {stop('c1 must be in [0,1/2], c2>0 and (1-c1)^2+c2^2 <=1')}

  if (!is.point(M) && !is.point(M,3) )
  {stop('M must be a numeric 2D point for Cartesian coordinates or 3D point for barycentric coordinates')}

  y1<-c(0,0); y2<-c(1,0); y3<-c(c1,c2); Tb<-rbind(y1,y2,y3)

  if (dimension(M)==3)
  {
    M<-bary2cart(M,Tb)
  }

  if (isTRUE(all.equal(M,circ.cent.tri(Tb)))==F & in.triangle(M,Tb,boundary=FALSE)$in.tri==F)
  {stop('center is not the circumcenter or not in the interior of the triangle')}

  if (isTRUE(all.equal(M,circ.cent.tri(Tb)))==T)
  {
    res<-cl2eVRCC(Dt,Tb)
    cent.name<-"CC"
  } else
  {
    cent.name<-"M"
    Ds<-cp2e.bastri(c1,c2,M)
    D1<-Ds[1,]; D2<-Ds[2,]; D3<-Ds[3,];
    L<-rbind(M,M,M); R<-Ds

    if (in.triangle(M,Tb,boundary=FALSE)$in.tri==F)
    {stop('center is not in the interior of the basic triangle')}

    mdt<-rep(1,3); #maximum distance from a point in the basic tri to its vertices (which is larger than distances to its edges)
    U<-matrix(NA,nrow=3,ncol=2);

    Dt<-matrix(Dt,ncol=2)
    n<-nrow(Dt)
    for (i in 1:n)
    {if (in.triangle(Dt[i,],Tb,boundary = TRUE)$in.tri)
    {rv<-rv.bastri.cent(Dt[i,],c1,c2,M)$rv;
    if (rv==1)
    {d1<-dp2l(Dt[i,],y2,y3)$dis;
    if (d1<=mdt[1]) {mdt[1]<-d1; U[1,]<-Dt[i,]};
    } else {
      if (rv==2)
      {d2<-dp2l(Dt[i,],y1,y3)$dis;
      if (d2<=mdt[2]) {mdt[2]<-d2; U[2,]<-Dt[i,]}
      } else {
        d3<-Dt[i,2];
        if (d3<=mdt[3]) {mdt[3]<-d3; U[3,]<-Dt[i,]}
      }}
    }
    }

    row.names(Tb)<-c("A","B","C")  #vertex labeling
    typ<-paste("Closest Points to Edges in the Respective M-Vertex Regions in the Basic Triangle with Vertices A=(0,0), B=(1,0), and C=(",c1,",",c2,")",sep="")
    txt1<-"Vertex labels are A=1, B=2, and C=3 (corresponds to row number in Extremum Points)"
    txt2<-"Distances to Edges in the Respective M-Vertex Regions"
    description<-"Closest Points to Edges in the Respective M-Vertex Regions \n (Row i corresponds to vertex i for i=1,2,3)"
    main.txt<-"Closest Points to Edges in\n Respective M-Vertex Regions"

    Dis<-rbind(ifelse(!is.na(U[1,1]),mdt[1],NA),ifelse(!is.na(U[2,1]),mdt[2],NA),ifelse(!is.na(U[3,1]),mdt[3],NA))
    #distances of the closest points to the edges in the respective vertex regions
    Regs<-list(vr1=rbind(y1,D3,M,D2), #regions inside the triangles
               vr2=rbind(y2,D1,M,D3),
               vr3=rbind(y3,D2,M,D1)
    )
    Reg.Cent<-vector()
    for (i in 1:length(Regs))
    { Reg.Cent<-rbind(Reg.Cent,apply(Regs[[i]],2,mean))}

    Reg.names<-c("vr=1","vr=2","vr=3")  #regions names

    res<-list(
      txt1=txt1, txt2=txt2,
      type=typ, desc=description,
      mtitle=main.txt, #main label in the plot
      ext=U, #closest points to edges in each associated vertex region
      X=Dt, num.points=n, #data points and its size
      ROI=Tb, # region of interest for X points
      cent=M, ncent=cent.name, #center and center name
      regions=Regs, region.names=Reg.names, region.centers=Reg.Cent,
      dist2ref=Dis #distances of closest points in vertex regions to the corresponding edges
    )

    class(res)<-"Extrema"
    res$call <-match.call()
  }
  res
} #end of the function
#'

#################################################################

#' @title The closest points among a data set in the vertex regions to the respective edges in a triangle
#'
#' @description
#' An object of class \code{"Extrema"}.
#' Returns the closest data points among the data set, \code{Dt}, to edge \eqn{i} in \code{M}-vertex region \eqn{i} for \eqn{i=1,2,3}
#' in the triangle \code{tri}\eqn{=T(A,B,C)}. Vertex labels are \eqn{A=1}, \eqn{B=2}, and \eqn{C=3}, and corresponding edge labels are
#' \eqn{BC=1}, \eqn{AC=2}, and \eqn{AB=3}.
#'
#' Vertex regions are based on center \eqn{M=(m_1,m_2)} in Cartesian coordinates or
#' \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the interior of the triangle \code{tri}
#' or based on the circumcenter of \code{tri}.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:dom-num-NPE-Spat2011;textual}{pcds}).
#'
#' @param Dt A set of 2D points representing the set of data points.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param M A 2D point in Cartesian coordinates or a 3D point in barycentric coordinates
#' which serves as a center in the interior of the triangle \code{tri} or the circumcenter of \code{tri}.
#'
#' @return A \code{list} with the elements
#' \item{txt1}{Vertex labels are \eqn{A=1}, \eqn{B=2}, and \eqn{C=3} (corresponds to row number in Extremum Points).}
#' \item{txt2}{A short description of the distances as \code{"Distances to Edges in the Respective \eqn{M}-Vertex Regions"}.}
#' \item{type}{Type of the extrema points}
#' \item{desc}{A short description of the extrema points}
#' \item{mtitle}{The \code{"main"} title for the plot of the extrema}
#' \item{ext}{The extrema points, here, closest points to edges in the respective vertex region.}
#' \item{X}{The input data, \code{Dt}, can be a \code{matrix} or \code{data frame}}
#' \item{num.points}{The number of data points, i.e., size of \code{Dt}}
#' \item{supp}{Support of the data points, here, it is \code{tri}}
#' \item{cent}{The center point used for construction of vertex regions}
#' \item{ncent}{Name of the center, \code{cent}, it is \code{"M"} or \code{"CC"} for this function}
#' \item{regions}{Vertex regions inside the triangle, \code{tri}, provided as a list}
#' \item{region.names}{Names of the vertex regions as \code{"vr=1"}, \code{"vr=2"}, and \code{"vr=3"}}
#' \item{region.centers}{Centers of mass of the vertex regions inside \code{tri}}
#' \item{dist2ref}{Distances of closest points in the \code{M}-vertex regions to corresponding edges.}
#'
#' @seealso \code{\link{cl2eTbVRcent}}, \code{\link{cl2eVRCM}}, and \code{\link{cl2edgesTe}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#'
#' Tr<-rbind(A,B,C);
#' n<-10  #try also n<-20
#'
#' set.seed(1)
#' dat<-runif.tri(n,Tr)$g
#'
#' M<-as.numeric(runif.tri(1,Tr)$g)  #try also M<-c(1.6,1.0)
#'
#' Ext<-cl2eVRcent(dat,Tr,M)
#' Ext
#' summary(Ext)
#' plot(Ext)
#'
#' cl2eVRcent(dat[1,],Tr,M)
#' cl2e<-cl2eVRcent(dat,Tr,M)
#' cl2e
#'
#' Ds<-cp2e.tri(Tr,M)
#'
#' Xlim<-range(Tr[,1],dat[,1])
#' Ylim<-range(Tr[,2],dat[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' if (dimension(M)==3) {M<-bary2cart(M,Tr)}
#' #need to run this when M is given in barycentric coordinates
#'
#' plot(Tr,pch=".",xlab="",ylab="",axes=TRUE,
#' xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tr)
#' points(dat,pch=1,col=1)
#' L<-rbind(M,M,M); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#' points(cl2e$ext,pch=3,col=2)
#'
#' xc<-Tr[,1]+c(-.02,.03,.02)
#' yc<-Tr[,2]+c(.02,.02,.04)
#' txt.str<-c("A","B","C")
#' text(xc,yc,txt.str)
#'
#' txt<-rbind(M,Ds)
#' xc<-txt[,1]+c(-.02,.05,-.02,-.01)
#' yc<-txt[,2]+c(-.03,.02,.08,-.07)
#' txt.str<-c("M","D1","D2","D3")
#' text(xc,yc,txt.str)
#'
#' cl2eVRcent(dat,Tr,M)
#'
#' dat.fr<-data.frame(a=dat)
#' cl2eVRcent(dat.fr,Tr,M)
#'
#' dat.fr<-data.frame(a=Tr)
#' cl2eVRcent(dat,dat.fr,M)
#'
#' @export cl2eVRcent
cl2eVRcent <- function(Dt,tri,M)
{
  if (!is.numeric(as.matrix(Dt)))
  {stop('Dt must be numeric')}

  if (is.point(Dt))
  { Dt<-matrix(Dt,ncol=2)
  } else
  {Dt<-as.matrix(Dt)
  if (ncol(Dt)!=2 )
  {stop('Dt must be of dimension nx2')}
  }

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  if (!is.point(M) && !is.point(M,3) )
  {stop('M must be a numeric 2D point for Cartesian coordinates or 3D point for barycentric coordinates')}

  if (dimension(M)==3)
  {
    M<-bary2cart(M,tri)
  }

  if (isTRUE(all.equal(M,circ.cent.tri(tri)))==F & in.triangle(M,tri,boundary=FALSE)$in.tri==F)
  {stop('center is not the circumcenter or not in the interior of the triangle')}

  y1<-tri[1,]; y2<-tri[2,]; y3<-tri[3,];

  if (isTRUE(all.equal(M,circ.cent.tri(tri)))==T)
  {
    res<-cl2eVRCC(Dt,tri)
    cent.name<-"CC"
  } else
  {
    cent.name<-"M"
    Ds<-cp2e.tri(tri,M)
    D1<-Ds[1,]; D2<-Ds[2,]; D3<-Ds[3,];
    L<-rbind(M,M,M); R<-Ds

    mdt<-c(dp2l(y1,y2,y3)$dis,dp2l(y2,y1,y3)$dis,dp2l(y3,y1,y2)$dis); #distances from each vertex to the opposite edge
    U<-matrix(NA,nrow=3,ncol=2);

    Dt<-matrix(Dt,ncol=2)
    n<-nrow(Dt);

    for (i in 1:n)
    {if (in.triangle(Dt[i,],tri,boundary = TRUE)$in.tri)
    {
      rv<-rv.tri.cent(Dt[i,],tri,M)$rv
      if (rv==1)
      {d1<-dp2l(Dt[i,],y2,y3)$dis;
      if (d1<=mdt[1]) {mdt[1]<-d1; U[1,]<-Dt[i,]};
      } else {
        if (rv==2)
        {d2<-dp2l(Dt[i,],y1,y3)$dis;
        if (d2<=mdt[2]) {mdt[2]<-d2; U[2,]<-Dt[i,]}
        } else {
          d3<-dp2l(Dt[i,],y1,y2)$dis;
          if (d3<=mdt[3]) {mdt[3]<-d3; U[3,]<-Dt[i,]}
        }}
    }
    }
    row.names(tri)<-c("A","B","C")  #vertex labeling
    typ<-"Closest Points to Edges in the Respective M-Vertex Regions in the Triangle with Vertices A, B, and C"
    txt1<-"Vertex labels are A=1, B=2, and C=3 (corresponds to row number in Extremum Points)"
    txt2<-"Distances to Edges in the Respective M-Vertex Regions"
    description<-"Closest Points to Edges in the Respective M-Vertex Regions \n (Row i corresponds to vertex i for i=1,2,3)  "
    main.txt<-"Closest Points to Edges in\n Respective  M-Vertex Regions"

    Dis<-rbind(ifelse(!is.na(U[1,1]),mdt[1],NA),ifelse(!is.na(U[2,1]),mdt[2],NA),ifelse(!is.na(U[3,1]),mdt[3],NA))
    #distances of the closest points to the edges in corresponding vertex regions
    Regs<-list(vr1=rbind(y1,D3,M,D2), #regions inside the triangles
               vr2=rbind(y2,D1,M,D3),
               vr3=rbind(y3,D2,M,D1)
    )
    Reg.Cent<-vector()
    for (i in 1:length(Regs))
    { Reg.Cent<-rbind(Reg.Cent,apply(Regs[[i]],2,mean))}

    Reg.names<-c("vr=1","vr=2","vr=3")  #regions names

    res<-list(
      txt1=txt1, txt2=txt2,
      type=typ, desc=description,
      mtitle=main.txt, #main label in the plot
      ext=U, #closest points to edges in each associated vertex region
      X=Dt, num.points=n, #data points and its size
      ROI=tri, # region of interest for X points
      cent=M, ncent=cent.name, #center and center name
      regions=Regs, region.names=Reg.names, region.centers=Reg.Cent,
      dist2ref=Dis #distances of closest points in vertex regions to the corresponding edges
    )

    class(res)<-"Extrema"
    res$call <-match.call()
  }
  res
} #end of the function
#'

#################################################################

#' @title An alternative function to the function \code{\link{cl2eVRcent}} which finds the closest points
#' among a data set in the vertex regions to the respective edges in a triangle
#'
#' @description
#' An object of class \code{"Extrema"}.
#' An alternative function to the function \code{\link{cl2eVRcent}}
#'
#' @param dat A set of 2D points representing the set of data points.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param M A 2D point in Cartesian coordinates or a 3D point in barycentric coordinates
#' which serves as a center in the interior of the triangle \code{tri} or the circumcenter of \code{tri}.
#'
#' @return A \code{list} with the elements
#' \item{txt1}{Vertex labels are \eqn{A=1}, \eqn{B=2}, and \eqn{C=3} (corresponds to row number in Extremum Points).}
#' \item{txt2}{A short description of the distances as \code{"Distances to Edges in the Respective \eqn{M}-Vertex Regions"}.}
#' \item{type}{Type of the extrema points}
#' \item{desc}{A short description of the extrema points}
#' \item{mtitle}{The \code{"main"} title for the plot of the extrema}
#' \item{ext}{The extrema points, here, closest points to edges in the respective vertex region.}
#' \item{X}{The input data, \code{Dt}, can be a \code{matrix} or \code{data frame}}
#' \item{num.points}{The number of data points, i.e., size of \code{Dt}}
#' \item{supp}{Support of the data points, here, it is \code{tri}}
#' \item{cent}{The center point used for construction of vertex regions}
#' \item{ncent}{Name of the center, \code{cent}, it is \code{"M"} or \code{"CC"} for this function}
#' \item{regions}{Vertex regions inside the triangle, \code{tri}, provided as a list}
#' \item{region.names}{Names of the vertex regions as \code{"vr=1"}, \code{"vr=2"}, and \code{"vr=3"}}
#' \item{region.centers}{Centers of mass of the vertex regions inside \code{tri}}
#' \item{dist2ref}{Distances of closest points in the \code{M}-vertex regions to corresponding edges.}
#'
#' @seealso \code{\link{cl2eVRcent}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#' M<-c(1.6,1.0)  #try also M<-c(1.3,1.3)
#'
#' n<-20
#' set.seed(1)
#' dat<-runif.tri(n,Tr)$g
#'
#' Ext<-cl2eVRcent.alt(dat,Tr,M)
#' Ext
#' summary(Ext)
#' plot(Ext)
#'
#' cl2eVRcent.alt(dat[1,],Tr,M)
#'
#' cl2e<-cl2eVRcent.alt(dat,Tr,M)
#' cl2e
#'
#' Ds<-cp2e.tri(Tr,M)
#'
#' Xlim<-range(Tr[,1])
#' Ylim<-range(Tr[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(Tr,pch=".",xlab="",ylab="",axes=TRUE,
#' xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tr)
#' points(dat,pch=1,col=1)
#' L<-rbind(M,M,M); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#' points(cl2e$ext,pch=3,col=2)
#'
#' xc<-Tr[,1]+c(-.02,.03,.02)
#' yc<-Tr[,2]+c(.02,.02,.04)
#' txt.str<-c("A","B","C")
#' text(xc,yc,txt.str)
#'
#' txt<-rbind(M,Ds)
#' xc<-txt[,1]+c(.02,.04,-.03,0)
#' yc<-txt[,2]+c(.07,.04,.06,-.08)
#' txt.str<-c("M","D1","D2","D3")
#' text(xc,yc,txt.str)
#'
#' dat.fr<-data.frame(a=dat)
#' cl2eVRcent.alt(dat.fr,Tr,M)
#'
#' dat.fr<-data.frame(a=Tr)
#' cl2eVRcent.alt(dat,dat.fr,M)
#'
#' @export
cl2eVRcent.alt <- function(dat,tri,M)
{
  if (!is.numeric(as.matrix(dat)))
  {stop('dat must be numeric')}

  if (is.point(dat))
  { dat<-matrix(dat,ncol=2)
  } else
  {dat<-as.matrix(dat)
  if (ncol(dat)!=2 )
  {stop('dat must be of dimension nx2')}
  }

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  if (!is.point(M))
  {stop('M must be a numeric 2D point')}

  n<-nrow(dat)
  A<-tri[1,]; B<-tri[2,]; C<-tri[3,]
  Cent<-(A+B+C)/3; Cname<-"CM"
  D1<-(B+C)/2; D2<-(A+C)/2; D3<-(A+B)/2;
  Ds<-rbind(D1,D2,D3)
  L<-rbind(Cent,Cent,Cent); R<-Ds

  clBC<-NA; clAC<-NA; clAB<-NA

  VRdt<-rverts.tri.cent(dat,tri,M)$rv
  dtA<-matrix(dat[VRdt==1,],ncol=2); dtB<-matrix(dat[VRdt==2,],ncol=2); dtC<-matrix(dat[VRdt==3,],ncol=2)
  distA<-distB<-distC<-vector()
  nA<-nrow(dtA); nB<-nrow(dtB); nC<-nrow(dtC);

  if (nA>0)
  {
    for (i in 1:nA)
    {distA<-c(distA,dp2l(dtA[i,],B,C)$dis)};
    clBC<-dtA[distA==min(distA),]
  }

  if (nB>0)
  {
    for (i in 1:nB)
    {distB<-c(distB,dp2l(dtB[i,],A,C)$dis)};
    clAC<-dtB[distB==min(distB),]
  }

  if (nC>0)
  {
    for (i in 1:nC)
    {distC<-c(distC,dp2l(dtC[i,],A,B)$dis)};
    clAB<-dtC[distC==min(distC),]
  }
  ce<-rbind(clBC,clAC,clAB)

  Dis<-c(ifelse(is.numeric(distA),min(distA),NA),ifelse(is.numeric(distB),min(distB),NA),
         ifelse(is.numeric(distC),min(distC),NA))

  row.names(tri)<-c("A","B","C")  #vertex labeling
  txt<-"Edge labels are AB=3, BC=1, and AC=2 (corresponds to row number in ce)"

  row.names(tri)<-c("A","B","C")  #vertex labeling
  row.names(ce)<-c("closest to edge 1:","closest to edge 2:","closest to edge 3:")  #extrema labeling
  typ<-"Closest Points to Edges in the Respective CM-Vertex Regions in the Triangle with vertices A, B, and C"
  txt1<-"Vertex labels are A=1, B=2, and C=3 (corresponds to row number in Extremum Points)"
  txt2<-"Distances to Edges in the Respective CM-Vertex Regions"
  main.txt<-paste("Closest Points to Edges in\n Respective ",Cname,"-Vertex Regions",sep="")

  Dis<-rbind(ifelse(is.numeric(distA),min(distA),NA),ifelse(is.numeric(distB),min(distB),NA),
             ifelse(is.numeric(distC),min(distC),NA))
  #distances of the closest points to the edges in the respective vertex regions
  Regs<-list(vr1=rbind(A,D3,Cent,D2), #regions inside the triangles
             vr2=rbind(B,D1,Cent,D3),
             vr3=rbind(C,D2,Cent,D1)
  )
  Reg.Cent<-vector()
  for (i in 1:length(Regs))
  { Reg.Cent<-rbind(Reg.Cent,apply(Regs[[i]],2,mean))}

  Reg.names<-c("vr=1","vr=2","vr=3")  #regions names

  res<-list(
    txt1=txt1, txt2=txt2,
    type=typ,
    mtitle=main.txt, #main label in the plot
    ext=ce, #furthest points from edges in each edge region
    X=dat, num.points=n, #data points and its size
    ROI=tri, # region of interest for X points
    cent=Cent, ncent=Cname, #center and center name
    regions=Regs, region.names=Reg.names, region.centers=Reg.Cent,
    dist2ref=Dis #distances of closest points to edges in the respective vertex region
  )
  class(res)<-"Extrema"

  res$call <-match.call()

  res
} #end of the function
#'

#################################################################

#' @title The indicator for the presence of an arc from a point to another for Proportional Edge Proximity Catch
#' Digraphs (PE-PCDs) - standard equilateral triangle case
#'
#' @description Returns \eqn{I(}\code{pt2} is in \eqn{N_{PE}(pt1,r))} for points \code{pt1} and \code{pt2}, that is, returns 1 if \code{pt2} is in \eqn{N_{PE}(pt1,r)},
#' returns 0 otherwise, where \eqn{N_{PE}(x,r)} is the PE proximity region for point \eqn{x} with expansion parameter \eqn{r \ge 1}.
#'
#' PE proximity region is defined with respect to the standard equilateral triangle
#' \eqn{T_e=T(v=1,v=2,v=3)=T((0,0),(1,0),(1/2,\sqrt{3}/2))} and vertex regions are based on the center \eqn{M=(m_1,m_2)}
#' in Cartesian coordinates or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the interior of \eqn{T_e};
#' default is \eqn{M=(1,1,1)} i.e., the center of mass of \eqn{T_e}.
#' \code{rv} is the index of the vertex region \code{pt1} resides, with default=\code{NULL}.
#'
#' If \code{pt1} and \code{pt2} are distinct and either of them are outside \eqn{T_e}, it returns 0,
#' but if they are identical, then it returns 1 regardless of their locations (i.e., it allows loops).
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:arc-density-CS;textual}{pcds}).
#'
#' @param pt1 A 2D point whose PE proximity region is constructed.
#' @param pt2 A 2D point. The function determines whether \code{pt2} is inside the PE proximity region of
#' \code{pt1} or not.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param M A 2D point in Cartesian coordinates or a 3D point in barycentric coordinates
#' which serves as a center in the interior of the standard equilateral triangle \eqn{T_e}; default is \eqn{M=(1,1,1)} i.e.
#' the center of mass of \eqn{T_e}.
#' @param rv The index of the vertex region in \eqn{T_e} containing the point, either \code{1,2,3} or \code{NULL}
#' (default is \code{NULL}).
#'
#' @return \eqn{I(}\code{pt2} is in \eqn{N_{PE}(pt1,r))} for points \code{pt1} and \code{pt2}, that is, returns 1 if \code{pt2} is in \eqn{N_{PE}(pt1,r)},
#' returns 0 otherwise
#'
#' @seealso \code{\link{IndNPEtri}}, \code{\link{IndNPEbastri}}, and \code{\link{IndCSTe}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(0,0); B<-c(1,0); C<-c(1/2,sqrt(3)/2);
#' Te<-rbind(A,B,C)
#' n<-10
#'
#' set.seed(1)
#' dat<-runifTe(n)$gen.points
#'
#' M<-as.numeric(runifTe(1)$g)  #try also M<-c(.6,.2)
#'
#' IndNPETe(dat[1,],dat[10,],r=2,M)
#' IndNPETe(c(0,1),dat[10,],r=2,M)
#'
#' IndNPETe(dat[1,],dat[10,],r=1.5,M)
#' IndNPETe(dat[1,],dat[5,],r=2,M)
#' IndNPETe(dat[1,],dat[5,],r=3,M)
#'
#' IndNPETe(c(1,1),dat[5,],r=3,M)
#' IndNPETe(c(1,1),c(1,1),r=3,M)
#'
#' #or try
#' Rv<-rvTeCM(dat[1,])$rv
#' IndNPETe(dat[1,],dat[10,],r=2,M,Rv)
#'
#' P1<-c(.4,.2)
#' P2<-c(.5,.26)
#' r<-2
#' IndNPETe(P1,P2,r,M)
#'
#' @export IndNPETe
IndNPETe <- function(pt1,pt2,r,M=c(1,1,1),rv=NULL)
{
  if (!is.point(pt1) || !is.point(pt2))
  {stop('pt1 and pt2 must be numeric 2D points')}

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if (!is.point(M) && !is.point(M,3) )
  {stop('M must be a numeric 2D point for Cartesian coordinates or 3D point for barycentric coordinates')}

  if (dimension(M)==3)
  {
    M<-bary2cart(M,Te)
  }

  A<-c(0,0); B<-c(1,0); C<-c(1/2,sqrt(3)/2);
  Te<-rbind(A,B,C);

  if (in.triangle(M,Te,boundary=FALSE)$in.tri==F)
  {stop('center is not in the interior of the triangle')}

  if (identical(pt1,pt2))
  {arc<-1; return(arc); stop}

  if (!in.triangle(pt1,Te,boundary=TRUE)$in.tri || !in.triangle(pt2,Te,boundary=TRUE)$in.tri)
  {arc<-0; return(arc); stop}

  if (is.null(rv))
  {rv<-rvTe.cent(pt1,M)$rv #vertex region for pt1
  } else
  {  if (!is.numeric(rv) || sum(rv==c(1,2,3))!=1)
  {stop('vertex index, rv, must be 1, 2 or 3')}}

  X1<-pt1[1]; Y1<-pt1[2];
  X2<-pt2[1]; Y2<-pt2[2];
  arc<-0;
  if (rv==1)
  {
    x1n<-X1*r; y1n<-Y1*r;
    if ( Y2 < y1n-3^(1/2)*X2+3^(1/2)*x1n ) {arc <-1}
  } else {
    if (rv==2)
    {
      x1n<-1+(X1-1)*r; y1n<-Y1*r;
      if ( Y2 < y1n+3^(1/2)*X2-3^(1/2)*x1n  ) {arc <-1}
    } else {
      y1n<-C[2]+(Y1-C[2])*r;
      if ( Y2 > y1n ) {arc<-1}
    }}
  arc
} #end of the function
#'

#################################################################

#' @title Number of arcs of Proportional Edge Proximity Catch Digraphs (PE-PCDs) - standard
#' equilateral triangle case
#'
#' @description Returns the number of arcs of Proportional Edge Proximity Catch Digraphs (PE-PCDs) whose vertices are the
#' given 2D numerical data set, \code{dat}.
#'
#' PE proximity region \eqn{N_{PE}(x,r)} is defined with respect to the standard
#' equilateral triangle \eqn{T_e=T(v=1,v=2,v=3)=T((0,0),(1,0),(1/2,\sqrt{3}/2))} with expansion parameter \eqn{r \ge 1}
#' and vertex regions are based on the center \eqn{M=(m_1,m_2)}
#' in Cartesian coordinates or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the interior of \eqn{T_e};
#' default is \eqn{M=(1,1,1)} i.e., the center of mass of \eqn{T_e}.
#' For the number of arcs, loops are not allowed so
#' arcs are only possible for points inside \eqn{T_e} for this function.
#'
#' See also (\insertCite{ceyhan:arc-density-PE;textual}{pcds}).
#'
#' @param dat A set of 2D points which constitute the vertices of the PE-PCD.
#' @param r A positive real number which serves as the expansion parameter for PE proximity region;
#' must be \eqn{\ge 1}.
#' @param M A 2D point in Cartesian coordinates or a 3D point in barycentric coordinates
#' which serves as a center in the interior of the standard equilateral triangle \eqn{T_e}; default is \eqn{M=(1,1,1)} i.e.
#' the center of mass of \eqn{T_e}.
#'
#' @return Number of arcs for the PE-PCD with vertices being 2D data set, \code{dat}, in \eqn{T_e}
#' with expansion parameter, \eqn{r \ge 1}, and \code{M}-vertex regions. PE proximity regions are defined only
#' for \code{dat} points inside \eqn{T_e}, i.e., arcs are possible for such points only.
#'
#' @seealso \code{\link{NumArcsPEtri}}, \code{\link{NumArcsPEMT}}, and \code{\link{NumArcsCSTe}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(0,0); B<-c(1,0); C<-c(1/2,sqrt(3)/2);
#' n<-10  #try also n<-20
#'
#' set.seed(1)
#' dat<-runifTe(n)$gen.points
#'
#' M<-c(.6,.2)  #try also M<-c(1,1,1)
#'
#' NumArcsPETe(dat,r=1.25,M)
#' NumArcsPETe(dat,r=1.5,M)
#' NumArcsPETe(dat,r=2,M)
#'
#' NumArcsPETe(rbind(dat,c(0,1)),r=2,M)
#' NumArcsPETe(c(.2,.3),r=2,M)
#'
#' NumArcsPETe(dat,r=1.5,M);
#'
#' dat.fr<-data.frame(a=dat)
#' NumArcsPETe(dat.fr,r=1.5,M);
#'
#' @export NumArcsPETe
NumArcsPETe <- function(dat,r,M=c(1,1,1))
{
  if (!is.numeric(as.matrix(dat)))
  {stop('dat must be numeric')}

  if (is.point(dat))
  { dat<-matrix(dat,ncol=2)
  } else
  {dat<-as.matrix(dat)
  if (ncol(dat)!=2 )
  {stop('dat must be of dimension nx2')}
  }

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if (!is.point(M) && !is.point(M,3) )
  {stop('M must be a numeric 2D point for Cartesian coordinates or 3D point for barycentric coordinates')}

  if (dimension(M)==3)
  {
    M<-bary2cart(M,Te)
  }

  A<-c(0,0); B<-c(1,0); C<-c(1/2,sqrt(3)/2);
  Te<-rbind(A,B,C);

  if (in.triangle(M,Te,boundary=FALSE)$in.tri==F)
  {stop('center is not in the interior of the triangle')}

  n<-nrow(dat)

  arcs<-0
  for (i in 1:n)
  {pt1<-dat[i,]
  if (!in.triangle(pt1,Te,boundary = TRUE)$in.tri)
  {arcs<-arcs+0
  } else
  {
    rv<-rvTe.cent(pt1,M)$rv
    for (j in ((1:n)[-i]) )
    {pt2<-dat[j,]
    if (!in.triangle(pt2,Te,boundary = TRUE)$in.tri)
    {arcs<-arcs+0
    } else
    {
      arcs<-arcs+IndNPETe(pt1,pt2,r,M,rv)
    }
    }
  }
  }
  arcs
} #end of the function
#'

#################################################################

#' @title Incidence matrix for Proportional Edge Proximity Catch Digraphs (PE-PCDs) - standard
#' equilateral triangle case
#'
#' @description Returns the incidence matrix for the PE-PCD whose vertices are the given 2D numerical data set, \code{dat},
#' in the standard equilateral triangle \eqn{T_e=T(v=1,v=2,v=3)=T((0,0),(1,0),(1/2,\sqrt{3}/2))}.
#'
#' PE proximity region is constructed with respect to the standard equilateral triangle \eqn{T_e} with
#' expansion parameter \eqn{r \ge 1} and vertex regions are based on
#' the center \eqn{M=(m_1,m_2)} in Cartesian coordinates or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates
#' in the interior of \eqn{T_e}; default is \eqn{M=(1,1,1)} i.e., the center of mass of \eqn{T_e}.
#' Loops are allowed, so the diagonal entries are all equal to 1.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:dom-num-NPE-Spat2011;textual}{pcds}).
#'
#' @param dat A set of 2D points which constitute the vertices of the PE-PCD.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param M A 2D point in Cartesian coordinates or a 3D point in barycentric coordinates
#' which serves as a center in the interior of the standard equilateral triangle \eqn{T_e}; default is \eqn{M=(1,1,1)} i.e.
#' the center of mass of \eqn{T_e}.
#'
#' @return Incidence matrix for the PE-PCD with vertices being 2D data set, \code{dat} and PE proximity
#' regions are defined in the standard equilateral triangle \eqn{T_e} with \code{M}-vertex regions.
#'
#' @seealso \code{\link{IncMatPEtri}}, \code{\link{IncMatPEMT}} and \code{\link{IncMatCSTe}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(0,0); B<-c(1,0); C<-c(1/2,sqrt(3)/2);
#' Te<-rbind(A,B,C)
#' n<-10
#'
#' set.seed(1)
#' dat<-runifTe(n)$gen.points
#'
#' M<-as.numeric(runifTe(1)$g)  #try also M<-c(.6,.2)
#'
#' NumArcsPETe(dat,r=1.25)
#'
#' inc.mat<-IncMatPETe(dat,r=1.25,M)
#' inc.mat
#' sum(inc.mat)-n
#'
#' dom.greedy(inc.mat)
#' IndUBdom(inc.mat,2)
#' dom.exact(inc.mat)
#'
#' inc.mat<-IncMatPETe(rbind(dat,c(0,1)),r=1.25,M)
#' inc.mat
#' sum(inc.mat)-(n+1)
#'
#' IncMatPETe(dat,r=1.5,M);
#'
#' IncMatPETe(rbind(dat,dat),r=1.5,M)
#'
#' dat.fr<-data.frame(a=dat)
#' IncMatPETe(dat.fr,r=1.5,M);
#'
#' @export IncMatPETe
IncMatPETe <- function(dat,r,M=c(1,1,1))
{
  if (!is.numeric(as.matrix(dat)))
  {stop('dat must be numeric')}

  if (is.point(dat))
  { dat<-matrix(dat,ncol=2)
  } else
  {dat<-as.matrix(dat)
  if (ncol(dat)!=2 )
  {stop('dat must be of dimension nx2')}
  }

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if (!is.point(M) && !is.point(M,3) )
  {stop('M must be a numeric 2D point for Cartesian coordinates or 3D point for barycentric coordinates')}

  A<-c(0,0); B<-c(1,0); C<-c(1/2,sqrt(3)/2);
  Te<-rbind(A,B,C);

  if (dimension(M)==3)
  {
    M<-bary2cart(M,Te)
  }

  if (in.triangle(M,Te,boundary=FALSE)$in.tri==F)
  {stop('center is not in the interior of the triangle')}

  n<-nrow(dat)

 inc.mat<-matrix(0, nrow=n, ncol=n)
  for (i in 1:n)
  {pt1<-dat[i,]
  rv<-rvTe.cent(pt1,M)$rv
  for (j in ((1:n)) )
  {pt2<-dat[j,]
 inc.mat[i,j]<-IndNPETe(pt1,pt2,r,M,rv)
  }
  }
 inc.mat
} #end of the function
#'

#################################################################

# funsMuVarPE2D
#'
#' @title Returns the mean and (asymptotic) variance of arc density of Proportional Edge Proximity Catch Digraph (PE-PCD)
#' for 2D uniform data in one triangle
#'
#' @description
#' Two functions: \code{muPE2D} and \code{asyvarPE2D}.
#'
#' \code{muPE2D} returns the mean of the (arc) density of PE-PCD
#' and \code{asyvarPE2D} returns the asymptotic variance of the arc density of PE-PCD
#' with expansion parameter \eqn{r \ge 1} for 2D uniform data in a triangle.
#'
#' PE proximity regions are defined with respect to the triangle and
#' vertex regions are based on center of mass, \eqn{CM} of the triangle.
#'
#' See also (\insertCite{ceyhan:arc-density-PE;textual}{pcds}).
#'
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#'
#' @return \code{muPE2D} returns the mean and \code{asyvarPE2D} returns the (asymptotic) variance of the
#' arc density of PE-PCD for uniform data in any triangle
#'
#' @name funsMuVarPE2D
NULL
#'
#' @seealso \code{\link{muCS2D}} and \code{\link{asyvarCS2D}}
#'
#' @rdname funsMuVarPE2D
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' #Examples for muPE2D
#' muPE2D(1.2)
#'
#' rseq<-seq(1.01,5,by=.05)
#' lrseq<-length(rseq)
#'
#' mu<-vector()
#' for (i in 1:lrseq)
#' {
#'   mu<-c(mu,muPE2D(rseq[i]))
#' }
#'
#' plot(rseq, mu,type="l",xlab="r",ylab=expression(mu(r)),lty=1,xlim=range(rseq),ylim=c(0,1))
#'
#' @export muPE2D
muPE2D <- function(r)
{
  if (!is.point(r,1) || r<1)
  {stop('the argument must be a scalar greater than 1')}

  mn<-0;
  if (r < 3/2)
  {
    mn<-(37/216)*r^2;
  } else {
    if (r < 2)
    {
      mn<--r^2/8-8/r+9/(2*r^2)+4;
    } else {
      mn<-1-3/(2*r^2);
    }}
  mn
} #end of the function
#'
#' @references
#' \insertAllCited{}
#'
#' @rdname funsMuVarPE2D
#'
#' @examples
#' #Examples for asyvarPE2D
#' asyvarPE2D(1.2)
#'
#' rseq<-seq(1.01,5,by=.05)
#' lrseq<-length(rseq)
#'
#' avar<-vector()
#' for (i in 1:lrseq)
#' {
#'   avar<-c(avar,asyvarPE2D(rseq[i]))
#' }
#'
#' oldpar <- par(no.readonly = TRUE)
#' par(mar=c(5,5,4,2))
#' plot(rseq, avar,type="l",xlab="r",ylab=expression(paste(sigma^2,"(r)")),lty=1,xlim=range(rseq))
#' par(oldpar)
#'
#' @export asyvarPE2D
asyvarPE2D <- function(r)
{
  if (!is.point(r,1) || r<1)
  {stop('the argument must be a scalar greater than 1')}

  asyvar<-0;
  if (r < 4/3)
  {
    asyvar<-(3007*r^(10)-13824*r^9+898*r^8+77760*r^7-117953*r^6+48888*r^5-24246*r^4+60480*r^3-38880*r^2+3888)/(58320*r^4);
  } else {
    if (r < 3/2)
    {
      asyvar<-(5467*r^(10)-37800*r^9+61912*r^8+46588*r^6-191520*r^5+13608*r^4+241920*r^3-155520*r^2+15552)/(233280*r^4);
    } else {
      if (r < 2)
      {
        asyvar<--(7*r^(12)-72*r^(11)+312*r^(10)-5332*r^8+15072*r^7+13704*r^6-139264*r^5+273600*r^4-242176*r^3+103232*r^2-27648*r+8640)/(960*r^6);
      } else {
        asyvar<-(15*r^4-11*r^2-48*r+25)/(15*r^6);
      }}}
  asyvar # no need to multiply this by 4
} #end of the function
#'

#################################################################

#' @title The indicator for a point being a dominating point or not for Proportional Edge Proximity Catch
#' Digraphs (PE-PCDs) - basic triangle case
#'
#' @description Returns \eqn{I(}\code{p} is a dominating point of the PE-PCD\eqn{)} where the vertices of the PE-PCD are the 2D data set \code{Dt} for data in the basic triangle
#' \eqn{T_b=T((0,0),(1,0),(c_1,c_2))}, that is, returns 1 if \code{p} is a dominating point of PE-PCD, returns 0 otherwise.
#'
#' PE proximity regions are defined with respect to the basic triangle \eqn{T_b}.
#' In the basic triangle, \eqn{T_b}, \eqn{c_1} is in \eqn{[0,1/2]}, \eqn{c_2>0} and \eqn{(1-c_1)^2+c_2^2 \le 1}.
#'
#' Any given triangle can be mapped to the basic triangle by a combination of rigid body motions
#' (i.e., translation, rotation and reflection) and scaling, preserving uniformity of the points in the
#' original triangle. Hence basic triangle is useful for simulation
#' studies under the uniformity hypothesis.
#'
#' Vertex regions are based on center \eqn{M=(m_1,m_2)} in Cartesian
#' coordinates or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the interior of a basic triangle
#' to the edges on the extension of the lines joining \code{M} to the vertices or based on the circumcenter of \eqn{T_b};
#' default is \eqn{M=(1,1,1)} i.e., the center of mass of \eqn{T_b}.
#' Point, \code{p}, is in the vertex region of vertex \code{rv} (default is \code{NULL}); vertices are labeled as \eqn{1,2,3}
#' in the order they are stacked row-wise.
#'
#' \code{ch.data.pnt} is for checking whether point \code{p} is a data point in \code{Dt} or not (default is \code{FALSE}),
#' so by default this function checks whether the point \code{p} would be a dominating point
#' if it actually were in the data set.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:dom-num-NPE-Spat2011;textual}{pcds}).
#'
#' @param p A 2D point that is to be tested for being a dominating point or not of the PE-PCD.
#' @param Dt A set of 2D points which constitutes the vertices of the PE-PCD.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param c1,c2 Positive real numbers which constitute the vertex of the basic triangle
#' adjacent to the shorter edges; \eqn{c_1} must be in \eqn{[0,1/2]}, \eqn{c_2>0} and \eqn{(1-c_1)^2+c_2^2 \le 1}.
#' @param M A 2D point in Cartesian coordinates or a 3D point in barycentric coordinates
#' which serves as a center in the interior of the basic triangle \eqn{T_b} or the circumcenter of \eqn{T_b};
#' default is \eqn{M=(1,1,1)} i.e., the center of mass of \eqn{T_b}.
#' @param rv Index of the vertex whose region contains point \code{p}, \code{rv} takes the vertex labels as \eqn{1,2,3} as
#' in the row order of the vertices in \eqn{T_b}.
#' @param ch.data.pnt A logical argument for checking whether point \code{p} is a data point in \code{Dt} or not
#' (default is \code{FALSE}).
#'
#' @return \eqn{I(}\code{p} is a dominating point of the PE-PCD\eqn{)} where the vertices of the PE-PCD are the 2D data set \code{Dt},
#' that is, returns 1 if \code{p} is a dominating point, returns 0 otherwise
#'
#' @seealso \code{\link{Gam1ASbastri}} and \code{\link{Gam1AStri}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' c1<-.4; c2<-.6;
#' A<-c(0,0); B<-c(1,0); C<-c(c1,c2);
#' Tb<-rbind(A,B,C)
#' n<-10  #try also n<-20
#'
#' set.seed(1)
#' dat<-runif.bastri(n,c1,c2)$g
#'
#' M<-as.numeric(runif.bastri(1,c1,c2)$g)  #try also M<-c(.6,.3)
#' r<-2
#'
#' P<-c(.4,.2)
#' Gam1PEbastri(P,dat,r,c1,c2,M)
#' Gam1PEbastri(P,P,r,c1,c2,M)
#'
#' Gam1PEbastri(dat[1,],dat,r,c1,c2,M)
#'
#' Gam1PEbastri(c(1,1),dat,r,c1,c2,M)
#'
#' Gam1PEbastri(c(1,1),dat,r,c1,c2,M,ch.data.pnt = FALSE)
#' #gives an error message if ch.data.pnt = TRUE since point p=c(1,1) is not a data point in Dt
#'
#' Gam1PEbastri(c(1,1),c(1,1),r,c1,c2,M)
#'
#' #or try
#' Rv<-rv.bastri.cent(dat[1,],c1,c2,M)$rv
#' Gam1PEbastri(dat[1,],dat,r,c1,c2,M,Rv)
#'
#' Gam1PEbastri(c(2,1),dat,r,c1,c2,M)
#'
#' Gam1PEbastri(c(.2,.1),dat,r,c1,c2,M)
#'
#' gam.vec<-vector()
#' for (i in 1:n)
#' {gam.vec<-c(gam.vec,Gam1PEbastri(dat[i,],dat,r,c1,c2,M))}
#'
#' ind.gam1<-which(gam.vec==1)
#' ind.gam1
#'
#' Xlim<-range(Tb[,1],dat[,1])
#' Ylim<-range(Tb[,2],dat[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' if (dimension(M)==3) {M<-bary2cart(M,Tb)}
#' #need to run this when M is given in barycentric coordinates
#'
#' if (identical(M,circ.cent.tri(Tb)))
#' {
#'   plot(Tb,pch=".",asp=1,xlab="",ylab="",axes=TRUE,
#'   xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#'   polygon(Tb)
#'   points(dat,pch=1,col=1)
#'   Ds<-rbind((B+C)/2,(A+C)/2,(A+B)/2)
#' } else
#' {plot(Tb,pch=".",xlab="",ylab="",axes=TRUE,
#' xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#'   polygon(Tb)
#'   points(dat,pch=1,col=1)
#'   Ds<-cp2e.bastri(c1,c2,M)}
#' L<-rbind(M,M,M); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#' points(rbind(dat[ind.gam1,]),pch=4,col=2)
#'
#' txt<-rbind(Tb,M,Ds)
#' xc<-txt[,1]+c(-.02,.02,.02,-.02,.03,-.03,.01)
#' yc<-txt[,2]+c(.02,.02,.02,-.02,.02,.02,-.03)
#' txt.str<-c("A","B","C","M","D1","D2","D3")
#' text(xc,yc,txt.str)
#'
#' P<-c(.4,.2)
#' Gam1PEbastri(P,dat,r,c1,c2,M)
#'
#' Gam1PEbastri(P,rbind(dat,dat),r,c1,c2,M)
#'
#' dat.fr<-data.frame(a=dat)
#' Gam1PEbastri(P,dat.fr,r,c1,c2,M)
#'
#' Gam1PEbastri(c(.2,.1),dat,r,c1,c2,M,ch.data.pnt=FALSE)
#' #gives an error message if ch.data.pnt=TRUE since point p is not a data point in Dt
#'
#' @export Gam1PEbastri
Gam1PEbastri <- function(p,Dt,r,c1,c2,M=c(1,1,1),rv=NULL,ch.data.pnt=FALSE)
{
  if (!is.point(p))
  {stop('p must be a numeric 2D point')}

  if (!is.numeric(as.matrix(Dt)))
  {stop('Dt must be numeric')}

  if (is.point(Dt))
  { Dt<-matrix(Dt,ncol=2)
  } else
  {Dt<-as.matrix(Dt)
  if (ncol(Dt)!=2 )
  {stop('Dt must be of dimension nx2')}
  }

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if (!is.point(c1,1) || !is.point(c2,1))
  {stop('c1 and c2 must be scalars')}

  if (c1<0 || c1>1/2 || c2<=0 || (1-c1)^2+c2^2 >1)
  {stop('c1 must be in [0,1/2], c2>0 and (1-c1)^2+c2^2 <=1')}

  if (!is.point(M) && !is.point(M,3) )
  {stop('M must be a numeric 2D point for Cartesian coordinates or 3D point for barycentric coordinates')}

  if (ch.data.pnt==TRUE)
  {
    if (!is.in.data(p,Dt))
    {stop('p is not a data point in Dt')}
  }

  y1<-c(0,0); y2<-c(1,0); y3<-c(c1,c2); Tb<-rbind(y1,y2,y3)

  if (dimension(M)==3)
  {
    M<-bary2cart(M,Tb)
  }

  if (isTRUE(all.equal(M,circ.cent.tri(Tb)))==F & in.triangle(M,Tb,boundary=FALSE)$in.tri==F)
  {stop('center is not the circumcenter or not in the interior of the triangle')}

  if (in.triangle(p,Tb)$in.tri==F)
  {dom<-0; return(dom); stop} #('point is not inside the triangle')}

  n<-nrow(Dt)
  dom<-1; i<-1;

  if (is.null(rv))
  { rv<-ifelse(isTRUE(all.equal(M,circ.cent.tri(Tb)))==T,rv.triCC(p,Tb)$rv,rv.tri.cent(p,Tb,M)$rv)  #vertex region for p
  } else
  {  if (!is.numeric(rv) || sum(rv==c(1,2,3))!=1)
  {stop('vertex index, rv, must be 1, 2 or 3')}}

  while (i <= n & dom==1)
  {if (IndNPEbastri(p,Dt[i,],r,c1,c2,M,rv)==0)
    dom<-0;
  i<-i+1;
  }
  dom
} #end of the function
#'

#################################################################

#' @title The indicator for two points being a dominating set for Proportional Edge Proximity Catch Digraphs
#' (PE-PCDs) - basic triangle case
#'
#' @description Returns \eqn{I(}\{\code{pt1,pt2}\} is a dominating set of the PE-PCD\eqn{)} where the vertices of the PE-PCD are the 2D data set \code{Dt} in the basic triangle
#' \eqn{T_b=T((0,0),(1,0),(c_1,c_2))}, that is, returns 1 if \{\code{pt1,pt2}\} is a dominating set of PE-PCD,
#' returns 0 otherwise.
#'
#' PE proximity regions are defined with respect to \eqn{T_b}.
#' In the basic triangle, \eqn{T_b}, \eqn{c_1} is in \eqn{[0,1/2]}, \eqn{c_2>0} and \eqn{(1-c_1)^2+c_2^2 \le 1}.
#'
#' Any given triangle can be mapped to the basic triangle by a combination of rigid body motions
#' (i.e., translation, rotation and reflection) and scaling, preserving uniformity of the points in the
#' original triangle. Hence basic triangle is useful for simulation
#' studies under the uniformity hypothesis.
#'
#' Vertex regions are based on center \eqn{M=(m_1,m_2)} in Cartesian
#' coordinates or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the interior of a basic triangle \eqn{T_b};
#' default is \eqn{M=(1,1,1)} i.e., the center of mass of \eqn{T_b}.
#' Point, \code{pt1}, is in the vertex region of vertex \code{rv1} (default is \code{NULL});
#' and point, \code{pt2}, is in the vertex region of vertex \code{rv2} (default is \code{NULL}); vertices are labeled as \eqn{1,2,3}
#' in the order they are stacked row-wise.
#'
#' \code{ch.data.pnts} is for checking whether points \code{pt1} and \code{pt2} are both data points in \code{Dt} or not (default is \code{FALSE}),
#' so by default this function checks whether the points \code{pt1} and \code{pt2} would constitute a dominating set
#' if they both were actually in the data set.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:dom-num-NPE-Spat2011;textual}{pcds}).
#'
#' @param pt1,pt2 Two 2D points to be tested for constituting a dominating set of the PE-PCD.
#' @param Dt A set of 2D points which constitutes the vertices of the PE-PCD.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param c1,c2 Positive real numbers which constitute the vertex of the basic triangle.
#' adjacent to the shorter edges; \eqn{c_1} must be in \eqn{[0,1/2]}, \eqn{c_2>0} and \eqn{(1-c_1)^2+c_2^2 \le 1}.
#' @param M A 2D point in Cartesian coordinates or a 3D point in barycentric coordinates
#' which serves as a center in the interior of the basic triangle \eqn{T_b} or the circumcenter of \eqn{T_b};
#' default is \eqn{M=(1,1,1)} i.e., the center of mass of \eqn{T_b}.
#' @param rv1,rv2 The indices of the vertices whose regions contains \code{pt1} and \code{pt2}, respectively.
#' They take the vertex labels as \eqn{1,2,3} as in the row order of the vertices in \eqn{T_b}
#' (default is \code{NULL} for both).
#' @param ch.data.pnts A logical argument for checking whether points \code{pt1} and \code{pt2} are data points in \code{Dt} or not
#' (default is \code{FALSE}).
#'
#' @return \eqn{I(}\{\code{pt1,pt2}\} is a dominating set of the PE-PCD\eqn{)} where the vertices of the PE-PCD are the 2D data set \code{Dt}),
#' that is, returns 1 if \{\code{pt1,pt2}\} is a dominating set of PE-PCD, returns 0 otherwise
#'
#' @seealso \code{\link{Gam2PEtri}}, \code{\link{Gam2ASbastri}}, and \code{\link{Gam2AStri}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' c1<-.4; c2<-.6;
#' A<-c(0,0); B<-c(1,0); C<-c(c1,c2);
#' Tb<-rbind(A,B,C)
#' n<-10  #try also n<-20
#'
#' set.seed(1)
#' dat<-runif.bastri(n,c1,c2)$g
#'
#' M<-as.numeric(runif.bastri(1,c1,c2)$g)  #try also M<-c(.6,.3)
#'
#' r<-2
#'
#' Gam2PEbastri(dat[1,],dat[2,],dat,r,c1,c2,M)
#' Gam2PEbastri(c(1,1),dat[2,],dat,r,c1,c2,M)
#'
#' Gam2PEbastri(c(1,2),dat[2,],dat,r,c1,c2,M)
#'
#' Gam2PEbastri(c(1,2),c(1,3),rbind(c(1,2),c(1,3)),r,c1,c2,M)
#' Gam2PEbastri(c(1,2),c(1,3),rbind(c(1,2),c(1,3)),r,c1,c2,M,ch.data.pnts = TRUE)
#'
#' ind.gam2<-vector()
#' for (i in 1:(n-1))
#'   for (j in (i+1):n)
#'   {if (Gam2PEbastri(dat[i,],dat[j,],dat,r,c1,c2,M)==1)
#'    ind.gam2<-rbind(ind.gam2,c(i,j))}
#' ind.gam2
#'
#' #or try
#' rv1<-rv.bastri.cent(dat[1,],c1,c2,M)$rv;
#' rv2<-rv.bastri.cent(dat[2,],c1,c2,M)$rv;
#' Gam2PEbastri(dat[1,],dat[2,],dat,r,c1,c2,M,rv1,rv2)
#'
#' #or try
#' rv1<-rv.bastri.cent(dat[1,],c1,c2,M)$rv;
#' Gam2PEbastri(dat[1,],dat[2,],dat,r,c1,c2,M,rv1)
#'
#' #or try
#' rv2<-rv.bastri.cent(dat[2,],c1,c2,M)$rv;
#' Gam2PEbastri(dat[1,],dat[2,],dat,r,c1,c2,M,rv2=rv2)
#'
#' P1<-c(.4,.2)
#' P2<-c(.6,.2)
#' Gam2PEbastri(P1,P2,dat,r,c1,c2,M)
#'
#' Gam2PEbastri(P1,P2,rbind(dat,dat),r,c1,c2,M)
#'
#' dat.fr<-data.frame(a=dat)
#' Gam2PEbastri(P1,P2,dat.fr,r,c1,c2,M)
#'
#' Gam2PEbastri(c(1,2),dat[2,],dat,r,c1,c2,M,ch.data.pnts = FALSE)
#' #gives an error message if ch.data.pnts = TRUE since not both points are data points in Dt
#'
#' @export Gam2PEbastri
Gam2PEbastri <- function(pt1,pt2,Dt,r,c1,c2,M=c(1,1,1),rv1=NULL,rv2=NULL,ch.data.pnts=FALSE)
{
  if (!is.point(pt1) || !is.point(pt2))
  {stop('pt1 and pt2 must be numeric 2D points')}

  if (!is.numeric(as.matrix(Dt)))
  {stop('Dt must be numeric')}

  if (is.point(Dt))
  { Dt<-matrix(Dt,ncol=2)
  } else
  {Dt<-as.matrix(Dt)
  if (ncol(Dt)!=2 )
  {stop('Dt must be of dimension nx2')}
  }

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if (!is.point(c1,1) || !is.point(c2,1))
  {stop('c1 and c2 must be scalars')}

  if (c1<0 || c1>1/2 || c2<=0 || (1-c1)^2+c2^2 >1)
  {stop('c1 must be in [0,1/2], c2>0 and (1-c1)^2+c2^2 <=1')}

  if (!is.point(M) && !is.point(M,3) )
  {stop('M must be a numeric 2D point for Cartesian coordinates or 3D point for barycentric coordinates')}

  if (ch.data.pnts==TRUE)
  {
    if (!is.in.data(pt1,Dt) || !is.in.data(pt2,Dt))
    {stop('not both points are data points in Dt')}
  }

  if (identical(pt1,pt2))
  {dom<-0; return(dom); stop}

  y1<-c(0,0); y2<-c(1,0); y3<-c(c1,c2); Tb<-rbind(y1,y2,y3)

  if (dimension(M)==3)
  {
    M<-bary2cart(M,Tb)
  }

  if (isTRUE(all.equal(M,circ.cent.tri(Tb)))==F & in.triangle(M,Tb,boundary=FALSE)$in.tri==F)
  {stop('center is not the circumcenter or not in the interior of the triangle')}

  if (is.null(rv1))
  {rv1<-ifelse(isTRUE(all.equal(M,circ.cent.tri(Tb)))==T,rv.triCC(pt1,Tb)$rv,rv.tri.cent(pt1,Tb,M)$rv)  #vertex region for point pt1
  }
  if (is.null(rv2))
  {rv2<-ifelse(isTRUE(all.equal(M,circ.cent.tri(Tb)))==T,rv.triCC(pt2,Tb)$rv,rv.tri.cent(pt2,Tb,M)$rv)  #vertex region for point pt2
  }

  Dt<-matrix(Dt,ncol=2)
  n<-nrow(Dt)
  dom<-1; i<-1;

  while (i <= n & dom==1)
  {if (max(IndNPEbastri(pt1,Dt[i,],r,c1,c2,M,rv1),IndNPEbastri(pt2,Dt[i,],r,c1,c2,M,rv2))==0)
    dom<-0;
  i<-i+1;
  }
  dom
} #end of the function
#'

#################################################################

#' @title The indices of the \eqn{CM}-vertex regions in a triangle that contains the points in a give data set
#'
#' @description Returns the indices of the vertices whose regions contain the points in data set \code{Dt} in
#' a triangle \code{tri}\eqn{=(A,B,C)} and vertex regions are based on the center of mass \eqn{CM} of \code{tri}.
#' (see the plots in the example for illustrations).
#'
#' The vertices of the  triangle \code{tri} are labeled as
#' \eqn{1=A}, \eqn{2=B}, and \eqn{3=C} also according to the row number the vertex is recorded in \code{tri}.
#' If a point in \code{Dt} is not inside \code{tri}, then the function yields \code{NA} as output for that entry.
#' The corresponding vertex region is the polygon
#' with the vertex, \eqn{CM}, and midpoints the edges crossing the vertex.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010,ceyhan:mcap2012;textual}{pcds}).
#'
#' @param Dt A set of 2D points representing the set of data points for which indices of the vertex regions
#' containing them are to be determined.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#'
#' @return A \code{list} with two elements
#' \item{rv}{Indices (i.e., a \code{vector} of indices) of the vertices whose region contains points in \code{Dt}
#' in the triangle \code{tri}}
#' \item{tri}{The vertices of the triangle, where row number corresponds to the vertex index in \code{rv}.}
#'
#' @seealso \code{\link{rverts.tri.cent}}, \code{\link{rverts.triCC}} and \code{\link{rverts.tri.nd}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#'
#' P<-c(.4,.2)
#' rverts.triCM(P,Tr)
#'
#' P<-c(1.8,.5)
#' rverts.triCM(P,Tr)
#'
#' P<-c(10.5,1.6)
#' rverts.triCM(P,Tr)
#'
#' n<-10  #try also n<-20
#' set.seed(1)
#' dat<-runif.tri(n,Tr)$g
#'
#' rverts.triCM(dat,Tr)
#' rverts.triCM(rbind(dat,c(2,2)),Tr)
#'
#' rv<-rverts.triCM(dat,Tr)
#' rv
#'
#' CM<-(A+B+C)/3
#' D1<-(B+C)/2; D2<-(A+C)/2; D3<-(A+B)/2;
#' Ds<-rbind(D1,D2,D3)
#'
#' Xlim<-range(Tr[,1],dat[,1])
#' Ylim<-range(Tr[,2],dat[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(Tr,pch=".",xlab="",ylab="",axes=TRUE,xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tr)
#' points(dat,pch=".",col=1)
#' L<-matrix(rep(CM,3),ncol=2,byrow=TRUE); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#'
#' xc<-Tr[,1]+c(-.04,.05,.05)
#' yc<-Tr[,2]+c(-.05,.05,.03)
#' txt.str<-c("rv=1","rv=2","rv=3")
#' text(xc,yc,txt.str)
#'
#' txt<-rbind(CM,Ds)
#' xc<-txt[,1]+c(.04,.04,-.03,0)
#' yc<-txt[,2]+c(-.07,.04,.06,-.08)
#' txt.str<-c("CM","D1","D2","D3")
#' text(xc,yc,txt.str)
#'
#' text(dat,labels=factor(rv$rv))
#'
#' P<-c(1.8,.5)
#' rverts.triCM(P,Tr)
#'
#' dat.fr<-data.frame(a=dat)
#' rverts.triCM(dat.fr,Tr)
#'
#' dat.fr<-data.frame(a=Tr)
#' rverts.triCM(dat,dat.fr)
#'
#' @export rverts.triCM
rverts.triCM <- function(Dt,tri)
{
  if (!is.numeric(as.matrix(Dt)))
  {stop('Dt must be numeric')}

  if (is.point(Dt))
  { Dt<-matrix(Dt,ncol=2)
  } else
  {Dt<-as.matrix(Dt)
  if (ncol(Dt)!=2 )
  {stop('Dt must be of dimension nx2')}
  }

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  nt<-nrow(Dt)

  if (nt==0)
  {ind.set<-NA}
  else
  {
    A<-tri[1,]; B<-tri[2,]; C<-tri[3,]
    CM<-(A+B+C)/3;
    M1<-(A+B)/2; M2<-(B+C)/2; M3<-(A+C)/2

    t.mA<-interp::tri.mesh(c(A[1],M1[1],CM[1],M3[1]),c(A[2],M1[2],CM[2],M3[2]),duplicate="remove")
    t.mB<-interp::tri.mesh(c(B[1],M2[1],CM[1],M1[1]),c(B[2],M2[2],CM[2],M1[2]),duplicate="remove")
    t.mC<-interp::tri.mesh(c(C[1],M3[1],CM[1],M2[1]),c(C[2],M3[2],CM[2],M2[2]),duplicate="remove")

   ind.set<-rep(NA,nt)
   ind.vA<-interp::in.convex.hull(t.mA,Dt[,1],Dt[,2])
   ind.vB<-interp::in.convex.hull(t.mB,Dt[,1],Dt[,2])
   ind.vC<-interp::in.convex.hull(t.mC,Dt[,1],Dt[,2])

   ind.set[ind.vA==TRUE]<-1
   ind.set[ind.vB==TRUE]<-2
   ind.set[ind.vC==TRUE]<-3
  }

  row.names(tri)<-c("vertex 1","vertex 2","vertex 3")  #vertex labeling

  list(rv=ind.set, #relative vertices
       tri=tri #vertex labeling
  )
} #end of the function
#'

#################################################################

#' @title Centers for non-degenerate asymptotic distribution of domination number of
#' Proportional Edge Proximity Catch Digraphs (PE-PCDs)
#'
#' @description Returns the centers which yield nondegenerate asymptotic distribution for the domination number
#' of PE-PCD for uniform data in a triangle, \code{tri}\eqn{=T(v_1,v_2,v_3)}.
#'
#' PE proximity region is defined with
#' respect to the triangle \code{tri} with expansion parameter \code{r} in \eqn{(1,1.5]}.
#'
#' Vertex regions are defined with the centers that are output of this function.
#' Centers are stacked row-wise with row number is corresponding to the vertex
#' row number in \code{tri} (see the examples for an illustration). The center labels 1,2,3 correspond
#' to the vertices \eqn{M_1}, \eqn{M_2}, and \eqn{M_3} (which are the three centers for \code{r} in \eqn{(1,1.5)} which becomes center of mass
#' for \eqn{r=1.5}.).
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:masa-2007,ceyhan:dom-num-NPE-Spat2011,ceyhan:mcap2012;textual}{pcds}).
#'
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be in \eqn{(1,1.5]} for this function.
#'
#' @return The centers (stacked row-wise) which give nondegenerate asymptotic distribution
#' for the domination number of PE-PCD for uniform data in a triangle, \code{tri}.
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#' r<-1.35
#'
#' Ms<-cent.nondeg(Tr,r)
#' Ms
#'
#' Xlim<-range(Tr[,1])
#' Ylim<-range(Tr[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(Tr,pch=".",xlab="",ylab="",axes=TRUE,xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tr)
#' points(Ms,pch=".",col=1)
#' polygon(Ms,lty=2)
#'
#' xc<-Tr[,1]+c(-.02,.02,.02)
#' yc<-Tr[,2]+c(.02,.02,.03)
#' txt.str<-c("A","B","C")
#' text(xc,yc,txt.str)
#'
#' xc<-Ms[,1]+c(-.04,.04,.03)
#' yc<-Ms[,2]+c(.02,.02,.05)
#' txt.str<-c("M1","M2","M3")
#' text(xc,yc,txt.str)
#'
#' cent.nondeg(Tr,r)
#'
#' dat.fr<-data.frame(a=Tr)
#' cent.nondeg(dat.fr,r)
#'
#' @export cent.nondeg
cent.nondeg <- function(tri,r)
{
  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  if (!is.point(r,1) || r<=1 || r>1.5)
  {stop('r must be a scalar in (1,1.5]')}

  A<-tri[1,]; a1<-A[1]; a2<-A[2];
  B<-tri[2,]; b1<-B[1]; b2<-B[2];
  C<-tri[3,]; c1<-C[1]; c2<-C[2];

  M1x<--(a1*r-b1*r-c1*r-2*a1+b1+c1)/r;
  M1y<--(a2*r-b2*r-c2*r-2*a2+b2+c2)/r;
  M1<-c(M1x,M1y);

  M2x<-(a1*r-b1*r+c1*r-a1+2*b1-c1)/r;
  M2y<-(a2*r-b2*r+c2*r-a2+2*b2-c2)/r;
  M2<-c(M2x,M2y);

  M3x<-(a1*r+b1*r-c1*r-a1-b1+2*c1)/r;
  M3y<-(a2*r+b2*r-c2*r-a2-b2+2*c2)/r;
  M3<-c(M3x,M3y);

  rbind(M1,M2,M3)
} #end of the function
#'

#################################################################

#' @title Projections of Centers for non-degenerate asymptotic distribution of domination number of
#' Proportional Edge Proximity Catch Digraphs (PE-PCDs) to its edges
#'
#' @description Returns the projections from  center \code{cent} to the edges on the extension of the lines joining \code{cent} to the vertices
#' in the triangle, \code{tri}. Here M is one of the three centers which gives nondegenerate asymptotic distribution
#' of the domination number of PE-PCD for uniform data in \code{tri} for a given expansion parameter \code{r} in \eqn{(1,1.5]}.
#' The center label \code{cent} values \code{1,2,3} correspond to the vertices \eqn{M_1}, \eqn{M_2}, and \eqn{M_3} (i.e., row numbers in the
#' output of \code{cent.nondeg(tri,r)}); default for \code{cent} is 1. \code{cent} becomes center of mass \eqn{CM} for \eqn{r=1.5}.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:masa-2007,ceyhan:dom-num-NPE-Spat2011;textual}{pcds}).
#'
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be in \eqn{(1,1.5]} for this function.
#' @param cent Index of the center (as \eqn{1,2,3} corresponding to \eqn{M_1,\,M_2,\,M_3}) which gives nondegenerate asymptotic
#' distribution of the domination number of PE-PCD for uniform data in \code{tri} for expansion parameter \code{r} in \eqn{(1,1.5]};
#' default \code{cent=1}.
#'
#' @return Three projection points (stacked row-wise) from one of the centers (as \eqn{1,2,3} corresponding to
#' \eqn{M_1,\,M_2,\,M_3}) which gives nondegenerate asymptotic distribution of the domination number of PE-PCD for
#' uniform data in \code{tri} for expansion parameter \code{r} in \eqn{(1,1.5]}.
#'
#' @seealso \code{\link{cp2e.bastri}} and \code{\link{cp2e.tri}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#' r<-1.35
#'
#' cp2edges.nd(Tr,r,cent=2)
#'
#' Ms<-cent.nondeg(Tr,r)
#'
#' Ds<-cp2edges.nd(Tr,r,cent=1)
#' D1<-Ds[1,]; D2<-Ds[2,]; D3<-Ds[3,]
#'
#' Xlim<-range(Tr[,1])
#' Ylim<-range(Tr[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(Tr,pch=".",xlab="",ylab="",axes=TRUE,xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tr)
#' points(Ms,pch=".",col=1)
#' polygon(Ms,lty=2)
#'
#' xc<-Tr[,1]
#' yc<-Tr[,2]
#' txt.str<-c("A","B","C")
#' text(xc,yc,txt.str)
#'
#' txt<-Ms
#' xc<-txt[,1]
#' yc<-txt[,2]
#' txt.str<-c("M1","M2","M3")
#' text(xc,yc,txt.str)
#'
#' points(Ds,pch=4,col=2)
#' L<-Tr; R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2,col=2)
#' txt<-Ds
#' xc<-txt[,1]
#' yc<-txt[,2]
#' txt.str<-c("D1","D2","D3")
#' text(xc,yc,txt.str)
#'
#' cp2edges.nd(Tr,r,1)
#'
#' dat.fr<-data.frame(a=Tr)
#' cp2edges.nd(dat.fr,r,1)
#'
#' cp2edges.nd(Tr,r,cent=3)
#' #gives an error message if center index, cent, is different from 1, 2 or 3
#' cp2edges.nd(Tr,r=1.49,cent=2)
#' #gives an error message if r>1.5
#'
#' @export cp2edges.nd
cp2edges.nd <- function(tri,r,cent=1)
{
  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  if (!is.point(r,1) || r<=1 || r>1.5)
  {stop('r must be a scalar in (1,1.5]')}

  if (cent!=1 & cent!=2 & cent!=3)
  {stop('center index, cent, must be 1, 2 or 3')}

  A<-tri[1,]; B<-tri[2,]; C<-tri[3,];

  if (cent==3)
  {
    D2<-A+(2-r)*(C-A)
    D1<-B+(2-r)*(C-B)
    D3<-(A+B)/2
  }
  else
  {
    if (cent==1)
    {
      D3<-B+(2-r)*(A-B)
      D2<-C+(2-r)*(A-C)
      D1<-(B+C)/2
    }
    else
    {
      D3<-A+(2-r)*(B-A)
      D1<-C+(2-r)*(B-C)
      D2<-(A+C)/2
    }
  }
  Ds<-rbind(D1,D2,D3)
  row.names(Ds)<-c()

  Ds
} #end of the function
#'

#################################################################

#' @title The indices of the vertex regions in a triangle that contains the points in a give data set
#'
#' @description Returns the indices of the vertices whose regions contain the points in data set \code{Dt} in
#' a triangle \code{tri}\eqn{=(A,B,C)} and vertex regions are based on the center \code{cent} which yields nondegenerate asymptotic
#' distribution of the domination number of PE-PCD for uniform data in \code{tri} for expansion parameter \code{r} in \eqn{(1,1.5]}.
#'
#' Vertices of triangle \code{tri} are labeled as \eqn{1,2,3} according to the row number the vertex is recorded
#' if a point in \code{Dt} is not inside \code{tri}, then the function yields \code{NA} as output for that entry.
#' The corresponding vertex region is the polygon with the vertex, \code{cent}, and projection points on the edges.
#' The center label \code{cent} values \code{1,2,3} correspond to the vertices \eqn{M_1}, \eqn{M_2}, and \eqn{M_3}; with default 1
#' (see the examples for an illustration).
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:dom-num-NPE-Spat2011,ceyhan:comp-geo-2010,ceyhan:mcap2012;textual}{pcds}).
#'
#' @param Dt A set of 2D points representing the set of data points for which indices of the vertex regions
#' containing them are to be determined.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be in \eqn{(1,1.5]} for this function.
#' @param cent Index of the center (as \eqn{1,2,3} corresponding to \eqn{M_1,\,M_2,\,M_3}) which gives nondegenerate asymptotic
#' distribution of the domination number of PE-PCD for uniform data in \code{tri} for expansion parameter \code{r} in \eqn{(1,1.5]};
#' default \code{cent=1}.
#'
#' @return A \code{list} with two elements
#' \item{rv}{Indices (i.e., a \code{vector} of indices) of the vertices whose region contains points in \code{Dt}
#' in the triangle \code{tri}}
#' \item{tri}{The vertices of the triangle, where row number corresponds to the vertex index in \code{rv}.}
#'
#' @seealso \code{\link{rverts.triCM}}, \code{\link{rverts.triCC}} and \code{\link{rverts.tri.cent}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#' r<-1.35
#' cent<-2
#'
#' P<-c(1.4,1.0)
#' rverts.tri.nd(P,Tr,r,cent)
#'
#' P<-c(1.8,.5)
#' rverts.tri.nd(P,Tr,r,cent)
#'
#' P<-c(10.5,1.6)
#' rverts.tri.nd(P,Tr,r,cent)
#'
#' n<-10  #try also n<-20
#' set.seed(1)
#' dat<-runif.tri(n,Tr)$g
#'
#' rverts.tri.nd(dat,Tr,r,cent)
#' rverts.tri.nd(rbind(dat,c(2,2)),Tr,r,cent)
#'
#' rv<-rverts.tri.nd(dat,Tr,r,cent)
#'
#' M<-cent.nondeg(Tr,r)[cent,];
#' Ds<-cp2edges.nd(Tr,r,cent)
#'
#' Xlim<-range(Tr[,1],dat[,1])
#' Ylim<-range(Tr[,2],dat[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(Tr,pch=".",xlab="",ylab="",axes=TRUE,xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tr)
#' points(dat,pch=".",col=1)
#' L<-rbind(M,M,M); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#'
#' xc<-Tr[,1]+c(-.03,.05,.05)
#' yc<-Tr[,2]+c(-.06,.02,.05)
#' txt.str<-c("rv=1","rv=2","rv=3")
#' text(xc,yc,txt.str)
#'
#' txt<-rbind(M,Ds)
#' xc<-txt[,1]+c(.02,.04,-.03,0)
#' yc<-txt[,2]+c(.07,.03,.05,-.07)
#' txt.str<-c("M","D1","D2","D3")
#' text(xc,yc,txt.str)
#'
#' text(dat,labels=factor(rv$rv))
#'
#' P<-c(1.4,1.0)
#' rverts.tri.nd(P,Tr,r,cent)
#' rverts.tri.nd(dat,Tr,r,cent)
#'
#' rverts.tri.nd(rbind(dat,dat),Tr,r,cent)
#'
#' dat.fr<-data.frame(a=dat)
#' rverts.tri.nd(dat.fr,Tr,r,1)
#'
#' dat.fr<-data.frame(a=Tr)
#' rverts.tri.nd(dat,dat.fr,r,1)
#'
#' @export rverts.tri.nd
rverts.tri.nd <- function(Dt,tri,r,cent=1)
{
  if (!is.numeric(as.matrix(Dt)))
  {stop('Dt must be numeric')}

  if (is.point(Dt))
  { Dt<-matrix(Dt,ncol=2)
  } else
  {Dt<-as.matrix(Dt)
  if (ncol(Dt)!=2 )
  {stop('Dt must be of dimension nx2')}
  }

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  if (!is.point(r,1) || r<=1 || r>1.5)
  {stop('r must be a scalar in (1,1.5]')}

  if (cent!=1 & cent!=2 & cent!=3)
  {stop('center index, cent, must be 1, 2 or 3')}

  Dt<-matrix(Dt,ncol=2)
  nt<-nrow(Dt)

  if (nt==0)
  {ind.set<-NA}
  else
  {
    A<-tri[1,]; B<-tri[2,]; C<-tri[3,];
    M<-cent.nondeg(tri,r)[cent,];
    Ds<-cp2edges.nd(tri,r,cent)
    D1<-Ds[1,]; D2<-Ds[2,]; D3<-Ds[3,];

    t.mA<-interp::tri.mesh(c(A[1],D3[1],M[1],D2[1]),c(A[2],D3[2],M[2],D2[2]),duplicate="remove")
    t.mB<-interp::tri.mesh(c(B[1],D1[1],M[1],D3[1]),c(B[2],D1[2],M[2],D3[2]),duplicate="remove")
    t.mC<-interp::tri.mesh(c(C[1],D2[1],M[1],D1[1]),c(C[2],D2[2],M[2],D1[2]),duplicate="remove")
   ind.set<-rep(NA,nt)
   ind.vA<-interp::in.convex.hull(t.mA,Dt[,1],Dt[,2])
   ind.vB<-interp::in.convex.hull(t.mB,Dt[,1],Dt[,2])
   ind.vC<-interp::in.convex.hull(t.mC,Dt[,1],Dt[,2])

   ind.set[ind.vA==TRUE]<-1
   ind.set[ind.vB==TRUE]<-2
   ind.set[ind.vC==TRUE]<-3
  }
  row.names(tri)<-c("vertex 1","vertex 2","vertex 3")  #vertex labeling

  list(rv=ind.set, #relative vertices
       tri=tri #vertex labeling
  )
} #end of the function
#'

#################################################################

#' @title The alternative function for the indices of the vertex regions in a triangle that contains the points
#' in a give data set
#'
#' @description An alternative function to the function \code{\link{rverts.tri.cent}}
#'
#' @param Dt A set of 2D points representing the set of data points for which indices of the vertex regions
#' containing them are to be determined.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param M A 2D point in Cartesian coordinates or a 3D point in barycentric coordinates
#' which serves as a center in the interior of the triangle \code{tri} or the circumcenter of \code{tri}.
#'
#' @return A \code{list} with two elements
#' \item{rv}{Indices of the vertices whose regions contains points in \code{Dt}.}
#' \item{tri}{The vertices of the triangle, where row number corresponds to the vertex index in \code{rv}.}
#'
#' @seealso \code{\link{rverts.tri.cent}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#' M<-c(1.6,1.0)
#'
#' P<-c(.4,.2)
#' rverts.tri.cent.alt(P,Tr,M)
#'
#' P<-c(1.8,.5)
#' rverts.tri.cent.alt(P,Tr,M)
#'
#' P<-c(1.5,1.6)
#' rverts.tri.cent.alt(P,Tr,M)
#'
#' n<-10  #try also n<-20
#' set.seed(1)
#' dat<-runif.tri(n,Tr)$g
#'
#' M<-c(1.6,1.0)  #try also M<-c(1.3,1.3)
#'
#' rv<-rverts.tri.cent.alt(dat,Tr,M)
#' rv
#'
#' Ds<-cp2e.tri(Tr,M)
#'
#' Xlim<-range(Tr[,1])
#' Ylim<-range(Tr[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(Tr,pch=".",xlab="",ylab="",axes=TRUE,xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tr)
#' points(dat,pch=".",col=1)
#' L<-rbind(M,M,M); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#'
#' xc<-Tr[,1]+c(-.03,.05,.05)
#' yc<-Tr[,2]+c(-.06,.02,.05)
#' txt.str<-c("rv=1","rv=2","rv=3")
#' text(xc,yc,txt.str)
#'
#' txt<-rbind(M,Ds)
#' xc<-txt[,1]+c(.02,.04,-.03,0)
#' yc<-txt[,2]+c(.07,.03,.05,-.07)
#' txt.str<-c("M","D1","D2","D3")
#' text(xc,yc,txt.str)
#'
#' text(dat,labels=factor(rv$rv))
#'
#' P<-c(1.4,1.0)
#' rverts.tri.cent.alt(P,Tr,M)
#' rverts.tri.cent.alt(dat,Tr,M)
#'
#' rverts.tri.cent.alt(rbind(dat,dat),Tr,M)
#'
#' dat.fr<-data.frame(a=dat)
#' rverts.tri.cent.alt(dat.fr,Tr,M)
#'
#' dat.fr<-data.frame(a=Tr)
#' rverts.tri.cent.alt(dat,dat.fr,M)
#'
#' @export
rverts.tri.cent.alt <- function(Dt,tri,M)
{
  if (!is.numeric(as.matrix(Dt)))
  {stop('Dt must be numeric')}

  if (is.point(Dt))
  { Dt<-matrix(Dt,ncol=2)
  } else
  {Dt<-as.matrix(Dt)
  if (ncol(Dt)!=2 )
  {stop('Dt must be of dimension nx2')}
  }

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  if (!is.point(M))
  {stop('M must be a numeric 2D point')}

  if (in.triangle(M,tri,boundary=FALSE)$in.tri==F)
  {stop('center is not in the interior of the basic triangle')}

  nt<-nrow(Dt)

  if (nt==0)
  {ind.set<-NA}
  else
  {
    rv<-rep(NA,nt)
    for (i in 1:nt)
    { if (in.triangle(Dt[i,],tri,boundary=TRUE)$in.tri)
      rv[i]<-rv.tri.cent(Dt[i,],tri,M)$rv
    }
  }
  row.names(tri)<-c("vertex 1","vertex 2","vertex 3")  #vertex labeling

  list(rv=rv, #relative vertices
       tri=tri #vertex labeling
  )
} #end of the function
#'

#################################################################

#' @title The closest points in a data set to edges in each \eqn{CM}-vertex region in a triangle
#'
#' @description
#' An object of class \code{"Extrema"}.
#' Returns the closest data points among the data set, \code{dat}, to edge \eqn{j} in \eqn{CM}-vertex region \eqn{j} for \eqn{j=1,2,3}
#' in the triangle, \code{tri}\eqn{=T(A,B,C)}, where \eqn{CM} stands for center of mass. Vertex labels are \eqn{A=1}, \eqn{B=2}, and \eqn{C=3},
#' and corresponding edge labels are \eqn{BC=1}, \eqn{AC=2}, and \eqn{AB=3}.
#' Function yields \code{NA} if there are no data points in a \eqn{CM}-vertex region.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:masa-2007,ceyhan:comp-geo-2010,ceyhan:dom-num-NPE-Spat2011;textual}{pcds}).
#'
#' @param dat A set of 2D points representing the set of data points.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#'
#' @return A \code{list} with the elements
#' \item{txt1}{Vertex labels are \eqn{A=1}, \eqn{B=2}, and \eqn{C=3} (corresponds to row number in Extremum Points).}
#' \item{txt2}{A short description of the distances as \code{"Distances to Edges in the Respective
#' CM-Vertex Regions"}.}
#' \item{type}{Type of the extrema points}
#' \item{desc}{A short description of the extrema points}
#' \item{mtitle}{The \code{"main"} title for the plot of the extrema}
#' \item{ext}{The extrema points, here, closest points to edges in the respective vertex region.}
#' \item{X}{The input data, \code{dat}, can be a \code{matrix} or \code{data frame}}
#' \item{num.points}{The number of data points, i.e., size of \code{dat}}
#' \item{supp}{Support of the data points, here, it is \code{tri}}
#' \item{cent}{The center point used for construction of vertex regions}
#' \item{ncent}{Name of the center, \code{cent}, it is \code{"CM"} for this function}
#' \item{regions}{Vertex regions inside the triangle, \code{tri}, provided as a list}
#' \item{region.names}{Names of the vertex regions as \code{"vr=1"}, \code{"vr=2"}, and \code{"vr=3"}}
#' \item{region.centers}{Centers of mass of the vertex regions inside \code{tri}}
#' \item{dist2ref}{Distances of closest points in the vertex regions to corresponding edges}
#'
#' @seealso \code{\link{cl2eTbVRcent}}, \code{\link{cl2eVRCC}}, \code{\link{cl2eVRcent}},
#' and \code{\link{cl2edgesTe}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#'
#' n<-10  #try also n<-20
#' set.seed(1)
#' dat<-runif.tri(n,Tr)$g
#'
#' Ext<-cl2eVRCM(dat,Tr)
#' Ext
#' summary(Ext)
#' plot(Ext)
#'
#' cl2eVRCM(dat[1,],Tr)
#'
#' cl2e<-cl2eVRCM(dat,Tr)
#' cl2e
#'
#' CM<-(A+B+C)/3;
#' D1<-(B+C)/2; D2<-(A+C)/2; D3<-(A+B)/2;
#' Ds<-rbind(D1,D2,D3)
#'
#' Xlim<-range(Tr[,1],dat[,1])
#' Ylim<-range(Tr[,2],dat[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(Tr,pch=".",xlab="",ylab="",axes=TRUE,xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tr)
#'
#' xc<-Tr[,1]+c(-.02,.02,.02)
#' yc<-Tr[,2]+c(.02,.02,.04)
#' txt.str<-c("A","B","C")
#' text(xc,yc,txt.str)
#'
#' points(dat,pch=1,col=1)
#' L<-matrix(rep(CM,3),ncol=2,byrow=TRUE); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#' points(cl2e$ext,pch=3,col=2)
#'
#' txt<-rbind(CM,Ds)
#' xc<-txt[,1]+c(-.04,.04,-.03,0)
#' yc<-txt[,2]+c(-.05,.04,.06,-.08)
#' txt.str<-c("CM","D1","D2","D3")
#' text(xc,yc,txt.str)
#'
#' P<-c(1.4,1.0)
#' cl2eVRCM(P,Tr)
#' cl2eVRCM(dat,Tr)
#'
#' cl2eVRCM(rbind(dat,dat),Tr)
#'
#' dat.fr<-data.frame(a=dat)
#' cl2eVRCM(dat.fr,Tr)
#'
#' dat.fr<-data.frame(a=Tr)
#' cl2eVRCM(dat,dat.fr)
#'
#' @export cl2eVRCM
cl2eVRCM <- function(dat,tri)
{
  if (!is.numeric(as.matrix(dat)))
  {stop('dat must be numeric')}

  if (is.point(dat))
  { dat<-matrix(dat,ncol=2)
  } else
  {dat<-as.matrix(dat)
  if (ncol(dat)!=2 )
  {stop('dat must be of dimension nx2')}
  }

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  n<-nrow(dat)

  A<-tri[1,]; B<-tri[2,]; C<-tri[3,]
  Cent<-(A+B+C)/3; Cname<-"CM"
  D1<-(B+C)/2; D2<-(A+C)/2; D3<-(A+B)/2;

  clBC<-NA; clAC<-NA; clAB<-NA
  VRdt<-rverts.triCM(dat,tri)$rv
  dtA<-matrix(dat[VRdt==1 & !is.na(VRdt),],ncol=2);
  dtB<-matrix(dat[VRdt==2 & !is.na(VRdt),],ncol=2);
  dtC<-matrix(dat[VRdt==3 & !is.na(VRdt),],ncol=2)
  distA<-distB<-distC<-vector()
  nA<-nrow(dtA); nB<-nrow(dtB); nC<-nrow(dtC);

  if (nA>0)
  {
    for (i in 1:nA)
    {distA<-c(distA,dp2l(dtA[i,],B,C)$dis)};
    clBC<-dtA[distA==min(distA),]
  }

  if (nB>0)
  {
    for (i in 1:nB)
    {distB<-c(distB,dp2l(dtB[i,],A,C)$dis)};
    clAC<-dtB[distB==min(distB),]
  }

  if (nC>0)
  {
    for (i in 1:nC)
    {distC<-c(distC,dp2l(dtC[i,],A,B)$dis)};
    clAB<-dtC[distC==min(distC),]
  }
  ce<-rbind(clBC,clAC,clAB)
  row.names(ce)<-c()

  Dis<-rbind(ifelse(is.numeric(distA),min(distA),NA),ifelse(is.numeric(distB),min(distB),NA),
             ifelse(is.numeric(distC),min(distC),NA))

  row.names(tri)<-c("A","B","C")  #vertex labeling
  txt<-"Edge labels are AB=3, BC=1, and AC=2 (corresponds to row number in ce)"

  row.names(tri)<-c("A","B","C")  #vertex labeling
  typ<-"Closest Points to Edges in the Respective CM-Vertex Regions in the Triangle with Vertices A, B, and C"
  txt1<-"Vertex labels are A=1, B=2, and C=3 (corresponds to row number in Extremum Points)"
  txt2<-"Distances to Edges in the Respective CM-Vertex Regions"
  description<-"Closest Points to Edges in the Respective CM-Vertex Regions \n (Row i corresponds to vertex i for i=1,2,3)"
  main.txt<-paste("Closest Points to Edges in\n Respective ",Cname,"-Vertex Regions",sep="")

  Dis<-c(ifelse(is.numeric(distA),min(distA),NA),ifelse(is.numeric(distB),min(distB),NA),
         ifelse(is.numeric(distC),min(distC),NA))
  #distances of the closest points to the edges in the respective vertex regions
  Regs<-list(vr1=rbind(A,D3,Cent,D2), #regions inside the triangles
             vr2=rbind(B,D1,Cent,D3),
             vr3=rbind(C,D2,Cent,D1)
  )
  Reg.Cent<-vector()
  for (i in 1:length(Regs))
  { Reg.Cent<-rbind(Reg.Cent,apply(Regs[[i]],2,mean))}

  Reg.names<-c("vr=1","vr=2","vr=3")  #regions names

  res<-list(
    txt1=txt1, txt2=txt2,
    type=typ, desc=description,
    mtitle=main.txt, #main label in the plot
    ext=ce,  #closest points to edges in each associated vertex region
    X=dat, num.points=n, #data points and its size
    ROI=tri, # region of interest for X points
    cent=Cent, ncent=Cname, #center and center name
    regions=Regs, region.names=Reg.names, region.centers=Reg.Cent,
    dist2ref=Dis #distances of closest points in vertex regions to the corresponding edges
  )

  class(res)<-"Extrema"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

#' @title The closest points in a data set to edges in each \eqn{CC}-vertex region in a triangle
#'
#' @description
#' An object of class \code{"Extrema"}.
#' Returns the closest data points among the data set, \code{dat}, to edge \eqn{j} in \eqn{CC}-vertex region \eqn{j} for \eqn{j=1,2,3}
#' in the triangle, \code{tri}\eqn{=T(A,B,C)}, where \eqn{CC} stands for circumcenter. Vertex labels are \eqn{A=1}, \eqn{B=2}, and \eqn{C=3},
#' and corresponding edge labels are \eqn{BC=1}, \eqn{AC=2}, and \eqn{AB=3}.
#' Function yields \code{NA} if there are no data points in a \eqn{CC}-vertex region.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010;textual}{pcds}).
#'
#' @param dat A set of 2D points representing the set of data points.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#'
#' @return A \code{list} with the elements
#' \item{txt1}{Vertex labels are \eqn{A=1}, \eqn{B=2}, and \eqn{C=3} (corresponds to row number in Extremum Points).}
#' \item{txt2}{A short description of the distances as \code{"Distances to Edges in the Respective
#' CC-Vertex Regions"}.}
#' \item{type}{Type of the extrema points}
#' \item{desc}{A short description of the extrema points}
#' \item{mtitle}{The \code{"main"} title for the plot of the extrema}
#' \item{ext}{The extrema points, here, closest points to edges in the respective vertex region.}
#' \item{X}{The input data, \code{dat}, can be a \code{matrix} or \code{data frame}}
#' \item{num.points}{The number of data points, i.e., size of \code{dat}}
#' \item{supp}{Support of the data points, here, it is \code{tri}}
#' \item{cent}{The center point used for construction of vertex regions}
#' \item{ncent}{Name of the center, \code{cent}, it is \code{"CC"} for this function}
#' \item{regions}{Vertex regions inside the triangle, \code{tri}, provided as a list}
#' \item{region.names}{Names of the vertex regions as \code{"vr=1"}, \code{"vr=2"}, and \code{"vr=3"}}
#' \item{region.centers}{Centers of mass of the vertex regions inside \code{tri}}
#' \item{dist2ref}{Distances of closest points in the vertex regions to corresponding edges}
#'
#' @seealso \code{\link{cl2eTbVRcent}}, \code{\link{cl2eVRCM}}, \code{\link{cl2eVRcent}},
#' and \code{\link{cl2edgesTe}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#'
#' n<-10  #try also n<-20
#' set.seed(1)
#' dat<-runif.tri(n,Tr)$g
#'
#' Ext<-cl2eVRCC(dat,Tr)
#' Ext
#' summary(Ext)
#' plot(Ext)
#'
#' cl2eVRCC(dat[1,],Tr)
#'
#' cl2e<-cl2eVRCC(dat,Tr)
#' cl2e
#'
#' CC<-circ.cent.tri(Tr);
#' D1<-(B+C)/2; D2<-(A+C)/2; D3<-(A+B)/2;
#' Ds<-rbind(D1,D2,D3)
#'
#' Xlim<-range(Tr[,1],dat[,1],CC[1])
#' Ylim<-range(Tr[,2],dat[,2],CC[2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(Tr,asp=1,pch=".",xlab="",ylab="",axes=TRUE,xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tr)
#'
#' xc<-Tr[,1]+c(-.02,.02,.02)
#' yc<-Tr[,2]+c(.02,.02,.04)
#' txt.str<-c("A","B","C")
#' text(xc,yc,txt.str)
#'
#' points(dat,pch=1,col=1)
#' L<-matrix(rep(CC,3),ncol=2,byrow=TRUE); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#' points(cl2e$ext,pch=3,col=2)
#'
#' txt<-rbind(CC,Ds)
#' xc<-txt[,1]+c(-.04,.04,-.03,0)
#' yc<-txt[,2]+c(-.05,.04,.06,-.08)
#' txt.str<-c("CC","D1","D2","D3")
#' text(xc,yc,txt.str)
#'
#' P<-c(1.4,1.0)
#' cl2eVRCC(P,Tr)
#' cl2eVRCC(dat,Tr)
#'
#' cl2eVRCC(rbind(dat,dat),Tr)
#'
#' dat.fr<-data.frame(a=dat)
#' cl2eVRCC(dat.fr,Tr)
#'
#' dat.fr<-data.frame(a=Tr)
#' cl2eVRCC(dat,dat.fr)
#'
#' @export cl2eVRCC
cl2eVRCC <- function(dat,tri)
{
  if (!is.numeric(as.matrix(dat)))
  {stop('dat must be numeric')}

  if (is.point(dat))
  { dat<-matrix(dat,ncol=2)
  } else
  {dat<-as.matrix(dat)
  if (ncol(dat)!=2 )
  {stop('dat must be of dimension nx2')}
  }

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  n<-nrow(dat)

  A<-tri[1,]; B<-tri[2,]; C<-tri[3,]
  Cent<-circ.cent.tri(tri) ; Cname<-"CC"
  D1<-(B+C)/2; D2<-(A+C)/2; D3<-(A+B)/2;

  clBC<-NA; clAC<-NA; clAB<-NA
  VRdt<-rverts.triCC(dat,tri)$rv
  dtA<-matrix(dat[VRdt==1 & !is.na(VRdt),],ncol=2);
  dtB<-matrix(dat[VRdt==2 & !is.na(VRdt),],ncol=2);
  dtC<-matrix(dat[VRdt==3 & !is.na(VRdt),],ncol=2)
  distA<-distB<-distC<-vector()
  nA<-nrow(dtA); nB<-nrow(dtB); nC<-nrow(dtC);

  if (nA>0)
  {
    for (i in 1:nA)
    {distA<-c(distA,dp2l(dtA[i,],B,C)$dis)};
    clBC<-dtA[distA==min(distA),]
  }

  if (nB>0)
  {
    for (i in 1:nB)
    {distB<-c(distB,dp2l(dtB[i,],A,C)$dis)};
    clAC<-dtB[distB==min(distB),]
  }

  if (nC>0)
  {
    for (i in 1:nC)
    {distC<-c(distC,dp2l(dtC[i,],A,B)$dis)};
    clAB<-dtC[distC==min(distC),]
  }
  ce<-rbind(clBC,clAC,clAB)

  Dis<-rbind(ifelse(is.numeric(distA),min(distA),NA),ifelse(is.numeric(distB),min(distB),NA),
             ifelse(is.numeric(distC),min(distC),NA))

  row.names(tri)<-c("A","B","C")  #vertex labeling
  txt<-"Edge labels are AB=3, BC=1, and AC=2 (corresponds to row number in ce)"

  row.names(tri)<-c("A","B","C")  #vertex labeling
  typ<-"Closest Points to Edges in the Respective CC-Vertex Regions in the Triangle with Vertices A, B, and C"
  txt1<-"Vertex labels are A=1, B=2, and C=3 (corresponds to row number in Extremum Points)"
  txt2<-"Distances to Edges in the Respective CC-Vertex Regions"
  description<-"Closest Points to Edges in the Respective CC-Vertex Regions \n (Row i corresponds to vertex i for i=1,2,3)"
  main.txt<-paste("Closest Points to Edges in\n Respective ",Cname,"-Vertex Regions",sep="")

  Dis<-rbind(ifelse(is.numeric(distA),min(distA),NA),ifelse(is.numeric(distB),min(distB),NA),
             ifelse(is.numeric(distC),min(distC),NA))
  #distances of the closest points to the edges in the respectivevertex regions
  if (in.triangle(Cent,tri,boundary = F)$i)
  {
    Regs<-list(vr1=rbind(A,D3,Cent,D2), #regions inside the triangles
               vr2=rbind(B,D1,Cent,D3),
               vr3=rbind(C,D2,Cent,D1))
    Reg.names<-c("vr=1","vr=2","vr=3")  #regions names
  } else
  { a1<-A[1]; a2<-A[2]; b1<-B[1]; b2<-B[2]; c1<-C[1]; c2<-C[2];
  dAB<-Dist(A,B); dBC<-Dist(B,C); dAC<-Dist(A,C); max.dis<-max(dAB,dBC,dAC)
  if (dAB==max.dis)
  {
    L1<-c((1/2)*(a1^3-a1^2*b1+a1*a2^2-2*a1*a2*b2+2*a1*b2*c2-a1*c1^2-a1*c2^2+a2^2*b1-2*a2*b1*c2+b1*c1^2+b1*c2^2)/(a1^2-a1*b1-a1*c1+a2^2-a2*b2-a2*c2+b1*c1+b2*c2), (1/2)*(a1^2*a2+a1^2*b2-2*a1*a2*b1-2*a1*b2*c1+a2^3-a2^2*b2+2*a2*b1*c1-a2*c1^2-a2*c2^2+b2*c1^2+b2*c2^2)/(a1^2-a1*b1-a1*c1+a2^2-a2*b2-a2*c2+b1*c1+b2*c2))
    L2<-c((1/2)*(a1*b1^2-a1*b2^2+2*a1*b2*c2-a1*c1^2-a1*c2^2+2*a2*b1*b2-2*a2*b1*c2-b1^3-b1*b2^2+b1*c1^2+b1*c2^2)/(a1*b1-a1*c1+a2*b2-a2*c2-b1^2+b1*c1-b2^2+b2*c2), (1/2)*(2*a1*b1*b2-2*a1*b2*c1-a2*b1^2+2*a2*b1*c1+a2*b2^2-a2*c1^2-a2*c2^2-b1^2*b2-b2^3+b2*c1^2+b2*c2^2)/(a1*b1-a1*c1+a2*b2-a2*c2-b1^2+b1*c1-b2^2+b2*c2))
    Regs<-list(vr1=rbind(A,L1,D2), #regions inside the triangles
               vr2=rbind(B,D1,L2),
               vr3=rbind(C,D2,L1,L2,D1),
               r4=rbind(Cent,L1,L2))  #only r4 is outside the triangle
  }

  if (dBC==max.dis)
  {
    L1<-c((1/2)*(a1^2*b1-a1^2*c1+a2^2*b1-a2^2*c1-2*a2*b1*c2+2*a2*b2*c1-b1^3+b1^2*c1-b1*b2^2+2*b1*b2*c2-b2^2*c1)/(a1*b1-a1*c1+a2*b2-a2*c2-b1^2+b1*c1-b2^2+b2*c2), (1/2)*(a1^2*b2-a1^2*c2+2*a1*b1*c2-2*a1*b2*c1+a2^2*b2-a2^2*c2-b1^2*b2-b1^2*c2+2*b1*b2*c1-b2^3+b2^2*c2)/(a1*b1-a1*c1+a2*b2-a2*c2-b1^2+b1*c1-b2^2+b2*c2))
    L2<-c((1/2)*(a1^2*b1-a1^2*c1+a2^2*b1-a2^2*c1-2*a2*b1*c2+2*a2*b2*c1-b1*c1^2+b1*c2^2-2*b2*c1*c2+c1^3+c1*c2^2)/(a1*b1-a1*c1+a2*b2-a2*c2-b1*c1-b2*c2+c1^2+c2^2), (1/2)*(a1^2*b2-a1^2*c2+2*a1*b1*c2-2*a1*b2*c1+a2^2*b2-a2^2*c2-2*b1*c1*c2+b2*c1^2-b2*c2^2+c1^2*c2+c2^3)/(a1*b1-a1*c1+a2*b2-a2*c2-b1*c1-b2*c2+c1^2+c2^2))
    Regs<-list(vr1=rbind(A,D3,L1,L2,D2), #regions inside the triangles
               vr2=rbind(B,L1,D3),
               vr3=rbind(C,D2,L2),
               r4=rbind(Cent,L1,L2))  #only r4 is outside the triangle
  }

  if (dAC==max.dis)
  {
    L1<-c((1/2)*(a1^3-a1^2*c1+a1*a2^2-2*a1*a2*c2-a1*b1^2-a1*b2^2+2*a1*b2*c2+a2^2*c1-2*a2*b2*c1+b1^2*c1+b2^2*c1)/(a1^2-a1*b1-a1*c1+a2^2-a2*b2-a2*c2+b1*c1+b2*c2), (1/2)*(a1^2*a2+a1^2*c2-2*a1*a2*c1-2*a1*b1*c2+a2^3-a2^2*c2-a2*b1^2+2*a2*b1*c1-a2*b2^2+b1^2*c2+b2^2*c2)/(a1^2-a1*b1-a1*c1+a2^2-a2*b2-a2*c2+b1*c1+b2*c2))
    L2<-c((1/2)*(a1*b1^2+a1*b2^2-2*a1*b2*c2-a1*c1^2+a1*c2^2+2*a2*b2*c1-2*a2*c1*c2-b1^2*c1-b2^2*c1+c1^3+c1*c2^2)/(a1*b1-a1*c1+a2*b2-a2*c2-b1*c1-b2*c2+c1^2+c2^2), (1/2)*(2*a1*b1*c2-2*a1*c1*c2+a2*b1^2-2*a2*b1*c1+a2*b2^2+a2*c1^2-a2*c2^2-b1^2*c2-b2^2*c2+c1^2*c2+c2^3)/(a1*b1-a1*c1+a2*b2-a2*c2-b1*c1-b2*c2+c1^2+c2^2))
    Regs<-list(vr1=rbind(A,D3,L1), #regions inside the triangles
               vr2=rbind(B,D1,L2,L1,D3),
               vr3=rbind(C,L2,D1),
               r4=rbind(Cent,L1,L2))  #only r4 is outside the triangle
  }
  Reg.names<-c("vr=1","vr=2","vr=3",NA)  #regions names
  }
  Reg.Cent<-vector()
  for (i in 1:length(Regs))
  { Reg.Cent<-rbind(Reg.Cent,apply(Regs[[i]],2,mean))}

  res<-list(
    txt1=txt1, txt2=txt2,
    type=typ, desc=description,
    mtitle=main.txt, #main label in the plot
    ext=ce,  #closest points to edges in each associated vertex region
    X=dat, num.points=n, #data points and its size
    ROI=tri, # region of interest for X points
    cent=Cent, ncent=Cname, #center and center name
    regions=Regs, region.names=Reg.names, region.centers=Reg.Cent,
    dist2ref=Dis #distances of closest points in vertex regions to the corresponding edges
  )

  class(res)<-"Extrema"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

#' @title The vertices of the Proportional Edge (PE) Proximity Region in a general triangle
#'
#' @description Returns the vertices of the PE proximity region (which is itself a triangle) for a point in the
#' triangle \code{tri}\eqn{=T(A,B,C)=}\code{(rv=1,rv=2,rv=3)}.
#'
#' PE proximity region is defined with respect to the triangle \code{tri}
#' with expansion parameter \eqn{r \ge 1} and vertex regions based on center \eqn{M=(m_1,m_2)} in Cartesian coordinates or
#' \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the interior of the triangle \code{tri}
#' or based on the circumcenter of \code{tri}; default is \eqn{M=(1,1,1)} i.e., the center of mass of \code{tri}.
#'
#' Vertex regions are labeled as \eqn{1,2,3} rowwise for the vertices
#' of the triangle \code{tri}. \code{rv} is the index of the vertex region \code{pt} resides, with default=\code{NULL}.
#' If \code{pt} is outside of \code{tri}, it returns \code{NULL} for the proximity region.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:arc-density-PE,ceyhan:dom-num-NPE-Spat2011;textual}{pcds}).
#'
#' @param pt A 2D point whose PE proximity region is to be computed.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param M A 2D point in Cartesian coordinates or a 3D point in barycentric coordinates
#' which serves as a center in the interior of the triangle \code{tri} or the circumcenter of \code{tri};
#' default is \eqn{M=(1,1,1)} i.e., the center of mass of \code{tri}.
#' @param rv Index of the \code{M}-vertex region containing the point \code{pt}, either \code{1,2,3} or \code{NULL}
#' (default is \code{NULL}).
#'
#' @return Vertices of the triangular region which constitutes the PE proximity region with expansion parameter
#' \code{r} and center \code{M} for a point \code{pt}
#'
#' @seealso \code{\link{NPEbastri}}, \code{\link{NAStri}}, \code{\link{NCStri}}
#' and \code{\link{IndNPEtri}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#'
#' M<-as.numeric(runif.tri(1,Tr)$g)  #try also M<-c(1.6,1.0)
#'
#' r<-1.5
#'
#' n<-10
#' set.seed(1)
#' dat<-runif.tri(n,Tr)$g
#'
#' NPEtri(dat[7,],r,Tr,M)
#'
#' P1<-as.numeric(runif.tri(1,Tr)$g)  #try also P1<-c(.4,.2)
#' NPEtri(P1,r,Tr,M)
#'
#' P2<-c(1.8,.5)
#' NPEtri(P2,r,Tr,M)
#'
#' P3<-c(1.7,.6)
#' NPEtri(P3,r,Tr,M)
#'
#' M<-c(1.3,1.3)
#' r<-2
#'
#' P1<-c(1.4,1.2)
#' P2<-c(1.5,1.26)
#' NPEtri(P1,r,Tr,M)
#' NPEtri(P2,r,Tr,M)
#'
#' #or try
#' Rv<-rv.tri.cent(P1,Tr,M)$rv
#' NPEtri(P1,r,Tr,M,Rv)
#'
#' dat.fr<-data.frame(a=Tr)
#' NPEtri(P2,r,dat.fr,M)
#'
#' @export NPEtri
NPEtri <- function(pt,r,tri,M=c(1,1,1),rv=NULL)
{
  if (!is.point(pt) )
  {stop('must be a numeric 2D point')}

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  if (!is.point(M) && !is.point(M,3) )
  {stop('M must be a numeric 2D point for Cartesian coordinates or 3D point for barycentric coordinates')}

  if (dimension(M)==3)
  {
    M<-bary2cart(M,tri)
  }

  if (isTRUE(all.equal(M,circ.cent.tri(tri)))==F & in.triangle(M,tri,boundary=FALSE)$in.tri==F)
  {stop('center is not the circumcenter or not in the interior of the triangle')}

  if (!in.triangle(pt,tri,boundary=TRUE)$in.tri)
  {reg<-NULL; return(reg); stop}

  if (is.null(rv))
  { rv<-ifelse(isTRUE(all.equal(M,circ.cent.tri(tri)))==T,rv.triCC(pt,tri)$rv,rv.tri.cent(pt,tri,M)$rv)  #vertex region for pt
  } else
  {  if (!is.numeric(rv) || sum(rv==c(1,2,3))!=1)
  {stop('vertex index, rv, must be 1, 2 or 3')}}

  A<-tri[1,];  B<-tri[2,];  C<-tri[3,];
  if (rv==1)
  {
    d1<-dp2l(pt,B,C)$dis
    d2<-dp2l(A,B,C)$dis
    sr<-d1/d2
    P1<-B+sr*(A-B); P2<-C+sr*(A-C)
    A1<-A+r*(P1-A)
    A2<-A+r*(P2-A)
    reg<-rbind(A,A1,A2)
  } else if (rv==2)
  {
    d1<-dp2l(pt,A,C)$dis
    d2<-dp2l(B,A,C)$dis
    sr<-d1/d2
    P1<-A+sr*(B-A); P2<-C+sr*(B-C)
    B1<-B+r*(P1-B)
    B2<-B+r*(P2-B)
    reg<-rbind(B,B1,B2)
  } else
  {
    d1<-dp2l(pt,A,B)$dis
    d2<-dp2l(C,A,B)$dis
    sr<-d1/d2
    P1<-A+sr*(C-A); P2<-B+sr*(C-B)
    C1<-C+r*(P1-C)
    C2<-C+r*(P2-C)
    reg<-rbind(C,C1,C2)
  }
  if (abs(area.polygon(reg))>abs(area.polygon(tri)))
  {reg<-tri}
  row.names(reg)<-c()
  reg
} #end of the function
#'

#################################################################

#' @title The indicator for the presence of an arc from a point to another for Proportional Edge Proximity
#' Catch Digraphs (PE-PCDs) - one triangle case
#'
#' @description Returns \eqn{I(}\code{pt2} is in \eqn{N_{PE}(pt1,r))} for points \code{pt1} and \code{pt2}, that is, returns 1 if \code{pt2} is in \eqn{N_{PE}(pt1,r)},
#' returns 0 otherwise, where \eqn{N_{PE}(x,r)} is the PE proximity region for point \eqn{x} with the expansion parameter \eqn{r \ge 1}.
#'
#' PE proximity region is constructed with respect to the triangle \code{tri} and
#' vertex regions are based on the center, \eqn{M=(m_1,m_2)} in Cartesian coordinates or
#' \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the interior of \code{tri}
#' or based on the circumcenter of \code{tri}; default is \eqn{M=(1,1,1)} i.e., the center of mass of \code{tri}.
#' \code{rv} is the index of the vertex region \code{pt1} resides, with default=\code{NULL}.
#'
#' If \code{pt1} and \code{pt2} are distinct and either of them are outside \code{tri}, it returns 0,
#' but if they are identical, then it returns 1 regardless of their locations (i.e., it allows loops).
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:arc-density-PE,ceyhan:dom-num-NPE-Spat2011;textual}{pcds}).
#'
#' @param pt1 A 2D point whose PE proximity region is constructed.
#' @param pt2 A 2D point. The function determines whether \code{pt2} is inside the PE proximity region of
#' \code{pt1} or not.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param M A 2D point in Cartesian coordinates or a 3D point in barycentric coordinates
#' which serves as a center in the interior of the triangle \code{tri} or the circumcenter of \code{tri};
#' default is \eqn{M=(1,1,1)} i.e., the center of mass of \code{tri}.
#' @param rv Index of the \code{M}-vertex region containing the point, either \code{1,2,3} or \code{NULL}
#' (default is \code{NULL}).
#'
#' @return \eqn{I(}\code{pt2} is in \eqn{N_{PE}(pt1,r))} for points \code{pt1} and \code{pt2}, that is, returns 1 if \code{pt2} is in \eqn{N_{PE}(pt1,r)},
#' returns 0 otherwise
#'
#' @seealso \code{\link{IndNPEbastri}}, \code{\link{IndNPETe}}, \code{\link{IndNAStri}}, and \code{\link{IndNCStri}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#'
#' M<-as.numeric(runif.tri(1,Tr)$g)  #try also M<-c(1.6,1.0);
#'
#' r<-1.5
#'
#' n<-10
#' set.seed(1)
#' dat<-runif.tri(n,Tr)$g
#'
#' IndNPEtri(dat[1,],dat[2,],r,Tr,M)
#'
#' P1<-as.numeric(runif.tri(1,Tr)$g)
#' P2<-as.numeric(runif.tri(1,Tr)$g)
#' IndNPEtri(P1,P2,r,Tr,M)
#'
#' P1<-c(.4,.2)
#' P2<-c(1.8,.5)
#' IndNPEtri(P1,P2,r,Tr,M)
#' IndNPEtri(P2,P1,r,Tr,M)
#' IndNPEtri(P1,P1,r,Tr,M)
#'
#' IndNPEtri(P2,P2,r,Tr,M)
#'
#' P3<-c(1.7,.6)
#' IndNPEtri(P2,P3,r,Tr,M)
#' IndNPEtri(P3,P2,r,Tr,M)
#'
#' M<-c(1.3,1.3)
#' r<-2
#'
#' P1<-c(1.4,1.2)
#' P2<-c(1.5,1.26)
#' IndNPEtri(P1,P2,r,Tr,M)
#' IndNPEtri(P2,P1,r,Tr,M)
#'
#' #or try
#' Rv<-rv.tri.cent(P1,Tr,M)$rv
#' IndNPEtri(P1,P2,r,Tr,M,Rv)
#'
#' P2<-c(1.8,.5)
#' P3<-c(1.7,.6)
#' IndNPEtri(P2,P3,r,Tr,M)
#'
#' dat.fr<-data.frame(a=Tr)
#' IndNPEtri(P2,P3,r,dat.fr,M)
#'
#' @export IndNPEtri
IndNPEtri <- function(pt1,pt2,r,tri,M=c(1,1,1),rv=NULL)
{
  if (!is.point(pt1) || !is.point(pt2))
  {stop('pt1 and pt2 must be numeric 2D points')}

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  if (!is.point(M) && !is.point(M,3) )
  {stop('M must be a numeric 2D point for Cartesian coordinates or 3D point for barycentric coordinates')}

  if (dimension(M)==3)
  {
    M<-bary2cart(M,tri)
  }

  if (isTRUE(all.equal(M,circ.cent.tri(tri)))==F & in.triangle(M,tri,boundary=FALSE)$in.tri==F)
  {stop('center is not the circumcenter or not in the interior of the triangle')}

  if (identical(pt1,pt2))
  {arc<-1; return(arc); stop}

  if (!in.triangle(pt1,tri,boundary=TRUE)$in.tri || !in.triangle(pt2,tri,boundary=TRUE)$in.tri)
  {arc<-0; return(arc); stop}

  if (is.null(rv))
  { rv<-ifelse(isTRUE(all.equal(M,circ.cent.tri(tri)))==T,rv.triCC(pt1,tri)$rv,rv.tri.cent(pt1,tri,M)$rv)  #vertex region for pt1
  } else
  {  if (!is.numeric(rv) || sum(rv==c(1,2,3))!=1)
  {stop('vertex index, rv, must be 1, 2 or 3')}}

  pr<-NPEtri(pt1,r,tri,M,rv)  #proximity region

  arc<-sum(in.triangle(pt2,pr,boundary=TRUE)$in.tri)
  arc
} #end of the function
#'

#################################################################

#' @title Number of arcs of Proportional Edge Proximity Catch Digraphs (PE-PCDs) - one triangle case
#'
#' @description Returns the number of arcs of Proportional Edge Proximity Catch Digraphs (PE-PCDs) whose vertices are the
#' given 2D numerical data set, \code{dat}.
#'
#' PE proximity region \eqn{N_{PE}(x,r)} is defined with respect to the triangle, \code{tri}
#' for this function. PE proximity region is constructed with expansion parameter \eqn{r \ge 1} and vertex regions are
#' based on the center \eqn{M=(m_1,m_2)} in Cartesian coordinates or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates
#' in the interior of the triangle \code{tri} or based on circumcenter of \code{tri}; default is \eqn{M=(1,1,1)} i.e., the center
#' of mass of \code{tri}. For the number of arcs, loops are not
#' allowed so arcs are only possible for points inside the triangle \code{tri} for this function.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:arc-density-PE;textual}{pcds}).
#'
#' @param dat A set of 2D points which constitute the vertices of PE-PCD.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param M A 2D point in Cartesian coordinates or a 3D point in barycentric coordinates
#' which serves as a center in the interior of the triangle \code{tri} or the circumcenter of \code{tri};
#' default is \eqn{M=(1,1,1)} i.e., the center of mass of \code{tri}.
#'
#' @return Number of arcs of the PE-PCD with vertices being 2D data set, \code{dat}, in \code{tri}
#' with expansion parameter, \eqn{r \ge 1}, and center \code{M}. PE proximity regions are defined only
#' for \code{dat} points inside \code{tri}, i.e., arcs are possible for such points only.
#'
#' @seealso \code{\link{NumArcsPETe}}, \code{\link{NumArcsPEMT}}, \code{\link{NumArcsCStri}},
#' and \code{\link{NumArcsAStri}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#'
#' n<-10  #try also n<-20
#' set.seed(1)
#' dat<-runif.tri(n,Tr)$g
#'
#' M<-as.numeric(runif.tri(1,Tr)$g)  #try also M<-c(1.6,1.0)
#'
#' NumArcsPEtri(dat,Tr,r=1.25,M)
#' NumArcsPEtri(dat,Tr,r=1.5,M)
#' NumArcsPEtri(dat,Tr,r=2.0,M)
#'
#' NumArcsPEtri(rbind(dat,c(0,2)),Tr,r=1.25,M)
#'
#' r<-2
#' NumArcsPEtri(dat,Tr,r,M)
#'
#' dat.fr<-data.frame(a=dat)
#' NumArcsPEtri(dat.fr,Tr,r,M)
#'
#' dat.fr<-data.frame(a=Tr)
#' NumArcsPEtri(dat,dat.fr,r,M)
#'
#' @export NumArcsPEtri
NumArcsPEtri <- function(dat,tri,r,M=c(1,1,1))
{
  if (!is.numeric(as.matrix(dat)))
  {stop('dat must be numeric')}

  if (is.point(dat))
  { dat<-matrix(dat,ncol=2)
  } else
  {dat<-as.matrix(dat)
  if (ncol(dat)!=2 )
  {stop('dat must be of dimension nx2')}
  }

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if (!is.point(M) && !is.point(M,3) )
  {stop('M must be a numeric 2D point for Cartesian coordinates or 3D point for barycentric coordinates')}

  if (dimension(M)==3)
  {
    M<-bary2cart(M,tri)
  }

  if (isTRUE(all.equal(M,circ.cent.tri(tri)))==F & in.triangle(M,tri,boundary=FALSE)$in.tri==F)
  {stop('center is not the circumcenter or not in the interior of the triangle')}

  n<-nrow(dat)
  arcs<-0
  if (n<=1)
  {
    arcs<-0
  } else
  {
    for (i in 1:n)
    {
      if (in.triangle(dat[i,],tri,boundary=TRUE)$in.tri)
      {  vert<-ifelse(isTRUE(all.equal(M,circ.cent.tri(tri)))==T,rv.triCC(dat[i,],tri)$rv,rv.tri.cent(dat[i,],tri,M)$rv)

      for (j in (1:n)[-i])  #to avoid loops
      {
        arcs<-arcs+IndNPEtri(dat[i,],dat[j,],r,tri,M,rv=vert)
      }
      }
    }
  }
  arcs
} #end of the function
#'

#################################################################

#' @title Arc density of Proportional Edge Proximity Catch Digraphs (PE-PCDs) - one triangle case
#'
#' @description Returns the arc density of PE-PCD whose vertex set is the given 2D numerical data set, \code{Xp},
#' (some of its members are) in the triangle \code{tri}.
#'
#' PE proximity regions is defined with respect to \code{tri} with
#' expansion parameter \eqn{r \ge 1} and vertex regions are based on center \eqn{M=(m_1,m_2)} in Cartesian coordinates or
#' \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the interior of the triangle \code{tri} or based on
#' circumcenter of \code{tri}; default is \eqn{M=(1,1,1)} i.e., the center of mass of \code{tri}.
#' The function also provides arc density standardized by the mean and asymptotic variance of the arc density
#' of PE-PCD for uniform data in the triangle \code{tri}. For the number of arcs, loops are not allowed.
#'
#' \code{tri.cor} is a logical argument for triangle correction (default is \code{TRUE}), if \code{TRUE}, only the points
#' inside the triangle are considered (i.e., digraph induced by these vertices are considered) in computing
#' the arc density, otherwise all points are considered (for the number of vertices in the denominator of arc
#' density).
#'
#' Caveat: The standardized arc density is only correct when \code{M} is the center of mass in the current version.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:arc-density-PE;textual}{pcds}).
#'
#' @param Xp A set of 2D points which constitute the vertices of the PE-PCD.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param M A 2D point in Cartesian coordinates or a 3D point in barycentric coordinates
#' which serves as a center in the interior of the triangle \code{tri} or the circumcenter of \code{tri};
#' default is \eqn{M=(1,1,1)} i.e., the center of mass of \code{tri}.
#' @param tri.cor A logical argument for computing the arc density for only the points inside the triangle,
#' \code{tri}.
#' (default=\code{TRUE}), i.e., if \code{TRUE} only the induced digraph with the vertices inside \code{tri} are considered in the
#' computation of arc density.
#'
#' @return A \code{list} with the elements
#' \item{arc.dens}{Arc density of PE-PCD whose vertices are the 2D numerical data set, \code{Xp};
#' PE proximity regions are defined with respect to the triangle \code{tri} and \code{M}-vertex regions}
#' \item{std.arc.dens}{Arc density standardized by the mean and asymptotic variance of the arc
#' density of PE-PCD for uniform data in the triangle \code{tri}.}
#' \item{caveat}{The warning as \code{"The standardized arc density is only correct when \code{M} is the center of mass
#' in the current version"}.}
#'
#' @seealso \code{\link{ASarcdens.tri}}, \code{\link{CSarcdens.tri}}, and \code{\link{NumArcsPEtri}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#' n<-10  #try also n<-20
#'
#' set.seed(1)
#' dat<-runif.tri(n,Tr)$g
#'
#' M<-as.numeric(runif.tri(1,Tr)$g)  #try also M<-c(1.6,1.0)
#'
#' NumArcsPEtri(dat,Tr,r=1.5,M)
#' PEarcdens.tri(dat,Tr,r=1.5,M)
#' PEarcdens.tri(dat,Tr,r=1.5,M,tri.cor = FALSE)
#'
#' NumArcsPEtri(dat,Tr,r=1,M)
#' PEarcdens.tri(dat,Tr,r=1,M)
#'
#' NumArcsPEtri(dat,Tr,r=1.5,M)
#' PEarcdens.tri(dat,Tr,r=1.5,M)
#'
#' r<-2
#' PEarcdens.tri(dat,Tr,r,M)
#'
#' dat.fr<-data.frame(a=dat)
#' PEarcdens.tri(dat.fr,Tr,r,M)
#'
#' dat.fr<-data.frame(a=Tr)
#' PEarcdens.tri(dat,dat.fr,r,M)
#'
#' @export PEarcdens.tri
PEarcdens.tri <- function(Xp,tri,r,M=c(1,1,1),tri.cor=TRUE)
{
  if (!is.numeric(as.matrix(Xp)) )
  {stop('Xp must be numeric')}

  if (is.point(Xp))
  { Xp<-matrix(Xp,ncol=2)
  } else
  {Xp<-as.matrix(Xp)
  if (ncol(Xp)!=2 )
  {stop('Xp must be of dimension nx2')}
  }

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if (!is.point(M) && !is.point(M,3) )
  {stop('M must be a numeric 2D point for Cartesian coordinates or 3D point for barycentric coordinates')}

  if (dimension(M)==3)
  {
    M<-bary2cart(M,tri)
  }

  if (isTRUE(all.equal(M,circ.cent.tri(tri)))==F & in.triangle(M,tri,boundary=FALSE)$in.tri==F)
  {stop('center is not the circumcenter or not in the interior of the triangle')}

  nx<-nrow(Xp)

  narcs<-NumArcsPEtri(Xp,tri,r,M)
  mean.rho<-muPE2D(r)
  var.rho<-asyvarPE2D(r)

  if (tri.cor==TRUE)
  {
   ind.it<-c()
    for (i in 1:nx)
    {
     ind.it<-c(ind.it,in.triangle(Xp[i,],tri)$in.tri)
    }
    Xp.it<-Xp[ind.it,] #Xp points inside the triangle
    NinTri<-nrow(Xp.it)
    if (NinTri<=1)
    {stop('not enough points in the triangle to compute the arc density')}
    n<-NinTri
  } else
  {
    n<-nx
  }
  rho<-narcs/(n*(n-1))
  std.rho<-sqrt(n)*(rho-mean.rho)/sqrt(var.rho)
  list(
    arc.dens=rho, #arc density
    std.arc.dens=std.rho, #standardized arc density
    caveat="The standardized arc density is only correct when M is the center of mass in the current version." #caveat
  )
} #end of the function
#'

#################################################################

#' @title Number of arcs of Proportional Edge Proximity Catch Digraphs (PE-PCDs) - multiple triangle case
#'
#' @description Returns the number of arcs of Proportional Edge Proximity Catch Digraph (PE-PCD) whose vertices are
#' the data points in \code{Xp} in the multiple triangle case.
#'
#' PE proximity regions are defined with respect to the
#' Delaunay triangles based on \code{Yp} points with expansion parameter \eqn{r \ge 1} and vertex regions in each triangle
#' is based on the center \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the interior of each
#' Delaunay triangle or based on circumcenter of each Delaunay triangle (default for \eqn{M=(1,1,1)}
#' which is the center of mass of the triangle).
#' Each Delaunay triangle is first converted to an (unscaled) basic triangle so that \code{M} will be the same
#' type of center for each Delaunay triangle (this conversion is not necessary when \code{M} is \eqn{CM}).
#'
#' Convex hull of \code{Yp} is partitioned by the Delaunay triangles based on \code{Yp} points
#' (i.e., multiple triangles are the set of these Delaunay triangles whose union constitutes the
#' convex hull of \code{Yp} points). For the number of arcs, loops are not allowed so arcs are only possible
#' for points inside the convex hull of \code{Yp} points.
#'
#' See (\insertCite{ceyhan:Phd-thesis,ceyhan:arc-density-PE;textual}{pcds}) for more on PE-PCDs.
#' Also see (\insertCite{okabe:2000,ceyhan:comp-geo-2010,sinclair:2016;textual}{pcds}) for more on Delaunay triangulation and the corresponding algorithm.
#'
#' @param Xp A set of 2D points which constitute the vertices of the PE-PCD.
#' @param Yp A set of 2D points which constitute the vertices of the Delaunay triangles.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param M A 3D point in barycentric coordinates which serves as a center in the interior of each Delaunay
#' triangle or circumcenter of each Delaunay triangle (for this argument should be set as \code{M="CC"}),
#' default for \eqn{M=(1,1,1)} which is the center of mass of each triangle.
#'
#' @return A \code{list} with the elements
#' \item{num.arcs}{Total number of arcs in all triangles}
#' \item{num.in.conhull}{Number of \code{Xp} points in the convex hull of \code{Yp} points}
#' \item{weight.vec}{The \code{vector} of the areas of Delaunay triangles based on \code{Yp} points}
#'
#' @seealso \code{\link{NumArcsPEtri}}, \code{\link{NumArcsPETe}}, \code{\link{NumArcsCSMT}},
#' and \code{\link{NumArcsASMT}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' #nx is number of X points (target) and ny is number of Y points (nontarget)
#' nx<-20; ny<-4;  #try also nx<-40; ny<-10 or nx<-1000; ny<-10;
#'
#' set.seed(1)
#' Xp<-cbind(runif(nx),runif(nx))
#' Yp<-cbind(runif(ny),runif(ny))
#'
#' oldpar <- par(no.readonly = TRUE)
#' plotDeltri(Xp,Yp,xlab="",ylab="")
#' par(oldpar)
#'
#' M<-c(1,1,1)  #try also M<-c(1,2,3)
#'
#' NumArcsPEMT(Xp,Yp,r=1.25,M)
#' NumArcsPEMT(Xp,Yp,r=1.5,M)
#' NumArcsPEMT(Xp,Yp,r=2,M)
#'
#' NumArcsPEMT(c(.4,.2),Yp,r=1.25)
#'
#' r<-2
#' NumArcsPEMT(Xp,Yp,r)
#' NumArcsPEMT(Xp,Yp[1:3,],r)
#'
#' NumArcsPEMT(Xp,rbind(Yp,Yp),r)
#'
#' dat.fr<-data.frame(a=Xp)
#' NumArcsPEMT(dat.fr,Yp,r)
#'
#' dat.fr<-data.frame(a=Yp)
#' NumArcsPEMT(Xp,dat.fr,r)
#'
#' @export NumArcsPEMT
NumArcsPEMT <- function(Xp,Yp,r,M=c(1,1,1))
{
  if (!is.numeric(as.matrix(Xp)) || !is.numeric(as.matrix(Yp)))
  {stop('Xp and Yp must be numeric')}

  if (is.point(Xp))
  { Xp<-matrix(Xp,ncol=2)
  } else
  {Xp<-as.matrix(Xp)
  if (ncol(Xp)!=2 )
  {stop('Xp must be of dimension nx2')}
  }

  Yp<-as.matrix(Yp)
  if (ncol(Yp)!=2 || nrow(Yp)<3)
  {stop('Yp must be of dimension kx2 with k>=3')}

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if (nrow(Yp)==3)
  {
    Tri.Ind<-indices.Del.tri(Xp,Yp)  #returns 1's if the points Xp[i,]'s are inside triangle based on Yp, NA otherwise

   inTri<-which(Tri.Ind==1)
    NinTri<-length(inTri)  #number of points in the triangle

    if (NinTri==0)
    {Tot.Arcs<-0;
    ListW<-vector()
    } else
    {
      Xdt<-matrix(Xp[inTri,],ncol=2)
      tri<-as.bastri(Yp)$tri #convert the triangle Yp into an unscaled basic triangle, see as.bastri help page
      ListW<-area.polygon(tri)
      Tot.Arcs<-NumArcsPEtri(Xdt,tri,r,M)  #number of arcs in the triangle Yp
    }
    res<-list(num.arcs=Tot.Arcs,
              num.in.conv.hull=NinTri,
              weight.vec=ListW)
  } else
  {

    if ((!is.point(M,3) && M!="CC") || !all(M>0))
    {stop('M must be a numeric 3D point with positive barycentric coordinates or "CC" for circumcenter')}

    #Delaunay triangulation of Yp points
    Ytrimesh<-interp::tri.mesh(Yp[,1],Yp[,2],duplicate="remove")
    Ytri<-matrix(interp::triangles(Ytrimesh)[,1:3],ncol=3); #the vertices of the Delaunay triangles
    nt<-nrow(Ytri)  #number of Delaunay triangles

   inCH<-interp::in.convex.hull(Ytrimesh,Xp[,1],Xp[,2])
    Ninch<-sum(inCH)  #number of points in the convex hull
    if (Ninch==0)
    {Tot.Arcs<-0;
    ListW<-vector()
    } else
    {
      Xdt<-matrix(Xp[inCH==TRUE,],ncol=2)

      Tri.Ind<-indices.Del.tri(Xdt,Yp,Ytrimesh)
      #indices of triangles in which the points in the data fall

      #calculation of the total number of arcs
      List.W<-ni<-arcs<-vector()

      for (i in 1:nt)
      {
        dati<-Xdt[Tri.Ind==i,] #points in ith Delaunay triangle
        ifelse(nt==1,Tri<-Yp[Ytri,],Tri<-Yp[Ytri[i,],])  #vertices of ith triangle
        tri<-as.bastri(Tri)$tri #convert the triangle Tri into an unscaled basic triangle, see as.bastri help page
        List.W<-c(List.W,area.polygon(tri))
        ni<-c(ni,length(dati)/2)  #number of points in ith delaunay triangle
        ifelse(identical(M,"CC"),cent<-circ.cent.tri(tri),cent<-M)
        num.arcs<-NumArcsPEtri(dati,tri,r,cent)  #number of arcs in ith triangle
        arcs<-c(arcs,num.arcs)  #number of arcs in all triangles as a vector

      }

      Tot.Arcs<-sum(arcs)  #the total number of arcs in all triangles
      ListW<-List.W[ni >= 1] #adjusted for triangles with one or more points in them
    }
    res<-list(num.arcs=Tot.Arcs,
              num.in.conhull=Ninch,
              weight.vec=ListW)
  }
  res
} #end of the function
#'

#################################################################

#' @title A test of segregation/association based on arc density of Proportional Edge Proximity Catch Digraph
#' (PE-PCD) for 2D data
#'
#' @description
#' An object of class \code{"htest"} (i.e., hypothesis test) function which performs a hypothesis test of complete spatial
#' randomness (CSR) or uniformity of \code{Xp} points in the convex hull of \code{Yp} points against the alternatives
#' of segregation (where \code{Xp} points cluster away from \code{Yp} points) and association (where \code{Xp} points cluster around
#' \code{Yp} points) based on the normal approximation of the arc density of the PE-PCD for uniform 2D data
#' in the convex hull of \code{Yp} points.
#'
#' The function yields the test statistic, \eqn{p}-value for the corresponding \code{alternative},
#' the confidence interval, estimate and null value for the parameter of interest (which is the arc density),
#' and method and name of the data set used.
#'
#' Under the null hypothesis of uniformity of \code{Xp} points in the convex hull of \code{Yp} points, arc density
#' of PE-PCD whose vertices are \code{Xp} points equals to its expected value under the uniform distribution and
#' \code{alternative} could be two-sided, or left-sided (i.e., data is accumulated around the \code{Yp} points, or association)
#' or right-sided (i.e., data is accumulated around the centers of the triangles, or segregation).
#'
#' PE proximity region is constructed with the expansion parameter \eqn{r \ge 1} and \eqn{CM}-vertex regions
#' (i.e., the test is not available for a general center \eqn{M} at this version of the function).
#' This test is more appropriate when supports of \code{Xp} and \code{Yp} has a substantial overlap,
#'
#' \code{ch.cor} is for convex hull correction (default is \code{"no convex hull correction"}, i.e., \code{ch.cor=FALSE})
#' which is recommended when both \code{Xp} and \code{Yp} have the same rectangular support.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:arc-density-PE;textual}{pcds}) for more on the test based on the arc
#' density of PE-PCDs.
#'
#' @param Xp A set of 2D points which constitute the vertices of the PE-PCD.
#' @param Yp A set of 2D points which constitute the vertices of the Delaunay triangles.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param ch.cor A logical argument for convex hull correction, default \code{ch.cor=FALSE},
#' recommended when both \code{Xp} and \code{Yp} have the same rectangular support.
#' @param alternative Type of the alternative hypothesis in the test, one of \code{"two.sided"}, \code{"less"}, \code{"greater"}.
#' @param conf.level Level of the confidence interval, default is \code{0.95}, for the arc density of PE-PCD based on
#' the 2D data set \code{Xp}.
#'
#' @return A \code{list} with the elements
#' \item{statistic}{Test statistic}
#' \item{p.value}{The \eqn{p}-value for the hypothesis test for the corresponding \code{alternative}}
#' \item{conf.int}{Confidence interval for the arc density at the given confidence level \code{conf.level} and
#' depends on the type of \code{alternative}.}
#' \item{estimate}{Estimate of the parameter, i.e., arc density}
#' \item{null.value}{Hypothesized value for the parameter, i.e., the null arc density, which is usually the
#' mean arc density under uniform distribution.}
#' \item{alternative}{Type of the alternative hypothesis in the test, one of \code{"two.sided"}, \code{"less"}, \code{"greater"}}
#' \item{method}{Description of the hypothesis test}
#' \item{data.name}{Name of the data set}
#'
#' @seealso \code{\link{TSArcDensCSMT}} and \code{\link{TSArcDensPE1D}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' #nx is number of X points (target) and ny is number of Y points (nontarget)
#' nx<-20; ny<-4;  #try also nx<-40; ny<-10 or nx<-1000; ny<-10;
#'
#' set.seed(1)
#' Xp<-cbind(runif(nx),runif(nx))
#' Yp<-cbind(runif(ny),runif(ny))
#'
#' oldpar <- par(no.readonly = TRUE)
#' plotDeltri(Xp,Yp,xlab="",ylab="")
#' par(oldpar)
#'
#' NumArcsPEMT(Xp,Yp,r=1.25)
#' TSArcDensPEMT(Xp,Yp,r=1.25)
#' TSArcDensPEMT(Xp,Yp,r=1.25,ch=TRUE)
#'
#' NumArcsPEMT(Xp,Yp,r=1.5)
#' TSArcDensPEMT(Xp,Yp,r=1.5,alt="l")
#' TSArcDensPEMT(Xp,Yp,r=1.5,ch=TRUE,alt="l")
#'
#' NumArcsPEMT(Xp,Yp,r=2)
#' TSArcDensPEMT(Xp,Yp,r=2)
#' TSArcDensPEMT(Xp,Yp,r=2,ch=TRUE)
#'
#' r<-2
#' TSArcDensPEMT(Xp,Yp,r)
#'
#' Xp<-runif.tri(nx,Yp[1:3,])$g
#' TSArcDensPEMT(Xp,Yp[1:3,],r)
#'
#'
#' TSArcDensPEMT(Xp,rbind(Yp,Yp),r)
#'
#' dat.fr<-data.frame(a=Xp)
#' TSArcDensPEMT(dat.fr,Yp,r)
#'
#' dat.fr<-data.frame(a=Yp)
#' TSArcDensPEMT(Xp,dat.fr,r)
#'
#' @export TSArcDensPEMT
TSArcDensPEMT <- function(Xp,Yp,r,ch.cor=FALSE,alternative = c("two.sided", "less", "greater"), conf.level = 0.95)
{
  dname <-deparse(substitute(Xp))

  alternative <-match.arg(alternative)
  if (length(alternative) > 1 || is.na(alternative))
    stop("alternative must be one \"greater\", \"less\", \"two.sided\"")

  if (!is.numeric(as.matrix(Xp)) || !is.numeric(as.matrix(Yp)))
  {stop('Xp and Yp must be numeric')}

  if (is.point(Xp))
  { Xp<-matrix(Xp,ncol=2)
  } else
  {Xp<-as.matrix(Xp)
  if (ncol(Xp)!=2 )
  {stop('Xp must be of dimension nx2')}
  }

  Yp<-as.matrix(Yp)
  if (ncol(Yp)!=2 || nrow(Yp)<3)
  {stop('Yp must be of dimension kx2 with k>=3')}

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if (!missing(conf.level))
    if (length(conf.level) != 1 || is.na(conf.level) || conf.level < 0 || conf.level > 1)
      stop("conf.level must be a number between 0 and 1")

  Num.arcs<-NumArcsPEMT(Xp,Yp,r,M=c(1,1,1))  #use the default, i.e., CM for the center M
  NinCH<-Num.arcs[[1]]
  if (NinCH<=1)
  {stop('not enough points in the convex hull of non-target points to compute arc density of the target points')}

  num.arcs<-Num.arcs[[2]]
  ListW<-Num.arcs[[3]]

  LW<-ListW/sum(ListW)
  arc.dens<-num.arcs/(NinCH*(NinCH-1))  #arc density
  estimate1<-arc.dens
  asy.mean0<-muPE2D(r)  #asy mean value for the r value
  asy.mean<-asy.mean0*sum(LW^2)
  estimate2<-asy.mean
  asy.var0<-asyvarPE2D(r)  #asy variance value for the r value
  asy.var<-asy.var0*sum(LW^3)+4*asy.mean0^2*(sum(LW^3)-(sum(LW^2))^2)

  TS0<-sqrt(NinCH)*(arc.dens-asy.mean)/sqrt(asy.var)  #standardized test stat

  if (ch.cor==F)
  {
    TS<-TS0
    method <-c("Large Sample z-Test Based on Arc Density for 2D Data\n without Convex Hull Correction")
  }
  else
  {
    n<-nrow(Xp)  #number of X points
    m<-nrow(Yp)  #number of Y points
    NoutCH<-n-NinCH #number of points outside of the convex hull

    prop.out<-NoutCH/n #observed proportion of points outside convex hull
    exp.prop.out<-1.66/m+1.256/sqrt(m)  #expected proportion of points outside convex hull

    TS<-TS0+abs(TS0)*sign(prop.out-exp.prop.out)*(prop.out-exp.prop.out)^2
    method <-c("Large Sample z-Test Based on Arc Density for 2D Data \n with Convex Hull Correction")
  }

  names(estimate1) <-c("arc density")
  null.dens<-asy.mean
  names(null.dens) <-"(expected) arc density"
  names(TS) <-"standardized arc density (i.e., Z)"

  if (alternative == "less") {
    pval <-pnorm(TS)
    cint <-arc.dens+c(-Inf, qnorm(conf.level))*sqrt(asy.var/NinCH)
  }
  else if (alternative == "greater") {
    pval <-pnorm(TS, lower.tail = FALSE)
    cint <-arc.dens+c(-qnorm(conf.level),Inf)*sqrt(asy.var/NinCH)
  }
  else {
    pval <-2 * pnorm(-abs(TS))
    alpha <-1 - conf.level
    cint <-qnorm(1 - alpha/2)
    cint <-arc.dens+c(-cint, cint)*sqrt(asy.var/NinCH)
  }

  attr(cint, "conf.level") <-conf.level

  rval <-list(
    statistic=TS,
    p.value=pval,
    conf.int = cint,
    estimate = estimate1,
    null.value = null.dens,
    alternative = alternative,
    method = method,
    data.name = dname
  )

  attr(rval, "class") <-"htest"
  return(rval)
} #end of the function
#'

#################################################################

#' @title Incidence matrix for Proportional Edge Proximity Catch Digraphs (PE-PCDs) - one triangle case
#'
#' @description Returns the incidence matrix for the PE-PCD whose vertices are the given 2D numerical data set, \code{dat},
#' in the triangle \code{tri}\eqn{=T(v=1,v=2,v=3)}.
#'
#' PE proximity regions are constructed with respect to triangle \code{tri}
#' with expansion parameter \eqn{r \ge 1} and vertex regions are based on the center \eqn{M=(m_1,m_2)} in Cartesian coordinates
#' or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the interior of the triangle \code{tri};
#' default is \eqn{M=(1,1,1)} i.e., the center of mass of \code{tri}.
#' Loops are allowed, so the diagonal entries are all equal to 1.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:arc-density-PE,ceyhan:dom-num-NPE-Spat2011;textual}{pcds}).
#'
#' @param dat A set of 2D points which constitute the vertices of PE-PCD.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param M A 2D point in Cartesian coordinates or a 3D point in barycentric coordinates
#' which serves as a center in the interior of the triangle \code{tri} or the circumcenter of \code{tri};
#' default is \eqn{M=(1,1,1)} i.e., the center of mass of \code{tri}.
#'
#' @return Incidence matrix for the PE-PCD with vertices being 2D data set, \code{dat},
#' in the triangle \code{tri} with vertex regions based on center \code{M}
#'
#' @seealso \code{\link{IncMatPEMT}}, \code{\link{IncMatCStri}}, and  \code{\link{IncMatAStri}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#' n<-10
#'
#' set.seed(1)
#' dat<-runif.tri(n,Tr)$g
#'
#' M<-as.numeric(runif.tri(1,Tr)$g)  #try also M<-c(1.6,1.0)
#'
#' IM<-IncMatPEtri(dat,Tr,r=1.25,M)
#'
#' IM
#' dom.greedy(IM)
#' dom.exact(IM)
#' IndUBdom(IM,2)
#' IndUBdom(IM,3)
#'
#' IncMatPEtri(dat,Tr,r=1.5,M)
#' IncMatPEtri(dat,Tr,r=2,M)
#'
#' r<-2
#' IncMatPEtri(dat,Tr,r,M)
#'
#' dat.fr<-data.frame(a=dat)
#' IncMatPEtri(dat.fr,Tr,r,M)
#'
#' dat.fr<-data.frame(a=Tr)
#' IncMatPEtri(dat,dat.fr,r,M)
#'
#' @export IncMatPEtri
IncMatPEtri <- function(dat,tri,r,M=c(1,1,1))
{
  if (!is.numeric(as.matrix(dat)))
  {stop('dat must be numeric')}

  if (is.point(dat))
  { dat<-matrix(dat,ncol=2)
  } else
  {dat<-as.matrix(dat)
  if (ncol(dat)!=2 )
  {stop('dat must be of dimension nx2')}
  }

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if (!is.point(M) && !is.point(M,3) )
  {stop('M must be a numeric 2D point for Cartesian coordinates or 3D point for barycentric coordinates')}

  if (dimension(M)==3)
  {
    M<-bary2cart(M,tri)
  }

  if (isTRUE(all.equal(M,circ.cent.tri(tri)))==F & in.triangle(M,tri,boundary=FALSE)$in.tri==F)
  {stop('center is not the circumcenter or not in the interior of the triangle')}

  n<-nrow(dat)

 inc.mat<-matrix(0, nrow=n, ncol=n)
  if (n>1)
  {
    for (i in 1:n)
    {pt1<-dat[i,]
    rv<-ifelse(isTRUE(all.equal(M,circ.cent.tri(tri)))==T,rv.triCC(pt1,tri)$rv,rv.tri.cent(pt1,tri,M)$rv)

    for (j in ((1:n)) )
    {pt2<-dat[j,]
   inc.mat[i,j]<-IndNPEtri(pt1,pt2,r,tri,M,rv=rv)
    }
    }
  }
 inc.mat
} #end of the function
#'

#################################################################

#' @title The indicator for a point being a dominating point for Proportional Edge Proximity Catch Digraphs (PE-PCDs)
#' - one triangle case
#'
#' @description Returns \eqn{I(}\code{p} is a dominating point of the PE-PCD\eqn{)} where the vertices of the PE-PCD are the 2D data set \code{Dt} in the triangle \code{tri}, that is,
#' returns 1 if \code{p} is a dominating point of PE-PCD, returns 0 otherwise.
#'
#' Point, \code{p}, is in the vertex region of vertex \code{rv} (default is \code{NULL}); vertices are labeled as \eqn{1,2,3}
#' in the order they are stacked row-wise in \code{tri}.
#'
#' PE proximity region is constructed with respect to the triangle \code{tri} with expansion parameter \eqn{r \ge 1}
#' and vertex regions are based on center \eqn{M=(m_1,m_2)} in Cartesian coordinates or
#' \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the interior of the triangle \code{tri}
#' or based on the circumcenter of \code{tri}; default is \eqn{M=(1,1,1)} i.e., the center of mass of \code{tri}.
#'
#' \code{ch.data.pnt} is for checking whether point \code{p} is a data point in \code{Dt} or not (default is \code{FALSE}),
#' so by default this function checks whether the point \code{p} would be a dominating point
#' if it actually were in the data set.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:masa-2007,ceyhan:dom-num-NPE-Spat2011,ceyhan:mcap2012;textual}{pcds}).
#'
#' @param p A 2D point that is to be tested for being a dominating point or not of the PE-PCD.
#' @param Dt A set of 2D points which constitutes the vertices of the PE-PCD.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param M A 2D point in Cartesian coordinates or a 3D point in barycentric coordinates
#' which serves as a center in the interior of the triangle \code{tri} or the circumcenter of \code{tri};
#' default is \eqn{M=(1,1,1)} i.e., the center of mass of \code{tri}.
#' @param rv Index of the vertex whose region contains point \code{p}, \code{rv} takes the vertex labels as \eqn{1,2,3} as
#' in the row order of the vertices in \code{tri}.
#' @param ch.data.pnt A logical argument for checking whether point \code{p} is a data point
#' in \code{Dt} or not (default is \code{FALSE}).
#'
#' @return \eqn{I(}\code{p} is a dominating point of the PE-PCD\eqn{)} where the vertices of the PE-PCD are the 2D data set \code{Dt},
#' that is, returns 1 if \code{p} is a dominating point, returns 0 otherwise
#'
#' @seealso \code{\link{Gam1PEbastri}} and \code{\link{Gam1AStri}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#' n<-10  #try also n<-20
#'
#' set.seed(1)
#' dat<-runif.tri(n,Tr)$g
#'
#' M<-as.numeric(runif.tri(1,Tr)$g)  #try also M<-c(1.6,1.0)
#'
#' r<-1.5  #try also r<-2
#'
#' Gam1PEtri(c(1.4,1),c(1.4,1),Tr,r,M)
#' Gam1PEtri(dat[1,],dat,Tr,r,M)
#' Gam1PEtri(c(1,2),dat,Tr,r,M)
#'
#' Gam1PEtri(c(1,2),c(1,2),Tr,r,M)
#' Gam1PEtri(c(1,2),c(1,2),Tr,r,M,ch.data.pnt = TRUE)
#'
#' gam.vec<-vector()
#' for (i in 1:n)
#' {gam.vec<-c(gam.vec,Gam1PEtri(dat[i,],dat,Tr,r,M))}
#'
#' ind.gam1<-which(gam.vec==1)
#' ind.gam1
#'
#' #or try
#' Rv<-rv.tri.cent(dat[1,],Tr,M)$rv
#' Gam1PEtri(dat[1,],dat,Tr,r,M,Rv)
#'
#' Ds<-cp2e.tri(Tr,M)
#'
#' if (dimension(M)==3) {M<-bary2cart(M,Tr)}
#' #need to run this when M is given in barycentric coordinates
#'
#' Xlim<-range(Tr[,1],dat[,1],M[1])
#' Ylim<-range(Tr[,2],dat[,2],M[2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(Tr,pch=".",xlab="",ylab="",axes=TRUE,
#' xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tr)
#' points(dat,pch=1,col=1)
#' L<-rbind(M,M,M); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#' points(rbind(dat[ind.gam1,]),pch=4,col=2)
#' #rbind is to insert the points correctly if there is only one dominating point
#'
#' txt<-rbind(Tr,M,Ds)
#' xc<-txt[,1]+c(-.02,.03,.02,-.02,.04,-.03,.0)
#' yc<-txt[,2]+c(.02,.02,.05,-.03,.04,.06,-.07)
#' txt.str<-c("A","B","C","M","D1","D2","D3")
#' text(xc,yc,txt.str)
#'
#' P<-c(1.4,1)
#' Gam1PEtri(P,P,Tr,r,M)
#' Gam1PEtri(dat[1,],dat,Tr,r,M)
#'
#' dat.fr<-data.frame(a=dat)
#' Gam1PEtri(P,dat.fr,Tr,r,M)
#'
#' dat.fr<-data.frame(a=Tr)
#' Gam1PEtri(P,dat,dat.fr,r,M)
#'
#' Gam1PEtri(c(1,2),dat,Tr,r,M,ch.data.pnt = FALSE)
#' #gives an error message if ch.data.pnt = TRUE since p is not a data point
#'
#' @export Gam1PEtri
Gam1PEtri <- function(p,Dt,tri,r,M=c(1,1,1),rv=NULL,ch.data.pnt=FALSE)
{
  if (!is.point(p))
  {stop('p must be a numeric 2D point')}

  if (!is.numeric(as.matrix(Dt)))
  {stop('Dt must be numeric')}

  if (is.point(Dt))
  { Dt<-matrix(Dt,ncol=2)
  } else
  {Dt<-as.matrix(Dt)
  if (ncol(Dt)!=2 )
  {stop('Dt must be of dimension nx2')}
  }

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if (!is.point(M) && !is.point(M,3) )
  {stop('M must be a numeric 2D point for Cartesian coordinates or 3D point for barycentric coordinates')}

  n<-nrow(Dt)

  if (ch.data.pnt==TRUE)
  {
    if (!is.in.data(p,Dt))
    {stop('p is not a data point in Dt')}
  }

  if (dimension(M)==3)
  {
    M<-bary2cart(M,tri)
  }

  if (isTRUE(all.equal(M,circ.cent.tri(tri)))==F & in.triangle(M,tri,boundary=FALSE)$in.tri==F)
  {stop('center is not the circumcenter or not in the interior of the triangle')}

  if (in.triangle(p,tri)$in.tri==F)
  {dom<-0; return(dom); stop}

  if (is.null(rv))
  {rv<-ifelse(isTRUE(all.equal(M,circ.cent.tri(tri)))==T,rv.triCC(p,tri)$rv,rv.tri.cent(p,tri,M)$rv)
  } else
  {  if (!is.numeric(rv) || sum(rv==c(1,2,3))!=1)
  {stop('vertex index, rv, must be 1, 2 or 3')}}

  if (n>=1)
  {
    dom<-1; i<-1;
    while (i <= n & dom==1)
    {
      if (IndNPEtri(p,Dt[i,],r,tri,M,rv)==0)
        dom<-0;
      i<-i+1;
    }
  } else
  {
    dom<-0
  }
  dom
} #end of the function
#'

#################################################################

#' @title The indicator for two points constituting a dominating set for Proportional Edge Proximity Catch Digraphs
#' (PE-PCDs) - one triangle case
#'
#' @description Returns \eqn{I(}\{\code{pt1,pt2}\} is a dominating set of the PE-PCD\eqn{)} where the vertices of the PE-PCD
#' are the 2D data set \code{Dt}), that is, returns 1 if \{\code{pt1,pt2}\} is a dominating set of PE-PCD,
#' returns 0 otherwise.
#'
#' Point, \code{pt1}, is in the region of vertex \code{rv1} (default is \code{NULL}) and point, \code{pt2}, is in the region of vertex \code{rv2}
#' (default is \code{NULL}); vertices (and hence \code{rv1} and \code{rv2}) are labeled as \eqn{1,2,3} in the order they are stacked
#' row-wise in \code{tri}.
#'
#' PE proximity regions are defined with respect to the triangle \code{tri} and vertex regions
#' are based on center \eqn{M=(m_1,m_2)} in Cartesian coordinates or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates
#' in the interior of the triangle \code{tri} or circumcenter of \code{tri}; default is \eqn{M=(1,1,1)} i.e., the center of mass of \code{tri}.
#'
#' \code{ch.data.pnts} is for checking whether points \code{pt1} and \code{pt2} are data points in \code{Dt} or not
#' (default is \code{FALSE}), so by default this function checks whether the points \code{pt1} and \code{pt2} would be a
#' dominating set if they actually were in the data set.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:masa-2007,ceyhan:dom-num-NPE-Spat2011,ceyhan:mcap2012;textual}{pcds}).
#'
#' @param pt1,pt2 Two 2D points to be tested for constituting a dominating set of the PE-PCD.
#' @param Dt A set of 2D points which constitutes the vertices of the PE-PCD.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param M A 2D point in Cartesian coordinates or a 3D point in barycentric coordinates
#' which serves as a center in the interior of the triangle \code{tri} or the circumcenter of \code{tri};
#' default is \eqn{M=(1,1,1)} i.e., the center of mass of \code{tri}.
#' @param rv1,rv2 The indices of the vertices whose regions contains \code{pt1} and \code{pt2}, respectively.
#' They take the vertex labels as \eqn{1,2,3} as in the row order of the vertices in \code{tri}
#' (default is \code{NULL} for both).
#' @param ch.data.pnts A logical argument for checking whether points \code{pt1} and \code{pt2} are
#' data points in \code{Dt} or not (default is \code{FALSE}).
#'
#' @return \eqn{I(}\{\code{pt1,pt2}\} is a dominating set of the PE-PCD\eqn{)} where the vertices of the PE-PCD are the 2D data set \code{Dt}),
#' that is, returns 1 if \{\code{pt1,pt2}\} is a dominating set of PE-PCD, returns 0 otherwise
#'
#' @seealso \code{\link{Gam2PEbastri}}, \code{\link{Gam2AStri}}, and \code{\link{Gam2PEtetra}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#' n<-10  #try also n<-20
#'
#' set.seed(1)
#' dat<-runif.tri(n,Tr)$g
#'
#' M<-as.numeric(runif.tri(1,Tr)$g)  #try also M<-c(1.6,1.0)
#'
#' r<-1.5  #try also r<-2
#'
#' Gam2PEtri(dat[1,],dat[2,],dat,Tr,r,M)
#' Gam2PEtri(dat[1,],dat[4,],dat,Tr,r,M)
#' Gam2PEtri(dat[4,],dat[4,],dat,Tr,r,M)
#'
#' Gam2PEtri(dat[1,],c(1,2),dat,Tr,r,M)
#'
#' Gam2PEtri(c(1,2),c(1,3),rbind(c(1,2),c(1,3)),Tr,r,M)
#'
#' ind.gam2<-vector()
#' for (i in 1:(n-1))
#'   for (j in (i+1):n)
#'   {if (Gam2PEtri(dat[i,],dat[j,],dat,Tr,r,M)==1)
#'    ind.gam2<-rbind(ind.gam2,c(i,j))}
#' ind.gam2
#'
#' #or try
#' rv1<-rv.tri.cent(dat[1,],Tr,M)$rv;
#' rv2<-rv.tri.cent(dat[2,],Tr,M)$rv
#' Gam2PEtri(dat[1,],dat[2,],dat,Tr,r,M,rv1,rv2)
#'
#' #or try
#' rv1<-rv.tri.cent(dat[1,],Tr,M)$rv;
#' Gam2PEtri(dat[1,],dat[2,],dat,Tr,r,M,rv1)
#'
#' #or try
#' rv2<-rv.tri.cent(dat[2,],Tr,M)$rv
#' Gam2PEtri(dat[1,],dat[2,],dat,Tr,r,M,rv2=rv2)
#'
#' P1<-c(1.4,1)
#' P2<-c(1.6,1)
#' Gam2PEtri(P1,P2,dat,Tr,r,M)
#' Gam2PEtri(dat[1,],dat[2,],dat,Tr,r,M)
#'
#' dat.fr<-data.frame(a=dat)
#' Gam2PEtri(P1,P2,dat.fr,Tr,r,M)
#'
#' dat.fr<-data.frame(a=Tr)
#' Gam2PEtri(P1,P2,dat,dat.fr,r,M)
#'
#' Gam2PEtri(dat[1,],c(1,2),dat,Tr,r,M,ch.data.pnts = FALSE)
#' #gives an error message if ch.data.pnts = TRUE
#' #since not both points, pt1 and pt2, are data points in Dt
#'
#' @export Gam2PEtri
Gam2PEtri <- function(pt1,pt2,Dt,tri,r,M=c(1,1,1),rv1=NULL,rv2=NULL,ch.data.pnts=FALSE)
{
  if (!is.point(pt1) || !is.point(pt2))
  {stop('pt1 and pt2 must be numeric 2D points')}

  if (!is.numeric(as.matrix(Dt)))
  {stop('Dt must be numeric')}

  if (is.point(Dt))
  { Dt<-matrix(Dt,ncol=2)
  } else
  {Dt<-as.matrix(Dt)
  if (ncol(Dt)!=2 )
  {stop('Dt must be of dimension nx2')}
  }

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if (!is.point(M) && !is.point(M,3) )
  {stop('M must be a numeric 2D point for Cartesian coordinates or 3D point for barycentric coordinates')}

  if (ch.data.pnts==TRUE)
  {
    if (!is.in.data(pt1,Dt) || !is.in.data(pt2,Dt))
    {stop('not both points, pt1 and pt2, are data points in Dt')}
  }

  if (dimension(M)==3)
  {
    M<-bary2cart(M,tri)
  }

  if (isTRUE(all.equal(M,circ.cent.tri(tri)))==F & in.triangle(M,tri,boundary=FALSE)$in.tri==F)
  {stop('center is not the circumcenter or not in the interior of the triangle')}

  if (identical(pt1,pt2))
  {dom<-0; return(dom); stop}

  if (is.null(rv1))
  {rv1<-ifelse(isTRUE(all.equal(M,circ.cent.tri(tri)))==T,rv.triCC(pt1,tri)$rv,rv.tri.cent(pt1,tri,M)$rv)  #vertex region for point pt1
  }
  if (is.null(rv2))
  {rv2<-ifelse(isTRUE(all.equal(M,circ.cent.tri(tri)))==T,rv.triCC(pt2,tri)$rv,rv.tri.cent(pt2,tri,M)$rv)  #vertex region for point pt2
  }

  n<-nrow(Dt)
  dom<-1; i<-1;
  while (i <= n & dom==1)
  {if (max(IndNPEtri(pt1,Dt[i,],r,tri,M,rv1),IndNPEtri(pt2,Dt[i,],r,tri,M,rv2))==0)
    dom<-0;
  i<-i+1;
  }
  dom
} #end of the function
#'

#################################################################

#' @title The domination number of Proportional Edge Proximity Catch Digraph (PE-PCD) - one triangle case
#'
#' @description Returns the domination number of PE-PCD whose vertices are the data points in \code{Xp}.
#'
#' PE proximity region is defined with respect to the triangle \code{tri} with expansion parameter \eqn{r \ge 1} and
#' vertex regions are constructed with center \eqn{M=(m_1,m_2)} in Cartesian coordinates or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates
#' in the interior of the triangle \code{tri} or the circumcenter of \code{tri}.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:masa-2007,ceyhan:dom-num-NPE-Spat2011,ceyhan:mcap2012;textual}{pcds}).
#'
#' @param Xp A set of 2D points which constitute the vertices of the digraph.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param M A 2D point in Cartesian coordinates or a 3D point in barycentric coordinates
#' which serves as a center in the interior of the triangle \code{tri} or the circumcenter of \code{tri},
#' default is \eqn{(1,1,1)} i.e., the center of mass.
#'
#' @return A \code{list} with two elements
#' \item{dom.num}{Domination number of PE-PCD with vertex set = \code{Xp} and expansion parameter \eqn{r \ge 1} and center \code{M}}
#' \item{mds}{A minimum dominating set of PE-PCD with vertex set = \code{Xp} and expansion parameter \eqn{r \ge 1} and center \code{M}}
#'
#' @seealso \code{\link{PEdomMTnd}}, \code{\link{PEdomMT}} and \code{\link{PEdom1D}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(0,0); B<-c(1,0); C<-c(1/2,sqrt(3)/2)
#' Tr<-rbind(A,B,C)
#' n<-10  #try also n<-20
#' dat<-runif.tri(n,Tr)$g
#'
#' M<-as.numeric(runif.tri(1,Tr)$g)  #try also M<-c(1,1,1)
#'
#' r<-1.4
#'
#' PEdomtri(dat,Tr,r,M)
#' IM<-IncMatPEtri(dat,Tr,r,M)
#' dom.greedy(IM)
#' dom.exact(IM)
#'
#' gr.gam<-dom.greedy(IM)
#' gr.gam
#' dat[gr.gam$i,]
#'
#' PEdomtri(rbind(dat,c(5,5)),Tr,r,M)
#'
#' PEdomtri(dat,Tr,r,M=c(.4,.4))
#'
#' PEdomtri(rbind(dat,c(5,5)),Tr,r,M=c(.4,.4))
#'
#' P1<-c(.5,.5)
#' PEdomtri(P1,Tr,r,M)
#'
#' @export PEdomtri
PEdomtri <- function(Xp,tri,r,M=c(1,1,1))
{
  if (!is.numeric(as.matrix(Xp)) )
  {stop('Xp must be numeric')}

  if (is.point(Xp))
  { Xp<-matrix(Xp,ncol=2)
  } else
  {Xp<-as.matrix(Xp)
  if (ncol(Xp)!=2 )
  {stop('Xp must be of dimension nx2')}
  }

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  if (!is.point(M) && !is.point(M,3) )
  {stop('M must be a numeric 2D point for Cartesian coordinates or 3D point for barycentric coordinates')}

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  n<-nrow(Xp)  #number of Xp points

 ind.tri<-c()
  for (i in 1:n)
  {
    if (in.triangle(Xp[i,],tri,boundary = T)$i)
     ind.tri<-c(ind.tri,i)
  }

  Xtri<-matrix(Xp[ind.tri,],ncol=2)

  ntri<-nrow(Xtri)  #number of points inside the triangle
  if (ntri==0)
  {gam<-0;
  res<-list(dom.num=gam, #domination number
            mds=vector())  #a minimum dominating set
  return(res); stop}

  if (dimension(M)==3)
  {
    M<-bary2cart(M,tri)
  }

  if (isTRUE(all.equal(M,circ.cent.tri(tri)))==F & in.triangle(M,tri,boundary=FALSE)$in.tri==F)
  {stop('center is not the circumcenter or not in the interior of the triangle')}

  Cl2e<-cl2eVRcent(Xtri,tri,M)$ext

  mds<-c()
  #Gamma=1 piece
  cnt<-0; j<-1;
  while (j<=3 & cnt==0)
  {
    if (Gam1PEtri(Cl2e[j,],Xtri,tri,r,M,rv=j)==1)
    {gam<-1; cnt<-1; mds<-rbind(mds,Cl2e[j,])
    } else
    {j<-j+1}
  }

  #Gamma=2 piece
  if (cnt==0)
  { k<-1; cnt2<-0;
  while (k<=2 & cnt2==0)
  {l<-k+1;
  while (l<=3 & cnt2==0)
  {
    if (Gam2PEtri(Cl2e[k,],Cl2e[l,],Xtri,tri,r,M,rv1=k,rv2=l)==1)
    {gam<-2;cnt2<-1 ; mds<-rbind(mds,Cl2e[c(k,l),])
    } else {l<-l+1};
  }
  k<-k+1;
  }
  }

  #Gamma=3 piece
  if (cnt==0 && cnt2==0)
  {gam <-3; mds<-rbind(mds,Cl2e)}

  row.names(mds)<-c()
  list(dom.num=gam, #domination number
       mds=mds #a minimum dominating set
  )
} #end of the function
#'

#################################################################

#' @title The indicator for the presence of an arc from a point in set \code{S} to the point \code{pt} for Proportional Edge
#' Proximity Catch Digraphs (PE-PCDs) - standard equilateral triangle case
#'
#' @description Returns \eqn{I(}\code{pt} in \eqn{N_{PE}(x,r)} for some \eqn{x} in \code{S}\eqn{)}, that is, returns 1 if \code{pt} is in \eqn{\cup_{x in S}N_{PE}(x,r)},
#' returns 0 otherwise.
#'
#' PE proximity region is constructed with respect to the standard equilateral triangle
#' \eqn{T_e=T(A,B,C)=T((0,0),(1,0),(1/2,\sqrt{3}/2))} with the expansion parameter \eqn{r \ge 1} and vertex regions are based
#' on center \eqn{M=(m_1,m_2)} in Cartesian coordinates or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the
#' interior of \eqn{T_e}; default is \eqn{M=(1,1,1)} i.e., the center of mass of \eqn{T_e} (which is equivalent to the circumcenter
#' for \eqn{T_e}).
#'
#' Vertices of \eqn{T_e} are also labeled as 1, 2, and 3, respectively.
#' If \code{pt} is not in \code{S} and either \code{pt} or all points in \code{S} are outside \eqn{T_e}, it returns 0,
#' but if \code{pt} is in \code{S}, then it always returns 1 regardless of its location (i.e., loops are allowed).
#'
#' @param S A set of 2D points. Presence of an arc from a point in \code{S} to point \code{pt} is checked
#' by the function.
#' @param pt A 2D point. Presence of an arc from a point in \code{S} to point \code{pt} is checked
#' by the function.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region in the
#' standard equilateral triangle \eqn{T_e=T((0,0),(1,0),(1/2,\sqrt{3}/2))}; must be \eqn{\ge 1}.
#' @param M A 2D point in Cartesian coordinates or a 3D point in barycentric coordinates
#' which serves as a center in the interior of the standard equilateral triangle \eqn{T_e}; default is \eqn{M=(1,1,1)}
#' i.e., the center of mass of \eqn{T_e}.
#'
#' @return I(\code{pt} is in U_{x in \code{S}} N_{PE}(x,r)), that is, returns 1 if \code{pt} is in \code{S} or inside \eqn{N_{PE}(x,r)} for at least
#' one \eqn{x} in \code{S}, returns 0 otherwise. PE proximity region is constructed with respect to the standard
#' equilateral triangle \eqn{T_e=T(A,B,C)=T((0,0),(1,0),(1/2,\sqrt{3}/2))} with \code{M}-vertex regions
#'
#' @seealso \code{\link{IndNPEtriSet}}, \code{\link{IndNPETe}}, \code{\link{IndNPEtri}}, and \code{\link{IndCSTeSet}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(0,0); B<-c(1,0); C<-c(1/2,sqrt(3)/2);
#' Te<-rbind(A,B,C);
#' n<-10
#'
#' set.seed(1)
#' dat<-runifTe(n)$gen.points
#'
#' M<-as.numeric(runifTe(1)$g)  #try also M<-c(.6,.2)
#'
#' r<-1.5
#'
#' S<-rbind(dat[1,],dat[2,])  #try also S<-c(.5,.5)
#' IndNPETeSet(S,dat[3,],r,M)
#' IndNPETeSet(S,dat[3,],r=1,M)
#' IndNPETeSet(S,dat[3,],r=1.5,M)
#'
#' S<-rbind(dat[1,],dat[2,],dat[3,],dat[5,])
#' IndNPETeSet(S,dat[3,],r,M)
#'
#' IndNPETeSet(S,dat[6,],r,M)
#' IndNPETeSet(S,dat[6,],r=1.25,M)
#'
#' S<-rbind(c(.1,.1),c(.3,.4),c(.5,.3))
#' IndNPETeSet(S,dat[3,],r,M)
#'
#' IndNPETeSet(c(.2,.5),dat[2,],r,M)
#' IndNPETeSet(dat,c(.2,.5),r,M)
#' IndNPETeSet(dat,dat[2,],r,M)
#' IndNPETeSet(c(.2,.5),c(.2,.5),r,M)
#' IndNPETeSet(dat[5,],dat[2,],r,M)
#'
#' S<-rbind(dat[1,],dat[2,],dat[3,],dat[5,],c(.2,.5))
#' IndNPETeSet(S,dat[3,],r,M)
#'
#' P<-c(.4,.2)
#' S<-dat[c(1,3,4),]
#' IndNPETeSet(dat,P,r,M)
#'
#' IndNPETeSet(rbind(S,S),P,r,M)
#'
#' dat.fr<-data.frame(a=S)
#' IndNPETeSet(dat.fr,P,r,M)
#'
#' @export IndNPETeSet
IndNPETeSet <- function(S,pt,r,M=c(1,1,1))
{
  if (!is.point(pt))
  {stop('pt must be a numeric 2D point')}

  if (!is.numeric(as.matrix(S)))
  {stop('S must be a matrix of numeric values')}

  if (is.point(S))
  { S<-matrix(S,ncol=2)
  } else
  {S<-as.matrix(S)
  if (ncol(S)!=2 )
  {stop('S must be of dimension nx2')}
  }

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if (!is.point(M) && !is.point(M,3) )
  {stop('M must be a numeric 2D point for Cartesian coordinates or 3D point for barycentric coordinates')}

  if (dimension(M)==3)
  {
    M<-bary2cart(M,Te)
  }

  A<-c(0,0); B<-c(1,0); C<-c(1/2,sqrt(3)/2);
  Te<-rbind(A,B,C);

  if (in.triangle(M,Te,boundary=FALSE)$in.tri==F)
  {stop('center is not in the interior of the triangle')}

  k<-nrow(S);
  dom<-0; i<-1;
  while (dom ==0 && i<= k)
  {
    if (IndNPETe(S[i,],pt,r,M)==1)
    {dom<-1};
    i<-i+1;
  }
  dom
} #end of the function
#'

#################################################################

#' @title The indicator for the presence of an arc from a point in set \code{S} to the point \code{pt} for
#' Proportional Edge Proximity Catch Digraphs (PE-PCDs) - one triangle case
#'
#' @description Returns \eqn{I(}\code{pt} in \eqn{N_{PE}(x,r)} for some \eqn{x} in \code{S}\eqn{)}, that is, returns 1 if \code{pt} is in \eqn{\cup_{x in S}N_{PE}(x,r)},
#' returns 0 otherwise.
#'
#' PE proximity region is constructed with respect to the triangle \code{tri} with
#' the expansion parameter \eqn{r \ge 1} and vertex regions are based on the center, \eqn{M=(m_1,m_2)} in Cartesian coordinates
#' or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the interior of the triangle \code{tri}
#' or based on the circumcenter of \code{tri}; default is \eqn{M=(1,1,1)} i.e., the center of mass of \code{tri}.
#' Vertices of \code{tri} are also labeled as 1, 2, and 3, respectively.
#'
#' If \code{pt} is not in \code{S} and either \code{pt} or all points in \code{S} are outside \code{tri}, it returns 0,
#' but if \code{pt} is in \code{S}, then it always returns 1 regardless of its location (i.e., loops are allowed).
#'
#' @param S A set of 2D points. Presence of an arc from a point in \code{S} to point \code{pt} is checked
#' by the function.
#' @param pt A 2D point. Presence of an arc from a point in \code{S} to point \code{pt} is checked
#' by the function.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region
#' constructed in the triangle \code{tri}; must be \eqn{\ge 1}.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param M A 2D point in Cartesian coordinates or a 3D point in barycentric coordinates
#' which serves as a center in the interior of the triangle \code{tri} or the circumcenter of \code{tri};
#' default is \eqn{M=(1,1,1)} i.e., the center of mass of \code{tri}.
#'
#' @return I(\code{pt} is in U_{x in \code{S}} N_{PE}(x,r)), that is, returns 1 if \code{pt} is in \code{S} or inside \eqn{N_{PE}(x,r)} for at least
#' one \eqn{x} in \code{S}, returns 0 otherwise where PE proximity region is constructed with respect to the triangle \code{tri}
#'
#' @seealso \code{\link{IndNPETeSet}}, \code{\link{IndNPEtri}}, \code{\link{IndNPETe}},
#' \code{\link{IndNAStriSet}}, and \code{\link{IndNCStriSet}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#' n<-10
#'
#' set.seed(1)
#' dat<-runif.tri(n,Tr)$gen.points
#'
#' M<-as.numeric(runif.tri(1,Tr)$g)  #try also M<-c(1.6,1.0)
#'
#' r<-1.5
#'
#' S<-rbind(dat[1,],dat[2,])  #try also S<-c(1.5,1)
#'
#' IndNPEtriSet(S,dat[3,],r,Tr,M)
#' IndNPEtriSet(S,dat[3,],r=1,Tr,M)
#' IndNPEtriSet(S,dat[3,],r=1.5,Tr,M)
#'
#' S<-rbind(dat[1,],dat[2,],dat[3,],dat[5,])
#' IndNPEtriSet(S,dat[3,],r,Tr,M)
#'
#' IndNPEtriSet(S,dat[6,],r,Tr,M)
#' IndNPEtriSet(S,dat[6,],r=1.25,Tr,M)
#'
#' S<-rbind(c(.1,.1),c(.3,.4),c(.5,.3))
#' IndNPEtriSet(S,dat[3,],r,Tr,M)
#'
#' IndNPEtriSet(c(.2,.5),dat[2,],r,Tr,M)
#' IndNPEtriSet(dat,c(.2,.5),r,Tr,M)
#' IndNPEtriSet(dat,dat[2,],r,Tr,M)
#' IndNPEtriSet(c(.2,.5),c(.2,.5),r,Tr,M)
#' IndNPEtriSet(dat[5,],dat[2,],r,Tr,M)
#'
#' S<-rbind(dat[1,],dat[2,],dat[3,],dat[5,],c(.2,.5))
#' IndNPEtriSet(S,dat[3,],r,Tr,M)
#'
#' P<-c(.4,.2)
#' S<-dat[c(1,3,4),]
#' IndNPEtriSet(dat,P,r,Tr,M)
#'
#' IndNPEtriSet(rbind(S,S),P,r,Tr,M)
#'
#' dat.fr<-data.frame(a=S)
#' IndNPEtriSet(dat.fr,P,r,Tr,M)
#'
#' @export IndNPEtriSet
IndNPEtriSet <- function(S,pt,r,tri,M=c(1,1,1))
{
  if (!is.point(pt))
  {stop('pt must be a numeric 2D point')}

  if (!is.numeric(as.matrix(S)))
  {stop('S must be a matrix of numeric values')}

  if (is.point(S))
  { S<-matrix(S,ncol=2)
  } else
  {S<-as.matrix(S)
  if (ncol(S)!=2 )
  {stop('S must be of dimension nx2')}
  }

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  if (!is.point(M) && !is.point(M,3) )
  {stop('M must be a numeric 2D point for Cartesian coordinates or 3D point for barycentric coordinates')}

  if (dimension(M)==3)
  {
    M<-bary2cart(M,tri)
  }

  if (isTRUE(all.equal(M,circ.cent.tri(tri)))==F & in.triangle(M,tri,boundary=FALSE)$in.tri==F)
  {stop('center is not the circumcenter or not in the interior of the triangle')}

  k<-nrow(S);
  dom<-0; i<-1;
  while (dom ==0 && i<= k)
  {
    if (IndNPEtri(S[i,],pt,r,tri,M)==1)
    {dom<-1};
    i<-i+1;
  }
  dom
} #end of the function
#'

#################################################################

#' @title The indicator for the set of points \code{S} being a dominating set or not for Proportional Edge Proximity
#' Catch Digraphs (PE-PCDs) - standard equilateral triangle case
#'
#' @description Returns \eqn{I(}\code{S} a dominating set of PE-PCD whose vertices are the data points \code{Dt}\eqn{)}, that is,
#' returns 1 if \code{S} is a dominating set of PE-PCD, returns 0 otherwise.
#'
#' PE proximity region is constructed
#' with respect to the standard equilateral triangle \eqn{T_e=T(A,B,C)=T((0,0),(1,0),(1/2,\sqrt{3}/2))} with
#' expansion parameter \eqn{r \ge 1} and vertex regions are based on the center \eqn{M=(m_1,m_2)}
#' in Cartesian coordinates or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the interior of \eqn{T_e};
#' default is \eqn{M=(1,1,1)} i.e., the center of mass of \eqn{T_e} (which is equivalent to the circumcenter of \eqn{T_e}).
#' Vertices of \eqn{T_e} are also labeled as 1, 2, and 3, respectively.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:masa-2007,ceyhan:dom-num-NPE-Spat2011,ceyhan:mcap2012;textual}{pcds}).
#'
#' @param S A set of 2D points whose PE proximity regions are considered.
#' @param Dt A set of 2D points which constitutes the vertices of the PE-PCD.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region in the
#' standard equilateral triangle \eqn{T_e=T((0,0),(1,0),(1/2,\sqrt{3}/2))}; must be \eqn{\ge 1}.
#' @param M A 2D point in Cartesian coordinates or a 3D point in barycentric coordinates
#' which serves as a center in the interior of the standard equilateral triangle \eqn{T_e}; default is \eqn{M=(1,1,1)} i.e.
#' the center of mass of \eqn{T_e}.
#'
#' @return \eqn{I(}\code{S} a dominating set of PE-PCD\eqn{)}, that is, returns 1 if \code{S} is a dominating set of PE-PCD,
#' returns 0 otherwise, where PE proximity region is constructed in the standard equilateral triangle \eqn{T_e}
#'
#' @seealso \code{\link{IndNPEtri.domset}} and \code{\link{IndCSTe.domset}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(0,0); B<-c(1,0); C<-c(1/2,sqrt(3)/2);
#' Te<-rbind(A,B,C);
#' n<-10
#'
#' set.seed(1)
#' dat<-runifTe(n)$gen.points
#'
#' M<-as.numeric(runifTe(1)$g)  #try also M<-c(.6,.2)
#'
#' r<-1.5
#'
#' S<-rbind(dat[1,],dat[2,])
#' IndNPETe.domset(S,dat,r,M)
#'
#' S<-rbind(dat[1,],dat[2,],dat[3,],dat[5,])
#' IndNPETe.domset(S,dat,r,M)
#'
#' S<-rbind(c(.1,.1),c(.3,.4),c(.5,.3))
#' IndNPETe.domset(S,dat,r,M)
#'
#' IndNPETe.domset(c(.2,.5),dat,r,M)
#' IndNPETe.domset(c(.2,.5),c(.2,.5),r,M)
#' IndNPETe.domset(dat[5,],dat[2,],r,M)
#'
#' S<-rbind(dat[1,],dat[2,],dat[3,],dat[5,],c(.2,.5))
#' IndNPETe.domset(S,dat[3,],r,M)
#'
#' IndNPETe.domset(dat,dat,r,M)
#'
#' P<-c(.4,.2)
#' S<-dat[c(1,3,4),]
#' IndNPETe.domset(dat,P,r,M)
#'
#' IndNPETe.domset(rbind(S,S),dat,r,M)
#'
#' dat.fr<-data.frame(a=dat)
#' IndNPETe.domset(S,dat.fr,r,M)
#'
#' @export IndNPETe.domset
IndNPETe.domset <- function(S,Dt,r,M=c(1,1,1))
{
  if (!is.numeric(as.matrix(S)) || !is.numeric(as.matrix(Dt)))
  {stop('Both arguments must be numeric')}

  if (is.point(S))
  { S<-matrix(S,ncol=2)
  } else
  {S<-as.matrix(S)
  if (ncol(S)!=2 )
  {stop('S must be of dimension nx2')}
  }

  if (is.point(Dt))
  { Dt<-matrix(Dt,ncol=2)
  } else
  {Dt<-as.matrix(Dt)
  if (ncol(Dt)!=2 )
  {stop('Dt must be of dimension nx2')}
  }

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if (!is.point(M) && !is.point(M,3) )
  {stop('M must be a numeric 2D point for Cartesian coordinates or 3D point for barycentric coordinates')}

  if (dimension(M)==3)
  {
    M<-bary2cart(M,Te)
  }

  A<-c(0,0); B<-c(1,0); C<-c(1/2,sqrt(3)/2);
  Te<-rbind(A,B,C);

  if (in.triangle(M,Te,boundary=FALSE)$in.tri==F)
  {stop('center is not in the interior of the triangle')}

  k<-nrow(S);
  n<-nrow(Dt);

  dom<-1; i<-1;
  while (dom ==1 && i<= n)
  {
    if (IndNPETeSet(S,Dt[i,],r,M)==0)  #this is where std equilateral triangle Te is implicitly used
    {dom<-0};
    i<-i+1;
  }
  dom
} #end of the function
#'

#################################################################

#' @title The indicator for the set of points \code{S} being a dominating set or not for Proportional Edge Proximity
#' Catch Digraphs (PE-PCDs) - one triangle case
#'
#' @description Returns \eqn{I(}\code{S} a dominating set of PE-PCD whose vertices are the data set \code{Dt}\eqn{)}, that is,
#' returns 1 if \code{S} is a dominating set of PE-PCD, returns 0 otherwise.
#'
#' PE proximity region is constructed with
#' respect to the triangle \code{tri} with the expansion parameter \eqn{r \ge 1} and vertex regions are based
#' on the center \eqn{M=(m_1,m_2)} in Cartesian coordinates or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates
#' in the interior of the triangle \code{tri} or based on the circumcenter of \code{tri};
#' default is \eqn{M=(1,1,1)} i.e., the center of mass of \code{tri}.
#' The triangle \code{tri}\eqn{=T(A,B,C)} has edges \eqn{AB}, \eqn{BC}, \eqn{AC} which are also labeled as edges 3, 1, and 2, respectively.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:masa-2007,ceyhan:dom-num-NPE-Spat2011,ceyhan:mcap2012;textual}{pcds}).
#'
#' @param S A set of 2D points which is to be tested for being a dominating set for the PE-PCDs.
#' @param Dt A set of 2D points which constitute the vertices of the PE-PCD.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region
#' constructed in the triangle \code{tri}; must be \eqn{\ge 1}.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param M A 2D point in Cartesian coordinates or a 3D point in barycentric coordinates
#' which serves as a center in the interior of the triangle \code{tri} or the circumcenter of \code{tri};
#' default is \eqn{M=(1,1,1)} i.e., the center of mass of \code{tri}.
#'
#' @return \eqn{I(}\code{S} a dominating set of PE-PCD\eqn{)}, that is, returns 1 if \code{S} is a dominating set of PE-PCD whose
#' vertices are the data points in \code{Dt}; returns 0 otherwise, where PE proximity region is constructed in
#' the triangle \code{tri}
#'
#' @seealso \code{\link{IndNPETe.domset}}, \code{\link{IndNPEtriSet}}, \code{\link{IndNCStri.domset}}
#' and \code{\link{IndNAStri.domset}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#' n<-10
#'
#' set.seed(1)
#' dat<-runif.tri(n,Tr)$gen.points
#'
#' M<-as.numeric(runif.tri(1,Tr)$g)  #try also M<-c(1.6,1.0)
#'
#' r<-1.5
#'
#' S<-rbind(dat[1,],dat[2,])
#' IndNPEtri.domset(S,dat,r,Tr,M)
#'
#' S<-rbind(dat[1,],dat[2,],dat[3,],dat[5,])
#' IndNPEtri.domset(S,dat,r,Tr,M)
#'
#' S<-rbind(c(.1,.1),c(.3,.4),c(.5,.3))
#' IndNPEtri.domset(S,dat,r,Tr,M)
#'
#' IndNPEtri.domset(c(.2,.5),dat,r,Tr,M)
#' IndNPEtri.domset(c(.2,.5),c(.2,.5),r,Tr,M)
#'
#' IndNPEtri.domset(dat[5,],dat[2,],r,Tr,M)
#'
#' S<-rbind(dat[1,],dat[2,],dat[3,],dat[5,],c(.2,.5))
#' IndNPEtri.domset(S,dat[3,],r,Tr,M)
#'
#' IndNPEtri.domset(dat,dat,r,Tr,M)
#'
#' P<-c(.4,.2)
#' S<-dat[c(1,3,4),]
#' IndNPEtri.domset(dat,P,r,Tr,M)
#'
#' IndNPEtri.domset(rbind(S,S),dat,r,Tr,M)
#'
#' dat.fr<-data.frame(a=dat)
#' IndNPEtri.domset(S,dat.fr,r,Tr,M)
#'
#' @export IndNPEtri.domset
IndNPEtri.domset <- function(S,Dt,r,tri,M=c(1,1,1))
{
  if (!is.numeric(as.matrix(S)) || !is.numeric(as.matrix(Dt)))
  {stop('Both arguments must be numeric')}

  if (is.point(S))
  { S<-matrix(S,ncol=2)
  } else
  {S<-as.matrix(S)
  if (ncol(S)!=2 )
  {stop('S must be of dimension nx2')}
  }

  if (is.point(Dt))
  { Dt<-matrix(Dt,ncol=2)
  } else
  {Dt<-as.matrix(Dt)
  if (ncol(Dt)!=2 )
  {stop('Dt must be of dimension nx2')}
  }

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  if (!is.point(M) && !is.point(M,3) )
  {stop('M must be a numeric 2D point for Cartesian coordinates or 3D point for barycentric coordinates')}

  if (dimension(M)==3)
  {
    M<-bary2cart(M,tri)
  }

  if (isTRUE(all.equal(M,circ.cent.tri(tri)))==F & in.triangle(M,tri,boundary=FALSE)$in.tri==F)
  {stop('center is not the circumcenter or not in the interior of the triangle')}

  k<-nrow(S);
  n<-nrow(Dt);

  dom<-1; i<-1;
  while (dom ==1 && i<= n)
  {
    if (IndNPEtriSet(S,Dt[i,],r,tri,M)==0)  #this is where tri is used
    {dom<-0};
    i<-i+1;
  }
  dom
} #end of the function
#'

#################################################################

#' @title The arcs of Proportional Edge Proximity Catch Digraph (PE-PCD) for 2D data - one triangle case
#'
#' @description
#' An object of class \code{"PCDs"}.
#' Returns arcs as tails (or sources) and heads (or arrow ends) for data set \code{Xp} as the vertices
#' of PE-PCD.
#'
#' PE proximity regions are constructed with respect to the triangle \code{tri} with expansion
#' parameter \eqn{r \ge 1}, i.e., arcs may exist for points only inside \code{tri}.
#' It also provides various descriptions and quantities about the arcs of the PE-PCD
#' such as number of arcs, arc density, etc.
#'
#' Vertex regions are based on center \eqn{M=(m_1,m_2)}
#' in Cartesian coordinates or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the interior of
#' the triangle \code{tri} or based on the circumcenter of \code{tri}; default is \eqn{M=(1,1,1)} i.e., the center of mass of \code{tri}.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:arc-density-PE;textual}{pcds}).
#'
#' @param Xp A set of 2D points which constitute the vertices of the PE-PCD.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param M A 2D point in Cartesian coordinates or a 3D point in barycentric coordinates
#' which serves as a center in the interior of the triangle \code{tri} or the circumcenter of \code{tri};
#' default is \eqn{M=(1,1,1)} i.e., the center of mass of \code{tri}.
#'
#' @return A \code{list} with the elements
#' \item{type}{A description of the type of the digraph}
#' \item{parameters}{Parameters of the digraph, here, it is the center used to construct the vertex regions.}
#' \item{tess.points}{Points on which the tessellation of the study region is performed, here, tessellation
#' is the support triangle.}
#' \item{tess.name}{Name of data set used in tessellation (i.e., vertices of the triangle)}
#' \item{vertices}{Vertices of the digraph, \code{Xp} points}
#' \item{vert.name}{Name of the data set which constitute the vertices of the digraph}
#' \item{S}{Tails (or sources) of the arcs of PE-PCD for 2D data set \code{Xp} as vertices of the digraph}
#' \item{E}{Heads (or arrow ends) of the arcs of PE-PCD for 2D data set \code{Xp} as vertices of the digraph}
#' \item{mtitle}{Text for \code{"main"} title in the plot of the digraph}
#' \item{quant}{Various quantities for the digraph: number of vertices, number of partition points,
#' number of intervals, number of arcs, and arc density.}
#'
#' @seealso \code{\link{ArcsPEMT}}, \code{\link{ArcsAStri}} and \code{\link{ArcsCStri}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#' n<-10
#'
#' set.seed(1)
#' dat<-runif.tri(n,Tr)$g
#'
#' M<-as.numeric(runif.tri(1,Tr)$g)  #try also M<-c(1.6,1.0)
#'
#' r<-1.5  #try also r<-2
#'
#' ArcsPEtri(dat,Tr,r,M)
#'
#' Arcs<-ArcsPEtri(dat,Tr,r,M)
#' Arcs
#' summary(Arcs)
#' plot(Arcs)
#'
#' S<-Arcs$S
#' E<-Arcs$E
#'
#' Xlim<-range(Tr[,1],dat[,1],M[1])
#' Ylim<-range(Tr[,2],dat[,2],M[2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' Ds<-cp2e.tri(Tr,M)
#'
#' if (dimension(M)==3) {M<-bary2cart(M,Tr)}
#' #need to run this when M is given in barycentric coordinates
#'
#' plot(Tr,pch=".",xlab="",ylab="",axes=TRUE,
#' xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' polygon(Tr)
#' points(dat,pch=1,col=1)
#' L<-rbind(M,M,M); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#' arrows(S[,1], S[,2], E[,1], E[,2], length = 0.1, col= 4)
#'
#' txt<-rbind(Tr,M,Ds)
#' xc<-txt[,1]+c(-.02,.03,.02,.03,.04,-.03,-.01)
#' yc<-txt[,2]+c(.02,.02,.03,.06,.04,.05,-.07)
#' txt.str<-c("A","B","C","M","D1","D2","D3")
#' text(xc,yc,txt.str)
#'
#' r<-2
#' ArcsPEtri(dat,Tr,r,M)
#'
#' dat.fr<-data.frame(a=dat)
#' ArcsPEtri(dat.fr,Tr,r,M)
#'
#' dat.fr<-data.frame(a=Tr)
#' ArcsPEtri(dat,dat.fr,r,M)
#'
#' @export ArcsPEtri
ArcsPEtri <- function(Xp,tri,r,M=c(1,1,1))
{
  if (!is.numeric(as.matrix(Xp)) )
  {stop('Xp must be numeric')}

  if (is.point(Xp))
  { Xp<-matrix(Xp,ncol=2)
  } else
  {Xp<-as.matrix(Xp)
  if (ncol(Xp)!=2 )
  {stop('Xp must be of dimension nx2')}
  }

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if (!is.point(M) && !is.point(M,3) )
  {stop('M must be a numeric 2D point for Cartesian coordinates or 3D point for barycentric coordinates')}

  if (dimension(M)==3)
  {
    M<-bary2cart(M,tri)
  }

  if (isTRUE(all.equal(M,circ.cent.tri(tri)))==F & in.triangle(M,tri,boundary=FALSE)$in.tri==F)
  {stop('center is not the circumcenter or not in the interior of the triangle')}

  n<-nrow(Xp)

 in.tri<-rep(0,n)
  for (i in 1:n)
   in.tri[i]<-in.triangle(Xp[i,],tri,boundary=TRUE)$in.tri #indices the Xp points inside the triangle

  Xtri<-Xp[in.tri==1,] #the Xp points inside the triangle
  n2<-length(Xtri)/2

  #the arcs of PE-PCDs
  S<-E<-NULL #S is for source and E is for end points for the arcs
  if (n2>1)
  {
    for (j in 1:n2)
    {
      pt1<-Xtri[j,];
      RV1<-ifelse(isTRUE(all.equal(M,circ.cent.tri(tri)))==T,rv.triCC(pt1,tri)$rv,rv.tri.cent(pt1,tri,M)$rv);
      for (k in (1:n2)[-j])  #to avoid loops
      {
        pt2<-Xtri[k,];
        if (IndNPEtri(pt1,pt2,r,tri,M,RV1)==1)
        {
          S <-rbind(S,Xtri[j,]); E <-rbind(E,Xtri[k,]);
        }
      }
    }
  }

  xname <-deparse(substitute(Xp))
  yname <-deparse(substitute(tri))
  cname <-ifelse(isTRUE(all.equal(M,circ.cent.tri(tri)))==T,"CC","M")
  rname <-deparse(substitute(r))

  param<-list(r,M)
  names(param)<-c("expansion parameter","center")
  typ<-paste("Proportional Edge Proximity Catch Digraph (PE-PCD) for 2D Points in the Triangle with Expansion Parameter ",rname, "=",r," and Center ", cname,"=(",M[1],",",M[2],")",sep="")

  Mr<-round(M,2)
  main.txt<-paste("Arcs of PE-PCD for Points in One Triangle \n with ",rname, "=",r," and Center ", cname,"=(",Mr[1],",",Mr[2],")",sep="")

  nvert<-n2; ny<-3; ntri<-1; narcs=ifelse(!is.null(S),nrow(S),0);
  arc.dens<-ifelse(nvert>1,narcs/(nvert*(nvert-1)),NA)

  quantities<-c(nvert,ny,ntri,narcs,arc.dens)
  names(quantities)<-c("number of vertices", "number of partition points",
                       "number of triangles","number of arcs", "arc density")

  res<-list(
    type=typ,
    parameters=param,
    tess.points=tri, tess.name=yname, #tessellation points
    vertices=Xp, vert.name=xname, #vertices of the digraph
    S=S,
    E=E,
    mtitle=main.txt,
    quant=quantities
  )

  class(res)<-"PCDs"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

#' @title The plot of the arcs of Proportional Edge Proximity Catch Digraph (PE-PCD) for a
#' 2D data set - one triangle case
#'
#' @description Plots the arcs of PE-PCD whose vertices are the data points, \code{Xp} and the triangle \code{tri}. PE proximity regions
#' are constructed with respect to the triangle \code{tri} with expansion parameter \eqn{r \ge 1}, i.e., arcs may exist only
#' for \code{Xp} points inside the triangle \code{tri}.
#'
#' Vertex regions are based on center \eqn{M=(m_1,m_2)} in Cartesian coordinates
#' or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the interior of the triangle \code{tri}
#' or based on the circumcenter of \code{tri}; default is \eqn{M=(1,1,1)} i.e., the center of mass of \code{tri}.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:arc-density-PE,ceyhan:dom-num-NPE-Spat2011;textual}{pcds}).
#'
#' @param Xp A set of 2D points which constitute the vertices of the PE-PCD.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param M A 2D point in Cartesian coordinates or a 3D point in barycentric coordinates
#' which serves as a center in the interior of the triangle \code{tri} or the circumcenter of \code{tri};
#' default is \eqn{M=(1,1,1)} i.e., the center of mass of \code{tri}.
#' @param asp A \code{numeric} value, giving the aspect ratio \eqn{y/x} (default is \code{NA}), see the official help page for \code{asp} by
#' typing "\code{? asp}".
#' @param main An overall title for the plot (default=\code{""}).
#' @param xlab,ylab Titles for the \eqn{x} and \eqn{y} axes, respectively (default=\code{""} for both).
#' @param xlim,ylim Two \code{numeric} vectors of length 2, giving the \eqn{x}- and \eqn{y}-coordinate ranges
#' (default=\code{NULL} for both).
#' @param \dots	Additional \code{plot} parameters.
#'
#' @return A plot of the arcs of the PE-PCD whose vertices are the points in data set \code{Xp} and the triangle \code{tri}
#'
#' @seealso \code{\link{plotPEarcsMT}}, \code{\link{plotASarcsTri}} and \code{\link{plotASarcsMT}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#' n<-10  #try also n<-20
#'
#' set.seed(1)
#' dat<-runif.tri(n,Tr)$g
#'
#' M<-as.numeric(runif.tri(1,Tr)$g)  #try also M<-c(1.6,1.0)
#'
#' r<-1.5  #try also r<-2
#'
#' ifelse(isTRUE(all.equal(M,circ.cent.tri(Tr))),
#' Ds<-rbind((B+C)/2,(A+C)/2,(A+B)/2),Ds<-cp2e.tri(Tr,M))
#'
#' dat<-matrix(dat,ncol=2)
#'
#' Xlim<-range(Tr[,1],dat[,1],M[1])
#' Ylim<-range(Tr[,2],dat[,2],M[2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' if (dimension(M)==3) {M<-bary2cart(M,Tr)}
#' #need to run this when M is given in barycentric coordinates
#'
#' plotPEarcsTri(dat,Tr,r,M,main="arcs of PE-PCD with r=1.5",
#' xlab="",ylab="",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' if (dimension(M)==3) {M<-bary2cart(M,Tr)}
#' #need to run this when M is given in barycentric coordinates
#' L<-rbind(M,M,M); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#'
#' txt<-rbind(Tr,M,Ds)
#' xc<-txt[,1]+c(-.02,.03,.03,.03,.05,-0.03,-.01)
#' yc<-txt[,2]+c(.02,.02,.02,.07,.02,.05,-.06)
#' txt.str<-c("A","B","C","M","D1","D2","D3")
#' text(xc,yc,txt.str)
#'
#' @export plotPEarcsTri
plotPEarcsTri <- function(Xp,tri,r,M=c(1,1,1),asp=NA,main="",xlab="",ylab="",xlim=NULL,ylim=NULL, ...)
{
  ArcsPE<-ArcsPEtri(Xp,tri,r,M)
  S<-ArcsPE$S
  E<-ArcsPE$E

  Xp<-matrix(Xp,ncol=2)
  if (is.null(xlim))
  {xlim<-range(tri[,1],Xp[,1])}
  if (is.null(ylim))
  {ylim<-range(tri[,2],Xp[,2])}

  plot(Xp,main=main,asp=asp, xlab=xlab, ylab=ylab,xlim=xlim,ylim=ylim,pch=".",cex=3, ...)
  polygon(tri)
  if (!is.null(S)) {arrows(S[,1], S[,2], E[,1], E[,2], length = 0.1, col= 4)}
} #end of the function
#'

#################################################################

#' @title The plot of the Proportional Edge (PE) Proximity Regions for a 2D data set - one triangle case
#'
#' @description Plots the points in and outside of the triangle \code{tri} and also the PE proximity regions
#' for points in data set \code{Xp}.
#'
#' PE proximity regions are defined with respect to the triangle \code{tri}
#' with expansion parameter \eqn{r \ge 1}, so PE proximity regions are defined only for points inside the
#' triangle \code{tri}.
#'
#' Vertex regions are based on center \eqn{M=(m_1,m_2)} in Cartesian coordinates
#' or \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates in the interior of the triangle \code{tri}
#' or based on the circumcenter of \code{tri}; default is \eqn{M=(1,1,1)} i.e., the center of mass of \code{tri}.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:arc-density-PE,ceyhan:dom-num-NPE-Spat2011;textual}{pcds}).
#'
#' @param Xp A set of 2D points for which PE proximity regions are constructed.
#' @param tri Three 2D points, stacked row-wise, each row representing a vertex of the triangle.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param M A 2D point in Cartesian coordinates or a 3D point in barycentric coordinates
#' which serves as a center in the interior of the triangle \code{tri} or the circumcenter of \code{tri};
#' default is \eqn{M=(1,1,1)} i.e., the center of mass of \code{tri}.
#' @param asp A \code{numeric} value, giving the aspect ratio \eqn{y/x} (default is \code{NA}),
#' see the official help page for \code{asp} by typing "\code{? asp}".
#' @param main An overall title for the plot (default=\code{""}).
#' @param xlab,ylab Titles for the \eqn{x} and \eqn{y} axes, respectively (default=\code{""} for both).
#' @param xlim,ylim Two \code{numeric} vectors of length 2, giving the \eqn{x}- and \eqn{y}-coordinate ranges
#' (default=\code{NULL} for both).
#' @param \dots Additional \code{plot} parameters.
#'
#' @return Plot of the PE proximity regions for points inside the triangle \code{tri}
#' (and just the points outside \code{tri})
#'
#' @seealso \code{\link{plotPEregsMT}}, \code{\link{plotASregsTri}} and \code{\link{plotCSregsTri}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(1,1); B<-c(2,0); C<-c(1.5,2);
#' Tr<-rbind(A,B,C);
#' n<-10
#'
#' set.seed(1)
#' dat<-runif.tri(n,Tr)$g
#'
#' M<-as.numeric(runif.tri(1,Tr)$g)  #try also M<-c(1.6,1.0)
#'
#' r<-1.5  #try also r<-2
#'
#' ifelse(identical(M,circ.cent.tri(Tr)),Ds<-rbind((B+C)/2,(A+C)/2,(A+B)/2),Ds<-cp2e.tri(Tr,M))
#'
#' plotPEregsTri(dat,Tr,r,M)
#' plotPEregsTri(dat[1,],Tr,r,M)
#'
#' dat<-matrix(dat,ncol=2)
#' Xlim<-range(Tr[,1],dat[,1],M[1])
#' Ylim<-range(Tr[,2],dat[,2],M[2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' if (dimension(M)==3) {M<-bary2cart(M,Tr)}
#' #need to run this when M is given in barycentric coordinates
#' plotPEregsTri(dat,Tr,r,M,main="PE Proximity Regions with r=1.5",
#' xlab="",ylab="",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#' L<-rbind(M,M,M); R<-Ds
#' segments(L[,1], L[,2], R[,1], R[,2], lty=2)
#'
#' txt<-rbind(Tr,M,Ds)
#' xc<-txt[,1]+c(-.02,.03,.03,.03,.05,-0.03,-.01)
#' yc<-txt[,2]+c(.02,.02,.02,.07,.02,.05,-.06)
#' txt.str<-c("A","B","C","M","D1","D2","D3")
#' text(xc,yc,txt.str)
#'
#' @export plotPEregsTri
plotPEregsTri <- function(Xp,tri,r,M=c(1,1,1),asp=NA,main="",xlab="",ylab="",xlim=NULL,ylim=NULL, ...)
{
  if (!is.numeric(as.matrix(Xp)) )
  {stop('Xp must be numeric')}

  if (is.point(Xp))
  { Xp<-matrix(Xp,ncol=2)
  } else
  {Xp<-as.matrix(Xp)
  if (ncol(Xp)!=2 )
  {stop('Xp must be of dimension nx2')}
  }

  tri<-as.matrix(tri)
  if (!is.numeric(tri) || nrow(tri)!=3 || ncol(tri)!=2)
  {stop('tri must be numeric and of dimension 3x2')}

  vec1<-rep(1,3);
  D0<-det(matrix(cbind(tri,vec1),ncol=3))
  if (round(D0,14)==0)
  {stop('the triangle is degenerate')}

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if (!is.point(M) && !is.point(M,3) )
  {stop('M must be a numeric 2D point for Cartesian coordinates or 3D point for barycentric coordinates')}

  if (dimension(M)==3)
  {
    M<-bary2cart(M,tri)
  }

  if (isTRUE(all.equal(M,circ.cent.tri(tri)))==F & in.triangle(M,tri,boundary=FALSE)$in.tri==F)
  {stop('center is not the circumcenter or not in the interior of the triangle')}

  n<-nrow(Xp)

 in.tri<-rep(0,n)
  for (i in 1:n)
   in.tri[i]<-in.triangle(Xp[i,],tri,boundary=TRUE)$in.tri #indices of the Xp points inside the triangle

  Xtri<-matrix(Xp[in.tri==1,],ncol=2)  #the Xp points inside the triangle
  nt<-length(Xtri)/2 #number of Xp points inside the triangle

  if (is.null(xlim))
  {xlim<-range(tri[,1],Xp[,1])}
  if (is.null(ylim))
  {ylim<-range(tri[,2],Xp[,2])}

  xr<-xlim[2]-xlim[1]
  yr<-ylim[2]-ylim[1]

  plot(Xp,main=main, asp=asp, xlab=xlab, ylab=ylab,xlim=xlim+xr*c(-.05,.05),
       ylim=ylim+yr*c(-.05,.05),pch=".",cex=3, ...)
  polygon(tri,lty=2)
  if (nt>=1)
  {
    for (i in 1:nt)
    {
      P1<-Xtri[i,]
      RV<-ifelse(isTRUE(all.equal(M,circ.cent.tri(tri)))==T,rv.triCC(P1,tri)$rv,rv.tri.cent(P1,tri,M)$rv)

      pr<-NPEtri(P1,r,tri,M,rv=RV)
      polygon(pr,border="blue")
    }
  }
} #end of the function
#'

#################################################################

#' @title The arcs of Proportional Edge Proximity Catch Digraph (PE-PCD) for 2D data - multiple triangle case
#'
#' @description
#' An object of class \code{"PCDs"}.
#' Returns arcs as tails (or sources) and heads (or arrow ends) of Proportional Edge Proximity Catch Digraph
#' (PE-PCD) whose vertices are the data points in \code{Xp} in the multiple triangle case.
#'
#' PE proximity regions are
#' defined with respect to the Delaunay triangles based on \code{Yp} points with expansion parameter \eqn{r \ge 1} and
#' vertex regions in each triangle are based on the center \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates
#' in the interior of each Delaunay triangle or based on circumcenter of each Delaunay triangle
#' (default for \eqn{M=(1,1,1)} which is the center of mass of the triangle).
#' Each Delaunay triangle is first converted to an (unscaled) basic triangle so that \code{M} will be the same
#' type of center for each Delaunay triangle (this conversion is not necessary when \code{M} is \eqn{CM}).
#'
#' Convex hull of \code{Yp} is partitioned by the Delaunay triangles based on \code{Yp} points
#' (i.e., multiple triangles are the set of these Delaunay triangles whose union constitutes the
#' convex hull of \code{Yp} points). For the number of arcs, loops are not allowed so arcs are only possible
#' for points inside the convex hull of \code{Yp} points.
#'
#' See (\insertCite{ceyhan:Phd-thesis,ceyhan:arc-density-PE,ceyhan:dom-num-NPE-Spat2011;textual}{pcds})
#' for more on the PE-PCDs.
#' Also see (\insertCite{okabe:2000,ceyhan:comp-geo-2010,sinclair:2016;textual}{pcds}) for more on Delaunay triangulation and the corresponding algorithm.
#'
#' @param Xp A set of 2D points which constitute the vertices of the PE-PCD.
#' @param Yp A set of 2D points which constitute the vertices of the Delaunay triangles.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param M A 3D point in barycentric coordinates which serves as a center in the interior of each Delaunay
#' triangle or circumcenter of each Delaunay triangle (for this argument should be set as \code{M="CC"}),
#' default for \eqn{M=(1,1,1)} which is the center of mass of each triangle.
#'
#' @return A \code{list} with the elements
#' \item{type}{A description of the type of the digraph}
#' \item{parameters}{Parameters of the digraph, here, it is the center used to construct the vertex regions.}
#' \item{tess.points}{Points on which the tessellation of the study region is performed, here, tessellation
#' is Delaunay triangulation based on \code{Yp} points.}
#' \item{tess.name}{Name of data set used in tessellation, it is \code{Yp} for this function}
#' \item{vertices}{Vertices of the digraph, \code{Xp} points}
#' \item{vert.name}{Name of the data set which constitute the vertices of the digraph}
#' \item{S}{Tails (or sources) of the arcs of PE-PCD for 2D data set \code{Xp} as vertices of the digraph}
#' \item{E}{Heads (or arrow ends) of the arcs of PE-PCD for 2D data set \code{Xp} as vertices of the digraph}
#' \item{mtitle}{Text for \code{"main"} title in the plot of the digraph}
#' \item{quant}{Various quantities for the digraph: number of vertices, number of partition points,
#' number of intervals, number of arcs, and arc density.}
#'
#' @seealso \code{\link{ArcsPEtri}}, \code{\link{ArcsASMT}} and \code{\link{ArcsCSMT}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' #nx is number of X points (target) and ny is number of Y points (nontarget)
#' nx<-20; ny<-4;  #try also nx<-40; ny<-10 or nx<-1000; ny<-10;
#'
#' set.seed(1)
#' Xp<-cbind(runif(nx,0,1),runif(nx,0,1))
#' Yp<-cbind(runif(ny,0,1),runif(ny,0,1))
#'
#' M<-c(1,1,1)  #try also M<-c(1,2,3)
#'
#' r<-1.5  #try also r<-2
#'
#' ArcsPEMT(Xp,Yp,r,M)
#'
#' Arcs<-ArcsPEMT(Xp,Yp,r,M)
#' Arcs
#' summary(Arcs)
#' plot(Arcs)
#'
#' S<-Arcs$S
#' E<-Arcs$E
#' DT<-interp::tri.mesh(Yp[,1],Yp[,2],duplicate="remove")
#'
#' Xlim<-range(Xp[,1],Yp[,1])
#' Ylim<-range(Xp[,2],Yp[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plot(Xp,main=" ", xlab=" ", ylab=" ",
#' xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05),pch=".",cex=3)
#' interp::plot.triSht(DT, add=TRUE, do.points = TRUE)
#' arrows(S[,1], S[,2], E[,1], E[,2], length = 0.1, col= 4)
#'
#' ArcsPEMT(Xp,Yp,r)
#' ArcsPEMT(Xp,Yp[1:3,],r)
#'
#' ArcsPEMT(Xp,rbind(Yp,Yp),r)
#'
#' dat.fr<-data.frame(a=Xp)
#' ArcsPEMT(dat.fr,Yp,r)
#'
#' dat.fr<-data.frame(a=Yp)
#' ArcsPEMT(Xp,dat.fr,r)
#'
#' @export ArcsPEMT
ArcsPEMT <- function(Xp,Yp,r,M=c(1,1,1))
{
  if (!is.numeric(as.matrix(Xp)) || !is.numeric(as.matrix(Yp)))
  {stop('Xp and Yp must be numeric')}

  if (is.point(Xp))
  { Xp<-matrix(Xp,ncol=2)
  } else
  {Xp<-as.matrix(Xp)
  if (ncol(Xp)!=2 )
  {stop('Xp must be of dimension nx2')}
  }

  Yp<-as.matrix(Yp)
  if (ncol(Yp)!=2 || nrow(Yp)<3)
  {stop('Yp must be of dimension kx2 with k>=3')}

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if (nrow(Yp)==3)
  {
    res<-ArcsPEtri(Xp,Yp,r,M)
  } else
  {

    if ((!is.point(M,3) && M!="CC") || !all(M>0))
    {stop('M must be a numeric 3D point with positive barycentric coordinates or "CC" for circumcenter')}

    DTmesh<-interp::tri.mesh(Yp[,1],Yp[,2],duplicate="remove")

    nx<-nrow(Xp)
    ch<-rep(0,nx)
    for (i in 1:nx)
      ch[i]<-interp::in.convex.hull(DTmesh,Xp[i,1],Xp[i,2])

    Xch<-matrix(Xp[ch==1,],ncol=2)  #the Xp points inside the convex hull of Yp

    DTr<-matrix(interp::triangles(DTmesh)[,1:3],ncol=3)
    nt<-nrow(DTr)
    nx2<-nrow(Xch)

    S<-E<-NULL #S is for source and E is for end points for the arcs
    if (nx2>1)
    {
      i.tr<-rep(0,nx2)  #the vector of indices for the triangles that contain the Xp points
      for (i in 1:nx2)
        for (j in 1:nt)
        {
          tri<-Yp[DTr[j,],]
          if (in.triangle(Xch[i,],tri,boundary=TRUE)$in.tri )
            i.tr[i]<-j
        }

      for (i in 1:nt)
      {
        Xl<-matrix(Xch[i.tr==i,],ncol=2)
        if (nrow(Xl)>1)
        {
          Yi.Tri<-Yp[DTr[i,],] #vertices of the ith triangle
          Yi.tri<-as.bastri(Yi.Tri)$tri #convert the triangle Yi.Tri into an unscaled basic triangle, see as.bastri help page
          nl<-nrow(Xl)
          ifelse(identical(M,"CC"), {rv.ind<-rverts.triCC(Xl,Yi.tri)$rv; cent<-circ.cent.tri(Yi.tri)},
                 {rv.ind<-rverts.tri.cent(Xl,Yi.tri,M)$rv; cent<-M})

          for (j in 1:nl)
          {RV<-rv.ind[j]
          for (k in (1:nl)[-j])  # to avoid loops
            if (IndNPEtri(Xl[j,],Xl[k,],r,Yi.tri,cent,rv=RV)==1 )
            {
              S <-rbind(S,Xl[j,]); E <-rbind(E,Xl[k,]);
            }
          }
        }
      }
    }

    xname <-deparse(substitute(Xp))
    yname <-deparse(substitute(Yp))
    rname <-deparse(substitute(r))

    param<-r
    names(param)<-"expansion parameter"
    typ<-paste("Proportional Edge Proximity Catch Digraph (PE-PCD) for 2D points in Multiple Triangles with Expansion parameter ", rname, "=",r," and Center M",sep="")
    main.txt<-paste("Arcs of PE-PCD for Points in Multiple Triangles\n with ", rname, "=",r," and Center M",sep="")

    nvert<-nx2; ny<-nrow(Yp); ntri<-nt; narcs=ifelse(!is.null(S),nrow(S),0);
    arc.dens<-ifelse(nvert>1,narcs/(nvert*(nvert-1)),NA)

    quantities<-c(nvert,ny,ntri,narcs,arc.dens)
    names(quantities)<-c("number of vertices", "number of partition points",
                         "number of triangles","number of arcs", "arc density")

    res<-list(
      type=typ,
      parameters=param,
      tess.points=Yp, tess.name=yname, #tessellation points
      vertices=Xp, vert.name=xname, #vertices of the digraph
      S=S,
      E=E,
      mtitle=main.txt,
      quant=quantities
    )

    class(res)<-"PCDs"
    res$call <-match.call()
  }
  res
} #end of the function
#'

#################################################################

#' @title Incidence matrix for Proportional Edge Proximity Catch Digraphs (PE-PCDs) - multiple triangle case
#'
#' @description Returns the incidence matrix of Proportional Edge Proximity Catch Digraph
#' (PE-PCD) whose vertices are the data points in \code{Xp} in the multiple triangle case.
#'
#' PE proximity regions are
#' defined with respect to the Delaunay triangles based on \code{Yp} points with expansion parameter \eqn{r \ge 1} and
#' vertex regions in each triangle are based on the center \eqn{M=(\alpha,\beta,\gamma)} in barycentric coordinates
#' in the interior of each Delaunay triangle or based on circumcenter of each Delaunay triangle
#' (default for \eqn{M=(1,1,1)} which is the center of mass of the triangle).
#'
#' Each Delaunay triangle is first converted to an (unscaled) basic triangle so that \code{M} will be the same
#' type of center for each Delaunay triangle (this conversion is not necessary when \code{M} is \eqn{CM}).
#'
#' Convex hull of \code{Yp} is partitioned by the Delaunay triangles based on \code{Yp} points
#' (i.e., multiple triangles are the set of these Delaunay triangles whose union constitutes the
#' convex hull of \code{Yp} points). For the incidence matrix loops are allowed,
#' so the diagonal entries are all equal to 1.
#'
#' See (\insertCite{ceyhan:Phd-thesis,ceyhan:arc-density-PE,ceyhan:dom-num-NPE-Spat2011;textual}{pcds}) for more
#' on the PE-PCDs.
#' Also see (\insertCite{okabe:2000,ceyhan:comp-geo-2010,sinclair:2016;textual}{pcds}) for more on Delaunay triangulation and the corresponding algorithm.
#'
#' @param Xp A set of 2D points which constitute the vertices of the PE-PCD.
#' @param Yp A set of 2D points which constitute the vertices of the Delaunay triangles.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param M A 3D point in barycentric coordinates which serves as a center in the interior of each Delaunay
#' triangle or circumcenter of each Delaunay triangle (for this argument should be set as \code{M="CC"}),
#' default for \eqn{M=(1,1,1)} which is the center of mass of each triangle.
#'
#' @return Incidence matrix for the PE-PCD with vertices being 2D data set, \code{Xp}.
#' PE proximity regions are constructed with respect to the Delaunay triangles and \code{M}-vertex regions.
#'
#' @seealso \code{\link{IncMatPEtri}}, \code{\link{IncMatPETe}}, \code{\link{IncMatASMT}}, and \code{\link{IncMatCSMT}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' nx<-20; ny<-4;  #try also nx<-40; ny<-10 or nx<-1000; ny<-10;
#'
#' set.seed(1)
#' Xp<-cbind(runif(nx,0,1),runif(nx,0,1))
#' Yp<-cbind(runif(ny,0,1),runif(ny,0,1))
#'
#' M<-c(1,1,1)  #try also M<-c(1,2,3)
#'
#' r<-1.5  #try also r<-2
#'
#' IM<-IncMatPEMT(Xp,Yp,r,M)
#' IM
#' dom.greedy(IM)
#' \donttest{
#' dom.exact(IM)  #might take a long time in this brute-force fashion ignoring the
#' #disconnected nature of the digraph inherent by the geometric construction of it
#' }
#' PEdomMTnd(Xp,Yp,r)
#'
#' Arcs<-ArcsPEMT(Xp,Yp,r,M)
#' Arcs
#' summary(Arcs)
#' plot(Arcs)
#'
#' IncMatPEMT(Xp,Yp,r,M)
#' IncMatPEMT(Xp,Yp[1:3,],r,M)
#'
#' IncMatPEMT(Xp,rbind(Yp,Yp),r,M)
#'
#' dat.fr<-data.frame(a=Xp)
#' IncMatPEMT(dat.fr,Yp,r,M)
#'
#' dat.fr<-data.frame(a=Yp)
#' IncMatPEMT(Xp,dat.fr,r,M)
#'
#' @export IncMatPEMT
IncMatPEMT <- function(Xp,Yp,r,M=c(1,1,1))
{
  if (!is.numeric(as.matrix(Xp)) || !is.numeric(as.matrix(Yp)))
  {stop('Xp and Yp must be numeric')}

  if (is.point(Xp))
  { Xp<-matrix(Xp,ncol=2)
  } else
  {Xp<-as.matrix(Xp)
  if (ncol(Xp)!=2 )
  {stop('Xp must be of dimension nx2')}
  }

  Yp<-as.matrix(Yp)
  if (ncol(Yp)!=2 || nrow(Yp)<3)
  {stop('Yp must be of dimension kx2 with k>=3')}

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if (nrow(Yp)==3)
  {
   inc.mat<-IncMatPEtri(Xp,Yp,r,M)
  } else
  {

    if ((!is.point(M,3) && M!="CC") || !all(M>0))
    {stop('M must be a numeric 3D point with positive barycentric coordinates or "CC" for circumcenter')}

    DTmesh<-interp::tri.mesh(Yp[,1],Yp[,2],duplicate="remove")

    nx<-nrow(Xp)
    ch<-rep(0,nx)
    for (i in 1:nx)
      ch[i]<-interp::in.convex.hull(DTmesh,Xp[i,1],Xp[i,2])

   inc.mat<-matrix(0, nrow=nx, ncol=nx)

    DTr<-matrix(interp::triangles(DTmesh)[,1:3],ncol=3)
    nt<-nrow(DTr)  #number of Delaunay triangles

    if (nx>1)
    {
      i.tr<-rep(0,nx)  #the vector of indices for the triangles that contain the Xp points
      for (i in 1:nx)
        for (j in 1:nt)
        {
          tri<-Yp[DTr[j,],]
          if (in.triangle(Xp[i,],tri,boundary=TRUE)$in.tri )
            i.tr[i]<-j
        }

      for (i in 1:nx)
      {pt1<-Xp[i,]
      if (i.tr[i]!=0)
      {
        Yi.Tri<-Yp[DTr[i.tr[i],],] #vertices of the ith triangle
        Yi.tri<-as.bastri(Yi.Tri)$tri #convert the triangle Yi.Tri into an unscaled basic triangle, see as.bastri help page
        ifelse(identical(M,"CC"),{vert<-rv.triCC(pt1,Yi.tri)$rv; cent<-circ.cent.tri(Yi.tri)},
               {vert<-rv.tri.cent(pt1,Yi.tri,M)$rv; cent<-M})

        for (j in 1:nx )
        {pt2<-Xp[j,]
       inc.mat[i,j]<-IndNPEtri(pt1,pt2,r,Yi.tri,cent,rv=vert)
        }
      }
      }
    }

    diag(inc.mat)<-1
  }
 inc.mat
} #end of the function
#'

#################################################################

#' @title Points from one class inside the convex hull of the points from the other class
#'
#' @description Given two 2D data sets, \code{Xp} and \code{Yp}, it returns the \code{Xp} points inside the convex hull of \code{Yp} points.
#'
#' See (\insertCite{okabe:2000,ceyhan:comp-geo-2010,sinclair:2016;textual}{pcds}) for more on Delaunay triangulation and the corresponding algorithm.
#'
#' @param Xp A set of 2D points which constitute the data set.
#' @param Yp A set of 2D points which constitute the vertices of the Delaunay triangles.
#'
#' @return \code{Xp} points inside the convex hull of \code{Yp} points
#'
#' @seealso \code{\link{plotDeltri}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' #nx is number of X points (target) and ny is number of Y points (nontarget)
#' nx<-20; ny<-4;  #try also nx<-40; ny<-10 or nx<-1000; ny<-10;
#'
#' set.seed(1)
#' Xp<-cbind(runif(nx,0,1),runif(nx,0,1))
#' Yp<-cbind(runif(ny,0,1),runif(ny,0,1))
#'
#' DT<-interp::tri.mesh(Yp[,1],Yp[,2],duplicate="remove")
#'
#' Xlim<-range(Xp[,1],Yp[,1])
#' Ylim<-range(Xp[,2],Yp[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' Xch<-XinCHY(Xp,Yp)
#'
#' plot(Xp,main=" ", xlab=" ", ylab=" ",
#' xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05),pch=".",cex=3)
#' interp::convex.hull(DT,plot.it = TRUE, add = TRUE)  # or try polygon(Yp[ch$i,])
#' points(Xch,pch=4,col="red")
#'
#' XinCHY(Xp,Yp)
#' XinCHY(Xp,Yp[1:3,])
#'
#' XinCHY(Xp,rbind(Yp,Yp))
#'
#' dat.fr<-data.frame(a=Xp)
#' XinCHY(dat.fr,Yp)
#'
#' dat.fr<-data.frame(a=Yp)
#' XinCHY(Xp,dat.fr)
#'
#' @export XinCHY
XinCHY <- function(Xp,Yp)
{
  if (!is.numeric(as.matrix(Xp)) || !is.numeric(as.matrix(Yp)))
  {stop('both arguments must be numeric')}

  if (is.point(Xp))
  { Xp<-matrix(Xp,ncol=2)
  } else
  {Xp<-as.matrix(Xp)
  if (ncol(Xp)!=2 )
  {stop('Xp must be of dimension nx2')}
  }

  Yp<-as.matrix(Yp)
  if (ncol(Yp)!=2 || nrow(Yp)<3)
  {stop('Yp must be of dimension kx2 with k>=3')}

  nx<-nrow(Xp)

  if (nrow(Yp)==3)
  {
    ch<-rep(0,nx)
    for (i in 1:nx)
    {ch[i]<-in.triangle(Xp[i,],Yp,boundary = TRUE)$in.tri}

    Xch<-Xp[ch==1,] #the Xp points inside the convex hull of Yp
  } else
  {
    DT<-interp::tri.mesh(Yp[,1],Yp[,2],duplicate="remove")

    ch<-rep(0,nx)
    for (i in 1:nx)
    {ch[i]<-interp::in.convex.hull(DT,Xp[i,1],Xp[i,2])}

    Xch<-Xp[ch==1,] #the Xp points inside the convex hull of Yp
  }
  Xch
} #end of the function
#'

#################################################################

#' @title The plot of the arcs of Proportional Edge Proximity Catch Digraph (PE-PCD) for a
#' 2D data set - multiple triangle case
#'
#' @description Plots the arcs of Proportional Edge Proximity Catch Digraph (PE-PCD) whose vertices are the data
#' points in \code{Xp} in the multiple triangle case and the Delaunay triangles based on \code{Yp} points.
#'
#' PE proximity regions are defined with respect to the Delaunay triangles based on \code{Yp} points with
#' expansion parameter \eqn{r \ge 1} and vertex regions in each triangle are based on the center \eqn{M=(\alpha,\beta,\gamma)}
#' in barycentric coordinates in the interior of each Delaunay triangle or based on circumcenter of
#' each Delaunay triangle (default for \eqn{M=(1,1,1)} which is the center of mass of the triangle).
#' Each Delaunay triangle is first converted to an (unscaled) basic triangle so that \code{M} will be the same
#' type of center for each Delaunay triangle (this conversion is not necessary when \code{M} is \eqn{CM}).
#'
#' Convex hull of \code{Yp} is partitioned by the Delaunay triangles based on \code{Yp} points
#' (i.e., multiple triangles are the set of these Delaunay triangles whose union constitutes the
#' convex hull of \code{Yp} points). Loops are not allowed so arcs are only possible
#' for points inside the convex hull of \code{Yp} points.
#'
#' See (\insertCite{ceyhan:Phd-thesis,ceyhan:arc-density-PE,ceyhan:dom-num-NPE-Spat2011;textual}{pcds}) for more
#' on the PE-PCDs.
#' Also see (\insertCite{okabe:2000,ceyhan:comp-geo-2010,sinclair:2016;textual}{pcds}) for more on Delaunay triangulation and the corresponding algorithm.
#'
#' @param Xp A set of 2D points which constitute the vertices of the PE-PCD.
#' @param Yp A set of 2D points which constitute the vertices of the Delaunay triangles.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param M A 3D point in barycentric coordinates which serves as a center in the interior of each Delaunay
#' triangle or circumcenter of each Delaunay triangle (for this argument should be set as \code{M="CC"}),
#' default for \eqn{M=(1,1,1)} which is the center of mass of each triangle.
#' @param asp A \code{numeric} value, giving the aspect ratio \eqn{y/x} (default is \code{NA}),
#' see the official help page for \code{asp} by typing "\code{? asp}".
#' @param main An overall title for the plot (default=\code{""}).
#' @param xlab,ylab Titles for the \eqn{x} and \eqn{y} axes, respectively (default=\code{""} for both).
#' @param xlim,ylim Two \code{numeric} vectors of length 2, giving the \eqn{x}- and \eqn{y}-coordinate ranges
#' (default=\code{NULL} for both).
#' @param \dots Additional \code{plot} parameters.
#'
#' @return A plot of the arcs of the PE-PCD whose vertices are the points in data set \code{Xp} and the Delaunay
#' triangles based on \code{Yp} points
#'
#' @seealso \code{\link{plotPEarcsTri}}, \code{\link{plotASarcsMT}}, and \code{\link{plotCSarcsMT}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' #nx is number of X points (target) and ny is number of Y points (nontarget)
#' nx<-20; ny<-4;  #try also nx<-40; ny<-10 or nx<-1000; ny<-10;
#'
#' set.seed(1)
#' Xp<-cbind(runif(nx,0,1),runif(nx,0,1))
#' Yp<-cbind(runif(ny,0,1),runif(ny,0,1))
#'
#' M<-c(1,1,1)  #try also M<-c(1,2,3)
#'
#' r<-1.5  #try also r<-2
#'
#' Xlim<-range(Xp[,1],Yp[,1])
#' Ylim<-range(Xp[,2],Yp[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' plotPEarcsMT(Xp,Yp,r,M,xlab="",ylab="",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#'
#' plotPEarcsMT(Xp,Yp[1:3,],r,M,xlab="",ylab="",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#'
#' @export plotPEarcsMT
plotPEarcsMT <- function(Xp,Yp,r,M=c(1,1,1),asp=NA,main="",xlab="",ylab="",xlim=NULL,ylim=NULL, ...)
{
  Yp<-as.matrix(Yp)
  if (ncol(Yp)!=2 || nrow(Yp)<3)
  {stop('Yp must be of dimension kx2 with k>=3')}

  if (nrow(Yp)==3)
  {
    plotPEarcsTri(Xp,Yp,r,M,asp,main,xlab,ylab,xlim,ylim)
  } else
  {
    ArcsPE<-ArcsPEMT(Xp,Yp,r,M)
    S<-ArcsPE$S
    E<-ArcsPE$E

    DTmesh<-interp::tri.mesh(Yp[,1],Yp[,2],duplicate="remove")

    Xch<-XinCHY(Xp,Yp)

    plot(rbind(Xp),asp=asp,main=main, xlab=xlab, ylab=ylab,xlim=xlim,ylim=ylim,pch=".",cex=3, ...)
   interp::plot.triSht(DTmesh, add=TRUE, do.points = TRUE)
    if (!is.null(S)) {arrows(S[,1], S[,2], E[,1], E[,2], length = 0.1, col= 4)}
  }
} #end of the function
#'

#################################################################

#' @title The plot of the Proportional Edge (PE) Proximity Regions for a 2D data set - multiple triangle case
#'
#' @description Plots the points in and outside of the Delaunay triangles based on \code{Yp} points which partition
#' the convex hull of \code{Yp} points and also plots the PE proximity regions
#' for \code{Xp} points and the Delaunay triangles based on \code{Yp} points.
#'
#' PE proximity regions are constructed with respect to the Delaunay triangles with the expansion parameter
#' \eqn{r \ge 1}.
#'
#' Vertex regions in each triangle is based on the center \eqn{M=(\alpha,\beta,\gamma)}
#' in barycentric coordinates in the interior of each Delaunay triangle or based on circumcenter of
#' each Delaunay triangle (default for \eqn{M=(1,1,1)} which is the center of mass of the triangle).
#'
#' See (\insertCite{ceyhan:Phd-thesis,ceyhan:arc-density-PE,ceyhan:dom-num-NPE-Spat2011;textual}{pcds}) for more
#' on the PE proximity regions.
#' Also see (\insertCite{okabe:2000,ceyhan:comp-geo-2010,sinclair:2016;textual}{pcds}) for more on Delaunay triangulation and the corresponding algorithm.
#'
#' @param Xp A set of 2D points for which PE proximity regions are constructed.
#' @param Yp A set of 2D points which constitute the vertices of the Delaunay triangles.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param M A 2D point in Cartesian coordinates or a 3D point in barycentric coordinates
#' which serves as a center in the interior of the triangle \code{tri} or the circumcenter of \code{tri}.
#' @param asp A \code{numeric} value, giving the aspect ratio \eqn{y/x} (default is \code{NA}),
#' see the official help page for \code{asp} by typing "\code{? asp}".
#' @param main An overall title for the plot (default=\code{""}).
#' @param xlab,ylab Titles for the \eqn{x} and \eqn{y} axes, respectively (default=\code{""} for both)
#' @param xlim,ylim Two \code{numeric} vectors of length 2, giving the \eqn{x}- and \eqn{y}-coordinate ranges
#' (default=\code{NULL} for both).
#' @param \dots Additional \code{plot} parameters.
#'
#' @return Plot of the \code{Xp} points, Delaunay triangles based on \code{Yp} points and also the PE proximity regions
#' for \code{Xp} points inside the convex hull of \code{Yp} points
#'
#' @seealso \code{\link{plotPEregsTri}}, \code{\link{plotASregsMT}} and \code{\link{plotCSregsMT}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' #nx is number of X points (target) and ny is number of Y points (nontarget)
#' nx<-20; ny<-4;  #try also nx<-40; ny<-10 or nx<-1000; ny<-10;
#'
#' set.seed(1)
#' Xp<-cbind(runif(nx,0,1),runif(nx,0,1))
#' Yp<-cbind(runif(ny,0,1),runif(ny,0,1))
#'
#' Xlim<-range(Xp[,1],Yp[,1])
#' Ylim<-range(Xp[,2],Yp[,2])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#'
#' M<-c(1,1,1)  #try also M<-c(1,2,3)
#'
#' r<-1.5  #try also r<-2
#'
#' plotPEregsMT(Xp,Yp,r,M,xlab="",ylab="",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#'
#' plotPEregsMT(Xp,Yp[1:3,],r,M,xlab="",ylab="",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05))
#'
#' @export plotPEregsMT
plotPEregsMT <- function(Xp,Yp,r,M=c(1,1,1),asp=NA,main="",xlab="",ylab="",xlim=NULL,ylim=NULL, ...)
{
  if (!is.numeric(as.matrix(Xp)) || !is.numeric(as.matrix(Yp)))
  {stop('Xp and Yp must be numeric')}

  if (is.point(Xp))
  { Xp<-matrix(Xp,ncol=2)
  } else
  {Xp<-as.matrix(Xp)
  if (ncol(Xp)!=2 )
  {stop('Xp must be of dimension nx2')}
  }

  Yp<-as.matrix(Yp)
  if (ncol(Yp)!=2 || nrow(Yp)<3)
  {stop('Yp must be of dimension kx2 with k>=3')}

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if (nrow(Yp)==3)
  {
    plotPEregsTri(Xp,Yp,r,M,asp,main,xlab,ylab,xlim,ylim)
  } else
  {

    if ((!is.point(M,3) && M!="CC") || !all(M>0))
    {stop('M must be a numeric 3D point with positive barycentric coordinates or "CC" for circumcenter')}

    DTmesh<-interp::tri.mesh(Yp[,1],Yp[,2],duplicate="remove")

    nx<-nrow(Xp)
    ch<-rep(0,nx)
    for (i in 1:nx)
      ch[i]<-interp::in.convex.hull(DTmesh,Xp[i,1],Xp[i,2])

    Xch<-matrix(Xp[ch==1,],ncol=2)  #the Xp points inside the convex hull of Yp points

    DTr<-matrix(interp::triangles(DTmesh)[,1:3],ncol=3)
    nt<-nrow(DTr)  #number of Delaunay triangles
    nx2<-nrow(Xch)  #number of Xp points inside the convex hull of Yp points

    if (nx2>=1)
    {
      i.tr<-rep(0,nx2)  #the vector of indices for the triangles that contain the Xp points
      for (i1 in 1:nx2)
        for (j1 in 1:nt)
        {
          Tri<-Yp[DTr[j1,],]
          if (in.triangle(Xch[i1,],Tri,boundary=TRUE)$in.tri )
            i.tr[i1]<-j1
        }
    }

    Xlim<-xlim; Ylim<-ylim
    if (is.null(xlim))
    {xlim<-range(Yp[,1],Xp[,1])
    xr<-xlim[2]-xlim[1]
    Xlim<-xlim+xr*c(-.05,.05)
    }
    if (is.null(ylim))
    {ylim<-range(Yp[,2],Xp[,2])
    yr<-ylim[2]-ylim[1]
    Ylim<-ylim+yr*c(-.05,.05)
    }

    plot(rbind(Xp),asp=asp,main=main, xlab=xlab, ylab=ylab,
         xlim=Xlim,ylim=Ylim,pch=".",cex=3, ...)

    for (i in 1:nt)
    {
      Tri<-Yp[DTr[i,],]  #vertices of the ith triangle
      tri<-as.bastri(Tri)$tri #convert the triangle Tri into an unscaled basic triangle, see as.bastri help page

      polygon(tri,lty=2)
      if (nx2>=1)
      {
        Xtri<-matrix(Xch[i.tr==i,],ncol=2)  #Xp points inside triangle i
        ni<-nrow(Xtri)
        if (ni>=1)
        {
          ################
          for (j in 1:ni)
          {
            P1<-Xtri[j,]
            ifelse(identical(M,"CC"),{RV<-rv.triCC(P1,tri)$rv; cent<-circ.cent.tri(tri)},
                   {RV<-rv.tri.cent(P1,tri,M)$rv; cent<-M})

            pr<-NPEtri(P1,r,tri,cent,rv=RV)
            polygon(pr,border="blue")

          }
          ################
        }
      }
    }
  }
} #end of the function
#'

#################################################################

#' @title The domination number of Proportional Edge Proximity Catch Digraph (PE-PCD) - multiple triangle case
#'
#' @description Returns the domination number and a minimum dominating set of PE-PCD whose vertices are the data
#' points in \code{Xp} in the multiple triangle case and the Delaunay triangles based on \code{Yp} points.
#'
#' PE proximity regions are defined with respect to the Delaunay triangles based on \code{Yp} points with
#' expansion parameter \eqn{r \ge 1} and vertex regions in each triangle are based on the center \eqn{M=(\alpha,\beta,\gamma)}
#' in barycentric coordinates in the interior of each Delaunay triangle or based on
#' circumcenter of each Delaunay triangle (default for \eqn{M=(1,1,1)} which is the center of mass of the
#' triangle). Each Delaunay triangle is first converted to an (unscaled) basic triangle so that \code{M} will be the
#' same type of center for each Delaunay triangle (this conversion is not necessary when \code{M} is \eqn{CM}).
#'
#' Convex hull of \code{Yp} is partitioned by the Delaunay triangles based on \code{Yp} points
#' (i.e., multiple triangles are the set of these Delaunay triangles whose union constitutes the
#' convex hull of \code{Yp} points). Loops are allowed for the domination number.
#'
#' See (\insertCite{ceyhan:Phd-thesis,ceyhan:masa-2007,ceyhan:dom-num-NPE-Spat2011,ceyhan:mcap2012;textual}{pcds}) for more on the domination number of
#' PE-PCDs. Also see (\insertCite{okabe:2000,ceyhan:comp-geo-2010,sinclair:2016;textual}{pcds}) for more on Delaunay triangulation and
#' the corresponding algorithm.
#'
#' @param Xp A set of 2D points which constitute the vertices of the PE-PCD.
#' @param Yp A set of 2D points which constitute the vertices of the Delaunay triangles.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param M A 3D point in barycentric coordinates which serves as a center in the interior of each Delaunay
#' triangle or circumcenter of each Delaunay triangle (for this argument should be set as \code{M="CC"}),
#' default for \eqn{M=(1,1,1)} which is the center of mass of each triangle.
#'
#' @return A \code{list} with two elements
#' \item{dom.num}{Domination number of the PE-PCD whose vertices are \code{Xp} points. PE proximity regions are
#' constructed with respect to the Delaunay triangles based on the \code{Yp} points with expansion parameter \eqn{r \ge 1}.}
#' \item{mds}{A minimum dominating set of the PE-PCD whose vertices are \code{Xp} points}
#'
#' @seealso \code{\link{PEdomtri}}, \code{\link{PEdom.tetra}}, \code{\link{dom.exact}}, and \code{\link{dom.greedy}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' #nx is number of X points (target) and ny is number of Y points (nontarget)
#' nx<-20; ny<-4;  #try also nx<-40; ny<-10 or nx<-1000; ny<-10;
#'
#' set.seed(1)
#' Xp<-cbind(runif(nx,0,1),runif(nx,0,1))
#' Yp<-cbind(runif(ny,0,1),runif(ny,0,1))
#'
#' M<-c(1,1,1)  #try also M<-c(1,2,3)
#'
#' r<-1.5  #try also r<-2
#'
#' PEdomMT(Xp,Yp,r,M)
#'
#' PEdomMT(Xp,Yp,r=1.4,M)
#' PEdomMT(Xp,Yp,r=2,M)
#'
#' r<-1.5  #try also r<-2
#'
#' PEdomMT(Xp,Yp,r,M)  #this may be different due to random selection of the center for r in (1,1.5)
#'
#' PEdomMT(Xp,Yp,r,M)
#' PEdomMT(Xp,Yp[1:3,],r,M)
#'
#' PEdomMT(Xp,rbind(Yp,Yp),r,M)
#'
#' dat.fr<-data.frame(a=Xp)
#' PEdomMT(dat.fr,Yp,r,M)
#'
#' dat.fr<-data.frame(a=Yp)
#' PEdomMT(Xp,dat.fr,r,M)
#'
#' @export PEdomMT
PEdomMT <- function(Xp,Yp,r,M=c(1,1,1))
{
  if (!is.numeric(as.matrix(Xp)) || !is.numeric(as.matrix(Yp)))
  {stop('Xp and Yp must be numeric')}

  if (is.point(Xp))
  { Xp<-matrix(Xp,ncol=2)
  } else
  {Xp<-as.matrix(Xp)
  if (ncol(Xp)!=2 )
  {stop('Xp must be of dimension nx2')}
  }

  Yp<-as.matrix(Yp)
  if (ncol(Yp)!=2 || nrow(Yp)<3)
  {stop('Yp must be of dimension kx2 with k>=3')}

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if(nrow(Yp)==3)
  {
    res<-PEdomtri(Xp,Yp,r,M)
  } else
  {

    if ((!is.point(M,3) && M!="CC") || !all(M>0))
    {stop('M must be a numeric 3D point with positive barycentric coordinates or "CC" for circumcenter')}

    n<-nrow(Xp)  #number of Xp points
    m<-nrow(Yp)  #number of Yp points

    Ytrimesh<-interp::tri.mesh(Yp[,1],Yp[,2],duplicate="remove")  #Delaunay triangulation
    Ytri<-matrix(interp::triangles(Ytrimesh)[,1:3],ncol=3); #the Delaunay triangles
    nt<-nrow(Ytri)  #number of Delaunay triangles
   inCH<-interp::in.convex.hull(Ytrimesh,Xp[,1],Xp[,2])  #logical indices for Xp points in convex hull of Yp points
    Xch<-matrix(Xp[inCH==TRUE,],ncol=2)

    gam<-rep(0,nt);  mds<-c()
    if (nrow(Xch)>=1)
    {
      Tri.Ind<-indices.Del.tri(Xch,Yp,Ytrimesh)  #indices of triangles in which the points in the data fall

      #calculation of the domination number
      for (i in 1:nt)
      {
        dati<-matrix(Xch[Tri.Ind==i,],ncol=2)  #points in ith Delaunay triangle
        ni<-nrow(dati)  #number of points in ith triangle
        if (ni==0)
        {
          gam[i]<-0
        } else
        {
          Yi.Tri<-Yp[Ytri[i,],] #vertices of ith triangle
          Yi.tri<-as.bastri(Yi.Tri)$tri
          ifelse(identical(M,"CC"), {cent<-circ.cent.tri(Yi.tri); Clvert<-cl2eVRCC(dati,Yi.tri)$ext},
                 {cent<-M; Clvert<-cl2eVRcent(dati,Yi.tri,cent)$ext})
          #closest points to edges in the respective vertex regions

          #Gamma=1 piece
          cnt<-0; j<-1;
          while (j<=3 & cnt==0)
          {
            if (Gam1PEtri(Clvert[j,],dati,Yi.tri,r,cent,rv=j)==1)
            {gam[i]<-1; cnt<-1; mds<-rbind(mds,Clvert[j,])
            } else
            {j<-j+1}
          }

          #Gamma=2 piece
          if (cnt==0)
          { k<-1; cnt2<-0;
          while (k<=2 & cnt2==0)
          {l<-k+1;
          while (l<=3 & cnt2==0)
          {
            if (Gam2PEtri(Clvert[k,],Clvert[l,],dati,Yi.tri,r,cent,rv1=k,rv2=l)==1)
            {gam[i]<-2;cnt2<-1; mds<-rbind(mds,Clvert[c(k,l),])
            } else {l<-l+1};
          }
          k<-k+1;
          }
          }

          if (cnt==0 && cnt2==0)
          {gam[i]<-3; mds<-rbind(mds,Clvert)}

        }

      }
    }

    Gam<-sum(gam)  #domination number for the entire digraph
    row.names(mds)<-c()

    res<-list(dom.num=Gam, #domination number
              mds=mds #a minimum dominating set
    )
  }
  res
} #end of the function
#'

#################################################################

#' @title The domination number of Proportional Edge Proximity Catch Digraph (PE-PCD) with
#' non-degeneracy centers - multiple triangle case
#'
#' @description Returns the domination number and a minimum dominating set of PE-PCD whose vertices are the data
#' points in \code{Xp} in the multiple triangle case and the Delaunay triangles based on \code{Yp} points.
#'
#' PE proximity regions are defined with respect to the Delaunay triangles based on \code{Yp} points with
#' expansion parameter \eqn{r \ge 1} and vertex regions in each triangle are based on the center \eqn{M=(\alpha,\beta,\gamma)}
#' in barycentric coordinates in the interior of each Delaunay triangle or based on
#' circumcenter of each Delaunay triangle (default for \eqn{M=(1,1,1)} which is the center \eqn{M}
#' where \code{M} is one of the 3 centers that renders the asymptotic distribution of domination number
#' to be non-degenerate for a given value of \code{r} in \eqn{(1,1.5)} and \code{M} is center of mass for \eqn{r=1.5}.
#'
#' Convex hull of \code{Yp} is partitioned by the Delaunay triangles based on \code{Yp} points
#' (i.e., multiple triangles are the set of these Delaunay triangles whose union constitutes the
#' convex hull of \code{Yp} points). Loops are allowed for the domination number.
#'
#' See (\insertCite{ceyhan:Phd-thesis,ceyhan:masa-2007,ceyhan:dom-num-NPE-Spat2011,ceyhan:mcap2012;textual}{pcds}) more on the domination number of
#' PE-PCDs. Also see (\insertCite{okabe:2000,ceyhan:comp-geo-2010,sinclair:2016;textual}{pcds}) for more on Delaunay triangulation and
#' the corresponding algorithm.
#'
#' @param Xp A set of 2D points which constitute the vertices of the PE-PCD.
#' @param Yp A set of 2D points which constitute the vertices of the Delaunay triangles.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#'
#' @return A \code{list} with two elements
#' \item{dom.num}{Domination number of the PE-PCD whose vertices are \code{Xp} points. PE proximity regions are
#' constructed with respect to the Delaunay triangles based on the \code{Yp} points with expansion parameter \eqn{r \ge 1}.}
#' \item{mds}{A minimum dominating set of the PE-PCD whose vertices are \code{Xp} points}
#'
#' @seealso \code{\link{PEdomtri}}, \code{\link{PEdom.tetra}}, \code{\link{dom.exact}}, and \code{\link{dom.greedy}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' #nx is number of X points (target) and ny is number of Y points (nontarget)
#' nx<-20; ny<-4;  #try also nx<-40; ny<-10 or nx<-1000; ny<-10;
#'
#' r<-1.5  #try also r<-2
#'
#' set.seed(1)
#' Xp<-cbind(runif(nx,0,1),runif(nx,0,1))
#' Yp<-cbind(runif(ny,0,1),runif(ny,0,1))
#'
#' PEdomMTnd(Xp,Yp,r)
#'
#' PEdomMTnd(Xp,Yp,r=1.4)
#'
#' r<-1.5  #try also  #r<-2
#' PEdomMTnd(Xp,Yp,r)  #this may be different due to random selection of the center for r in (1,1.5)
#'
#' PEdomMTnd(Xp,Yp[1:3,],r)
#'
#' PEdomMTnd(Xp,rbind(Yp,Yp),r)
#'
#' dat.fr<-data.frame(a=Xp)
#' PEdomMTnd(dat.fr,Yp,r)
#'
#' dat.fr<-data.frame(a=Yp)
#' PEdomMTnd(Xp,dat.fr,r)
#'
#' @export PEdomMTnd
PEdomMTnd <- function(Xp,Yp,r)
{
  if (!is.numeric(as.matrix(Xp)) || !is.numeric(as.matrix(Yp)))
  {stop('Xp and Yp must be numeric')}

  if (is.point(Xp))
  { Xp<-matrix(Xp,ncol=2)
  } else
  {Xp<-as.matrix(Xp)
  if (ncol(Xp)!=2 )
  {stop('Xp must be of dimension nx2')}
  }

  Yp<-as.matrix(Yp)
  if (ncol(Yp)!=2 || nrow(Yp)<3)
  {stop('Yp must be of dimension kx2 with k>=3')}

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if(nrow(Yp)==3)
  { rcent<-sample(1:3,1)  #random center selection from M1,M2,M3
  cent.nd<-cent.nondeg(Yp,r)[rcent,]
  res<-PEdomtri(Xp,Yp,r,cent.nd)
  } else
  {
    Ytrimesh<-interp::tri.mesh(Yp[,1],Yp[,2],duplicate="remove")  #Delaunay triangulation
    Ytri<-matrix(interp::triangles(Ytrimesh)[,1:3],ncol=3); #the Delaunay triangles
    nt<-nrow(Ytri)  #number of Delaunay triangles
   inCH<-interp::in.convex.hull(Ytrimesh,Xp[,1],Xp[,2])  #logical indices for Xp points in convex hull of Yp points
    Xch<-matrix(Xp[inCH==TRUE,],ncol=2)

    gam<-rep(0,nt);  mds<-c()
    if (nrow(Xch)>=1)
    {
      Tri.Ind<-indices.Del.tri(Xch,Yp,Ytrimesh)  #indices of triangles in which the points in the data fall

      #calculation of the domination number
      for (i in 1:nt)
      {
        dati<-matrix(Xch[Tri.Ind==i,],ncol=2)  #points in ith Delaunay triangle
        ni<-nrow(dati)  #number of points in ith triangle
        if (ni==0)
        {
          gam[i]<-0
        } else
        {
          Yi.tri<-Yp[Ytri[i,],] #vertices of ith triangle
          rcent<-sample(1:3,1)  #random center selection from M1,M2,M3
          Centi<-cent.nondeg(Yi.tri,r)[rcent,]
          Clvert<-cl2eVRcent(dati,Yi.tri,Centi)$ext  #for general r

          #Gamma=1 piece
          cnt<-0; j<-1;
          while (j<=3 & cnt==0)
          {
            if (Gam1PEtri(Clvert[j,],dati,Yi.tri,r,Centi,rv=j)==1)
            {gam[i]<-1; cnt<-1; mds<-rbind(mds,Clvert[j,])
            } else
            {j<-j+1}
          }

          #Gamma=2 piece
          if (cnt==0)
          { k<-1; cnt2<-0;
          while (k<=2 & cnt2==0)
          {l<-k+1;
          while (l<=3 & cnt2==0)
          {
            if (Gam2PEtri(Clvert[k,],Clvert[l,],dati,Yi.tri,r,Centi,rv1=k,rv2=l)==1)
            {gam[i]<-2;cnt2<-1; mds<-rbind(mds,Clvert[c(k,l),])
            } else {l<-l+1};
          }
          k<-k+1;
          }
          }

          if (cnt==0 && cnt2==0)
          {gam[i]<-3; mds<-rbind(mds,Clvert)}

        }

      }
    }

    Gam<-sum(gam)  #domination number for the entire digraph
    row.names(mds)<-c()

    res<-list(dom.num=Gam, #domination number
              mds=mds #a minimum dominating set
    )
  }
  res
} #end of the function
#'

#################################################################

#' @title Asymptotic probability that domination number of Proportional Edge Proximity Catch Digraphs (PE-PCDs)
#' equals 2 where vertices of the digraph are uniform points in a triangle
#'
#' @description Returns \eqn{P(}domination number\eqn{=2)} for PE-PCD for uniform data in a triangle, when the sample size \eqn{n} goes to
#' infinity (i.e., asymptotic probability of domination number \eqn{= 2}).
#'
#' PE proximity regions are constructed with respect to the triangle with the expansion parameter \eqn{r \ge 1} and
#' \eqn{M}-vertex regions where \eqn{M} is the vertex that renders the asymptotic distribution of the domination
#' number non-degenerate for the given value of \code{r} in \eqn{(1,1.5]}.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:masa-2007,ceyhan:dom-num-NPE-Spat2011;textual}{pcds}).
#'
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be in \eqn{(1,1.5]} to attain non-degenerate asymptotic distribution for the domination number.
#'
#' @return \eqn{P(}domination number\eqn{=2)} for PE-PCD for uniform data on an triangle as the sample size n
#' goes to infinity
#'
#' @seealso \code{\link{PG2PE1D}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' PG2PEtri(r=1.5)
#' PG2PEtri(r=1.4999999999)
#'
#' PG2PEtri(r=1.5) / PG2PEtri(r=1.4999999999)
#'
#' rseq<-seq(1.01,1.49999999999,l=20)  #try also l=100
#' lrseq<-length(rseq)
#'
#' pg2<-vector()
#' for (i in 1:lrseq)
#' {
#'   pg2<-c(pg2,PG2PEtri(rseq[i]))
#' }
#'
#' plot(rseq, pg2,type="l",xlab="r",ylab=expression(paste("P(", gamma, "=2)")),
#'      lty=1,xlim=range(rseq)+c(0,.01),ylim=c(0,1))
#' points(rbind(c(1.50,PG2PEtri(1.50))),pch=".",cex=3)
#'
#' @export PG2PEtri
PG2PEtri <- function(r)
{
  if (!is.point(r,1) || r<=1 || r>1.5)
  {stop('the argument must be a scalar in (1,1.5]')}

  if (r==1.5)
  {pg2<-.7413}
  else
  {pg2<--(1/2)*(pi*r^2-2*atan(r*(r-1)/sqrt(-r^4+2*r^3-r^2+1))*r^2-pi*r+2*atan(r*(r-1)/sqrt(-r^4+2*r^3-r^2+1))*r-2*sqrt(-r^4+2*r^3-r^2+1))/(-r^4+2*r^3-r^2+1)^(3/2)}
  pg2
} #end of the function
#'

#################################################################

#' @title Number of Delaunay triangles based on a 2D data set
#'
#' @description Returns the number of Delaunay triangles based on the 2D set of points \code{Yp}.
#' See (\insertCite{okabe:2000,sinclair:2016;textual}{pcds}) for more on Delaunay triangulation and
#' the corresponding algorithm.
#'
#' @param Yp A set of 2D points which constitute the vertices of Delaunay triangles.
#'
#' @return Number of Delaunay triangles based on \code{Yp} points.
#'
#' @seealso \code{\link{plotDeltri}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' ny<-10
#'
#' set.seed(1)
#' Yp<-cbind(runif(ny,0,1),runif(ny,0,1))
#'
#' NumDelTri(Yp)
#' NumDelTri(Yp[1:3,])
#'
#' dat.fr<-data.frame(a=Yp)
#' NumDelTri(dat.fr)
#'
#' @export NumDelTri
NumDelTri <- function(Yp)
{
  if (!is.numeric(as.matrix(Yp)))
  {stop('the argument must be numeric')}

  Yp<-as.matrix(Yp)
  if (ncol(Yp)!=2 || nrow(Yp)<3)
  {stop('the argument must be of dimension kx2 with k>=3')}

  if (nrow(Yp)==3)
  {
    vec1<-rep(1,3);
    D0<-det(matrix(cbind(Yp,vec1),ncol=3))
    if (round(D0,14)==0)
    {stop('the Delaunay triangle is degenerate')}
    nt<-1
  } else
  {
    Ytrimesh<-interp::tri.mesh(Yp[,1],Yp[,2],duplicate="remove")  #Delaunay triangulation
    Ytri<-matrix(interp::triangles(Ytrimesh)[,1:3],ncol=3); #the Delaunay triangles
    nt<-nrow(Ytri)  #number of Delaunay triangles
  }
  nt
} #end of the function
#'

#################################################################

#' @title A test of segregation/association based on domination number of Proportional Edge Proximity Catch Digraph
#' (PE-PCD) for 2D data - Binomial Approximation
#'
#' @description
#' An object of class \code{"htest"} (i.e., hypothesis test) function which performs a hypothesis test of complete spatial
#' randomness (CSR) or uniformity of \code{Xp} points in the convex hull of \code{Yp} points against the alternatives
#' of segregation (where \code{Xp} points cluster away from \code{Yp} points i.e., cluster around the centers of the Delaunay
#' triangles) and association (where \code{Xp} points cluster around \code{Yp} points) based on the (asymptotic) binomial
#'d distribution of the domination number of PE-PCD for uniform 2D data
#' in the convex hull of \code{Yp} points.
#'
#' The function yields the test statistic, \eqn{p}-value for the corresponding \code{alternative},
#' the confidence interval, estimate and null value for the parameter of interest
#' (which is \eqn{Pr(}domination number\eqn{=3)}), and method and name of the data set used.
#'
#' Under the null hypothesis of uniformity of \code{Xp} points in the convex hull of \code{Yp} points, probability of success
#' (i.e., \eqn{Pr(}domination number\eqn{=3)}) equals to its expected value under the uniform distribution) and
#' \code{alternative} could be two-sided, or left-sided (i.e., data is accumulated around the \code{Yp} points, or association)
#' or right-sided (i.e., data is accumulated around the centers of the triangles, or segregation).
#'
#' PE proximity region is constructed with the expansion parameter \eqn{r \ge 1} and \eqn{M}-vertex regions where M
#' is a center that yields non-degenerate asymptotic distribution of the domination number.
#'
#' The test statistic is based on the binomial distribution, when domination number is scaled to have value
#' 0 and 1 in the one triangle case (i.e., Domination Number minus 2 for the one triangle case).
#' That is, the test statistic is based on the domination number for \code{Xp} points inside convex hull of \code{Yp} points
#' for the PE-PCD and default convex hull correction, \code{ch.cor}, is \code{FALSE}
#' where \code{M} is the center that yields nondegenerate asymptotic distribution for the domination number.
#' For this approximation to work, \code{Yp} must be at least 10 (i.e., about 15 or more Delaunay triangles)
#' and \code{Xp} must be at least 7 times more than \code{Yp} points.
#'
#' See also (\insertCite{ceyhan:dom-num-NPE-Spat2011;textual}{pcds}).
#'
#' @param Xp A set of 2D points which constitute the vertices of the PE-PCD.
#' @param Yp A set of 2D points which constitute the vertices of the Delaunay triangles.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be in \eqn{(1,1.5]}.
#' @param ch.cor A logical argument for convex hull correction, default \code{ch.cor=FALSE},
#' recommended when both \code{Xp} and \code{Yp} have the same rectangular support.
#' @param nt Number of Delaunay triangles based on \code{Yp} points, default is \code{NULL}.
#' @param alternative Type of the alternative hypothesis in the test, one of \code{"two.sided"}, \code{"less"}, \code{"greater"}.
#' @param conf.level Level of the confidence interval, default is \code{0.95}, for the probability of success
#' (i.e., \eqn{Pr(}domination number\eqn{=3)} for PE-PCD whose vertices are the 2D data set \code{Xp}.
#'
#' @return A \code{list} with the elements
#' \item{statistic}{Test statistic}
#' \item{p.value}{The \eqn{p}-value for the hypothesis test for the corresponding \code{alternative}}
#' \item{conf.int}{Confidence interval for \eqn{Pr(}Domination Number\eqn{=3)} at the given level \code{conf.level} and
#' depends on the type of \code{alternative}.}
#' \item{estimate}{A \code{vector} with two entries: first is is the estimate of the parameter, i.e.,
#' \eqn{Pr(}Domination Number\eqn{=3)} and second is the domination number}
#' \item{null.value}{Hypothesized value for the parameter, i.e., the null value for \eqn{Pr(}Domination Number\eqn{=3)}}
#' \item{alternative}{Type of the alternative hypothesis in the test, one of \code{"two.sided"}, \code{"less"}, \code{"greater"}}
#' \item{method}{Description of the hypothesis test}
#' \item{data.name}{Name of the data set}
#'
#' @seealso \code{\link{TSDomPENor}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' nx<-20; ny<-4  #try also nx<-1000; ny<-10
#' r<-1.4  #try also r<-1.5
#'
#' set.seed(1)
#' Xp<-cbind(runif(nx,0,1),runif(nx,0,1))
#' Yp<-cbind(runif(ny,0,1),runif(ny,0,1))
#'
#' oldpar <- par(no.readonly = TRUE)
#' plotDeltri(Xp,Yp,xlab="",ylab="")
#' par(oldpar)
#'
#' PEdomMTnd(Xp,Yp,r)
#'
#' TSDomPEBin(Xp,Yp,r,alt="t")
#' TSDomPEBin(Xp,Yp,r,alt="l")
#' TSDomPEBin(Xp,Yp,r,alt="g")
#' TSDomPEBin(Xp,Yp,r,ch=TRUE)
#' TSDomPEBin(Xp,Yp,r=1.25)
#'
#' #or try
#' ndt<-NumDelTri(Yp)
#' TSDomPEBin(Xp,Yp,r,nt=ndt)
#' #values might differ due to the random of choice of the three centers M1,M2,M3
#' #for the non-degenerate asymptotic distribution of the domination number
#'
#' TSDomPEBin(Xp,Yp,r)
#' TSDomPEBin(Xp,Yp[1:3,],r)
#'
#' TSDomPEBin(Xp,rbind(Yp,Yp),r)
#'
#' dat.fr<-data.frame(a=Xp)
#' TSDomPEBin(dat.fr,Yp,r)
#'
#' dat.fr<-data.frame(a=Yp)
#' TSDomPEBin(Xp,dat.fr,r)
#'
#' @export TSDomPEBin
TSDomPEBin <- function(Xp,Yp,r,ch.cor=F,nt=NULL,alternative=c("two.sided", "less", "greater"),conf.level = 0.95)
{
  alternative <-match.arg(alternative)
  if (length(alternative) > 1 || is.na(alternative))
    stop("alternative must be one \"greater\", \"less\", \"two.sided\"")

  dname <-deparse(substitute(Xp))

  if (!is.numeric(as.matrix(Xp)) || !is.numeric(as.matrix(Yp)))
  {stop('Xp and Yp must be numeric')}

  if (is.point(Xp))
  { Xp<-matrix(Xp,ncol=2)
  } else
  {Xp<-as.matrix(Xp)
  if (ncol(Xp)!=2 )
  {stop('Xp must be of dimension nx2')}
  }

  Yp<-as.matrix(Yp)
  if (ncol(Yp)!=2 || nrow(Yp)<3)
  {stop('Yp must be of dimension kx2 with k>=3')}

  if (!is.point(r,1) || r<=1 || r>1.5)
  {stop('r must be in (1,1.5] for domination number to be asymptotically non-degenerate')}

  if (!missing(conf.level))
    if (length(conf.level) != 1 || is.na(conf.level) || conf.level < 0 || conf.level > 1)
      stop("conf.level must be a number between 0 and 1")

  if (is.null(nt))
  {nt<-NumDelTri(Yp)} #number of Delaunay triangles

  Gam<-PEdomMTnd(Xp,Yp,r)$d #domination number
  estimate1<-Gam
  Bm<-Gam-2*nt; #the binomial test statistic

  if (Bm<0)
    warning('The adjusted binomial test statistic is negative! So, 0 is taken as its value')

  Bm<-max(0,Bm)  # to avoid negative Bm values

  method <-c("Exact Binomial Test for the Domination Number for Testing uniformity of 2D Data \n without Convex Hull Correction")

  if (ch.cor==T)  #the part for the convex hull correction
  {
    nx<-nrow(Xp)  #number of Xp points
    ny<-nrow(Yp)  #number of Yp points
    Ytrimesh<-interp::tri.mesh(Yp[,1],Yp[,2],duplicate="remove")  #Delaunay triangulation

   inCH<-interp::in.convex.hull(Ytrimesh,Xp[,1],Xp[,2])  #logical indices for Xp points in convex hull of Yp points

    outch<-nx-sum(inCH)
    prop.out<-outch/nx #observed proportion of points outside convex hull
    exp.prop.out<-1.7932/ny+1.2229/sqrt(ny)  #expected proportion of points outside convex hull

    Bm<-Bm*(1-(prop.out-exp.prop.out))
    method <-c("Exact Binomial Test for the Domination Number for Testing uniformity of 2D Data \n with Convex Hull Correction")
  }

  pg2<-PG2PEtri(r)
  p<-1-pg2
  x<-round(Bm)
  pval <-switch(alternative, less = pbinom(x, nt, p),
                greater = pbinom(x - 1, nt, p, lower.tail = FALSE),
                two.sided = {if (p == 0) (x == 0) else if (p == 1) (x == nt)
                  else { relErr <-1 + 1e-07
                  d <-dbinom(x, nt, p)
                  m <-nt * p
                  if (x == m) 1 else if (x < m)
                  {i <-seq.int(from = ceiling(m), to = nt)
                  y <-sum(dbinom(i, nt, p) <= d * relErr)
                  pbinom(x, nt, p) + pbinom(nt - y, nt, p, lower.tail = FALSE)
                  } else {
                    i <-seq.int(from = 0, to = floor(m))
                    y <-sum(dbinom(i, nt, p) <= d * relErr)
                    pbinom(y - 1, nt, p) + pbinom(x - 1, nt, p, lower.tail = FALSE)
                  }
                  }
                })

  p.L <- function(x, alpha) {
    if (x == 0)
      0
    else qbeta(alpha, x, nt - x + 1)
  }
  p.U <- function(x, alpha) {
    if (x == nt)
      1
    else qbeta(1 - alpha, x + 1, nt - x)
  }
  cint <-switch(alternative, less = c(0, p.U(x, 1 - conf.level)),
                greater = c(p.L(x, 1 - conf.level), 1), two.sided = {
                  alpha <-(1 - conf.level)/2
                  c(p.L(x, alpha), p.U(x, alpha))
                })
  attr(cint, "conf.level") <-conf.level

  estimate2 <-x/nt
  names(x) <-"domination number - 2 * number of Delaunay triangles"
  names(nt) <-"number of Delaunay triangles based on Yp"
  names(p) <-"Pr(Domination Number=3)"
  names(estimate1) <-c(" Pr(domination number = 3)")
  names(estimate2) <-c("|| domination number")
  structure(
    list(statistic = x,
         p.value = pval,
         conf.int = cint,
         estimate = c(estimate1,estimate2),
         null.value = p,
         alternative = alternative,
         method = method,
         data.name = dname
    ),
    class = "htest")

} #end of the function
#'

#################################################################

#' @title A test of segregation/association based on domination number of Proportional Edge Proximity Catch Digraph
#' (PE-PCD) for 2D data - Normal Approximation
#'
#' @description
#' An object of class \code{"htest"} (i.e., hypothesis test) function which performs a hypothesis test of complete spatial
#' randomness (CSR) or uniformity of \code{Xp} points in the convex hull of \code{Yp} points against the alternatives
#' of segregation (where \code{Xp} points cluster away from \code{Yp} points i.e., cluster around the centers of the Delaunay
#' triangles) and association (where \code{Xp} points cluster around \code{Yp} points) based on the normal approximation
#' to the binomial distribution of the domination number of PE-PCD for uniform 2D data
#' in the convex hull of \code{Yp} points
#'
#' The function yields the test statistic, \eqn{p}-value for the corresponding \code{alternative},
#' the confidence interval, estimate and null value for the parameter of interest (which is Pr(Domination
#' Number=3)), and method and name of the data set used.
#'
#' Under the null hypothesis of uniformity of \code{Xp} points in the convex hull of \code{Yp} points, probability of success
#' (i.e., \eqn{Pr(}domination number\eqn{=3)}) equals to its expected value under the uniform distribution) and
#' \code{alternative} could be two-sided, or left-sided (i.e., data is accumulated around the \code{Yp} points, or association)
#' or right-sided (i.e., data is accumulated around the centers of the triangles, or segregation).
#'
#' PE proximity region is constructed with the expansion parameter \eqn{r \ge 1} and \eqn{M}-vertex regions where M
#' is a center that yields non-degenerate asymptotic distribution of the domination number.
#'
#' The test statistic is based on the normal approximation to the binomial distribution, when domination number
#' is scaled to have value  0 and 1 in the one triangle case (i.e., Domination Number minus 1 for the one triangle
#' case). That is, the test statistic is based on the domination number for \code{Xp} points inside convex hull of \code{Yp}
#' points for the PE-PCD and default convex hull correction, \code{ch.cor}, is \code{FALSE}
#' where \code{M} is the center that yields nondegenerate asymptotic distribution for the domination number.
#'
#' For this approximation to work, \code{Yp} must be at least 10 (actually about 15 or more Delaunay triangles)
#' and \code{Xp} must be at least 10 times more than \code{Yp} points.
#'
#' See also (\insertCite{ceyhan:dom-num-NPE-Spat2011;textual}{pcds}).
#'
#' @param Xp A set of 2D points which constitute the vertices of the PE-PCD.
#' @param Yp A set of 2D points which constitute the vertices of the Delaunay triangles.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be in \eqn{(1,1.5]}.
#' @param ch.cor A logical argument for convex hull correction, default \code{ch.cor=FALSE},
#' recommended when both \code{Xp} and \code{Yp} have the same rectangular support.
#' @param nt Number of Delaunay triangles based on \code{Yp} points, default is \code{NULL}.
#' @param alternative Type of the alternative hypothesis in the test, one of \code{"two.sided"}, \code{"less"}, \code{"greater"}.
#' @param conf.level Level of the confidence interval, default is \code{0.95}, for the domination number of
#' PE-PCD whose vertices are the 2D data set \code{Xp}.
#'
#' @return A \code{list} with the elements
#' \item{statistic}{Test statistic}
#' \item{p.value}{The \eqn{p}-value for the hypothesis test for the corresponding \code{alternative}}
#' \item{conf.int}{Confidence interval for the domination number at the given level \code{conf.level} and
#' depends on the type of \code{alternative}.}
#' \item{estimate}{A \code{vector} with two entries: first is the domination number,
#' and second is the estimate of the parameter, i.e., \eqn{Pr(}Domination Number\eqn{=3)}}
#' \item{null.value}{Hypothesized value for the parameter, i.e., the null value for expected domination number}
#' \item{alternative}{Type of the alternative hypothesis in the test, one of \code{"two.sided"}, \code{"less"}, \code{"greater"}}
#' \item{method}{Description of the hypothesis test}
#' \item{data.name}{Name of the data set}
#'
#' @seealso \code{\link{TSDomPEBin}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' nx<-20; ny<-4  #try also nx<-1000; ny<-10
#' r<-1.5  #try also r<-2 or r<-1.25
#'
#' set.seed(1)
#' Xp<-cbind(runif(nx,0,1),runif(nx,0,1))
#' Yp<-cbind(runif(ny,0,1),runif(ny,0,1))
#'
#' oldpar <- par(no.readonly = TRUE)
#' plotDeltri(Xp,Yp,xlab="",ylab="")
#' par(oldpar)
#'
#' PEdomMTnd(Xp,Yp,r)
#'
#' TSDomPENor(Xp,Yp,r)
#' TSDomPENor(Xp,Yp,r)
#'
#' TSDomPENor(Xp,Yp,1.25,ch=TRUE)
#'
#' #or try
#' ndt<-NumDelTri(Yp)
#' TSDomPENor(Xp,Yp,r,nt=ndt)
#' #values might differ due to the random of choice of the three centers M1,M2,M3
#' #for the non-degenerate asymptotic distribution of the domination number
#'
#' TSDomPENor(Xp,Yp,r)
#' TSDomPENor(Xp,Yp[1:3,],r)
#'
#' TSDomPENor(Xp,rbind(Yp,Yp),r)
#'
#' dat.fr<-data.frame(a=Xp)
#' TSDomPENor(dat.fr,Yp,r)
#'
#' dat.fr<-data.frame(a=Yp)
#' TSDomPENor(Xp,dat.fr,r)
#'
#' @export TSDomPENor
TSDomPENor <- function(Xp,Yp,r,ch.cor=F,nt=NULL,alternative=c("two.sided", "less", "greater"),conf.level = 0.95)
{
  alternative <-match.arg(alternative)
  if (length(alternative) > 1 || is.na(alternative))
    stop("alternative must be one \"greater\", \"less\", \"two.sided\"")

  dname <-deparse(substitute(Xp))

  if (!is.numeric(as.matrix(Xp)) || !is.numeric(as.matrix(Yp)))
  {stop('Xp and Yp must be numeric')}

  if (is.point(Xp))
  { Xp<-matrix(Xp,ncol=2)
  } else
  {Xp<-as.matrix(Xp)
  if (ncol(Xp)!=2 )
  {stop('Xp must be of dimension nx2')}
  }

  Yp<-as.matrix(Yp)
  if (ncol(Yp)!=2 || nrow(Yp)<3)
  {stop('Yp must be of dimension kx2 with k>=3')}

  if (!is.point(r,1) || r<=1 || r>1.5)
  {stop('r must be in (1,1.5] for domination number to be asymptotically non-degenerate')}

  if (!missing(conf.level))
    if (length(conf.level) != 1 || is.na(conf.level) || conf.level < 0 || conf.level > 1)
      stop("conf.level must be a number between 0 and 1")

  if (is.null(nt))
  {nt<-NumDelTri(Yp)} #number of Delaunay triangles

  Gam<-PEdomMTnd(Xp,Yp,r)$d #domination number
  estimate1<-Gam
  p<-PG2PEtri(r)
  Exp.Gam <-nt*(3-p)  #expected domination number
  estimate2<-1-p #Pr(gamma=3)
  TS0<-(Gam-Exp.Gam)/sqrt(nt*p*(1-p))  #the standardized test statistic

  if (ch.cor==F)  #the part for the convex hull correction
  {
    TS<-TS0
    method <-c("Normal Approximation to the Domination Number for Testing uniformity of 2D Data\n without Convex Hull Correction")
  } else
  {
    n<-nrow(Xp)  #number of Xp points
    m<-nrow(Yp)  #number of Yp points
    Ytrimesh<-interp::tri.mesh(Yp[,1],Yp[,2],duplicate="remove")  #Delaunay triangulation

   inCH<-interp::in.convex.hull(Ytrimesh,Xp[,1],Xp[,2])  #logical indices for Xp points in convex hull of Yp points

    outch<-n-sum(inCH)
    prop.out<-outch/n #observed proportion of points outside convex hull
    exp.prop.out<-1.7932/m+1.2229/sqrt(m)  #expected proportion of points outside convex hull

    TS<-TS0*(1-(prop.out-exp.prop.out))
    method <-c("Normal Approximation to the Domination Number for Testing uniformity of 2D Data\n with Convex Hull Correction")
  }

  st.err<-sqrt(nt*p*(1-p))

  names(estimate1) <-c(" domination number  ")
  names(estimate2) <-c("|| Pr(domination number = 3)")

  null.gam<-Exp.Gam

  names(null.gam) <-c("expected domination number")

  names(TS) <-"standardized domination number (i.e., Z)"

  if (alternative == "less") {
    pval <-pnorm(TS)
    cint <-Gam+c(-Inf, qnorm(conf.level))*st.err
  }
  else if (alternative == "greater") {
    pval <-pnorm(TS, lower.tail = FALSE)
    cint <-Gam+c(-qnorm(conf.level),Inf)*st.err
  }
  else {
    pval <-2 * pnorm(-abs(TS))
    alpha <-1 - conf.level
    cint <-qnorm(1 - alpha/2)
    cint <-Gam+c(-cint, cint)*st.err
  }

  attr(cint, "conf.level") <-conf.level

  rval <-list(
    statistic=TS,
    p.value=pval,
    conf.int = cint,
    estimate = c(estimate1,estimate2),
    null.value = null.gam,
    alternative = alternative,
    method = method,
    data.name = dname
  )

  attr(rval, "class") <-"htest"

  return(rval)
} #end of the function
#'

#################################################################
#Functions for NPE in R^3
#################################################################

#' @title The line crossing 3D point \code{r0} in the direction of \code{vector} \code{v} (or if \code{v} is a point,
#' in direction of \eqn{v-r_0})
#'
#' @description
#' An object of class \code{"Lines3D"}.
#' Returns the equation, \eqn{x}-, \eqn{y}-, and \eqn{z}-coordinates of the line crossing 3D point \eqn{r_0}
#' in the direction of \code{vector} \code{v} (of if \code{v} is a point, in the direction of \eqn{v-r_0})
#' with the parameter \code{t} being provided in \code{vector} \code{t}.
#'
#' @param r0 A 3D point through which the straight line passes.
#' @param v A 3D \code{vector} which determines the direction of the straight line (i.e., the straight line would be
#' parallel to this vector) if the \code{dir.vec=TRUE}, otherwise it is 3D point and \eqn{v-r_0} determines the direction of the
#' the straight line.
#' @param t A scalar or a \code{vector} of scalars representing the parameter of the coordinates of the line
#' (for the form: \eqn{x=x_0 + a t}, \eqn{y=y_0 + b t}, and \eqn{z=z_0 + c t} where \eqn{r_0=(x_0,y_0,z_0)}
#' and \eqn{v=(a,b,c)} if \code{dir.vec=TRUE}, else \eqn{v-r_0=(a,b,c)}).
#' @param dir.vec A logical argument about \code{v}, if \code{TRUE} \code{v} is treated as a vector,
#' else \code{v} is treated as a point and so the direction \code{vector} is taken to be \eqn{v-r_0}.
#'
#' @return A \code{list} with the elements
#' \item{desc}{A description of the line}
#' \item{mtitle}{The \code{"main"} title for the plot of the line}
#' \item{pts}{The input points that determine a line and/or a plane, \code{NULL} for this function.}
#' \item{pnames}{The names of the input points that determine a line and/or a plane, \code{NULL} for this function.}
#' \item{vecs}{The point \code{r0} and the \code{vector} \code{v} (if \code{dir.vec=TRUE}) or the point \code{v} (if \code{dir.vec=FALSE}). The first row is \code{r0}
#' and the second row is \code{v}.}
#' \item{vec.names}{The names of the point \code{r0} and the \code{vector} \code{v} (if \code{dir.vec=TRUE}) or the point \code{v} (if \code{dir.vec=FALSE}).}
#' \item{x,y,z}{The \eqn{x}-, \eqn{y}-, and \eqn{z}-coordinates of the point(s) of interest on the line.}
#' \item{tsq}{The scalar or the \code{vector} of the parameter in defining each coordinate of the line for the form:
#' \eqn{x=x_0 + a t}, \eqn{y=y_0 + b t}, and \eqn{z=z_0 + c t} where \eqn{r_0=(x_0,y_0,z_0)} and \eqn{v=(a,b,c)} if \code{dir.vec=TRUE}, else \eqn{v-r_0=(a,b,c)}.}
#' \item{equation}{Equation of the line passing through point \code{r0} in the direction of the \code{vector} \code{v} (if \code{dir.vec=TRUE})
#' else in the direction of \eqn{v-r_0}. The line equation is in the form: \eqn{x=x_0 + a t}, \eqn{y=y_0 + b t}, and \eqn{z=z_0 + c t} where
#' \eqn{r_0=(x_0,y_0,z_0)} and \eqn{v=(a,b,c)} if \code{dir.vec=TRUE}, else \eqn{v-r_0=(a,b,c)}.}
#'
#' @seealso \code{\link{line}}, \code{\link{paraline3D}}, and \code{\link{Plane}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' \donttest{
#' A<-c(1,10,3); B<-c(1,1,3);
#'
#' vecs<-rbind(A,B)
#'
#' Line3D(A,B,.1)
#' Line3D(A,B,.1,dir.vec=FALSE)
#'
#' tr<-range(vecs);
#' tf<-(tr[2]-tr[1])*.1 #how far to go at the lower and upper ends in the x-coordinate
#' tsq<-seq(-tf*10-tf,tf*10+tf,l=20)  #try also l=100
#'
#' lnAB3D<-Line3D(A,B,tsq)
#' lnAB3D
#' summary(lnAB3D)
#' plot(lnAB3D)
#'
#' Line3D(A,B,c(.1,.2))
#'
#' x<-lnAB3D$x
#' y<-lnAB3D$y
#' z<-lnAB3D$z
#'
#' zr<-range(z)
#' zf<-(zr[2]-zr[1])*.2
#' Bv<-B*tf*5
#'
#' Xlim<-range(x)
#' Ylim<-range(y)
#' Zlim<-range(z)
#'
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#' zd<-Zlim[2]-Zlim[1]
#'
#' Dr<-A+min(tsq)*B
#'
#' plot3D::lines3D(x, y, z, phi = 0, bty = "g",
#' xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05),zlim=Zlim+zd*c(-.1,.1),
#'         pch = 20, cex = 2, ticktype = "detailed")
#' plot3D::arrows3D(Dr[1],Dr[2],Dr[3]+zf,Dr[1]+Bv[1],Dr[2]+Bv[2],Dr[3]+zf+Bv[3], add=TRUE)
#' plot3D::points3D(A[1],A[2],A[3],add=TRUE)
#' plot3D::arrows3D(A[1],A[2],A[3]-2*zf,A[1],A[2],A[3],lty=2, add=TRUE)
#' plot3D::text3D(A[1],A[2],A[3]-2*zf,labels="initial point",add=TRUE)
#' plot3D::text3D(A[1],A[2],A[3]+zf/2,labels=expression(r[0]),add=TRUE)
#' plot3D::arrows3D(Dr[1]+Bv[1]/2,Dr[2]+Bv[2]/2,Dr[3]+3*zf+Bv[3]/2,
#' Dr[1]+Bv[1]/2,Dr[2]+Bv[2]/2,Dr[3]+zf+Bv[3]/2,lty=2, add=TRUE)
#' plot3D::text3D(Dr[1]+Bv[1]/2,Dr[2]+Bv[2]/2,Dr[3]+3*zf+Bv[3]/2,
#' labels="direction vector",add=TRUE)
#' plot3D::text3D(Dr[1]+Bv[1]/2,Dr[2]+Bv[2]/2,Dr[3]+zf+Bv[3]/2,labels="v",add=TRUE)
#' plot3D::text3D(0,0,0,labels="O",add=TRUE)
#' }
#'
#' @export Line3D
Line3D <- function(r0,v,t,dir.vec=TRUE)
{
  if (!is.point(r0,3) || !is.point(v,3))
  {stop('r0 and v must be numeric vectors of dimension 3')}

  if (!is.point(t,length(t)))
  {stop('t must be a numeric vector')}

  if (isTRUE(all.equal(v,c(0,0,0))))
  {stop('The line is not well-defined, as the direction vector is the zero vector!')}

  x0<-r0[1]; y0<-r0[2]; z0<-r0[3];

  if (dir.vec!=T) {v<-v-r0}

  a<-v[1]; b<-v[2]; c<-v[3];

  x=x0+a*t
  y=y0+b*t
  z=z0+c*t

  vecs<-rbind(r0,v)
  row.names(vecs) <- c("initial point","direction vector")
  r0name <-deparse(substitute(r0))
  vname <-deparse(substitute(v))

  line.desc<-paste("The Line Passing through Point", r0name, "in the direction of", vname, "(i.e., parallel to", vname, ")")
  main.txt<-paste("Line Passing thru Point", r0name, "in the direction of", vname)

  eqn<- rbind(paste("x=",x0,ifelse(sign(a)<0,"","+"),a,"t",sep=""),
              paste("y=",y0,ifelse(sign(b)<0,"","+"),b,"t",sep=""),
              paste("z=",z0,ifelse(sign(c)<0,"","+"),c,"t",sep=""))

  res<-list(
    desc=line.desc,
    mtitle=main.txt,
    pts=NULL,
    pnames=NULL,
    vecs=vecs,
    vec.names=c(r0name,vname),
    x=x,
    y=y,
    z=z,
    tsq=t,
    equation=eqn
  )
  class(res)<-"Lines3D"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

#' @title The line crossing the 3D point \code{P} and parallel to line joining 3D points \code{A} and \code{B}
#'
#' @description
#' An object of class \code{"Lines3D"}.
#' Returns the equation, \eqn{x}-, \eqn{y}-, and \eqn{z}-coordinates of the line crossing 3D point \code{P} and parallel to the line
#' joining 3D points \code{A} and \code{B} (i.e., the line is in the direction of \code{vector} \code{B}-\code{A})
#' with the parameter \code{t} being provided in \code{vector} \code{t}.
#'
#' @param P A 3D point through which the straight line passes.
#' @param A,B 3D points which determine the straight line to which the line passing through point \code{P} would be
#' parallel (i.e., \eqn{B-A} determines the direction of the straight line passing through \code{P}).
#' @param t A scalar or a \code{vector} of scalars representing the parameter of the coordinates of the line
#' (for the form: \eqn{x=x_0 + a t}, \eqn{y=y_0 + b t}, and \eqn{z=z_0 + c t} where \eqn{P=(x_0,y_0,z_0)}
#' and \eqn{B-A=(a,b,c)}).
#'
#' @return A \code{list} with the elements
#' \item{desc}{A description of the line}
#' \item{mtitle}{The \code{"main"} title for the plot of the line}
#' \item{points}{The input points that determine the line to which the line crossing point \code{P} would be parallel.}
#' \item{pnames}{The names of the input points that determine the line to which the line crossing point \code{P} would
#' be parallel.}
#' \item{vecs}{The points \code{P}, \code{A}, and \code{B} stacked row-wise in this order.}
#' \item{vec.names}{The names of the points \code{P}, \code{A}, and \code{B}.}
#' \item{x,y,z}{The \eqn{x}-, \eqn{y}-, and \eqn{z}-coordinates of the point(s) of interest on the line parallel to the line
#' determined by points \code{A} and \code{B}.}
#' \item{tsq}{The scalar or the \code{vector} of the parameter in defining each coordinate of the line for the form:
#' \eqn{x=x_0 + a t}, \eqn{y=y_0 + b t}, and \eqn{z=z_0 + c t} where \eqn{P=(x_0,y_0,z_0)} and \eqn{B-A=(a,b,c)}.}
#' \item{equation}{Equation of the line passing through point \code{P} and parallel to the line joining points \code{A} and \code{B}
#' (i.e., in the direction of the \code{vector} \code{B}-\code{A}). The line equation is in the form: \eqn{x=x_0 + a t}, \eqn{y=y_0 + b t}, and
#' \eqn{z=z_0 + c t} where \eqn{P=(x_0,y_0,z_0)} and \eqn{B-A=(a,b,c)}.}
#'
#' @seealso \code{\link{Line3D}}, \code{\link{perp.ln2pl}}, and \code{\link{paraline}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' \donttest{
#' P<-c(1,10,4); A<-c(1,1,3); B<-c(3,9,12)
#'
#' vecs<-rbind(P,B-A)
#' pts<-rbind(P,A,B)
#' paraline3D(P,A,B,.1)
#'
#' tr<-range(pts,vecs);
#' tf<-(tr[2]-tr[1])*.1 #how far to go at the lower and upper ends in the x-coordinate
#' tsq<-seq(-tf*10-tf,tf*10+tf,l=20)  #try also l=100
#'
#' pln3D<-paraline3D(P,A,B,tsq)
#' pln3D
#' summary(pln3D)
#' plot(pln3D)
#'
#' paraline3D(P,A,B,c(.1,.2))
#'
#' x<-pln3D$x
#' y<-pln3D$y
#' z<-pln3D$z
#'
#' zr<-range(z)
#' zf<-(zr[2]-zr[1])*.2
#' Av<-(B-A)*tf*5
#'
#' Xlim<-range(x,pts[,1])
#' Ylim<-range(y,pts[,2])
#' Zlim<-range(z,pts[,3])
#'
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#' zd<-Zlim[2]-Zlim[1]
#'
#' Dr<-P+min(tsq)*(B-A)
#'
#' plot3D::lines3D(x, y, z, phi = 0, bty = "g",
#' xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05),zlim=Zlim+zd*c(-.1,.1)+c(-zf,zf),
#'         pch = 20, cex = 2, ticktype = "detailed")
#' plot3D::arrows3D(Dr[1],Dr[2],Dr[3]+zf,Dr[1]+Av[1],Dr[2]+Av[2],Dr[3]+zf+Av[3], add=TRUE)
#' plot3D::points3D(pts[,1],pts[,2],pts[,3],add=TRUE)
#' plot3D::text3D(pts[,1],pts[,2],pts[,3],labels=c("P","A","B"),add=TRUE)
#' plot3D::arrows3D(P[1],P[2],P[3]-2*zf,P[1],P[2],P[3],lty=2, add=TRUE)
#' plot3D::text3D(P[1],P[2],P[3]-2*zf,labels="initial point",add=TRUE)
#' plot3D::arrows3D(Dr[1]+Av[1]/2,Dr[2]+Av[2]/2,Dr[3]+3*zf+Av[3]/2,Dr[1]+Av[1]/2,
#' Dr[2]+Av[2]/2,Dr[3]+zf+Av[3]/2,lty=2, add=TRUE)
#' plot3D::text3D(Dr[1]+Av[1]/2,Dr[2]+Av[2]/2,Dr[3]+3*zf+Av[3]/2,labels="direction vector",add=TRUE)
#' plot3D::text3D(Dr[1]+Av[1]/2,Dr[2]+Av[2]/2,Dr[3]+zf+Av[3]/2,labels="B-A",add=TRUE)
#' }
#'
#' @export paraline3D
paraline3D <- function(P,A,B,t)
{
  if (!is.point(P,3) || !is.point(A,3) || !is.point(B,3))
  {stop('P, A and B must be numeric vectors of dimension 3')}

  if (!is.point(t,length(t)))
  {stop('t must be a numeric vector')}

  if (isTRUE(all.equal(A,B)))
  {stop('The lines are not well defined, the two points to define the line are concurrent!')}

  v<-B-A
  x0<-P[1]; y0<-P[2]; z0<-P[3];
  a<-v[1]; b<-v[2]; c<-v[3];

  x=x0+a*t
  y=y0+b*t
  z=z0+c*t

  vecs<-rbind(P,v)
  row.names(vecs) <- c("initial point","direction vector")
  Pname <-deparse(substitute(P))
  Aname <-deparse(substitute(A))
  Bname <-deparse(substitute(B))

  line.desc<-paste("The Line Passing through Point", Pname, "parallel to the line joining points", Aname, "and", Bname)
  main.txt<-paste("Line Passing thru Point", Pname, "in the direction of", Bname, "-", Aname)

  eqn<- rbind(paste("x=",x0,ifelse(sign(a)<0,"","+"),a,"t",sep=""),
              paste("y=",y0,ifelse(sign(b)<0,"","+"),b,"t",sep=""),
              paste("z=",z0,ifelse(sign(c)<0,"","+"),c,"t",sep=""))

  res<-list(
    desc=line.desc,
    mtitle=main.txt,
    pts=rbind(P,A,B),
    pnames=c(Pname,Aname,Bname),
    vecs=vecs,
    vec.names=c(Pname,paste(Bname,"-",Aname)),
    x=x,
    y=y,
    z=z,
    tsq=t,
    equation=eqn
  )
  class(res)<-"Lines3D"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

#' @title The line crossing the 3D point \code{P} and perpendicular to the plane spanned by 3D points \code{A},
#' \code{B}, and \code{C}
#'
#' @description
#' An object of class \code{"Lines3D"}.
#' Returns the equation, \eqn{x}-, \eqn{y}-, and \eqn{z}-coordinates of the line crossing 3D point \code{P} and perpendicular to the plane
#' spanned by 3D points \code{A}, \code{B}, and \code{C} (i.e., the line is in the direction of normal \code{vector} of this plane)
#' with the parameter \code{t} being provided in \code{vector} \code{t}.
#'
#' @param P A 3D point through which the straight line passes.
#' @param A,B,C 3D points which determine the plane to which the line passing through point \code{P} would be
#' perpendicular (i.e., the normal \code{vector} of this plane determines the direction of the straight line
#' passing through \code{P}).
#' @param t A scalar or a \code{vector} of scalars representing the parameter of the coordinates of the line
#' (for the form: \eqn{x=x_0 + a t}, \eqn{y=y_0 + b t}, and \eqn{z=z_0 + c t} where \eqn{P=(x_0,y_0,z_0)}
#' and normal vector\eqn{=(a,b,c)}).
#'
#' @return A \code{list} with the elements
#' \item{desc}{A description of the line}
#' \item{mtitle}{The \code{"main"} title for the plot of the line}
#' \item{points}{The input points that determine the line and plane, line crosses point \code{P} and plane is determined
#' by 3D points \code{A}, \code{B}, and \code{C}.}
#' \item{pnames}{The names of the input points that determine the line and plane; line would be perpendicular
#' to the plane.}
#' \item{vecs}{The point \code{P} and normal vector.}
#' \item{vec.names}{The names of the point \code{P} and the second entry is "normal vector".}
#' \item{x,y,z}{The \eqn{x}-, \eqn{y}-, and \eqn{z}-coordinates of the point(s) of interest on the line perpendicular to the plane
#' determined by points \code{A}, \code{B}, and \code{C}.}
#' \item{tsq}{The scalar or the \code{vector} of the parameter in defining each coordinate of the line for the form:
#' \eqn{x=x_0 + a t}, \eqn{y=y_0 + b t}, and \eqn{z=z_0 + c t} where \eqn{P=(x_0,y_0,z_0)} and normal vector\eqn{=(a,b,c)}.}
#' \item{equation}{Equation of the line passing through point \code{P} and perpendicular to the plane determined by
#' points \code{A}, \code{B}, and \code{C} (i.e., line is in the direction of the normal \code{vector} N of the plane). The line equation
#' is in the form: \eqn{x=x_0 + a t}, \eqn{y=y_0 + b t}, and \eqn{z=z_0 + c t} where \eqn{P=(x_0,y_0,z_0)} and normal vector\eqn{=(a,b,c)}.}
#'
#' @seealso \code{\link{Line3D}}, \code{\link{paraline3D}} and \code{\link{perpline}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' \donttest{
#' P<-c(1,1,1); A<-c(1,10,4); B<-c(1,1,3); C<-c(3,9,12)
#'
#' cf<-as.numeric(Plane(A,B,C,1,1)$coeff)
#' a<-cf[1]; b<-cf[2]; c<- -1;
#'
#' vecs<-rbind(A,c(a,b,c))
#' pts<-rbind(P,A,B,C)
#' perp.ln2pl(P,A,B,C,.1)
#'
#' tr<-range(pts,vecs);
#' tf<-(tr[2]-tr[1])*.1 #how far to go at the lower and upper ends in the x-coordinate
#' tsq<-seq(-tf*10-tf,tf*10+tf,l=20)  #try also l=100
#'
#' pln2pl<-perp.ln2pl(P,A,B,C,tsq)
#' pln2pl
#' summary(pln2pl)
#' plot(pln2pl)
#'
#' perp.ln2pl(P,A,B,C,c(.1,.2))
#'
#' xc<-pln2pl$x
#' yc<-pln2pl$y
#' zc<-pln2pl$z
#'
#' zr<-range(zc)
#' zf<-(zr[2]-zr[1])*.2
#' Bv<- -c(a,b,c)*zf*5
#'
#' Dr<-(A+B+C)/3
#'
#' pts2<-rbind(A,B,C)
#' xr<-range(pts2[,1],xc); yr<-range(pts2[,2],yc)
#' xf<-(xr[2]-xr[1])*.1 #how far to go at the lower and upper ends in the x-coordinate
#' yf<-(yr[2]-yr[1])*.1 #how far to go at the lower and upper ends in the y-coordinate
#' xs<-seq(xr[1]-xf,xr[2]+xf,l=20)  #try also l=100
#' ys<-seq(yr[1]-yf,yr[2]+yf,l=20)  #try also l=100
#'
#' plABC<-Plane(A,B,C,xs,ys)
#' z.grid<-plABC$z
#'
#' Xlim<-range(xc,xs,pts[,1])
#' Ylim<-range(yc,ys,pts[,2])
#' Zlim<-range(zc,z.grid,pts[,3])
#'
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#' zd<-Zlim[2]-Zlim[1]
#'
#' plot3D::persp3D(z = z.grid, x = xs, y = ys, theta =225, phi = 30,
#' col="lightblue", ticktype = "detailed",
#'         xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05),zlim=Zlim+zd*c(-.05,.05))
#'         #plane spanned by points A, B, C
#' plot3D::lines3D(xc, yc, zc, bty = "g",pch = 20, cex = 2,col="red", ticktype = "detailed",add=TRUE)
#' plot3D::arrows3D(Dr[1],Dr[2],Dr[3],Dr[1]+Bv[1],Dr[2]+Bv[2],Dr[3]+Bv[3], add=TRUE)
#' plot3D::points3D(pts[,1],pts[,2],pts[,3],add=TRUE)
#' plot3D::text3D(pts[,1],pts[,2],pts[,3],labels=c("P","A","B","C"),add=TRUE)
#' plot3D::arrows3D(P[1],P[2],P[3]-zf,P[1],P[2],P[3],lty=2, add=TRUE)
#' plot3D::text3D(P[1],P[2],P[3]-zf,labels="initial point",add=TRUE)
#' plot3D::text3D(P[1],P[2],P[3]+zf/2,labels="P",add=TRUE)
#' plot3D::arrows3D(Dr[1],Dr[2],Dr[3],Dr[1]+Bv[1]/2,Dr[2]+Bv[2]/2,Dr[3]+Bv[3]/2,lty=2, add=TRUE)
#' plot3D::text3D(Dr[1]+Bv[1]/2,Dr[2]+Bv[2]/2,Dr[3]+Bv[3]/2,labels="normal vector",add=TRUE)
#' }
#' @export perp.ln2pl
perp.ln2pl <- function(P,A,B,C,t)
{
  if (!is.point(P,3) || !is.point(A,3) || !is.point(B,3) || !is.point(C,3))
  {stop('P, A, B, and C must be numeric vectors of dimension 3')}

  if (!is.point(t,length(t)))
  {stop('t must be a numeric vector')}

  dab<-Dist(A,B); dac<-Dist(A,C); dbc<-Dist(B,C);
  sd<-sort(c(dab,dac,dbc))

  if (isTRUE(all.equal(sd[3] , sd[1]+sd[2])))
  {stop('The three points, A, B, and C are collinear; plane is not well-defined')}

  cf<-as.numeric(Plane(A,B,C,1,1)$coeff)
  a<-cf[1]; b<-cf[2]; c<--1;
  x0<-P[1]; y0<-P[2]; z0<-P[3];

  x=x0+a*t
  y=y0+b*t
  z=z0+c*t

  vecs<-rbind(P,c(a,b,c))
  row.names(vecs) <- c("initial point","normal vector")
  Pname <-deparse(substitute(P))
  Aname <-deparse(substitute(A))
  Bname <-deparse(substitute(B))
  Cname <-deparse(substitute(C))

  line.desc<-paste("The line passing through point", Pname, "perpendicular to the plane spanned by points", Aname, ",", Bname, "and", Cname)
  main.txt<-paste("Line Passing thru Point", Pname, "Perpendicular to the Plane\n Spanned by Points", Aname, ",", Bname, "and", Cname)

  eqn<- rbind(paste("x=",x0,ifelse(sign(a)<0,"","+"),a,"t",sep=""),
              paste("y=",y0,ifelse(sign(b)<0,"","+"),b,"t",sep=""),
              paste("z=",z0,ifelse(sign(c)<0,"","+"),c,"t",sep=""))

  res<-list(
    desc=line.desc,
    mtitle=main.txt,
    pts=rbind(P,A,B,C),
    pnames=c(Pname,Aname,Bname,Cname),
    vecs=vecs,
    vec.names=c(Pname,"normal vector"),
    x=x,
    y=y,
    z=z,
    tsq=t,
    equation=eqn
  )
  class(res)<-"Lines3D"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

#' @title The point of intersection of a line and a plane
#'
#' @description Returns the point of the intersection of the line determined by the 3D points \eqn{x_1} and \eqn{x_2} and the plane spanned
#' by 3D points \code{x3}, \code{x4}, and \code{x5}.
#'
#' @param x1,x2 3D points that determine the straight line (i.e., through which the straight line passes).
#' @param x3,x4,x5 3D points that determine the plane (i.e., through which the plane passes).
#'
#' @return The coordinates of the point of intersection the line determined by the 3D points \eqn{x_1} and \eqn{x_2} and the
#' plane determined by 3D points \code{x3}, \code{x4}, and \code{x5}.
#'
#' @seealso \code{\link{int.2lines}} and \code{\link{int.circ.line}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' \donttest{
#' L1<-c(2,4,6); L2<-c(1,3,5); A<-c(1,10,3); B<-c(1,1,3); C<-c(3,9,12)
#'
#' Pint<-int.line.plane(L1,L2,A,B,C)
#' Pint
#' pts<-rbind(L1,L2,A,B,C,Pint)
#'
#' tr<-max(Dist(L1,L2),Dist(L1,Pint),Dist(L2,Pint))
#' tf<-tr*1.1 #how far to go at the lower and upper ends in the x-coordinate
#' tsq<-seq(-tf,tf,l=20)  #try also l=100
#'
#' lnAB3D<-Line3D(L1,L2,tsq)
#' xl<-lnAB3D$x
#' yl<-lnAB3D$y
#' zl<-lnAB3D$z
#'
#' xr<-range(pts[,1]); yr<-range(pts[,2])
#' xf<-(xr[2]-xr[1])*.1 #how far to go at the lower and upper ends in the x-coordinate
#' yf<-(yr[2]-yr[1])*.1 #how far to go at the lower and upper ends in the y-coordinate
#' xp<-seq(xr[1]-xf,xr[2]+xf,l=20)  #try also l=100
#' yp<-seq(yr[1]-yf,yr[2]+yf,l=20)  #try also l=100
#'
#' plABC<-Plane(A,B,C,xp,yp)
#' z.grid<-plABC$z
#'
#' res<-persp(xp,yp,z.grid, xlab="x",ylab="y",zlab="z",theta = -30, phi = 30, expand = 0.5,
#' col = "lightblue", ltheta = 120, shade = 0.05, ticktype = "detailed")
#' lines (trans3d(xl, yl, zl, pmat = res), col = 3)
#'
#' Xlim<-range(xl,pts[,1])
#' Ylim<-range(yl,pts[,2])
#' Zlim<-range(zl,pts[,3])
#'
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#' zd<-Zlim[2]-Zlim[1]
#'
#' plot3D::persp3D(z = z.grid, x = xp, y = yp, theta =225, phi = 30, ticktype = "detailed"
#'         ,xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05),zlim=Zlim+zd*c(-.1,.1))
#'         #plane spanned by points A, B, C
#' #add the defining points
#' plot3D::points3D(pts[,1],pts[,2],pts[,3], pch = ".", col = "black", bty = "f", cex = 5,add=TRUE)
#' plot3D::points3D(Pint[1],Pint[2],Pint[3], pch = "*", col = "red", bty = "f", cex = 5,add=TRUE)
#' plot3D::lines3D(xl, yl, zl, bty = "g", cex = 2, ticktype = "detailed",add=TRUE)
#' }
#'
#' @export int.line.plane
int.line.plane <- function(x1,x2,x3,x4,x5)
{
  if (!is.point(x1,3) || !is.point(x2,3) || !is.point(x3,3) || !is.point(x4,3) || !is.point(x5,3))
  {stop('all arguments must be numeric vectors/points of dimension 3')}

  if (isTRUE(all.equal(x1,x2)))
  {stop('The line is not well defined.')}

  d34<-Dist(x3,x4); d35<-Dist(x3,x5); d45<-Dist(x4,x5);
  sd<-sort(c(d34,d35,d45))

  if (isTRUE(all.equal(sd[3] , sd[1]+sd[2])))
  {stop('the three points, x3, x4 and x5 are collinear; plane is not well-defined.')}

  num<- rbind(rep(1,4),cbind(x3,x4,x5,x1))
  denom<-rbind(c(1,1,1,0),cbind(x3,x4,x5,x2-x1))
  t<- - det(num)/det(denom)

  if (abs(t)==Inf)
  {stop('The line and plane are parallel, hence they do not intersect.')}

  v<-x2-x1
  x0<-x1[1]; y0<-x1[2]; z0<-x1[3];
  a<-v[1]; b<-v[2]; c<-v[3];

  x=x0+a*t
  y=y0+b*t
  z=z0+c*t

  c(x,y,z)
} #end of the function
#'

#################################################################

#' @title The plane passing through three distinct 3D points \code{a}, \code{b}, and \code{c}
#'
#' @description
#' An object of class \code{"Planes"}.
#' Returns the equation and \eqn{z}-coordinates of the plane passing through three distinct 3D points \code{a}, \code{b}, and \code{c}
#' with \eqn{x}- and \eqn{y}-coordinates are provided in vectors \code{x} and \code{y}, respectively.
#'
#' @param a,b,c 3D points that determine the plane (i.e., through which the plane is passing).
#' @param x,y Scalars or vectors of scalars representing the \eqn{x}- and \eqn{y}-coordinates of the plane.
#'
#' @return A \code{list} with the elements
#' \item{desc}{A description of the plane}
#' \item{points}{The input points \code{a}, \code{b}, and \code{c} through which the plane is passing
#' (stacked row-wise, i.e., row 1 is point \code{a}, row 2 is point \code{b} and row 3 is point \code{c}).}
#' \item{x,y}{The input vectors which constitutes the \eqn{x}- and \eqn{y}-coordinates of the point(s) of interest on the
#' plane. \code{x} and \code{y} can be scalars or vectors of scalars.}
#' \item{z}{The output \code{vector} which constitutes the \eqn{z}-coordinates of the point(s) of interest on the plane.
#' If \code{x} and \code{y} are scalars, \code{z} will be a scalar and
#' if \code{x} and \code{y} are vectors of scalars, then \code{z} needs to be a \code{matrix} of scalars,
#' containing the \eqn{z}-coordinate for each pair of \code{x} and \code{y} values.}
#' \item{coeff}{Coefficients of the plane (in the \eqn{z = A x+B y+C} form).}
#' \item{equation}{Equation of the plane in long form}
#' \item{equation2}{Equation of the plane in short form, to be inserted on the plot}
#'
#' @seealso \code{\link{paraplane}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' \donttest{
#' A<-c(1,10,3); B<-c(1,1,3); C<-c(3,9,12)
#'
#' pts<-rbind(A,B,C)
#' Plane(A,B,C,.1,.2)
#'
#' xr<-range(pts[,1]); yr<-range(pts[,2])
#' xf<-(xr[2]-xr[1])*.1 #how far to go at the lower and upper ends in the x-coordinate
#' yf<-(yr[2]-yr[1])*.1 #how far to go at the lower and upper ends in the y-coordinate
#' x<-seq(xr[1]-xf,xr[2]+xf,l=20)  #try also l=100
#' y<-seq(yr[1]-yf,yr[2]+yf,l=20)  #try also l=100
#'
#' plABC<-Plane(A,B,C,x,y)
#' plABC
#' summary(plABC)
#' plot(plABC)
#'
#' Plane(A,B,A+B,.1,.2)
#'
#' z.grid<-plABC$z
#'
#' persp(x,y,z.grid, xlab="x",ylab="y",zlab="z",
#' theta = -30, phi = 30, expand = 0.5, col = "lightblue",
#'       ltheta = 120, shade = 0.05, ticktype = "detailed")
#'
#' zr<-max(z.grid)-min(z.grid)
#' Pts<-rbind(A,B,C)+rbind(c(0,0,zr*.1),c(0,0,zr*.1),c(0,0,zr*.1))
#' Mn.pts<-apply(Pts,2,mean)
#'
#' plot3D::persp3D(z = z.grid, x = x, y = y, theta =225, phi = 30, ticktype = "detailed")
#' #plane spanned by points A, B, C
#' #add the defining points
#' plot3D::points3D(Pts[,1],Pts[,2],Pts[,3], add=TRUE)
#' plot3D::text3D(Pts[,1],Pts[,2],Pts[,3], c("A","B","C"),add=TRUE)
#' plot3D::text3D(Mn.pts[1],Mn.pts[2],Mn.pts[3],plABC$equation,add=TRUE)
#' plot3D::polygon3D(Pts[,1],Pts[,2],Pts[,3], add=TRUE)
#'
#' Plane(A,B,C,.1,.2)
#' }
#'
#' @export Plane
Plane <- function(a,b,c,x,y)
{
  if (!is.point(a,3) || !is.point(b,3) || !is.point(c,3))
  {stop('a, b,and c must be numeric points of dimension 3')}

  if (!is.point(x,length(x)) || !is.point(y,length(y)))
  {stop('x and y must be numeric vectors')}

  dab<-Dist(a,b); dac<-Dist(a,c); dbc<-Dist(b,c);
  sd<-sort(c(dab,dac,dbc))

  if (isTRUE(all.equal(sd[3] , sd[1]+sd[2])))
  {stop('the three points, a, b, and c, are collinear; plane is not well-defined')}

  mat<-rbind(a,b,c)
  D<-det(mat)
  matx<-cbind(rep(1,3),mat[,2],mat[,3])
  maty<-cbind(mat[,1],rep(1,3),mat[,3])
  matz<-cbind(mat[,1],mat[,2],rep(1,3))
  a0<-(-1/D)*det(matx)
  b0<-(-1/D)*det(maty)
  c0<-(-1/D)*det(matz)

  A<- -a0/c0; B<- -b0/c0; C<- -1/c0;

  lx<-length(x)
  ly<-length(y)

  z.grid<-matrix(0,nrow=lx,ncol=ly)
  for (i in 1:lx)
    for (j in 1:ly)
    {
      z.grid[i,j]<-A*x[i]+B*y[j]+C
    }

  aname <-deparse(substitute(a))
  bname <-deparse(substitute(b))
  cname <-deparse(substitute(c))

  plane.desc<-paste("Plane Passing through Points ", aname,", ", bname,",", " and ", cname,sep="")

  pts<-rbind(a,b,c)
  row.names(pts)<-c(aname,bname,cname)

  coeffs<-c(A,B,C)
  names(coeffs)<-c("A","B","C")

  res<-list(
    desc=plane.desc,
    points=pts,
    x=x,
    y=y,
    z=z.grid,
    coeff=coeffs,
    equation=paste("z=",A,"x",ifelse(sign(B)<0,"","+"),B,"y",ifelse(sign(C)<0,"","+"),C,sep=""),
    equation2=paste("z=",round(A,2),"x",ifelse(sign(B)<0,"","+"),round(B,2),"y",
                    ifelse(sign(C)<0,"","+"),round(C,2),sep="")
  )

  class(res)<-"Planes"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

#' @title The distance from a point to a plane
#'
#' @description Returns the distance from a point \code{p} to the plane passing through points \code{a}, \code{b}, and \code{c} in 3D space.
#'
#' @param p A 3D point, distance from \code{p} to the plane passing through points \code{a}, \code{b},
#' and \code{c} are to be computed.
#' @param a,b,c 3D points that determine the plane (i.e., through which the plane is passing).
#'
#' @return A \code{list} with two elements
#' \item{dis}{Distance from point \code{p} to the plane spanned by 3D points \code{a}, \code{b}, and \code{c}}
#' \item{cl2p}{The closest point on the plane spanned by 3D points \code{a}, \code{b}, and \code{c} to the point \code{p}}
#'
#' @seealso \code{\link{dp2l}}, \code{\link{dist.pt2set}}  and \code{\link{Dist}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' \donttest{
#' A<-c(1,2,3); B<-c(3,9,12); C<-c(1,1,3); P<-c(5,2,40)
#'
#' dis<-dp2pl(P,A,B,C);
#' dis
#' Pr<-dis$Pr2pl #projection on the plane
#'
#' P2<-c(0,0,0)
#' dp2pl(P2,A,B,C);
#'
#' dp2pl(A,A,B,C);
#'
#' xseq<-seq(0,10,l=20)  #try also l=100
#' yseq<-seq(0,10,l=20)  #try also l=100
#'
#' pl.grid<-Plane(A,B,C,xseq,yseq)$z
#'
#' plot3D::persp3D(z = pl.grid, x = xseq, y = yseq, theta =225, phi = 30, ticktype = "detailed")
#' #plane spanned by points A, B, C
#' #add the vertices of the tetrahedron
#' plot3D::points3D(P[1],P[2],P[3], add=TRUE)
#' plot3D::points3D(Pr[1],Pr[2],Pr[3], add=TRUE)
#' plot3D::segments3D(P[1], P[2], P[3], Pr[1], Pr[2],Pr[3], add=TRUE,lwd=2)
#'
#' plot3D::text3D(P[1]-.5,P[2],P[3]+1, c("P"),add=TRUE)
#' plot3D::text3D(Pr[1]-.5,Pr[2],Pr[3]+2, c("Pr"),add=TRUE)
#'
#' persp(xseq,yseq,pl.grid, xlab="x",ylab="y",zlab="z",theta = -30,
#' phi = 30, expand = 0.5, col = "lightblue",
#'       ltheta = 120, shade = 0.05, ticktype = "detailed")
#'
#' dp2pl(P,A,B,C)
#' }
#'
#' @export dp2pl
dp2pl <- function(p,a,b,c)
{
  if (!is.point(p,3) || !is.point(a,3) || !is.point(b,3) || !is.point(c,3))
  {stop('all arguments must be numeric points of dimension 3')}

  p1<-p[1]; p2<-p[2]; p3<-p[3];
  a1<-a[1]; a2<-a[2]; a3<-a[3];
  b1<-b[1]; b2<-b[2]; b3<-b[3];
  c1<-c[1]; c2<-c[2]; c3<-c[3];

  Cx <-(-a2*b3+a2*c3+a3*b2-a3*c2-b2*c3+b3*c2)/(a1*b2-a1*c2-a2*b1+a2*c1+b1*c2-b2*c1);
  Cy <-(a1*b3-a1*c3-a3*b1+a3*c1+b1*c3-b3*c1)/(a1*b2-a1*c2-a2*b1+a2*c1+b1*c2-b2*c1);
  Cz <--1;
  if (abs(Cx)==Inf || abs(Cy)==Inf)  #this part is for when the plane is vertical so Cx or Cy are Inf or -Inf
  {ra<-range(a); a<-a+(ra[2]-ra[1])*runif(3,0,10^(-10))
  rb<-range(b); b<-b+(rb[2]-rb[1])*runif(3,0,10^(-10))
  rc<-range(c); c<-c+(rc[2]-rc[1])*runif(3,0,10^(-10))
  a1<-a[1]; a2<-a[2]; a3<-a[3];
  b1<-b[1]; b2<-b[2]; b3<-b[3];
  c1<-c[1]; c2<-c[2]; c3<-c[3];
  Cx <-(-a2*b3+a2*c3+a3*b2-a3*c2-b2*c3+b3*c2)/(a1*b2-a1*c2-a2*b1+a2*c1+b1*c2-b2*c1);
  Cy <-(a1*b3-a1*c3-a3*b1+a3*c1+b1*c3-b3*c1)/(a1*b2-a1*c2-a2*b1+a2*c1+b1*c2-b2*c1);
  }
  C0 <-(a1*b2*c3-a1*b3*c2-a2*b1*c3+a2*b3*c1+a3*b1*c2-a3*b2*c1)/(a1*b2-a1*c2-a2*b1+a2*c1+b1*c2-b2*c1);

  num.dis<-abs(Cx*p1+Cy*p2+Cz*p3+C0); #numerator of the distance equation

  t<-(Cx*a1-Cx*p1+Cy*a2-Cy*p2+Cz*a3-Cz*p3)/(Cx^2+Cy^2+Cz^2)
  prj<-p+t*c(Cx,Cy,Cz)

  dis<-num.dis/sqrt(Cx^2+Cy^2+Cz^2);

  list(d=dis, #distance
       Pr2pl=prj #c(pr.x,pr.y,pr.z)
  )
} #end of the function
#'

#################################################################

#' @title Generation of Uniform Points in the Standard Regular Tetrahedron \eqn{T_h}
#'
#' @description
#' An object of class \code{"Uniform"}.
#' Generates \code{k} points uniformly in the standard regular tetrahedron
#' \eqn{T_h=T((0,0,0),(1,0,0),(1/2,\sqrt{3}/2,0),(1/2,\sqrt{3}/6,\sqrt{6}/3))}.
#'
#' @param k A positive integer representing the number of uniform points to be generated in the
#' standard regular tetrahedron \eqn{T_h}.
#'
#' @return A \code{list} with the elements
#' \item{type}{The type of the pattern from which points are to be generated}
#' \item{mtitle}{The \code{"main"} title for the plot of the point pattern}
#' \item{tess.points}{The vertices of the support region of the uniformly generated points, it is the
#' standard regular tetrahedron \eqn{T_h} for this function}
#' \item{gen.points}{The output set of generated points uniformly in the standard regular tetrahedron \eqn{T_h}.}
#' \item{out.region}{The outer region which contains the support region, \code{NULL} for this function.}
#' \item{desc.pat}{Description of the point pattern from which points are to be generated}
#' \item{num.points}{The \code{vector} of two numbers, which are the number of generated points and the number
#' of vertices of the support points (here it is 4).}
#' \item{txt4pnts}{Description of the two numbers in \code{num.points}}
#' \item{xlimit,ylimit,zlimit}{The ranges of the \eqn{x}-, \eqn{y}-, and \eqn{z}-coordinates of the support, \eqn{T_h}}
#'
#' @seealso \code{\link{runif.tetra}}, \code{\link{runif.tri}} and \code{\link{runifMT}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' \donttest{
#' A<-c(0,0,0); B<-c(1,0,0); C<-c(1/2,sqrt(3)/2,0); D<-c(1/2,sqrt(3)/6,sqrt(6)/3)
#' tetra<-rbind(A,B,C,D)
#' n<-10  #try also n<-100
#'
#' set.seed(1)
#' Xdt<-runif.stdtetra(n)
#' Xdt
#' summary(Xdt)
#' plot(Xdt)
#'
#' Dt<-runif.stdtetra(n)$g
#'
#' Xlim<-range(tetra[,1])
#' Ylim<-range(tetra[,2])
#' Zlim<-range(tetra[,3])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#' zd<-Zlim[2]-Zlim[1]
#'
#' plot3D::scatter3D(Dt[,1],Dt[,2],Dt[,3],
#' phi =20,theta=15, bty = "g", pch = 20, cex = 1, ticktype = "detailed",
#' xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05), zlim=Zlim+zd*c(-.05,.05))
#' #add the vertices of the tetrahedron
#' plot3D::points3D(tetra[,1],tetra[,2],tetra[,3], add=TRUE)
#' L<-rbind(A,A,A,B,B,C); R<-rbind(B,C,D,C,D,D)
#' plot3D::segments3D(L[,1], L[,2], L[,3], R[,1], R[,2],R[,3], add=TRUE,lwd=2)
#'
#' plot3D::text3D(tetra[,1]+c(.05,0,0,0),tetra[,2],tetra[,3],
#' labels=c("A","B","C","D"), add=TRUE)
#' }
#'
#' \dontrun{
#' #need to install scatterplot3d package and call "library(scatterplot3d)"
#' s3d<-scatterplot3d(Dt, highlight.3d=TRUE,xlab="x",ylab="y",zlab="z",
#'               col.axis="blue", col.grid="lightblue",
#'                main="3D Scatterplot of the data", pch=20)
#' s3d$points3d(tetra,pch=20,col="blue")
#' }
#'
#' @export runif.stdtetra
runif.stdtetra <- function(k)
{
  X <-matrix(0,k,3);
  for (i in 1:k)
  {ct<-0;
  while (ct==0)
  {
    x<-runif(1,0,1); y<-runif(1,0,.866025404); z<-runif(1,0,.816496581)
    if (y < 1.732050808*x & y < 1.732050808-1.732050808*x &
        z < -2.449489743*x+2.449489743-1.414213562*y &
        z < 2.449489743*x-1.414213562*y & z < 2.828427124*y)
    {X[i,]<-c(x,y,z);
    ct<-1;
    }
  }
  }
  A<-c(0,0,0); B<-c(1,0,0); C<-c(1/2,sqrt(3)/2,0); D<-c(1/2,sqrt(3)/6,sqrt(6)/3)
  Th<-rbind(A,B,C,D)  #std regular tetrahedron

  typ<-"Uniform Distribution in the Standard Regular Tetrahedron with Vertices A=(0,0,0), B=(1,0,0), C=(1/2,sqrt(3)/2,0), and D=(1/2,sqrt(3)/6,sqrt(6)/3)"
  main.txt<-"Uniform Points in the \n Standard Regular Tetrahedron"
  txt<-paste(k, "uniform points in the standard regular tetrahedron")

  ny<-4
  npts<-c(k,ny)
  names(npts)<-c("nx","ny")
  txt4pnts<-paste(names(npts)[1], "is the number of Uniform points \n",names(npts)[2], "is the number of vertices of the support region")

  res<-list(
    type=typ,
    mtitle=main.txt,
    tess.points=Th, #tessellation points whose convex hull consititues the support of the uniform points
    gen.points=X, #uniformly generated points in the std regular tetrahedron
    out.region=NULL, #outer region for Te
    desc.pat=txt, #description of the pattern
    num.points=npts,  txt4pnts=txt4pnts,
    xlimit=range(Th[,1]),
    ylimit=range(Th[,2]),
    zlimit=range(Th[,3])
  )

  class(res)<-"Uniform"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

#' @title The plane parallel to the plane spanned by three distinct 3D points \code{a}, \code{b}, and \code{c}
#'
#' @description
#' An object of class \code{"Planes"}.
#' Returns the equation and \eqn{z}-coordinates of the plane passing through point \code{p} and parallel to the plane spanned
#' by three distinct 3D points \code{a}, \code{b}, and \code{c} with \eqn{x}- and \eqn{y}-coordinates are provided in vectors \code{x} and \code{y},
#' respectively.
#'
#' @param p A 3D point which the plane parallel to the plane spanned by
#' three distinct 3D points \code{a}, \code{b}, and \code{c} crosses.
#' @param a,b,c 3D points that determine the plane to which the plane crossing point \code{p} is parallel to.
#' @param x,y A scalar or a \code{vector} of scalars representing the \eqn{x}- and \eqn{y}-coordinates of the plane parallel to
#' the plane spanned by points \code{a}, \code{b}, and \code{c} and passing through point \code{p}.
#'
#' @return A \code{list} with the elements
#' \item{desc}{Description of the plane passing through point \code{p} and parallel to plane spanned by points
#' \code{a}, \code{b} and \code{c}}
#' \item{points}{The input points \code{a}, \code{b}, \code{c}, and \code{p}. Plane is parallel to the plane spanned by \code{a}, \code{b}, and \code{c}
#' and passes through point \code{p} (stacked row-wise, i.e., row 1 is point \code{a}, row 2 is point \code{b},
#'  row 3 is point \code{c}, and row 4 is point \code{p}).}
#' \item{x,y}{The input vectors which constitutes the \eqn{x}- and \eqn{y}-coordinates of the point(s) of interest on the
#' plane. \code{x} and \code{y} can be scalars or vectors of scalars.}
#' \item{z}{The output \code{vector} which constitutes the \eqn{z}-coordinates of the point(s) of interest on the plane.
#' If \code{x} and \code{y} are scalars, \code{z} will be a scalar and
#' if \code{x} and \code{y} are vectors of scalars, then \code{z} needs to be a \code{matrix} of scalars,
#' containing the \eqn{z}-coordinate for each pair of \code{x} and \code{y} values.}
#' \item{coeff}{Coefficients of the plane (in the \eqn{z = A x+B y+C} form).}
#' \item{equation}{Equation of the plane in long form}
#' \item{equation2}{Equation of the plane in short form, to be inserted on the plot}
#'
#' @seealso \code{\link{Plane}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' \donttest{
#' A<-c(1,10,3); B<-c(1,1,3); C<-c(3,9,12); P<-c(1,1,0)
#'
#' Plane(A,B,C,.1,.2)
#'
#' pts<-rbind(A,B,C,P)
#' paraplane(P,A,B,C,.1,.2)
#' paraplane(P,A,B,C,0,0)
#'
#' xr<-range(pts[,1]); yr<-range(pts[,2])
#' xf<-(xr[2]-xr[1])*.25 #how far to go at the lower and upper ends in the x-coordinate
#' yf<-(yr[2]-yr[1])*.25 #how far to go at the lower and upper ends in the y-coordinate
#' x<-seq(xr[1]-xf,xr[2]+xf,l=20)  #try also l=100
#' y<-seq(yr[1]-yf,yr[2]+yf,l=20)  #try also l=100
#'
#' plP2ABC<-paraplane(P,A,B,C,x,y)
#' plP2ABC
#' summary(plP2ABC)
#' plot(plP2ABC)
#'
#' paraplane(P,A,B,A+B,.1,.2)
#'
#' z.grid<-plP2ABC$z
#'
#' plABC<-Plane(A,B,C,x,y)
#' plABC
#' pl.grid<-plABC$z
#'
#' zr<-max(z.grid)-min(z.grid)
#' Pts<-rbind(A,B,C,P)+rbind(c(0,0,zr*.1),c(0,0,zr*.1),c(0,0,zr*.1),c(0,0,zr*.1))
#' Mn.pts<-apply(Pts[1:3,],2,mean)
#'
#' plot3D::persp3D(z = pl.grid, x = x, y = y, theta =225, phi = 30, ticktype = "detailed")
#' #plane spanned by points A, B, C
#' plot3D::persp3D(z = z.grid, x = x, y = y,add=TRUE)
#' #plane parallel to the original plane and passing thru point \code{P}
#'
#' plot3D::persp3D(z = z.grid, x = x, y = y, theta =225, phi = 30, ticktype = "detailed")
#' #plane spanned by points A, B, C
#' #add the defining points
#' plot3D::points3D(Pts[,1],Pts[,2],Pts[,3], add=TRUE)
#' plot3D::text3D(Pts[,1],Pts[,2],Pts[,3], c("A","B","C","P"),add=TRUE)
#' plot3D::text3D(Mn.pts[1],Mn.pts[2],Mn.pts[3],plP2ABC$equation,add=TRUE)
#' plot3D::polygon3D(Pts[1:3,1],Pts[1:3,2],Pts[1:3,3], add=TRUE)
#'
#' P<-c(1,1,1)
#' paraplane(P,A,B,C,.1,.2)
#' }
#'
#' @export paraplane
paraplane <- function(p,a,b,c,x,y)
{
  if (!is.point(p,3) || !is.point(a,3) || !is.point(b,3) || !is.point(c,3))
  {stop('p, a, b, and c must be numeric points of dimension 3')}

  dab<-Dist(a,b); dac<-Dist(a,c); dbc<-Dist(b,c);
  sd<-sort(c(dab,dac,dbc))

  if (isTRUE(all.equal(sd[3] , sd[1]+sd[2])))
  {stop('the three defining points, a, b, and c are collinear; plane is not well-defined')}

  if (!is.point(x,length(x)) || !is.point(y,length(y)))
  {stop('x and y must be numeric vectors')}

  p1<-p[1]; p2<-p[2]; p3<-p[3];
  a1<-a[1]; a2<-a[2]; a3<-a[3];
  b1<-b[1]; b2<-b[2]; b3<-b[3];
  c1<-c[1]; c2<-c[2]; c3<-c[3];

  A <- (-a2*b3+a2*c3+a3*b2-a3*c2-b2*c3+b3*c2)/(a1*b2-a1*c2-a2*b1+a2*c1+b1*c2-b2*c1)
  B <- (a1*b3-a1*c3-a3*b1+a3*c1+b1*c3-b3*c1)/(a1*b2-a1*c2-a2*b1+a2*c1+b1*c2-b2*c1)
  C <- (a1*b2*p3-a1*b3*p2-a1*c2*p3+a1*c3*p2-a2*b1*p3+a2*b3*p1+a2*c1*p3-a2*c3*p1+a3*b1*p2-a3*b2*p1-a3*c1*p2+a3*c2*p1+b1*c2*p3-b1*c3*p2-b2*c1*p3+b2*c3*p1+b3*c1*p2-b3*c2*p1)/(a1*b2-a1*c2-a2*b1+a2*c1+b1*c2-b2*c1)

  lx<-length(x)
  ly<-length(y)

  z.grid<-matrix(0,nrow=lx,ncol=ly)
  for (i in 1:lx)
    for (j in 1:ly)
    {
      z.grid[i,j]<-A*x[i]+B*y[j]+C
    }

  aname <-deparse(substitute(a))
  bname <-deparse(substitute(b))
  cname <-deparse(substitute(c))
  pname <-deparse(substitute(p))

  plane.desc<-paste("Plane Passing through Point ", pname," Parallel to the Plane \n Passing through Points ", aname,", ",bname, " and ", cname,sep="")

  pts<-rbind(a,b,c,p)
  row.names(pts)<-c(aname,bname,cname,pname)

  coeffs<-c(A,B,C)
  names(coeffs)<-c("A","B","C")

  res<-list(
    desc=plane.desc,
    points=pts,
    x=x,
    y=y,
    z=z.grid,
    coeff=coeffs,
    equation=paste("z=",A,"x",ifelse(sign(B)<0,"","+"),B,"y",ifelse(sign(C)<0,"","+"),C,sep=""),
    equation2=paste("z=",round(A,2),"x",ifelse(sign(B)<0,"","+"),round(B,2),"y",
                    ifelse(sign(C)<0,"","+"),round(C,2),sep="")
  )

  class(res)<-"Planes"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

#' @title Check whether a point is inside a tetrahedron
#'
#' @description Checks if the point \code{p} lies in the tetrahedron, \code{th}, using the barycentric coordinates, generally denoted as
#' \eqn{(\alpha,\beta,\gamma)}. If all (normalized or non-normalized) barycentric coordinates are positive then the
#' point \code{p} is inside the tetrahedron, if all are nonnegative with one or more are zero, then \code{p} falls on the
#' boundary. If some of the barycentric coordinates are negative, then \code{p} falls outside the tetrahedron.
#'
#' \code{boundary} is a logical argument (default=\code{FALSE}) to include boundary or not, so if it is \code{TRUE},
#' the function checks if the point, \code{p}, lies in the closure of the tetrahedron (i.e., interior and boundary
#' combined) else it checks if \code{p} lies in the interior of the tetrahedron.
#'
#' @param p A 3D point to be checked whether it is inside the tetrahedron or not.
#' @param th Four 3D points, stacked row-wise, each row representing a vertex of the tetrahedron.
#' @param boundary A logical parameter (default=\code{FALSE}) to include boundary or not, so if it is \code{TRUE},
#' the function checks if the point, \code{p}, lies in the closure of the tetrahedron (i.e., interior and boundary
#' combined) else it checks if \code{p} lies in the interior of the tetrahedron.
#'
#' @return A \code{list} with two elements
#' \item{in.tetra}{A logical output, if the point, \code{p}, is inside the tetrahedron, \code{th}, it is \code{TRUE},
#' else it is \code{FALSE}.}
#' \item{barycentric}{The barycentric coordinates of the point \code{p} with respect to the tetrahedron, \code{th}.}
#'
#' @seealso \code{\link{in.triangle}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' \donttest{
#' A<-c(0,0,0); B<-c(1,0,0); C<-c(1/2,sqrt(3)/2,0); D<-c(1/2,sqrt(3)/6,sqrt(6)/3); P<-c(.1,.1,.1)
#' tetra<-rbind(A,B,C,D)
#'
#' in.tetrahedron(P,tetra,boundary=FALSE)
#'
#' in.tetrahedron(C,tetra,boundary=FALSE)
#' in.tetrahedron(C,tetra,boundary=TRUE)
#'
#' n1<-5; n2<-5; n<-n1+n2
#' Dt<-rbind(cbind(runif(n1),runif(n1,0,sqrt(3)/2),runif(n1,0,sqrt(6)/3)),
#'           runif.tetra(n2,tetra)$g)
#'
#' in.tetra<-vector()
#' for (i in 1:n)
#' {in.tetra<-c(in.tetra,in.tetrahedron(Dt[i,],tetra,boundary=TRUE)$in.tetra) }
#'
#' in.tetra
#' Dt.tet<-Dt[in.tetra,]
#' if (is.vector(Dt.tet)) {Dt.tet<-matrix(Dt.tet,nrow=1)}
#'
#' Xlim<-range(tetra[,1],Dt[,1])
#' Ylim<-range(tetra[,2],Dt[,2])
#' Zlim<-range(tetra[,3],Dt[,3])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#' zd<-Zlim[2]-Zlim[1]
#'
#' plot3D::scatter3D(Dt[,1],Dt[,2],Dt[,3], phi=40,theta=40, bty = "g", pch = 20, cex = 1,
#' ticktype="detailed",xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05),zlim=Zlim+zd*c(-.05,.05))
#' #add the vertices of the tetrahedron
#' plot3D::points3D(tetra[,1],tetra[,2],tetra[,3], add=TRUE)
#' plot3D::points3D(Dt.tet[,1],Dt.tet[,2],Dt.tet[,3],pch=4, add=TRUE)
#' L<-rbind(A,A,A,B,B,C); R<-rbind(B,C,D,C,D,D)
#' plot3D::segments3D(L[,1], L[,2], L[,3], R[,1], R[,2],R[,3], add=TRUE,lwd=2)
#'
#' plot3D::text3D(tetra[,1],tetra[,2],tetra[,3], labels=c("A","B","C","D"), add=TRUE)
#'
#' in.tetrahedron(P,tetra) #this works fine
#'
#' dat.fr<-data.frame(a=tetra) #this works fine
#' in.tetrahedron(P,dat.fr)
#' }
#'
#' @export in.tetrahedron
in.tetrahedron <- function(p,th,boundary=FALSE)
{
  if (!is.point(p,3))
  {stop('p must be a numeric 3D point')}

  th<-as.matrix(th)
  if (!is.numeric(th) || nrow(th)!=4 || ncol(th)!=3)
  {stop('th must be numeric and of dimension 4x3')}

  vec1<-rep(1,4);
  D0<-det(matrix(cbind(th,vec1),ncol=4))
  if (round(D0,14)==0)
  {stop('the tetrahedron is degenerate')}

  if (p[1]=="NA" || p[2]=="NA"|| p[3]=="NA")
  {ind.tetra<-FALSE; b1<-b2<-b3<-b4<-NA;
  } else
  {
    V1<-th[1,];V2<-th[2,];V3<-th[3,];V4<-th[4,];

    D1<-det(matrix(cbind(rbind(p,V2,V3,V4),vec1),ncol=4))
    D2<-det(matrix(cbind(rbind(V1,p,V3,V4),vec1),ncol=4))
    D3<-det(matrix(cbind(rbind(V1,V2,p,V4),vec1),ncol=4))
    D4<-det(matrix(cbind(rbind(V1,V2,V3,p),vec1),ncol=4))
    b1<-D1/D0; b2<-D2/D0; b3<-D3/D0; b4<-D4/D0; # barycentric coordinates

    #if all bi are greater than 0, then the point p lies within the tetrahedron
    if (boundary==FALSE)
    {ifelse(all(c(b1,b2,b3,b4)>0), ind.tetra<-TRUE, ind.tetra<-FALSE)
    }  else
    {ifelse(all(c(b1,b2,b3,b4)>=0), ind.tetra<-TRUE, ind.tetra<-FALSE)}
  }

  list(
   in.tetra=ind.tetra,
    barycentric=c(b1,b2,b3,b4)
  )
}
#'

#################################################################

#' @title Generation of Uniform Points in a tetrahedron
#'
#' @description
#' An object of class \code{"Uniform"}.
#' Generates \code{k} points uniformly in the general tetrahedron \code{th} whose vertices are stacked row-wise.
#'
#' @param k A positive integer representing the number of uniform points to be generated in the tetrahedron.
#' @param th Four 3D points, stacked row-wise, each row representing a vertex of the tetrahedron.
#'
#' @return A \code{list} with the elements
#' \item{type}{The type of the pattern from which points are to be generated}
#' \item{mtitle}{The \code{"main"} title for the plot of the point pattern}
#' \item{tess.points}{The vertices of the support of the uniformly generated points, it is the tetrahedron
#' \code{th} for this function}
#' \item{gen.points}{The output set of generated points uniformly in the tetrahedron, \code{th}.}
#' \item{out.region}{The outer region which contains the support region, \code{NULL} for this function.}
#' \item{desc.pat}{Description of the point pattern from which points are to be generated}
#' \item{num.points}{The \code{vector} of two numbers, which are the number of generated points and the number
#' of vertices of the support points (here it is 4).}
#' \item{txt4pnts}{Description of the two numbers in \code{num.points}}
#' \item{xlimit, ylimit, zlimit}{The ranges of the \eqn{x}-, \eqn{y}-, and \eqn{z}-coordinates of the support, \code{th}}
#'
#' @seealso \code{\link{runif.stdtetra}} and \code{\link{runif.tri}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' \donttest{
#' A<-sample(1:12,3); B<-sample(1:12,3); C<-sample(1:12,3); D<-sample(1:12,3)
#' tetra<-rbind(A,B,C,D)
#'
#' n<-10  #try also n<-100
#'
#' set.seed(1)
#' Xdt<-runif.tetra(n,tetra)
#' Xdt
#' summary(Xdt)
#' plot(Xdt)
#'
#' Dt<-Xdt$g
#'
#' Xlim<-range(tetra[,1],Dt[,1])
#' Ylim<-range(tetra[,2],Dt[,2])
#' Zlim<-range(tetra[,3],Dt[,3])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#' zd<-Zlim[2]-Zlim[1]
#'
#' plot3D::scatter3D(Dt[,1],Dt[,2],Dt[,3], theta =225, phi = 30, bty = "g",
#' xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05), zlim=Zlim+zd*c(-.05,.05),
#'           pch = 20, cex = 1, ticktype = "detailed")
#' #add the vertices of the tetrahedron
#' plot3D::points3D(tetra[,1],tetra[,2],tetra[,3], add=TRUE)
#' L<-rbind(A,A,A,B,B,C); R<-rbind(B,C,D,C,D,D)
#' plot3D::segments3D(L[,1], L[,2], L[,3], R[,1], R[,2],R[,3], add=TRUE,lwd=2)
#'
#' plot3D::text3D(tetra[,1],tetra[,2],tetra[,3], labels=c("A","B","C","D"), add=TRUE)
#'
#' runif.tetra(n,tetra)
#'
#' dat.fr<-data.frame(a=tetra)
#' runif.tetra(n,dat.fr)
#' }
#'
#' \dontrun{
#' #need to install scatterplot3d package and call "library(scatterplot3d)"
#' s3d<-scatterplot3d(Dt, highlight.3d=TRUE,xlab="x",ylab="y",zlab="z",
#'                     col.axis="blue", col.grid="lightblue",
#'                     main="3D Scatterplot of the data", pch=20)
#'  s3d$points3d(tetra,pch=20,col="blue")
#'  }
#'
#' @export runif.tetra
runif.tetra <- function(k,th)
{
  th<-as.matrix(th)
  if (!is.numeric(th) || nrow(th)!=4 || ncol(th)!=3)
  {stop('th must be numeric and of dimension 4x3')}

  vec1<-rep(1,4);
  D0<-det(matrix(cbind(th,vec1),ncol=4))
  if (round(D0,14)==0)
  {stop('the tetrahedron is degenerate')}

  X <-matrix(0,k,3);
  for (i in 1:k)
  {ct<-0;
  while (ct==0)
  {
    x<-runif(1,min(th[,1]),max(th[,1]));
    y<-runif(1,min(th[,2]),max(th[,2]));
    z<-runif(1,min(th[,3]),max(th[,3]))
    p<-c(x,y,z)
    if (in.tetrahedron(p,th)$in.tetra)
    {X[i,]<-p;
    ct<-1;
    }
  }
  }
  typ<-"Uniform Distribution in the Tetrahedron"
  main.txt<-"Uniform Points in the Tetrahedron"
  txt<-paste(k, " uniform points in the tetrahedron with vertices (",th[1,1],",",th[1,2],",",th[1,3],")",", ",
             "(",th[2,1],",",th[2,2],",",th[2,3],")",", ","(",th[3,1],",",th[3,2],",",th[3,3],")",
             " and ","(",th[4,1],",",th[4,2],",",th[4,3],")",sep="")

  ny<-4
  npts<-c(k,ny)
  names(npts)<-c("nx","ny")
  txt4pnts<-paste(names(npts)[1], "is the number of Uniform points \n",names(npts)[2], "is the number of vertices of the support region")

  res<-list(
    type=typ,
    mtitle=main.txt,
    gen.points=X, #uniformly generated points in the std regular tetrahedron
    tess.points=th, #tessellation points whose convex hull consititues the support of the uniform points
    desc.pat=txt, #description of the pattern
    out.region=NULL, #outer region for th
    num.points=npts,  txt4pnts=txt4pnts,
    xlimit=range(th[,1]),
    ylimit=range(th[,2]),
    zlimit=range(th[,3])
  )

  class(res)<-"Uniform"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

#' @title Circumcenter of a general tetrahedron
#'
#' @description Returns the circumcenter a given tetrahedron \code{th} with vertices stacked row-wise.
#'
#' @param th Four 3D points, stacked row-wise, each row representing a vertex of the tetrahedron.
#'
#' @return circumcenter of the tetrahedron \code{th}
#'
#' @seealso \code{\link{circ.cent.tri}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' \donttest{
#' A<-c(0,0,0); B<-c(1,0,0); C<-c(1/2,sqrt(3)/2,0); D<-c(1/2,sqrt(3)/6,sqrt(6)/3)
#' tetra<-rbind(A,B,C,D)
#'
#' CC<-circ.cent.tetra(tetra)
#' CC
#'
#' Xlim<-range(tetra[,1],CC[1])
#' Ylim<-range(tetra[,2],CC[2])
#' Zlim<-range(tetra[,3],CC[3])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#' zd<-Zlim[2]-Zlim[1]
#'
#' plot3D::scatter3D(tetra[,1],tetra[,2],tetra[,3], phi =0,theta=40, bty = "g",
#' xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05), zlim=Zlim+zd*c(-.05,.05),
#'           pch = 20, cex = 1, ticktype = "detailed")
#' #add the vertices of the tetrahedron
#' plot3D::points3D(CC[1],CC[2],CC[3], add=TRUE)
#' L<-rbind(A,A,A,B,B,C); R<-rbind(B,C,D,C,D,D)
#' plot3D::segments3D(L[,1], L[,2], L[,3], R[,1], R[,2],R[,3], add=TRUE,lwd=2)
#'
#' plot3D::text3D(tetra[,1],tetra[,2],tetra[,3], labels=c("A","B","C","D"), add=TRUE)
#'
#' D1<-(A+B)/2; D2<-(A+C)/2; D3<-(A+D)/2; D4<-(B+C)/2; D5<-(B+D)/2; D6<-(C+D)/2;
#' L<-rbind(D1,D2,D3,D4,D5,D6); R<-
#' plot3D::segments3D(L[,1], L[,2], L[,3], R[,1], R[,2],R[,3], add=TRUE,lty=2)
#'
#' plot3D::text3D(CC[1],CC[2],CC[3], labels="CC", add=TRUE)
#'
#' dat.fr<-data.frame(a=tetra)
#' circ.cent.tetra(dat.fr)
#' }
#'
#' @export circ.cent.tetra
circ.cent.tetra <- function(th)
{
  th<-as.matrix(th)
  if (!is.numeric(th) || nrow(th)!=4 || ncol(th)!=3)
  {stop('the argument must be numeric and of dimension 4x3')}

  vec1<-rep(1,4);
  D0<-det(matrix(cbind(th,vec1),ncol=4))
  if (round(D0,14)==0)
  {stop('the tetrahedron is degenerate')}

  A<-th[1,]; B<-th[2,]; C<-th[3,]; D<-th[4,];
  a1<-A[1]; a2<-A[2]; a3<-A[3];
  b1<-B[1]; b2<-B[2]; b3<-B[3];
  c1<-C[1]; c2<-C[2]; c3<-C[3];

  Dx<-det(matrix(cbind(c(sum(A^2),sum(B^2),sum(C^2),sum(D^2)),th[,2],th[,3],vec1),ncol=4))
  Dy<--det(matrix(cbind(c(sum(A^2),sum(B^2),sum(C^2),sum(D^2)),th[,1],th[,3],vec1),ncol=4))
  Dz<-det(matrix(cbind(c(sum(A^2),sum(B^2),sum(C^2),sum(D^2)),th[,1],th[,2],vec1),ncol=4))

  cc1<-Dx/(2*D0); cc2<-Dy/(2*D0); cc3<-Dz/(2*D0);
  c(cc1,cc2,cc3)
} #end of the function
#'

#################################################################

#' @title The index of the \eqn{CC}-vertex region in a tetrahedron that contains a point
#'
#' @description Returns the index of the vertex whose region contains point \code{p} in
#' a tetrahedron \eqn{th=T(A,B,C,D)} and vertex regions are based on the circumcenter \eqn{CC} of \code{th}.
#' (see the plots in the example for illustrations).
#'
#' The vertices of the tetrahedron \code{th} are labeled as
#' \eqn{1=A}, \eqn{2=B}, \eqn{3=C}, and \eqn{4=C} also according to the row number the vertex is recorded in \code{th}.
#'
#' If the point, \code{p}, is not inside \code{th}, then the function yields \code{NA} as output.
#' The corresponding vertex region is the polygon whose interior points are closest to that vertex.
#' If \code{th} is regular tetrahedron, then \eqn{CC} and \eqn{CM} (center of mass) coincide.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010;textual}{pcds}).
#'
#' @param p A 3D point for which \eqn{CC}-vertex region it resides in is to be determined in the
#' tetrahedron \code{th}.
#' @param th Four 3D points, stacked row-wise, each row representing a vertex of the tetrahedron.
#'
#' @return A \code{list} with two elements
#' \item{rv}{Index of the \eqn{CC}-vertex region that contains point, \code{p} in the tetrahedron \code{th}}
#' \item{tri}{The vertices of the tetrahedron, where row number corresponds to the vertex index in \code{rv}.}
#'
#' @seealso \code{\link{rv.tetraCM}} and \code{\link{rv.triCC}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' \donttest{
#' A<-c(0,0,0); B<-c(1,0,0); C<-c(1/2,sqrt(3)/2,0); D<-c(1/2,sqrt(3)/6,sqrt(6)/3)
#' tetra<-rbind(A,B,C,D)
#'
#' n<-10  #try also n<-20
#'
#' Dt<-runif.stdtetra(n)$g
#'
#' rv.tetraCC(Dt[1,],tetra)
#' rv.tetraCC(Dt[5,],tetra)
#' rv.tetraCC(c(2,2,2),tetra)
#'
#' Rv<-vector()
#' for (i in 1:n)
#'  Rv<-c(Rv,rv.tetraCC(Dt[i,],tetra)$rv)
#' Rv
#'
#' CC<-circ.cent.tetra(tetra)
#' CC
#'
#' Xlim<-range(tetra[,1],Dt[,1],CC[1])
#' Ylim<-range(tetra[,2],Dt[,2],CC[2])
#' Zlim<-range(tetra[,3],Dt[,3],CC[3])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#' zd<-Zlim[2]-Zlim[1]
#'
#' plot3D::scatter3D(tetra[,1],tetra[,2],tetra[,3], phi =0,theta=40, bty = "g",
#' xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05), zlim=Zlim+zd*c(-.05,.05),
#'           pch = 20, cex = 1, ticktype = "detailed")
#' L<-rbind(A,A,A,B,B,C); R<-rbind(B,C,D,C,D,D)
#' plot3D::segments3D(L[,1], L[,2], L[,3], R[,1], R[,2],R[,3], add=TRUE,lwd=2)
#' #add the data points
#' plot3D::points3D(Dt[,1],Dt[,2],Dt[,3], add=TRUE)
#'
#' plot3D::text3D(tetra[,1],tetra[,2],tetra[,3], labels=c("A","B","C","D"), add=TRUE)
#' plot3D::text3D(CC[1],CC[2],CC[3], labels=c("CC"), add=TRUE)
#'
#' D1<-(A+B)/2; D2<-(A+C)/2; D3<-(A+D)/2; D4<-(B+C)/2; D5<-(B+D)/2; D6<-(C+D)/2;
#' L<-rbind(D1,D2,D3,D4,D5,D6); R<-matrix(rep(CC,6),ncol=3,byrow=TRUE)
#' plot3D::segments3D(L[,1], L[,2], L[,3], R[,1], R[,2],R[,3], add=TRUE,lty=2)
#'
#' F1<-int.line.plane(A,CC,B,C,D)
#' L<-matrix(rep(F1,4),ncol=3,byrow=TRUE); R<-rbind(D4,D5,D6,CC)
#' plot3D::segments3D(L[,1], L[,2], L[,3], R[,1], R[,2],R[,3],col=2, add=TRUE,lty=2)
#'
#' F2<-int.line.plane(B,CC,A,C,D)
#' L<-matrix(rep(F2,4),ncol=3,byrow=TRUE); R<-rbind(D2,D3,D6,CC)
#' plot3D::segments3D(L[,1], L[,2], L[,3], R[,1], R[,2],R[,3],col=3, add=TRUE,lty=2)
#'
#' F3<-int.line.plane(C,CC,A,B,D)
#' L<-matrix(rep(F3,4),ncol=3,byrow=TRUE); R<-rbind(D3,D5,D6,CC)
#' plot3D::segments3D(L[,1], L[,2], L[,3], R[,1], R[,2],R[,3],col=4, add=TRUE,lty=2)
#'
#' F4<-int.line.plane(D,CC,A,B,C)
#' L<-matrix(rep(F4,4),ncol=3,byrow=TRUE); R<-rbind(D1,D2,D4,CC)
#' plot3D::segments3D(L[,1], L[,2], L[,3], R[,1], R[,2],R[,3],col=5, add=TRUE,lty=2)
#'
#' plot3D::text3D(Dt[,1],Dt[,2],Dt[,3], labels=factor(Rv), add=TRUE)
#'
#' P<-c(.1,.1,.1)
#' rv.tetraCC(P,tetra)
#'
#' dat.fr<-data.frame(a=tetra)
#' rv.tetraCC(P,dat.fr)
#' }
#'
#' @export rv.tetraCC
rv.tetraCC <- function(p,th)
{
  if (!is.point(p,3))
  {stop('p must be a numeric 3D point')}

  th<-as.matrix(th)
  if (!is.numeric(th) || nrow(th)!=4 || ncol(th)!=3)
  {stop('th must be numeric and of dimension 4x3')}

  vec1<-rep(1,4);
  D0<-det(matrix(cbind(th,vec1),ncol=4))
  if (round(D0,14)==0)
  {stop('the tetrahedron is degenerate')}

  if (!in.tetrahedron(p,th,boundary=TRUE)$in.tetra)
  {rv<-NA
  } else
  {
    y1<-th[1,]; y2<-th[2,]; y3<-th[3,]; y4<-th[4,];
    mdt<-max(Dist(y1,y2),Dist(y1,y3),Dist(y1,y4),Dist(y2,y3),Dist(y2,y4),Dist(y3,y4))
    for (i in 1:4)
    {
      d1<-Dist(p,th[i,]);
      if (d1<mdt)
      {mdt<-d1; rv<-i }
    }
  }
  row.names(th)<-c("vertex 1","vertex 2","vertex 3","vertex 4")  #vertex labeling

  list(rv=rv, #relative vertex
       tetra=th #vertex labeling
  )
} #end of the function
#'

#################################################################

#' @title The index of the \eqn{CM}-vertex region in a tetrahedron that contains a point
#'
#' @description Returns the index of the vertex whose region contains point \code{p} in
#' a tetrahedron \eqn{th=T(A,B,C,D)} and vertex regions are based on the center of mass \eqn{CM=(A+B+C+D)/4} of \code{th}.
#' (see the plots in the example for illustrations).
#'
#' The vertices of the tetrahedron \code{th} are labeled as
#' \eqn{1=A}, \eqn{2=B}, \eqn{3=C}, and \eqn{4=C} also according to the row number the vertex is recorded in \code{th}.
#'
#' If the point, \code{p}, is not inside \code{th}, then the function yields \code{NA} as output.
#' The corresponding vertex region is the simplex with the vertex, \eqn{CM}, and
#' midpoints of the edges adjacent to the vertex.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010;textual}{pcds}).
#'
#' @param p A 3D point for which \eqn{CM}-vertex region it resides in is to be determined in the
#' tetrahedron \code{th}.
#' @param th Four 3D points, stacked row-wise, each row representing a vertex of the tetrahedron.
#'
#' @return A \code{list} with two elements
#' \item{rv}{Index of the \eqn{CM}-vertex region that contains point, \code{p} in the tetrahedron \code{th}}
#' \item{th}{The vertices of the tetrahedron, where row number corresponds to the vertex index in \code{rv}.}
#'
#' @seealso \code{\link{rv.tetraCC}} and \code{\link{rv.triCM}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' \donttest{
#' A<-c(0,0,0); B<-c(1,0,0); C<-c(1/2,sqrt(3)/2,0); D<-c(1/2,sqrt(3)/6,sqrt(6)/3)
#' tetra<-rbind(A,B,C,D)
#'
#' n<-10  #try also n<-20
#'
#' Dt<-runif.stdtetra(n)$g
#'
#' rv.tetraCM(Dt[1,],tetra)
#' rv.tetraCM(Dt[5,],tetra)
#' rv.tetraCM(c(2,2,2),tetra)
#'
#' Rv<-vector()
#' for (i in 1:n)
#'   Rv<-c(Rv, rv.tetraCM(Dt[i,],tetra)$rv )
#' Rv
#'
#' Xlim<-range(tetra[,1],Dt[,1])
#' Ylim<-range(tetra[,2],Dt[,2])
#' Zlim<-range(tetra[,3],Dt[,3])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#' zd<-Zlim[2]-Zlim[1]
#'
#' CM<-apply(tetra,2,mean)
#'
#' plot3D::scatter3D(tetra[,1],tetra[,2],tetra[,3], phi =0,theta=40, bty = "g",
#' xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05), zlim=Zlim+zd*c(-.05,.05),
#'           pch = 20, cex = 1, ticktype = "detailed")
#' L<-rbind(A,A,A,B,B,C); R<-rbind(B,C,D,C,D,D)
#' plot3D::segments3D(L[,1], L[,2], L[,3], R[,1], R[,2],R[,3], add=TRUE,lwd=2)
#' #add the data points
#' plot3D::points3D(Dt[,1],Dt[,2],Dt[,3], add=TRUE)
#'
#' plot3D::text3D(tetra[,1],tetra[,2],tetra[,3], labels=c("A","B","C","D"), add=TRUE)
#' plot3D::text3D(CM[1],CM[2],CM[3], labels=c("CM"), add=TRUE)
#'
#' D1<-(A+B)/2; D2<-(A+C)/2; D3<-(A+D)/2; D4<-(B+C)/2; D5<-(B+D)/2; D6<-(C+D)/2;
#' L<-rbind(D1,D2,D3,D4,D5,D6); R<-matrix(rep(CM,6),ncol=3,byrow=TRUE)
#' plot3D::segments3D(L[,1], L[,2], L[,3], R[,1], R[,2],R[,3], add=TRUE,lty=2)
#'
#' F1<-int.line.plane(A,CM,B,C,D)
#' L<-matrix(rep(F1,4),ncol=3,byrow=TRUE); R<-rbind(D4,D5,D6,CM)
#' plot3D::segments3D(L[,1], L[,2], L[,3], R[,1], R[,2],R[,3],col=2, add=TRUE,lty=2)
#'
#' F2<-int.line.plane(B,CM,A,C,D)
#' L<-matrix(rep(F2,4),ncol=3,byrow=TRUE); R<-rbind(D2,D3,D6,CM)
#' plot3D::segments3D(L[,1], L[,2], L[,3], R[,1], R[,2],R[,3],col=3, add=TRUE,lty=2)
#'
#' F3<-int.line.plane(C,CM,A,B,D)
#' L<-matrix(rep(F3,4),ncol=3,byrow=TRUE); R<-rbind(D3,D5,D6,CM)
#' plot3D::segments3D(L[,1], L[,2], L[,3], R[,1], R[,2],R[,3],col=4, add=TRUE,lty=2)
#'
#' F4<-int.line.plane(D,CM,A,B,C)
#' L<-matrix(rep(F4,4),ncol=3,byrow=TRUE); R<-rbind(D1,D2,D4,CM)
#' plot3D::segments3D(L[,1], L[,2], L[,3], R[,1], R[,2],R[,3],col=5, add=TRUE,lty=2)
#'
#' plot3D::text3D(Dt[,1],Dt[,2],Dt[,3], labels=factor(Rv), add=TRUE)
#'
#' P<-c(.1,.1,.1)
#' rv.tetraCM(P,tetra)
#'
#' dat.fr<-data.frame(a=tetra)
#' rv.tetraCM(P,dat.fr)
#' }
#'
#' @export rv.tetraCM
rv.tetraCM <- function(p,th)
{
  if (!is.point(p,3))
  {stop('p must be a numeric 3D point')}

  th<-as.matrix(th)
  if (!is.numeric(th) || nrow(th)!=4 || ncol(th)!=3)
  {stop('th must be numeric and of dimension 4x3')}

  vec1<-rep(1,4);
  D0<-det(matrix(cbind(th,vec1),ncol=4))
  if (round(D0,14)==0)
  {stop('the tetrahedron is degenerate')}

  if (!in.tetrahedron(p,th,boundary=TRUE)$in.tetra)
  {rv<-NA
  } else
  {
    CM<-apply(th,2,mean)
    i<-1
    while (i <=4)
    { th0<-th[-i,] #remove ith vertex
    y1<-th[i,]; y2<-th0[1,]; y3<-th0[2,]; y4<-th0[3,];
    M1<-(y1+y2)/2; M2<-(y1+y3)/2; M3<-(y1+y4)/2;
    th1<-rbind(y1,M1,y3,y4); th2<-rbind(y1,M2,y2,y4); th3<-rbind(y1,M3,y2,y3);
    if (in.tetrahedron(p,th1,boundary=TRUE)$i &&
       in.tetrahedron(p,th2,boundary=TRUE)$i &&
       in.tetrahedron(p,th3,boundary=TRUE)$i)
    {rv<-i; i<-5
    } else
    {i<-i+1}
    }
  }
  row.names(th)<-c("vertex 1","vertex 2","vertex 3","vertex 4")  #vertex labeling

  list(rv=rv, #relative vertex
       tetra=th #vertex labeling
  )
} #end of the function
#'

#################################################################

#' @title The closest points among a data set in the vertex regions to the respective faces in a tetrahedron
#'
#' @description
#' An object of class \code{"Extrema"}.
#' Returns the closest data points among the data set, \code{Dt}, to face \eqn{i} in \code{M}-vertex region \eqn{i} for \eqn{i=1,2,3,4}
#' in the tetrahedron \eqn{th=T(A,B,C,D)}. Vertex labels are \eqn{A=1}, \eqn{B=2}, \eqn{C=3}, and \eqn{D=4} and corresponding face
#' labels are \eqn{BCD=1}, \eqn{ACD=2}, \eqn{ABD=3}, and \eqn{ABC=4}.
#'
#' Vertex regions are based on center \code{M} which can be the center
#' of mass (\code{"CM"}) or circumcenter (\code{"CC"}) of \code{th}.
#'
#' @param Dt A set of 3D points representing the set of data points.
#' @param th Four 3D points, stacked row-wise, each row representing a vertex of the tetrahedron.
#' @param M The center to be used in the construction of the vertex regions in the tetrahedron, \code{th}.
#' Currently it only takes \code{"CC"} for circumcenter and \code{"CM"} for center of mass; default=\code{"CM"}.
#'
#' @return A \code{list} with the elements
#' \item{txt1}{Vertex labels are \eqn{A=1}, \eqn{B=2}, \eqn{C=3}, and \eqn{D=4} (corresponds to row number in Extremum Points).}
#' \item{txt2}{A short description of the distances as \code{"Distances from Closest Points to Faces ..."}.}
#' \item{type}{Type of the extrema points}
#' \item{desc}{A short description of the extrema points}
#' \item{mtitle}{The \code{"main"} title for the plot of the extrema}
#' \item{ext}{The extrema points, here, closest points to faces in the respective vertex region.}
#' \item{X}{The input data, \code{Dt}, can be a \code{matrix} or \code{data frame}}
#' \item{num.points}{The number of data points, i.e., size of \code{Dt}}
#' \item{supp}{Support of the data points, here, it is \code{th}}
#' \item{cent}{The center point used for construction of vertex regions, it is circumcenter of center of mass
#' for this function}
#' \item{ncent}{Name of the center, it is circumcenter \code{"CC"} or  center of mass \code{"CM"} for this function.}
#' \item{regions}{Vertex regions inside the tetrahedron \code{th} provided as a list.}
#' \item{region.names}{Names of the vertex regions as \code{"vr=1","vr=2","vr=3","vr=4"}}
#' \item{region.centers}{Centers of mass of the vertex regions inside \code{th}.}
#' \item{dist2ref}{Distances from closest points in each vertex region to the corresponding face.}
#'
#' @seealso \code{\link{fr2vVRCC}}, \code{\link{fr2eTeER}}, \code{\link{Kfr2vTbVRCC}} and \code{\link{Kfr2vVRCC}}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' \donttest{
#' A<-c(0,0,0); B<-c(1,0,0); C<-c(1/2,sqrt(3)/2,0); D<-c(1/2,sqrt(3)/6,sqrt(6)/3)
#' tetra<-rbind(A,B,C,D)
#' Cent<-"CC"  #try also "CM"
#'
#' n<-10  #try also n<-20
#' Dt<-runif.tetra(n,tetra)$g  #try also Dt<-cbind(runif(n),runif(n),runif(n))
#'
#' Ext<-cl2fVRth(Dt,tetra,Cent)
#' Ext
#' summary(Ext)
#' plot(Ext)
#'
#' cl2fVRth(c(.5,.5,.5),tetra,Cent)
#'
#' cl2fVRth(Dt,tetra,Cent)
#' clf<-cl2fVRth(Dt,tetra,Cent)$ext
#' clf
#'
#' if (Cent=="CC") {M<-circ.cent.tetra(tetra)}
#' if (Cent=="CM") {M<-apply(tetra,2,mean)}
#'
#' Xlim<-range(tetra[,1],Dt[,1],M[1])
#' Ylim<-range(tetra[,2],Dt[,2],M[2])
#' Zlim<-range(tetra[,3],Dt[,3],M[3])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#' zd<-Zlim[2]-Zlim[1]
#'
#' plot3D::scatter3D(Dt[,1],Dt[,2],Dt[,3], phi =0,theta=40, bty = "g",
#' xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05), zlim=Zlim+zd*c(-.05,.05),
#'           pch = 20, cex = 1, ticktype = "detailed")
#' #add the vertices of the tetrahedron
#' plot3D::points3D(tetra[,1],tetra[,2],tetra[,3], add=TRUE)
#' L<-rbind(A,A,A,B,B,C); R<-rbind(B,C,D,C,D,D)
#' plot3D::segments3D(L[,1], L[,2], L[,3], R[,1], R[,2],R[,3], add=TRUE,lwd=2)
#' plot3D::points3D(clf[,1],clf[,2],clf[,3], pch=4,col="red", add=TRUE)
#'
#' plot3D::text3D(tetra[,1],tetra[,2],tetra[,3], labels=c("A","B","C","D"), add=TRUE)
#'
#' #for center of mass use #Cent<-apply(tetra,2,mean)
#' D1<-(A+B)/2; D2<-(A+C)/2; D3<-(A+D)/2; D4<-(B+C)/2; D5<-(B+D)/2; D6<-(C+D)/2;
#' L<-rbind(D1,D2,D3,D4,D5,D6); R<-rbind(M,M,M,M,M,M)
#' plot3D::segments3D(L[,1], L[,2], L[,3], R[,1], R[,2],R[,3], add=TRUE,lty=2)
#'
#' cl2fVRth(Dt,tetra,Cent)
#'
#' P<-c(.1,.1,.1)
#' cl2fVRth(P,tetra,Cent)
#'
#' dat.fr<-data.frame(a=Dt)
#' cl2fVRth(dat.fr,tetra,Cent)
#'
#' dat.fr<-data.frame(a=tetra)
#' cl2fVRth(Dt,dat.fr,Cent)
#' }
#'
#' @export cl2fVRth
cl2fVRth <- function(Dt,th,M="CM")
{
  if (!is.numeric(as.matrix(Dt)))
  {stop('Dt must be numeric')}

  if (is.point(Dt,3))
  { Dt<-matrix(Dt,ncol=3)
  } else
  {Dt<-as.matrix(Dt)
  if (ncol(Dt)!=3 )
  {stop('Dt must be of dimension nx3')}
  }

  th<-as.matrix(th)
  if (!is.numeric(th) || nrow(th)!=4 || ncol(th)!=3)
  {stop('th must be numeric and of dimension 4x3')}

  vec1<-rep(1,4);
  D0<-det(matrix(cbind(th,vec1),ncol=4))
  if (round(D0,14)==0)
  {stop('the tetrahedron is degenerate')}

  if (length(M) > 1 || sum(M==c("CM","CC"))==0)
    stop("M must be one of \"CC\", \"CM\"")

  y1<-A<-th[1,]; y2<-B<-th[2,]; y3<-C<-th[3,]; y4<-D<-th[4,];
  ifelse(identical(M,"CC"), {Cent<-circ.cent.tetra(th); cent.name<-"CC"},
         {Cent<-apply(th, 2, mean); cent.name<-"CM"})
  D1<-(y1+y2)/2; D2<-(y1+y3)/2; D3<-(y1+y4)/2; D4<-(y2+y3)/2; D5<-(y2+y4)/2; D6<-(y3+y4)/2;
  Ds<-rbind(D1,D2,D3,D4,D5,D6)

  mdt<-c(dp2pl(y1,y2,y3,y4)$d,dp2pl(y2,y1,y3,y4)$d,dp2pl(y3,y1,y2,y4)$d,dp2pl(y4,y1,y2,y3)$d);
  U<-matrix(NA,nrow=4,ncol=3);

  n<-nrow(Dt);

  for (i in 1:n)
  {ifelse(identical(M,"CC"),rv<-rv.tetraCC(Dt[i,],th)$rv,rv<-rv.tetraCM(Dt[i,],th)$rv);
    if (!is.na(rv))
    {
      if (rv==1)
      {d1<-dp2pl(Dt[i,],y2,y3,y4)$d;
      if (d1<=mdt[1]) {mdt[1]<-d1; U[1,]<-Dt[i,]};
      } else if (rv==2)
      {d2<-dp2pl(Dt[i,],y1,y3,y4)$d;
      if (d2<=mdt[2]) {mdt[2]<-d2; U[2,]<-Dt[i,]}
      } else if (rv==3)
      {d3<-dp2pl(Dt[i,],y1,y2,y4)$d;
      if (d3<=mdt[3]) {mdt[3]<-d3; U[3,]<-Dt[i,]}
      } else {
        d4<-dp2pl(Dt[i,],y1,y2,y3)$d;
        if (d4<=mdt[4]) {mdt[4]<-d4; U[4,]<-Dt[i,]}
      }
    }
  }

  row.names(th)<-c("A","B","C","D")  #vertex labeling
  typ<-paste("Closest Points to Faces in the Respective ",cent.name,"-Vertex Regions in the Tetrahedron with Vertices A, B, C and D",sep="")
  description<-paste("Closest Points to Faces in the Respective ",cent.name,"-Vertex Regions\n (Row i corresponds to face i for i=1,2,3,4)",sep="")
  txt1<-"Vertex labels are A=1, B=2, C=3, and D=4 (corresponds to row number in Extremum Points)"
  txt2<-paste("Distances from Closest Points to Faces in the Respective ",cent.name,"-Vertex Regions\n (Row i corresponds to vertex i for i=1,2,3)",sep="")
  main.txt<-paste("Closest Points to Faces in\n Respective ",cent.name,"-Vertex Regions",sep="")

  Dis<-rbind(ifelse(!is.na(U[1,1]),mdt[1],NA),ifelse(!is.na(U[2,1]),mdt[2],NA),
             ifelse(!is.na(U[3,1]),mdt[3],NA),ifelse(!is.na(U[4,1]),mdt[4],NA))
  #distances of the closest points to the faces in the respectivevertex regions
  Regs<-list(vr1=rbind(A,D1,D2,D3,Cent), #regions inside the triangles
             vr2=rbind(B,D1,D4,D5,Cent),
             vr3=rbind(C,D4,D2,D6,Cent),
             vr4=rbind(D,D3,D5,D6,Cent)
  )
  Reg.Cent<-vector()
  for (i in 1:length(Regs))
  { Reg.Cent<-rbind(Reg.Cent,apply(Regs[[i]],2,mean))}

  Reg.names<-c("vr=1","vr=2","vr=3","vr=4")  #regions names

  res<-list(
    txt1=txt1, txt2=txt2,
    type=typ, desc=description,
    mtitle=main.txt, #main label in the plot
    ext=U, #furthest points from edges in each edge region
    X=Dt, num.points=n, #data points and its size
    ROI=th, # region of interest for X points
    cent=Cent, ncent=cent.name, #center and center name
    regions=Regs, region.names=Reg.names, region.centers=Reg.Cent,
    dist2ref=Dis #distances of closest points to edges in the respective vertex region
  )

  class(res)<-"Extrema"
  res$call <-match.call()
  res
} #end of the function
#'

#################################################################

#' @title The vertices of the Proportional Edge (PE) Proximity Region in the standard regular tetrahedron
#'
#' @description Returns the vertices of the PE proximity region (which is itself a tetrahedron) for a point in the
#' standard regular tetrahedron \eqn{T_h=T((0,0,0),(1,0,0),(1/2,\sqrt{3}/2,0),(1/2,\sqrt{3}/6,\sqrt{6}/3))=}
#' \code{(rv=1,rv=2,rv=3,rv=4)}.
#'
#' PE proximity region is defined with respect to the tetrahedron \eqn{T_h}
#' with expansion parameter \eqn{r \ge 1} and vertex regions based on the circumcenter of \eqn{T_h} (which is equivalent
#' to the center of mass in the standard regular tetrahedron).
#'
#' Vertex regions are labeled as \code{1,2,3,4} rowwise for the vertices of the tetrahedron \eqn{T_h}.
#' \code{rv} is the index of the vertex region \code{pt} resides, with default=\code{NULL}.
#' If \code{pt} is outside of \eqn{T_h}, it returns \code{NULL} for the proximity region.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010;textual}{pcds}).
#'
#' @param pt A 3D point whose PE proximity region is to be computed.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param rv Index of the vertex region containing the point, either \code{1,2,3,4} or \code{NULL}
#' (default is \code{NULL}).
#'
#' @return Vertices of the tetrahedron which constitutes the PE proximity region with expansion parameter
#' \code{r} and circumcenter (or center of mass) for a point \code{pt} in the standard regular tetrahedron
#'
#' @seealso \code{\link{NPEtetra}}, \code{\link{NPEtri}} and \code{\link{NPEint}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(0,0,0); B<-c(1,0,0); C<-c(1/2,sqrt(3)/2,0); D<-c(1/2,sqrt(3)/6,sqrt(6)/3)
#' tetra<-rbind(A,B,C,D)
#'
#' n<-10  #try also n<-20
#' dat<-runif.stdtetra(n)$g
#' r<-1.5
#' NPEstdtetra(dat[1,],r)
#'
#' NPEstdtetra(dat[5,],r)
#'
#' NPEstdtetra(c(.4,.4,.4),r)
#' NPEstdtetra(c(.5,.5,.5),r)
#' NPEstdtetra(c(.5,.5,5),r)
#'
#' #or try
#' RV<-rv.tetraCC(dat[1,],tetra)$rv
#' NPEstdtetra(dat[1,],r,rv=RV)
#'
#' NPEstdtetra(c(-1,-1,-1),r,rv=NULL)
#'
#' P1<-c(.1,.1,.1)
#' P2<-c(.5,.5,.5)
#' NPEstdtetra(P1,r)
#'
#' @export NPEstdtetra
NPEstdtetra <- function(pt,r,rv=NULL)
{
  if (!is.point(pt,3) )
  {stop('pt must be a numeric 3D point')}

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  A<-c(0,0,0); B<-c(1,0,0); C<-c(1/2,sqrt(3)/2,0); D<-c(1/2,sqrt(3)/6,sqrt(6)/3)
  th<-rbind(A,B,C,D)  #standard regular tetrahedron

  if (!in.tetrahedron(pt,th,boundary=TRUE)$in.tetra)
  {reg<-NULL; return(reg); stop}

  if (is.null(rv))
  {rv<-rv.tetraCC(pt,th)$rv #vertex region for pt
  } else
  {  if (!is.numeric(rv) || sum(rv==c(1,2,3,4))!=1)
  {stop('vertex index, rv, must be 1, 2, 3 or 4')}}

  p1<-pt[1]; p2<-pt[2]; p3<-pt[3];

  if (rv==1)
  {
    A1<-c((1/6*(sqrt(3)*sqrt(2)*p3+2*sqrt(3)*p2+6*p1))*r, 0, 0);
    A2<-c((1/12*(sqrt(3)*sqrt(2)*p3+2*sqrt(3)*p2+6*p1))*r, (1/12)*sqrt(3)*(sqrt(3)*sqrt(2)*p3+2*sqrt(3)*p2+6*p1)*r, 0);
    A3<-c((1/12*(sqrt(3)*sqrt(2)*p3+2*sqrt(3)*p2+6*p1))*r, (1/36)*sqrt(3)*(sqrt(3)*sqrt(2)*p3+2*sqrt(3)*p2+6*p1)*r, (1/18)*sqrt(3)*sqrt(2)*(sqrt(3)*sqrt(2)*p3+2*sqrt(3)*p2+6*p1)*r);
    reg<-rbind(A,A1,A2,A3)
  } else if (rv==2)
  {
    B1<-c((1/2)*p1*r-(1/2)*r-(1/6)*sqrt(3)*p2*r+1-(1/12)*sqrt(3)*sqrt(2)*p3*r, (1/12)*sqrt(3)*(sqrt(3)*sqrt(2)*p3+2*sqrt(3)*p2-6*p1+6)*r, 0);
    B2<-c((1/2)*p1*r-(1/2)*r-(1/6)*sqrt(3)*p2*r+1-(1/12)*sqrt(3)*sqrt(2)*p3*r, (1/36)*sqrt(3)*(sqrt(3)*sqrt(2)*p3+2*sqrt(3)*p2-6*p1+6)*r, (1/18)*sqrt(3)*sqrt(2)*(sqrt(3)*sqrt(2)*p3+2*sqrt(3)*p2-6*p1+6)*r);
    B3<-c(p1*r-r-(1/3)*sqrt(3)*p2*r+1-(1/6)*sqrt(3)*sqrt(2)*p3*r, 0, 0);
    reg<-rbind(B,B1,B2,B3)
  } else if (rv==3)
  {
    C1<-c(-(1/12)*sqrt(3)*(sqrt(2)*p3*r+2*sqrt(3)*r-4*p2*r-2*sqrt(3)), -(1/4)*sqrt(2)*p3*r-(1/2)*sqrt(3)*r+p2*r+(1/2)*sqrt(3), 0);
    C2<-c(1/2, -(1/3)*sqrt(3)*r+(2/3)*p2*r+(1/2)*sqrt(3)-(1/6)*sqrt(2)*p3*r, (1/3)*sqrt(3)*sqrt(2)*r-(2/3)*sqrt(2)*p2*r+(1/3)*p3*r);
    C3<-c(-(1/12)*sqrt(3)*(-sqrt(2)*p3*r-2*sqrt(3)*r+4*p2*r-2*sqrt(3)), -(1/4)*sqrt(2)*(p3*r+sqrt(3)*sqrt(2)*r-2*sqrt(2)*p2*r-sqrt(3)*sqrt(2)), 0);
    reg<-rbind(C,C1,C2,C3)
  } else {
    D1<-c(-(1/12)*sqrt(6)*(sqrt(6)*r-3*p3*r-sqrt(6)), -(1/12)*sqrt(2)*(sqrt(3)*sqrt(2)*r-sqrt(3)*sqrt(2)-3*p3*r), -(1/3)*sqrt(3)*sqrt(2)*r+(1/3)*sqrt(3)*sqrt(2)+p3*r);
    D2<-c((1/12*(sqrt(6)*r-3*p3*r+sqrt(6)))*sqrt(6), -(1/12)*sqrt(2)*(sqrt(3)*sqrt(2)*r-sqrt(3)*sqrt(2)-3*p3*r), -(1/3)*sqrt(3)*sqrt(2)*r+(1/3)*sqrt(3)*sqrt(2)+p3*r);
    D3<-c(1/2, -(1/36)*sqrt(6)*sqrt(3)*(6*p3*r-2*r*sqrt(6)-sqrt(6)), -(1/3)*r*sqrt(6)+p3*r+(1/3)*sqrt(6));
    reg<-rbind(D,D1,D2,D3)
  }

  vec1<-rep(1,4);
  D0<-abs(det(matrix(cbind(th,vec1),ncol=4)))
  D1<-abs(det(matrix(cbind(reg,vec1),ncol=4)))
  if (D1>=D0)
  {reg<-th}
  row.names(reg)<-c()
  reg
} #end of the function
#'

#################################################################

#' @title The indicator for the presence of an arc from a point to another for Proportional Edge Proximity Catch
#' Digraphs (PE-PCDs) - standard regular tetrahedron case
#'
#' @description Returns \eqn{I(}\code{pt2} is in \eqn{N_{PE}(pt1,r))} for points \code{pt1} and \code{pt2}, that is, returns 1 if \code{pt2} is in \eqn{N_{PE}(pt1,r)},
#' returns 0 otherwise, where \eqn{N_{PE}(x,r)} is the PE proximity region for point \eqn{x} with expansion parameter \eqn{r \ge 1}.
#'
#' PE proximity region is defined with respect to the standard regular tetrahedron
#' \eqn{T_h=T(v=1,v=2,v=3,v=4)=T((0,0,0),(1,0,0),(1/2,\sqrt{3}/2,0),(1/2,\sqrt{3}/6,\sqrt{6}/3))} and vertex regions
#' are based on the circumcenter (which is equivalent to the center of mass for standard regular tetrahedron)
#' of \eqn{T_h}. \code{rv} is the index of the vertex region \code{pt1} resides, with default=\code{NULL}.
#'
#' If \code{pt1} and \code{pt2} are distinct and either of them are outside \eqn{T_h}, it returns 0,
#' but if they are identical, then it returns 1 regardless of their locations (i.e., it allows loops).
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010;textual}{pcds}).
#'
#' @param pt1 A 3D point whose PE proximity region is constructed.
#' @param pt2 A 3D point. The function determines whether \code{pt2} is inside the PE proximity region of
#' \code{pt1} or not.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param rv Index of the vertex region containing the point, either \code{1,2,3,4} (default is \code{NULL}).
#'
#' @return \eqn{I(}\code{pt2} is in \eqn{N_{PE}(pt1,r))} for points \code{pt1} and \code{pt2}, that is, returns 1 if \code{pt2} is in \eqn{N_{PE}(pt1,r)},
#' returns 0 otherwise
#'
#' @seealso \code{\link{IndNPEtetra}}, \code{\link{IndNPEtri}} and \code{\link{IndNPEint}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(0,0,0); B<-c(1,0,0); C<-c(1/2,sqrt(3)/2,0); D<-c(1/2,sqrt(3)/6,sqrt(6)/3)
#' tetra<-rbind(A,B,C,D)
#'
#' n<-10  #try also n<-20
#' dat<-runif.stdtetra(n)$g
#' r<-1.5
#' IndNPEstdtetra(dat[1,],dat[1,],r)
#'
#' IndNPEstdtetra(dat[1,],dat[5,],r)
#'
#' IndNPEstdtetra(c(.4,.4,.4),c(.5,.5,.5),r)
#'
#' #or try
#' RV<-rv.tetraCC(dat[1,],tetra)$rv
#' IndNPEstdtetra(dat[1,],dat[5,],r,rv=RV)
#'
#' IndNPEstdtetra(dat[1,],c(-1,-1,-1),r,rv=NULL)
#'
#' IndNPEstdtetra(c(-1,-1,-1),dat[1,],r,rv=NULL)
#' IndNPEstdtetra(c(-1,-1,-1),c(-1,-1,-1),r)
#'
#' IndNPEstdtetra(dat[1,],dat[5,],r)
#'
#' P1<-c(.1,.1,.1)
#' P2<-c(.5,.5,.5)
#' IndNPEstdtetra(P1,P2,r)
#'
#' @export IndNPEstdtetra
IndNPEstdtetra <- function(pt1,pt2,r,rv=NULL)
{
  if (!is.point(pt1,3) || !is.point(pt2,3))
  {stop('pt1 and pt2 must be a numeric 3D point')}

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if (identical(pt1,pt2))
  {arc<-1; return(arc); stop}

  A<-c(0,0,0); B<-c(1,0,0); C<-c(1/2,sqrt(3)/2,0); D<-c(1/2,sqrt(3)/6,sqrt(6)/3)
  th<-rbind(A,B,C,D)  #standard regular tetrahedron

  if (!in.tetrahedron(pt1,th,boundary=TRUE)$in.tetra || !in.tetrahedron(pt2,th,boundary=TRUE)$in.tetra)
  {arc<-0
  } else
  {
    if (is.null(rv))
    {rv<-rv.tetraCC(pt1,th)$rv #vertex region for pt1
    } else
    {  if (!is.numeric(rv) || sum(rv==c(1,2,3,4))!=1)
    {stop('vertex index, rv, must be 1, 2, 3 or 4')}}

    y1<-th[1,]; y2<-th[2,]; y3<-th[3,]; y4<-th[4,];
    X1<-pt1[1]; Y1<-pt1[2]; Z1<-pt1[3];
    X2<-pt2[1]; Y2<-pt2[2]; Z2<-pt2[3];
    arc<-0;
    if (rv==1)
    {
      x1n<-X1*r; y1n<-Y1*r; z1n<-Z1*r;
      if ( Z2 < as.vector(paraplane(c(x1n,y1n,z1n),y2,y3,y4,X2,Y2)$z) ) {arc <-1}
    } else {
      if (rv==2)
      {
        x1n<-1+(X1-1)*r; y1n<-Y1*r;  z1n<-Z1*r;
        if ( Z2 < as.vector(paraplane(c(x1n,y1n,z1n),y1,y3,y4,X2,Y2)$z) ) {arc <-1}
      } else {
        if (rv==3)
        {
          x1n<-1/2+(X1-1/2)*r; y1n<-sqrt(3)/2+(Y1-sqrt(3)/2)*r; z1n<-Z1*r;
          if ( Z2 < as.vector(paraplane(c(x1n,y1n,z1n),y1,y2,y4,X2,Y2)$z) ) {arc<-1}
        } else {
          z1n<-y4[3]+(Z1-y4[3])*r;
          if ( Z2 > z1n ) {arc<-1}
        }}}
  }
  arc
} #end of the function
#'

#################################################################

#' @title The plot of the Proportional Edge (PE) Proximity Regions for a 3D data set - standard
#' regular tetrahedron case
#'
#' @description Plots the points in and outside of the standard regular tetrahedron
#' \eqn{T_h=T((0,0,0),(1,0,0),(1/2,\sqrt{3}/2,0),(1/2,\sqrt{3}/6,\sqrt{6}/3))} and also the PE proximity regions
#' for points in data set \code{Xp}.
#'
#' PE proximity regions are defined with respect to the standard regular tetrahedron \eqn{T_h}
#' with expansion parameter \eqn{r \ge 1}, so PE proximity regions are defined only for points inside \eqn{T_h}.
#'
#' Vertex regions are based on circumcenter (which is equivalent to the center of mass for the standard
#' regular tetrahedron) of \eqn{T_h}.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010;textual}{pcds}).
#'
#' @param Xp A set of 3D points for which PE proximity regions are constructed.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param main An overall title for the plot (default=\code{""}).
#' @param xlab,ylab,zlab titles for the \eqn{x}, \eqn{y}, and \eqn{z} axes, respectively  (default=\code{""} for all).
#' @param xlim,ylim,zlim Two \code{numeric} vectors of length 2, giving the \eqn{x}-, \eqn{y}-, and \eqn{z}-coordinate ranges
#' (default=\code{NULL} for all).
#' @param \dots Additional \code{scatter3D} parameters.
#'
#' @return Plot of the PE proximity regions for points inside the standard regular tetrahedron \eqn{T_h}
#' (and just the points outside \eqn{T_h})
#'
#' @seealso \code{\link{plotPEregsMT}}, \code{\link{plotASregsTri}}, \code{\link{plotASregsMT}},
#' \code{\link{plotCSregsTri}}, and \code{\link{plotCSregsMT}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' \donttest{
#' A<-c(0,0,0); B<-c(1,0,0); C<-c(1/2,sqrt(3)/2,0); D<-c(1/2,sqrt(3)/6,sqrt(6)/3)
#' tetra<-rbind(A,B,C,D)
#' r<-1.5
#'
#' n<-10  #try also n<-20
#' dat<-runif.stdtetra(n)$g  #try also dat[,1]<-dat[,1]+1
#'
#' plotPEregsStdTH(dat[1,],r)
#'
#' plotPEregsStdTH(dat[5,],r)
#'
#' plotPEregsStdTH(c(.4,.4,.4),r)
#'
#' plotPEregsStdTH(c(.5,.5,.5),r)
#'
#' plotPEregsStdTH(dat[1:3,],r)
#'
#' P1<-c(.1,.1,.1)
#' plotPEregsStdTH(rbind(P1,P1),r)
#' }
#'
#' @export plotPEregsStdTH
plotPEregsStdTH <- function(Xp,r,main="",xlab="",ylab="",zlab="",xlim=NULL,ylim=NULL,zlim=NULL, ...)
{
  if (!is.numeric(as.matrix(Xp)) )
  {stop('Xp must be numeric')}

  if (is.point(Xp,dim=length(Xp)))
  { Xp<-matrix(Xp,ncol=3)
  } else
  {Xp<-as.matrix(Xp)
  if (ncol(Xp)!=3 )
  {stop('Xp must be of dimension nx3')}
  }

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  n<-nrow(Xp)

  A<-c(0,0,0); B<-c(1,0,0); C<-c(1/2,sqrt(3)/2,0); D<-c(1/2,sqrt(3)/6,sqrt(6)/3)
  th<-rbind(A,B,C,D)  #standard regular tetrahedron

 in.tetra<-rep(0,n)
  for (i in 1:n)
   in.tetra[i]<-in.tetrahedron(Xp[i,],th,boundary=TRUE)$in.tetra #indices of the Xp points inside the tetrahedron

  Xtetra<-matrix(Xp[in.tetra==1,],ncol=3)  #the Xp points inside the tetrahedron
  nt<-length(Xtetra)/3 #number of Xp points inside the tetrahedron

  if (is.null(xlim))
  {xlim<-range(th[,1],Xp[,1])}
  if (is.null(ylim))
  {ylim<-range(th[,2],Xp[,2])}
  if (is.null(zlim))
  {zlim<-range(th[,3],Xp[,3])}

  xr<-xlim[2]-xlim[1]
  yr<-ylim[2]-ylim[1]
  zr<-zlim[2]-zlim[1]

  plot3D::scatter3D(Xp[,1],Xp[,2],Xp[,3], phi =0,theta=40, bty = "g",xlab=xlab, ylab=ylab, zlab=zlab,
                    xlim=xlim+xr*c(-.05,.05), ylim=ylim+yr*c(-.05,.05), zlim=zlim+zr*c(-.05,.05),
                    pch = 20, cex = 1, ticktype = "detailed", ...)
  #add the vertices of the tetrahedron
  plot3D::points3D(th[,1],th[,2],th[,3], add = TRUE)
  L<-rbind(A,A,A,B,B,C); R<-rbind(B,C,D,C,D,D)
  plot3D::segments3D(L[,1], L[,2], L[,3], R[,1], R[,2],R[,3], add=TRUE,lwd=1,lty=2)
  if (nt>=1)
  {
    for (i in 1:nt)
    {
      P1<-as.vector(Xtetra[i,])
      RV<-rv.tetraCC(P1,th)$rv

      pr<-NPEstdtetra(P1,r,rv=RV)
      L<-rbind(pr[1,],pr[1,],pr[1,],pr[2,],pr[2,],pr[3,]);
      R<-rbind(pr[2,],pr[3,],pr[4,],pr[3,],pr[4,],pr[4,])
      plot3D::segments3D(L[,1], L[,2], L[,3], R[,1], R[,2],R[,3], add=TRUE,lwd=2,col="blue")
    }
  }
} #end of the function
#'

#################################################################

#' @title The vertices of the Proportional Edge (PE) Proximity Region in a tetrahedron
#'
#' @description Returns the vertices of the PE proximity region (which is itself a tetrahedron) for a point in the
#' tetrahedron \code{th}.
#'
#' PE proximity region is defined with respect to the tetrahedron \code{th}
#' with expansion parameter \eqn{r \ge 1} and vertex regions based on the center \code{M} which is circumcenter (\code{"CC"}) or
#' center of mass (\code{"CM"}) of \code{th} with default=\code{"CM"}.
#'
#' Vertex regions are labeled as \code{1,2,3,4} rowwise for the vertices of the tetrahedron \code{th}.
#' \code{rv} is the index of the vertex region \code{pt} resides, with default=\code{NULL}.
#' If \code{pt} is outside of \code{th}, it returns \code{NULL} for the proximity region.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010;textual}{pcds}).
#'
#' @param pt A 3D point whose PE proximity region is to be computed.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param th Four 3D points, stacked row-wise, each row representing a vertex of the tetrahedron.
#' @param M The center to be used in the construction of the vertex regions in the tetrahedron, \code{th}.
#' Currently it only takes \code{"CC"} for circumcenter and \code{"CM"} for center of mass; default=\code{"CM"}.
#' @param rv Index of the vertex region containing the point, either \code{1,2,3,4} (default is \code{NULL}).
#'
#' @return Vertices of the tetrahedron which constitutes the PE proximity region with expansion parameter
#' \code{r} and circumcenter (or center of mass) for a point \code{pt} in the tetrahedron
#'
#' @seealso \code{\link{NPEstdtetra}}, \code{\link{NPEtri}} and \code{\link{NPEint}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(0,0,0); B<-c(1,0,0); C<-c(1/2,sqrt(3)/2,0); D<-c(1/2,sqrt(3)/6,sqrt(6)/3)
#' tetra<-rbind(A,B,C,D)
#' n<-10  #try also n<-20
#'
#' dat<-runif.tetra(n,tetra)$g
#'
#' M<-"CM"  #try also M<-"CC"
#' r<-1.5
#'
#' NPEtetra(dat[1,],r,tetra)  #uses the default M="CM"
#' NPEtetra(dat[1,],r,tetra,M)
#'
#' NPEtetra(dat[5,],r,tetra,M)
#'
#' NPEtetra(c(.4,.4,.4),r,tetra,M)
#' NPEtetra(c(.5,.5,.5),r,tetra,M)
#'
#' #or try
#' RV<-rv.tetraCM(dat[1,],tetra)$rv
#' NPEtetra(dat[1,],r,tetra,M,rv=RV)
#'
#' NPEtetra(c(-1,-1,-1),r,tetra,M,rv=NULL)
#'
#' P1<-c(.1,.1,.1)
#' P2<-c(.5,.5,.5)
#' NPEtetra(P1,r,tetra,M)
#'
#' @export NPEtetra
NPEtetra <- function(pt,r,th,M="CM",rv=NULL)
{
  if (!is.point(pt,3) )
  {stop('pt must be a numeric 3D point')}

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  th<-as.matrix(th)
  if (!is.numeric(th) || nrow(th)!=4 || ncol(th)!=3)
  {stop('th must be numeric and of dimension 4x3')}

  if (length(M) > 1 || sum(M==c("CM","CC"))==0)
    stop("M must be one of \"CC\", \"CM\"")

  if (!in.tetrahedron(pt,th,boundary=TRUE)$in.tetra)
  {reg<-NULL; return(reg); stop}

  Rv<-rv
  if (is.null(Rv))
  {Rv<-ifelse(identical(M,"CC"),rv.tetraCC(pt,th)$rv,rv.tetraCM(pt,th)$rv)  #vertex region for pt
  } else
  {  if (!is.numeric(Rv) || sum(Rv==c(1,2,3,4))!=1)
  {stop('vertex index, rv, must be 1, 2, 3 or 4')}}

  A<-th[Rv,];
  th0<-th[-Rv,]; B<-th0[1,]; C<-th0[2,]; D<-th0[3,]
  d1<-dp2pl(A,B,C,D)$d; d2<-dp2pl(pt,B,C,D)$d;
  rd<-(d1-d2)/d1; #distance ratio
  A1p<-A+rd*(B-A); A1<-A+r*(A1p-A);
  A2p<-A+rd*(C-A); A2<-A+r*(A2p-A);
  A3p<-A+rd*(D-A); A3<-A+r*(A3p-A);
  reg<-rbind(A,A1,A2,A3)

  vec1<-rep(1,4);
  D0<-abs(det(matrix(cbind(th,vec1),ncol=4)))
  D1<-abs(det(matrix(cbind(reg,vec1),ncol=4)))
  if (D1>=D0)
  {reg<-th}
  row.names(reg)<-c()
  reg
} #end of the function
#'

#################################################################

#' @title The indicator for the presence of an arc from one 3D point to another 3D point for
#' Proportional Edge Proximity Catch Digraphs (PE-PCDs)
#'
#' @description Returns \eqn{I(}\code{pt2} is in \eqn{N_{PE}(pt1,r))} for 3D points \code{pt1} and \code{pt2}, that is, returns 1 if \code{pt2} is in \eqn{N_{PE}(pt1,r)},
#' returns 0 otherwise, where N_{PE}(x,r) is the PE proximity region for point \eqn{x} with the expansion parameter \eqn{r \ge 1}.
#'
#' PE proximity region is constructed with respect to the tetrahedron \code{th} and
#' vertex regions are based on the center \code{M} which is circumcenter (\code{"CC"}) or
#' center of mass (\code{"CM"}) of \code{th} with default=\code{"CM"}.
#' \code{rv} is the index of the vertex region \code{pt1} resides, with default=\code{NULL}.
#'
#' If \code{pt1} and \code{pt2} are distinct and either of them are outside \code{th}, it returns 0,
#' but if they are identical, then it returns 1 regardless of their locations (i.e., it allows loops).
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010;textual}{pcds}).
#'
#' @param pt1 A 3D point whose PE proximity region is constructed.
#' @param pt2 A 3D point. The function determines whether \code{pt2} is inside the PE proximity region of
#' \code{pt1} or not.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param th Four 3D points, stacked row-wise, each row representing a vertex of the tetrahedron.
#' @param M The center to be used in the construction of the vertex regions in the tetrahedron, \code{th}.
#' Currently it only takes \code{"CC"} for circumcenter and \code{"CM"} for center of mass; default=\code{"CM"}.
#' @param rv Index of the \code{M}-vertex region containing the point, either \code{1,2,3,4}
#' (default is \code{NULL}).
#'
#' @return \eqn{I(}\code{pt2} is in \eqn{N_{PE}(pt1,r))} for \code{pt1}, that is, returns 1 if \code{pt2} is in \eqn{N_{PE}(pt1,r)}, returns 0 otherwise
#'
#' @seealso \code{\link{IndNPEstdtetra}}, \code{\link{IndNPEtri}} and \code{\link{IndNPEint}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(0,0,0); B<-c(1,0,0); C<-c(1/2,sqrt(3)/2,0); D<-c(1/2,sqrt(3)/6,sqrt(6)/3)
#' tetra<-rbind(A,B,C,D)
#' n<-10  #try also n<-20
#'
#' dat<-runif.tetra(n,tetra)$g
#'
#' M<-"CM"  #try also M<-"CC"
#' r<-1.5
#'
#' IndNPEtetra(dat[1,],dat[2,],r,tetra)  #uses the default M="CM"
#' IndNPEtetra(dat[1,],dat[2,],r,tetra,M)
#'
#' IndNPEtetra(dat[1,],dat[1,],r,tetra,M)
#'
#' IndNPEtetra(c(.4,.4,.4),c(.5,.5,.5),r,tetra,M)
#'
#' #or try
#' RV<-rv.tetraCC(dat[1,],tetra)$rv
#' IndNPEtetra(dat[1,],dat[5,],r,tetra,M,rv=RV)
#'
#' IndNPEtetra(dat[1,],c(-1,-1,-1),r,tetra,M,rv=NULL)
#'
#' IndNPEtetra(c(-1,-1,-1),dat[1,],r,tetra,M,rv=NULL)
#' IndNPEtetra(c(-1,-1,-1),c(-1,-1,-1),r,tetra,M)
#'
#' P1<-c(.1,.1,.1)
#' P2<-c(.5,.5,.5)
#' IndNPEtetra(P1,P2,r,tetra,M)
#'
#' @export IndNPEtetra
IndNPEtetra <- function(pt1,pt2,r,th,M="CM",rv=NULL)
{
  if (!is.point(pt1,3) || !is.point(pt2,3))
  {stop('pt1 and pt2 must be a numeric 3D point')}

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if (identical(pt1,pt2))
  {arc<-1; return(arc); stop}

  th<-as.matrix(th)
  if (!is.numeric(th) || nrow(th)!=4 || ncol(th)!=3)
  {stop('th must be numeric and of dimension 4x3')}

  if (length(M) > 1 || sum(M==c("CM","CC"))==0)
    stop("M must be one of \"CC\", \"CM\"")

  if (!in.tetrahedron(pt1,th,boundary=TRUE)$in.tetra || !in.tetrahedron(pt2,th,boundary=TRUE)$in.tetra)
  {arc<-0
  } else
  {
    if (is.null(rv))
    {rv<-ifelse(identical(M,"CC"),rv.tetraCC(pt1,th)$rv,rv.tetraCM(pt1,th)$rv)  #vertex region for pt1
    } else
    {  if (!is.numeric(rv) || sum(rv==c(1,2,3,4))!=1)
    {stop('vertex index, rv, must be 1, 2, 3 or 4')}}

    pr<-NPEtetra(pt1,r,th,M,rv)
    ifelse(in.tetrahedron(pt2,pr,boundary = T)$i,arc<-1,arc<-0)
  }
  arc
} #end of the function
#'

################################################################

#' @title Incidence matrix for Proportional Edge Proximity Catch Digraphs (PE-PCDs) - one tetrahedron case
#'
#' @description Returns the incidence matrix for the PE-PCD whose vertices are the given 3D numerical data set, \code{dat},
#' in the tetrahedron \eqn{th=T(v=1,v=2,v=3,v=4)}.
#'
#' PE proximity regions are constructed with respect to tetrahedron
#' \code{th} with expansion parameter \eqn{r \ge 1} and vertex regions are based on the center \code{M} which is circumcenter (\code{"CC"})
#' or center of mass (\code{"CM"}) of \code{th} with default=\code{"CM"}.
#' Loops are allowed, so the diagonal entries are all equal to 1.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010;textual}{pcds}).
#'
#' @param dat A set of 3D points which constitute the vertices of PE-PCD.
#' @param th Four 3D points, stacked row-wise, each row representing a vertex of the tetrahedron.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param M The center to be used in the construction of the vertex regions in the tetrahedron, \code{th}.
#' Currently it only takes \code{"CC"} for circumcenter and \code{"CM"} for center of mass; default=\code{"CM"}.
#'
#' @return Incidence matrix for the PE-PCD with vertices being 3D data set, \code{dat},
#' in the tetrahedron \code{th} with vertex regions based on circumcenter or center of mass
#'
#' @seealso \code{\link{IncMatPEtri}}, \code{\link{IncMatPE1D}}, and  \code{\link{IncMatPEMT}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' \donttest{
#' A<-c(0,0,0); B<-c(1,0,0); C<-c(1/2,sqrt(3)/2,0); D<-c(1/2,sqrt(3)/6,sqrt(6)/3)
#' tetra<-rbind(A,B,C,D)
#' n<-10
#'
#' dat<-runif.tetra(n,tetra)$g  #try also dat<-c(.5,.5,.5)
#'
#' M<-"CM"  #try also M<-"CC"
#' r<-1.5
#'
#' IM<-IncMatPEtetra(dat,tetra,r=1.25)  #uses the default M="CM"
#' IM<-IncMatPEtetra(dat,tetra,r=1.25,M)
#' IM
#' dom.greedy(IM)
#' IndUBdom(IM,3)
#' dom.exact(IM)  #this might take a long time for large n
#'
#' IncMatPEtetra(dat,tetra,r=1.5)
#' IncMatPEtetra(dat,tetra,r=2)
#'
#' r<-2
#' IncMatPEtetra(dat,tetra,r,M)
#'
#' dat.fr<-data.frame(a=dat)
#' IncMatPEtetra(dat.fr,tetra,r,M)
#'
#' dat.fr<-data.frame(a=tetra)
#' IncMatPEtetra(dat,dat.fr,r,M)
#' }
#'
#' @export IncMatPEtetra
IncMatPEtetra <- function(dat,th,r,M="CM")
{
  if (!is.numeric(as.matrix(dat)))
  {stop('dat must be numeric')}

  if (is.point(dat,dim=length(dat)))
  { dat<-matrix(dat,ncol=3)
  } else
  {dat<-as.matrix(dat)
  if (ncol(dat)!=3 )
  {stop('dat must be of dimension nx3')}
  }

  th<-as.matrix(th)
  if (!is.numeric(th) || nrow(th)!=4 || ncol(th)!=3)
  {stop('th must be numeric and of dimension 4x3')}

  vec1<-rep(1,4);
  D0<-det(matrix(cbind(th,vec1),ncol=4))
  if (round(D0,14)==0)
  {stop('the tetrahedron is degenerate')}

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if (length(M) > 1 || sum(M==c("CM","CC"))==0)
    stop("M must be one of \"CC\", \"CM\"")

  n<-nrow(dat)

 inc.mat<-matrix(0, nrow=n, ncol=n)
  if (n>1)
  {
    for (i in 1:n)
    {pt1<-dat[i,]
    v<-ifelse(identical(M,"CC"),rv.tetraCC(pt1,th)$rv,rv.tetraCM(pt1,th)$rv)  #vertex region for pt1
    for (j in ((1:n)) )
    {pt2<-dat[j,]
   inc.mat[i,j]<-IndNPEtetra(pt1,pt2,r,th,M,rv=v)
    }
    }
  }
  diag(inc.mat)<-1
 inc.mat
} #end of the function
#'

#################################################################

#' @title The plot of the Proportional Edge (PE) Proximity Regions for a 3D data set - one tetrahedron case
#'
#' @description Plots the points in and outside of the tetrahedron \code{th} and also the PE proximity regions  (which are also
#' tetrahedrons) for points inside the tetrahedron \code{th}.
#'
#' PE proximity regions are constructed with respect to
#' tetrahedron \code{th} with expansion parameter \eqn{r \ge 1} and vertex regions are based on the center \code{M} which is
#' circumcenter (\code{"CC"}) or center of mass (\code{"CM"}) of \code{th} with default=\code{"CM"}, so PE proximity regions are defined
#' only for points inside the tetrahedron \code{th}.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010;textual}{pcds}).
#'
#' @param Xp A set of 3D points for which PE proximity regions are constructed.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param th Four 3D points, stacked row-wise, each row representing a vertex of the tetrahedron.
#' @param M The center to be used in the construction of the vertex regions in the tetrahedron, \code{th}.
#' Currently it only takes \code{"CC"} for circumcenter and \code{"CM"} for center of mass; default=\code{"CM"}.
#' @param main An overall title for the plot (default=\code{""}).
#' @param xlab,ylab,zlab Titles for the \eqn{x}, \eqn{y}, and \eqn{z} axes, respectively  (default=\code{""} for all).
#' @param xlim,ylim,zlim Two \code{numeric} vectors of length 2, giving the \eqn{x}-, \eqn{y}-, and \eqn{z}-coordinate ranges
#' (default=\code{NULL} for all).
#' @param \dots Additional \code{scatter3D} parameters.
#'
#' @return Plot of the PE proximity regions for points inside the tetrahedron \code{th}
#' (and just the points outside \code{th})
#'
#' @seealso \code{\link{plotPEregsStdTH}}, \code{\link{plotPEregsTri}} and \code{\link{plotPEregsInt}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' \donttest{
#' A<-c(0,0,0); B<-c(1,0,0); C<-c(1/2,sqrt(3)/2,0); D<-c(1/2,sqrt(3)/6,sqrt(6)/3)
#' tetra<-rbind(A,B,C,D)
#' n<-10  #try also n<-20
#'
#' dat<-runif.tetra(n,tetra)$g  #try also dat[,1]<-dat[,1]+1
#'
#' M<-"CM"  #try also M<-"CC"
#' r<-1.5
#'
#' plotPEregsTH(dat[1,],r,tetra)  #uses the default M="CM"
#' plotPEregsTH(dat[1,],r,tetra,M)
#'
#' plotPEregsTH(dat[5,],r,tetra,M)
#'
#' plotPEregsTH(c(.4,.4,.4),r,tetra,M)
#'
#' plotPEregsTH(c(.5,.5,.5),r,tetra,M)
#'
#' plotPEregsTH(dat[1:3,],r,tetra,M)
#'
#' P1<-c(.1,.1,.1)
#' plotPEregsTH(rbind(P1,P1),r,tetra,M)
#' }
#'
#' @export plotPEregsTH
plotPEregsTH <- function(Xp,r,th,M="CM",main="",xlab="",ylab="",zlab="",xlim=NULL,ylim=NULL,zlim=NULL, ...)
{
  if (!is.numeric(as.matrix(Xp)) )
  {stop('Xp must be numeric')}

  if (is.point(Xp,dim=length(Xp)))
  { Xp<-matrix(Xp,ncol=3)
  } else
  {Xp<-as.matrix(Xp)
  if (ncol(Xp)!=3 )
  {stop('Xp must be of dimension nx3')}
  }

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  th<-as.matrix(th)
  if (!is.numeric(th) || nrow(th)!=4 || ncol(th)!=3)
  {stop('th must be numeric and of dimension 4x3')}

  vec1<-rep(1,4);
  D0<-det(matrix(cbind(th,vec1),ncol=4))
  if (round(D0,14)==0)
  {stop('the tetrahedron is degenerate')}

  if (length(M) > 1 || sum(M==c("CM","CC"))==0)
    stop("M must be one of \"CC\", \"CM\"")

  n<-nrow(Xp)

 in.tetra<-rep(0,n)
  for (i in 1:n)
   in.tetra[i]<-in.tetrahedron(Xp[i,],th,boundary=TRUE)$in.tetra #indices of the Xp points inside the tetrahedron

  Xtetra<-matrix(Xp[in.tetra==1,],ncol=3)  #the Xp points inside the tetrahedron
  nt<-length(Xtetra)/3 #number of Xp points inside the tetrahedron

  if (is.null(xlim))
  {xlim<-range(th[,1],Xp[,1])}
  if (is.null(ylim))
  {ylim<-range(th[,2],Xp[,2])}
  if (is.null(zlim))
  {zlim<-range(th[,3],Xp[,3])}

  xr<-xlim[2]-xlim[1]
  yr<-ylim[2]-ylim[1]
  zr<-zlim[2]-zlim[1]

  plot3D::scatter3D(Xp[,1],Xp[,2],Xp[,3], phi =0,theta=40, bty = "g",xlab=xlab, ylab=ylab, zlab=zlab,
                    xlim=xlim+xr*c(-.05,.05), ylim=ylim+yr*c(-.05,.05), zlim=zlim+zr*c(-.05,.05),
                    pch = 20, cex = 1, ticktype = "detailed", ...)
  #add the vertices of the tetrahedron
  plot3D::points3D(th[,1],th[,2],th[,3], add = TRUE)
  A<-th[1,]; B<-th[2,]; C<-th[3,]; D<-th[4,]
  L<-rbind(A,A,A,B,B,C); R<-rbind(B,C,D,C,D,D)
  plot3D::segments3D(L[,1], L[,2], L[,3], R[,1], R[,2],R[,3], add=TRUE,lwd=1,lty=2)
  if (nt>=1)
  {
    for (i in 1:nt)
    {
      P1<-as.vector(Xtetra[i,])
      RV<-ifelse(identical(M,"CC"),rv.tetraCC(P1,th)$rv,rv.tetraCM(P1,th)$rv)
      pr<-NPEtetra(P1,r,th,M,rv=RV)
      L<-rbind(pr[1,],pr[1,],pr[1,],pr[2,],pr[2,],pr[3,]);
      R<-rbind(pr[2,],pr[3,],pr[4,],pr[3,],pr[4,],pr[4,])
      plot3D::segments3D(L[,1], L[,2], L[,3], R[,1], R[,2],R[,3], add=TRUE,lwd=2,col="blue")
    }
  }
} #end of the function
#'

#################################################################

#' @title The indicator for a 3D point being a dominating point for Proportional Edge Proximity Catch Digraphs
#' (PE-PCDs) - standard regular tetrahedron case
#'
#' @description Returns \eqn{I(}\code{p} is a dominating point of the PE-PCD\eqn{)} where the vertices of the PE-PCD are the 3D data set \code{Dt} in the
#' standard regular tetrahedron \eqn{T_h=T((0,0,0),(1,0,0),(1/2,\sqrt{3}/2,0),(1/2,\sqrt{3}/6,\sqrt{6}/3))}, that is,
#' returns 1 if \code{p} is a dominating point of PE-PCD, returns 0 otherwise.
#'
#' Point, \code{p}, is in the vertex region of vertex \code{rv} (default is \code{NULL}); vertices are labeled as \code{1,2,3,4}
#' in the order they are stacked row-wise in \eqn{T_h}.
#'
#' PE proximity region is constructed with respect to the tetrahedron \eqn{T_h} with expansion parameter \eqn{r \ge 1}
#' and vertex regions are based on center of mass \eqn{CM} (equivalent to circumcenter in this case).
#'
#' \code{ch.data.pnt} is for checking whether point \code{p} is a data point in \code{Dt} or not (default is \code{FALSE}),
#' so by default this function checks whether the point \code{p} would be a dominating point
#' if it actually were in the data set.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010;textual}{pcds}).
#'
#' @param p A 3D point that is to be tested for being a dominating point or not of the PE-PCD.
#' @param Dt A set of 3D points which constitutes the vertices of the PE-PCD.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param rv Index of the vertex whose region contains point \code{p}, \code{rv} takes the vertex labels
#' as \code{1,2,3,4} as in the row order of the vertices in standard regular tetrahedron, default is \code{NULL}.
#' @param ch.data.pnt A logical argument for checking whether point \code{p} is a data point
#' in \code{Dt} or not (default is \code{FALSE}).
#'
#' @return \eqn{I(}\code{p} is a dominating point of the PE-PCD\eqn{)} where the vertices of the PE-PCD are the 3D data set \code{Dt},
#' that is, returns 1 if \code{p} is a dominating point, returns 0 otherwise
#'
#' @seealso \code{\link{Gam1PEtetra}}, \code{\link{Gam1PEtri}} and \code{\link{Gam1PEbastri}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' set.seed(123)
#' A<-c(0,0,0); B<-c(1,0,0); C<-c(1/2,sqrt(3)/2,0); D<-c(1/2,sqrt(3)/6,sqrt(6)/3)
#' tetra<-rbind(A,B,C,D)
#' #'
#' n<-20
#' dat<-runif.stdtetra(n)$g  #try also dat<-cbind(runif(n),runif(n),runif(n))
#' r<-1.5
#' #'
#' P<-c(.4,.1,.2)
#' Gam1PEstdTetra(P,P,r)
#' Gam1PEstdTetra(dat[1,],dat,r)
#' Gam1PEstdTetra(P,dat,r)
#' #'
#' Gam1PEstdTetra(dat[1,],dat,r)
#' Gam1PEstdTetra(dat[1,],dat[1,],r)
#' #'
#' #or try
#' RV<-rv.tetraCC(dat[1,],tetra)$rv
#' Gam1PEstdTetra(dat[1,],dat,r,rv=RV)
#' #'
#' Gam1PEstdTetra(c(-1,-1,-1),dat,r)
#' Gam1PEstdTetra(c(-1,-1,-1),c(-1,-1,-1),r)
#' #'
#' gam.vec<-vector()
#' for (i in 1:n)
#' {gam.vec<-c(gam.vec,Gam1PEstdTetra(dat[i,],dat,r))}
#' #'
#' ind.gam1<-which(gam.vec==1)
#' ind.gam1
#' g1.pts<-dat[ind.gam1,]
#' #'
#' Xlim<-range(tetra[,1],dat[,1])
#' Ylim<-range(tetra[,2],dat[,2])
#' Zlim<-range(tetra[,3],dat[,3])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#' zd<-Zlim[2]-Zlim[1]
#' #'
#' plot3D::scatter3D(dat[,1],dat[,2],dat[,3], phi =0,theta=40, bty = "g",
#' xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05), zlim=Zlim+zd*c(-.05,.05),
#'          pch = 20, cex = 1, ticktype = "detailed")
#' #add the vertices of the tetrahedron
#' plot3D::points3D(tetra[,1],tetra[,2],tetra[,3], add=TRUE)
#' L<-rbind(A,A,A,B,B,C); R<-rbind(B,C,D,C,D,D)
#' plot3D::segments3D(L[,1], L[,2], L[,3], R[,1], R[,2],R[,3], add=TRUE,lwd=2)
#' if (length(g1.pts)!=0)
#' {
#'   if (length(g1.pts)==3) g1.pts<-matrix(g1.pts,nrow=1)
#'   plot3D::points3D(g1.pts[,1],g1.pts[,2],g1.pts[,3], pch=4,col="red", add=TRUE)}
#' #'
#' plot3D::text3D(tetra[,1],tetra[,2],tetra[,3], labels=c("A","B","C","D"), add=TRUE)
#' #'
#' CM<-apply(tetra,2,mean)
#' D1<-(A+B)/2; D2<-(A+C)/2; D3<-(A+D)/2; D4<-(B+C)/2; D5<-(B+D)/2; D6<-(C+D)/2;
#' L<-rbind(D1,D2,D3,D4,D5,D6); R<-matrix(rep(CM,6),ncol=3,byrow=TRUE)
#' plot3D::segments3D(L[,1], L[,2], L[,3], R[,1], R[,2],R[,3], add=TRUE,lty=2)
#' #'
#' P<-c(.4,.1,.2)
#' Gam1PEstdTetra(P,dat,r)
#' #'
#' dat.fr<-data.frame(a=dat)
#' Gam1PEstdTetra(P,dat.fr,r)
#' #'
#' Gam1PEstdTetra(c(-1,-1,-1),dat,r,ch.data.pnt = FALSE)
#' #gives an error message if ch.data.pnt = TRUE
#'
#' @export
Gam1PEstdTetra <- function(p,Dt,r,rv=NULL,ch.data.pnt=FALSE)
{
  if (!is.point(p,3))
  {stop('p must be a numeric 3D point')}

  if (!is.numeric(as.matrix(Dt)))
  {stop('Dt must be numeric')}

  if (is.point(Dt,3))
  { Dt<-matrix(Dt,ncol=3)
  } else
  {Dt<-as.matrix(Dt)
  if (ncol(Dt)!=3 )
  {stop('Dt must be of dimension nx3')}
  }

  if (ch.data.pnt==TRUE)
  {
    if (!is.in.data(p,Dt))
    {stop('point, p, is not a data point in Dt')}
  }

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if (identical(matrix(p,ncol=3),matrix(Dt,ncol=3)))
  {dom<-1;return(dom);stop}

  A<-c(0,0,0); B<-c(1,0,0); C<-c(1/2,sqrt(3)/2,0); D<-c(1/2,sqrt(3)/6,sqrt(6)/3)
  th<-rbind(A,B,C,D)  #standard regular tetrahedron
  if (in.tetrahedron(p,th)$in.tetra==F)
  {dom<-0; return(dom); stop}

  if (is.null(rv))
  {rv<-rv.tetraCC(p,th)$rv #vertex region for p
  } else
  {  if (!is.numeric(rv) || sum(rv==c(1,2,3,4))!=1)
  {stop('vertex index, rv, must be 1, 2, 3 or 4')}}

  Dt<-matrix(Dt,ncol=3)
  n<-nrow(Dt)
  dom<-1; i<-1;
  while (i <= n & dom==1)
  {if (IndNPEstdtetra(p,Dt[i,],r,rv=rv)==0)
    dom<-0;
  i<-i+1;
  }
  dom
} #end of the function
#'

#################################################################

#' @title The indicator for two 3D points constituting a dominating set for Proportional Edge Proximity Catch Digraphs
#' (PE-PCDs) - standard regular tetrahedron case
#'
#' @description Returns \eqn{I(}\{\code{pt1,pt2}\} is a dominating set of the PE-PCD\eqn{)} where the vertices of the PE-PCD are the 3D data set \code{Dt} in the standard regular
#' tetrahedron \eqn{T_h=T((0,0,0),(1,0,0),(1/2,\sqrt{3}/2,0),(1/2,\sqrt{3}/6,\sqrt{6}/3))}, that is,
#' returns 1 if \{\code{pt1,pt2}\} is a dominating set of PE-PCD, returns 0 otherwise.
#'
#' Point, \code{pt1}, is in the region of vertex \code{rv1} (default is \code{NULL}) and point, \code{pt2}, is in the region of vertex \code{rv2}
#' (default is \code{NULL}); vertices (and hence \code{rv1} and \code{rv2}) are labeled as \code{1,2,3,4} in the order they are stacked
#' row-wise in \eqn{T_h}.
#'
#' PE proximity region is constructed with respect to the tetrahedron \eqn{T_h} with expansion parameter \eqn{r \ge 1}
#' and vertex regions are based on center of mass \eqn{CM} (equivalent to circumcenter in this case).
#'
#' \code{ch.data.pnts} is for checking whether points \code{pt1} and \code{pt2} are data points in \code{Dt} or not (default is \code{FALSE}),
#' so by default this function checks whether the points \code{pt1} and \code{pt2} would constitute a dominating set
#' if they actually were both in the data set.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010;textual}{pcds}).
#'
#' @param pt1,pt2 Two 3D points to be tested for constituting a dominating set of the PE-PCD.
#' @param Dt A set of 3D points which constitutes the vertices of the PE-PCD.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param rv1,rv2 The indices of the vertices whose regions contains \code{pt1} and \code{pt2}, respectively.
#' They take the vertex labels as \code{1,2,3,4} as in the row order of the vertices in \eqn{T_h}
#' (default is \code{NULL} for both).
#' @param ch.data.pnts A logical argument for checking whether points \code{pt1} and \code{pt2} are
#' data points in \code{Dt} or not (default is \code{FALSE}).
#'
#' @return \eqn{I(}\{\code{pt1,pt2}\} is a dominating set of the PE-PCD\eqn{)} where the vertices of the PE-PCD are the 3D data set \code{Dt}),
#' that is, returns 1 if \{\code{pt1,pt2}\} is a dominating set of PE-PCD, returns 0 otherwise
#'
#' @seealso \code{\link{Gam2PEtetra}}, \code{\link{Gam2PEtri}} and \code{\link{Gam2PEbastri}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(0,0,0); B<-c(1,0,0); C<-c(1/2,sqrt(3)/2,0); D<-c(1/2,sqrt(3)/6,sqrt(6)/3)
#' tetra<-rbind(A,B,C,D)
#'
#' n<-20
#' dat<-runif.stdtetra(n)$g  #try also dat<-cbind(runif(n),runif(n),runif(n))
#' r<-1.5
#'
#' Gam2PEstdTetra(dat[1,],dat[2,],dat,r)
#' Gam2PEstdTetra(dat[1,],dat[1,],dat,r)
#'
#' Gam2PEstdTetra(c(-1,-1,-1),dat[2,],dat,r)
#'
#' Gam2PEstdTetra(c(-1,-1,-1),c(-1,-1,-2),rbind(c(-1,-1,-1),c(-1,-1,-2)),r)
#'
#' ind.gam2<-vector()
#' for (i in 1:(n-1))
#'  for (j in (i+1):n)
#'  {if (Gam2PEstdTetra(dat[i,],dat[j,],dat,r)==1)
#'   ind.gam2<-rbind(ind.gam2,c(i,j))}
#'
#' ind.gam2
#'
#' #or try
#' rv1<-rv.tetraCC(dat[1,],tetra)$rv;rv2<-rv.tetraCC(dat[2,],tetra)$rv
#' Gam2PEstdTetra(dat[1,],dat[2,],dat,r,rv1,rv2)
#'
#' #or try
#' rv1<-rv.tetraCC(dat[1,],tetra)$rv;
#' Gam2PEstdTetra(dat[1,],dat[2,],dat,r,rv1)
#'
#' #or try
#' rv2<-rv.tetraCC(dat[2,],tetra)$rv
#' Gam2PEstdTetra(dat[1,],dat[2,],dat,r,rv2=rv2)
#'
#' P1<-c(.1,.1,.1)
#' P2<-c(.4,.1,.2)
#' Gam2PEstdTetra(P1,P2,dat,r)
#'
#' Gam2PEstdTetra(dat[1,],dat[2,],dat,r)
#'
#' dat.fr<-data.frame(a=dat)
#' Gam2PEstdTetra(P1,P2,dat.fr,r)
#'
#' Gam2PEstdTetra(c(-1,-1,-1),dat[2,],dat,r,ch.data.pnts = FALSE)
#' #gives an error message if ch.data.pnts = TRUE
#' #since not both points, pt1 and pt2, are data points in Dt
#'
#' @export
Gam2PEstdTetra <- function(pt1,pt2,Dt,r,rv1=NULL,rv2=NULL,ch.data.pnts=FALSE)
{
  if (!is.point(pt1,3) || !is.point(pt2,3))
  {stop('pt1 and pt2 must be a numeric 3D point')}

  if (!is.numeric(as.matrix(Dt)))
  {stop('third argument must be numeric')}

  if (is.point(Dt,3))
  { Dt<-matrix(Dt,ncol=3)
  } else
  {Dt<-as.matrix(Dt)
  if (ncol(Dt)!=3 )
  {stop('Dt must be of dimension nx3')}
  }

  if (ch.data.pnts==TRUE)
  {
    if (!is.in.data(pt1,Dt) || !is.in.data(pt2,Dt))
    {stop('not both points are data points in Dt')}
  }

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  if (identical(pt1,pt2))
  {dom<-0; return(dom); stop}

  A<-c(0,0,0); B<-c(1,0,0); C<-c(1/2,sqrt(3)/2,0); D<-c(1/2,sqrt(3)/6,sqrt(6)/3)
  th<-rbind(A,B,C,D)  #standard regular tetrahedron

  if (is.null(rv1))
  {rv1<-rv.tetraCC(pt1,th)$rv #vertex region for point pt1
  }
  if (is.null(rv2))
  {rv2<-rv.tetraCC(pt2,th)$rv  #vertex region for point pt2
  }
  Dt<-matrix(Dt,ncol=3)
  n<-nrow(Dt)
  dom<-1; i<-1;
  while (i <= n & dom==1)
  {if (max(IndNPEstdtetra(pt1,Dt[i,],r,rv1),IndNPEstdtetra(pt2,Dt[i,],r,rv2))==0)
    dom<-0;
  i<-i+1;
  }
  dom
} #end of the function
#'

#################################################################

#' @title The indicator for three 3D points constituting a dominating set for Proportional Edge Proximity Catch Digraphs
#' (PE-PCDs) - standard regular tetrahedron case
#'
#' @description Returns \eqn{I(\{}\code{pt1,pt2,pt3}\} is a dominating set of the PE-PCD\eqn{)} where the vertices of the PE-PCD are the 3D data set \code{Dt} in the standard regular
#' tetrahedron \eqn{T_h=T((0,0,0),(1,0,0),(1/2,\sqrt{3}/2,0),(1/2,\sqrt{3}/6,\sqrt{6}/3))}, that is,
#' returns 1 if \{\code{pt1,pt2,pt3}\} is a dominating set of PE-PCD, returns 0 otherwise.
#'
#' Point, \code{pt1}, is in the region of vertex \code{rv1} (default is \code{NULL}), point, \code{pt2}, is in the region of vertex \code{rv2}
#' (default is \code{NULL}); point, \code{pt3}), is in the region of vertex \code{rv3}) (default is \code{NULL}); vertices (and hence \code{rv1, rv2} and
#' \code{rv3}) are labeled as \code{1,2,3,4} in the order they are stacked row-wise in \eqn{T_h}.
#'
#' PE proximity region is constructed with
#' respect to the tetrahedron \eqn{T_h} with expansion parameter \eqn{r \ge 1}
#' and vertex regions are based on center of mass \eqn{CM} (equivalent to circumcenter in this case).
#'
#'  \code{ch.data.pnts} is for checking whether points \code{pt1}, \code{pt2} and \code{pt3} are all data points in \code{Dt} or not (default is \code{FALSE}),
#' so by default this function checks whether the points \code{pt1}, \code{pt2} and \code{pt3} would constitute a dominating set
#' if they actually were all in the data set.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010;textual}{pcds}).
#'
#' @param pt1,pt2,pt3 Three 3D points to be tested for constituting a dominating set of the PE-PCD.
#' @param Dt A set of 3D points which constitutes the vertices of the PE-PCD.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param rv1,rv2,rv3 The indices of the vertices whose regions contains \code{pt1}, \code{pt2} and \code{pt3},
#' respectively.
#' They take the vertex labels as \code{1,2,3,4} as in the row order of the vertices in \eqn{T_h}
#' (default is \code{NULL} for all).
#' @param ch.data.pnts A logical argument for checking whether points \code{pt1} and \code{pt2} are
#' data points in \code{Dt} or not (default is \code{FALSE}).
#'
#' @return \eqn{I(\{}\code{pt1,pt2,pt3}\} is a dominating set of the PE-PCD\eqn{)} where the vertices of the PE-PCD are the 3D data set \code{Dt}),
#' that is, returns 1 if \{\code{pt1,pt2,pt3}\} is a dominating set of PE-PCD, returns 0 otherwise
#'
#' @seealso \code{\link{Gam3PEtetra}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' set.seed(123)
#' A<-c(0,0,0); B<-c(1,0,0); C<-c(1/2,sqrt(3)/2,0); D<-c(1/2,sqrt(3)/6,sqrt(6)/3)
#' tetra<-rbind(A,B,C,D)
#' n<-10 #try 20, 40, 100 (larger n may take a long time)
#' dat<-runif.stdtetra(n)$g  #try also dat<-cbind(runif(n),runif(n),runif(n))
#' r<-1.25
#'
#' Gam3PEstdTetra(dat[1,],dat[2,],dat[3,],dat,r)
#'
#' Gam3PEstdTetra(dat[1,],dat[2,],dat[2,],dat,r)
#'
#' Gam3PEstdTetra(dat[1,],c(1,1,1),dat[3,],dat,r)
#'
#' Gam3PEstdTetra(c(-1,1,1),c(1,1,1),c(1,1,-1),rbind(c(-1,1,1),c(1,1,1),c(1,1,-1)),r)
#'
#' ind.gam3<-vector()
#' for (i in 1:(n-2))
#'  for (j in (i+1):(n-1))
#'    for (k in (j+1):n)
#'  {if (Gam3PEstdTetra(dat[i,],dat[j,],dat[k,],dat,r)==1)
#'   ind.gam3<-rbind(ind.gam3,c(i,j,k))}
#'
#' ind.gam3
#'
#' #or try
#' rv1<-rv.tetraCC(dat[1,],tetra)$rv; rv2<-rv.tetraCC(dat[2,],tetra)$rv;
#' rv3<-rv.tetraCC(dat[3,],tetra)$rv
#' Gam3PEstdTetra(dat[1,],dat[2,],dat[3,],dat,r,rv1,rv2,rv3)
#'
#' #or try
#' rv1<-rv.tetraCC(dat[1,],tetra)$rv;
#' Gam3PEstdTetra(dat[1,],dat[2,],dat[3,],dat,r,rv1)
#'
#' #or try
#' rv2<-rv.tetraCC(dat[2,],tetra)$rv
#' Gam3PEstdTetra(dat[1,],dat[2,],dat[3,],dat,r,rv2=rv2)
#'
#' P1<-c(.1,.1,.1)
#' P2<-c(.3,.3,.3)
#' P3<-c(.4,.1,.2)
#' Gam3PEstdTetra(P1,P2,P3,dat,r)
#'
#' dat.fr<-data.frame(a=dat)
#' Gam3PEstdTetra(P1,P2,P3,dat.fr,r)
#'
#' Gam3PEstdTetra(dat[1,],c(1,1,1),dat[3,],dat,r,ch.data.pnts = FALSE)
#' #gives an error message if ch.data.pnts = TRUE since not all points are data points in Dt
#'
#' @export
Gam3PEstdTetra <- function(pt1,pt2,pt3,Dt,r,rv1=NULL,rv2=NULL,rv3=NULL,ch.data.pnts=FALSE)
{
  if (!is.point(pt1,3) || !is.point(pt2,3) || !is.point(pt3,3))
  {stop('pt1, pt2, and pt3 must be a numeric 3D points')}

  if (!is.numeric(as.matrix(Dt)))
  {stop('Dt must be numeric')}

  if (is.point(Dt,3))
  { Dt<-matrix(Dt,ncol=3)
  } else
  {Dt<-as.matrix(Dt)
  if (ncol(Dt)!=3 )
  {stop('Dt must be of dimension nx3')}
  }

  if (ch.data.pnts==TRUE)
  {
    if (!is.in.data(pt1,Dt) || !is.in.data(pt2,Dt) || !is.in.data(pt3,Dt))
    {stop('not all points are data points in Dt')}
  }

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}


  if (identical(pt1,pt2) || identical(pt1,pt3) || identical(pt2,pt3))
  {dom<-0; return(dom); stop}

  A<-c(0,0,0); B<-c(1,0,0); C<-c(1/2,sqrt(3)/2,0); D<-c(1/2,sqrt(3)/6,sqrt(6)/3)
  th<-rbind(A,B,C,D)  #standard regular tetrahedron

  if (is.null(rv1))
  {rv1<-rv.tetraCC(pt1,th)$rv #vertex region for point pt1
  }
  if (is.null(rv2))
  {rv2<-rv.tetraCC(pt2,th)$rv #vertex region for point pt2
  }
  if (is.null(rv3))
  {rv3<-rv.tetraCC(pt3,th)$rv  #vertex region for point pt3
  }

  Dt<-matrix(Dt,ncol=3)
  n<-nrow(Dt)
  dom<-1; i<-1;
  while (i <= n & dom==1)
  {if (max(IndNPEstdtetra(pt1,Dt[i,],r,rv1),IndNPEstdtetra(pt2,Dt[i,],r,rv2),IndNPEstdtetra(pt3,Dt[i,],r,rv3))==0)
    dom<-0;
  i<-i+1;
  }
  dom
} #end of the function
#'

#################################################################

#' @title The indicator for a 3D point being a dominating point for Proportional Edge Proximity Catch Digraphs (PE-PCDs)
#' - one tetrahedron case
#'
#' @description Returns \eqn{I(}\code{p} is a dominating point of the PE-PCD\eqn{)} where the vertices of the PE-PCD are the 2D data set \code{Dt} in the tetrahedron \code{th}, that is,
#' returns 1 if \code{p} is a dominating point of PE-PCD, returns 0 otherwise.
#'
#' Point, \code{p}, is in the vertex region of vertex \code{rv} (default is \code{NULL}); vertices are labeled as \code{1,2,3,4}
#' in the order they are stacked row-wise in \code{th}.
#'
#' PE proximity region is constructed with respect to the tetrahedron \code{th} with expansion parameter \eqn{r \ge 1}
#' and vertex regions are based on center of mass (\code{M="CM"}) or circumcenter (\code{M="CC"}) only.
#' and vertex regions are based on center of mass \eqn{CM} (equivalent to circumcenter in this case).
#'
#' \code{ch.data.pnt} is for checking whether point \code{p} is a data point in \code{Dt} or not (default is \code{FALSE}),
#' so by default this function checks whether the point \code{p} would be a dominating point
#' if it actually were in the data set.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010;textual}{pcds}).
#'
#' @param p A 3D point that is to be tested for being a dominating point or not of the PE-PCD.
#' @param Dt A set of 3D points which constitutes the vertices of the PE-PCD.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param th Four 3D points, stacked row-wise, each row representing a vertex of the tetrahedron.
#' @param M The center to be used in the construction of the vertex regions in the tetrahedron, \code{th}.
#' Currently it only takes \code{"CC"} for circumcenter and \code{"CM"} for center of mass; default=\code{"CM"}.
#' @param rv Index of the vertex whose region contains point \code{p}, \code{rv} takes the vertex labels as \code{1,2,3,4} as
#' in the row order of the vertices in standard tetrahedron, default is \code{NULL}.
#' @param ch.data.pnt A logical argument for checking whether point \code{p} is a data point
#' in \code{Dt} or not (default is \code{FALSE}).
#'
#' @return \eqn{I(}\code{p} is a dominating point of the PE-PCD\eqn{)} where the vertices of the PE-PCD are the 2D data set \code{Dt},
#' that is, returns 1 if \code{p} is a dominating point, returns 0 otherwise
#'
#' @seealso \code{\link{Gam1PEstdTetra}}, \code{\link{Gam1PEtri}} and \code{\link{Gam1PEbastri}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(0,0,0); B<-c(1,0,0); C<-c(1/2,sqrt(3)/2,0); D<-c(1/2,sqrt(3)/6,sqrt(6)/3)
#' tetra<-rbind(A,B,C,D)
#' n<-20
#'
#' dat<-runif.tetra(n,tetra)$g  #try also dat<-cbind(runif(n),runif(n),runif(n))
#'
#' M<-"CM"; cent<-apply(tetra,2,mean)  #center of mass
#' #try also M<-"CC"; cent<-circ.cent.tetra(tetra)  #circumcenter
#'
#' r<-2
#'
#' P<-c(.4,.1,.2)
#' Gam1PEtetra(P,P,r,tetra,M)
#' Gam1PEtetra(dat[1,],dat,r,tetra,M)
#' Gam1PEtetra(P,dat,r,tetra,M)
#'
#' Gam1PEtetra(dat[1,],dat,r,tetra,M)
#' Gam1PEtetra(dat[1,],dat[1,],r,tetra,M)
#'
#' #or try
#' RV<-rv.tetraCC(dat[1,],tetra)$rv
#' Gam1PEtetra(dat[1,],dat,r,tetra,M,rv=RV)
#'
#' Gam1PEtetra(c(-1,-1,-1),dat,r,tetra,M)
#' Gam1PEtetra(c(-1,-1,-1),c(-1,-1,-1),r,tetra,M)
#'
#' gam.vec<-vector()
#' for (i in 1:n)
#' {gam.vec<-c(gam.vec,Gam1PEtetra(dat[i,],dat,r,tetra,M))}
#'
#' ind.gam1<-which(gam.vec==1)
#' ind.gam1
#' g1.pts<-dat[ind.gam1,]
#'
#' Xlim<-range(tetra[,1],dat[,1],cent[1])
#' Ylim<-range(tetra[,2],dat[,2],cent[2])
#' Zlim<-range(tetra[,3],dat[,3],cent[3])
#' xd<-Xlim[2]-Xlim[1]
#' yd<-Ylim[2]-Ylim[1]
#' zd<-Zlim[2]-Zlim[1]
#'
#' plot3D::scatter3D(dat[,1],dat[,2],dat[,3], phi =0,theta=40, bty = "g",
#' xlim=Xlim+xd*c(-.05,.05),ylim=Ylim+yd*c(-.05,.05), zlim=Zlim+zd*c(-.05,.05),
#'          pch = 20, cex = 1, ticktype = "detailed")
#' #add the vertices of the tetrahedron
#' plot3D::points3D(tetra[,1],tetra[,2],tetra[,3], add=TRUE)
#' L<-rbind(A,A,A,B,B,C); R<-rbind(B,C,D,C,D,D)
#' plot3D::segments3D(L[,1], L[,2], L[,3], R[,1], R[,2],R[,3], add=TRUE,lwd=2)
#' if (length(g1.pts)!=0)
#' {plot3D::points3D(g1.pts[,1],g1.pts[,2],g1.pts[,3], pch=4,col="red", add=TRUE)}
#'
#' plot3D::text3D(tetra[,1],tetra[,2],tetra[,3], labels=c("A","B","C","D"), add=TRUE)
#'
#' D1<-(A+B)/2; D2<-(A+C)/2; D3<-(A+D)/2; D4<-(B+C)/2; D5<-(B+D)/2; D6<-(C+D)/2;
#' L<-rbind(D1,D2,D3,D4,D5,D6); R<-rbind(cent,cent,cent,cent,cent,cent)
#' plot3D::segments3D(L[,1], L[,2], L[,3], R[,1], R[,2],R[,3], add=TRUE,lty=2)
#'
#' P<-c(.4,.1,.2)
#' Gam1PEtetra(P,dat,r,tetra,M)
#'
#' dat.fr<-data.frame(a=dat)
#' Gam1PEtetra(P,dat.fr,r,tetra,M)
#'
#' Gam1PEtetra(c(-1,-1,-1),dat,r,tetra,M,ch.data.pnt = FALSE)
#' #gives an error message if ch.data.pnt = TRUE since p is not a data point
#'
#' @export
Gam1PEtetra <- function(p,Dt,r,th,M="CM",rv=NULL,ch.data.pnt=FALSE)
{
  if (!is.point(p,3))
  {stop('p must be a numeric 3D point')}

  if (!is.numeric(as.matrix(Dt)))
  {stop('Dt must be numeric')}

  if (is.point(Dt,3))
  { Dt<-matrix(Dt,ncol=3)
  } else
  {Dt<-as.matrix(Dt)
  if (ncol(Dt)!=3 )
  {stop('Dt must be of dimension nx3')}
  }

  if (ch.data.pnt==TRUE)
  {
    if (!is.in.data(p,Dt))
    {stop('point, p, is not a data point in Dt')}
  }

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  th<-as.matrix(th)
  if (!is.numeric(th) || nrow(th)!=4 || ncol(th)!=3)
  {stop('th must be numeric and of dimension 4x3')}

  vec1<-rep(1,4);
  D0<-det(matrix(cbind(th,vec1),ncol=4))
  if (round(D0,14)==0)
  {stop('the tetrahedron is degenerate')}

  if (length(M) > 1 || sum(M==c("CM","CC"))==0)
    stop("M must be one of \"CC\", \"CM\"")

  if (identical(matrix(p,ncol=3),matrix(Dt,ncol=3)))
  {dom<-1;return(dom);stop}

  if (in.tetrahedron(p,th)$in.tetra==F)
  {dom<-0; return(dom); stop}

  if (is.null(rv))
  {rv<-ifelse(identical(M,"CC"),rv.tetraCC(p,th)$rv,rv.tetraCM(p,th)$rv); #vertex region for p
  } else
  {  if (!is.numeric(rv) || sum(rv==c(1,2,3,4))!=1)
  {stop('vertex index, rv, must be 1, 2, 3 or 4')}}

  Dt<-matrix(Dt,ncol=3)
  n<-nrow(Dt)
  dom<-1; i<-1;
  while (i <= n & dom==1)
  {if (IndNPEtetra(p,Dt[i,],r,th,M,rv=rv)==0)
    dom<-0;
  i<-i+1;
  }
  dom
} #end of the function
#'

#################################################################

#' @title The indicator for two 3D points constituting a dominating set for Proportional Edge Proximity Catch Digraphs
#' (PE-PCDs) - one tetrahedron case
#'
#' @description Returns \eqn{I(}\{\code{pt1,pt2}\} is a dominating set of the PE-PCD\eqn{)} where the vertices of the PE-PCD are the 3D data set \code{Dt} in the tetrahedron \code{th}, that is,
#' returns 1 if \{\code{pt1,pt2}\} is a dominating set of PE-PCD, returns 0 otherwise.
#'
#' Point, \code{pt1}, is in the region of vertex \code{rv1} (default is \code{NULL}) and point, \code{pt2}, is in the region of vertex \code{rv2}
#' (default is \code{NULL}); vertices (and hence \code{rv1} and \code{rv2}) are labeled as \code{1,2,3,4} in the order they are stacked
#' row-wise in \code{th}.
#'
#' PE proximity region is constructed with respect to the tetrahedron \code{th} with expansion parameter \eqn{r \ge 1}
#' and vertex regions are based on center of mass (\code{M="CM"}) or circumcenter (\code{M="CC"}) only.
#'
#' \code{ch.data.pnts} is for checking whether points \code{pt1} and \code{pt2} are both data points in \code{Dt} or not (default is \code{FALSE}),
#' so by default this function checks whether the points \code{pt1} and \code{pt2} would constitute a dominating set
#' if they actually were both in the data set.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010;textual}{pcds}).
#'
#' @param pt1,pt2 Two 3D points to be tested for constituting a dominating set of the PE-PCD.
#' @param Dt A set of 3D points which constitutes the vertices of the PE-PCD.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param th Four 3D points, stacked row-wise, each row representing a vertex of the tetrahedron.
#' @param M The center to be used in the construction of the vertex regions in the tetrahedron, \code{th}.
#' Currently it only takes \code{"CC"} for circumcenter and \code{"CM"} for center of mass; default=\code{"CM"}.
#' @param rv1,rv2 The indices of the vertices whose regions contains \code{pt1} and \code{pt2}, respectively.
#' They take the vertex labels as \code{1,2,3,4} as in the row order of the vertices in \code{th}
#' (default is \code{NULL} for both).
#' @param ch.data.pnts A logical argument for checking whether both points \code{pt1} and \code{pt2} are
#' data points in \code{Dt} or not (default is \code{FALSE}).
#'
#' @return \eqn{I(}\{\code{pt1,pt2}\} is a dominating set of the PE-PCD\eqn{)} where the vertices of the PE-PCD are the 3D data set \code{Dt}),
#' that is, returns 1 if \{\code{pt1,pt2}\} is a dominating set of PE-PCD, returns 0 otherwise
#'
#' @seealso \code{\link{Gam2PEstdTetra}}, \code{\link{Gam2PEtri}} and \code{\link{Gam2PEbastri}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' A<-c(0,0,0); B<-c(1,0,0); C<-c(1/2,sqrt(3)/2,0); D<-c(1/2,sqrt(3)/6,sqrt(6)/3)
#' tetra<-rbind(A,B,C,D)
#' n<-20
#'
#' dat<-runif.tetra(n,tetra)$g  #try also dat<-cbind(runif(n),runif(n),runif(n))
#'
#' M<-"CM";  #try also M<-"CC";
#' r<-1.5
#'
#' Gam2PEtetra(dat[1,],dat[2,],dat,r,tetra,M)
#' Gam2PEtetra(dat[1,],dat[1,],dat,r,tetra,M)
#'
#' Gam2PEtetra(c(-1,-1,-1),dat[2,],dat,r,tetra,M)
#'
#' Gam2PEtetra(c(-1,-1,-1),c(-1,-1,-2),rbind(c(-1,-1,-1),c(-1,-1,-2)),r,tetra,M)
#'
#' ind.gam2<-ind.gamn2<-vector()
#' for (i in 1:(n-1))
#'  for (j in (i+1):n)
#'  {if (Gam2PEtetra(dat[i,],dat[j,],dat,r,tetra,M)==1)
#'  {ind.gam2<-rbind(ind.gam2,c(i,j))
#'  }
#'  }
#' ind.gam2
#'
#' #or try
#' rv1<-rv.tetraCC(dat[1,],tetra)$rv;rv2<-rv.tetraCC(dat[2,],tetra)$rv
#' Gam2PEtetra(dat[1,],dat[2,],dat,r,tetra,M,rv1,rv2)
#'
#' #or try
#' rv1<-rv.tetraCC(dat[1,],tetra)$rv;
#' Gam2PEtetra(dat[1,],dat[2,],dat,r,tetra,M,rv1)
#'
#' #or try
#' rv2<-rv.tetraCC(dat[2,],tetra)$rv
#' Gam2PEtetra(dat[1,],dat[2,],dat,r,tetra,M,rv2=rv2)
#'
#' P1<-c(.1,.1,.1)
#' P2<-c(.4,.1,.2)
#' Gam2PEtetra(P1,P2,dat,r,tetra,M)
#'
#' dat.fr<-data.frame(a=dat)
#' Gam2PEtetra(P1,P2,dat.fr,r,tetra,M)
#'
#' Gam2PEtetra(c(-1,-1,-1),dat[2,],dat,r,tetra,M,ch.data.pnts = FALSE)
#' #gives an error message if ch.data.pnts = TRUE
#' #since not both points, pt1 and pt2, are data points in Dt
#'
#' @export
Gam2PEtetra <- function(pt1,pt2,Dt,r,th,M="CM",rv1=NULL,rv2=NULL,ch.data.pnts=FALSE)
{
  if (!is.point(pt1,3) || !is.point(pt2,3))
  {stop('pt1 and pt2 must be a numeric 3D point')}

  if (!is.numeric(as.matrix(Dt)))
  {stop('third argument must be numeric')}

  if (is.point(Dt,3))
  { Dt<-matrix(Dt,ncol=3)
  } else
  {Dt<-as.matrix(Dt)
  if (ncol(Dt)!=3 )
  {stop('Dt must be of dimension nx3')}
  }

  if (ch.data.pnts==TRUE)
  {
    if (!is.in.data(pt1,Dt) || !is.in.data(pt2,Dt))
    {stop('not both points are data points in Dt')}
  }

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  th<-as.matrix(th)
  if (!is.numeric(th) || nrow(th)!=4 || ncol(th)!=3)
  {stop('th must be numeric and of dimension 4x3')}

  vec1<-rep(1,4);
  D0<-det(matrix(cbind(th,vec1),ncol=4))
  if (round(D0,14)==0)
  {stop('the tetrahedron is degenerate')}

  if (length(M) > 1 || sum(M==c("CM","CC"))==0)
    stop("M must be one of \"CC\", \"CM\"")

  if (identical(pt1,pt2))
  {dom<-0; return(dom); stop}

  if (is.null(rv1))
  {rv1<-ifelse(identical(M,"CC"),rv.tetraCC(pt1,th)$rv,rv.tetraCM(pt1,th)$rv);} #vertex region for pt1

  if (is.null(rv2))
  {rv2<-ifelse(identical(M,"CC"),rv.tetraCC(pt2,th)$rv,rv.tetraCM(pt2,th)$rv);} #vertex region for pt1

  Dt<-matrix(Dt,ncol=3)
  n<-nrow(Dt)
  dom<-1; i<-1;
  while (i <= n & dom==1)
  {if (max(IndNPEtetra(pt1,Dt[i,],r,th,M,rv1),IndNPEtetra(pt2,Dt[i,],r,th,M,rv2))==0)
    dom<-0;
  i<-i+1;
  }
  dom
} #end of the function
#'

#################################################################

#' @title The indicator for three 3D points constituting a dominating set for Proportional Edge Proximity Catch Digraphs
#' (PE-PCDs) - one tetrahedron case
#'
#' @description Returns \eqn{I(\{}\code{pt1,pt2,pt3}\} is a dominating set of the PE-PCD\eqn{)} where the vertices of the PE-PCD are the 3D data set \code{Dt} in the tetrahedron \code{th}, that is,
#' returns 1 if \{\code{pt1,pt2,pt3}\} is a dominating set of PE-PCD, returns 0 otherwise.
#'
#' Point, \code{pt1}, is in the region of vertex \code{rv1} (default is \code{NULL}), point, \code{pt2}, is in the region of vertex \code{rv2}
#' (default is \code{NULL}); point, \code{pt3}), is in the region of vertex \code{rv3}) (default is \code{NULL}); vertices (and hence \code{rv1, rv2} and
#' \code{rv3}) are labeled as \code{1,2,3,4} in the order they are stacked row-wise in \code{th}.
#'
#' PE proximity region is constructed with
#' respect to the tetrahedron \code{th} with expansion parameter \eqn{r \ge 1}
#' and vertex regions are based on center of mass \eqn{CM} (equivalent to circumcenter in this case).
#'
#' \code{ch.data.pnts} is for checking whether points \code{pt1}, \code{pt2} and \code{pt3} are all data points in \code{Dt} or not (default is \code{FALSE}),
#' so by default this function checks whether the points \code{pt1}, \code{pt2} and \code{pt3} would constitute a dominating set
#' if they actually were all in the data set.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010;textual}{pcds}).
#'
#' @param pt1,pt2,pt3 Three 3D points to be tested for constituting a dominating set of the PE-PCD.
#' @param Dt A set of 3D points which constitutes the vertices of the PE-PCD.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param th Four 3D points, stacked row-wise, each row representing a vertex of the tetrahedron.
#' @param M The center to be used in the construction of the vertex regions in the tetrahedron, \code{th}.
#' Currently it only takes \code{"CC"} for circumcenter and \code{"CM"} for center of mass; default=\code{"CM"}.
#' @param rv1,rv2,rv3 The indices of the vertices whose regions contains \code{pt1}, \code{pt2} and \code{pt3},
#' respectively.
#' They take the vertex labels as \code{1,2,3,4} as in the row order of the vertices in \code{th}
#' ( default is \code{NULL} for all).
#' @param ch.data.pnts A logical argument for checking whether points \code{pt1} and \code{pt2} are
#' data points in \code{Dt} or not (default is \code{FALSE}).
#'
#' @return \eqn{I(\{}\code{pt1,pt2,pt3}\} is a dominating set of the PE-PCD\eqn{)} where the vertices of the PE-PCD are the 3D data set \code{Dt}),
#' that is, returns 1 if \{\code{pt1,pt2,pt3}\} is a dominating set of PE-PCD, returns 0 otherwise
#'
#' @seealso \code{\link{Gam3PEstdTetra}}
#'
#' @references
#' \insertAllCited{}
#'
#' @author Elvan Ceyhan
#'
#' @examples
#' set.seed(123)
#' A<-c(0,0,0); B<-c(1,0,0); C<-c(1/2,sqrt(3)/2,0); D<-c(1/2,sqrt(3)/6,sqrt(6)/3)
#' tetra<-rbind(A,B,C,D)
#' n<-10 #try 20, 40, 100 (larger n may take a long time)
#' dat<-runif.tetra(n,tetra)$g
#'
#' M<-"CM";  #try also M<-"CC";
#' r<-1.25
#'
#' Gam3PEtetra(dat[1,],dat[2,],dat[3,],dat,r,tetra,M)
#'
#' Gam3PEtetra(dat[1,],dat[2,],dat[2,],dat,r,tetra,M)
#'
#' Gam3PEtetra(dat[1,],c(1,1,1),dat[3,],dat,r,tetra,M)
#'
#' Gam3PEtetra(c(-1,1,1),c(1,1,1),c(1,1,-1),rbind(c(-1,1,1),c(1,1,1),c(1,1,-1)),r,tetra,M)
#'
#' ind.gam3<-vector()
#' for (i in 1:(n-2))
#'  for (j in (i+1):(n-1))
#'    for (k in (j+1):n)
#'    {if (Gam3PEtetra(dat[i,],dat[j,],dat[k,],dat,r,tetra,M)==1)
#'     ind.gam3<-rbind(ind.gam3,c(i,j,k))}
#'
#' ind.gam3
#'
#' #or try
#' rv1<-rv.tetraCC(dat[1,],tetra)$rv; rv2<-rv.tetraCC(dat[2,],tetra)$rv;
#' rv3<-rv.tetraCC(dat[3,],tetra)$rv
#' Gam3PEtetra(dat[1,],dat[2,],dat[3,],dat,r,tetra,M,rv1,rv2,rv3)
#'
#' #or try
#' rv1<-rv.tetraCC(dat[1,],tetra)$rv;
#' Gam3PEtetra(dat[1,],dat[2,],dat[3,],dat,r,tetra,M,rv1)
#'
#' #or try
#' rv2<-rv.tetraCC(dat[2,],tetra)$rv
#' Gam3PEtetra(dat[1,],dat[2,],dat[3,],dat,r,tetra,M,rv2=rv2)
#'
#' P1<-c(.1,.1,.1)
#' P2<-c(.3,.3,.3)
#' P3<-c(.4,.1,.2)
#' Gam3PEtetra(P1,P2,P3,dat,r,tetra,M)
#'
#' dat.fr<-data.frame(a=dat)
#' Gam3PEtetra(P1,P2,P3,dat.fr,r,tetra,M)
#'
#' Gam3PEtetra(dat[1,],c(1,1,1),dat[3,],dat,r,tetra,M,ch.data.pnts = FALSE)
#' #gives an error message if ch.data.pnts = TRUE since not all points are data points in Dt
#'
#' @export
Gam3PEtetra <- function(pt1,pt2,pt3,Dt,r,th,M="CM",rv1=NULL,rv2=NULL,rv3=NULL,ch.data.pnts=FALSE)
{
  if (!is.point(pt1,3) || !is.point(pt2,3) || !is.point(pt3,3))
  {stop('pt1, pt2, and pt3 must be a numeric 3D points')}

  if (!is.numeric(as.matrix(Dt)))
  {stop('Dt must be numeric')}

  if (is.point(Dt,3))
  { Dt<-matrix(Dt,ncol=3)
  } else
  {Dt<-as.matrix(Dt)
  if (ncol(Dt)!=3 )
  {stop('Dt must be of dimension nx3')}
  }

  if (ch.data.pnts==TRUE)
  {
    if (!is.in.data(pt1,Dt) || !is.in.data(pt2,Dt) || !is.in.data(pt3,Dt))
    {stop('not all points are data points in Dt')}
  }

  if (!is.point(r,1) || r<1)
  {stop('r must be a scalar >= 1')}

  th<-as.matrix(th)
  if (!is.numeric(th) || nrow(th)!=4 || ncol(th)!=3)
  {stop('th must be numeric and of dimension 4x3')}

  vec1<-rep(1,4);
  D0<-det(matrix(cbind(th,vec1),ncol=4))
  if (round(D0,14)==0)
  {stop('the tetrahedron is degenerate')}

  if (length(M) > 1 || sum(M==c("CM","CC"))==0)
    stop("M must be one of \"CC\", \"CM\"")

  if (identical(pt1,pt2) || identical(pt1,pt3) || identical(pt2,pt3))
  {dom<-0; return(dom); stop}

  if (is.null(rv1))
  {rv1<-ifelse(identical(M,"CC"),rv.tetraCC(pt1,th)$rv,rv.tetraCM(pt1,th)$rv);} #vertex region for pt1

  if (is.null(rv2))
  {rv2<-ifelse(identical(M,"CC"),rv.tetraCC(pt2,th)$rv,rv.tetraCM(pt2,th)$rv);} #vertex region for pt2

  if (is.null(rv3))
  {rv3<-ifelse(identical(M,"CC"),rv.tetraCC(pt3,th)$rv,rv.tetraCM(pt3,th)$rv);} #vertex region for pt3

  Dt<-matrix(Dt,ncol=3)
  n<-nrow(Dt)
  dom<-1; i<-1;
  while (i <= n & dom==1)
  {if (max(IndNPEtetra(pt1,Dt[i,],r,th,M,rv1),IndNPEtetra(pt2,Dt[i,],r,th,M,rv2),
           IndNPEtetra(pt3,Dt[i,],r,th,M,rv3))==0)
    dom<-0;
  i<-i+1;
  }
  dom
} #end of the function
#'

#################################################################

#' @title The domination number of Proportional Edge Proximity Catch Digraph (PE-PCD) - one tetrahedron case
#'
#' @description Returns the domination number of PE-PCD whose vertices are the data points in \code{Xp}.
#'
#' PE proximity region is defined with respect to the tetrahedron \code{th} with expansion parameter \eqn{r \ge 1} and
#' vertex regions are based on the center \code{M} which is circumcenter (\code{"CC"}) or center of mass (\code{"CM"}) of \code{th}
#' with default=\code{"CM"}.
#'
#' See also (\insertCite{ceyhan:Phd-thesis,ceyhan:comp-geo-2010;textual}{pcds}).
#'
#' @param Xp A set of 3D points which constitute the vertices of the digraph.
#' @param th Four 3D points, stacked row-wise, each row representing a vertex of the tetrahedron.
#' @param r A positive real number which serves as the expansion parameter in PE proximity region;
#' must be \eqn{\ge 1}.
#' @param M The center to be used in the construction of the vertex regions in the tetrahedron, \code{th}.
#' Currently it only takes \code{"CC"} for circumcenter and \code{"CM"} for center of mass; default=\code{"CM"}.
#'
#' @return A \code{list} with two elements
#' \item{dom.num}{Domination number of PE-PCD with vertex set = \code{Xp} and expansion parameter \eqn{r \ge 1} and center \code{M}}
#' \item{mds}{A m