% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/model_parameters.FactoMineR.R,
%   R/model_parameters.psych.R
\name{model_parameters.PCA}
\alias{model_parameters.PCA}
\alias{model_parameters.principal}
\alias{model_parameters.omega}
\title{Structural Models (PCA, EFA, ...)}
\usage{
\method{model_parameters}{PCA}(model, sort = FALSE, threshold = NULL, labels = NULL, ...)

\method{model_parameters}{principal}(model, sort = FALSE, threshold = NULL, labels = NULL, ...)

\method{model_parameters}{omega}(model, ...)
}
\arguments{
\item{model}{PCA or FA created by the \pkg{psych} or \pkg{FactoMineR} packages (e.g. through \code{psych::principal},  \code{psych::fa} or \code{psych::omega}).}

\item{sort}{Sort the loadings.}

\item{threshold}{A value between 0 and 1 indicates which (absolute) values from the loadings should be removed. An integer higher than 1 indicates the n strongest loadings to retain. Can also be \code{"max"}, in which case it will only display the maximum loading per variable (the most simple structure).}

\item{labels}{A character vector containing labels to be added to the loadings data. Usually, the question related to the item.}

\item{...}{Arguments passed to or from other methods.}
}
\value{
A data frame of loadings.
}
\description{
Format structural models from the \pkg{psych} or \pkg{FactoMineR} packages.
}
\details{
For the structural models obtained with \pkg{psych}, the following indices are present:
 \itemize{
   \item \strong{Complexity} (\cite{Hoffman's, 1978; Pettersson and Turkheimer, 2010}) represents the number of latent components needed to account for the observed variables. Whereas a perfect simple structure solution has a complexity of 1 in that each item would only load on one factor, a solution with evenly distributed items has a complexity greater than 1.
   \item \strong{Uniqueness} represents the variance that is 'unique' to the variable and not shared with other variables. It is equal to \code{1 – communality} (variance that is shared with other variables). A uniqueness of \code{0.20} suggests that 20\% or that variable's variance is not shared with other variables in the overall factor model. The greater 'uniqueness' the lower the relevance of the variable in the factor model.
   \item \strong{MSA} represents the Kaiser-Meyer-Olkin Measure of Sampling Adequacy (\cite{Kaiser and Rice, 1974}) for each item. It indicates whether there is enough data for each factor give reliable results for the PCA. The value should be > 0.6, and desirable values are > 0.8 (\cite{Tabachnick and Fidell, 2013}).
}
}
\examples{
library(parameters)
library(psych)

# Principal Component Analysis (PCA) ---------
pca <- psych::principal(attitude)
model_parameters(pca)

pca <- psych::principal(attitude, nfactors = 3, rotate = "none")
model_parameters(pca, sort = TRUE, threshold = 0.2)

principal_components(attitude, n = 3, sort = TRUE, threshold = 0.2)
\donttest{
# Exploratory Factor Analysis (EFA) ---------
efa <- psych::fa(attitude, nfactors = 3)
model_parameters(efa, threshold = "max", sort = TRUE, labels = as.character(1:ncol(attitude)))
}

# Omega ---------
omega <- psych::omega(mtcars, nfactors = 3)
params <- model_parameters(omega)
params
summary(params)


# FactoMineR ---------
\dontrun{
library(FactoMineR)

model <- FactoMineR::PCA(iris[, 1:4], ncp = 2)
model_parameters(model)
attributes(model_parameters(model))$scores

model <- FactoMineR::FAMD(iris, ncp = 2)
model_parameters(model)
}
}
\references{
\itemize{
  \item Kaiser, H.F. and Rice. J. (1974). Little jiffy, mark iv. Educational and Psychological Measurement, 34(1):111–117
  \item Pettersson, E., \& Turkheimer, E. (2010). Item selection, evaluation, and simple structure in personality data. Journal of research in personality, 44(4), 407-420.
  \item Revelle, W. (2016). How To: Use the psych package for Factor Analysis and data reduction.
  \item Tabachnick, B. G., and Fidell, L. S. (2013). Using multivariate statistics (6th ed.). Boston: Pearson Education.
}
}
