#' An R6 Class Interface to OpenAI API
#'
#' @description Provides methods to interact with OpenAI API including
#'   fetching model details, generating completions, managing files, and more.
#'   Always ensure that the API key is kept private.
#' @examples
#' if(file.exists("openai-key")){
#'   key <- readLines(file("openai-key"))
#'   #NOTE To retrieve the correct data, it is necessary to provide an 'openaikey';
#'   # otherwise, an exception will be returned.
#' }else{
#'   key <- "you_api_key"
#' }
#' aaa <- openai$new(key)
#' #if need proxy
#' #aaa$set_proxy("127.0.0.1", 10890)
#' # List model
#' aaa$get_model_list()
#' \donttest{
#' #The following request will cause a timeout when the correct key is not present,
#' # so it is not necessary to test it during the check.
#' # Upload a file
#' train_file_path<-system.file("exdata","train.jsonl", package = "openaistream")
#' file_id <- aaa$files_upload(path = train_file_path,purpose = "fine-tune")
#' # Retrieve a file's details
#' aaa$files_retrieve(NULL, verbosity = 0)
#' # Delete a file
#' #aaa$files_delete(file_id$id, verbosity = 0)
#' # Get a list of files
#' aaa$files_get_list()
#' #aaa
#' job<-aaa$fine_tuning_jobs_create(model = "gpt-3.5-turbo",training_file = file_id$id)
#' aaa$files_delete(file_id$id, verbosity = 0)
#' # List fine-tuning jobs
#' aaa$fine_tuning_jobs_list(limit = 2, verbosity = 3)
#' # Retrieve details of a specific fine-tuning job
#' aaa$fine_tuning_jobs_retrieve(job$id, verbosity = 0)
#' # Get events related to a specific fine-tuning job
#' aaa$fine_tuning_jobs_events(job$id, verbosity = 3)
#' # Compute embeddings for input data
#' aaa$embeddings(model = "text-embedding-ada-002", input = "who are you?")
#' # chat test not stream
#' streamlg <- aaa$get_completions_query(
#'   prompt = "Please explain the World Wars?",
#'   model = "davinci-002",
#'   stream = FALSE,
#'   max_tokens = 20,
#'   num = 4,
#'   verbosity = 0
#' )
#'
#' # chat test stream
#' streamlg <- aaa$get_chat_completions_query(
#'   messages = data.frame(role = c("system", "user"),
#'                         content = c("You are a assistant.", "How's the weather today?")),
#'   model = "gpt-3.5-turbo",
#'   stream = TRUE,
#'   max_tokens = 10
#' )
#'
#' streamlg$get_state()
#' streamlg$next_value
#' streamlg$close()
#' streamlg$next_value
#' streamlg$get_state()
#' sss<-aaa$audio_speech(input = "Hi, this is a voice transmission test.",response_format="mp3")
#' #writeBin(sss,"test.mp3")
#' #text_E1<-aaa$audio_transcriptions(path = "test.mp3")
#' #text_E2<-aaa$audio_translations(path = "test.mp3")
#' pic1<-aaa$images_generations(prompt = "A small bird flies over the ocean")
#' #req <- request(pic1$data$url) %>%req_perform(path = "pic1.png")
#' #Here is the image test.
#' #For simple test,you need install 'png' package.
#' #Note that when a mask is not provided, OpenAI defaults to using areas of the uploaded image
#' #  with a transparency of 0 as the modification area.
#' #The following test adds full image transparency of 0 (modifying the entire image) to pic1.
#' # pic1<-readPNG("pic1.png")
#' # if (dim(pic1)[3] == 3) {  # Checking if the image is RGB
#' #   # Adding an alpha channel (fully opaque)
#' #   alpha_channel <- matrix(0, nrow = dim(pic1)[1], ncol = dim(pic1)[2])
#' #   pic1_b<-array(c(pic1,alpha_channel), c(1024,1024,4))
#' # }
#' # writePNG(pic1_b,target = "pic1_b.png")
#' # pic3<-aaa$images_edits(image = "pic1_b.png",
#' # prompt = "Please modify the bird in the picture to a flock of birds."
#' #  ,verbosity = 3,n="2",size="256x256")
#' # request(pic3$data$url[1]) %>%req_perform(path = "pic3_1.png")
#' # request(pic3$data$url[2]) %>%req_perform(path = "pic3_2.png")
#' #
#' # pic4<-aaa$images_variations(image = "pic1.png",n="2",size="256x256")
#' # request(pic4$data$url[1]) %>%req_perform(path = "pic4_1.png")
#' # request(pic4$data$url[2]) %>%req_perform(path = "pic4_2.png")
#' }
#' @export
openai <- R6Class(
  "openai",
  private = list(
    api_key = NULL,
    proxy=list(),
    api_endpoints = list(
      chat_completions = "https://api.openai.com/v1/chat/completions",
      completions = "https://api.openai.com/v1/completions",
      files = "https://api.openai.com/v1/files",
      models = "https://api.openai.com/v1/models",
      fine_tuning_jobs ="https://api.openai.com/v1/fine_tuning/jobs",
      embeddings = "https://api.openai.com/v1/embeddings",
      audio = "https://api.openai.com/v1/audio",
      images = "https://api.openai.com/v1/images"
    ),
    #base_call
    base_call = function(endpoint, path="", body=NULL, method="GET", headers=list(), query=list()){
      url <- private$api_endpoints[[endpoint]]
      req <- request(url) %>%
        req_headers(Authorization = paste0("Bearer ", private$api_key),!!!headers) %>%
        req_method(method)
      if (path != "") {
        req <- req %>% req_url_path_append(strsplit(path, "/")[[1]])
      }
      if (length(query) > 0) {
        req <- req %>%req_url_query(!!!query)
      }

      if(length(private$proxy) == 2) {
        req <- req %>%
          req_proxy(private$proxy$ip, private$proxy$port)
      }
      req
    },
    #get and post and delete call
    api_call = function(endpoint, path="", body=NULL, method="GET", headers=list(), query=list(), verbosity=0,fpath=NULL) {
      req<-private$base_call(endpoint, path=path, method=method, headers=headers, query=query)
      #browser()
      if (!is.null(body)) {
        req <- req %>%req_body_json(body)
      }
      tryCatch({
        parsed <- req %>% req_perform(verbosity = verbosity,path=fpath)
        if(path=="/speech"){
          return(list(success=TRUE, data=parsed$body))
        }else{
          return(list(success=TRUE, data=fromJSON(rawToChar(parsed$body))))
        }
      }, error=function(e) {
        return(openai_error$new(as.character(e)))
      })
    },
    #stream call
    handle_call=function(endpoint, path="", body=NULL, headers=list(), query=list()){
      req<-private$base_call(endpoint, path=path, method="POST", headers=headers, query=query)
      if (!is.null(body)) {
        req <- req %>%req_body_json(body)
      }else{
        return(list(success=FALSE, data="not find body"))
      }
      return(curl::curl(req$url, handle = httr2:::req_handle(req)))
    },
    #file call
    file_call=function(endpoint, path="", body=NULL, headers=list(), query=list(), verbosity=0){
      req<-private$base_call(endpoint, path=path, method="POST", headers=headers, query=query)
      if (!is.null(body)) {
        req<-req %>% req_body_multipart(!!!body)
      }else{
        return(list(success=FALSE, data="not find body"))
      }
      tryCatch({
        parsed <- req %>% req_perform(verbosity = verbosity)
        return(list(success=TRUE, data=fromJSON(rawToChar(parsed$body))))
      }, error=function(e) {
        return(openai_error$new(as.character(e)))
      })
    }
  ),
  public = list(
    #' @description Initialize the OpenAI API interface with the provided API key.
    #' @param api_key The OpenAI API key.
    initialize = function(api_key) {
      private$api_key<-api_key
    },
    #' @description Configure the proxy settings.
    #' @param proxy_ip The IP address of the proxy.
    #' @param proxy_port The port number of the proxy.
    set_proxy = function(proxy_ip,proxy_port){
      if(!grepl("^([0-9]{1,3}\\.){3}[0-9]{1,3}$", proxy_ip)) {
        stop("Invalid proxy IP address.")
      }
      ip_components <- unlist(strsplit(proxy_ip, split = "\\."))
      if(any(as.numeric(ip_components) > 255) || any(as.numeric(ip_components) < 0)) {
        stop("Invalid proxy IP address.")
      }
      if(!is.numeric(proxy_port) || proxy_port < 1 || proxy_port > 65535) {
        stop("Invalid proxy port number.")
      }
      private$proxy$ip = proxy_ip
      private$proxy$port = proxy_port
    },
    #' @description Retrieve a list of available models from OpenAI.
    #' @param verbosity Verbosity level for the API call.
    #' @return A list of available models.
    get_model_list=function(verbosity=0){
      result <- private$api_call("models", headers = list(`Content-Type` = "application/json"), verbosity = verbosity)
      if (inherits(result, "openai_error")) {
        return(list(success=FALSE, message=result$get_message(), type=result$get_type()))
      }else{
        return(result$data$data)
      }
    },
    #' @description Retrieve details of a specific model from OpenAI.
    #' @param model The model ID.
    #' @param verbosity Verbosity level for the API call.
    #' @return Details of the specified model.
    get_model_retrieve=function(model,verbosity=0){
      result <- private$api_call("models", paste0("/", model), verbosity = verbosity)
      if (inherits(result, "openai_error")) {
        return(list(success=FALSE, message=result$get_message(), type=result$get_type()))
      }else{
        return(result$data$data)
      }
    },
    #' @description Delete a fine-tuned model from OpenAI.
    #' @param model_id The ID of the file to delete.
    #' @param verbosity Verbosity level for the API call.
    #' @return Response indicating the success or failure of the delete operation.
    fine_tuning_jobs_delete=function(model_id,verbosity=0){
      result <- private$api_call("models", paste0("/", model_id), method = "DELETE", verbosity = verbosity)
      if (inherits(result, "openai_error")) {
        return(list(success=FALSE, message=result$get_message(), type=result$get_type()))
      }else{
        return(result$data)
      }
    },
    #' @description Get completions for a given prompt using a specific model.
    #' @param prompt The input text to get completions for.
    #' @param model The model to use for generating completions.
    #' @param stream A boolean indicating whether to stream the results.
    #' @param num The num parameter controls the number of text entries returned by a stream in one go.
    #'            Note that this is different from the n parameter, which specifies the number of results returned.
    #'            For detailed information on the n parameter, please refer to OpenAI's API documentation.
    #' @param verbosity Verbosity level for the API call.
    #' @param ... Additional parameters as required by the OpenAI API.For example:max_tokens;n;stop;temperature......
    #' @return Completions based on the input prompt.
    get_completions_query=function(prompt,model,stream=F,num=2,verbosity = 0,...){
      option = list(...)
      option$prompt = prompt
      option$model = model
      option$stream = stream
      if (option$stream) {
        handle <- private$handle_call("completions", body=option, headers=list(Accept="text/event-stream", `Content-Type` = "application/json"))
        return(DataStream$new(requery = handle, num = num))
      } else {
        result <- private$api_call("completions", body=option, headers=list(`Content-Type` = "application/json"),method = "POST", verbosity=verbosity)
        if (inherits(result, "openai_error")) {
          return(list(success=FALSE, message=result$get_message(), type=result$get_type()))
        }else{
          return(list(all_resp=result$data, vres = result$data$choices))
        }
      }
    },
    #' @description Generate conversational completions using chat models.
    #' @param messages A list of message objects, where each message has a role and content.
    #' @param model The model to use for generating chat completions.
    #' @param stream A boolean indicating whether to stream the results.
    #' @param num The num parameter controls the number of text entries returned by a stream in one go.
    #'            Note that this is different from the n parameter, which specifies the number of results returned.
    #'            For detailed information on the n parameter, please refer to OpenAI's API documentation.
    #' @param verbosity Verbosity level for the API call.
    #' @param ... Additional parameters as required by the OpenAI API.For example:max_tokens;n;stop;temperature......
    #' @return Completions based on the conversational context.
    get_chat_completions_query=function(messages,model,stream=F,num=2,verbosity = 0,...){
      option = list(...)
      option$messages = messages
      option$model = model
      option$stream = stream
      if (option$stream) {
        handle <- private$handle_call("chat_completions", body=option, headers=list(Accept="text/event-stream", `Content-Type` = "application/json"))
        return(DataStream$new(requery = handle , num = num))
      } else {
        result <- private$api_call("chat_completions", body=option, headers=list(`Content-Type` = "application/json"),method = "POST", verbosity=verbosity)
        if (inherits(result, "openai_error")) {
          return(list(success=FALSE, message=result$get_message(), type=result$get_type()))
        }else{
          return(list(all_resp=result$data, vres = result$data$choices))
        }
      }
    },
    #' @description Upload a file to OpenAI.
    #' @param path Path to the file that needs to be uploaded.
    #' @param purpose Purpose of the file (e.g., "fine-tune").
    #' @param verbosity Verbosity level for the API call.
    #' @return Response indicating the success or failure of the upload operation.
    files_upload=function(path,verbosity=0,purpose){
      result<-private$file_call(endpoint = "files",body = list(file = curl::form_file(path),purpose=purpose),verbosity=verbosity)
      if (inherits(result, "openai_error")) {
        return(list(success=FALSE, message=result$get_message(), type=result$get_type()))
      }else{
        return(result$data)
      }
    },
    #' @description Retrieve a list of files from OpenAI.
    #' @param verbosity Verbosity level for the API call.
    #' @return A list of files.
    files_get_list=function(verbosity=0){
      result <- private$api_call("files", verbosity = verbosity)
      if (inherits(result, "openai_error")) {
        return(list(success=FALSE, message=result$get_message(), type=result$get_type()))
      }else{
        return(result$data)
      }
    },
    #' @description Delete a file from OpenAI.
    #' @param file_id The ID of the file to delete.
    #' @param verbosity Verbosity level for the API call.
    #' @return Response indicating the success or failure of the delete operation.
    files_delete=function(file_id,verbosity=0){
      result <- private$api_call("files", paste0("/", file_id), method = "DELETE", verbosity = verbosity)
      if (inherits(result, "openai_error")) {
        return(list(success=FALSE, message=result$get_message(), type=result$get_type()))
      }else{
        return(result$data)
      }
    },
    #' @description Retrieve details of a specific file from OpenAI.
    #' @param file_id The ID of the file to retrieve details for.
    #' @param verbosity Verbosity level for the API call.
    #' @return Details of the specified file.
    files_retrieve=function(file_id,verbosity=0){
      result <- private$api_call("files", paste0("/", file_id), verbosity = verbosity)
      if (inherits(result, "openai_error")) {
        return(list(success=FALSE, message=result$get_message(), type=result$get_type()))
      }else{
        return(result$data)
      }
    },
    #' @description Create a fine-tuning job in OpenAI.
    #' @param model The model ID.
    #' @param training_file The file used for training.
    #' @param hyperparameters params list for the fine-tuning include batch_size;learning_rate_multiplier;n_epochs.
    #' @param verbosity Verbosity level for the API call.
    #' @param ... Additional parameters as required by the OpenAI API.For example:validation_file......
    #' @return Response indicating the success or failure of the fine-tuning job creation.
    fine_tuning_jobs_create=function(model,training_file,hyperparameters=list(n_epochs=1),verbosity=0,...){
      option <- list(...)
      option$model <- model
      option$training_file <- training_file
      option$hyperparameters <- hyperparameters
      result <- private$api_call("fine_tuning_jobs", body = option, method = "POST", headers = list(`Content-Type` = "application/json"), verbosity = verbosity)
      if (inherits(result, "openai_error")) {
        return(list(success=FALSE, message=result$get_message(), type=result$get_type()))
      }else{
        return(result$data)
      }
    },
    #' @description List fine-tuning jobs from OpenAI.
    #' @param verbosity Verbosity level for the API call.
    #' @param ... Additional parameters as required by the OpenAI API.
    #' @return A list of fine-tuning jobs.
    fine_tuning_jobs_list=function(verbosity=0,...){
      option = list(...)
      result <- private$api_call("fine_tuning_jobs", query = option, verbosity = verbosity)
      if (inherits(result, "openai_error")) {
        return(list(success=FALSE, message=result$get_message(), type=result$get_type()))
      }else{
        return(result$data)
      }
    },
    #' @description Retrieve details of a specific fine-tuning job from OpenAI.
    #' @param job_id The ID of the fine-tuning job.
    #' @param verbosity Verbosity level for the API call.
    #' @return Details of the specified fine-tuning job.
    fine_tuning_jobs_retrieve=function(job_id,verbosity=0){
      result <- private$api_call("fine_tuning_jobs", paste0("/", job_id), verbosity = verbosity)
      if (inherits(result, "openai_error")) {
        return(list(success=FALSE, message=result$get_message(), type=result$get_type()))
      }else{
        return(result$data)
      }
    },
    #' @description Cancel a specific fine-tuning job in OpenAI.
    #' @param job_id The ID of the fine-tuning job to cancel.
    #' @param verbosity Verbosity level for the API call.
    #' @return Response indicating the success or failure of the cancel operation.
    fine_tuning_jobs_cancel=function(job_id,verbosity=0){
      result <- private$api_call("fine_tuning_jobs", paste0("/", job_id, "/", "cancel"), method = "POST", verbosity = verbosity)
      if (inherits(result, "openai_error")) {
        return(list(success=FALSE, message=result$get_message(), type=result$get_type()))
      }else{
        return(result$data)
      }
    },
    #' @description Get events related to a specific fine-tuning job from OpenAI.
    #' @param job_id The ID of the fine-tuning job to retrieve events for.
    #' @param verbosity Verbosity level for the API call.
    #' @return A list of events related to the specified fine-tuning job.
    fine_tuning_jobs_events=function(job_id,verbosity=0){
      result <- private$api_call("fine_tuning_jobs", paste0("/", job_id, "/", "events"),method = "GET", verbosity = verbosity)
      if (inherits(result, "openai_error")) {
        return(list(success=FALSE, message=result$get_message(), type=result$get_type()))
      }else{
        return(result$data)
      }
    },
    #' @description Compute embeddings for input data using a specified model in OpenAI.
    #' @param model The model to use for generating embeddings.
    #' @param input Input data to generate embeddings for.
    #' @param verbosity Verbosity level for the API call.
    #' @param ... Additional parameters as required by the OpenAI API.For example:encoding_format;user....
    #' @return Embeddings for the input data.
    embeddings=function(model,input,verbosity=0,...){
      option <- list(...)
      option$model <- model
      option$input <- input
      result <- private$api_call("embeddings", body = option,method = "POST", headers = list(`Content-Type` = "application/json"), verbosity = verbosity)
      if (inherits(result, "openai_error")) {
        return(list(success=FALSE, message=result$get_message(), type=result$get_type()))
      }else{
        return(result$data)
      }
    },
    #' @description Generates audio from the input text.
    #' @param model One of the available TTS models: tts-1 or tts-1-hd
    #' @param input The text to generate audio for. The maximum length is 4096 characters.
    #' @param voice The voice to use when generating the audio. Supported voices are alloy, echo, fable, onyx, nova, and shimmer.
    #' @param verbosity Verbosity level for the API call.
    #' @param ... Additional parameters as required by the OpenAI API.For example:response_format;speed....
    #' @return The audio file content.
    audio_speech=function(model="tts-1",input,voice="alloy",verbosity=0,...){
      option <- list(...)
      option$model <- model
      option$input <- input
      option$voice <- voice
      result <- private$api_call("audio", body = option,"/speech",method = "POST", headers = list(`Content-Type` = "application/json"), verbosity = verbosity)
      if (inherits(result, "openai_error")) {
        return(list(success=FALSE, message=result$get_message(), type=result$get_type()))
      }else{
        return(result$data)
      }
    },
    #' @description Transcribes audio into the input language.
    #' @param path The audio file object (not file name) to transcribe, in one of these formats: flac, mp3, mp4, mpeg, mpga, m4a, ogg, wav, or webm.
    #' @param model ID of the model to use. Only whisper-1 is currently available.
    #' @param verbosity Verbosity level for the API call.
    #' @param ... Additional parameters as required by the OpenAI API.For example:language;prompt;response_format;temperature....
    #' @return The transcribed text.
    audio_transcriptions=function(path,model="whisper-1",verbosity=0,...){
      option <- list(...)
      option$model <- model
      option$file <- curl::form_file(path)
      result<-private$file_call(endpoint = "audio",path = "/transcriptions",body = option,verbosity=verbosity)
      if (inherits(result, "openai_error")) {
        return(list(success=FALSE, message=result$get_message(), type=result$get_type()))
      }else{
        return(result$data)
      }
    },
    #' @description Translates audio into English.
    #' @param path The audio file object (not file name) to transcribe, in one of these formats: flac, mp3, mp4, mpeg, mpga, m4a, ogg, wav, or webm.
    #' @param model ID of the model to use. Only whisper-1 is currently available.
    #' @param verbosity Verbosity level for the API call.
    #' @param ... Additional parameters as required by the OpenAI API.For example:prompt;response_format;temperature....
    #' @return The transcribed text.
    audio_translations=function(path,model="whisper-1",verbosity=0,...){
      option <- list(...)
      option$model <- model
      option$file <- curl::form_file(path)
      result<-private$file_call(endpoint = "audio",path = "/translations",body = option,verbosity=verbosity)
      if (inherits(result, "openai_error")) {
        return(list(success=FALSE, message=result$get_message(), type=result$get_type()))
      }else{
        return(result$data)
      }
    },
    #' @description Creates an image given a prompt.
    #' @param prompt A text description of the desired image(s). The maximum length is 1000 characters for dall-e-2 and 4000 characters for dall-e-3
    #' @param verbosity  Verbosity level for the API call.
    #' @param ...  Additional parameters as required by the OpenAI API.For example:n;quality;response_format...
    #' @return Returns a list of image objects.
    images_generations=function(prompt,verbosity=0,...){
      option <- list(...)
      option$prompt <- prompt
      result<-private$api_call(endpoint = "images",path = "/generations", headers = list(`Content-Type` = "application/json"),method = "POST",body = option,verbosity=verbosity)
      if (inherits(result, "openai_error")) {
        return(list(success=FALSE, message=result$get_message(), type=result$get_type()))
      }else{
        return(result$data)
      }
    },
    #' @description Creates an edited or extended image given an original image and a prompt.
    #' @param image The image to edit. Must be a valid PNG file, less than 4MB, and square. If mask is not provided, image must have transparency, which will be used as the mask.
    #' @param prompt A text description of the desired image(s). The maximum length is 1000 characters.
    #' @param verbosity Verbosity level for the API call.
    #' @param ...  Additional parameters as required by the OpenAI API.For example:mask;model;n...
    #' @return Returns a list of image objects.
    images_edits=function(image,prompt,verbosity=0,...){
      option <- list(...)
      option$prompt <- prompt
      option$image<-curl::form_file(image)
      if(!is.null(option$mask)){
        option$mask<-curl::form_file(option$mask)
      }
      result<-private$file_call(endpoint = "images",path = "/edits",body = option,verbosity=verbosity)
      if (inherits(result, "openai_error")) {
        return(list(success=FALSE, message=result$get_message(), type=result$get_type()))
      }else{
        return(result$data)
      }
    },
    #' @description Creates a variation of a given image.
    #' @param image The image to use as the basis for the variation(s). Must be a valid PNG file, less than 4MB, and square.
    #' @param verbosity Verbosity level for the API call.
    #' @param ...  Additional parameters as required by the OpenAI API.For example:model;n;response_format
    #' @return Returns a list of image objects.
    images_variations=function(image,verbosity=0,...){
      option <- list(...)
      option$image<-curl::form_file(image)
      result<-private$file_call(endpoint = "images",path = "/variations",body = option,verbosity=verbosity)
      if (inherits(result, "openai_error")) {
        return(list(success=FALSE, message=result$get_message(), type=result$get_type()))
      }else{
        return(result$data)
      }
    }
  )
)

