##' Bivariate polarAnnulus plot
##'
##' Typically plots the concentration of a pollutant by wind direction and as a
##' function of time as an annulus. The function is good for visualising how
##' concentrations of pollutants vary by wind direction and a time period e.g.
##' by month, day of week.
##'
##' The \code{polarAnnulus} function shares many of the properties of the
##' \code{polarPlot}. However, \code{polarAnnulus} is focussed on displaying
##' information on how concentrations of a pollutant (values of another
##' variable) vary with wind direction and time. Plotting as an annulus helps
##' to reduce compression of information towards the centre of the plot. The
##' circular plot is easy to interpret because wind direction is most easily
##' understood in polar rather than Cartesian coordinates.
##'
##' The inner part of the annulus represents the earliest time and the outer
##' part of the annulus the latest time. The time dimension can be shown in
##' many ways including "trend", "hour" (hour or day), "season" (month of the
##' year) and "weekday" (day of the week). Taking hour as an example, the plot
##' will show how concentrations vary by hour of the day and wind direction.
##' Such plots can be very useful for understanding how different source
##' influences affect a location.
##'
##' For \code{type = "trend"} the amount of smoothing does not vary linearly
##' with the length of the time series i.e. a certain amount of smoothing per
##' unit interval in time. This is a deliberate choice because should one be
##' interested in a subset (in time) of data, more detail will be provided for
##' the subset compared with the full data set. This allows users to
##' investigate specific periods in more detail. Full flexibility is given
##' through the smoothing parameter \code{k}.
##'
##' @param mydata A data frame minimally containing \code{date}, \code{wd} and
##'   a pollutant.
##' @param pollutant Mandatory. A pollutant name corresponding to a variable in
##'   a data frame should be supplied e.g. \code{pollutant = "nox"}. There can
##'   also be more than one pollutant specified e.g. \code{pollutant = c("nox",
##'   "no2")}. The main use of using two or more pollutants is for model
##'   evaluation where two species would be expected to have similar
##'   concentrations. This saves the user stacking the data and it is possible
##'   to work with columns of data directly. A typical use would be
##'   \code{pollutant = c("obs", "mod")} to compare two columns "obs" (the
##'   observations) and "mod" (modelled values).
##' @param resolution Two plot resolutions can be set: \code{"normal"} and
##'   \code{"fine"} (the default).
##' @param local.time Should the results be calculated in local time? The
##'   default is \code{TRUE}. Emissions activity tends to occur at local time
##'   e.g. rush hour is at 8 am every day. When the clocks go forward in
##'   spring, the emissions are effectively released into the atmosphere at BST
##'   - 1 hour during the summer. When plotting diurnal profiles, this has the
##'   effect of "smearing-out" the concentrations. A better approach is to
##'   express time as local time, which here is defined as BST (British Summer
##'   Time). This correction tends to produce better-defined diurnal profiles
##'   of concentration (or other variables) and allows a better comparison to
##'   be made with emissions/activity data. If set to \code{FALSE} then GMT is
##'   used.
##' @param period This determines the temporal period to consider. Options are
##'   "hour" (the default, to plot diurnal variations), "season" to plot
##'   variation throughout the year, "weekday" to plot day of the week
##'   variation and "trend" to plot the trend by wind direction.
##' @param type \code{type} determines how the data are split i.e. conditioned,
##'   and then plotted. The default is will produce a single plot using the
##'   entire data. Type can be one of the built-in types as detailed in
##'   \code{cutData} e.g. "season", "year", "weekday" and so on. For example,
##'   \code{type = "season"} will produce four plots --- one for each season.
##'
##' It is also possible to choose \code{type} as another variable in the data
##'   frame. If that variable is numeric, then the data will be split into four
##'   quantiles (if possible) and labelled accordingly. If type is an existing
##'   character or factor variable, then those categories/levels will be used
##'   directly. This offers great flexibility for understanding the variation
##'   of different variables and how they depend on one another.
##'
##' Type can be up length two e.g. \code{type = c("season", "site")} will
##'   produce a 2x2 plot split by season and site. The use of two types is
##'   mostly meant for situations where there are several sites. Note, when two
##'   types are provided the first forms the columns and the second the rows.
##'
##' Also note that for the \code{polarAnnulus} function some type/period
##'   combinations are forbidden or make little sense. For example, \code{type
##'   = "season"} and \code{period = "trend"} (which would result in a plot
##'   with too many gaps in it for sensible smoothing), or \code{type =
##'   "weekday"} and \code{period = "weekday"}.
##'
##' @param limits Limits for colour scale.
##' @param cols Colours to be used for plotting. Options include "default",
##'   "increment", "heat", "jet" and user defined. For user defined the user
##'   can supply a list of colour names recognised by R (type \code{colours()}
##'   to see the full list). An example would be \code{cols = c("yellow",
##'   "green", "blue")}
##' @param width The width of the annulus; can be "normal" (the default),
##'   "thin" or "fat".
##' @param min.bin The minimum number of points allowed in a wind speed/wind
##'   direction bin.  The default is 1. A value of two requires at least 2
##'   valid records in each bin an so on; bins with less than 2 valid records
##'   are set to NA. Care should be taken when using a value > 1 because of the
##'   risk of removing real data points. It is recommended to consider your
##'   data with care. Also, the \code{polarFreq} function can be of use in such
##'   circumstances.
##' @param exclude.missing Setting this option to \code{TRUE} (the default)
##'   removes points from the plot that are too far from the original data. The
##'   smoothing routines will produce predictions at points where no data exist
##'   i.e. they predict. By removing the points too far from the original data
##'   produces a plot where it is clear where the original data lie. If set to
##'   \code{FALSE} missing data will be interpolated.
##' @param date.pad For \code{type = "trend"} (default), \code{date.pad = TRUE}
##'   will pad-out missing data to the beginning of the first year and the end
##'   of the last year. The purpose is to ensure that the trend plot begins and
##'   ends at the beginning or end of year.
##' @param force.positive The default is \code{TRUE}. Sometimes if smoothing
##'   data with steep gradients it is possible for predicted values to be
##'   negative. \code{force.positive = TRUE} ensures that predictions remain
##'   postive. This is useful for several reasons. First, with lots of missing
##'   data more interpolation is needed and this can result in artifacts
##'   because the predictions are too far from the original data. Second, if it
##'   is known beforehand that the data are all postive, then this option
##'   carries that assumption through to the prediction. The only likely time
##'   where setting \code{force.positive = FALSE} would be if background
##'   concentrations were first subtracted resulting in data that is
##'   legitimately negative. For the vast majority of situations it is expected
##'   that the user will not need to alter the default option.
##' @param k The smoothing value supplied to \code{gam} for the temporal and
##'   wind direction components, respectively. In some cases e.g. a trend plot
##'   with less than 1-year of data the smoothing with the default values may
##'   become too noisy and affected more by outliers. Choosing a lower value of
##'   \code{k} (say 10) may help produce a better plot.
##' @param normalise If \code{TRUE} concentrations are normalised by dividing
##'   by their mean value. This is done \emph{after} fitting the smooth
##'   surface. This option is particularly useful if one is interested in the
##'   patterns of concentrations for several pollutants on different scales
##'   e.g. NOx and CO. Often useful if more than one \code{pollutant} is
##'   chosen.
##' @param key.header Adds additional text/labels to the scale key.
##'   For example, passing the options \code{key.header = "header", key.footer
##'   = "footer1"} adds addition text above and below the scale key. These
##'   arguments are passed to \code{drawOpenKey} via \code{quickText}, applying
##'   the \code{auto.text} argument, to handle formatting.
##' @param key.footer see \code{key.header}.
##' @param key.position Location where the scale key is to plotted.  Allowed
##'   arguments currently include \code{"top"}, \code{"right"}, \code{"bottom"}
##'   and \code{"left"}.
##' @param key Fine control of the scale key via \code{drawOpenKey}. See
##'   \code{drawOpenKey} for further details.
##' @param auto.text Either \code{TRUE} (default) or \code{FALSE}. If
##'   \code{TRUE} titles and axis labels will automatically try and format
##'   pollutant names and units properly e.g.  by subscripting the \sQuote{2}
##'   in NO2.
##' @param ... Other graphical parameters passed onto \code{lattice:levelplot}
##'   and \code{cutData}. For example, \code{polarAnnulus} passes the option
##'   \code{hemisphere = "southern"} on to \code{cutData} to provide southern
##'   (rather than default northern) hemisphere handling of \code{type = "season"}.
##'   Similarly, common axis and title labelling options (such as \code{xlab},
##'   \code{ylab}, \code{main}) are passed to \code{levelplot} via \code{quickText}
##'   to handle routine formatting.
##' @export
##' @return As well as generating the plot itself, \code{polarAnnulus} also
##'   returns an object of class ``openair''. The object includes three main
##'   components: \code{call}, the command used to generate the plot;
##'   \code{data}, the data frame of summarised information used to make the
##'   plot; and \code{plot}, the plot itself. If retained, e.g. using
##'   \code{output <- polarAnnulus(mydata, "nox")}, this output can be used to
##'   recover the data, reproduce or rework the original plot or undertake
##'   further analysis.
##'
##' An openair output can be manipulated using a number of generic operations,
##'   including \code{print}, \code{plot} and \code{summary}. See
##'   \code{\link{openair.generics}} for further details.
##' @author David Carslaw
##' @seealso \code{\link{polarPlot}}, \code{\link{polarFreq}},
##'   \code{\link{pollutionRose}} and \code{\link{percentileRose}}
##' @keywords methods
##' @examples
##'
##'
##' # load example data from package
##' data(mydata)
##'
##' # diurnal plot for PM10 at Marylebone Rd
##' polarAnnulus(mydata, pollutant = "pm10", main = "diurnal variation in pm10 at Marylebone Road")
##'
##' # seasonal plot for PM10 at Marylebone Rd
##' \dontrun{polarAnnulus(mydata, poll="pm10", period = "season")}
##'
##' # trend in coarse particles (PMc = PM10 - PM2.5), calculate PMc first
##'
##' mydata$pmc <- mydata$pm10 - mydata$pm25
##' \dontrun{polarAnnulus(mydata, poll="pmc", period = "trend", main = "trend in pmc at Marylebone Road")}
##'
##'
polarAnnulus <- function(mydata,
                         pollutant = "nox",
                         resolution = "fine",
                         local.time = FALSE,
                         period = "hour",
                         type = "default",
                         limits = c(0, 100),
                         cols = "default",
                         width = "normal",
                         min.bin = 1,
                         exclude.missing = TRUE,
                         date.pad = FALSE,
                         force.positive = TRUE,
                         k = c(20, 10),
                         normalise = FALSE,
                         key.header = "",
                         key.footer = pollutant,
                         key.position = "right",
                         key = TRUE,
                         auto.text = TRUE,...) {


    ## extract variables of interest
    vars <- c("wd", "date", pollutant)

    if (period == "trend" & "season" %in% type) stop ("Cannot have same type as 'season' and period as 'trend'.")
    if (length(type) > 2) stop("Cannot have more than two types.")

    ## greyscale handling
    if (length(cols) == 1 && cols == "greyscale") {
        ## strip only
        current.strip <- trellis.par.get("strip.background")
        trellis.par.set(list(strip.background = list(col = "white")))
    }

    ##extra.args setup
    extra.args <- list(...)

    #label controls
    extra.args$xlab <- if("xlab" %in% names(extra.args))
                           quickText(extra.args$xlab, auto.text) else quickText("", auto.text)
    extra.args$ylab <- if("ylab" %in% names(extra.args))
                           quickText(extra.args$ylab, auto.text) else quickText("", auto.text)
    extra.args$main <- if("main" %in% names(extra.args))
                           quickText(extra.args$main, auto.text) else quickText("", auto.text)


    ## check data
    mydata <- openair:::checkPrep(mydata, vars, type, remove.calm = FALSE)

    ## if more than one pollutant, need to stack the data and set type = "variable"
    ## this case is most relevent for model-measurement compasrions where data are in columns
    if (length(pollutant) > 1) {
        mydata <- melt(mydata, measure.vars = pollutant)
        ## now set pollutant to "value"
        pollutant <- "value"
        type <- "variable"
    }


    d <- 10      ## d + upper = 1/2 width of annulus; do not need to adjust

    if (width == "normal") upper <- 10
    if (width == "thin") upper <- 15
    if (width == "fat") upper <- 5

    ## add extra wds - reduces discontinuity at 0/360
    zero.wd <- subset(mydata, wd == 360)

    if (nrow(zero.wd) > 0) {
        zero.wd$wd <- 0
        mydata <- rbind(mydata, zero.wd)
    }


    ## remove NAs
    mydata <- na.omit(mydata)
    mydata <- cutData(mydata, type, ...)

    ## convert to local time
    if (local.time) attr(mydata$date, "tzone") <- "Europe/London"

    ## for resolution of grid plotting (default = 0.2; fine = 0.1)
    if (resolution == "normal") int <- 0.2
    if (resolution == "fine") int <- 0.1
    if (resolution == "ultra.fine") int <- 0.05  # very large files!

    len.int <- 20 / int + 1 ## number of x and y points to make up surfacexb

    prepare.grid <- function(mydata) {

        ## for padding to beginning of first year, end of last year
        if (date.pad) {

            min.year <- as.numeric(format(min(mydata$date, na.rm = TRUE), "%Y"))
            max.year <- as.numeric(format(max(mydata$date, na.rm = TRUE), "%Y"))

            all.dates <- data.frame(date = seq(ISOdate(min.year, 1, 1, 0, 0, 0, tz = "GMT"),
                                    ISOdate(max.year, 12, 31, 23, 0, 0, tz = "GMT"),
                                    by = "hour"))

            all.dates <- data.frame(date = all.dates)
        }

        ## different date components, others available
        if (period == "trend")
        {
            if (date.pad) {
                ## for new limits with padding
                day <- as.numeric(format(all.dates$date, "%j"))
                year <- as.numeric(format(all.dates$date, "%Y"))
                trend2 <- year + day / 366
                min.trend <- min(trend2, na.rm = TRUE)
                max.trend <- max(trend2, na.rm = TRUE)

                ## actual data
                day <- as.numeric(format(mydata$date, "%j"))
                year <- as.numeric(format(mydata$date, "%Y"))
                trend <- year + day / 366

            } else {

                year <- as.numeric(format(mydata$date, "%Y"))
                day <- as.numeric(format(mydata$date, "%j"))
                trend <- year + day / 366
                min.trend <- min(trend, na.rm = TRUE)
                max.trend <- max(trend, na.rm = TRUE)
            }
        }

        if (period == "weekday")
        {
            hour <- as.numeric(format(mydata$date, "%H"))
            weekday <- as.numeric(format(mydata$date, "%w"))
            trend <- weekday + hour / 23
            min.trend <- 0
            max.trend <- 7
        }

        if (period == "season")
        {
            week <- as.numeric(format(mydata$date, "%W"))
            trend <- week
            min.trend <- 0
            max.trend <- 53
        }

        if (period == "hour")
        {
            hour <- as.numeric(format(mydata$date, "%H"))
            trend <- hour
            min.trend <- 0
            max.trend <- 23
        }

        trend <- 10 * (trend - min.trend) / (max.trend - min.trend)

        mydata <- cbind(mydata, trend)

        time.seq <- seq(0, 10, length = 24)

        wd <- seq(from = 0, to = 360, 10) #wind directions from 10 to 360
        ws.wd <- expand.grid(time.seq = time.seq, wd = wd)


        ## identify which ws and wd bins the data belong
        ## wd.cut <- cut(mydata$wd, seq(0, 360, 10))
        wd.cut <- cut(mydata$wd, seq(0, 360, length = 38), include.lowest = TRUE)

        ## divide-up the data for the annulus
        time.cut <- cut(mydata$trend, seq(0, 10, length = 25), include.lowest = TRUE)

        binned <- tapply(mydata[, pollutant], list(time.cut, wd.cut), mean, na.rm = TRUE)
        binned <- as.vector(binned)

        ## frequency - remove points with freq < min.bin
        bin.len <- tapply(mydata[, pollutant], list(time.cut, wd.cut), length)
        binned.len <- as.vector(bin.len)

        ids <- which(binned.len < min.bin)
        binned[ids] <- NA

        ## data to predict over
        time.seq <- ws.wd$time.seq
        wd <- ws.wd$wd

        input.data <- expand.grid(time.seq = seq(0, 10, length = len.int),
                                  wd = seq(0, 360, length = len.int))
######################Smoothing#################################################

        ## run GAM to make a smooth surface
        if (force.positive) n <- 0.5 else n <- 1

        input <- data.frame(binned, time.seq, wd)

        ## note use of cyclic smooth for the wind direction component
        Mgam <- gam(binned ^ n ~ te(time.seq, wd, k = k, bs = c("tp", "cc")), data = input)


        pred <- predict.gam(Mgam, input.data)
        pred <- pred ^ (1 / n)

        input.data <- cbind(input.data, pred)

        if (exclude.missing) {

            ## exclude predictions too far from data (from mgcv)
            x <- seq(0, 10, length = len.int)
            y <- seq(0, 360, length = len.int)
            res <- len.int
            wsp <- rep(x, res)
            wdp <- rep(y, rep(res, res))

          #  ind <- exclude.too.far(wsp, wdp, mydata$trend, mydata$wd, dist = 0.03)
            ## data with gaps caused by min.bin
            all.data <- na.omit(data.frame(time.seq = ws.wd$time.seq, wd = ws.wd$wd, binned))
            ind <- with(all.data, exclude.too.far(wsp, wdp, time.seq, wd, dist = 0.03))

            input.data$pred[ind] <- NA
            pred <- input.data$pred
        }

#############################################################################
        ## transform to new coords - probably more efficient way of doing this
        ## need to transform to/from annulus to a cartesian coords

        ## new grid for plotting (the annulus)
        new.data <- expand.grid(u = seq(-upper - d, upper + d, by = int),
                                v = seq(-upper - d, upper + d, by = int), z = NA, wd = NA,
                                time.val = NA)

        new.data$id <- seq(nrow(new.data))

        ## calculate wd and time.val in on annulus in original units (helps with debugging)
        ## essentially start with final grid (annulus) and work backwards to find original data point in
        ## orginal coordinates
        new.data <- within(new.data, time.val <- (u ^ 2 + v ^ 2) ^ 0.5  - upper)
        new.data <- within(new.data, wd <- 180 * atan2(u , v) / pi)
        new.data <- within(new.data, wd[wd < 0] <- wd[wd < 0] + 360)  ## correct negative wds

        ## remove ids outside range
        ids <- with(new.data, which((u ^ 2 + v ^ 2) ^ 0.5 > d + upper | (u ^ 2 + v ^ 2) ^ 0.5
                                    < upper))
        new.data$wd[ids] <- NA
        new.data$time.val[ids] <- NA

        ## ids where there are data
        ids <- new.data$id[!is.na(new.data$wd)]

        ## indexes in orginal (cartesian) coordinates
        id.time <- round((2 * d / int) * new.data$time.val[ids] / 10) + 1
        id.wd <- round((2 * d / int) * new.data$wd[ids] / 360) + 1

        ## predicted values as a matrix
        pred <- matrix(pred, nrow = len.int, ncol = len.int)

        ## match the ids with predictions
        new.data$z[ids] <- sapply(seq_along(ids), function(x) pred[id.time[x], id.wd[x]])

        new.data
    }

    ## more compact way?  Need to test
    results.grid <- ddply(mydata, type, prepare.grid)


    ## normalise by divining by mean conditioning value if needed
    if (normalise){
        results.grid <- ddply(results.grid, type, transform, z = z / mean(z, na.rm = TRUE))
        if (missing(key.footer)) key.footer <- "normalised \nlevel"
    }

    ## proper names of labelling ##############################################################################
    pol.name <- sapply(levels(results.grid[ , type[1]]), function(x) quickText(x, auto.text))
    strip <- strip.custom(factor.levels = pol.name)

    if (length(type) == 1 ) {

        strip.left <- FALSE

    } else { ## two conditioning variables

        pol.name <- sapply(levels(results.grid[ , type[2]]), function(x) quickText(x, auto.text))
        strip.left <- strip.custom(factor.levels = pol.name)
    }
    if (length(type) == 1 & type[1] == "default") strip <- FALSE ## remove strip

########################################################################################################


    ## auto-scaling
    nlev = 200  #preferred number of intervals
    ## handle missing breaks arguments
    if(missing(limits)) breaks <- pretty(results.grid$z, n = nlev) else breaks <- pretty(limits,
                                                         n = nlev)

    nlev2 = length(breaks)

    col <- openColours(cols, (nlev2 - 1))
    col.scale = breaks

#################
    ## scale key setup
#################
    legend <- list(col = col, at = col.scale, space = key.position,
                   auto.text = auto.text, footer = key.footer, header = key.header,
                   height = 1, width = 1.5, fit = "all")
    legend <- openair:::makeOpenKeyLegend(key, legend, "polarAnnulus")

    temp <- paste(type, collapse = "+")
    myform <- formula(paste("z ~ u * v | ", temp, sep = ""))

    levelplot.args <- list(x = myform, results.grid, axes = FALSE,
                     as.table = TRUE,
                     aspect = 1,
                     colorkey = FALSE, legend = legend,
                     at = col.scale, col.regions = col,
                     par.strip.text = list(cex = 0.8),
                     scales = list(draw = FALSE),
                     strip = strip,

                     len <- upper + d + 3,
                     xlim = c(-len, len), ylim = c(-len, len),

                     panel = function(x, y, z,subscripts,...) {
                         panel.levelplot(x, y, z, subscripts, at = col.scale,
                                         lwd = 1, col.regions = col, labels = FALSE)

                         ## add axis line to central polarPlot
                         llines(c(upper, upper + d), c(0, 0), col = "grey20")
                         llines(c(0, 0), c(-upper, -upper - d), col = "grey20")

                         ## add axis line to central polarPlot
                         llines(c(0, 0), c(upper, upper + d), col = "grey20")
                         llines(c(-upper, -upper - d), c(0, 0), col = "grey20")

                         add.tick <- function(n, start = 0, end = 0) {
                             ## start is an offset if time series does not begin in January

                             ## left
                             lsegments(seq(-upper - start, -upper - d + end, length = n),
                                       rep(-.5, n),
                                       seq(-upper - start, -upper - d + end, length = n),
                                       rep(.5, n), col = "grey20")
                             ## top
                             lsegments(rep(-.5, n),
                                       seq(upper + start, upper + d - end, length = n),
                                       rep(.5, n),
                                       seq(upper + start, upper + d - end, length = n))
                             ## right
                             lsegments(seq(upper + start, upper + d - end, length = n),
                                       rep(-.5, n),
                                       seq(upper + start, upper + d - end, length = n),
                                       rep(.5, n), col = "grey20")
                             ## bottom
                             lsegments(rep(-.5, n),
                                       seq(-upper - start, -upper - d + end, length = n),
                                       rep(.5, n),
                                       seq(-upper - start, -upper - d + end, length = n))
                         }

                         label.axis <- function(x, lab1, lab2, ticks)
                         {
                             ltext(x, upper, lab1, cex = 0.7, pos = 4)
                             ltext(x, upper + d, lab2, cex = 0.7, pos = 4)
                             ## at bottom
                             ltext(-x, -upper, lab1, cex = 0.7, pos = 2)
                             ltext(-x, -upper - d, lab2, cex = 0.7, pos = 2)
                             add.tick(ticks)
                         }

                         if (period == "trend")
                         {
                             if (date.pad) {
                                 date.start <- min(all.dates$date)
                                 date.end <- max(all.dates$date)
                             } else {
                                 date.start <- min(mydata$date)
                                 date.end <- max(mydata$date)
                             }

                             label.axis(0, format(date.start, "%d-%b-%Y"), format(date.end, "%d-%b-%Y"), 2)

                             ## add ticks at 1-Jan each year (could be missing dates)
                             days <- seq(as.Date(date.start), as.Date(date.end), by = "day")

                             if (length(days) > 365) {
                                 ## find number of Januarys
                                 num.jans <- which(format(days, "%j") == "001")
                                 ticks <- length(num.jans)
                                 start <- 10 * (num.jans[1] - 1) / length(days)
                                 end <- 10 - 10 * (num.jans[ticks] - 1) / length(days)
                                 if (length(num.jans) > 0) add.tick(ticks, start, end)

                                 ## mark montly intervals (approx)
                             } else {
                                 ## find number of 01s
                                 num.jans <- which(format(days, "%d") == "01")
                                 ticks <- length(num.jans)
                                 start <- 10 * (num.jans[1] - 1) / length(days)
                                 end <- 10 - 10 * (num.jans[ticks] - 1) / length(days)
                                 if (length(num.jans) > 0) add.tick(ticks, start, end)

                             }
                         }

                         if (period == "season") label.axis(0, format(ISOdate(2000, 1, 1), "%B"),
                             format(ISOdate(2000, 12, 1), "%B"), 13)

                         if (period == "hour")	label.axis(0, "0", "23", 7)

                         if (period == "weekday") {
                             local.weekdays <- format(ISOdate(2000, 1, 1:14), "%A")[order(format(ISOdate(2000, 1, 1:14), "%w"))]
                             loc.sunday <- local.weekdays[1]
                             loc.saturday <- local.weekdays[length(local.weekdays)]
                             label.axis(0, loc.sunday, loc.saturday, 8)
                         }

                         ## text for directions
                         ltext(-upper -d - 1.5, 0, "W", cex = 0.7)
                         ltext(0, -upper - d - 1.5, "S", cex = 0.7)
                         ltext(0, upper + d + 1.5, "N", cex = 0.7)
                         ltext(upper + d + 1.5, 0, "E", cex = 0.7)

                     })

    #reset for extra.args
    levelplot.args<- openair:::listUpdate(levelplot.args, extra.args)

    #plot
    plt <- do.call(levelplot, levelplot.args)


#################
                                        #output
#################
    if (length(type) == 1) plot(plt) else plot(useOuterStrips(plt, strip = strip, strip.left = strip.left))
    newdata <- results.grid
    output <- list(plot = plt, data = newdata, call = match.call())
    class(output) <- "openair"

                                        #reset if greyscale
    if (length(cols) == 1 && cols == "greyscale")
        trellis.par.set("strip.background", current.strip)

    invisible(output)

}


