% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulate_regression.R
\name{simulate_regression}
\alias{simulate_regression}
\title{Simulates a 3D Spherical Regression.}
\usage{
simulate_regression(
  explanatory_points,
  local_rotation_composer,
  local_error_sampler
)
}
\arguments{
\item{explanatory_points}{An \emph{m}-by-3 matrix whose rows contain
the Cartesian coordinates of the points at which the regression
will be simulated.}

\item{local_rotation_composer}{A function that returns a
3-length numeric vector representing the independent components of a
skew symmetric matrix local to an explanatory point, given its
Cartesian coordinates.}

\item{local_error_sampler}{A function that returns a 3-length numeric vector
representing a sampled error term local to an explanatory point,
given its Cartesian coordinates.}
}
\value{
An \emph{m}-by-\emph{3} matrix whose rows contain
the Cartesian coordinates of the response points corresponding
to the explanatory points.
}
\description{
Returns the response points corresponding to the specified explanatory
points, given a model for local rotations
and an error term sampler.
}
\details{
Let \eqn{E} be
the \emph{m}-by-3 matrix of explanatory points.
This function will return
an \emph{m}-by-\emph{3} matrix whose \eqn{i}-th row is obtained by
transposition of the following expression:

\deqn{exp(\Phi(\epsilon(x))) exp(\Phi(s(x))) x}

where \eqn{x} is the transpose of the \eqn{i}-th row of
\eqn{E}. Terms \eqn{\epsilon(x)} and \eqn{s(x)} are obtained by
evaluating at \eqn{x} functions \code{local_error_sampler} and
\code{local_rotation_composer}, respectively, while
matrix \eqn{\Phi(c)}, for a 3-length numeric vector \eqn{c}, is
the skew symmetric matrix having its independent components
represented by the entries of \eqn{c} (for a thorough discussion,
see function
\code{\link{get_skew_symmetric_matrix}}).

Functions \code{local_error_sampler} and \code{local_rotation_composer}
must be prototyped as having one argument, \code{point},
representing the Cartesian coordinates of a point on a 3D sphere,
and returning a non \code{NULL} numerical object having length
equal to \code{3}.
}
\examples{
library(nprotreg)

# Define a matrix of explanatory points.

explanatory_points <- rbind(
  cbind(.5, 0, .8660254),
  cbind(-.5, 0, .8660254),
  cbind(1, 0, 0),
  cbind(0, 1, 0),
  cbind(-1, 0, 0),
  cbind(0, -1, 0),
  cbind(.5, 0, -.8660254),
  cbind(-.5, 0, -.8660254)
)

# Define a local rotation composer.

local_rotation_composer <- function(point) {
  independent_components <- (1 / 2) *
    c(exp(2.0 * point[3]), - exp(2.0 * point[2]), exp(2.0 * point[1]))
}

# Define a local error sampler.

local_error_sampler <- function(point) {
  rnorm(3)
}

# Get the corresponding 8-by-3 matrix of response points.
# Rows corresponds to explanatory points,
# columns to Cartesian coordinates.

response_points <- simulate_regression(explanatory_points,
                                       local_rotation_composer,
                                       local_error_sampler)

# Get the response point corresponding to the second
# explanatory point.

cat("Response point corresponding to the second explanatory point: \n")
cat(response_points[2, ], "\n")
}
\seealso{
Other Regression functions: 
\code{\link{cross_validate_concentration}()},
\code{\link{fit_regression}()},
\code{\link{get_equally_spaced_points}()},
\code{\link{get_skew_symmetric_matrix}()},
\code{\link{simulate_rigid_regression}()},
\code{\link{weight_explanatory_points}()}
}
\concept{Regression functions}
