% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/glmer.mp.con.R
\name{glmer.mp.con}
\alias{glmer.mp.con}
\title{Contrast tests for multinomial-Poisson GLMMs}
\usage{
glmer.mp.con(
  model,
  formula,
  adjust = c("holm", "hochberg", "hommel", "bonferroni", "BH", "BY", "fdr", "none"),
  ...
)
}
\arguments{
\item{model}{A multinomial-Poisson generalized linear mixed model created by \code{\link{glmer.mp}}.}

\item{formula}{A formula object in the style of, e.g., \code{pairwise ~ X1*X2}, where \code{X1} and
\code{X2} are factors in \code{model}. The \code{pairwise} keyword \strong{must} be used on the left-hand
side of the formula. See the \code{specs} entry for \code{\link[emmeans]{emmeans}}.}

\item{adjust}{A string indicating the \emph{p}-value adjustment to use. Defaults to \code{"holm"}. See the
Details section for \code{\link[stats]{p.adjust}}.}

\item{...}{Additional arguments to be passed to \code{\link[lme4]{glmer}}. Generally, these are
unnecessary but are provided for advanced users. They should not specify \code{formula}, \code{data},
or \code{family} arguments. See \code{\link[lme4]{glmer}} for valid arguments.}
}
\value{
Pairwise comparisons for all levels indicated by the factors in \code{formula}.
}
\description{
This function conducts \emph{post hoc} pairwise comparisons on generalized linear mixed models (GLMMs)
built with \code{\link{glmer.mp}}. Such models have \strong{nominal response} types, i.e., \code{factor}s
with unordered categories.
}
\details{
\emph{Post hoc} pairwise comparisons should be conducted \emph{only} after a statistically significant
omnibus test using \code{\link{Anova.mp}}. Comparisons are conducted in the style of
\code{\link[emmeans]{emmeans}} but not using this function; rather, the multinomial-Poisson trick is used
on a subset of the data relevant to each pairwise comparison.

Users wishing to verify the correctness of \code{glmer.mp.con} should compare its results to
\code{\link[emmeans]{emmeans}} results for models built with \code{\link[lme4]{glmer}} using
\code{family=binomial} for dichotomous responses. Factor contrasts should be set to sum-to-zero
contrasts (i.e., \code{"contr.sum"}).
}
\note{
It is common to receive a \code{boundary (singular) fit} message. This generally can be ignored
provided the test output looks sensible. Less commonly, the procedure can fail to converge, which
can happen when counts of one or more categories are very small or zero in some conditions. In such
cases, any results should be regarded with caution.

Sometimes, convergence issues can be remedied by changing the optimizer or its control parameters.
For example, the following code uses the \code{bobyqa} optimizer instead of the \code{Nelder_Mead} optimizer
and increases the maximum number of function evaluations:

\code{m = glmer.mp.con(m, pairwise ~ X1*X2, adjust="holm", control=glmerControl(optimizer="bobyqa", optCtrl=list(maxfun=100000)))}

See \code{\link[lme4]{glmerControl}} for more information.
}
\examples{
library(car)
library(lme4)
library(lmerTest)
library(emmeans)

## within-subjects factors (x1,X2) with dichotomous response (Y)
data(ws2, package="multpois")

ws2$PId = factor(ws2$PId)
ws2$Y = factor(ws2$Y)
ws2$X1 = factor(ws2$X1)
ws2$X2 = factor(ws2$X2)
contrasts(ws2$X1) <- "contr.sum"
contrasts(ws2$X2) <- "contr.sum"

m1 = glmer(Y ~ X1*X2 + (1|PId), data=ws2, family=binomial)
Anova(m1, type=3)
emmeans(m1, pairwise ~ X1*X2, adjust="holm")

m2 = glmer.mp(Y ~ X1*X2 + (1|PId), data=ws2)
Anova.mp(m2, type=3)
glmer.mp.con(m2, pairwise ~ X1*X2, adjust="holm") # compare

\donttest{
## within-subjects factors (x1,X2) with polytomous response (Y)
data(ws3, package="multpois")

ws3$PId = factor(ws3$PId)
ws3$Y = factor(ws3$Y)
ws3$X1 = factor(ws3$X1)
ws3$X2 = factor(ws3$X2)
contrasts(ws3$X1) <- "contr.sum"
contrasts(ws3$X2) <- "contr.sum"

m3 = glmer.mp(Y ~ X1*X2 + (1|PId), data=ws3)
Anova.mp(m3, type=3)
glmer.mp.con(m3, pairwise ~ X1*X2, adjust="holm")
}

}
\references{
Baker, S.G. (1994). The multinomial-Poisson transformation.
\emph{The Statistician 43} (4), pp. 495-504. \doi{10.2307/2348134}

Chen, Z. and Kuo, L. (2001). A note on the estimation of the
multinomial logit model with random effects. \emph{The American Statistician
55} (2), pp. 89-95. \url{https://www.jstor.org/stable/2685993}

Guimaraes, P. (2004). Understanding the multinomial-Poisson
transformation. \emph{The Stata Journal 4} (3), pp. 265-273.
\url{https://www.stata-journal.com/article.html?article=st0069}

Lee, J.Y.L., Green, P.J.,and Ryan, L.M. (2017). On the “Poisson
trick” and its extensions for fitting multinomial regression models. \emph{arXiv
preprint} available at \doi{10.48550/arXiv.1707.08538}
}
\seealso{
\code{\link[=Anova.mp]{Anova.mp()}}, \code{\link[=glmer.mp]{glmer.mp()}}, \code{\link[=glm.mp]{glm.mp()}}, \code{\link[=glm.mp.con]{glm.mp.con()}}, \code{\link[emmeans:emmeans]{emmeans::emmeans()}}
}
\author{
Jacob O. Wobbrock
}
