% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/callFromMap.R
\name{callFromMap}
\alias{callFromMap}
\title{Call markers based on an existing map}
\usage{
callFromMap(
  rawData,
  thresholdChromosomes = 100,
  thresholdAlleleClusters = c(1e-10, 1e-20, 1e-30, 1e-40),
  maxChromosomes = 2,
  existingImputations,
  tDistributionPValue = 0.6,
  useOnlyExtraImputationPoints = TRUE,
  ...
)
}
\arguments{
\item{rawData}{Raw data for a genetic marker.}

\item{thresholdChromosomes}{The test-statistic threshold for declaring a marker to be polymorphic on a chromosome.}

\item{thresholdAlleleClusters}{The p-value threshold for declaring two underlying founder alleles to have different marker alleles. Multiple possible values should be input.}

\item{maxChromosomes}{The maximum number of chromosomes that a marker can be polymorphic on}

\item{existingImputations}{An object of class mpcrossMapped from the mpMap2 package, containing data about imputed underlying genotypes.}

\item{tDistributionPValue}{Paramater controlling the size of each detected cluster, ranging from 0 to 1. Small values result in small clusters, and large values result in large clusters.}

\item{useOnlyExtraImputationPoints}{Should we only use the non-marker positions to identify the correct locations?}

\item{...}{Extra arguments. Only \code{existingLocalisationStatistics} is supported, mostly so the example can run quickly.}
}
\value{
At the minimum, a list containing an entry \code{called} indicating whether the marker could be successfully called. If it could, other entries are returned. 
\describe{
	\item{overallAssignment}{Defines clusters within the data. }
	\item{classificationsPerPosition}{Defines genotype calls per genetic location to which the marker was mapped. }
	\item{clusterBoundaries}{Contours giving the boundaries of each cluster in \code{overallAssignment}.}
	\item{preliminaryGroups}{The preliminary groups based on IBD imputations, which the final genotype calls are built from.}
\item{pValuesMatrices}{The matrices of p-values used to form a graph, and therefore identify founder alleles.}
}
}
\description{
This function uses an existing genetic map to call genetic markers, including markers polymorphic on multiple chromosomes.
}
\details{
This function uses an existing genetic map to call a genetic marker. There are a number of advantages to this approach
\describe{
	\item{1.}{It can correctly call markers which are polymorphic on multiple chromosomes, therefore converting one marker into two.}
\item{2.}{It avoids incorrectly calling markers polymorphic on multiple chromosomes. Incorrect calling can lead to supurious genetic interactions.}
	\item{3.}{It can call markers that initially appear to be monomorphic in the population. }
	\item{4.}{It can call additional marker alleles for markers that would otherwise be ignored. }
}

Once a genetic map has been constructed, it should be used to impute underlying founder genotypes at an equally spaced grid of points using function \code{\link[mpMap2]{imputeFounders}}. 
The steps in the algorithm are as follows:
\describe{
	\item{1.}{Determine which chromosomes the marker is associated to, and where on those chromosomes. This is determined using function \code{\link[mpMap2]{addExtraMarkerFromRawCall}}, which is itself based on a manova model. 
  The marker is assumed associated to chromosomes for which the test statistic is greater than \code{thresholdChromosomes}. 
	  An appropriate value for \code{thresholdChromosomes} can be determined by looking at the results of \code{\link[mpMap2]{addExtraMarkerFromRawCall}}, for a number of different markers.}
	\item{2.}{Determine the distribution of marker alleles, at all the associated genetic locations. 
   This is done by taking the founders to be the vertices of a graph, and connecting founders which seem to part of the same marker allele. 
	  The resulting graph should be a union of disjoint complete graphs (cliques). }
	\item{3.}{We now have a preliminary assignment of marker alleles to lines, where the assignment may be of 1, 2, 3 or more \emph{different} marker alleles, depending on how many chromosomes the marker is associated with.
	  For example, if the marker is associated with two chromosomes, then there will be two marker alleles for each line. 
	  For each unique combination of marker alleles, we take the lines which have that assignment of marker alleles, and fit a skew-t distribution. }
	\item{4.}{For each fitted distribution, determine a confidence region using p-value \code{tDistributionPValue}. }
	\item{5.}{Use these confidence regions to construct marker calls at each associated location.}
}
}
\examples{
data(eightParentSubsetMap)
data(wsnp_Ku_rep_c103074_89904851)
data(callFromMapExampleLocalisationStatistics)
library(ggplot2)
library(gridExtra)
#We use an existing set of localisation statistics, to make the example faster
called <- callFromMap(rawData = as.matrix(wsnp_Ku_rep_c103074_89904851), existingImputations = 
    eightParentSubsetMap, useOnlyExtraImputationPoints = TRUE, tDistributionPValue = 0.8, 
    thresholdChromosomes = 80, existingLocalisationStatistics = existingLocalisationStatistics)
plotData <- wsnp_Ku_rep_c103074_89904851
plotData$genotype1B <- factor(called$classificationsPerPosition$Chr1BLoc31$finals)
plotData$imputed1B <- factor(imputationData(eightParentSubsetMap)[, "Chr1BLoc31"])
plotData$genotype1D <- factor(called$classificationsPerPosition$Chr1DLoc16$finals)
plotData$imputed1D <- factor(imputationData(eightParentSubsetMap)[, "Chr1DLoc16"])

plotImputations1B <- ggplot(plotData, mapping = aes(x = theta, y = r, color = imputed1B)) + 
    geom_point() + theme_bw() + ggtitle("Imputed genotype, 1B") + 
    guides(color=guide_legend(title="IBD genotype"))

called1B <- ggplot(plotData, mapping = aes(x = theta, y = r, color = genotype1B)) + 
    geom_point() + theme_bw() + ggtitle("Called genotype, 1B") + 
    guides(color=guide_legend(title="Called cluster")) + scale_color_manual(values = 
    c("black", RColorBrewer::brewer.pal(n = 4, name = "Set1")))

plotImputations1D <- ggplot(plotData, mapping = aes(x = theta, y = r, color = imputed1D)) + 
    geom_point() + theme_bw() + ggtitle("Imputed genotype, 1D") + 
    guides(color=guide_legend(title="IBD genotype"))

called1D <- ggplot(plotData, mapping = aes(x = theta, y = r, color = genotype1D)) + 
    geom_point() + theme_bw() + ggtitle("Called genotype, 1D") + 
    guides(color=guide_legend(title="Called cluster")) + 
    scale_color_manual(values = c("black",RColorBrewer::brewer.pal(n=3,name = "Set1")[1:2]))

\donttest{grid.arrange(plotImputations1B, plotImputations1D, called1B, called1D)}
}
