% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hv.R
\name{hypervolume}
\alias{hypervolume}
\title{Hypervolume metric}
\usage{
hypervolume(x, reference, maximise = FALSE)
}
\arguments{
\item{x}{\code{matrix()}|\code{data.frame()}\cr Matrix or data frame of numerical
values, where each row gives the coordinates of a point.}

\item{reference}{\code{numeric()}\cr Reference point as a vector of numerical
values.}

\item{maximise}{\code{logical()}\cr Whether the objectives must be maximised
instead of minimised. Either a single logical value that applies to all
objectives or a vector of logical values, with one value per objective.}
}
\value{
\code{numeric(1)} A single numerical value.
}
\description{
Compute the hypervolume metric with respect to a given reference point
assuming minimization of all objectives.
}
\details{
The hypervolume of a set of multidimensional points \eqn{A \subset
\mathbb{R}^d}{A in R^m} with respect to a reference point \eqn{\vec{r} \in
\mathbb{R}^d}{r in R^m} is the volume of the region dominated by the set and
bounded by the reference point \citep{ZitThi1998ppsn}.  Points in \eqn{A}
that do not strictly dominate \eqn{\vec{r}} do not contribute to the
hypervolume value, thus, ideally, the reference point must be strictly
dominated by all points in the true Pareto front.

More precisely, the hypervolume is the \href{https://en.wikipedia.org/wiki/Lebesgue_measure}{Lebesgue measure} of the union of
axis-aligned hyperrectangles
(\href{https://en.wikipedia.org/wiki/Hyperrectangle}{orthotopes}), where each
hyperrectangle is defined by one point from \eqn{\vec{a} \in A} and the
reference point.  The union of axis-aligned hyperrectangles is also called
an \emph{orthogonal polytope}.

The hypervolume is compatible with Pareto-optimality
\citep{KnoCor2002cec,ZitThiLauFon2003:tec}, that is, \eqn{\nexists A,B
\subset \mathbb{R}^m}{it does not exist A,B subsets of R^d}, such that
\eqn{A} is better than \eqn{B} in terms of Pareto-optimality and
\eqn{\text{hyp}(A) \leq \text{hyp}(B)}{hyp(A) <= hyp(B)}. In other words, if
a set is better than another in terms of Pareto-optimality, the hypervolume
of the former must be strictly larger than the hypervolume of the latter.
Conversely, if the hypervolume of a set is larger than the hypervolume of
another, then we know for sure than the latter set cannot be better than the
former in terms of Pareto-optimality.

For 2D and 3D, the algorithms used
\citep{FonPaqLop06:hypervolume,BeuFonLopPaqVah09:tec} have \eqn{O(n \log n)}
complexity, where \eqn{n} is the number of input points. The 3D case uses
the \eqn{\text{HV3D}^{+}} algorithm \citep{GueFon2017hv4d}, which has the
sample complexity as the HV3D algorithm
\citep{FonPaqLop06:hypervolume,BeuFonLopPaqVah09:tec}, but it is faster in
practice.

For 4D, the algorithm used is \eqn{\text{HV4D}^{+}} \citep{GueFon2017hv4d},
which has \eqn{O(n^2)} complexity.  Compared to the \href{https://github.com/apguerreiro/HVC/}{original implementation}, this implementation
correctly handles weakly dominated points and has been further optimized for
speed.

For 5D or higher, it uses a recursive algorithm
\citep{FonPaqLop06:hypervolume} with \eqn{\text{HV4D}^{+}} as the base case,
resulting in a \eqn{O(n^{d-2})} time complexity and \eqn{O(n)} space
complexity in the worst-case, where \eqn{d} is the dimension of the points.
Experimental results show that the pruning techniques used may reduce the
time complexity even further.  The original proposal
\citep{FonPaqLop06:hypervolume} had the HV3D algorithm as the base case,
giving a time complexity of \eqn{O(n^{d-2} \log n)}.  Andreia P. Guerreiro
enhanced the numerical stability of the algorithm by avoiding floating-point
comparisons of partial hypervolumes.
}
\examples{

data(SPEA2minstoptimeRichmond)
# The second objective must be maximized
# We calculate the hypervolume of the union of all sets.
hypervolume(SPEA2minstoptimeRichmond[, 1:2], reference = c(250, 0),
            maximise = c(FALSE, TRUE))

}
\references{
\insertAllCited{}
}
\author{
Manuel \enc{López-Ibáñez}{Lopez-Ibanez}
}
\concept{metrics}
