#' Create control structures for feature selection.
#' 
#' The following methods are available:
#'
#' \describe{
#'   \item{FeatSelControlExhaustive}{Exhaustive search. All feature sets (up to a certain size) are searched.}
#'   \item{FeatSelControlRandom}{Random search. Features vectors are randomly drawn.}
#'   \item{FeatSelControlSequential}{Deterministic forward or backward search.}
#'   \item{FeatSelControlGA}{Search via genetic algorithm.}
#' }
#' 
#' The GA is a simple (\code{mu}, \code{lambda}) or (\code{mu} + \code{lambda}) algorithm, depending on the
#' \code{comma} setting. 
#' A comma strategy selects a new population of size \code{mu} out of the \code{lambda} > \code{mu} offspring.
#' A plus strategy uses the joint pool of \code{mu} parents and \code{lambda} offspring for selecting 
#' \code{mu} new candidates. 
#' Out of those \code{mu} features, the new \code{lambda} features are generated by randomly choosing 
#' pairs of parents. These are crossed over and \code{crossover.rate}
#' represents the probability of choosing a feature from the first parent instead of
#' the second parent. 
#' The resulting offspring is mutated, i.e. its bits are flipped with probability \code{mutation.rate}.
#' If \code{max.features} is set, offspring are repeatedly generated until the setting is satisfied. 
#' 
#' @param same.resampling.instance [\code{logical(1)}]\cr
#'   Should the same resampling instance be used for all evaluations to reduce variance?
#'   Default is \code{TRUE}.
#' @param maxit [\code{integer(1)}]\cr
#'   Maximal number of iterations. Note, that this is usually not equal to the number
#'   of function evaluations.
#' @param max.features [\code{integer(1)}]\cr
#'   Maximal number of features. 
#' @param crossover.rate [\code{numeric(1)}]\cr
#'   Parameter of the GA feature selection. Probability of choosing a bit from the first parent 
#'   within the crossover mutation.
#' @param mutation.rate [\code{numeric(1)}]\cr
#'   Parameter of the GA feature selection. Probability of flipping a feature bit, i.e. switch 
#'   between selecting / deselecting a feature.
#' @param comma [\code{logical(1)}]\cr
#'   Parameter of the GA feature selection, indicating whether to use a (\code{mu}, \code{lambda})
#'   or (\code{mu} + \code{lambda}) GA. The default is \code{FALSE}.
#' @param mu [\code{integer(1)}]\cr
#'   Parameter of the GA feature selection. Size of the parent population.
#' @param lambda [\code{integer(1)}]\cr
#'   Parameter of the GA feature selection. Size of the children population (should be smaller 
#'   or equal to \code{mu}).
#' @param prob [\code{numeric(1)}]\cr
#'   Parameter of the random feature selection. Probability of choosing a feature.
#' @param method [\code{character(1)}]\cr
#'   Parameter of the sequential feature selection. A character representing the method. Possible 
#'   values are \code{sfs} (forward search), \code{sbs} (backward search), \code{sffs}
#'   (floating forward search) and \code{sfbs} (floating backward search).
#' @param alpha [\code{numeric(1)}]\cr
#'   Parameter of the sequential feature selection. Minimal value of improvement. 
#' @param beta [\code{numeric(1)}]\cr
#'   Parameter of the sequential feature selection. Maximal value of setback. 
#'  
#' @return [\code{\link{FeatSelControl}}]. The specific subclass is one of
#'   \code{\link{FeatSelControlExhaustive}}, \code{\link{FeatSelControlRandom}}, 
#'   \code{\link{FeatSelControlSequential}}, \code{\link{FeatSelControlGA}}.
#' @name FeatSelControl
#' @rdname FeatSelControl
#' @aliases FeatSelControlExhaustive FeatSelControlRandom FeatSelControlSequential FeatSelControlGA
NULL

makeFeatSelControl = function(same.resampling.instance, maxit, max.features, ..., cl) {
  checkArg(same.resampling.instance, "logical", len=1, na.ok=FALSE)
  maxit = convertInteger(maxit)
  checkArg(maxit, "integer", len=1L, lower=1L, na.ok=TRUE)
  max.features = convertInteger(max.features)
  checkArg(max.features, "integer", len=1L, lower=1L, na.ok=TRUE)
	x = makeOptControl(same.resampling.instance=same.resampling.instance, ...)
  x$maxit = maxit
  x$max.features = max.features
  class(x) = c(cl, "FeatSelControl", class(x))
  return(x)
}

#' @S3method print FeatSelControl
print.FeatSelControl = function(x, ...) {
  catf("FeatSel control: %s", class(x)[1])
  catf("Same resampling instance: %s", x$same.resampling.instance)
  if (is.na(x$max.features))
    catf("Max. features: <not used>")
  else  
    catf("Max. features: %i", x$max.features)
  catf("Max. iterations: %i", x$maxit)
  if (length(x$extra.args) > 0)
    catf("Further arguments: %s", listToShortString(x$extra.args))
  else
    catf("Further arguments: <not used>")
}

