% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/collin.diag.R
\name{collin.diag}
\alias{collin.diag}
\title{Collinearity Diagnostics}
\usage{
collin.diag(model, print = c("all", "vif", "eigen"), digits = 3, p.digits = 3,
            check = TRUE, output = TRUE)
}
\arguments{
\item{model}{a fitted model of class \code{"lm"}, \code{"glm"}, \code{"lmerMod"}, \code{"lmerModLmerTest"},
\code{"glmerMod"}, \code{"lme"}, or \code{"glmmTMB"}.}

\item{print}{a character vector indicating which results to show, i.e. \code{"all"}, for all results,
\code{"vif"} for tolerance, std. error inflation factor, and variance inflation factor,
or \code{eigen} for eigenvalue, condition index, and variance proportions.}

\item{digits}{an integer value indicating the number of decimal places to be used for displaying
results.}

\item{p.digits}{an integer value indicating the number of decimal places to be used for displaying the
\emph{p}-value.}

\item{check}{logical: if \code{TRUE}, argument specification is checked.}

\item{output}{logical: if \code{TRUE}, output is shown on the console.}
}
\value{
Returns an object of class \code{misty.object}, which is a list with following entries:
function call (\code{call}), type of analysis \code{type}, model specified in the
\code{model} argument (\code{model}), specification of function arguments (\code{args}),
list with results (\code{result}).
}
\description{
This function computes tolerance, standard error inflation factor, variance inflation factor, eigenvalues,
condition index, and variance proportions for linear, generalized linear, and mixed-effects models.
}
\details{
Collinearity diagnostics can be conducted for objects returned from the \code{lm()} and \code{glm()}
function, but also from objects returned from the \code{lmer()} and \code{glmer()} function from
the \pkg{lme4} package, \code{lme()} function from the \pkg{nlme} package, and the \code{glmmTMB()}
function from the \pkg{glmmTMB} package.

The generalized variance inflation factor (Fox & Monette, 1992) is computed for terms with more than
1 df resulting from factors with more than two levels. The generalized VIF (GVIF) is interpretable
as the inflation in size of the confidence ellipse or ellipsoid for the coefficients of the term
in comparison with what would be obtained for orthogonal data. GVIF is invariant to the coding of
the terms in the model. In order to adjust for the dimension of the confidence ellipsoid,
GVIF\eqn{^\frac{1}{2df}} is computed. Note that the adjusted GVIF (aGVIF) is actually a generalized
standard error inflation factor (GSIF). Thus, the aGIF needs to be squared before applying a common
cutoff threshold for the VIF (e.g., VIF > 10). Note that the output of \code{collin.diag()} function
reports either the variance inflation factor or the squared generalized variance inflation factor
in the column \code{VIF}, while the standard error inflation factor or the adjusted generalized
variance inflation factor is reported in the column \code{SIF}.
}
\note{
The computation of the VIF and the GVIF is based on the \code{vif()} function in the
\pkg{car} package by John Fox, Sanford Weisberg and Brad Price (2020), and the computation
of eigenvalues, condition index, and variance proportions is based on the \code{ols_eigen_cindex()}
function in the \pkg{olsrr} package by Aravind Hebbali (2020).
}
\examples{
dat <- data.frame(group = c(1, 1, 1, 2, 2, 2, 3, 3, 3, 4, 4, 4),
                  x1 = c(3, 2, 4, 9, 5, 3, 6, 4, 5, 6, 3, 5),
                  x2 = c(1, 4, 3, 1, 2, 4, 3, 5, 1, 7, 8, 7),
                  x3 = c(7, 3, 4, 2, 5, 6, 4, 2, 3, 5, 2, 8),
                  x4 = c("a", "b", "a", "c", "c", "c", "a", "b", "b", "c", "a", "c"),
                  y1 = c(2, 7, 4, 4, 7, 8, 4, 2, 5, 1, 3, 8),
                  y2 = c(0, 1, 0, 1, 1, 1, 0, 0, 0, 1, 0, 1),
                  stringsAsFactors = TRUE)

#----------------------------
# Linear model

# Estimate linear model with continuous predictors
mod.lm1 <- lm(y1 ~ x1 + x2 + x3, data = dat)

# Tolerance, std. error, and variance inflation factor
collin.diag(mod.lm1)

# Tolerance, std. error, and variance inflation factor
# Eigenvalue, Condition index, and variance proportions
collin.diag(mod.lm1, print = "all")

# Estimate model with continuous and categorical predictors
mod.lm2 <- lm(y1 ~ x1 + x2 + x3 + x4, data = dat)

# Tolerance, generalized std. error, and variance inflation factor
collin.diag(mod.lm2)

#----------------------------
# Generalized linear model

# Estimate logistic regression model with continuous predictors
mod.glm <- glm(y2 ~ x1 + x2 + x3, data = dat, family = "binomial")

# Tolerance, std. error, and variance inflation factor
collin.diag(mod.glm)

\dontrun{
#----------------------------
# Linear mixed-effects model

# Estimate linear mixed-effects model with continuous predictors using lme4 package
mod.lmer <- lme4::lmer(y1 ~ x1 + x2 + x3 + (1|group), data = dat)

# Tolerance, std. error, and variance inflation factor
collin.diag(mod.lmer)

# Estimate linear mixed-effects model with continuous predictors using nlme package
mod.lme <- nlme::lme(y1 ~ x1 + x2 + x3, random = ~ 1 | group, data = dat)

# Tolerance, std. error, and variance inflation factor
collin.diag(mod.lme)

# Estimate linear mixed-effects model with continuous predictors using glmmTMB package
mod.glmmTMB1 <- glmmTMB::glmmTMB(y1 ~ x1 + x2 + x3 + (1|group), data = dat)

# Tolerance, std. error, and variance inflation factor
collin.diag(mod.glmmTMB1)

#----------------------------
# Generalized linear mixed-effects model

# Estimate mixed-effects logistic regression model with continuous predictors using lme4 package
mod.glmer <- lme4::glmer(y2 ~ x1 + x2 + x3 + (1|group), data = dat, family = "binomial")

# Tolerance, std. error, and variance inflation factor
collin.diag(mod.glmer)

# Estimate mixed-effects logistic regression model with continuous predictors using glmmTMB package
mod.glmmTMB2 <- glmmTMB::glmmTMB(y2 ~ x1 + x2 + x3 + (1|group), data = dat, family = "binomial")

# Tolerance, std. error, and variance inflation factor
collin.diag(mod.glmmTMB2)
}
}
\references{
Fox, J., & Monette, G. (1992). Generalized collinearity diagnostics.
\emph{Journal of the American Statistical Association, 87}, 178-183.

Fox, J., Weisberg, S., & Price, B. (2020). \emph{car: Companion to Applied Regression}.
R package version 3.0-8. https://cran.r-project.org/web/packages/car/

Hebbali, A. (2020). \emph{olsrr: Tools for building OLS regression models}.
R package version 0.5.3. https://cran.r-project.org/web/packages/olsrr/
}
\author{
Takuya Yanagida \email{takuya.yanagida@univie.ac.at}
}
