% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/motif_census.R
\name{node_census}
\alias{node_census}
\alias{node_tie_census}
\alias{node_triad_census}
\alias{node_quad_census}
\alias{node_path_census}
\title{Censuses of nodes' motifs}
\usage{
node_tie_census(.data)

node_triad_census(.data)

node_quad_census(.data)

node_path_census(.data)
}
\arguments{
\item{.data}{An object of a \code{{manynet}}-consistent class:
\itemize{
\item matrix (adjacency or incidence) from \code{{base}} R
\item edgelist, a data frame from \code{{base}} R or tibble from \code{{tibble}}
\item igraph, from the \code{{igraph}} package
\item network, from the \code{{network}} package
\item tbl_graph, from the \code{{tidygraph}} package
}}
}
\description{
These functions include ways to take a census of the positions of nodes
in a network. These include a triad census based on the triad profile
of nodes, but also a tie census based on the particular tie partners
of nodes. Included also are group census functions for summarising
the profiles of clusters of nodes in a network.

#' @export
node_igraph_census <- function(.data, normalized = FALSE){
out <- igraph::motifs(manynet::as_igraph(.data), 4)
if(manynet::is_labelled(.data))
rownames(out) <- manynet::node_names(.data)
colnames(out) <- c("co-K4",
"co-diamond",
"co-C4",
"co-paw",
"co-claw",
"P4",
"claw",
"paw",
"C4",
"diamond",
"K4")
make_node_motif(out, .data)
}
}
\details{
The quad census uses the \code{{oaqc}} package to do
the heavy lifting of counting the number of each orbits.
See \code{vignette('oaqc')}.
However, our function relabels some of the motifs
to avoid conflicts and improve some consistency with
other census-labelling practices.
The letter-number pairing of these labels indicate
the number and configuration of ties.
For now, we offer a rough translation:\tabular{ll}{
   migraph \tab Ortmann and Brandes \cr
   E4 \tab co-K4 \cr
   I40, I41 \tab co-diamond \cr
   H4 \tab co-C4 \cr
   L42, L41, L40 \tab co-paw \cr
   D42, D40 \tab co-claw \cr
   U42, U41 \tab P4 \cr
   Y43, Y41 \tab claw \cr
   P43, P42, P41 \tab paw \cr
   04 \tab C4 \cr
   Z42, Z43 \tab diamond \cr
   X4 \tab K4 \cr
}


See also \href{https://www.graphclasses.org/smallgraphs.html#nodes4}{this list of graph classes}.
}
\section{Functions}{
\itemize{
\item \code{node_tie_census()}: Returns a census of the ties in a network.
For directed networks, out-ties and in-ties are bound together.

\item \code{node_triad_census()}: Returns a census of the triad configurations
nodes are embedded in.

\item \code{node_quad_census()}: Returns a census of nodes' positions
in motifs of four nodes.

\item \code{node_path_census()}: Returns the shortest path lengths
of each node to every other node in the network.

}}
\examples{
task_eg <- manynet::to_named(manynet::to_uniplex(manynet::ison_algebra, "task_tie"))
(tie_cen <- node_tie_census(task_eg))
(triad_cen <- node_triad_census(task_eg))
node_quad_census(manynet::ison_southern_women)
node_path_census(manynet::ison_adolescents)
node_path_census(manynet::ison_southern_women)
}
\references{
Davis, James A., and Samuel Leinhardt. 1967.
“\href{https://files.eric.ed.gov/fulltext/ED024086.pdf}{The Structure of Positive Interpersonal Relations in Small Groups}.” 55.

Ortmann, Mark, and Ulrik Brandes. 2017.
“Efficient Orbit-Aware Triad and Quad Census in Directed and Undirected Graphs.”
\emph{Applied Network Science} 2(1):13.
\doi{10.1007/s41109-017-0027-2}.

Dijkstra, Edsger W. 1959.
"A note on two problems in connexion with graphs".
\emph{Numerische Mathematik} 1, 269-71.
\doi{10.1007/BF01386390}.

Opsahl, Tore, Filip Agneessens, and John Skvoretz. 2010.
"Node centrality in weighted networks: Generalizing degree and shortest paths".
\emph{Social Networks} 32(3): 245-51.
\doi{10.1016/j.socnet.2010.03.006}.
}
\seealso{
Other motifs: 
\code{\link{brokerage_census}},
\code{\link{network_census}}
}
\concept{motifs}
